Attribute VB_Name = "Sea_Ice_Vap_4_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules:
'     Constants_0_Mdl,  file Constants_0.bas
'     Maths_0_Mdl,      file Maths_0.bas
'     Flu_1_Mdl,        file Flu_1.bas
'     Ice_1_Mdl,        file Ice_1.bas
'     Sal_1_Mdl,        file Sal_1.bas
'     Sal_2_Mdl,        file Sal_2.bas

'#########################################################################

'=========================================================================
'This module implements the phase equilibrium properties of sea ice with vapour
'computed from IAPWS-95, IAPWS-06 and IAPWS-08:

'Release on the IAPWS Formulation 1995 for the Thermodynamic Properties of
'Ordinary Water Substance for General and Scientific Use
'The International Association for the Properties of Water and Steam
'Fredericia, Denmark, September 1996

'Release on an Equation of State for H2O Ice Ih
'The International Association for the Properties of Water and Steam
'Witney, UK, September 2006

'Release on the IAPWS Formulation 2008 for the Thermodynamic Properties of Seawater
'The International Association for the Properties of Water and Steam
'Berlin, Germany, September 2008

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98
'Private Const IsOK = -1

'Control parameters of the sea-ice-vap iteration
Private ctrl_initialized As Integer
'
Private ctrl_mode_salinity As Integer
Private ctrl_mode_temperature As Integer
Private ctrl_mode_pressure As Integer
Private ctrl_mode_liquid As Integer
Private ctrl_mode_vapour As Integer
'
Private ctrl_loop_maximum As Long
'
Private ctrl_init_s As Double        'initial brine salinity
Private ctrl_init_t As Double        'initial temperature
Private ctrl_init_p As Double        'initial pressure
Private ctrl_init_d_liq As Double    'initial liquid density
Private ctrl_init_d_vap As Double    'initial vapour density
'
Private ctrl_eps_exit_t As Double    'temperature exit tolerance
Private ctrl_eps_exit_p As Double    'pressure exit tolerance
'
'Properties of the current sea-ice-vapour equilibrium state
Private equi_sea_ice_vap_done As Integer  'flag = IsOK if equilibrium was successfully computed
'
Private equi_sea_ice_vap_t As Double      'equilibrium temperature
Private equi_sea_ice_vap_p As Double      'equilibrium pressure
Private equi_sea_ice_vap_s As Double      'equilibrium brine salinity
Private equi_sea_ice_vap_d_liq As Double  'equilibrium density of liquid water at s, t, p
Private equi_sea_ice_vap_d_vap As Double  'equilibrium density of water vapour at s, t, p

Private Const Version = "16 Sep 2009"

'==========================================================================
Public Function set_sea_ice_vap_eq_at_s(ByVal sa_si As Double) As Double

'this function computes the sea-ice-vapour equilibrium at given salinity sa_si in kg/kg
'by Newton iteration

Const Tt = TP_temperature_si
Const Pt = TP_pressure_IAPWS95_si
Const dt_liq = TP_density_liq_IAPWS95_si  'triple point density of liquid water
Const dt_vap = TP_density_vap_IAPWS95_si  'triple point density of water vapour
'
Dim t As Double, p As Double, dl As Double, dv As Double, eps As Double
Dim maxit As Long

If equi_sea_ice_vap_done = IsOK And _
   sa_si = equi_sea_ice_vap_s Then
  'the requested state has already been computed earlier
  set_sea_ice_vap_eq_at_s = IsOK
  Exit Function
End If

clear_sea_ice_vap_state 'waste any previous state

set_sea_ice_vap_eq_at_s = ErrorReturn

If sa_si < 0 Then Exit Function
If sa_si >= 1 Then Exit Function

If sa_si = 0 Then  'pure water: set IAPWS-95 triple point
  set_sea_ice_vap_state sa_si, Tt, Pt, dt_vap, dt_liq
  set_sea_ice_vap_eq_at_s = IsOK
  Exit Function
End If

init_it_ctrl_sea_ice_vap

'set initial temperature guess
Select Case ctrl_mode_temperature
  Case 0:  t = aux_temperature_si(sa_si)
  Case -1: t = Tt
  Case 1:  t = ctrl_init_t
  Case Else: Exit Function
End Select
If t <= 0 Then Exit Function
If t = ErrorReturn Then Exit Function

'set initial pressure guess
Select Case ctrl_mode_pressure
  Case 0:  p = aux_pressure_si(sa_si)
  Case -1: p = Pt
  Case 1:  p = ctrl_init_p
  Case Else: Exit Function
End Select
If p <= 0 Then Exit Function
If p = ErrorReturn Then Exit Function

'set initial liquid density guess
Select Case ctrl_mode_liquid
  Case 0:  dl = dt_liq
  Case -1: dl = dt_liq
  Case 1:  dl = ctrl_init_d_liq
  Case Else: Exit Function
End Select
If dl <= 0 Then Exit Function
If dl = ErrorReturn Then Exit Function

'set initial vapour density guess
Select Case ctrl_mode_vapour
  Case 0:  dv = aux_density_ideal_si(t, p)
  Case -1: dv = dt_vap
  Case 1:  dv = ctrl_init_d_vap
  Case Else: Exit Function
End Select
If dv <= 0 Then Exit Function
If dv = ErrorReturn Then Exit Function

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: set_sea_ice_vap_state sa_si, t, p, dv, dl
           set_sea_ice_vap_eq_at_s = IsOK
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit
eps = ctrl_eps_exit_t
If eps = 0 Then Exit Function

'run iteration loop
If sea_ice_vap_iteration_at_s(sa_si, maxit, eps, t, p, dv, dl) = ErrorReturn Then
  Exit Function
End If

set_sea_ice_vap_state sa_si, t, p, dv, dl

set_sea_ice_vap_eq_at_s = IsOK

End Function

'==========================================================================
Public Function set_sea_ice_vap_eq_at_t(ByVal t_si As Double) As Double

'this function computes the sea-ice-vapour equilibrium at given absolute temperature t_si in K
'by Newton iteration

Const Tt = TP_temperature_si
Const Pt = TP_pressure_IAPWS95_si
Const dt_liq = TP_density_liq_IAPWS95_si  'triple point density of liquid water
Const dt_vap = TP_density_vap_IAPWS95_si  'triple point density of water vapour
'
Dim S As Double, p As Double, dl As Double, dv As Double, eps As Double
Dim maxit As Long

If equi_sea_ice_vap_done = IsOK And _
   t_si = equi_sea_ice_vap_t Then
  'the requested state has already been computed earlier
  set_sea_ice_vap_eq_at_t = IsOK
  Exit Function
End If

clear_sea_ice_vap_state 'waste any previous state

set_sea_ice_vap_eq_at_t = ErrorReturn

If t_si <= 0 Then Exit Function
If t_si > Tt Then Exit Function

If t_si = Tt Then  'pure water: set IAPWS-95 triple point
  set_sea_ice_vap_state 0, Tt, Pt, dt_vap, dt_liq
  set_sea_ice_vap_eq_at_t = IsOK
  Exit Function
End If

init_it_ctrl_sea_ice_vap

'set initial salinity guess
Select Case ctrl_mode_salinity
  Case 0:  S = aux_salinity_of_t_si(t_si)
  Case -1: S = SO_salinity_si
  Case 1:  S = ctrl_init_s
  Case Else: Exit Function
End Select
If S <= 0 Or S >= 1 Then Exit Function
If S = ErrorReturn Then Exit Function

'set initial pressure guess
Select Case ctrl_mode_pressure
  Case 0:  p = aux_pressure_si(S)
  Case -1: p = Pt
  Case 1:  p = ctrl_init_p
  Case Else: Exit Function
End Select
If p <= 0 Then Exit Function
If p = ErrorReturn Then Exit Function

'set initial liquid density guess
Select Case ctrl_mode_liquid
  Case 0:  dl = dt_liq
  Case -1: dl = dt_liq
  Case 1:  dl = ctrl_init_d_liq
  Case Else: Exit Function
End Select
If dl <= 0 Then Exit Function
If dl = ErrorReturn Then Exit Function

'set initial vapour density guess
Select Case ctrl_mode_vapour
  Case 0:  dv = aux_density_ideal_si(t_si, p)
  Case -1: dv = dt_vap
  Case 1:  dv = ctrl_init_d_vap
  Case Else: Exit Function
End Select
If dv <= 0 Then Exit Function
If dv = ErrorReturn Then Exit Function

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: set_sea_ice_vap_state S, t_si, p, dv, dl
           set_sea_ice_vap_eq_at_t = IsOK
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit
eps = ctrl_eps_exit_p
If eps = 0 Then Exit Function

'run iteration loop
If sea_ice_vap_iteration_at_t(t_si, maxit, eps, S, p, dv, dl) = ErrorReturn Then
  Exit Function
End If

set_sea_ice_vap_state S, t_si, p, dv, dl

set_sea_ice_vap_eq_at_t = IsOK

End Function

'==========================================================================
Public Function set_sea_ice_vap_eq_at_p(ByVal p_si As Double) As Double

'this function computes the sea-ice-vapour equilibrium at given absolute pressure p_si in Pa
'by Newton iteration

Const Tt = TP_temperature_si
Const Pt = TP_pressure_IAPWS95_si
Const dt_liq = TP_density_liq_IAPWS95_si  'triple point density of liquid water
Const dt_vap = TP_density_vap_IAPWS95_si  'triple point density of water vapour
'
Dim S As Double, t As Double, dl As Double, dv As Double, eps As Double
Dim maxit As Long

If equi_sea_ice_vap_done = IsOK And _
   p_si = equi_sea_ice_vap_p Then
  'the requested state has already been computed earlier
  set_sea_ice_vap_eq_at_p = IsOK
  Exit Function
End If

clear_sea_ice_vap_state 'waste any previous state

set_sea_ice_vap_eq_at_p = ErrorReturn

If p_si <= 0 Then Exit Function
If p_si > Pt Then Exit Function

If p_si = Pt Then  'pure water: set IAPWS-95 triple point
  set_sea_ice_vap_state 0, Tt, Pt, dt_vap, dt_liq
  set_sea_ice_vap_eq_at_p = IsOK
  Exit Function
End If

init_it_ctrl_sea_ice_vap

'set initial salinity guess
Select Case ctrl_mode_salinity
  Case 0:  S = aux_salinity_of_p_si(p_si)
  Case -1: S = SO_salinity_si
  Case 1:  S = ctrl_init_s
  Case Else: Exit Function
End Select
If S <= 0 Or S >= 1 Then Exit Function
If S = ErrorReturn Then Exit Function

'set initial temperature guess
Select Case ctrl_mode_temperature
  Case 0:  t = aux_temperature_si(S)
  Case -1: t = Tt
  Case 1:  t = ctrl_init_t
  Case Else: Exit Function
End Select
If t <= 0 Then Exit Function
If t = ErrorReturn Then Exit Function

'set initial liquid density guess
Select Case ctrl_mode_liquid
  Case 0:  dl = dt_liq
  Case -1: dl = dt_liq
  Case 1:  dl = ctrl_init_d_liq
  Case Else: Exit Function
End Select
If dl <= 0 Then Exit Function
If dl = ErrorReturn Then Exit Function

'set initial vapour density guess
Select Case ctrl_mode_vapour
  Case 0:  dv = aux_density_ideal_si(t, p_si)
  Case -1: dv = dt_vap
  Case 1:  dv = ctrl_init_d_vap
  Case Else: Exit Function
End Select
If dv <= 0 Then Exit Function
If dv = ErrorReturn Then Exit Function

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: set_sea_ice_vap_state S, t, p_si, dv, dl
           set_sea_ice_vap_eq_at_p = IsOK
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit
eps = ctrl_eps_exit_t
If eps = 0 Then Exit Function

'run iteration loop
If sea_ice_vap_iteration_at_p(p_si, maxit, eps, S, t, dv, dl) = ErrorReturn Then
  Exit Function
End If

set_sea_ice_vap_state S, t, p_si, dv, dl

set_sea_ice_vap_eq_at_p = IsOK

End Function

'==========================================================================
Public Function sea_ice_vap_salinity_si() As Double

'this function returns the absolute brine salinity in kg/kg of sea ice at equilibrium with vapour,
'set by a previous call of either set_sea_ice_vap_eq_at_s,
'set_sea_ice_vap_eq_at_t or set_sea_ice_vap_eq_at_p

'Note: the accuracy of this function depends on the iteration settings in this module

'check values with default settings:
'set_sea_ice_vap_eq_at_s 0.035
'sea_ice_vap_salinity_si = 0.035

'set_sea_ice_vap_eq_at_t 270
'sea_ice_vap_salinity_si = 5.61489288505582E-02

'set_sea_ice_vap_eq_at_p 500
'sea_ice_vap_salinity_si = 4.3895587882|7614E-02

sea_ice_vap_salinity_si = ErrorReturn

If equi_sea_ice_vap_done <> IsOK Then Exit Function

sea_ice_vap_salinity_si = equi_sea_ice_vap_s

End Function

'==========================================================================
Public Function sea_ice_vap_density_vap_si() As Double

'this function returns the vapour density in kg m-3 of sea ice at equilibrium with vapour,
'set by a previous call of either set_sea_ice_vap_eq_at_s,
'set_sea_ice_vap_eq_at_t or set_sea_ice_vap_eq_at_p

'Note: the accuracy of this function depends on the iteration settings in this module

'check values with default settings:
'set_sea_ice_vap_eq_at_s 0.035
'sea_ice_vap_density_vap_si = 4.17156419317765E-03

'set_sea_ice_vap_eq_at_t 270
'sea_ice_vap_density_vap_si = 3.7740614077202E-03

'set_sea_ice_vap_eq_at_p 500
'sea_ice_vap_density_vap_si = 4.00364833230323E-03

sea_ice_vap_density_vap_si = ErrorReturn

If equi_sea_ice_vap_done <> IsOK Then Exit Function

sea_ice_vap_density_vap_si = equi_sea_ice_vap_d_vap

End Function

'==========================================================================
Public Function sea_ice_vap_temperature_si() As Double

'this function returns the absolute temperature in K of sea ice at equilibrium with vapour,
'set by a previous call of either set_sea_ice_vap_eq_at_s,
'set_sea_ice_vap_eq_at_t or set_sea_ice_vap_eq_at_p

'Note: the accuracy of this function depends on the iteration settings in this module

'check values with default settings:
'set_sea_ice_vap_eq_at_s 0.035
'sea_ice_vap_temperature_si = 271.247815056812

'set_sea_ice_vap_eq_at_t 270
'sea_ice_vap_temperature_si = 270

'set_sea_ice_vap_eq_at_p 500
'sea_ice_vap_temperature_si = 270.734430916974

sea_ice_vap_temperature_si = ErrorReturn

If equi_sea_ice_vap_done <> IsOK Then Exit Function

sea_ice_vap_temperature_si = equi_sea_ice_vap_t

End Function

'==========================================================================
Public Function sea_ice_vap_pressure_si() As Double

'this function returns the absolute pressure in Pa of sea ice at equilibrium with vapour,
'set by a previous call of either set_sea_ice_vap_eq_at_s,
'set_sea_ice_vap_eq_at_t or set_sea_ice_vap_eq_at_p

'Note: the accuracy of this function depends on the iteration settings in this module

'check values with default settings:
'set_sea_ice_vap_eq_at_s 0.035
'sea_ice_vap_pressure_si = 521.950349224785

'set_sea_ice_vap_eq_at_t 270
'sea_ice_vap_pressure_si = 470.059067980885

'set_sea_ice_vap_eq_at_p 500
'sea_ice_vap_pressure_si = 500

sea_ice_vap_pressure_si = ErrorReturn

If equi_sea_ice_vap_done <> IsOK Then Exit Function

sea_ice_vap_pressure_si = equi_sea_ice_vap_p

End Function

'==========================================================================
Private Sub init_it_ctrl_sea_ice_vap()

Const s_norm = SO_salinity_si  'kg/kg of KCl normalized seawater
Const Tt = TP_temperature_si
Const Pt = TP_pressure_IAPWS95_si
Const dlt = TP_density_liq_IAPWS95_si  'triple point density of liquid water
Const dvt = TP_density_vap_IAPWS95_si  'triple point density of water vapour

If ctrl_initialized = IsOK Then Exit Sub

ctrl_initialized = IsOK

'Set default values and modes for the iteration
ctrl_mode_liquid = 0
ctrl_mode_vapour = 0
ctrl_mode_salinity = 0
ctrl_mode_temperature = 0
ctrl_mode_pressure = 0
ctrl_loop_maximum = 100
ctrl_init_s = s_norm
ctrl_init_t = Tt
ctrl_init_p = Pt
ctrl_init_d_liq = dlt
ctrl_init_d_vap = dvt
'
ctrl_eps_exit_t = -0.0000001 'relative, 0.1 ppm
ctrl_eps_exit_p = -0.0000001 'relative, 0.1 ppm

End Sub

'==========================================================================
Public Sub set_it_ctrl_sea_ice_vap(ByVal key As String, ByVal value As Double)

'this sub sets control parameters for the iteration used to compute
'sea-ice-vapour equilibrium

'key              value
'it_steps         0           set iteration number to default (100)
'it_steps         n > 0       set iteration number to n
'it_steps        -1           do not iterate, use initial values to compute equlibrium

'init_liq_dens    0           use default liquid density to start, d = triple point liquid density
'init_liq_dens   -1           use triple point liquid density to start
'init_liq_dens    d > 0       use value d as liquid density to start

'init_vap_dens    0           use default vapour density to start, d = aux_density_ideal_si(T, P)
'init_vap_dens   -1           use triple point vapour density to start
'init_vap_dens    d > 0       use value d as vapour density to start

'init_brine_sa    0           use default brine salinity to start (fitted guess function aux_salinity)
'init_brine_sa   -1           use normal salinity to start
'init_brine_sa    s > 0       use value s as temperature to start

'init_temp        0           use default temperature to start (fitted guess function aux_temperature)
'init_temp       -1           use triple point temperature to start
'init_temp        t > 0       use value t as temperature to start

'init_press       0           use default pressure to start (fitted guess function aux_pressure)
'init_press      -1           use triple point pressure to start
'init_press       p > 0       use value p as pressure to start

'tol_temp         0           use default exit accuracy for freezing temperature (0.1 ppm)
'tol_temp         eps         use eps as exit accuracy for temperature (eps < 0 means relative error)

'tol_press        0           use default exit accuracy for melting pressure (0.1 ppm)
'tol_press        eps         use eps as exit accuracy for pressure (eps < 0 means relative error)

init_it_ctrl_sea_ice_vap

clear_sea_ice_vap_state

Select Case LCase(Trim(key))

  Case "it_steps":   'iteration steps
    Select Case value
      Case 0:      ctrl_loop_maximum = 100  'default = 100
      Case Is < 0: ctrl_loop_maximum = -1
      Case Else:   ctrl_loop_maximum = value
    End Select

  Case "init_vap_dens":   'starting vapour density
    Select Case CLng(value)
      Case 0:       ctrl_mode_vapour = 0    'default = ideal gas
      Case Is < -1: 'ignore it
      Case Is < 0:  ctrl_mode_vapour = value
      Case Else:    ctrl_mode_vapour = 1
                    ctrl_init_d_vap = value
    End Select

  Case "init_liq_dens":   'starting liquid density
    Select Case CLng(value)
      Case 0:       ctrl_mode_liquid = 0    'default = triple point
      Case Is < -1: 'ignore it
      Case Is < 0:  ctrl_mode_liquid = value
      Case Else:    ctrl_mode_liquid = 1
                    ctrl_init_d_liq = value
    End Select

  Case "init_brine_sa":   'starting brine salinity
    Select Case CLng(value)
      Case 0:       ctrl_mode_salinity = 0    'default = aux function
      Case Is < -1: 'ignore it
      Case Is < 0:  ctrl_mode_salinity = value
      Case Else:    ctrl_mode_salinity = 1
                    ctrl_init_s = value
    End Select

  Case "init_temp":   'starting temperature
    Select Case CLng(value)
      Case 0:       ctrl_mode_temperature = 0    'default = aux function
      Case Is < -1: 'ignore it
      Case Is < 0:  ctrl_mode_temperature = value
      Case Else:    ctrl_mode_temperature = 1
                    ctrl_init_t = value
    End Select

  Case "init_press":   'starting pressure
    Select Case CLng(value)
      Case 0:       ctrl_mode_pressure = 0    'default = aux function
      Case Is < -1: 'ignore it
      Case Is < 0:  ctrl_mode_pressure = value
      Case Else:    ctrl_mode_pressure = 1
                    ctrl_init_p = value
    End Select

  Case "tol_temp":      'required temperature tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_t = -0.0000001   'default = 0.1 ppm relative
      Case Else:   ctrl_eps_exit_t = value
    End Select

  Case "tol_press":      'required pressure tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_p = -0.0000001   'default = 0.1 ppm relative
      Case Else:   ctrl_eps_exit_p = value
    End Select

End Select

End Sub

'==========================================================================
Private Function sea_ice_vap_iteration_at_s(ByVal sa_si As Double, _
                                            ByVal maxit As Long, _
                                            ByVal eps As Double, _
                                            ByRef t_si As Double, _
                                            ByRef p_si As Double, _
                                            ByRef d_vap_si As Double, _
                                            ByRef d_liq_si As Double) As Double

'this function computes the seawater-ice-vapour phase equilibrium from
'equal chemical potentials of water at given salinity, sa_si,
'from initial guesses for the temperature, t_si, pressure, p_si,
'the vapour density, d_vap_si and the liquid water density, d_liq_si.
'The iteration limit eps refers to the error in temperature.

'output:    sea_ice_vap_iteration_at_s = IsOK if successfully done
'           sea_ice_vap_iteration_at_s = ErrorReturn is returned if
'           - the maximum number of iterations is exceeded without meeting the exit criterion
'           - the function call to a Gibbs/Helmholtz function has returned an error
'           - density, temperature or pressure have taken a zero or negative value during the iteration
'           - s-t-p values do not permit the stable existence of ice
'  t_si:    absolute temperature in K at the seawater-ice equilibrium
'  p_si:    absolute pressure in Pa at the seawater-ice equilibrium
' d_vap_si: density in kg/m3 of water vapour at (t_si, p_si)
' d_liq_si: density in kg/m3 of liquid water at (t_si, p_si)

'input: sa_si: absolute salinity in kg/kg
'       maxit: maximum number of iteration steps to be done
'         eps: required accuracy of temperature
'              eps > 0: absolute temperature tolerance in K
'              eps < 0: relative temperature tolerance
'        t_si: initial guess for absolute (freezing) temperature in K
'        p_si: initial guess for absolute pressure in Pa
'    d_vap_si: initial guess for density in kg/m3 of water vapour at (t_si, p_si)
'    d_liq_si: initial guess for density in kg/m3 of liquid water at (t_si, p_si)

Dim dv As Double, dl As Double, t As Double, p As Double
Dim ddv As Double, ddl As Double, dt As Double, dp As Double
Dim gi As Double, gi_t As Double, gi_p As Double
Dim gs As Double, gs_s As Double
Dim gs_t As Double, gs_st As Double
Dim gs_p As Double, gs_sp As Double
Dim fv As Double, fv_t As Double, fv_td As Double
Dim fv_d As Double, fv_dd As Double
Dim fl As Double, fl_t As Double, fl_td As Double
Dim fl_d As Double, fl_dd As Double

Dim it As Long

Dim a(4, 4) As Double, b(4) As Double, x(4) As Double

sea_ice_vap_iteration_at_s = ErrorReturn

If d_vap_si <= 0 Or _
   d_liq_si <= 0 Or _
   t_si <= 0 Or _
   p_si <= 0 Then
  t_si = ErrorReturn
  p_si = ErrorReturn
  d_liq_si = ErrorReturn
  d_vap_si = ErrorReturn
  Exit Function
End If

If check_limits <> 1 Then
  If sa_si <= 0 Or sa_si >= 1 Then
    t_si = ErrorReturn
    p_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_vap_si = ErrorReturn
    Exit Function
  End If
Else
  'SAL_LIMITS
  If sa_si < sal_smin Or sa_si > sal_smax Then
    t_si = ErrorReturn
    p_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_vap_si = ErrorReturn
    Exit Function
  End If
End If

check_limits = check_limits - 1

t = t_si
p = p_si
dl = d_liq_si
dv = d_vap_si

For it = 1 To maxit

  'Derivatives of the Helmholtz function of water vapour for 4D Newton iteration
  fv = flu_f_si(0, 0, t, dv)
  If fv = ErrorReturn Then Exit For
  fv_t = flu_f_si(1, 0, t, dv)
  If fv_t = ErrorReturn Then Exit For
  fv_d = flu_f_si(0, 1, t, dv)
  If fv_d = ErrorReturn Then Exit For
  fv_td = flu_f_si(1, 1, t, dv)
  If fv_td = ErrorReturn Then Exit For
  fv_dd = flu_f_si(0, 2, t, dv)
  If fv_dd = ErrorReturn Then Exit For

  'Derivatives of the Helmholtz function of liquid water for Newton iteration
  fl = flu_f_si(0, 0, t, dl)
  If fl = ErrorReturn Then Exit For
  fl_t = flu_f_si(1, 0, t, dl)
  If fl_t = ErrorReturn Then Exit For
  fl_d = flu_f_si(0, 1, t, dl)
  If fl_d = ErrorReturn Then Exit For
  fl_td = flu_f_si(1, 1, t, dl)
  If fl_td = ErrorReturn Then Exit For
  fl_dd = flu_f_si(0, 2, t, dl)
  If fl_dd = ErrorReturn Then Exit For

  'Derivatives of the Gibbs function of ice for Newton iteration
  gi = ice_g_si(0, 0, t, p)
  If gi = ErrorReturn Then Exit For
  gi_t = ice_g_si(1, 0, t, p)
  If gi_t = ErrorReturn Then Exit For
  gi_p = ice_g_si(0, 1, t, p)
  If gi_p = ErrorReturn Then Exit For

  'Derivatives of the saline Gibbs function for Newton iteration
  gs = sal_g_si(0, 0, 0, sa_si, t, p)
  If gs = ErrorReturn Then Exit For
  gs_s = sal_g_si(1, 0, 0, sa_si, t, p)
  If gs_s = ErrorReturn Then Exit For
  gs_t = sal_g_si(0, 1, 0, sa_si, t, p)
  If gs_t = ErrorReturn Then Exit For
  gs_st = sal_g_si(1, 1, 0, sa_si, t, p)
  If gs_st = ErrorReturn Then Exit For
  gs_p = sal_g_si(0, 0, 1, sa_si, t, p)
  If gs_p = ErrorReturn Then Exit For
  gs_sp = sal_g_si(1, 0, 1, sa_si, t, p)
  If gs_sp = ErrorReturn Then Exit For

  'coefficient matrix
  a(1, 1) = -fl_t - dl * fl_td + fv_t + dv * fv_td - gs_t + sa_si * gs_st
  a(1, 2) = -gs_p + sa_si * gs_sp
  a(1, 3) = 2 * fv_d + dv * fv_dd
  a(1, 4) = -2 * fl_d - dl * fl_dd
  
  a(2, 1) = fv_t + dv * fv_td - gi_t
  a(2, 2) = -gi_p
  a(2, 3) = 2 * fv_d + dv * fv_dd
  a(2, 4) = 0
  
  a(3, 1) = -dv * fv_td
  a(3, 2) = 1 / dv
  a(3, 3) = -2 * fv_d - dv * fv_dd
  a(3, 4) = 0
  
  a(4, 1) = -dl * fl_td
  a(4, 2) = 1 / dl
  a(4, 3) = 0
  a(4, 4) = -2 * fl_d - dl * fl_dd

  'right-hand sides, must vanish at equilibrium
  b(1) = fl + dl * fl_d - fv - dv * fv_d + gs - sa_si * gs_s
  b(2) = gi - fv - dv * fv_d
  b(3) = dv * fv_d - p / dv
  b(4) = dl * fl_d - p / dl

  'solve equations
  If matrix_solve(a(), b(), x(), 4) <> 0 Then Exit For 'matrix singular
  dt = x(1)
  dp = x(2)
  ddv = x(3)
  ddl = x(4)
  
  'update unknowns
  dv = dv + ddv
  If dv <= 0 Then Exit For
  dl = dl + ddl
  If dl <= 0 Then Exit For
  t = t + dt
  If t <= 0 Then Exit For
  p = p + dp
  If p <= 0 Then Exit For
  
  'check absolute or relative error limit
  If (eps > 0 And Abs(dt) < eps) Or _
     (eps < 0 And Abs(dt) < -eps * Abs(t)) Then
    sea_ice_vap_iteration_at_s = IsOK
    t_si = t
    p_si = p
    d_vap_si = dv
    d_liq_si = dl
    Exit For
  End If

Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Or _
     d_liq_si <= flu_dmin Or d_liq_si > flu_dmax Then
    t_si = ErrorReturn
    p_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_vap_si = ErrorReturn
    sea_ice_vap_iteration_at_s = ErrorReturn
    Exit Function
  End If
  'SAL_LIMITS
  If t_si < sal_tmin Or t_si > sal_tmax Or _
     sa_si < sal_smin Or sa_si > sal_smax Then
    t_si = ErrorReturn
    p_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_vap_si = ErrorReturn
    sea_ice_vap_iteration_at_s = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Function sea_ice_vap_iteration_at_t(ByVal t_si As Double, _
                                            ByVal maxit As Long, _
                                            ByVal eps As Double, _
                                            ByRef sa_si As Double, _
                                            ByRef p_si As Double, _
                                            ByRef d_vap_si As Double, _
                                            ByRef d_liq_si As Double) As Double

'this function computes the seawater-ice-vapour phase equilibrium from
'equal chemical potentials of water at given temperature, t_si,
'from initial guesses for the salinity, sa_si, pressure, p_si,
'the vapour density, d_vap_si and the liquid water density, d_liq_si.
'The iteration limit eps refers to the error in pressure.

'output:   sea_ice_vap_iteration_at_t = IsOK if successfully done
'          sea_ice_vap_iteration_at_t = ErrorReturn is returned if
'          - the maximum number of iterations is exceeded without meeting the exit criterion
'          - the function call to a Gibbs/Helmholtz function has returned an error
'          - density, salinity or pressure have taken a zero or negative value during the iteration
'          - s-t-p values do not permit the stable existence of ice
'   sa_si: absolute salinity in kg/kg at the seawater-ice equilibrium
'    p_si: absolute pressure in Pa at the seawater-ice equilibrium
'd_vap_si: density in kg/m3 of water vapour at (t_si, p_si)
'd_liq_si: density in kg/m3 of liquid water at (t_si, p_si)

'input:  t_si: absolute temperature in K
'       maxit: maximum number of iteration steps to be done
'         eps: required accuracy of pressure
'              eps > 0: absolute pressure tolerance in Pa
'              eps < 0: relative pressure tolerance
'       sa_si: initial guess for absolute salinity in kg/kg
'        p_si: initial guess for absolute pressure in Pa
'    d_vap_si: initial guess for density in kg/m3 of water vapour at (t_si, p_si)
'    d_liq_si: initial guess for density in kg/m3 of liquid water at (t_si, p_si)

Dim dv As Double, dl As Double, S As Double, p As Double
Dim ddv As Double, ddl As Double, ds As Double, dp As Double
Dim gi As Double, gi_p As Double
Dim gs As Double, gs_s As Double, gs_ss As Double
Dim gs_p As Double, gs_sp As Double
Dim fv As Double
Dim fv_d As Double, fv_dd As Double
Dim fl As Double
Dim fl_d As Double, fl_dd As Double

Dim it As Long

Dim a(4, 4) As Double, b(4) As Double, x(4) As Double

sea_ice_vap_iteration_at_t = ErrorReturn

If d_vap_si <= 0 Or _
   d_liq_si <= 0 Or _
   p_si <= 0 Or _
   sa_si <= 0 Or sa_si >= 1 Then
  sa_si = ErrorReturn
  p_si = ErrorReturn
  d_liq_si = ErrorReturn
  d_vap_si = ErrorReturn
  Exit Function
End If

If check_limits <> 1 Then
  If t_si <= 0 Then
    sa_si = ErrorReturn
    p_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_vap_si = ErrorReturn
    Exit Function
  End If
Else
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Then
    sa_si = ErrorReturn
    p_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_vap_si = ErrorReturn
    Exit Function
  End If
End If

check_limits = check_limits - 1

S = sa_si
p = p_si
dl = d_liq_si
dv = d_vap_si

For it = 1 To maxit

  'Derivatives of the Helmholtz function of water vapour for 4D Newton iteration
  fv = flu_f_si(0, 0, t_si, dv)
  If fv = ErrorReturn Then Exit For
  fv_d = flu_f_si(0, 1, t_si, dv)
  If fv_d = ErrorReturn Then Exit For
  fv_dd = flu_f_si(0, 2, t_si, dv)
  If fv_dd = ErrorReturn Then Exit For

  'Derivatives of the Helmholtz function of liquid water for Newton iteration
  fl = flu_f_si(0, 0, t_si, dl)
  If fl = ErrorReturn Then Exit For
  fl_d = flu_f_si(0, 1, t_si, dl)
  If fl_d = ErrorReturn Then Exit For
  fl_dd = flu_f_si(0, 2, t_si, dl)
  If fl_dd = ErrorReturn Then Exit For

  'Derivatives of the Gibbs function of ice for Newton iteration
  gi = ice_g_si(0, 0, t_si, p)
  If gi = ErrorReturn Then Exit For
  gi_p = ice_g_si(0, 1, t_si, p)
  If gi_p = ErrorReturn Then Exit For

  'Derivatives of the saline Gibbs function for Newton iteration
  gs = sal_g_si(0, 0, 0, S, t_si, p)
  If gs = ErrorReturn Then Exit For
  gs_s = sal_g_si(1, 0, 0, S, t_si, p)
  If gs_s = ErrorReturn Then Exit For
  gs_ss = sal_g_si(2, 0, 0, S, t_si, p)
  If gs_ss = ErrorReturn Then Exit For
  gs_p = sal_g_si(0, 0, 1, S, t_si, p)
  If gs_p = ErrorReturn Then Exit For
  gs_sp = sal_g_si(1, 0, 1, S, t_si, p)
  If gs_sp = ErrorReturn Then Exit For

  'coefficient matrix
  a(1, 1) = S * gs_ss
  a(1, 2) = -gs_p + S * gs_sp
  a(1, 3) = 2 * fv_d + dv * fv_dd
  a(1, 4) = -2 * fl_d - dl * fl_dd
  
  a(2, 1) = 0
  a(2, 2) = -gi_p
  a(2, 3) = 2 * fv_d + dv * fv_dd
  a(2, 4) = 0
  
  a(3, 1) = 0
  a(3, 2) = 1 / dv
  a(3, 3) = -2 * fv_d - dv * fv_dd
  a(3, 4) = 0
  
  a(4, 1) = 0
  a(4, 2) = 1 / dl
  a(4, 3) = 0
  a(4, 4) = -2 * fl_d - dl * fl_dd

  'right-hand sides, must vanish at equilibrium
  b(1) = fl + dl * fl_d - fv - dv * fv_d + gs - S * gs_s
  b(2) = gi - fv - dv * fv_d
  b(3) = dv * fv_d - p / dv
  b(4) = dl * fl_d - p / dl

  'solve equations
  If matrix_solve(a(), b(), x(), 4) <> 0 Then Exit For 'matrix singular
  ds = x(1)
  dp = x(2)
  ddv = x(3)
  ddl = x(4)
  
  'update unknowns
  dv = dv + ddv
  If dv <= 0 Then Exit For
  dl = dl + ddl
  If dl <= 0 Then Exit For
  S = S + ds
  If S <= 0 Then Exit For
  If S >= 1 Then Exit For
  p = p + dp
  If p <= 0 Then Exit For
  
  'check absolute or relative error limit
  If (eps > 0 And Abs(dp) < eps) Or _
     (eps < 0 And Abs(dp) < -eps * Abs(p)) Then
    sea_ice_vap_iteration_at_t = IsOK
    sa_si = S
    p_si = p
    d_vap_si = dv
    d_liq_si = dl
    Exit For
  End If

Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'FLU_LIMITS
  If d_liq_si <= flu_dmin Or d_liq_si > flu_dmax Then
    sa_si = ErrorReturn
    p_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_vap_si = ErrorReturn
    sea_ice_vap_iteration_at_t = ErrorReturn
    Exit Function
  End If
  'SAL_LIMITS
  If sa_si < sal_smin Or sa_si > sal_smax Or _
     p_si < sal_pmin Or p_si > sal_pmax Then
    sa_si = ErrorReturn
    p_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_vap_si = ErrorReturn
    sea_ice_vap_iteration_at_t = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Function sea_ice_vap_iteration_at_p(ByVal p_si As Double, _
                                            ByVal maxit As Long, _
                                            ByVal eps As Double, _
                                            ByRef sa_si As Double, _
                                            ByRef t_si As Double, _
                                            ByRef d_vap_si As Double, _
                                            ByRef d_liq_si As Double) As Double

'this function computes the seawater-ice-vapour phase equilibrium from
'equal chemical potentials of water at given pressure, p_si,
'from initial guesses for the salinity, sa_si, temperature, t_si,
'the vapour density, d_vap_si and the liquid density, d_liq_si.
'The iteration limit eps refers to the error in temperature.

'output:    sea_ice_vap_iteration_at_p = IsOK if successfully done
'           sea_ice_vap_iteration_at_p = ErrorReturn is returned if
'           - the maximum number of iterations is exceeded without meeting the exit criterion
'           - the function call to a Gibbs/Helmholtz function has returned an error
'           - density, salnity or temperature have taken a zero or negative value during the iteration
'           - s-t-p values do not permit the stable existence of ice
'    sa_si: absolute salinity in kg/kg at the seawater-ice equilibrium
'     t_si: absolute temperature in K at the seawater-ice equilibrium
' d_vap_si: density in kg/m3 of water vapour at (t_si, p_si)
' d_liq_si: density in kg/m3 of liquid water at (t_si, p_si)

'input:  p_si: absolute pressure in Pa
'       maxit: maximum number of iteration steps to be done
'         eps: required accuracy of temperature
'              eps > 0: absolute temperature tolerance in K
'              eps < 0: relative temperature tolerance
'       sa_si: initial guess for absolute salinity in kg/kg
'        t_si: initial guess for absolute temperature in K
'    d_vap_si: initial guess for density in kg/m3 of water vapour at (t_si, p_si)
'    d_liq_si: initial guess for density in kg/m3 of liquid water at (t_si, p_si)

Dim dv As Double, dl As Double, S As Double, t As Double
Dim ddv As Double, ddl As Double, ds As Double, dt As Double
Dim gi As Double, gi_t As Double
Dim gs As Double, gs_s As Double, gs_ss As Double
Dim gs_t As Double, gs_st As Double
Dim fv As Double, fv_t As Double, fv_td As Double
Dim fv_d As Double, fv_dd As Double
Dim fl As Double, fl_t As Double, fl_td As Double
Dim fl_d As Double, fl_dd As Double

Dim it As Long

Dim a(4, 4) As Double, b(4) As Double, x(4) As Double

sea_ice_vap_iteration_at_p = ErrorReturn

If d_vap_si <= 0 Or _
   d_liq_si <= 0 Or _
   t_si <= 0 Or _
   sa_si <= 0 Or sa_si >= 1 Then
  sa_si = ErrorReturn
  t_si = ErrorReturn
  d_liq_si = ErrorReturn
  d_vap_si = ErrorReturn
  Exit Function
End If

If check_limits <> 1 Then
  If p_si <= 0 Then
    sa_si = ErrorReturn
    t_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_vap_si = ErrorReturn
    Exit Function
  End If
Else
  'SAL_LIMITS
  If p_si < sal_pmin Or p_si > sal_pmax Then
    sa_si = ErrorReturn
    t_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_vap_si = ErrorReturn
    Exit Function
  End If
End If

check_limits = check_limits - 1

S = sa_si
t = t_si
dl = d_liq_si
dv = d_vap_si

For it = 1 To maxit

  'Derivatives of the Helmholtz function of water vapour for 4D Newton iteration
  fv = flu_f_si(0, 0, t, dv)
  If fv = ErrorReturn Then Exit For
  fv_t = flu_f_si(1, 0, t, dv)
  If fv_t = ErrorReturn Then Exit For
  fv_d = flu_f_si(0, 1, t, dv)
  If fv_d = ErrorReturn Then Exit For
  fv_td = flu_f_si(1, 1, t, dv)
  If fv_td = ErrorReturn Then Exit For
  fv_dd = flu_f_si(0, 2, t, dv)
  If fv_dd = ErrorReturn Then Exit For

  'Derivatives of the Helmholtz function of liquid water for Newton iteration
  fl = flu_f_si(0, 0, t, dl)
  If fl = ErrorReturn Then Exit For
  fl_t = flu_f_si(1, 0, t, dl)
  If fl_t = ErrorReturn Then Exit For
  fl_d = flu_f_si(0, 1, t, dl)
  If fl_d = ErrorReturn Then Exit For
  fl_td = flu_f_si(1, 1, t, dl)
  If fl_td = ErrorReturn Then Exit For
  fl_dd = flu_f_si(0, 2, t, dl)
  If fl_dd = ErrorReturn Then Exit For

  'Derivatives of the Gibbs function of ice for Newton iteration
  gi = ice_g_si(0, 0, t, p_si)
  If gi = ErrorReturn Then Exit For
  gi_t = ice_g_si(1, 0, t, p_si)
  If gi_t = ErrorReturn Then Exit For

  'Derivatives of the saline Gibbs function for Newton iteration
  gs = sal_g_si(0, 0, 0, S, t, p_si)
  If gs = ErrorReturn Then Exit For
  gs_s = sal_g_si(1, 0, 0, S, t, p_si)
  If gs_s = ErrorReturn Then Exit For
  gs_ss = sal_g_si(2, 0, 0, S, t, p_si)
  If gs_ss = ErrorReturn Then Exit For
  gs_t = sal_g_si(0, 1, 0, S, t, p_si)
  If gs_t = ErrorReturn Then Exit For
  gs_st = sal_g_si(1, 1, 0, S, t, p_si)
  If gs_st = ErrorReturn Then Exit For

  'coefficient matrix
  a(1, 1) = S * gs_ss
  a(1, 2) = -fl_t - dl * fl_td + fv_t + dv * fv_td - gs_t + sa_si * gs_st
  a(1, 3) = 2 * fv_d + dv * fv_dd
  a(1, 4) = -2 * fl_d - dl * fl_dd
  
  a(2, 1) = 0
  a(2, 2) = fv_t + dv * fv_td - gi_t
  a(2, 3) = 2 * fv_d + dv * fv_dd
  a(2, 4) = 0
  
  a(3, 1) = 0
  a(3, 2) = -dv * fv_td
  a(3, 3) = -2 * fv_d - dv * fv_dd
  a(3, 4) = 0
  
  a(4, 1) = 0
  a(4, 2) = -dl * fl_td
  a(4, 3) = 0
  a(4, 4) = -2 * fl_d - dl * fl_dd

  'right-hand sides, must vanish at equilibrium
  b(1) = fl + dl * fl_d - fv - dv * fv_d + gs - S * gs_s
  b(2) = gi - fv - dv * fv_d
  b(3) = dv * fv_d - p_si / dv
  b(4) = dl * fl_d - p_si / dl

  'solve equations
  If matrix_solve(a(), b(), x(), 4) <> 0 Then Exit For 'matrix singular
  ds = x(1)
  dt = x(2)
  ddv = x(3)
  ddl = x(4)
  
  'update unknowns
  dv = dv + ddv
  If dv <= 0 Then Exit For
  dl = dl + ddl
  If dl <= 0 Then Exit For
  S = S + ds
  If S <= 0 Then Exit For
  If S >= 1 Then Exit For
  t = t + dt
  If t <= 0 Then Exit For
  
  'check absolute or relative error limit
  If (eps > 0 And Abs(dt) < eps) Or _
     (eps < 0 And Abs(dt) < -eps * Abs(t)) Then
    sea_ice_vap_iteration_at_p = IsOK
    sa_si = S
    t_si = t
    d_vap_si = dv
    d_liq_si = dl
    Exit For
  End If

Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Or _
     d_liq_si <= flu_dmin Or d_liq_si > flu_dmax Then
    sa_si = ErrorReturn
    t_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_vap_si = ErrorReturn
    sea_ice_vap_iteration_at_p = ErrorReturn
    Exit Function
  End If
  'SAL_LIMITS
  If t_si < sal_tmin Or t_si > sal_tmax Or _
     sa_si < sal_smin Or sa_si > sal_smax Then
    sa_si = ErrorReturn
    t_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_vap_si = ErrorReturn
    sea_ice_vap_iteration_at_p = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Sub clear_sea_ice_vap_state()

'clears the current equilibrium state descriptor

equi_sea_ice_vap_done = 0

End Sub

'==========================================================================
Private Sub set_sea_ice_vap_state(ByVal sa_si As Double, _
                                  ByVal t_si As Double, _
                                  ByVal p_si As Double, _
                                  ByVal d_vap_si As Double, _
                                  ByVal d_liq_si As Double)
                                  
'stores the actual properties as the current equilibrium state descriptor

equi_sea_ice_vap_done = IsOK

equi_sea_ice_vap_s = sa_si         'triple-point salinity
equi_sea_ice_vap_t = t_si          'triple-point temperature
equi_sea_ice_vap_p = p_si          'triple-point pressure
equi_sea_ice_vap_d_liq = d_liq_si  'density of liquid pure water
equi_sea_ice_vap_d_vap = d_vap_si  'density of water vapour

End Sub

'==========================================================================
Private Function aux_temperature_si(ByVal sa_si As Double) As Double

'function returns the estimated freezing temperature in K at given salinity in kg/kg
Const Tt = TP_temperature_si

aux_temperature_si = Tt - 60 * sa_si

End Function

'==========================================================================
Private Function aux_salinity_of_t_si(ByVal t_si As Double) As Double

'function returns the estimated salinity in kg/kg at given temperature in K
Const Tt = TP_temperature_si

aux_salinity_of_t_si = (Tt - t_si) / 60

End Function

'==========================================================================
Private Function aux_pressure_si(ByVal sa_si As Double) As Double

'function returns the estimated freezing pressure in Pa at given salinity in kg/kg
Const Pt = TP_pressure_IAPWS95_si

aux_pressure_si = Pt - 3000 * sa_si

End Function

'==========================================================================
Private Function aux_salinity_of_p_si(ByVal p_si As Double) As Double

'function returns the estimated freezing salinity in kg/kg at given pressure in Pa
Const Pt = TP_pressure_IAPWS95_si

aux_salinity_of_p_si = (Pt - p_si) / 3000

End Function

'==========================================================================
Private Function aux_density_ideal_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'function returns the ideal-gas density in kg m-3 at given
'absolute temperature in K and absolute pressure in Pa

Const Pt = TP_pressure_IAPWS95_si
Const RW = Gas_constant_H2O_si

aux_density_ideal_si = ErrorReturn
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

aux_density_ideal_si = p_si / (RW * t_si)

End Function
