Attribute VB_Name = "Sea_3a_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules:
'     Constants_0_Mdl, file Constants_0.bas
'     Maths_0_Mdl,     file Maths_0.bas
'     Sal_1_Mdl,       file Sal_1.bas
'     Sal_2_Mdl,       file Sal_2.bas
'     Flu_3a_Mdl,      file Flu_3a.bas

'#########################################################################

'=========================================================================
'This module implements the Gibbs function of seawater depending on temperature
'and pressure, as well as their partial derivatives,
'as defined in IAPWS-08:

'Release on the IAPWS Formulation 2008 for the Thermodynamic Properties of Seawater
'The International Association for the Properties of Water and Steam
'Berlin, Germany, September 2008

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98

Private Const Version = "20 Nov 2009"

'==========================================================================
Public Function sea_g_si(ByVal drv_s As Integer, _
                         ByVal drv_t As Integer, _
                         ByVal drv_p As Integer, _
                         ByVal sa_si As Double, _
                         ByVal t_si As Double, _
                         ByVal p_si As Double) As Double

'this function implements the Gibbs function of seawater, IAPWS-08

'returns sea_g_si as the S-T-P derivative

'(d/dS)^drv_s (d/dT)^drv_t (d/dP)^drv_p g(S,T,P)

'of the specific Gibbs energy of seawater, g(S,T,P), in J/kg

'sa_si     absolute salinity in kg/kg
't_si      absolute temperature in K
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check values with default settings:
'sea_g_si( 0, 0, 0, 0.035, 300, 1E5) =-5114.9961985|6615
'sea_g_si( 1, 0, 0, 0.035, 300, 1E5) = 78593.7757371468
'sea_g_si( 0, 1, 0, 0.035, 300, 1E5) =-374.452240540063
'sea_g_si( 0, 0, 1, 0.035, 300, 1E5) = 9.7785861518227E-04
'sea_g_si( 2, 0, 0, 0.035, 300, 1E5) = 2247550.41118143
'sea_g_si( 1, 1, 0, 0.035, 300, 1E5) = 789.934255688123
'sea_g_si( 1, 0, 1, 0.035, 300, 1E5) =-7.16682401264827E-04
'sea_g_si( 0, 2, 0, 0.035, 300, 1E5) =-13.3358324654699
'sea_g_si( 0, 1, 1, 0.035, 300, 1E5) = 3.04605539768293E-07
'sea_g_si( 0, 0, 2, 0.035, 300, 1E5) =-4.10945807960411E-13

Dim gw As Double, gs As Double

sea_g_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

If drv_s < 0 Or drv_s > 3 Then Exit Function
If drv_t < 0 Then Exit Function
If drv_p < 0 Then Exit Function

'water part
gw = 0
If drv_s = 0 Then
  gw = liq_g_si(drv_t, drv_p, t_si, p_si)
  If gw = ErrorReturn Then Exit Function
End If

'saline part
gs = sal_g_si(drv_s, drv_t, drv_p, sa_si, t_si, p_si)
If gs = ErrorReturn Then Exit Function

sea_g_si = gw + gs

End Function

'=========================================================================
Public Function sea_chempot_h2o_si(ByVal sa_si As Double, _
                                   ByVal t_si As Double, _
                                   ByVal p_si As Double) As Double

'returns   _W(S,T,P) = g - S*(dg/dS)_T_P  chem. potential of water in seawater in J/kg,
'sa_si     absolute salinity in kg/kg
't_si      absolute temperature in K
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: sea_chempot_h2o_si(0.035, 300, 1E5) = -7865.7783493|6628

Dim g As Double, mu_s As Double

sea_chempot_h2o_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g = liq_g_si(0, 0, t_si, p_si)
If g = ErrorReturn Then Exit Function

mu_s = sal_chempot_h2o_si(sa_si, t_si, p_si)
If mu_s = ErrorReturn Then Exit Function

sea_chempot_h2o_si = g + mu_s

End Function

'=========================================================================
Public Function sea_chempot_rel_si(ByVal sa_si As Double, _
                                   ByVal t_si As Double, _
                                   ByVal p_si As Double) As Double

'returns   (S,T,P) = (dg/dS)_T_P  relative chem. potential of seawater in J/kg,
'sa_si     absolute salinity in kg/kg
't_si      absolute temperature in K
'p_si      absolute pressure in Pa

'Check value with default settings:
'sea_chempot_rel_si(0.035, 300, 1E5) = 78593.7757371468

Dim g As Double, mu_s As Double

sea_chempot_rel_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

sea_chempot_rel_si = sal_chempot_rel_si(sa_si, t_si, p_si)

End Function

'=========================================================================
Public Function sea_g_contraction_t_si(ByVal sa_si As Double, _
                                       ByVal t_si As Double, _
                                       ByVal p_si As Double) As Double

'returns   beta(S,T,P) = (1/D)*(dD/dS)_T_P isothermal haline contraction of seawater in kg/kg,
'sa_si     absolute salinity in kg/kg
't_si      absolute temperature in K
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: sea_g_contraction_t_si(0.035, 300, 1E5) = 0.732910044599075


Dim gp As Double, gsp As Double

sea_g_contraction_t_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

gp = sea_g_si(0, 0, 1, sa_si, t_si, p_si)
If gp = ErrorReturn Then Exit Function
If gp <= 0 Then Exit Function

gsp = sea_g_si(1, 0, 1, sa_si, t_si, p_si)
If gsp = ErrorReturn Then Exit Function

sea_g_contraction_t_si = -gsp / gp

End Function

'=========================================================================
Public Function sea_cp_si(ByVal sa_si As Double, _
                          ByVal t_si As Double, _
                          ByVal p_si As Double) As Double

'returns   cp(S,T,P) = T * (deta/dT)_P  isobaric heat capacity of seawater in J/(kg K),
'sa_si     absolute salinity in kg/kg
't_si      absolute temperature in K
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: sea_cp_si(0.035, 300, 1E5) = 4000.74973964098

Dim gtt As Double

sea_cp_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

gtt = sea_g_si(0, 2, 0, sa_si, t_si, p_si)
If gtt = ErrorReturn Then Exit Function

sea_cp_si = -t_si * gtt

End Function

'=========================================================================
Public Function sea_density_si(ByVal sa_si As Double, _
                               ByVal t_si As Double, _
                               ByVal p_si As Double) As Double

'returns   d(S,T,P) = 1/(dg/dP)_S_T  density of seawater in kg/m^3
'sa_si     absolute salinity in kg/kg
't_si      absolute temperature in K
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: sea_density_si(0.035, 300, 1E5) = 1022.6427261303

Dim gp As Double

sea_density_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

gp = sea_g_si(0, 0, 1, sa_si, t_si, p_si)
If gp = ErrorReturn Then Exit Function
If gp <= 0 Then Exit Function

sea_density_si = 1# / gp

End Function

'=========================================================================
Public Function sea_enthalpy_si(ByVal sa_si As Double, _
                                ByVal t_si As Double, _
                                ByVal p_si As Double) As Double

'returns   h(S,T,P) = g - T*(dg/dT)_S_P enthalpy of seawater in J/kg,
'sa_si     absolute salinity in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: sea_enthalpy_si(0.035, 300, 1E5) = 107220.675963453

Dim g As Double, gt As Double

sea_enthalpy_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g = sea_g_si(0, 0, 0, sa_si, t_si, p_si)
If g = ErrorReturn Then Exit Function

gt = sea_g_si(0, 1, 0, sa_si, t_si, p_si)
If gt = ErrorReturn Then Exit Function

sea_enthalpy_si = g - t_si * gt

End Function

'==========================================================================
Public Function sea_entropy_si(ByVal sa_si As Double, _
                               ByVal t_si As Double, _
                               ByVal p_si As Double) As Double
                                 
'returns   eta(S,T,P) = -(dg/dT)_S_P  specific entropy of seawater in J/(kg K),
'sa_si     absolute salinity in kg/kg
't_si      absolute temperature in K
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: sea_entropy_si(0.035, 300, 1E5) = 374.452240540063

Dim gt As Double

sea_entropy_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

gt = sea_g_si(0, 1, 0, sa_si, t_si, p_si)
If gt = ErrorReturn Then Exit Function

sea_entropy_si = -gt

End Function

'=========================================================================
Public Function sea_g_expansion_t_si(ByVal sa_si As Double, _
                                     ByVal t_si As Double, _
                                     ByVal p_si As Double) As Double

'returns   alpha(S,T,P) = -(1/D) * (dD/dT)_S_P  thermal expansion of seawater in 1/K,
'sa_si     absolute salinity in kg/kg
't_si      absolute temperature in K
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: sea_g_expansion_t_si(0.035, 300, 1E5) = 3.11502639583039E-04

Dim gtp As Double, gp As Double

sea_g_expansion_t_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

gtp = sea_g_si(0, 1, 1, sa_si, t_si, p_si)
If gtp = ErrorReturn Then Exit Function

gp = sea_g_si(0, 0, 1, sa_si, t_si, p_si)
If gp = ErrorReturn Then Exit Function
If gp <= 0 Then Exit Function

sea_g_expansion_t_si = gtp / gp

End Function

'=========================================================================
Public Function sea_gibbs_energy_si(ByVal sa_si As Double, _
                                    ByVal t_si As Double, _
                                    ByVal p_si As Double) As Double

'returns   g(S,T,P) = g   Gibbs energy of seawater in J/kg,
'sa_si     absolute salinity in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: sea_gibbs_energy_si(0.035, 300, 1E5) = -5114.9961985|6615

sea_gibbs_energy_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

sea_gibbs_energy_si = sea_g_si(0, 0, 0, sa_si, t_si, p_si)

End Function

'=========================================================================
Public Function sea_internal_energy_si(ByVal sa_si As Double, _
                                       ByVal t_si As Double, _
                                       ByVal p_si As Double) As Double

'returns   u(S,T,P) = g - T*(dg/dT)_S_P - P*(dg/dP)_S_T internal energy of seawater in J/kg,
'sa_si     absolute salinity in kg/kg
't_si      absolute temperature in K
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: sea_internal_energy_si(0.035, 300, 1E5) = 107122.890101934

Dim g As Double, gt As Double, gp As Double

sea_internal_energy_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g = sea_g_si(0, 0, 0, sa_si, t_si, p_si)
If g = ErrorReturn Then Exit Function

gt = sea_g_si(0, 1, 0, sa_si, t_si, p_si)
If gt = ErrorReturn Then Exit Function

gp = sea_g_si(0, 0, 1, sa_si, t_si, p_si)
If gp = ErrorReturn Then Exit Function
If gp <= 0 Then Exit Function

sea_internal_energy_si = g - t_si * gt - p_si * gp

End Function

'=========================================================================
Public Function sea_kappa_s_si(ByVal sa_si As Double, _
                               ByVal t_si As Double, _
                               ByVal p_si As Double) As Double

'returns   kappa_s(S,T,P) = (1/D) * (dD/dP)_S_eta  isentropic compressibility of seawater in 1/Pa,
'sa_si     absolute salinity in kg/kg
't_si      absolute temperature in K
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: sea_kappa_s_si(0.035, 300, 1E5) = 4.13135667732431E-10

Dim gp As Double
Dim gtp As Double, gtt As Double, gpp As Double

sea_kappa_s_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

gp = sea_g_si(0, 0, 1, sa_si, t_si, p_si)
If gp = ErrorReturn Then Exit Function
If gp <= 0 Then Exit Function

gtp = sea_g_si(0, 1, 1, sa_si, t_si, p_si)
If gtp = ErrorReturn Then Exit Function

gtt = sea_g_si(0, 2, 0, sa_si, t_si, p_si)
If gtt = ErrorReturn Then Exit Function
If gtt = 0 Then Exit Function

gpp = sea_g_si(0, 0, 2, sa_si, t_si, p_si)
If gpp = ErrorReturn Then Exit Function

sea_kappa_s_si = (gtp * gtp - gtt * gpp) / (gp * gtt)

End Function

'=========================================================================
Public Function sea_kappa_t_si(ByVal sa_si As Double, _
                               ByVal t_si As Double, _
                               ByVal p_si As Double) As Double

'returns   kappa_t(S,T,P) = (1/D) * (dD/dP)_S_T  isothermal compressibility of seawater in 1/K,
'sa_si     absolute salinity in kg/kg
't_si      absolute temperature in K
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: sea_kappa_t_si(0.035, 300, 1E5) = 4.20250741344455E-10

Dim gp As Double, gpp As Double

sea_kappa_t_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

gp = sea_g_si(0, 0, 1, sa_si, t_si, p_si)
If gp = ErrorReturn Then Exit Function
If gp <= 0 Then Exit Function

gpp = sea_g_si(0, 0, 2, sa_si, t_si, p_si)
If gpp = ErrorReturn Then Exit Function

sea_kappa_t_si = -gpp / gp

End Function

'=========================================================================
Public Function sea_lapserate_si(ByVal sa_si As Double, _
                                 ByVal t_si As Double, _
                                 ByVal p_si As Double) As Double
                                 
'returns   gamma(S,T,P) = (dT/dP)_S_eta  adiabatic lapse rate of seawater in K/Pa,
'sa_si     absolute salinity in kg/kg
't_si      absolute temperature in K
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: sea_lapserate_si(0.035, 300, 1E5) = 2.28411342566727E-08

Dim gtp As Double, gtt As Double

sea_lapserate_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

gtp = sea_g_si(0, 1, 1, sa_si, t_si, p_si)
If gtp = ErrorReturn Then Exit Function

gtt = sea_g_si(0, 2, 0, sa_si, t_si, p_si)
If gtt = ErrorReturn Then Exit Function
If gtt = 0 Then Exit Function

sea_lapserate_si = -gtp / gtt

End Function

'=========================================================================
Public Function sea_osm_coeff_si(ByVal sa_si As Double, _
                                 ByVal t_si As Double, _
                                 ByVal p_si As Double) As Double

'returns the osmotic coefficient of seawater, phi

'sa_si:   absolute salinity in kg/kg
't_si:    absolute temperature in K
'p_si:    absolute pressure in Pa

'Check value with default settings: sea_osm_coeff_si(0.035, 300, 1E5) = 0.902777495349254

sea_osm_coeff_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

sea_osm_coeff_si = sal_osm_coeff_si(sa_si, t_si, p_si)

End Function

'=========================================================================
Public Function sea_soundspeed_si(ByVal sa_si As Double, _
                                  ByVal t_si As Double, _
                                  ByVal p_si As Double) As Double
                                 
'returns   c(S,T,P) = sqr[ (dP/dD)_S_eta ] sound speed of seawater in m/s,
'sa_si     absolute salinity in kg/kg
't_si      absolute temperature in K
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: sea_soundspeed_si(0.035, 300, 1E5) = 1538.47940766397

Dim d As Double, gp As Double, c As Double
Dim gtp As Double, gtt As Double, gpp As Double
Dim ctrl0 As Double, ctrl1 As Double

sea_soundspeed_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

gp = sea_g_si(0, 0, 1, sa_si, t_si, p_si)
If gp = ErrorReturn Then Exit Function
If gp <= 0 Then Exit Function

gtp = sea_g_si(0, 1, 1, sa_si, t_si, p_si)
If gtp = ErrorReturn Then Exit Function

gtt = sea_g_si(0, 2, 0, sa_si, t_si, p_si)
If gtt = ErrorReturn Then Exit Function
If gtt = 0 Then Exit Function

gpp = sea_g_si(0, 0, 2, sa_si, t_si, p_si)
If gpp = ErrorReturn Then Exit Function

c = gtp * gtp - gtt * gpp
If c = 0 Then Exit Function
c = gtt / c
If c < 0 Then Exit Function

sea_soundspeed_si = gp * Sqr(c)

End Function

'=========================================================================
Public Function sea_temp_maxdensity_si(ByVal sa_si As Double, _
                                       ByVal p_si As Double) As Double

'returns   t_md(S,P) temperature of maximum density in K, by Newton iteration of d2g/dtdp = 0
'input:
'sa_si     absolute salinity in kg/kg
'p_si      absolute pressure in Pa

'Check value with default settings:
'sea_temp_maxdensity_si(0.01, 1E5) = 274.950121503097

Const h = 0.000001   'finite difference in t to estimate d3g/dt2dp
Const T0 = 300#      'initial temperature value
Const eps = 0.000001 'temperature tolerance
Const maxit = 20&    'max iteration steps

Dim g_tp As Double, g_ttp As Double, t As Double, dt As Double
Dim iter As Long

sea_temp_maxdensity_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function

t = T0
For iter = 1 To maxit

  g_tp = sea_g_si(0, 1, 1, sa_si, t, p_si)
  If g_tp = ErrorReturn Then Exit Function
  
  g_ttp = sea_g_si(0, 1, 1, sa_si, t + h, p_si)
  If g_ttp = ErrorReturn Then Exit Function
  g_ttp = (g_ttp - g_tp) / h                    'finite difference approximation for 3rd derivative
  If g_ttp = 0 Then Exit Function
  
  'improve estimate
  dt = -g_tp / g_ttp
  t = t + dt
  If t <= 0 Then Exit Function
  
  If Abs(dt) < eps Then
    sea_temp_maxdensity_si = t
    Exit Function
  End If

Next iter

End Function

'=========================================================================
Public Function chk_IAPWS08_Table8a() As String

'TABLE 8  Numerical check values for the water part computed from
'gW and its derivatives, Table 4, for the saline part computed from
'gS and its derivatives, Eq. (4), and for the seawater property
'computed from the Gibbs function g = gW + gS and its derivatives, Eq. (3).
'The numerical functions evaluated here at given points (S, T, p) are defined in Tables 3-6.

'a)  Properties at S = Sn = 0.035 165 04 kg kg-1,  T = T0 = 273.15 K, p = p0 = 101325 Pa

'Quantity         Water part       Saline part      Seawater        Unit
'g                0.101342742E3   -0.101342742E3    0.0             J kg-1
'(dg/dS)_T,p      0.0              0.639974067E5    0.639974067E5   J kg-1
'(dg/dT)_S,p      0.147643376     -0.147643376      0.0             J kg-1 K-1
'(dg/dp)-S,T      0.100015694E-2  -0.274957224E-4   0.972661217E-3  m3 kg-1
'(d2g/dSdp)-T     0.0             -0.759615412E-3  -0.759615412E-3  m3 kg-1
'(d2g/dT 2)_S,p  -0.154473542E2    0.852861151     -0.145944931E2   J kg-1 K-2
'(d2g/dTdp)_S    -0.677700318E-7   0.119286787E-6   0.515167556E-7  m3 kg-1 K-1
'(d2g/dp2)_S,T   -0.508928895E-12  0.581535172E-13 -0.450775377E-12 m3 kg-1 Pa-1
'h                0.610139535E2   -0.610139535E2    0.0             J kg-1
'f                0.18399E-2      -0.985567377E2   -0.985548978E2   J kg-1
'u               -0.403269484E2   -0.582279494E2   -0.985548978E2   J kg-1
's               -0.147643376      0.147643376      0.0             J kg-1 K-1
'rho              0.99984386E3     -                0.102810720E4   kg m - 3
'cp               0.421944481E4   -0.232959023E3    0.398648579E4   J kg-1 K-1
'w                0.140238253E4    -                0.144900246E4   m s-1
'W               0.101342742E3   -0.235181411E4   -0.225047137E4   J kg-1


Dim CRLF As String
Dim txt As String, row As String
Dim S As Double, t As Double, p As Double, q(3) As Double
Dim i As Integer, j As Integer

CRLF = Chr(13) + Chr(10)

txt = " Implementation of IAPWS-08 in Visual Basic" + CRLF
txt = txt + " for Publication in Ocean Science, 2009" + CRLF
txt = txt + " R. Feistel, IOW, Version " + Version + CRLF
txt = txt + " Compiled on " + CStr(Now) + CRLF + CRLF

txt = txt + " Function values as given in Table 8a of IAPWS-08:" + CRLF
txt = txt + " Properties at S = 0.03516504 kg kg-1,  T = 273.15 K, P = 101325 Pa" + CRLF + CRLF

txt = txt + "Quantity         Water part        Saline part       Seawater        Unit" + CRLF

t = 273.15
p = 101325

For i = 1 To 16
  Select Case i
    Case 1:  row = "g                0.101342742E+3   -0.101342742E+3    0.0             J kg-1"
    Case 2:  row = "(dg/dS)_T,P      0.0               0.639974067E+5    0.639974067E+5  J kg-1"
    Case 3:  row = "(dg/dT)_S,P      0.147643376      -0.147643376       0.0             J kg-1 K-1"
    Case 4:  row = "(dg/dP)_S,T      0.100015694E-2   -0.274957224E-4    0.972661217E-3  m3 kg-1"
    Case 5:  row = "(d2g/dSdP)_T     0.0              -0.759615412E-3   -0.759615412E-3  m3 kg-1"
    Case 6:  row = "(d2g/dT2)_S,P   -0.154473542E+2    0.852861151      -0.145944931E+2  J kg-1 K-2"
    Case 7:  row = "(d2g/dTdP)_S    -0.677700318E-7    0.119286787E-6    0.515167556E-7  m3 kg-1 K-1"
    Case 8:  row = "(d2g/dP2)_S,T   -0.508928895E-12   0.581535172E-13  -0.450775377E-12 m3 kg-1 Pa-1"
    Case 9:  row = "h                0.610139535E+2   -0.610139535E+2    0.0             J kg-1"
    Case 10: row = "f                0.18399xxxxE-2   -0.985567377E+2   -0.985548978E+2  J kg-1"
    Case 11: row = "u               -0.403269484E+2   -0.582279494E+2   -0.985548978E+2  J kg-1"
    Case 12: row = "s               -0.147643376       0.147643376       0.0             J kg-1 K-1"
    Case 13: row = "rho              0.999843086E+3    -                 0.102810720E+4  kg m - 3"
    Case 14: row = "cp               0.421944481E+4   -0.232959023E+3    0.398648579E+4  J kg-1 K-1"
    Case 15: row = "w                0.140238253E+4    -                 0.144900246E+4  m s-1"
    Case 16: row = "W               0.101342742E+3   -0.235181411E+4   -0.225047137E+4  J kg-1"
  End Select

  txt = txt + row + CRLF
  txt = txt + "this code:      "
  
  For j = 1 To 3 Step 2
    S = IIf(j = 3, 0.03516504, 0)
    q(j) = 0
    Select Case i
      Case 1: q(j) = sea_g_si(0, 0, 0, S, t, p)
      Case 2: If S > 0 Then q(j) = sea_g_si(1, 0, 0, S, t, p) 'no s-derivative of pure water
      Case 3: q(j) = sea_g_si(0, 1, 0, S, t, p)
      Case 4: q(j) = sea_g_si(0, 0, 1, S, t, p)
      Case 5: If S > 0 Then q(j) = sea_g_si(1, 0, 1, S, t, p) 'no s-derivative of pure water
      Case 6: q(j) = sea_g_si(0, 2, 0, S, t, p)
      Case 7: q(j) = sea_g_si(0, 1, 1, S, t, p)
      Case 8: q(j) = sea_g_si(0, 0, 2, S, t, p)

      Case 9: q(j) = sea_enthalpy_si(S, t, p)
      Case 10: q(j) = sea_g_si(0, 0, 0, S, t, p) - p * sea_g_si(0, 0, 1, S, t, p)
      Case 11: q(j) = sea_g_si(0, 0, 0, S, t, p) - t * sea_g_si(0, 1, 0, S, t, p) - p * sea_g_si(0, 0, 1, S, t, p)
      Case 12: q(j) = sea_entropy_si(S, t, p)
      Case 13: q(j) = sea_density_si(S, t, p)
      Case 14: q(j) = sea_cp_si(S, t, p)
      Case 15: q(j) = sea_soundspeed_si(S, t, p)
      Case 16: q(j) = sea_chempot_h2o_si(S, t, p)
    End Select

  Next j
  txt = txt + Left(EFormat(q(1), 9) + Space(18), 18)
  Select Case i
    Case 13, 15: txt = txt + Left(" -" + Space(18), 18)  'saline part only if linear in g
    Case Else: txt = txt + Left(EFormat(q(3) - q(1), 9) + Space(18), 18)
  End Select
  txt = txt + EFormat(q(3), 9)
  txt = txt + CRLF + CRLF
  
Next i

chk_IAPWS08_Table8a = txt

End Function

'=========================================================================
Public Function chk_IAPWS08_Table8b() As String

'TABLE 8  Numerical check values for the water part computed from
'gW and its derivatives, Table 4, for the saline part computed from
'gS and its derivatives, Eq. (4), and for the seawater property
'computed from the Gibbs function g = gW + gS and its derivatives, Eq. (3).
'The numerical functions evaluated here at given points (S, T, p) are defined in Tables 3-6.

'b)  Properties at S = 0.1 kg kg-1, T = 353 K, p = p0 = 101325 Pa.
'This point is located in the regions (C) and (F) of Fig. 1 with restricted validity

'Quantity  Water part  Saline part Property of seawater  Unit
'g               -0.446114969E+5   0.150871740E+5  -0.295243229E+5  J kg-1
'(dg/dS)T,P       0.0              0.251957276E+6   0.251957276E+6  J kg-1
'(dg/dT)S,P      -0.107375993E+4   0.156230907E+3  -0.917529024E+3  J kg-1 K-1
'(dg/dP)S,T       0.102892956E-2  -0.579227286E-4   0.971006828E-3  m3 kg-1
'(d2g/dSdP)T      0.0             -0.305957802E-3  -0.305957802E-3  m3 kg-1
'(d2g/dT2)S,P   - 0.118885000E+2   0.127922649E+1  -0.106092735E+2  J kg-1 K-2
'(d2g/dTdP)S      0.659051552E-6   0.803061596E-6   0.146211315E-5  m3 kg-1 K-1
'(d2g/dP2)S,T    -0.474672819E-12  0.213086154E-12 -0.261586665E-12 m3 kg-1 Pa-1
'h                0.334425759E+6  -0.400623363E+5   0.294363423E+6  J kg-1
'f               -0.447157532E+5   0.150930430E+5  -0.296227102E+5  J kg-1
'u                0.334321503E+6  -0.400564673E+5   0.294265035E+6  J kg-1
's                0.107375993E+4  -0.156230907E+3   0.917529024E+3  J kg-1 K-1
'rho              0.971883832E+3   -                0.102985888E+4  kg m - 3
'cP               0.419664050E+4  -0.451566952E+3   0.374507355E+4  J kg-1 K-1
'w                0.155446297E+4   -                0.396127835E+4  m s-1
'W              -0.446114969E+5  -0.101085536E+5  -0.547200505E+5  J kg-1

Dim CRLF As String
Dim txt As String, row As String
Dim S As Double, t As Double, p As Double, q(3) As Double
Dim i As Integer, j As Integer

CRLF = Chr(13) + Chr(10)

txt = " Implementation of IAPWS-08 in Visual Basic" + CRLF
txt = txt + " for Publication in Ocean Science, 2009" + CRLF
txt = txt + " R. Feistel, IOW, Version " + Version + CRLF
txt = txt + " Compiled on " + CStr(Now) + CRLF + CRLF

txt = txt + " Function values as given in Table 8b of IAPWS-08:" + CRLF
txt = txt + " Properties at S = 0.03516504 kg kg-1,  T = 273.15 K, P = 101325 Pa" + CRLF + CRLF

txt = txt + "Quantity         Water part        Saline part       Seawater        Unit" + CRLF

t = 353
p = 101325

For i = 1 To 16
  Select Case i
    Case 1:  row = "g               -0.446114969E+5    0.150871740E+5   -0.295243229E+5  J kg-1"
    Case 2:  row = "(dg/dS)T,P       0.0               0.251957276E+6    0.251957276E+6  J kg-1"
    Case 3:  row = "(dg/dT)S,P      -0.107375993E+4    0.156230907E+3   -0.917529024E+3  J kg-1 K-1"
    Case 4:  row = "(dg/dP)S,T       0.102892956E-2   -0.579227286E-4    0.971006828E-3  m3 kg-1"
    Case 5:  row = "(d2g/dSdP)T      0.0              -0.305957802E-3   -0.305957802E-3  m3 kg-1"
    Case 6:  row = "(d2g/dT2)S,P   - 0.118885000E+2    0.127922649E+1   -0.106092735E+2  J kg-1 K-2"
    Case 7:  row = "(d2g/dTdP)S      0.659051552E-6    0.803061596E-6    0.146211315E-5  m3 kg-1 K-1"
    Case 8:  row = "(d2g/dP2)S,T    -0.474672819E-12   0.213086154E-12  -0.261586665E-12 m3 kg-1 Pa-1"
    Case 9:  row = "h                0.334425759E+6   -0.400623363E+5    0.294363423E+6  J kg-1"
    Case 10: row = "f               -0.447157532E+5    0.150930430E+5   -0.296227102E+5  J kg-1"
    Case 11: row = "u                0.334321503E+6   -0.400564673E+5    0.294265035E+6  J kg-1"
    Case 12: row = "s                0.107375993E+4   -0.156230907E+3    0.917529024E+3  J kg-1 K-1"
    Case 13: row = "rho              0.971883832E+3    -                 0.102985888E+4  kg m - 3"
    Case 14: row = "cP               0.419664050E+4   -0.451566952E+3    0.374507355E+4  J kg-1 K-1"
    Case 15: row = "w                0.155446297E+4    -                 0.396127835E+4  m s-1"
    Case 16: row = "W              -0.446114969E+5   -0.101085536E+5   -0.547200505E+5  J kg-1"
  End Select

  txt = txt + row + CRLF
  txt = txt + "this code:      "
  
  For j = 1 To 3 Step 2
    S = IIf(j = 3, 0.1, 0)
    q(j) = 0
    Select Case i
      Case 1: q(j) = sea_g_si(0, 0, 0, S, t, p)
      Case 2: If S > 0 Then q(j) = sea_g_si(1, 0, 0, S, t, p) 'no s-derivative of pure water
      Case 3: q(j) = sea_g_si(0, 1, 0, S, t, p)
      Case 4: q(j) = sea_g_si(0, 0, 1, S, t, p)
      Case 5: If S > 0 Then q(j) = sea_g_si(1, 0, 1, S, t, p) 'no s-derivative of pure water
      Case 6: q(j) = sea_g_si(0, 2, 0, S, t, p)
      Case 7: q(j) = sea_g_si(0, 1, 1, S, t, p)
      Case 8: q(j) = sea_g_si(0, 0, 2, S, t, p)

      Case 9: q(j) = sea_enthalpy_si(S, t, p)
      Case 10: q(j) = sea_g_si(0, 0, 0, S, t, p) - p * sea_g_si(0, 0, 1, S, t, p)
      Case 11: q(j) = sea_g_si(0, 0, 0, S, t, p) - t * sea_g_si(0, 1, 0, S, t, p) - p * sea_g_si(0, 0, 1, S, t, p)
      Case 12: q(j) = sea_entropy_si(S, t, p)
      Case 13: q(j) = sea_density_si(S, t, p)
      Case 14: q(j) = sea_cp_si(S, t, p)
      Case 15: q(j) = sea_soundspeed_si(S, t, p)
      Case 16: q(j) = sea_chempot_h2o_si(S, t, p)
    End Select

  Next j
  txt = txt + Left(EFormat(q(1), 9) + Space(18), 18)
  Select Case i
    Case 13, 15: txt = txt + Left(" -" + Space(18), 18)  'saline part only if linear in g
    Case Else: txt = txt + Left(EFormat(q(3) - q(1), 9) + Space(18), 18)
  End Select
  txt = txt + EFormat(q(3), 9)
  txt = txt + CRLF + CRLF
  
Next i

chk_IAPWS08_Table8b = txt

End Function

'=========================================================================
Public Function chk_IAPWS08_Table8c() As String

'TABLE 8  Numerical check values for the water part computed from
'gW and its derivatives, Table 4, for the saline part computed from
'gS and its derivatives, Eq. (4), and for the seawater property
'computed from the Gibbs function g = gW + gS and its derivatives, Eq. (3).
'The numerical functions evaluated here at given points (S, T, p) are defined in Tables 3-6.

'c)  Properties at S = Sn = 0.035 165 04 kg kg-1, T = T0 = 273.15 K, p = 1E8 Pa
'
'Quantity  Water part  Saline part Property of seawater  Unit
'g                0.977303862E+5  -0.260093051E+4   0.951294557E+5  J kg-1
'(dg/dS)_T,P      0.0             -0.545861581E+4  -0.545861581E+4  J kg-1
'(dg/dT)_S,P      0.851466502E-    0.754045685E-    0.160551219E+2  J kg-1 K-1
'(dg/dP)_S,T      0.956683329E-3  -0.229123842E-4   0.933770945E-3  m3 kg-1
'(d2g/dSdP)_T     0.0             -0.640757619E-3  -0.640757619E-3  m3 kg-1
'(d2g/dT2)_S,P   -0.142969873E+2   0.488076974     -0.138089104E+2  J kg-1 K-2
'(d2g/dTdP)_S     0.199079571E-6   0.466284412E-7   0.245708012E-6  m3 kg-1 K-1
'(d2g/dP2)_S,T   -0.371530889E-12  0.357345736E-13 -0.335796316E-12 m3 kg-1 Pa-1
'h                0.954046055E+5  -0.466060630E+4   0.907439992E+5  J kg-1
'f                0.206205330E+4  -0.309692089E+3   0.175236121E+4  J kg-1
'u               -0.263727446E+3  -0.236936788E+4  -0.263309532E+4  J kg-1
's               -0.851466502E+1  -0.754045685E+1  -0.160551219E+2  J kg-1 K-1
'rho              0.104527796E+4   -                0.107092645E+4  kg m-3
'cP               0.390522209E+4  -0.133318225E+3   0.377190387E+4  J kg-1 K-1
'w                0.157542240E+4   -                0.162198998E+4  m s-1
'W               0.977303862E+5  -0.240897806E+4   0.953214082E+5  J kg-1

Dim CRLF As String
Dim txt As String, row As String
Dim S As Double, t As Double, p As Double, q(3) As Double
Dim i As Integer, j As Integer

CRLF = Chr(13) + Chr(10)

txt = " Implementation of IAPWS-08 in Visual Basic" + CRLF
txt = txt + " for Publication in Ocean Science, 2009" + CRLF
txt = txt + " R. Feistel, IOW, Version " + Version + CRLF
txt = txt + " Compiled on " + CStr(Now) + CRLF + CRLF

txt = txt + " Function values as given in Table 8c of IAPWS-08:" + CRLF
txt = txt + " Properties at S = 0.03516504 kg kg-1,  T = 273.15 K, P = 100 MPa" + CRLF + CRLF

txt = txt + "Quantity         Water part        Saline part       Seawater        Unit" + CRLF

t = 273.15
p = 100000000#

For i = 1 To 16
  Select Case i
    Case 1:  row = "g                0.977303862E+5   -0.260093051E+4    0.951294557E+5  J kg-1"
    Case 2:  row = "(dg/dS)_T,P      0.0              -0.545861581E+4   -0.545861581E+4  J kg-1"
    Case 3:  row = "(dg/dT)_S,P      0.851466502E+1    0.754045685E+1    0.160551219E+2  J kg-1 K-1"
    Case 4:  row = "(dg/dP)_S,T      0.956683329E-3   -0.229123842E-4    0.933770945E-3  m3 kg-1"
    Case 5:  row = "(d2g/dSdP)_T     0.0              -0.640757619E-3   -0.640757619E-3  m3 kg-1"
    Case 6:  row = "(d2g/dT2)_S,P   -0.142969873E+2    0.488076974      -0.138089104E+2  J kg-1 K-2"
    Case 7:  row = "(d2g/dTdP)_S     0.199079571E-6    0.466284412E-7    0.245708012E-6  m3 kg-1 K-1"
    Case 8:  row = "(d2g/dP2)_S,T   -0.371530889E-12   0.357345736E-13  -0.335796316E-12 m3 kg-1 Pa-1"
    Case 9:  row = "h                0.954046055E+5   -0.466060630E+4    0.907439992E+5  J kg-1"
    Case 10: row = "f                0.206205330E+4   -0.309692089E+3    0.175236121E+4  J kg-1"
    Case 11: row = "u               -0.263727446E+3   -0.236936788E+4   -0.263309532E+4  J kg-1"
    Case 12: row = "s               -0.851466502E+1   -0.754045685E+1   -0.160551219E+2  J kg-1 K-1"
    Case 13: row = "rho              0.104527796E+4    -                 0.107092645E+4  kg m-3"
    Case 14: row = "cP               0.390522209E+4   -0.133318225E+3    0.377190387E+4  J kg-1 K-1"
    Case 15: row = "w                0.157542240E+4    -                 0.162198998E+4  m s-1"
    Case 16: row = "W               0.977303862E+5   -0.240897806E+4    0.953214082E+5  J kg-1"
  End Select

  txt = txt + row + CRLF
  txt = txt + "this code:      "
  
  For j = 1 To 3 Step 2
    S = IIf(j = 3, 0.03516504, 0)
    q(j) = 0
    Select Case i
      Case 1: q(j) = sea_g_si(0, 0, 0, S, t, p)
      Case 2: If S > 0 Then q(j) = sea_g_si(1, 0, 0, S, t, p) 'no s-derivative of pure water
      Case 3: q(j) = sea_g_si(0, 1, 0, S, t, p)
      Case 4: q(j) = sea_g_si(0, 0, 1, S, t, p)
      Case 5: If S > 0 Then q(j) = sea_g_si(1, 0, 1, S, t, p) 'no s-derivative of pure water
      Case 6: q(j) = sea_g_si(0, 2, 0, S, t, p)
      Case 7: q(j) = sea_g_si(0, 1, 1, S, t, p)
      Case 8: q(j) = sea_g_si(0, 0, 2, S, t, p)

      Case 9: q(j) = sea_enthalpy_si(S, t, p)
      Case 10: q(j) = sea_g_si(0, 0, 0, S, t, p) - p * sea_g_si(0, 0, 1, S, t, p)
      Case 11: q(j) = sea_g_si(0, 0, 0, S, t, p) - t * sea_g_si(0, 1, 0, S, t, p) - p * sea_g_si(0, 0, 1, S, t, p)
      Case 12: q(j) = sea_entropy_si(S, t, p)
      Case 13: q(j) = sea_density_si(S, t, p)
      Case 14: q(j) = sea_cp_si(S, t, p)
      Case 15: q(j) = sea_soundspeed_si(S, t, p)
      Case 16: q(j) = sea_chempot_h2o_si(S, t, p)
    End Select

  Next j
  txt = txt + Left(EFormat(q(1), 9) + Space(18), 18)
  Select Case i
    Case 13, 15: txt = txt + Left(" -" + Space(18), 18)  'saline part only if linear in g
    Case Else: txt = txt + Left(EFormat(q(3) - q(1), 9) + Space(18), 18)
  End Select
  txt = txt + EFormat(q(3), 9)
  txt = txt + CRLF + CRLF
  
Next i

chk_IAPWS08_Table8c = txt

End Function

'==========================================================================
Public Function Sea_3a_example_call(ByVal sa_si As Double, _
                                    ByVal t_si As Double, _
                                    ByVal p_si As Double) As String

Dim CRLF As String, TB As String
Dim txt As String

CRLF = Chr(13) + Chr(10)
TB = Chr(9)

txt = " Implementation of thermodynamic properties of seawater in Visual Basic" + CRLF
txt = txt + " for Publication in Ocean Science, 2009" + CRLF
txt = txt + " R. Feistel, IOW, Version " + Version + CRLF
txt = txt + " Compiled on " + CStr(Now) + CRLF + CRLF

If p_si <= 0 Then
  txt = txt + "incorrect: negative pressure"
  Sea_3a_example_call = txt
  Exit Function
End If

If t_si <= 0 Then
  txt = txt + "incorrect: negative temperature"
  Sea_3a_example_call = txt
  Exit Function
End If

If sa_si < 0 Then
  txt = txt + "incorrect: negative salinity"
  Sea_3a_example_call = txt
  Exit Function
End If

If sa_si >= 1 Then
  txt = txt + "incorrect: salinity >= 1000 g/kg"
  Sea_3a_example_call = txt
  Exit Function
End If


If p_si > sal_pmax Then
  txt = txt + "Warning: pressure >" + Str(sal_pmax) + " Pa is outside validity" + CRLF + CRLF
End If

If p_si < sal_pmin Then
  txt = txt + "Warning: pressure <" + Str(sal_pmin) + " Pa is outside validity" + CRLF + CRLF
End If

If t_si > sal_tmax Then
  txt = txt + "Warning: temperature >" + Str(sal_tmax) + " K is outside validity" + CRLF + CRLF
End If

If t_si < sal_tmin Then
  txt = txt + "Warning: temperature <" + Str(sal_tmin) + " K is outside validity" + CRLF + CRLF
End If

If sa_si > sal_smax Then
  txt = txt + "Warning: salinity >" + Str(sal_smax) + " K is outside validity" + CRLF + CRLF
End If

If sa_si < sal_smin Then
  txt = txt + "Warning: salinity >" + Str(sal_smin) + " K is outside validity" + CRLF + CRLF
End If

txt = txt + " Absolute salinity:          " + TB + Str(sa_si) + TB + "kg/kg" + CRLF
txt = txt + " Absolute temperature:       " + TB + Str(t_si) + TB + "K" + CRLF
txt = txt + " Absolute pressure:          " + TB + Str(p_si) + TB + "Pa" + CRLF + CRLF
txt = txt + " Isobaric heat capacity:     " + TB + Str(sea_cp_si(sa_si, t_si, p_si)) + TB + "J/(kg K)" + CRLF
txt = txt + " Density:                    " + TB + Str(sea_density_si(sa_si, t_si, p_si)) + TB + "kg/m3" + CRLF
txt = txt + " Enthalpy:                   " + TB + Str(sea_enthalpy_si(sa_si, t_si, p_si)) + TB + "J/kg" + CRLF
txt = txt + " Entropy:                    " + TB + Str(sea_entropy_si(sa_si, t_si, p_si)) + TB + "J/(kg K)" + CRLF
txt = txt + " Thermal expansion:          " + TB + Str(sea_g_expansion_t_si(sa_si, t_si, p_si)) + TB + "1/K" + CRLF
txt = txt + " Haline contraction:         " + TB + Str(sea_g_contraction_t_si(sa_si, t_si, p_si)) + TB + "kg/kg" + CRLF
txt = txt + " Gibbs energy:               " + TB + Str(sea_gibbs_energy_si(sa_si, t_si, p_si)) + TB + "J/kg" + CRLF
txt = txt + " Internal energy:            " + TB + Str(sea_internal_energy_si(sa_si, t_si, p_si)) + TB + "J/kg" + CRLF
txt = txt + " Adiabatic compressibility:  " + TB + Str(sea_kappa_s_si(sa_si, t_si, p_si)) + TB + "1/Pa" + CRLF
txt = txt + " Isothermal compressibility: " + TB + Str(sea_kappa_t_si(sa_si, t_si, p_si)) + TB + "1/Pa" + CRLF
txt = txt + " Adiabatic lapse rate:       " + TB + Str(sea_lapserate_si(sa_si, t_si, p_si)) + TB + "K/Pa" + CRLF
txt = txt + " Osmotic coefficient:        " + TB + Str(sea_osm_coeff_si(sa_si, t_si, p_si)) + TB + "1" + CRLF
txt = txt + " Sound speed:                " + TB + Str(sea_soundspeed_si(sa_si, t_si, p_si)) + TB + "m/s" + CRLF + CRLF

txt = txt + "Note: Absolute energy and entropy are defined by IAPWS-95 and IAPWS-08" + CRLF + CRLF

Sea_3a_example_call = txt

End Function


