Attribute VB_Name = "Liq_Ice_Air_4_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules:
'     Constants_0_Mdl,  file Constants_0.bas
'     Convert_0_Mdl,    file Convert_0.bas
'     Maths_0_Mdl,      file Maths_0.bas
'     Air_1_Mdl,        file Air_1.bas
'     Air_2_Mdl,        file Air_2.bas
'     Air_3b_Mdl,       file Air_3b.bas
'     Ice_1_Mdl,        file Ice_1.bas
'     Ice_2_Mdl,        file Ice_2.bas
'     Ice_Liq_4_Mdl,    file Ice_Liq_4.bas
'     Flu_1_Mdl,        file Flu_1.bas
'     Flu_2_Mdl,        file Flu_2.bas

'#########################################################################

'=========================================================================
'This module implements the equilibrium properties of wet ice air, i.e., between
'liquid water, ice and humid air

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009

'Feistel, R., Kretzschmar, H.-J., Span, R., Hagen, E., Wright, D.G., Herrmann, S.:
'Thermodynamic Properties of Sea Air.
'Ocean Science Discussion 6(2009)21932325.
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98
'Private Const IsOK = -1

'Control parameters of the liq-ice-air iteration
Private ctrl_initialized As Integer

Private ctrl_mode_humidity As Integer
Private ctrl_mode_temperature As Integer
Private ctrl_mode_pressure As Integer
Private ctrl_mode_liquid As Integer
Private ctrl_mode_air As Integer
Private ctrl_loop_maximum As Long

Private ctrl_init_a As Double        'initial dry-air fraction of humid air
Private ctrl_init_t As Double        'initial temperature
Private ctrl_init_p As Double        'initial pressure
Private ctrl_init_d_liq As Double    'initial liquid density
Private ctrl_init_d_air As Double    'initial humid-air density

Private ctrl_eps_exit_t As Double    'temperature exit tolerance
Private ctrl_eps_exit_p As Double    'pressure exit tolerance

'Properties of the current water-ice-humid-air equilibrium state
Private equi_liq_ice_air_done As Integer  'flag = IsOK if equilibrium was successfully computed

Private equi_liq_ice_air_t As Double      'equilibrium temperature
Private equi_liq_ice_air_p As Double      'equilibrium pressure
Private equi_liq_ice_air_a As Double      'equilibrium dry-air fraction in humid air
Private equi_liq_ice_air_d_liq As Double  'equilibrium density of liquid water at a, t, p
Private equi_liq_ice_air_d_air As Double  'equilibrium density of humid air at a, t, p

'if only a-t-p was fixed by equilibrium setting, the following values may remain unspecified
Private equi_liq_ice_air_wa As Double     'equilibrium dry-air fraction
Private equi_liq_ice_air_wet As Double    'equilibrium liquid fraction of the condensed part
Private equi_liq_ice_air_eta As Double    'equilibrium entropy

Private Const Version = "28 May 2010"

'==========================================================================
Public Function liq_ice_air_entropy_si() As Double

'function computes the entropy of wet ice air after previously calling
'set_liq_ice_air_eq_at_wa_wl_wi,... etc

'Note: the accuracy of this function depends on the iteration settings in this module

'check values with default settings:  v. 1.0
'set_liq_ice_air_eq_at_a 0.99
'liq_ice_air_entropy_si = 9.99999999E+98

'set_liq_ice_air_eq_at_p 1E4
'liq_ice_air_entropy_si = 9.99999999E+98

'set_liq_ice_air_eq_at_t 273.155
'liq_ice_air_entropy_si = 9.99999999E+98

'set_liq_ice_air_eq_at_wa_eta_wt 0.99, 0, 0.5
'liq_ice_air_entropy_si = 0

'set_liq_ice_air_eq_at_wa_wl_wi .1, .2, .3
'liq_ice_air_entropy_si = 3496.20368786436


'check values with default settings:  v. 1.1
'set_liq_ice_air_eq_at_a 0.99
'liq_ice_air_entropy_si = 9.99999999E+98

'set_liq_ice_air_eq_at_p 1E4
'liq_ice_air_entropy_si = 9.99999999E+98

'set_liq_ice_air_eq_at_t 273.155
'liq_ice_air_entropy_si = 9.99999999E+98

'set_liq_ice_air_eq_at_wa_eta_wt 0.99, 0, 0.5
'liq_ice_air_entropy_si = 0

'set_liq_ice_air_eq_at_wa_wl_wi .1, .2, .3
'liq_ice_air_entropy_si = 3496.16306902868

liq_ice_air_entropy_si = ErrorReturn

If equi_liq_ice_air_done <> IsOK Then Exit Function

liq_ice_air_entropy_si = equi_liq_ice_air_eta


End Function

'==========================================================================
Public Function liq_ice_air_enthalpy_si() As Double

'function computes the enthalpy of wet ice air after previously calling
'set_liq_ice_air_eq_at_wa_wl_wi,... etc

'Note: the accuracy of this function depends on the iteration settings in this module

'check values with default settings:  v. 1.0
'set_liq_ice_air_eq_at_a 0.99
'liq_ice_air_enthalpy_si = 9.99999999E+98

'set_liq_ice_air_eq_at_p 1E4
'liq_ice_air_enthalpy_si = 9.99999999E+98

'set_liq_ice_air_eq_at_t 273.155
'liq_ice_air_enthalpy_si = 9.99999999E+98

'set_liq_ice_air_eq_at_wa_eta_wt 0.99, 0, 0.5
'liq_ice_air_enthalpy_si = 7358.21843392192

'set_liq_ice_air_eq_at_wa_wl_wi .1, .2, .3
'liq_ice_air_enthalpy_si = 900361.142093772


'check values with default settings:  v. 1.1
'set_liq_ice_air_eq_at_a 0.99
'liq_ice_air_enthalpy_si = 9.99999999E+98

'set_liq_ice_air_eq_at_p 1E4
'liq_ice_air_enthalpy_si = 9.99999999E+98

'set_liq_ice_air_eq_at_t 273.155
'liq_ice_air_enthalpy_si = 9.99999999E+98

'set_liq_ice_air_eq_at_wa_eta_wt 0.99, 0, 0.5
'liq_ice_air_enthalpy_si = 7356.1294372357

'set_liq_ice_air_eq_at_wa_wl_wi .1, .2, .3
'liq_ice_air_enthalpy_si = 900361.135280164

Dim a As Double, t As Double, p As Double
Dim dh As Double, dl As Double, wet As Double
Dim wa As Double, wh As Double, wl As Double, wi As Double
Dim hh As Double, hl As Double, hi As Double

liq_ice_air_enthalpy_si = ErrorReturn

If equi_liq_ice_air_done <> IsOK Then Exit Function

If equi_liq_ice_air_wa = ErrorReturn Or _
   equi_liq_ice_air_wet = ErrorReturn Then Exit Function

a = equi_liq_ice_air_a
t = equi_liq_ice_air_t
p = equi_liq_ice_air_p
dh = equi_liq_ice_air_d_air
dl = equi_liq_ice_air_d_liq
wa = equi_liq_ice_air_wa
wet = equi_liq_ice_air_wet

If a = 0 Then Exit Function  'the air-free triple point cannot be computed here

wh = wa / a                'wh is the humid-air fraction, 1-wh is the condensed fraction
If wh < 0 Or wh > 1 Then Exit Function
wl = wet * (1 - wh)                       'the liquid fraction
If wl < 0 Or wl > 1 Then Exit Function
wi = (1 - wet) * (1 - wh)                 'the ice fraction
If wi < 0 Or wi > 1 Then Exit Function

hh = air_f_enthalpy_si(a, t, dh)          'enthalpy of humid air
If hh = ErrorReturn Then Exit Function
hl = flu_enthalpy_si(t, dl)               'enthalpy of liquid water
If hl = ErrorReturn Then Exit Function
hi = ice_enthalpy_si(t, p)                'enthalpy of ice
If hi = ErrorReturn Then Exit Function

liq_ice_air_enthalpy_si = wh * hh + wl * hl + wi * hi

End Function

'==========================================================================
Public Function liq_ice_air_density_si() As Double

'function computes the density of wet ice air after previously calling
'set_liq_ice_air_eq_at_wa_wl_wi,... etc

'Note: the accuracy of this function depends on the iteration settings in this module

'check values with default settings:  v. 1.0
'set_liq_ice_air_eq_at_a 0.99
'liq_ice_air_density_si = 9.99999999E+98

'set_liq_ice_air_eq_at_p 1E4
'liq_ice_air_density_si = 9.99999999E+98

'set_liq_ice_air_eq_at_t 273.155
'liq_ice_air_density_si = 9.99999999E+98

'set_liq_ice_air_eq_at_wa_eta_wt 0.99, 0, 0.5
'liq_ice_air_density_si = 7.74649959201826

'set_liq_ice_air_eq_at_wa_wl_wi .1, .2, .3
'liq_ice_air_density_si = 474.974398771744


'check values with default settings:  v. 1.1
'set_liq_ice_air_eq_at_a 0.99
'liq_ice_air_density_si = 9.99999999E+98

'set_liq_ice_air_eq_at_p 1E4
'liq_ice_air_density_si = 9.99999999E+98

'set_liq_ice_air_eq_at_t 273.155
'liq_ice_air_density_si = 9.99999999E+98

'set_liq_ice_air_eq_at_wa_eta_wt 0.99, 0, 0.5
'liq_ice_air_density_si = 7.7475797940365

'set_liq_ice_air_eq_at_wa_wl_wi .1, .2, .3
'liq_ice_air_density_si = 474.974398768669

Dim t As Double, p As Double, a As Double
Dim dh As Double, dl As Double, di As Double
Dim wl As Double, wi As Double

liq_ice_air_density_si = ErrorReturn

If equi_liq_ice_air_done <> IsOK Then Exit Function

p = liq_ice_air_pressure_si
If p = ErrorReturn Then Exit Function
t = liq_ice_air_temperature_si
If t = ErrorReturn Then Exit Function
dh = equi_liq_ice_air_d_air
If dh = ErrorReturn Then Exit Function
dl = equi_liq_ice_air_d_liq
If dl = ErrorReturn Then Exit Function

If equi_liq_ice_air_wet = ErrorReturn Then Exit Function
If equi_liq_ice_air_wa = ErrorReturn Then Exit Function
If equi_liq_ice_air_a = ErrorReturn Then Exit Function
If equi_liq_ice_air_a = 0 Then 'air-free triple point is not supported here
  Exit Function
End If
wl = equi_liq_ice_air_wet * (1 - equi_liq_ice_air_wa / equi_liq_ice_air_a)
wi = (1 - equi_liq_ice_air_wet) * (1 - equi_liq_ice_air_wa / equi_liq_ice_air_a)
di = ice_density_si(t, p)
If di = ErrorReturn Then Exit Function

liq_ice_air_density_si = (1 - wi - wl) * dh + wl * dl + wi * di

End Function

'==========================================================================
Public Function liq_ice_air_ifl_si(ByVal wa_si As Double, _
                                   ByVal eta_si As Double) As Double

'this function computes the isentropic freezing level (IFL) in Pa
'from the dry-air fraction, wa_si in kg/kg, and the entropy, eta_si in J/(kg K)
'of wet air

'Note: the accuracy of this function depends on the iteration settings in this module

'check value with default settings:
'liq_ice_air_ifl_si(.99, 100) = 83240.7163329918  v. 1.0
'liq_ice_air_ifl_si(.99, 100) = 83234.7314360022  v. 1.1

liq_ice_air_ifl_si = ErrorReturn

If set_liq_ice_air_eq_at_wa_eta_wt(wa_si, eta_si, 1) = ErrorReturn Then Exit Function

liq_ice_air_ifl_si = equi_liq_ice_air_p

End Function

'==========================================================================
Public Function liq_ice_air_iml_si(ByVal wa_si As Double, _
                                   ByVal eta_si As Double) As Double

'this function computes the isentropic melting level (IML) in Pa
'from the dry-air fraction, wa_si in kg/kg, and the entropy, eta_si in J/(kg K)
'of ice air

'Note: the accuracy of this function depends on the iteration settings in this module

'check value with default settings:
'liq_ice_air_iml_si(.99, 100) = 81612.0101048005  v. 1.0
'liq_ice_air_iml_si(.99, 100) = 81605.5557727816  v. 1.1

liq_ice_air_iml_si = ErrorReturn

If set_liq_ice_air_eq_at_wa_eta_wt(wa_si, eta_si, 0) = ErrorReturn Then Exit Function

liq_ice_air_iml_si = equi_liq_ice_air_p

End Function

'==========================================================================
Public Function liq_ice_air_pressure_si() As Double

'this function returns the pressure in Pa of water + ice at equilibrium with humid air,
'set by a previous call of set_liq_ice_air_eq_at_a, etc

'Note: the accuracy of this function depends on the iteration settings in this module

'check values with default settings:  v. 1.0
'set_liq_ice_air_eq_at_a 0.99
'liq_ice_air_pressure_si = 38347.8983146684

'set_liq_ice_air_eq_at_p 1E4
'liq_ice_air_pressure_si = 10000

'set_liq_ice_air_eq_at_t 273.155
'liq_ice_air_pressure_si = 67931.6010|763921

'set_liq_ice_air_eq_at_wa_eta_wt 0.99, 0, 0.5
'liq_ice_air_pressure_si = 112016.572991626

'set_liq_ice_air_eq_at_wa_wl_wi .1, .2, .3
'liq_ice_air_pressure_si = 706.839968365434


'check values with default settings:  v. 1.1
'set_liq_ice_air_eq_at_a 0.99
'liq_ice_air_pressure_si = 38338.9622423812

'set_liq_ice_air_eq_at_p 1E4
'liq_ice_air_pressure_si = 10000

'set_liq_ice_air_eq_at_t 273.155
'liq_ice_air_pressure_si = 67931.6010|763921

'set_liq_ice_air_eq_at_wa_eta_wt 0.99, 0, 0.5
'liq_ice_air_pressure_si = 112016.075795051

'set_liq_ice_air_eq_at_wa_wl_wi .1, .2, .3
'liq_ice_air_pressure_si = 706.817425300617

liq_ice_air_pressure_si = ErrorReturn

If equi_liq_ice_air_done <> IsOK Then Exit Function

liq_ice_air_pressure_si = equi_liq_ice_air_p

End Function

'==========================================================================
Public Function liq_ice_air_airfraction_si() As Double

'this function returns the fraction a in kg/kg of dry air in humid air
'when water + ice are at equilibrium with humid air,
'set by a previous call of set_liq_ice_air_eq_at_a, etc

'Note: the accuracy of this function depends on the iteration settings in this module

'check values with default settings:  v. 1.0
'set_liq_ice_air_eq_at_a 0.99
'liq_ice_air_airfraction_si = 0.99

'set_liq_ice_air_eq_at_p 1E4
'liq_ice_air_airfraction_si = 0.961015437340699

'set_liq_ice_air_eq_at_t 273.155
'liq_ice_air_airfraction_si = 0.99436473982|5772

'set_liq_ice_air_eq_at_wa_eta_wt 0.99, 0, 0.5
'liq_ice_air_airfraction_si = 0.996582564510175

'set_liq_ice_air_eq_at_wa_wl_wi .1, .2, .3
'liq_ice_air_airfraction_si = 0.2


'check values with default settings:  v. 1.1
'set_liq_ice_air_eq_at_a 0.99
'liq_ice_air_airfraction_si = 0.99

'set_liq_ice_air_eq_at_p 1E4
'liq_ice_air_airfraction_si = 0.961024307543784

'set_liq_ice_air_eq_at_t 273.155
'liq_ice_air_airfraction_si = 0.994366063923|088

'set_liq_ice_air_eq_at_wa_eta_wt 0.99, 0, 0.5
'liq_ice_air_airfraction_si = 0.996583352944083

'set_liq_ice_air_eq_at_wa_wl_wi .1, .2, .3
'liq_ice_air_airfraction_si = 0.2

liq_ice_air_airfraction_si = ErrorReturn

If equi_liq_ice_air_done <> IsOK Then Exit Function

liq_ice_air_airfraction_si = equi_liq_ice_air_a

End Function

'==========================================================================
Public Function liq_ice_air_dryairfraction_si() As Double

'this function returns the fraction a of dry air in wet ice air in kg/kg
'when water + ice are in equilibrium with humid air,
'set by a previous call of set_liq_ice_air_eq_at_a, etc

'Note: the accuracy of this function depends on the iteration settings in this module

'check values with default settings: v. 1.0 and v. 1.1
'set_liq_ice_air_eq_at_a 0.99
'liq_ice_air_dryairfraction_si = 9.99999999E+98

'set_liq_ice_air_eq_at_p 1E4
'liq_ice_air_dryairfraction_si = 9.99999999E+98

'set_liq_ice_air_eq_at_t 273.155
'liq_ice_air_dryairfraction_si = 9.99999999E+98

'set_liq_ice_air_eq_at_wa_eta_wt 0.99, 0, 0.5
'liq_ice_air_dryairfraction_si = 0.99

'set_liq_ice_air_eq_at_wa_wl_wi .1, .2, .3
'liq_ice_air_dryairfraction_si = 0.1

liq_ice_air_dryairfraction_si = ErrorReturn

If equi_liq_ice_air_done <> IsOK Then Exit Function

liq_ice_air_dryairfraction_si = equi_liq_ice_air_wa

End Function


'==========================================================================
Public Function liq_ice_air_solidfraction_si() As Double


'this function returns the fraction of ice in wet ice air in kg/kg
'when water + ice are at equilibrium with humid air,
'set by a previous call of set_liq_ice_air_eq_at_wa_eta_wt, etc

'Note: the accuracy of this function depends on the iteration settings in this module

'check values with default settings:  v. 1.0
'set_liq_ice_air_eq_at_a 0.99
'liq_ice_air_solidfraction_si = 9.99999999E+98

'set_liq_ice_air_eq_at_p 1E4
'liq_ice_air_solidfraction_si = 9.99999999E+98

'set_liq_ice_air_eq_at_t 273.155
'liq_ice_air_solidfraction_si = 9.99999999E+98

'set_liq_ice_air_eq_at_wa_eta_wt 0.99, 0, 0.5
'liq_ice_air_solidfraction_si = 3.30256857012676E-03

'set_liq_ice_air_eq_at_wa_wl_wi .1, .2, .3
'liq_ice_air_solidfraction_si = 0.3


'check values with default settings:  v. 1.1
'set_liq_ice_air_eq_at_a 0.99
'liq_ice_air_solidfraction_si = 9.99999999E+98

'set_liq_ice_air_eq_at_p 1E4
'liq_ice_air_solidfraction_si = 9.99999999E+98

'set_liq_ice_air_eq_at_t 273.155
'liq_ice_air_solidfraction_si = 9.99999999E+98

'set_liq_ice_air_eq_at_wa_eta_wt 0.99, 0, 0.5
'liq_ice_air_solidfraction_si = 3.30296152581446E-03

'set_liq_ice_air_eq_at_wa_wl_wi .1, .2, .3
'liq_ice_air_solidfraction_si = 0.3

liq_ice_air_solidfraction_si = ErrorReturn

If equi_liq_ice_air_done <> IsOK Then Exit Function
If equi_liq_ice_air_wa = ErrorReturn Or _
   equi_liq_ice_air_wet = ErrorReturn Then Exit Function
If equi_liq_ice_air_a = 0 Then Exit Function

liq_ice_air_solidfraction_si = (1 - equi_liq_ice_air_wet) * (1 - equi_liq_ice_air_wa / equi_liq_ice_air_a)

End Function

'==========================================================================
Public Function liq_ice_air_liquidfraction_si() As Double

'this function returns the fraction of liquid water in wet ice air in kg/kg
'when water + ice are at equilibrium with humid air,
'set by a previous call of set_liq_ice_air_eq_at_wa_eta_wt, etc

'Note: the accuracy of this function depends on the iteration settings in this module

'check values with default settings:  v. 1.0
'set_liq_ice_air_eq_at_a 0.99
'liq_ice_air_liquidfraction_si = 9.99999999E+98

'set_liq_ice_air_eq_at_p 1E4
'liq_ice_air_liquidfraction_si = 9.99999999E+98

'set_liq_ice_air_eq_at_t 273.155
'liq_ice_air_liquidfraction_si = 9.99999999E+98

'set_liq_ice_air_eq_at_wa_eta_wt 0.99, 0, 0.5
'liq_ice_air_liquidfraction_si = 3.30256857012676E-03

'set_liq_ice_air_eq_at_wa_wl_wi .1, .2, .3
'liq_ice_air_liquidfraction_si = 0.2


'check values with default settings:  v. 1.1
'set_liq_ice_air_eq_at_a 0.99
'liq_ice_air_liquidfraction_si = 9.99999999E+98

'set_liq_ice_air_eq_at_p 1E4
'liq_ice_air_liquidfraction_si = 9.99999999E+98

'set_liq_ice_air_eq_at_t 273.155
'liq_ice_air_liquidfraction_si = 9.99999999E+98

'set_liq_ice_air_eq_at_wa_eta_wt 0.99, 0, 0.5
'liq_ice_air_liquidfraction_si = 3.30296152581446E-03

'set_liq_ice_air_eq_at_wa_wl_wi .1, .2, .3
'liq_ice_air_liquidfraction_si = 0.2

liq_ice_air_liquidfraction_si = ErrorReturn

If equi_liq_ice_air_done <> IsOK Then Exit Function
If equi_liq_ice_air_wa = ErrorReturn Or _
   equi_liq_ice_air_wet = ErrorReturn Then Exit Function
If equi_liq_ice_air_a = 0 Then Exit Function

liq_ice_air_liquidfraction_si = equi_liq_ice_air_wet * (1 - equi_liq_ice_air_wa / equi_liq_ice_air_a)

End Function

'==========================================================================
Public Function liq_ice_air_vapourfraction_si() As Double

'this function returns the fraction of vapour in wet ice air in kg/kg
'when water + ice are at equilibrium with humid air,
'set by a previous call of set_liq_ice_air_eq_at_wa_eta_wt, etc

'Note: the accuracy of this function depends on the iteration settings in this module

'check values with default settings:  v. 1.0
'set_liq_ice_air_eq_at_a 0.99
'liq_ice_air_vapourfraction_si = 9.99999999E+98

'set_liq_ice_air_eq_at_p 1E4
'liq_ice_air_vapourfraction_si = 9.99999999E+98

'set_liq_ice_air_eq_at_t 273.155
'liq_ice_air_vapourfraction_si = 9.99999999E+98

'set_liq_ice_air_eq_at_wa_eta_wt 0.99, 0, 0.5
'liq_ice_air_vapourfraction_si = 3.3948628597465E-03

'set_liq_ice_air_eq_at_wa_wl_wi .1, .2, .3
'liq_ice_air_vapourfraction_si = 0.4


'check values with default settings:  v. 1.1
'set_liq_ice_air_eq_at_a 0.99
'liq_ice_air_vapourfraction_si = 9.99999999E+98

'set_liq_ice_air_eq_at_p 1E4
'liq_ice_air_vapourfraction_si = 9.99999999E+98

'set_liq_ice_air_eq_at_t 273.155
'liq_ice_air_vapourfraction_si = 9.99999999E+98

'set_liq_ice_air_eq_at_wa_eta_wt 0.99, 0, 0.5
'liq_ice_air_vapourfraction_si = 3.39407694837108E-03

'set_liq_ice_air_eq_at_wa_wl_wi .1, .2, .3
'liq_ice_air_vapourfraction_si = 0.4

liq_ice_air_vapourfraction_si = ErrorReturn

If equi_liq_ice_air_done <> IsOK Then Exit Function
If equi_liq_ice_air_wa = ErrorReturn Or _
   equi_liq_ice_air_wet = ErrorReturn Then Exit Function
If equi_liq_ice_air_a = 0 Then Exit Function

liq_ice_air_vapourfraction_si = equi_liq_ice_air_wa * (1 / equi_liq_ice_air_a - 1)

End Function


'==========================================================================
Public Function liq_ice_air_temperature_si() As Double

'this function returns the temperature in K of water + ice at equilibrium with humid air,
'set by a previous call of set_liq_ice_air_eq_at_a, etc

'Note: the accuracy of this function depends on the iteration settings in this module

'check values with default settings:  v. 1.0
'set_liq_ice_air_eq_at_a 0.99
'liq_ice_air_temperature_si = 273.157197423614

'set_liq_ice_air_eq_at_p 1E4
'liq_ice_air_temperature_si = 273.159302793328

'set_liq_ice_air_eq_at_t 273.155
'liq_ice_air_temperature_si = 273.155

'set_liq_ice_air_eq_at_wa_eta_wt 0.99, 0, 0.5
'liq_ice_air_temperature_si = 273.151724933173

'set_liq_ice_air_eq_at_wa_wl_wi .1, .2, .3
'liq_ice_air_temperature_si = 273.159992931404


'check values with default settings:  v. 1.1
'set_liq_ice_air_eq_at_a 0.99
'liq_ice_air_temperature_si = 273.157198087326

'set_liq_ice_air_eq_at_p 1E4
'liq_ice_air_temperature_si = 273.159302793328

'set_liq_ice_air_eq_at_t 273.155
'liq_ice_air_temperature_si = 273.155

'set_liq_ice_air_eq_at_wa_eta_wt 0.99, 0, 0.5
'liq_ice_air_temperature_si = 273.151724970113

'set_liq_ice_air_eq_at_wa_wl_wi .1, .2, .3
'liq_ice_air_temperature_si = 273.159992933078

liq_ice_air_temperature_si = ErrorReturn

If equi_liq_ice_air_done <> IsOK Then Exit Function

liq_ice_air_temperature_si = equi_liq_ice_air_t

End Function

'==========================================================================
Public Function set_liq_ice_air_eq_at_a(ByVal a_si As Double) As Double

'this function computes the water-ice-humid-air equilibrium at given dry-air fraction
'in humid air by Newton iteration

Const Tt = TP_temperature_si
Const Pt = TP_pressure_IAPWS95_si
Const dt_liq = TP_density_liq_IAPWS95_si  'triple point density of liquid water
Const dt_vap = TP_density_vap_IAPWS95_si  'triple point density of water vapour

Dim t As Double, p As Double, dl As Double, dh As Double, eps As Double
Dim wa As Double, wet As Double, eta As Double
Dim maxit As Long

If equi_liq_ice_air_done = IsOK And _
   a_si = equi_liq_ice_air_a Then
  'the requested state has already been computed earlier
  set_liq_ice_air_eq_at_a = IsOK
  Exit Function
End If

clear_liq_ice_air_state 'waste any previous state

set_liq_ice_air_eq_at_a = ErrorReturn

If a_si < 0 Then Exit Function
If a_si >= 1 Then Exit Function

'at this equilibrium, the mass fractions of dry air, wa, liquid, wet,
'and entropy, eta, are still ambiguous and must be specified by additional constraints
wa = ErrorReturn
wet = ErrorReturn
eta = ErrorReturn

If a_si = 0 Then  'pure water: set IAPWS-95 triple point
  set_liq_ice_air_state a_si, Tt, Pt, dt_vap, dt_liq, wa, wet, eta
  set_liq_ice_air_eq_at_a = IsOK
  Exit Function
End If

init_it_ctrl_liq_ice_air

'set initial temperature guess
Select Case ctrl_mode_temperature
  Case 0:  t = Tt
  Case -1: t = Tt
  Case 1:  t = ctrl_init_t
  Case Else: Exit Function
End Select
If t <= 0 Then Exit Function
If t = ErrorReturn Then Exit Function

'set initial pressure guess
Select Case ctrl_mode_pressure
  Case 0:  p = aux_pressure_si(a_si)
  Case -1: p = Pt
  Case 1:  p = ctrl_init_p
  Case Else: Exit Function
End Select
If p <= 0 Then Exit Function
If p = ErrorReturn Then Exit Function

'set initial liquid density guess
Select Case ctrl_mode_liquid
  Case 0:  dl = dt_liq
  Case -1: dl = dt_liq
  Case 1:  dl = ctrl_init_d_liq
  Case Else: Exit Function
End Select
If dl <= 0 Then Exit Function
If dl = ErrorReturn Then Exit Function

'set initial humid-air density guess
Select Case ctrl_mode_air
  Case 0:  dh = air_g_density_si(a_si, t, p)
  Case -1: dh = dt_vap
  Case 1:  dh = ctrl_init_d_air
  Case Else: Exit Function
End Select
If dh <= 0 Then Exit Function
If dh = ErrorReturn Then Exit Function

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: set_liq_ice_air_state a_si, t, p, dh, dl, wa, wet, eta
           set_liq_ice_air_eq_at_a = IsOK
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit
eps = ctrl_eps_exit_p
If eps = 0 Then Exit Function

'run iteration loop
If liq_ice_air_iter_at_a(a_si, maxit, eps, t, p, dh, dl) = ErrorReturn Then
  Exit Function
End If

set_liq_ice_air_state a_si, t, p, dh, dl, wa, wet, eta

set_liq_ice_air_eq_at_a = IsOK

End Function

'==========================================================================
Public Function set_liq_ice_air_eq_at_p(ByVal p_si As Double) As Double

'this function computes the water-ice-humid-air equilibrium at given pressure
' by Newton iteration

Const Tt = TP_temperature_si
Const Pt = TP_pressure_IAPWS95_si
Const dt_liq = TP_density_liq_IAPWS95_si  'triple point density of liquid water
Const dt_vap = TP_density_vap_IAPWS95_si  'triple point density of water vapour

Dim t As Double, a As Double, dl As Double, dh As Double, eps As Double
Dim wa As Double, wet As Double, eta As Double
Dim maxit As Long

If equi_liq_ice_air_done = IsOK And _
   p_si = equi_liq_ice_air_p Then
  'the requested state has already been computed earlier
  set_liq_ice_air_eq_at_p = IsOK
  Exit Function
End If

clear_liq_ice_air_state 'waste any previous state

set_liq_ice_air_eq_at_p = ErrorReturn

If p_si < Pt Then Exit Function

'at this equilibrium, the mass fractions of dry air, wa, liquid, wet,
'and entropy, eta, are still ambiguous and must be specified by additional constraints
wa = ErrorReturn
wet = ErrorReturn
eta = ErrorReturn

If p_si = Pt Then  'pure water: set IAPWS-95 triple point
  set_liq_ice_air_state 0, Tt, Pt, dt_vap, dt_liq, wa, wet, eta
  set_liq_ice_air_eq_at_p = IsOK
  Exit Function
End If

init_it_ctrl_liq_ice_air

'set initial temperature guess
Select Case ctrl_mode_temperature
  Case 0:  t = Tt
  Case -1: t = Tt
  Case 1:  t = ctrl_init_t
  Case Else: Exit Function
End Select
If t <= 0 Then Exit Function
If t = ErrorReturn Then Exit Function

a = air_massfraction_air_si(1 - Pt / p_si)
If a = ErrorReturn Then Exit Function

'set initial liquid density guess
Select Case ctrl_mode_liquid
  Case 0:  dl = dt_liq
  Case -1: dl = dt_liq
  Case 1:  dl = ctrl_init_d_liq
  Case Else: Exit Function
End Select
If dl <= 0 Then Exit Function
If dl = ErrorReturn Then Exit Function

'set initial humid-air density guess
Select Case ctrl_mode_air
  Case 0:  dh = air_g_density_si(a, t, p_si)
  Case -1: dh = dt_vap
  Case 1:  dh = ctrl_init_d_air
  Case Else: Exit Function
End Select
If dh <= 0 Then Exit Function
If dh = ErrorReturn Then Exit Function

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: set_liq_ice_air_state a, t, p_si, dh, dl, wa, wet, eta
           set_liq_ice_air_eq_at_p = IsOK
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit
eps = ctrl_eps_exit_p
If eps = 0 Then Exit Function

'run iteration loop
If liq_ice_air_iter_at_p(p_si, maxit, eps, a, t, dh, dl) = ErrorReturn Then
  Exit Function
End If

set_liq_ice_air_state a, t, p_si, dh, dl, wa, wet, eta

set_liq_ice_air_eq_at_p = IsOK

End Function

'==========================================================================
Public Function set_liq_ice_air_eq_at_t(ByVal t_si As Double) As Double

'this function computes the water-ice-humid-air equilibrium at given temperature
' by Newton iteration

Const Tt = TP_temperature_si
Const Pt = TP_pressure_IAPWS95_si
Const dt_liq = TP_density_liq_IAPWS95_si  'triple point density of liquid water
Const dt_vap = TP_density_vap_IAPWS95_si  'triple point density of water vapour

Dim p As Double, a As Double, dl As Double, dh As Double, eps As Double
Dim wa As Double, wet As Double, eta As Double
Dim maxit As Long

If equi_liq_ice_air_done = IsOK And _
   t_si = equi_liq_ice_air_t Then
  'the requested state has already been computed earlier
  set_liq_ice_air_eq_at_t = IsOK
  Exit Function
End If

clear_liq_ice_air_state 'waste any previous state

set_liq_ice_air_eq_at_t = ErrorReturn

If t_si > Tt Then Exit Function

'at this equilibrium, the mass fractions of dry air, wa, liquid, wet,
'and entropy, eta, are still ambiguous and must be specified by additional constraints
wa = ErrorReturn
wet = ErrorReturn
eta = ErrorReturn

If t_si = Tt Then  'pure water: set IAPWS-95 triple point
  set_liq_ice_air_state 0, Tt, Pt, dt_vap, dt_liq, wa, wet, eta
  set_liq_ice_air_eq_at_t = IsOK
  Exit Function
End If

init_it_ctrl_liq_ice_air

'set initial pressure guess
Select Case ctrl_mode_pressure
  Case 0:  p = ice_liq_meltingpressure_si(t_si)
  Case -1: p = Pt
  Case 1:  p = ctrl_init_p
  Case Else: Exit Function
End Select
If p <= 0 Then Exit Function
If p = ErrorReturn Then Exit Function

a = air_massfraction_air_si(1 - Pt / p)
If a = ErrorReturn Then Exit Function

'set initial liquid density guess
Select Case ctrl_mode_liquid
  Case 0:  dl = dt_liq
  Case -1: dl = dt_liq
  Case 1:  dl = ctrl_init_d_liq
  Case Else: Exit Function
End Select
If dl <= 0 Then Exit Function
If dl = ErrorReturn Then Exit Function

'set initial humid-air density guess
Select Case ctrl_mode_air
  Case 0:  dh = air_g_density_si(a, t_si, p)
  Case -1: dh = dt_vap
  Case 1:  dh = ctrl_init_d_air
  Case Else: Exit Function
End Select
If dh <= 0 Then Exit Function
If dh = ErrorReturn Then Exit Function

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: set_liq_ice_air_state a, t_si, p, dh, dl, wa, wet, eta
           set_liq_ice_air_eq_at_t = IsOK
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit
eps = ctrl_eps_exit_p
If eps = 0 Then Exit Function

'run iteration loop
If liq_ice_air_iter_at_t(t_si, maxit, eps, a, p, dh, dl) = ErrorReturn Then
  Exit Function
End If

set_liq_ice_air_state a, t_si, p, dh, dl, wa, wet, eta

set_liq_ice_air_eq_at_t = IsOK

End Function

'==========================================================================
Public Function set_liq_ice_air_eq_at_wa_eta_wt(ByVal wa_si As Double, _
                                                ByVal eta_si As Double, _
                                                ByVal wet_si As Double) As Double

'this function computes the water-ice-humid-air equilibrium at given air fraction wa_si,
'entropy eta_si, and the liquid fraction, wet_si, of the condensed part by Newton iteration.
'In particular, at wet_si = 0, the isentropic melting level IML is computed,
'and at wet_si = 1, the isentropic freezing level IFL

Const Tt = TP_temperature_si
Const Pt = TP_pressure_IAPWS95_si
Const dt_liq = TP_density_liq_IAPWS95_si  'triple point density of liquid water
Const dt_vap = TP_density_vap_IAPWS95_si  'triple point density of water vapour

Dim a As Double, t As Double, p As Double, dl As Double, dh As Double, eps As Double
Dim maxit As Long

If equi_liq_ice_air_done = IsOK And _
   wa_si = equi_liq_ice_air_wa And _
   eta_si = equi_liq_ice_air_eta And _
   wet_si = equi_liq_ice_air_wet Then
  'the requested state has already been computed earlier
  set_liq_ice_air_eq_at_wa_eta_wt = IsOK
  Exit Function
End If

clear_liq_ice_air_state 'waste any previous state

set_liq_ice_air_eq_at_wa_eta_wt = ErrorReturn

If wa_si < 0 Or wa_si >= 1 Then Exit Function

If wa_si = 0 Then  'pure water: set IAPWS-95 triple point
  set_liq_ice_air_state 0, Tt, Pt, dt_vap, dt_liq, wa_si, wet_si, eta_si
  set_liq_ice_air_eq_at_wa_eta_wt = IsOK
  Exit Function
End If

init_it_ctrl_liq_ice_air

'set initial mass fractions guess
Select Case ctrl_mode_humidity
  Case 0:  a = aux_airfraction_si(wa_si)
  Case -1: a = 0.999
  Case 1:  a = ctrl_init_a
  Case Else: Exit Function
End Select
If a < 0 Or a > 1 Then Exit Function
If a = ErrorReturn Then Exit Function

'set initial temperature guess
Select Case ctrl_mode_temperature
  Case 0:  t = Tt
  Case -1: t = Tt
  Case 1:  t = ctrl_init_t
  Case Else: Exit Function
End Select
If t <= 0 Then Exit Function
If t = ErrorReturn Then Exit Function

'set initial pressure guess
Select Case ctrl_mode_pressure
  Case 0:  p = aux_pressure_si(a)
  Case -1: p = Pt
  Case 1:  p = ctrl_init_p
  Case Else: Exit Function
End Select
If p <= 0 Then Exit Function
If p = ErrorReturn Then Exit Function

'set initial liquid density guess
Select Case ctrl_mode_liquid
  Case 0:  dl = dt_liq
  Case -1: dl = dt_liq
  Case 1:  dl = ctrl_init_d_liq
  Case Else: Exit Function
End Select
If dl <= 0 Then Exit Function
If dl = ErrorReturn Then Exit Function

'set initial humid-air density guess
Select Case ctrl_mode_air
  Case 0:  dh = air_g_density_si(a, t, p)
  Case -1: dh = dt_vap
  Case 1:  dh = ctrl_init_d_air
  Case Else: Exit Function
End Select
If dh <= 0 Then Exit Function
If dh = ErrorReturn Then Exit Function

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: set_liq_ice_air_state a, t, p, dh, dl, wa_si, wet_si, eta_si
           set_liq_ice_air_eq_at_wa_eta_wt = IsOK
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit
eps = ctrl_eps_exit_p
If eps = 0 Then Exit Function

'run iteration loop
If liq_ice_air_iter_at_wa_eta_wet(wa_si, eta_si, wet_si, maxit, eps, a, t, p, dh, dl) = ErrorReturn Then
  Exit Function
End If

set_liq_ice_air_state a, t, p, dh, dl, wa_si, wet_si, eta_si

set_liq_ice_air_eq_at_wa_eta_wt = IsOK

End Function

'==========================================================================
Public Function set_liq_ice_air_eq_at_wa_wl_wi(ByVal wa_si As Double, _
                                               ByVal wl_si As Double, _
                                               ByVal wi_si As Double) As Double

'this function computes by Newton interation the water-ice-humid-air equilibrium
'at given air fraction wa_si (0 <= wa_si < 1) of air in kg/kg,
'given liquid fraction wl_si (0 <= wl_si < 1) in kg/kg,
'and the ice fraction wi_si (0 <= wi_si < 1) in kg/kg.
'Note that wa + wl + wi < 1 must hold for a vapour fraction wv > 0


Dim t As Double, p As Double, a As Double, S As Double, wet As Double
Dim sh As Double, si As Double, sl As Double
Dim dh As Double, dl As Double

set_liq_ice_air_eq_at_wa_wl_wi = ErrorReturn

If wa_si < 0 Or wa_si >= 1 Then Exit Function
If wl_si < 0 Or wl_si >= 1 Then Exit Function
If wi_si < 0 Or wi_si >= 1 Then Exit Function
If wa_si + wl_si + wi_si >= 1 Then Exit Function
If wl_si + wi_si = 0 Then Exit Function

a = wa_si / (1 - wl_si - wi_si)  'air fraction in humid air
wet = wl_si / (wl_si + wi_si)    'liquid fraction of the condensed part

'iteratively compute equilibrium temperature and pressure
If set_liq_ice_air_eq_at_a(a) = ErrorReturn Then Exit Function

p = liq_ice_air_pressure_si
If p = ErrorReturn Then Exit Function
t = liq_ice_air_temperature_si
If t = ErrorReturn Then Exit Function

'compute entropies of the components
'humid air:
dh = equi_liq_ice_air_d_air  '= air_g_density_si(a, t, p)
If dh = ErrorReturn Then Exit Function
sh = air_f_entropy_si(a, t, dh)
If sh = ErrorReturn Then Exit Function

'ice:
si = ice_entropy_si(t, p)
If si = ErrorReturn Then Exit Function

'liquid water
dl = equi_liq_ice_air_d_liq  'liq_density_si(t, p)
If dl = ErrorReturn Then Exit Function
sl = flu_entropy_si(t, dl)
If sl = ErrorReturn Then Exit Function

'total entropy
S = (1 - wl_si - wi_si) * sh + wl_si * sl + wi_si * si

'store the computed equilibrium values
set_liq_ice_air_state a, t, p, dh, dl, wa_si, wet, S

set_liq_ice_air_eq_at_wa_wl_wi = IsOK

End Function

'==========================================================================
Public Sub set_it_ctrl_liq_ice_air(ByVal key As String, ByVal value As Double)

'this sub sets control parameters for the iteration used to compute
'wet ice air properties

'key              value
'it_steps         0           set iteration number to default (100)
'it_steps         n > 0       set iteration number to n
'it_steps        -1           do not iterate, use initial values

'init_air         0           use default air density to start ( = rho(ctrl_init_p, ctrl_init_t))
'init_air         d > 0       use value d as air density to start

'init_liq         0           use default liquid density to start ( = rho(ctrl_init_p, ctrl_init_t))
'init_liq         d > 0       use value d as liquid density to start

'init_hum         0           use default air fraction to start ( = aux_airfraction_si(wa_si))
'init_hum         a > 0       use value a as air fraction to start

'init_temp        0           use default temperature to start ( = triple point temperature)
'init_temp        t > 0       use value t as temperature to start

'init_press       0           use default pressure to start ( = aux_pressure_si(wa_si))
'init_press       p > 0       use value p as pressure to start

'tol_temp         0           use default exit accuracy for temperature (0.1 ppm)
'tol_temp         eps         use eps as exit accuracy for temperature (eps < 0 means relative error)

'tol_press        0           use default exit accuracy for pressure (0.1 ppm)
'tol_press        eps         use eps as exit accuracy for pressure (eps < 0 means relative error)

init_it_ctrl_liq_ice_air

clear_liq_ice_air_state

Select Case LCase(Trim(key))

  Case "it_steps":   'iteration steps
    Select Case value
      Case 0:      ctrl_loop_maximum = 100  'default = 100
      Case Is < 0: ctrl_loop_maximum = -1
      Case Else:   ctrl_loop_maximum = value
    End Select

  Case "init_air":   'start air density
    Select Case CLng(value)
      Case 0:       ctrl_mode_air = 0    'default = rho(ctrl_init_p, ctrl_init_t)
      Case Else:    ctrl_mode_air = 1
                    ctrl_init_d_air = value
    End Select
    
  Case "init_liq":   'start liquid water density
    Select Case CLng(value)
      Case 0:       ctrl_mode_liquid = 0    'default = rho(ctrl_init_p, ctrl_init_t)
      Case Else:    ctrl_mode_liquid = 1
                    ctrl_init_d_liq = value
    End Select

  Case "init_hum":   'start air fraction
    Select Case CLng(value)
      Case 0:       ctrl_mode_humidity = 0    'default = aux function
      Case Else:    ctrl_mode_humidity = 1
                    ctrl_init_a = value
    End Select
    
  Case "init_temp":   'start temperature
    Select Case CLng(value)
      Case 0:       ctrl_mode_temperature = 0    'default = triple point temperature
      Case Else:    ctrl_mode_temperature = 1
                    ctrl_init_t = value
    End Select

  Case "init_press":   'start pressure
    Select Case CLng(value)
      Case 0:       ctrl_mode_pressure = 0    'default = aux function
      Case Else:    ctrl_mode_pressure = 1
                    ctrl_init_p = value
    End Select

  Case "tol_temp":      'required temperature tolerance
    Select Case value
      Case 0:       ctrl_eps_exit_t = -0.0000001   'default = 0.1 ppm relative
      Case Else:    ctrl_eps_exit_t = value
    End Select

  Case "tol_press":      'required total pressure tolerance
    Select Case value
      Case 0:       ctrl_eps_exit_p = -0.0000001   'default = 0.1 ppm relative
      Case Else:    ctrl_eps_exit_p = value
    End Select

End Select


End Sub

'==========================================================================
Private Sub clear_liq_ice_air_state()

'clears the current equilibrium state descriptor

equi_liq_ice_air_done = 0

End Sub

'==========================================================================
Private Sub set_liq_ice_air_state(ByVal a_si As Double, _
                                  ByVal t_si As Double, _
                                  ByVal p_si As Double, _
                                  ByVal d_air_si As Double, _
                                  ByVal d_liq_si As Double, _
                                  ByVal wa_si As Double, _
                                  ByVal wet_si As Double, _
                                  ByVal eta_si As Double)
                                  
'stores the actual properties as the current equilibrium state descriptor

equi_liq_ice_air_done = IsOK

'intensive properties, well defined at the triple point
equi_liq_ice_air_a = a_si          'triple-point air fraction
equi_liq_ice_air_t = t_si          'triple-point temperature
equi_liq_ice_air_p = p_si          'triple-point pressure
equi_liq_ice_air_d_liq = d_liq_si  'density of pure water
equi_liq_ice_air_d_air = d_air_si  'density of humid air

'"extensive properties", depend on additional constraints
'and may be undefined (i.e. = ErrorReturn) in certain computed states

equi_liq_ice_air_wa = wa_si     'dry-air mass fraction in kg/kg of wet ice air
equi_liq_ice_air_wet = wet_si   'liquid-water mass fraction in kg/kg of the condensed part
equi_liq_ice_air_eta = eta_si   'total specific entropy of the parcel

End Sub

'==========================================================================
Private Sub init_it_ctrl_liq_ice_air()

Const Tt = TP_temperature_si
Const Pt = TP_pressure_IAPWS95_si
Const dlt = TP_density_liq_IAPWS95_si  'triple point density of liquid water
Const dvt = TP_density_vap_IAPWS95_si  'triple point density of water vapour

If ctrl_initialized = IsOK Then Exit Sub

ctrl_initialized = IsOK

'Set default values and modes for iteration
ctrl_mode_humidity = 0
ctrl_mode_liquid = 0
ctrl_mode_air = 0
ctrl_mode_temperature = 0
ctrl_mode_pressure = 0
ctrl_loop_maximum = 100

ctrl_init_a = 0.999
ctrl_init_t = Tt
ctrl_init_p = Pt
ctrl_init_d_liq = dlt
ctrl_init_d_air = dvt

ctrl_eps_exit_t = -0.0000001 'relative, 0.1 ppm
ctrl_eps_exit_p = -0.0000001 'relative, 0.1 ppm

End Sub

'==========================================================================
Private Function aux_airfraction_si(ByVal wa_si As Double) As Double

'function returns the estimated dry-air fraction a in kg/kg of humid air
'at given dry-air fraction wa in kg/kg of wet ice air

'a = wa /(wa + wv)                   'dry-air fraction of humid air, wv = vapour fraction
'wl = wet / (1 - wa / a)             'liquid fraction of wet ice air
'wi = (1 - wet) / (1 - wa / a)       'ice fraction of wet ice air
'wa + wv + wl + wi = 1               'mass balance: air + vapour + liquid + ice = 100%

Dim wv As Double

aux_airfraction_si = ErrorReturn

 'wi+wl=wv, rough guess for the vapour fraction wv, 50% of the water may be vapour
wv = 0.5 * (1 - wa_si)
If wa_si + wv <= 0 Then Exit Function

aux_airfraction_si = wa_si / (wa_si + wv)

End Function
'==========================================================================
Private Function aux_pressure_si(ByVal a_si As Double) As Double

'function returns the estimated triple-point pressure in Pa at given dry-air fraction a_si
'in kg/kg of the humid-air part
Const Pt = TP_pressure_IAPWS95_si

Dim xv As Double

aux_pressure_si = ErrorReturn

'assume the partial vapour pressure = triple-point pressure
xv = air_molfraction_vap_si(a_si)
If xv = ErrorReturn Then Exit Function
If xv <= 0 Or xv > 1 Then Exit Function

aux_pressure_si = Pt / xv

End Function

'==========================================================================
Private Function liq_ice_air_iter_at_a(ByVal a_si As Double, _
                                       ByVal maxit As Long, _
                                       ByVal eps As Double, _
                                       ByRef t_si As Double, _
                                       ByRef p_si As Double, _
                                       ByRef d_air_si As Double, _
                                       ByRef d_liq_si As Double) As Double

'this function returns the water-ice-humid-air phase equilibrium from
'equal chemical potentials of water at given air fraction, a_si,
'from an initial guesses for the temperature, t_si, pressure, p_si,
'the humid-air density, d_air_si and the liquid density, d_liq_si.
'The iteration limit eps refers to the error in pressure.

'output:    liq_ice_air_iter_at_a = IsOK if successfully done
'           liq_ice_air_iter_at_a = ErrorReturn is returned if
'           - the maximum number of iterations is exceeded without meeting the exit criterion
'           - the function call to a Gibbs/Helmholtz function has returned an error
'           - density, temperature or pressure have taken a zero or negative value during the iteration
'           - a-t-p values do not permit the stable existence of ice or water
'     t_si: temperature in K of the water-ice-air equilibrium
'     p_si: pressure in Pa of the water-ice-air equilibrium
' d_air_si: density of humid air at (a_si, t_si, p_si)
' d_liq_si: density of liquid water at (t_si, p_si)

'input:  a_si: air fraction of dry air in humid air in kg/kg
'       maxit: maximum number of iteration steps to be done
'         eps: required accuracy of pressure
'              eps > 0: absolute pressure tolerance in Pa
'              eps < 0: relative pressure tolerance
'        t_si: initial guess for absolute temperature in K
'        p_si: initial guess for absolute pressure in Pa
'    d_air_si: initial guess for density of humid air at (a_si, t_si, p_si)
'    d_liq_si: initial guess for density of liquid at (t_si, p_si)

Dim dh As Double, dl As Double, t As Double, p As Double
Dim ddh As Double, ddl As Double, dt As Double, dp As Double
Dim gi As Double, gi_t As Double, gi_p As Double
Dim fh As Double, fh_a As Double, fh_t As Double, fh_td As Double
Dim fh_d As Double, fh_dd As Double, fh_at As Double, fh_ad As Double
Dim fl As Double, fl_t As Double, fl_td As Double
Dim fl_d As Double, fl_dd As Double

Dim it As Long

Dim a(4, 4) As Double, b(4) As Double, x(4) As Double

liq_ice_air_iter_at_a = ErrorReturn

If a_si <= 0 Or a_si >= 1 Or _
   d_air_si <= 0 Or _
   d_liq_si <= 0 Or _
   t_si <= 0 Or _
   p_si <= 0 Then
  t_si = ErrorReturn
  p_si = ErrorReturn
  d_air_si = ErrorReturn
  d_liq_si = ErrorReturn
  Exit Function
End If

check_limits = check_limits - 1
 
t = t_si
p = p_si
dl = d_liq_si
dh = d_air_si

For it = 1 To maxit

  'Derivatives of the Helmholtz function of humid air for 4D-Newton iteration
  fh = air_f_si(0, 0, 0, a_si, t, dh)
  If fh = ErrorReturn Then Exit For
  fh_a = air_f_si(1, 0, 0, a_si, t, dh)
  If fh_a = ErrorReturn Then Exit For
  fh_t = air_f_si(0, 1, 0, a_si, t, dh)
  If fh_t = ErrorReturn Then Exit For
  fh_d = air_f_si(0, 0, 1, a_si, t, dh)
  If fh_d = ErrorReturn Then Exit For
  fh_at = air_f_si(1, 1, 0, a_si, t, dh)
  If fh_at = ErrorReturn Then Exit For
  fh_ad = air_f_si(1, 0, 1, a_si, t, dh)
  If fh_ad = ErrorReturn Then Exit For
  fh_td = air_f_si(0, 1, 1, a_si, t, dh)
  If fh_td = ErrorReturn Then Exit For
  fh_dd = air_f_si(0, 0, 2, a_si, t, dh)
  If fh_dd = ErrorReturn Then Exit For

  'Derivatives of the Helmholtz function of liquid water for Newton iteration
  fl = flu_f_si(0, 0, t, dl)
  If fl = ErrorReturn Then Exit For
  fl_t = flu_f_si(1, 0, t, dl)
  If fl_t = ErrorReturn Then Exit For
  fl_d = flu_f_si(0, 1, t, dl)
  If fl_d = ErrorReturn Then Exit For
  fl_td = flu_f_si(1, 1, t, dl)
  If fl_td = ErrorReturn Then Exit For
  fl_dd = flu_f_si(0, 2, t, dl)
  If fl_dd = ErrorReturn Then Exit For

  'Derivatives of the Gibbs function of ice for Newton iteration
  gi = ice_g_si(0, 0, t, p)
  If gi = ErrorReturn Then Exit For
  gi_t = ice_g_si(1, 0, t, p)
  If gi_t = ErrorReturn Then Exit For
  gi_p = ice_g_si(0, 1, t, p)
  If gi_p = ErrorReturn Then Exit For

  'coefficient matrix
  a(1, 1) = gi_t - fl_t
  a(1, 2) = gi_p - 1 / dl
  a(1, 3) = p / dl ^ 2 - fl_d
  a(1, 4) = 0
  a(2, 1) = -dl * fl_td
  a(2, 2) = 1 / dl
  a(2, 3) = -fl_d - dl * fl_dd - p / dl ^ 2
  a(2, 4) = 0
  a(3, 1) = gi_t - fh_t + a_si * fh_at
  a(3, 2) = gi_p - 1 / dh
  a(3, 3) = 0
  a(3, 4) = p / dh ^ 2 + a_si * fh_ad - fh_d
  a(4, 1) = -dh * fh_td
  a(4, 2) = 1 / dh
  a(4, 3) = 0
  a(4, 4) = -p / dh ^ 2 - fh_d - dh * fh_dd

  'right-hand sides, must vanish at equilibrium
  b(1) = fl + p / dl - gi
  b(2) = dl * fl_d - p / dl
  b(3) = -gi + fh + p / dh - a_si * fh_a
  b(4) = dh * fh_d - p / dh
  
  'solve equations
  If matrix_solve(a(), b(), x(), 4) <> 0 Then Exit For 'matrix singular
  dt = x(1)
  dp = x(2)
  ddl = x(3)
  ddh = x(4)
  
  'update unknowns
  dh = dh + ddh
  If dh <= 0 Then Exit For
  dl = dl + ddl
  If dl <= 0 Then Exit For
  t = t + dt
  If t <= 0 Then Exit For
  p = p + dp
  If p <= 0 Then Exit For
  
  'check absolute or relative error limit
  If (eps > 0 And Abs(dp) < eps) Or _
     (eps < 0 And Abs(dp) < -eps * Abs(p)) Then
    liq_ice_air_iter_at_a = IsOK
    t_si = t
    p_si = p
    d_air_si = dh
    d_liq_si = dl
    Exit For
  End If

Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Or _
     d_liq_si <= flu_dmin Or d_liq_si > flu_dmax Then
    t_si = ErrorReturn
    p_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_air_si = ErrorReturn
    liq_ice_air_iter_at_a = ErrorReturn
    Exit Function
  End If
  'ICE_LIMITS
  If t_si <= ice_tmin Or t_si > ice_tmax Or _
     p_si <= ice_pmin Or p_si > ice_pmax Then
    t_si = ErrorReturn
    p_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_air_si = ErrorReturn
    liq_ice_air_iter_at_a = ErrorReturn
    Exit Function
  End If
  'AIR_LIMITS
  If t_si < dry_air_tmin Or t_si > dry_air_tmax Or _
     d_air_si <= dry_air_dmin Or d_air_si > dry_air_dmax Then
    t_si = ErrorReturn
    p_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_air_si = ErrorReturn
    liq_ice_air_iter_at_a = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Function liq_ice_air_iter_at_p(ByVal p_si As Double, _
                                       ByVal maxit As Long, _
                                       ByVal eps As Double, _
                                       ByRef a_si As Double, _
                                       ByRef t_si As Double, _
                                       ByRef d_air_si As Double, _
                                       ByRef d_liq_si As Double) As Double

'this function returns the water-ice-humid-air phase equilibrium from
'equal chemical potentials of water at given pressure, p_si,
'from an initial guesses for the temperature, t_si, air fraction, a_si,
'the humid-air density, d_air_si and the liquid density, d_liq_si.
'The iteration limit eps refers to the error in air fraction.

'output:    liq_ice_air_iter_at_p = IsOK if successfully done
'           liq_ice_air_iter_at_p = ErrorReturn is returned if
'           - the maximum number of iterations is exceeded without meeting the exit criterion
'           - the function call to a Gibbs/Helmholtz function has returned an error
'           - density, temperature or air fraction have taken an invalid value during the iteration
'           - a-t-p values do not permit the stable existence of ice or water
'     t_si: temperature in K of the water-ice-air equilibrium
'     a_si: air fraction of dry air in humid air in kg/kg
' d_air_si: density of humid air at (a_si, t_si, p_si)
' d_liq_si: density of liquid water at (t_si, p_si)

'input:  p_si: absolute pressure in Pa
'       maxit: maximum number of iteration steps to be done
'         eps: required accuracy of air fraction
'              eps > 0: air fraction tolerance in kg/kg
'              eps < 0: relative air fraction tolerance
'        a_si: air fraction of dry air in humid air in kg/kg
'        t_si: initial guess for absolute temperature in K
'    d_air_si: initial guess for density of humid air at (a_si, t_si, p_si)
'    d_liq_si: initial guess for density of liquid at (t_si, p_si)

Dim dh As Double, dl As Double, t As Double
Dim ddh As Double, ddl As Double, dt As Double, dp As Double
Dim gi As Double, gi_t As Double, a As Double
Dim fh As Double, fh_a As Double, fh_t As Double, fh_td As Double
Dim fh_d As Double, fh_dd As Double, fh_at As Double, fh_ad As Double
Dim fl As Double, fl_t As Double, fl_td As Double, fh_aa As Double
Dim fl_d As Double, fl_dd As Double

Dim it As Long

Dim aa(4, 4) As Double, b(4) As Double, x(4) As Double

liq_ice_air_iter_at_p = ErrorReturn

If d_air_si <= 0 Then Exit Function
If d_liq_si <= 0 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function
If a_si <= 0 Then Exit Function
If a_si >= 1 Then Exit Function

If d_air_si <= 0 Or _
   d_liq_si <= 0 Or _
   t_si <= 0 Or _
   a_si <= 0 Or a_si >= 1 Then
  t_si = ErrorReturn
  a_si = ErrorReturn
  d_air_si = ErrorReturn
  d_liq_si = ErrorReturn
  Exit Function
End If

If check_limits <> 1 Then
  If p_si <= 0 Then
    t_si = ErrorReturn
    a_si = ErrorReturn
    d_air_si = ErrorReturn
    d_liq_si = ErrorReturn
    Exit Function
  End If
Else
  'ICE_LIMITS
  If p_si <= ice_pmin Or p_si > ice_pmax Then
    t_si = ErrorReturn
    a_si = ErrorReturn
    d_air_si = ErrorReturn
    d_liq_si = ErrorReturn
    Exit Function
  End If
End If

check_limits = check_limits - 1

t = t_si
a = a_si
dl = d_liq_si
dh = d_air_si

For it = 1 To maxit

  'Derivatives of the Helmholtz function of humid air for 4D-Newton iteration
  fh = air_f_si(0, 0, 0, a, t, dh)
  If fh = ErrorReturn Then Exit For
  fh_a = air_f_si(1, 0, 0, a, t, dh)
  If fh_a = ErrorReturn Then Exit For
  fh_t = air_f_si(0, 1, 0, a, t, dh)
  If fh_t = ErrorReturn Then Exit For
  fh_d = air_f_si(0, 0, 1, a, t, dh)
  If fh_d = ErrorReturn Then Exit For
  fh_aa = air_f_si(2, 0, 0, a, t, dh)
  If fh_aa = ErrorReturn Then Exit For
  fh_at = air_f_si(1, 1, 0, a, t, dh)
  If fh_at = ErrorReturn Then Exit For
  fh_ad = air_f_si(1, 0, 1, a, t, dh)
  If fh_ad = ErrorReturn Then Exit For
  fh_td = air_f_si(0, 1, 1, a, t, dh)
  If fh_td = ErrorReturn Then Exit For
  fh_dd = air_f_si(0, 0, 2, a, t, dh)
  If fh_dd = ErrorReturn Then Exit For

  'Derivatives of the Helmholtz function of liquid water for Newton iteration
  fl = flu_f_si(0, 0, t, dl)
  If fl = ErrorReturn Then Exit For
  fl_t = flu_f_si(1, 0, t, dl)
  If fl_t = ErrorReturn Then Exit For
  fl_d = flu_f_si(0, 1, t, dl)
  If fl_d = ErrorReturn Then Exit For
  fl_td = flu_f_si(1, 1, t, dl)
  If fl_td = ErrorReturn Then Exit For
  fl_dd = flu_f_si(0, 2, t, dl)
  If fl_dd = ErrorReturn Then Exit For

  'Derivatives of the Gibbs function of ice for Newton iteration
  gi = ice_g_si(0, 0, t, p_si)
  If gi = ErrorReturn Then Exit For
  gi_t = ice_g_si(1, 0, t, p_si)
  If gi_t = ErrorReturn Then Exit For

  'coefficient matrix
  aa(1, 1) = gi_t - fl_t
  aa(1, 2) = p_si / dl ^ 2 - fl_d
  aa(1, 3) = 0
  aa(1, 4) = 0
  
  aa(2, 1) = -dl * fl_td
  aa(2, 2) = -fl_d - dl * fl_dd - p_si / dl ^ 2
  aa(2, 3) = 0
  aa(2, 4) = 0
  
  aa(3, 1) = gi_t - fh_t + a * fh_at
  aa(3, 2) = 0
  aa(3, 3) = p_si / dh ^ 2 + a * fh_ad - fh_d
  aa(3, 4) = a * fh_aa
  
  aa(4, 1) = -dh * fh_td
  aa(4, 2) = 0
  aa(4, 3) = -p_si / dh ^ 2 - fh_d - dh * fh_dd
  aa(4, 4) = -dh * fh_ad

  'right-hand sides, must vanish at equilibrium
  b(1) = fl + p_si / dl - gi
  b(2) = dl * fl_d - p_si / dl
  b(3) = -gi + fh + p_si / dh - a * fh_a
  b(4) = dh * fh_d - p_si / dh

  'solve equations
  If matrix_solve(aa(), b(), x(), 4) <> 0 Then Exit For 'matrix singular
  
  'update unknowns
  t = t + x(1)
  If t <= 0 Then Exit For
  dl = dl + x(2)
  If dl <= 0 Then Exit For
  dh = dh + x(3)
  If dh <= 0 Then Exit For
  a = a + x(4)
  If a <= 0 Or a >= 1 Then Exit For

  'check absolute or relative error limit
  If (eps > 0 And Abs(x(4)) < eps) Or _
     (eps < 0 And Abs(x(4)) < -eps * a) Then
    liq_ice_air_iter_at_p = IsOK
    a_si = a
    t_si = t
    d_air_si = dh
    d_liq_si = dl
    Exit For
  End If

Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Or _
     d_liq_si <= flu_dmin Or d_liq_si > flu_dmax Then
    t_si = ErrorReturn
    a_si = ErrorReturn
    d_air_si = ErrorReturn
    d_liq_si = ErrorReturn
    liq_ice_air_iter_at_p = ErrorReturn
    Exit Function
  End If
  'ICE_LIMITS
  If t_si <= ice_tmin Or t_si > ice_tmax Then
    t_si = ErrorReturn
    a_si = ErrorReturn
    d_air_si = ErrorReturn
    d_liq_si = ErrorReturn
    liq_ice_air_iter_at_p = ErrorReturn
    Exit Function
  End If
  'AIR_LIMITS
  If t_si < dry_air_tmin Or t_si > dry_air_tmax Or _
     a_si < 0 Or a_si > 1 Or _
     d_air_si <= dry_air_dmin Or d_air_si > dry_air_dmax Then
    t_si = ErrorReturn
    a_si = ErrorReturn
    d_air_si = ErrorReturn
    d_liq_si = ErrorReturn
    liq_ice_air_iter_at_p = ErrorReturn
    Exit Function
  End If
End If

End Function

'==========================================================================
Private Function liq_ice_air_iter_at_t(ByVal t_si As Double, _
                                       ByVal maxit As Long, _
                                       ByVal eps As Double, _
                                       ByRef a_si As Double, _
                                       ByRef p_si As Double, _
                                       ByRef d_air_si As Double, _
                                       ByRef d_liq_si As Double) As Double

'this function returns the water-ice-humid-air phase equilibrium from
'equal chemical potentials of water at given temperature, t_si,
'from an initial guesses for the pressure, p_si, air fraction, a_si,
'the humid-air density, d_air_si and the liquid density, d_liq_si.
'The iteration limit eps refers to the error in air fraction.

'output:    liq_ice_air_iter_at_t = IsOK if successfully done
'           liq_ice_air_iter_at_t = ErrorReturn is returned if
'           - the maximum number of iterations is exceeded without meeting the exit criterion
'           - the function call to a Gibbs/Helmholtz function has returned an error
'           - density, pressure or air fraction have taken an invalid value during the iteration
'           - a-t-p values do not permit the stable existence of ice or water
'     p_si: pressure in Pa of the water-ice-air equilibrium
'     a_si: air fraction of dry air in humid air in kg/kg
' d_air_si: density of humid air at (a_si, t_si, p_si)
' d_liq_si: density of liquid water at (t_si, p_si)

'input:  t_si: absolute temperature in K
'       maxit: maximum number of iteration steps to be done
'         eps: required accuracy of pressure
'              eps > 0: absolute pressure tolerance in Pa
'              eps < 0: relative pressure tolerance
'        a_si: air fraction of dry air in humid air in kg/kg
'        p_si: initial guess for absolute pressure in Pa
'    d_air_si: initial guess for density of humid air at (a_si, t_si, p_si)
'    d_liq_si: initial guess for density of liquid at (t_si, p_si)

Dim dh As Double, dl As Double, p As Double
Dim ddh As Double, ddl As Double, dt As Double, dp As Double
Dim gi As Double, gi_p As Double, a As Double
Dim fh As Double, fh_a As Double, fh_td As Double
Dim fh_d As Double, fh_dd As Double, fh_ad As Double
Dim fl As Double, fh_aa As Double
Dim fl_d As Double, fl_dd As Double

Dim it As Long

Dim aa(4, 4) As Double, b(4) As Double, x(4) As Double

liq_ice_air_iter_at_t = ErrorReturn

If d_air_si <= 0 Or _
   d_liq_si <= 0 Or _
   p_si <= 0 Or _
   a_si <= 0 Or a_si >= 1 Then
  p_si = ErrorReturn
  a_si = ErrorReturn
  d_air_si = ErrorReturn
  d_liq_si = ErrorReturn
  Exit Function
End If

If check_limits <> 1 Then
  If t_si <= 0 Then
    p_si = ErrorReturn
    a_si = ErrorReturn
    d_air_si = ErrorReturn
    d_liq_si = ErrorReturn
    Exit Function
  End If
Else
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Then
    p_si = ErrorReturn
    a_si = ErrorReturn
    d_air_si = ErrorReturn
    d_liq_si = ErrorReturn
    Exit Function
  End If
  'ICE_LIMITS
  If t_si <= ice_tmin Or t_si > ice_tmax Then
    p_si = ErrorReturn
    a_si = ErrorReturn
    d_air_si = ErrorReturn
    d_liq_si = ErrorReturn
    Exit Function
  End If
  'AIR_LIMITS
  If t_si < dry_air_tmin Or t_si > dry_air_tmax Then
    p_si = ErrorReturn
    a_si = ErrorReturn
    d_air_si = ErrorReturn
    d_liq_si = ErrorReturn
    Exit Function
  End If
End If

check_limits = check_limits - 1

p = p_si
a = a_si
dl = d_liq_si
dh = d_air_si

For it = 1 To maxit

  'Derivatives of the Helmholtz function of humid air for 4D-Newton iteration
  fh = air_f_si(0, 0, 0, a, t_si, dh)
  If fh = ErrorReturn Then Exit For
  fh_a = air_f_si(1, 0, 0, a, t_si, dh)
  If fh_a = ErrorReturn Then Exit For
  fh_d = air_f_si(0, 0, 1, a, t_si, dh)
  If fh_d = ErrorReturn Then Exit For
  fh_aa = air_f_si(2, 0, 0, a, t_si, dh)
  If fh_aa = ErrorReturn Then Exit For
  fh_ad = air_f_si(1, 0, 1, a, t_si, dh)
  If fh_ad = ErrorReturn Then Exit For
  fh_dd = air_f_si(0, 0, 2, a, t_si, dh)
  If fh_dd = ErrorReturn Then Exit For

  'Derivatives of the Helmholtz function of liquid water for Newton iteration
  fl = flu_f_si(0, 0, t_si, dl)
  If fl = ErrorReturn Then Exit For
  fl_d = flu_f_si(0, 1, t_si, dl)
  If fl_d = ErrorReturn Then Exit For
  fl_dd = flu_f_si(0, 2, t_si, dl)
  If fl_dd = ErrorReturn Then Exit For

  'Derivatives of the Gibbs function of ice for Newton iteration
  gi = ice_g_si(0, 0, t_si, p)
  If gi = ErrorReturn Then Exit For
  gi_p = ice_g_si(0, 1, t_si, p)
  If gi_p = ErrorReturn Then Exit For

  'coefficient matrix
  aa(1, 1) = gi_p - 1 / dl
  aa(1, 2) = p / dl ^ 2 - fl_d
  aa(1, 3) = 0
  aa(1, 4) = 0
  
  aa(2, 1) = 1 / dl
  aa(2, 2) = -fl_d - dl * fl_dd - p / dl ^ 2
  aa(2, 3) = 0
  aa(2, 4) = 0
  
  aa(3, 1) = gi_p - 1 / dh
  aa(3, 2) = 0
  aa(3, 3) = p / dh ^ 2 + a * fh_ad - fh_d
  aa(3, 4) = a * fh_aa
  
  aa(4, 1) = 1 / dh
  aa(4, 2) = 0
  aa(4, 3) = -p / dh ^ 2 - fh_d - dh * fh_dd
  aa(4, 4) = -dh * fh_ad

  'right-hand sides, must vanish at equilibrium
  b(1) = fl + p / dl - gi
  b(2) = dl * fl_d - p / dl
  b(3) = -gi + fh + p / dh - a * fh_a
  b(4) = dh * fh_d - p / dh

  'solve equations
  If matrix_solve(aa(), b(), x(), 4) <> 0 Then Exit For 'matrix singular
  
  'update unknowns
  p = p + x(1)
  If p <= 0 Then Exit For
  dl = dl + x(2)
  If dl <= 0 Then Exit For
  dh = dh + x(3)
  If dh <= 0 Then Exit For
  a = a + x(4)
  If a <= 0 Then Exit For

  'check absolute or relative error limit
  If (eps > 0 And Abs(x(1)) < eps) Or _
     (eps < 0 And Abs(x(1)) < -eps * p) Then
    liq_ice_air_iter_at_t = IsOK
    a_si = a
    p_si = p
    d_air_si = dh
    d_liq_si = dl
    Exit For
  End If

Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'FLU_LIMITS
  If d_liq_si <= flu_dmin Or d_liq_si > flu_dmax Or _
     p_si <= 0 Then
    p_si = ErrorReturn
    a_si = ErrorReturn
    d_air_si = ErrorReturn
    d_liq_si = ErrorReturn
    liq_ice_air_iter_at_t = ErrorReturn
    Exit Function
  End If
  'ICE_LIMITS
  If p_si <= ice_pmin Or p_si > ice_pmax Then
    p_si = ErrorReturn
    a_si = ErrorReturn
    d_air_si = ErrorReturn
    d_liq_si = ErrorReturn
    liq_ice_air_iter_at_t = ErrorReturn
    Exit Function
  End If
  'AIR_LIMITS
  If a_si < 0 Or a_si > 1 Or _
     p_si <= 0 Or _
     d_air_si <= dry_air_dmin Or d_air_si > dry_air_dmax Then
    p_si = ErrorReturn
    a_si = ErrorReturn
    d_air_si = ErrorReturn
    d_liq_si = ErrorReturn
    liq_ice_air_iter_at_t = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Function liq_ice_air_iter_at_wa_eta_wet(ByVal wa_si As Double, _
                                                ByVal eta_si As Double, _
                                                ByVal wet_si As Double, _
                                                ByVal maxit As Long, _
                                                ByVal eps As Double, _
                                                ByRef a_si As Double, _
                                                ByRef t_si As Double, _
                                                ByRef p_si As Double, _
                                                ByRef d_air_si As Double, _
                                                ByRef d_liq_si As Double) As Double

'this function returns the water-ice-humid-air phase equilibrium from
'equal chemical potentials of water at given air fraction, wa_si,
'entropy, eta_si, and the wetness, wet_si, of the condensate, from initial guesses
'for the air fraction a_si of the humid-air part, temperature, t_si, pressure, p_si,
'the humid-air density, d_air_si, and the liquid density, d_liq_si.
'wet_si = 0 means the condensed part is ice, wet_si = 1 means the condensed part is liquid.
'The iteration limit eps refers to the error in pressure.

'output:    liq_ice_air_iter_at_wa_eta_wet = IsOK if successfully done
'           liq_ice_air_iter_at_wa_eta_wet = ErrorReturn is returned if
'           - the maximum number of iterations is exceeded without meeting the exit criterion
'           - the function call to a Gibbs/Helmholtz function has returned an error
'           - density, temperature, pressure or air raction have taken an invalid value during the iteration
'           - a-t-p values do not permit the stable existence of ice or water
'     a_si: mass fraction of dry air in humid air in kg/kg
'     t_si: temperature in K of the water-ice-air equilibrium
'     p_si: pressure in Pa of the water-ice-air equilibrium
' d_air_si: density of humid air at (a_si, t_si, p_si)
' d_liq_si: density of liquid water at (t_si, p_si)

'input: wa_si: air fraction of dry air in wet ice air in kg/kg
'      eta_si: specific entropy of wet ice air in J/(kg K)
'      wet_si: liquid fraction of wet ice in kg/kg, (1-wet_si) is the ice fraction
'       maxit: maximum number of iteration steps to be done
'         eps: required accuracy of pressure
'              eps > 0: absolute pressure tolerance in Pa
'              eps < 0: relative pressure tolerance
'        a_si: initial guess for the dry-air fraction in humid air in kg/kg
'        t_si: initial guess for absolute temperature in K
'        p_si: initial guess for absolute pressure in Pa
'    d_air_si: initial guess for density of humid air at (a_si, t_si, p_si)
'    d_liq_si: initial guess for density of liquid at (t_si, p_si)

Dim dh As Double, dl As Double, t As Double, p As Double, a As Double
Dim gi As Double, gi_t As Double, gi_p As Double, gi_tt As Double, gi_tp As Double
Dim fh As Double, fh_a As Double, fh_t As Double, fh_td As Double
Dim fh_d As Double, fh_dd As Double, fh_at As Double, fh_ad As Double
Dim fh_aa As Double, fh_tt As Double
Dim fl As Double, fl_t As Double, fl_tt As Double, fl_td As Double
Dim fl_d As Double, fl_dd As Double

Dim it As Long

'the 5 unknowns are, index 1-5: t, p, dl, dh, a
Dim aa(5, 5) As Double, b(5) As Double, x(5) As Double

liq_ice_air_iter_at_wa_eta_wet = ErrorReturn

If d_air_si <= 0 Then Exit Function
If d_liq_si <= 0 Then Exit Function
If a_si < 0 Or a_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function
If wa_si < 0 Or wa_si >= a_si Then Exit Function
If wet_si < 0 Or wet_si > 1 Then Exit Function

If d_air_si <= 0 Or _
   d_liq_si <= 0 Or _
   a_si < 0 Or a_si >= 1 Or _
   wa_si < 0 Or wa_si > 1 Or _
   wet_si < 0 Or wet_si > 1 Or _
   t_si <= 0 Or _
   p_si <= 0 Then
  a_si = ErrorReturn
  t_si = ErrorReturn
  p_si = ErrorReturn
  d_air_si = ErrorReturn
  d_liq_si = ErrorReturn
  Exit Function
End If

check_limits = check_limits - 1

a = a_si
t = t_si
p = p_si
dl = d_liq_si
dh = d_air_si

For it = 1 To maxit

  'Derivatives of the Helmholtz function of humid air for 5D-Newton iteration
  fh = air_f_si(0, 0, 0, a, t, dh)
  If fh = ErrorReturn Then Exit For
  fh_a = air_f_si(1, 0, 0, a, t, dh)
  If fh_a = ErrorReturn Then Exit For
  fh_t = air_f_si(0, 1, 0, a, t, dh)
  If fh_t = ErrorReturn Then Exit For
  fh_d = air_f_si(0, 0, 1, a, t, dh)
  If fh_d = ErrorReturn Then Exit For
  fh_aa = air_f_si(2, 0, 0, a, t, dh)
  If fh_aa = ErrorReturn Then Exit For
  fh_at = air_f_si(1, 1, 0, a, t, dh)
  If fh_at = ErrorReturn Then Exit For
  fh_ad = air_f_si(1, 0, 1, a, t, dh)
  If fh_ad = ErrorReturn Then Exit For
  fh_tt = air_f_si(0, 2, 0, a, t, dh)
  If fh_tt = ErrorReturn Then Exit For
  fh_td = air_f_si(0, 1, 1, a, t, dh)
  If fh_td = ErrorReturn Then Exit For
  fh_dd = air_f_si(0, 0, 2, a, t, dh)
  If fh_dd = ErrorReturn Then Exit For

  'Derivatives of the Helmholtz function of liquid water for Newton iteration
  fl = flu_f_si(0, 0, t, dl)
  If fl = ErrorReturn Then Exit For
  fl_t = flu_f_si(1, 0, t, dl)
  If fl_t = ErrorReturn Then Exit For
  fl_d = flu_f_si(0, 1, t, dl)
  If fl_d = ErrorReturn Then Exit For
  fl_tt = flu_f_si(2, 0, t, dl)
  If fl_tt = ErrorReturn Then Exit For
  fl_td = flu_f_si(1, 1, t, dl)
  If fl_td = ErrorReturn Then Exit For
  fl_dd = flu_f_si(0, 2, t, dl)
  If fl_dd = ErrorReturn Then Exit For

  'Derivatives of the Gibbs function of ice for Newton iteration
  gi = ice_g_si(0, 0, t, p)
  If gi = ErrorReturn Then Exit For
  gi_t = ice_g_si(1, 0, t, p)
  If gi_t = ErrorReturn Then Exit For
  gi_p = ice_g_si(0, 1, t, p)
  If gi_p = ErrorReturn Then Exit For
  gi_tt = ice_g_si(2, 0, t, p)
  If gi_tt = ErrorReturn Then Exit For
  gi_tp = ice_g_si(1, 1, t, p)
  If gi_tp = ErrorReturn Then Exit For

  'coefficient matrix
  aa(1, 1) = gi_t - fl_t
  aa(1, 2) = gi_p - 1 / dl
  aa(1, 3) = p / dl ^ 2 - fl_d
  aa(1, 4) = 0
  aa(1, 5) = 0
  
  aa(2, 1) = -dl * fl_td
  aa(2, 2) = 1 / dl
  aa(2, 3) = -fl_d - dl * fl_dd - p / dl ^ 2
  aa(2, 4) = 0
  aa(2, 5) = 0

  aa(3, 1) = gi_t - fh_t + a * fh_at
  aa(3, 2) = gi_p - 1 / dh
  aa(3, 3) = 0
  aa(3, 4) = p / dh ^ 2 + a * fh_ad - fh_d
  aa(3, 5) = a * fh_aa

  aa(4, 1) = -dh * fh_td
  aa(4, 2) = 1 / dh
  aa(4, 3) = 0
  aa(4, 4) = -p / dh ^ 2 - fh_d - dh * fh_dd
  aa(4, 5) = -dh * fh_ad

  aa(5, 1) = (a - wa_si) * (wet_si * fl_tt + (1 - wet_si) * gi_tt) + wa_si * fh_tt
  aa(5, 2) = (a - wa_si) * (1 - wet_si) * gi_tp
  aa(5, 3) = (a - wa_si) * wet_si * fl_td
  aa(5, 4) = wa_si * fh_td
  aa(5, 5) = wet_si * fl_t + (1 - wet_si) * gi_t + wa_si * fh_at

  'right-hand sides, must vanish at equilibrium
  b(1) = fl + p / dl - gi
  b(2) = dl * fl_d - p / dl
  b(3) = -gi + fh + p / dh - a * fh_a
  b(4) = dh * fh_d - p / dh
  b(5) = (wa_si - a) * (wet_si * fl_t + (1 - wet_si) * gi_t) - wa_si * fh_t - a * eta_si
  
  'solve equations
  If matrix_solve(aa(), b(), x(), 5) <> 0 Then Exit For 'matrix singular
  
  'update unknowns
  t = t + x(1)
  If t <= 0 Then Exit For
  p = p + x(2)
  If p <= 0 Then Exit For
  dl = dl + x(3)
  If dl <= 0 Then Exit For
  dh = dh + x(4)
  If dh <= 0 Then Exit For
  a = a + x(5)
  If a < 0 Or a >= 1 Then Exit For

  'check absolute or relative error limit
  If (eps > 0 And Abs(x(2)) < eps) Or _
     (eps < 0 And Abs(x(2)) < -eps * Abs(p)) Then
    liq_ice_air_iter_at_wa_eta_wet = IsOK
    a_si = a
    t_si = t
    p_si = p
    d_air_si = dh
    d_liq_si = dl
    Exit For
  End If

Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Or _
     d_liq_si <= flu_dmin Or d_liq_si > flu_dmax Then
    a_si = ErrorReturn
    t_si = ErrorReturn
    p_si = ErrorReturn
    d_air_si = ErrorReturn
    d_liq_si = ErrorReturn
    liq_ice_air_iter_at_wa_eta_wet = ErrorReturn
    Exit Function
  End If
  'ICE_LIMITS
  If t_si <= ice_tmin Or t_si > ice_tmax Or _
     p_si <= ice_pmin Or p_si > ice_pmax Then
    a_si = ErrorReturn
    t_si = ErrorReturn
    p_si = ErrorReturn
    d_air_si = ErrorReturn
    d_liq_si = ErrorReturn
    liq_ice_air_iter_at_wa_eta_wet = ErrorReturn
    Exit Function
  End If
  'AIR_LIMITS
  If t_si < dry_air_tmin Or t_si > dry_air_tmax Or _
  d_air_si <= dry_air_dmin Or d_air_si > dry_air_dmax Then
    a_si = ErrorReturn
    t_si = ErrorReturn
    p_si = ErrorReturn
    d_air_si = ErrorReturn
    d_liq_si = ErrorReturn
    liq_ice_air_iter_at_wa_eta_wet = ErrorReturn
  End If
End If

End Function


