Attribute VB_Name = "Ice_Vap_4_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules:
'     Constants_0_Mdl, file Constants_0.bas
'     Flu_1_Mdl,       file Flu_1.bas
'     Ice_1_Mdl,       file Ice_1.bas
'     Flu_2_Mdl,       file Flu_2.bas
'     Ice_2_Mdl,       file Ice_2.bas
'     Maths_0_Mdl,     file Maths_0.bas

'#########################################################################

'=========================================================================
'This module implements the phase equilibrim properties of ice with water vapour
'computed from IAPWS-95 and IAPWS-06:

'Release on the IAPWS Formulation 1995 for the Thermodynamic Properties of
'Ordinary Water Substance for General and Scientific Use
'The International Association for the Properties of Water and Steam
'Fredericia, Denmark, September 1996

'Release on an Equation of State for H2O Ice Ih
'The International Association for the Properties of Water and Steam
'Witney, UK, September 2006

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98
'Private Const IsOK = -1

'Control parameters of the vapour pressure iteration
Private ctrl_initialized As Integer

Private ctrl_mode_ice As Integer
Private ctrl_mode_vapour As Integer
Private ctrl_mode_temperature As Integer
Private ctrl_loop_maximum As Long
Private ctrl_init_d_vap As Double
Private ctrl_init_t As Double
Private ctrl_eps_exit_p_vap As Double

'Properties of the current ice-vapour equilibrium state
Private equi_ice_vap_done As Integer

Private equi_ice_vap_t As Double
Private equi_ice_vap_p_vap As Double
Private equi_ice_vap_g_ice As Double
Private equi_ice_vap_g_vap As Double
Private equi_ice_vap_d_ice As Double
Private equi_ice_vap_d_vap As Double

Private Const Version = "16 Sep 2009"

'==========================================================================
Public Function ice_vap_sublimationpressure_si(ByVal t_si As Double) As Double

'returns the absolute pressure in Pa of water vapour at vapour-ice equilibrium
'as a function of absolute temperature t_si in K

'note: the accuracy of this function depends on the iteration settings in ths module

'check value with default settings: ice_vap_sublimationpressure_si(270) = 470.059067980884

ice_vap_sublimationpressure_si = ErrorReturn

If set_ice_vap_eq_at_t(t_si) = ErrorReturn Then Exit Function

ice_vap_sublimationpressure_si = equi_ice_vap_p_vap

End Function

'==========================================================================
Public Function ice_vap_sublimationtemp_si(ByVal p_si As Double) As Double

'returns the absolute temperature in K of vapour-ice equilibrium
'as a function of absolute pressure p_si in Pa

'note: the accuracy of this function depends on the iteration settings in ths module

'check value with default settings: ice_vap_sublimationtemp_si(100) = 252.817910214512

ice_vap_sublimationtemp_si = ErrorReturn

If set_ice_vap_eq_at_p(p_si) = ErrorReturn Then Exit Function

ice_vap_sublimationtemp_si = equi_ice_vap_t

End Function

'==========================================================================
Public Function ice_vap_pressure_vap_si() As Double

'returns the absolute pressure in Pa of water vapour at vapour-ice equilibrium
'after execution of set_ice_vap_eq_at_t or set_ice_vap_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_ice_vap_eq_at_t 270
'ice_vap_pressure_vap_si = 470.059067980884

'set_ice_vap_eq_at_p 100
'ice_vap_pressure_vap_si = 100

ice_vap_pressure_vap_si = ErrorReturn

If equi_ice_vap_done <> IsOK Then Exit Function

ice_vap_pressure_vap_si = equi_ice_vap_p_vap

End Function

'==========================================================================
Public Function ice_vap_temperature_si() As Double

'returns the absolute temperature in K of water vapour at vapour-ice equilibrium
'after execution of set_ice_vap_eq_at_t or set_ice_vap_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_ice_vap_eq_at_t 270
'ice_vap_temperature_si = 270

'set_ice_vap_eq_at_p 100
'ice_vap_temperature_si = 252.817910214512

ice_vap_temperature_si = ErrorReturn

If equi_ice_vap_done <> IsOK Then Exit Function

ice_vap_temperature_si = equi_ice_vap_t

End Function

'==========================================================================
Public Function ice_vap_chempot_si() As Double

'returns the chemical potential in J/kg of water vapour at vapour-ice equilibrium
'after execution of set_ice_vap_eq_at_t or set_ice_vap_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_ice_vap_eq_at_t 270
'ice_vap_chempot_si = -3895.2674739233

'set_ice_vap_eq_at_p 100
'ice_vap_chempot_si = -26421.2820403233

ice_vap_chempot_si = ErrorReturn

If equi_ice_vap_done <> IsOK Then Exit Function

ice_vap_chempot_si = equi_ice_vap_g_vap


End Function

'==========================================================================
Public Function ice_vap_density_ice_si() As Double

'returns the density in kg/m3 of ice at vapour-ice equilibrium
'after execution of set_ice_vap_eq_at_t or set_ice_vap_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_ice_vap_eq_at_t 270
'ice_vap_density_ice_si = 917.170465732797

'set_ice_vap_eq_at_p 100
'ice_vap_density_ice_si = 919.600269744796

ice_vap_density_ice_si = ErrorReturn

If equi_ice_vap_done <> IsOK Then Exit Function

ice_vap_density_ice_si = equi_ice_vap_d_ice

End Function

'==========================================================================
Public Function ice_vap_density_vap_si() As Double

'returns the density in kg/m3 of water vapour at vapour-ice equilibrium
'after execution of set_ice_vap_eq_at_t or set_ice_vap_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_ice_vap_eq_at_t 270
'ice_vap_density_vap_si = 3.7740614077202E-03

'set_ice_vap_eq_at_p 100
'ice_vap_density_vap_si = 8.57185487853061E-04

ice_vap_density_vap_si = ErrorReturn

If equi_ice_vap_done <> IsOK Then Exit Function

ice_vap_density_vap_si = equi_ice_vap_d_vap

End Function

'==========================================================================
Public Function ice_vap_entropy_ice_si() As Double

'returns the specific entropy in J/(kg K) of ice at vapour-ice equilibrium
'after execution of set_ice_vap_eq_at_t or set_ice_vap_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_ice_vap_eq_at_t 270
'ice_vap_entropy_ice_si = -1244.95617472139

'set_ice_vap_eq_at_p 100
'ice_vap_entropy_ice_si = -1377.09771246549

ice_vap_entropy_ice_si = ErrorReturn

If equi_ice_vap_done <> IsOK Then Exit Function

ice_vap_entropy_ice_si = ice_entropy_si(equi_ice_vap_t, equi_ice_vap_p_vap)

End Function

'==========================================================================
Public Function ice_vap_entropy_vap_si() As Double

'returns the specific entropy in J/(kg K) of water vapour at vapour-ice equilibrium
'after execution of set_ice_vap_eq_at_t or set_ice_vap_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_ice_vap_eq_at_t 270
'ice_vap_entropy_vap_si = 9255.65736017602

'set_ice_vap_eq_at_p 100
'ice_vap_entropy_vap_si = 9848.7740691229

ice_vap_entropy_vap_si = ErrorReturn

If equi_ice_vap_done <> IsOK Then Exit Function

ice_vap_entropy_vap_si = flu_entropy_si(equi_ice_vap_t, equi_ice_vap_d_vap)

End Function

'==========================================================================
Public Function ice_vap_enthalpy_ice_si() As Double

'returns the specific enthalpy in J/kg of ice at vapour-ice equilibrium
'after execution of set_ice_vap_eq_at_t or set_ice_vap_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_ice_vap_eq_at_t 270
'ice_vap_enthalpy_ice_si = -340033.434648698

'set_ice_vap_eq_at_p 100
'ice_vap_enthalpy_ice_si = -374576.247867035

ice_vap_enthalpy_ice_si = ErrorReturn

If equi_ice_vap_done <> IsOK Then Exit Function

ice_vap_enthalpy_ice_si = ice_enthalpy_si(equi_ice_vap_t, equi_ice_vap_p_vap)

End Function

'==========================================================================
Public Function ice_vap_enthalpy_vap_si() As Double

'returns the specific enthalpy in J/kg of water vapour at vapour-ice equilibrium
'after execution of set_ice_vap_eq_at_t or set_ice_vap_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_ice_vap_eq_at_t 270
'ice_vap_enthalpy_vap_si = 2495132.2197736

'set_ice_vap_eq_at_p 100
'ice_vap_enthalpy_vap_si = 2463525.19629021

ice_vap_enthalpy_vap_si = ErrorReturn

If equi_ice_vap_done <> IsOK Then Exit Function

ice_vap_enthalpy_vap_si = flu_enthalpy_si(equi_ice_vap_t, equi_ice_vap_d_vap)

End Function

'==========================================================================
Public Function ice_vap_volume_subl_si() As Double

'returns the specific sublimation volume in m3/kg of water
'after execution of set_ice_vap_eq_at_t or set_ice_vap_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_ice_vap_eq_at_t 270
'ice_vap_volume_subl_si = 264.965451557852

'set_ice_vap_eq_at_p 100
'ice_vap_volume_subl_si = 1166.60755699012

ice_vap_volume_subl_si = ErrorReturn

If equi_ice_vap_done <> IsOK Then Exit Function

If equi_ice_vap_d_ice <= 0 Then Exit Function
If equi_ice_vap_d_vap <= 0 Then Exit Function

ice_vap_volume_subl_si = 1# / equi_ice_vap_d_vap - 1# / equi_ice_vap_d_ice

End Function

'==========================================================================
Public Function ice_vap_entropy_subl_si() As Double

'returns the specific sublimation entropy in J/(kg K) of water
'after execution of set_ice_vap_eq_at_t or set_ice_vap_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_ice_vap_eq_at_t 270
'ice_vap_entropy_subl_si = 10500.6135348974

'set_ice_vap_eq_at_p 100
'ice_vap_entropy_subl_si = 11225.8717815884

Dim si As Double, sv As Double

ice_vap_entropy_subl_si = ErrorReturn

If equi_ice_vap_done <> IsOK Then Exit Function

si = ice_entropy_si(equi_ice_vap_t, equi_ice_vap_p_vap)
If si = ErrorReturn Then Exit Function
sv = flu_entropy_si(equi_ice_vap_t, equi_ice_vap_d_vap)
If sv = ErrorReturn Then Exit Function

ice_vap_entropy_subl_si = sv - si

End Function

'==========================================================================
Public Function ice_vap_enthalpy_subl_si() As Double

'returns the specific sublimation enthalpy in J/kg of water
'after execution of set_ice_vap_eq_at_t or set_ice_vap_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_ice_vap_eq_at_t 270
'ice_vap_enthalpy_subl_si = 2835165.6544223

'set_ice_vap_eq_at_p 100
'ice_vap_enthalpy_subl_si = 2838101.44415724

Dim hi As Double, hv As Double

ice_vap_enthalpy_subl_si = ErrorReturn

If equi_ice_vap_done <> IsOK Then Exit Function

hi = ice_enthalpy_si(equi_ice_vap_t, equi_ice_vap_p_vap)
If hi = ErrorReturn Then Exit Function
hv = flu_enthalpy_si(equi_ice_vap_t, equi_ice_vap_d_vap)
If hv = ErrorReturn Then Exit Function

ice_vap_enthalpy_subl_si = hv - hi

End Function

'==========================================================================
Public Function set_ice_vap_eq_at_p(ByVal p_si As Double) As Double

'this function computes the ice-vapour equilibrium at given pressure
'by 2-dimensional Newton iteration

'triple point
Const dv_triple = TP_density_vap_IAPWS95_si
Const t_triple = TP_temperature_si

Dim t_si As Double
Dim pv As Double
Dim gi As Double, gv As Double
Dim di As Double, dv As Double

Dim fv_d As Double, eps As Double
Dim maxit As Long

If equi_ice_vap_done = IsOK And _
   p_si = equi_ice_vap_p_vap Then
  'the requested state has already been computed earlier
  set_ice_vap_eq_at_p = IsOK
  Exit Function
End If

clear_ice_vap_state 'waste any previous state

set_ice_vap_eq_at_p = ErrorReturn

If p_si <= 0 Then Exit Function

init_it_ctrl_ice_vap

'set initial temperature guess
Select Case ctrl_mode_temperature
  Case 0:  t_si = aux_temperature_correlation(p_si)
           If t_si = ErrorReturn Then Exit Function
  Case -1: t_si = t_triple
  Case 1:  t_si = ctrl_init_t
  Case Else: Exit Function
End Select

'set initial vapour density guess
Select Case ctrl_mode_vapour
  Case 0:  dv = aux_vap_density_correlation(t_si)
  Case -1: dv = dv_triple
  Case 1:  dv = ctrl_init_d_vap
  Case Else: Exit Function
End Select

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: pv = flu_pressure_si(t_si, dv)
           gi = ice_g_si(0, 0, t_si, pv)
           gv = flu_gibbs_energy_si(t_si, dv)
           di = ice_g_si(0, 1, t_si, pv)
           If di = ErrorReturn Or di <= 0 Then Exit Function
           di = 1# / di
           set_ice_vap_state t_si, pv, gi, gv, di, dv
           set_ice_vap_eq_at_p = IsOK
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit
eps = ctrl_eps_exit_p_vap
If eps = 0 Then Exit Function

'run iteration loop
If ice_vap_iteration_at_p(p_si, maxit, eps, t_si, di, dv, gi, gv, pv) = ErrorReturn Then
  Exit Function
End If

set_ice_vap_state t_si, pv, gi, gv, di, dv

set_ice_vap_eq_at_p = IsOK

End Function

'==========================================================================
Public Function set_ice_vap_eq_at_t(ByVal t_si As Double) As Double

'this function computes the ice-vapour equilibrium at given temperature
'by Newton iteration

'triple point
Const t_triple = TP_temperature_si
Const dv_triple = TP_density_vap_IAPWS95_si

Dim pv As Double
Dim gi As Double, gv As Double
Dim di As Double, dv As Double

Dim fv_d As Double, eps As Double
Dim maxit As Long

If equi_ice_vap_done = IsOK And _
   t_si = equi_ice_vap_t Then
  'the requested state has already been computed earlier
  set_ice_vap_eq_at_t = IsOK
  Exit Function
End If

clear_ice_vap_state 'waste any previous state

set_ice_vap_eq_at_t = ErrorReturn

If t_si <= 0 Then Exit Function
If t_si > t_triple Then Exit Function

init_it_ctrl_ice_vap

'set initial vapour density guess
Select Case ctrl_mode_vapour
  Case 0: dv = aux_vap_density_correlation(t_si)
  Case -1: dv = dv_triple
  Case 1:  dv = ctrl_init_d_vap
  Case Else: Exit Function
End Select

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: pv = flu_pressure_si(t_si, dv)
           gi = ice_g_si(0, 0, t_si, pv)
           gv = flu_gibbs_energy_si(t_si, dv)
           di = ice_g_si(0, 1, t_si, pv)
           If di = ErrorReturn Or di <= 0 Then Exit Function
           di = 1# / di
           set_ice_vap_state t_si, pv, gi, gv, di, dv
           set_ice_vap_eq_at_t = IsOK
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit
eps = ctrl_eps_exit_p_vap
If eps = 0 Then Exit Function

'run iteration loop
If ice_vap_iteration_at_t(t_si, maxit, eps, di, dv, gi, gv, pv) = ErrorReturn Then
  Exit Function
End If

set_ice_vap_state t_si, pv, gi, gv, di, dv

set_ice_vap_eq_at_t = IsOK

End Function

'==========================================================================
Private Function ice_vap_iteration_at_p(ByVal p_si As Double, _
                                        ByVal maxit As Long, _
                                        ByVal eps As Double, _
                                        ByRef t_si As Double, _
                                        ByRef d_ice_si As Double, _
                                        ByRef d_vap_si As Double, _
                                        ByRef g_ice_si As Double, _
                                        ByRef g_vap_si As Double, _
                                        ByRef p_vap_si As Double) As Double

'this function returns the ice-vapour phase equilibrium from equal pressures,
'temperatures and chemical potentials of the two phases at given pressure, p_si,
'from initial guesses for the vapour density, d_vap_si, and the temperature, t_si
'The iteration limit eps refers to the error in vapour pressure

'output:    ice_vap_iteration_at_p = IsOK if successful
'           ice_vap_iteration_at_p = ErrorReturn is returned if
'           - the maximum number of iterations is exceeded without meeting the exit criterion
'           - the function call to flu_f_si or to ice_g_si has returned an error
'           - density or temperature have taken a zero or negative value during the iteration
'     t_si: absolute temperature in K
' d_ice_si: ice density at the ice-vapour equilibrium in kg/m3
' d_vap_si: vapour density at the ice-vapour equilibrium in kg/m3
' g_ice_si: ice chemical potential at the ice-vapour equilibrium in J/kg
' g_vap_si: vapour chemical potential at the ice-vapour equilibrium in J/kg
' p_vap_si: vapour pressure at the ice-vapour equilibrium in Pa

'input: p_si: absolute pressure in Pa
'      maxit: maximum number of iteration steps to be done
'        eps: required accuracy of vapour pressure
'             eps > 0: absolute vapour pressure tolerance in Pa
'             eps < 0: relative vapour pressure tolerance
'       t_si: initial guess of absolute temperature in K
'   d_vap_si: initial guess of vapour density in kg/m3

Const Tt = TP_temperature_si 'triple point in K

Dim vi As Double, dv As Double
Dim pv As Double, gv As Double
Dim gi As Double, gi_t As Double
Dim t As Double
Dim fv As Double, fv_d As Double, fv_dd As Double
Dim fv_t As Double, fv_td As Double

Dim it As Long

Dim a(2, 2) As Double, b(2) As Double, x(2) As Double
Dim ddv As Double, dt As Double, pv_old As Double

ice_vap_iteration_at_p = ErrorReturn

If t_si <= 0 Or d_vap_si <= 0 Then
  t_si = ErrorReturn
  d_ice_si = ErrorReturn
  d_vap_si = ErrorReturn
  g_ice_si = ErrorReturn
  g_vap_si = ErrorReturn
  p_vap_si = ErrorReturn
  Exit Function
End If

If check_limits = 1 Then
'ICE_LIMITS
  If p_si <= ice_pmin Or p_si > ice_pmax Then
    t_si = ErrorReturn
    d_ice_si = ErrorReturn
    d_vap_si = ErrorReturn
    g_ice_si = ErrorReturn
    g_vap_si = ErrorReturn
    p_vap_si = ErrorReturn
    Exit Function
  End If
Else
  If p_si <= 0 Then
    t_si = ErrorReturn
    d_ice_si = ErrorReturn
    d_vap_si = ErrorReturn
    g_ice_si = ErrorReturn
    g_vap_si = ErrorReturn
    p_vap_si = ErrorReturn
    Exit Function
  End If
End If

check_limits = check_limits - 1

dv = d_vap_si
t = t_si

For it = 0 To maxit

  'Derivatives of the Gibbs & Helmholtz functions for 2D-Newton iteration
  fv = flu_f_si(0, 0, t, dv)
  If fv = ErrorReturn Then Exit For
  
  fv_d = flu_f_si(0, 1, t, dv)
  If fv_d = ErrorReturn Then Exit For
  fv_dd = flu_f_si(0, 2, t, dv)
  If fv_dd = ErrorReturn Then Exit For
  
  fv_t = flu_f_si(1, 0, t, dv)
  If fv_t = ErrorReturn Then Exit For
  fv_td = flu_f_si(1, 1, t, dv)
  If fv_td = ErrorReturn Then Exit For

  pv_old = pv  'keep previous pv to check accuracy
  'vapour pressure and ice + vapour chemical potentials
  pv = dv ^ 2 * fv_d     'pressure of vapour
  gv = fv + dv * fv_d    'chem. pot. of vapour
  
  gi = ice_g_si(0, 0, t, pv)
  If gi = ErrorReturn Then Exit For
  gi_t = ice_g_si(1, 0, t, pv)
  If gi_t = ErrorReturn Then Exit For

  If it > 0 Then
    'check absolute or relative error limit
    If (eps > 0 And Abs(pv - pv_old) < eps) Or _
       (eps < 0 And Abs(pv - pv_old) < -eps * pv) Then
      p_vap_si = pv
      g_ice_si = gi
      g_vap_si = gv
      vi = ice_g_si(0, 1, t, pv)
      If vi = ErrorReturn Or vi <= 0 Then Exit For
      d_ice_si = 1# / vi
      d_vap_si = dv
      t_si = t
      ice_vap_iteration_at_p = IsOK
      Exit For
    End If
  End If
  
  If it = maxit Then Exit For
  
  'coefficient matrix
  a(1, 1) = dv * (2 * fv_d + dv * fv_dd)
  a(1, 2) = dv ^ 2 * fv_td
  a(2, 1) = (2 * fv_d + dv * fv_dd)
  a(2, 2) = -gi_t + fv_t + dv * fv_td

  'right-hand sides, must vanish in equilibrium
  b(1) = p_si - pv
  b(2) = gi - gv

  'solve equations
  If matrix_solve(a(), b(), x(), 2) <> 0 Then Exit For 'matrix singular
  ddv = x(1)
  dt = x(2)
  
  'update density & temperature
  dv = dv + ddv
  If dv <= 0 Then Exit For
  t = t + dt
  If t <= 0 Then Exit For
  
Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Or _
     d_vap_si <= flu_dmin Or d_vap_si > flu_dmax Or _
     p_vap_si < 0 Then
    t_si = ErrorReturn
    d_ice_si = ErrorReturn
    d_vap_si = ErrorReturn
    g_ice_si = ErrorReturn
    g_vap_si = ErrorReturn
    p_vap_si = ErrorReturn
    ice_vap_iteration_at_p = ErrorReturn
    Exit Function
  End If
  'ICE_LIMITS
  If t_si <= ice_tmin Or t_si > ice_tmax Or _
     d_ice_si <= 0 Then
    t_si = ErrorReturn
    d_ice_si = ErrorReturn
    d_vap_si = ErrorReturn
    g_ice_si = ErrorReturn
    g_vap_si = ErrorReturn
    p_vap_si = ErrorReturn
    ice_vap_iteration_at_p = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Function ice_vap_iteration_at_t(ByVal t_si As Double, _
                                        ByVal maxit As Long, _
                                        ByVal eps As Double, _
                                        ByRef d_ice_si As Double, _
                                        ByRef d_vap_si As Double, _
                                        ByRef g_ice_si As Double, _
                                        ByRef g_vap_si As Double, _
                                        ByRef p_vap_si As Double) As Double

'this function returns the ice-vapour phase equilibrium from equal pressures
'and chemical potentials of the two phases at given temperature, t_si, in K
'from an initial guess for the vapour density, d_vap_si, in kg/m3
'The iteration limit eps refers to the error in vapour pressure

'output:    ice_vap_iteration = IsOK if successfully done
'           ice_vap_iteration = ErrorReturn is returned if
'           - the maximum number of iterations is exceeded without meeting the exit criterion
'           - the function call to flu_f_si or ice_g_si has returned an error
'           - density has taken a zero or negative value during the iteration
' d_vap_si: vapour density at the ice-vapour equilibrium in kg/m3
' d_ice_si: ice density at the ice-vapour equilibrium in kg/m3
' g_vap_si: vapour chemical potential at the ice-vapour equilibrium in J/kg
' g_ice_si: ice chemical potential at the ice-vapour equilibrium in J/kg
' p_vap_si: vapour pressure at the ice-vapour equilibrium in Pa

'input: t_si: absolute temperature in K
'      maxit: maximum number of iteration steps to be done
'        eps: required accuracy of vapour pressure
'             eps > 0: absolute pressure tolerance in Pa
'             eps < 0: relative pressure tolerance
'   d_vap_si: initial guess of vapour density in kg/m3

Dim vi As Double, dv As Double
Dim pv As Double
Dim gi As Double, gv As Double
Dim fv As Double, fv_d As Double, fv_dd As Double

Dim it As Long

Dim ddv As Double, pv_old As Double

ice_vap_iteration_at_t = ErrorReturn

If d_vap_si <= 0 Then Exit Function
If t_si <= 0 Then Exit Function

If check_limits = 1 Then
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Then
    d_vap_si = ErrorReturn
    d_ice_si = ErrorReturn
    g_vap_si = ErrorReturn
    g_ice_si = ErrorReturn
    p_vap_si = ErrorReturn
    Exit Function
  End If
  'ICE_LIMITS
  If t_si <= ice_tmin Or t_si > ice_tmax Then
    d_vap_si = ErrorReturn
    d_ice_si = ErrorReturn
    g_vap_si = ErrorReturn
    g_ice_si = ErrorReturn
    p_vap_si = ErrorReturn
    Exit Function
  End If
Else
  If t_si <= 0 Then
    d_vap_si = ErrorReturn
    d_ice_si = ErrorReturn
    g_vap_si = ErrorReturn
    g_ice_si = ErrorReturn
    p_vap_si = ErrorReturn
    Exit Function
  End If
End If

check_limits = check_limits - 1

dv = d_vap_si

For it = 0 To maxit

  'Derivatives of the Helmholtz function for Newton iteration
  fv = flu_f_si(0, 0, t_si, dv)
  If fv = ErrorReturn Then Exit For
  fv_d = flu_f_si(0, 1, t_si, dv)
  If fv_d = ErrorReturn Then Exit For
  fv_dd = flu_f_si(0, 2, t_si, dv)
  If fv_dd = ErrorReturn Then Exit For
  
  pv_old = pv  'keep previous pv to check accuracy
  'vapour pressure and ice + vapour chemical potentials
  pv = dv ^ 2 * fv_d     'pressure of vapour
  gv = fv + dv * fv_d    'chem. pot. of vapour
  
  'Derivatives of the Gibbs function of ice for Newton iteration
  gi = ice_g_si(0, 0, t_si, pv)
  If gi = ErrorReturn Then Exit For
  vi = ice_g_si(0, 1, t_si, pv)
  If vi = ErrorReturn Then Exit For
  If vi <= 0 Then Exit For
  
  If it > 0 Then
    'check absolute or relative error limit
    If (eps > 0 And Abs(pv - pv_old) < eps) Or _
       (eps < 0 And Abs(pv - pv_old) < -eps * pv) Then
      ice_vap_iteration_at_t = IsOK
      p_vap_si = pv
      g_ice_si = gi
      g_vap_si = gv
      d_ice_si = 1# / vi
      d_vap_si = dv
      Exit For
    End If
  End If

  If it = maxit Then Exit For
  
  'iteration step
  ddv = (2 * fv_d + dv * fv_dd) * (vi * dv - 1#)
  If ddv = 0 Then Exit For
  ddv = (gv - gi) / ddv
  
  'update vapour density
  dv = dv + ddv
  If dv <= 0 Then Exit For

Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'FLU_LIMITS
  If d_vap_si <= flu_dmin Or d_vap_si > flu_dmax Or _
     p_vap_si <= 0 Then
    d_vap_si = ErrorReturn
    d_ice_si = ErrorReturn
    g_vap_si = ErrorReturn
    g_ice_si = ErrorReturn
    p_vap_si = ErrorReturn
    ice_vap_iteration_at_t = ErrorReturn
    Exit Function
  End If
  'ICE_LIMITS
  If d_ice_si <= 0 Then
    d_vap_si = ErrorReturn
    d_ice_si = ErrorReturn
    g_vap_si = ErrorReturn
    g_ice_si = ErrorReturn
    p_vap_si = ErrorReturn
    ice_vap_iteration_at_t = ErrorReturn
  End If
  
End If

End Function

'==========================================================================
Private Sub init_it_ctrl_ice_vap()

'triple point
Const dv_triple = TP_density_vap_IAPWS95_si
Const t_triple = TP_temperature_si

If ctrl_initialized = IsOK Then Exit Sub

ctrl_initialized = IsOK

'Set default values and modes for the iteration
ctrl_mode_ice = 0
ctrl_mode_vapour = 0
ctrl_mode_temperature = 0
ctrl_loop_maximum = 100
ctrl_init_d_vap = dv_triple
ctrl_init_t = t_triple
ctrl_eps_exit_p_vap = -0.0000001 'relative, 0.1 ppm

End Sub

'==========================================================================
Public Sub set_it_ctrl_ice_vap(ByVal key As String, ByVal value As Double)

'this sub sets control parameters for the iteration used to compute
'ice-vapour equilibrium

'key             value
'it_steps        0           set iteration number to default (100)
'it_steps        n > 0       set iteration number to n
'it_steps       -1           do not iterate, use initial vapour density to compute vapour pressure

'init_vap_dens   0           use default vapour density to start =
'                            vapour density correlation function, d = aux_vap_density_correlation(T)
'init_vap_dens  -1           use triple point vapour density to start
'init_vap_dens   d > 0       use value d as vapour density to start

'init_temp       0           use default temperature to start ( = aux_subl_temperature(p))
'init_temp      -1           use triple point temperature to start
'init_temp       t > 0       use value t as temperature to start

'tol_vap_press   0           use default exit tolerance for vapour pressure (0.1 ppm)
'tol_vap_press   eps         use eps as exit tolerance for vapour pressure (eps < 0 means relative error)


init_it_ctrl_ice_vap

clear_ice_vap_state

Select Case LCase(Trim(key))

  Case "it_steps":   'iteration steps
    Select Case value
      Case 0:      ctrl_loop_maximum = 100  'default = 100
      Case Is < 0: ctrl_loop_maximum = -1
      Case Else:   ctrl_loop_maximum = value
    End Select

  Case "init_vap_dens":   'start vapour density
    Select Case CLng(value)
      Case 0:       ctrl_mode_vapour = 0    'default = aux polynomial
      Case Is < -1: 'ignore it
      Case Is < 0:  ctrl_mode_vapour = value
      Case Else:    ctrl_mode_vapour = 1
                    ctrl_init_d_vap = value
    End Select

  Case "init_temp":   'start temperature
    Select Case CLng(value)
      Case 0:       ctrl_mode_temperature = 0    'default = aux polynomial
      Case Is < -1: 'ignore it
      Case Is < 0:  ctrl_mode_temperature = value
      Case Else:    ctrl_mode_temperature = 1
                    ctrl_init_t = value
    End Select

  Case "tol_vap_press":      'required vapour pressure tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_p_vap = -0.0000001   'default = 0.1 ppm relative
      Case Else:   ctrl_eps_exit_p_vap = value
    End Select

End Select

End Sub

'==========================================================================
Private Sub set_ice_vap_state(ByVal t As Double, _
                              ByVal pv As Double, _
                              ByVal gi As Double, _
                              ByVal gv As Double, _
                              ByVal di As Double, _
                              ByVal dv As Double)

'stores the actual properties as the current equilibrium state descriptor

equi_ice_vap_done = IsOK

equi_ice_vap_t = t
equi_ice_vap_p_vap = pv
equi_ice_vap_g_ice = gi
equi_ice_vap_g_vap = gv
equi_ice_vap_d_ice = di
equi_ice_vap_d_vap = dv

End Sub

'==========================================================================
Private Sub clear_ice_vap_state()

'clears the current equilibrium state descriptor

equi_ice_vap_done = 0

End Sub

'==========================================================================
Private Function aux_vap_density_correlation(ByVal t_si As Double) As Double

'this function implements the vapour density of sublimation from the
'Clausius-Clapeyron equation as a first guess for the sublimation iteration

Const R = Gas_constant_H2O_si  'specific gas constant of H2O in J/(kg K)
Const q = TP_enthalpy_vap_si - TP_enthalpy_ice_si ' = 2834359.44543354   'Sublimation enthalpy in J/kg
Const Pt = TP_pressure_IAPWS95_si  'IAPWS-95 triple point pressure in Pa
Const Tt = TP_temperature_si

Dim RT As Double

RT = R * t_si

aux_vap_density_correlation = (Pt / RT) * Exp((t_si / Tt - 1#) * q / RT)

End Function

'==========================================================================
Private Function aux_temperature_correlation(ByVal p_si As Double) As Double

'this function implements the sublimation temperature computed from the pressure by a
'Clausius-Clapeyron equation as a first guess for the sublimation iteration

Const R = Gas_constant_H2O_si  'specific gas constant of H2O in J/(kg K)
Const q = TP_enthalpy_vap_si - TP_enthalpy_ice_si ' = 2834359.44543354   'Sublimation enthalpy in J/kg
Const Pt = TP_pressure_IAPWS95_si  'IAPWS-95 triple point pressure in Pa
Const Tt = TP_temperature_si

Dim t As Double

aux_temperature_correlation = ErrorReturn

If p_si <= 0 Then Exit Function

t = 1# / Tt - (R / q) * Log(p_si / Pt)
If t <= 0 Then Exit Function

aux_temperature_correlation = 1# / t

End Function

