Attribute VB_Name = "Ice_Flu_5_Mdl"
Option Explicit

'#########################################################################

'This module requires the library module
'     Constants_0_Mdl, file Constants_0.bas

'#########################################################################

'=========================================================================
'This module implements the correlation functions for melting and sublimation of ice,
'as proposed in

'Rainer Feistel:
'Development of New Equations for Melting Pressure and Sublimation Pressure.
'Taskgroup report,
'IAPWS Thermophysical Properties of Water and Steam WG
'Witney, UK, 3 to 8 September 2006

'and

'Revised Release on the Pressure along the Melting and
'Sublimation Curves of Ordinary Water Substance
'The International Association for the Properties of Water and Steam
'Berlin, Germany, September 2008

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98

Private Const Version = "04 Nov 2009"

'==========================================================================
Public Function fit_ice_liq_pressure_si(ByVal t_si As Double) As Double

'this implements the melting pressure in Pa as a function of temperature in K
'determined from a fit of the function ice_liq_pressure_si,
'as defined in eq. (1) of

'Revised Release on the Pressure along the Melting and
'Sublimation Curves of Ordinary Water Substance
'The International Association for the Properties of Water and Steam
'Berlin, Germany, September 2008

Const Pt = TP_pressure_exp_si
Const Tt = TP_temperature_si

Const a1 = 1195393.37
Const b1 = 3#
Const a2 = 80818.3159
Const b2 = 25.75
Const a3 = 3338.2686
Const b3 = 103.75

Dim tr As Double, pr As Double

fit_ice_liq_pressure_si = ErrorReturn

If t_si <= 0 Then Exit Function

tr = t_si / Tt
pr = 1# + a1 * (1# - tr ^ b1) + a2 * (1# - tr ^ b2) + a3 * (1# - tr ^ b3)

fit_ice_liq_pressure_si = Pt * pr

End Function

'==========================================================================
Public Function fit_ice_liq_temperature_si(ByVal p_si As Double) As Double

'this implements the melting temperature of ice in Pa as a function of pressure in Pa
'determined from a fit of the function ice_liq_temperature_si,
'as defined in

'Rainer Feistel:
'Development of New Equations for Melting Pressure and Sublimation Pressure.
'Task group report, IAPWS Thermophysical Properties of Water and Steam WG
'Witney, UK, 3 to 8 September 2006

Const Pt = TP_pressure_exp_si
Const Tt = TP_temperature_si

Const a1 = -1.66356104484551E-07
Const a2 = -2.13519241979406E-13
Const a3 = 3.52967405341877E-20
Const a4 = -2.73184525236281E-26

Dim tr As Double, pr As Double

fit_ice_liq_temperature_si = ErrorReturn

If p_si <= 0 Then Exit Function

pr = p_si / Pt

tr = 1# + a1 * (pr - 1#) + a2 * (pr - 1#) ^ 2 + a3 * (pr - 1#) ^ 3 + a4 * (pr - 1#) ^ 4

fit_ice_liq_temperature_si = Tt * tr

End Function

'==========================================================================
Public Function fit_ice_vap_pressure_si(ByVal t_si As Double) As Double

'this implements the sublimation pressure in Pa as a function of temperature in K
'determined from a fit of the function ice_vap_pressure_si,
'as defined in eq. (6) of

'Revised Release on the Pressure along the Melting and
'Sublimation Curves of Ordinary Water Substance
'The International Association for the Properties of Water and Steam
'Berlin, Germany, September 2008

Const Pt = TP_pressure_exp_si
Const Tt = TP_temperature_si

Const a1 = -21.2144006
Const b1 = 0.00333333333
Const a2 = 27.3203819
Const b2 = 1.20666667
Const a3 = -6.1059813
Const b3 = 1.70333333

Dim tr As Double, pr As Double

fit_ice_vap_pressure_si = ErrorReturn

If t_si <= 0 Then Exit Function

tr = t_si / Tt
pr = a1 * tr ^ b1 + a2 * tr ^ b2 + a3 * tr ^ b3

fit_ice_vap_pressure_si = Pt * Exp(pr / tr)

End Function


