Attribute VB_Name = "Ice_2_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules
'     Constants_0_Mdl, file Constants_0.bas
'     Ice_1_Mdl,     file Ice_1.bas

'#########################################################################

'=========================================================================
'This module implements thermodynamic properties of hexagonal ice I
'as functions of absolute temperature in K and absolute pressure in Pa,
'computed from the Gibbs potential ice_g_si(drv_t, drv_p, t_si, p_si)

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science 2008, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98

Private Const Version = "11 Nov 2009"

'==========================================================================
Public Function ice_chempot_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns    = g(T,P) Gibbs energy = chemical potential in J/kg,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value: ice_chempot_si(270,1E5) = -3786.7496312781

ice_chempot_si = ice_g_si(0, 0, t_si, p_si)

End Function

'==========================================================================
Public Function ice_cp_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   cp(T,P) =  - T*(d2g/dT2)_P isobaric heat capacity in J/(kg K),
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value: ice_cp_si(270,1E5) = 2073.4794621103

Dim g_tt As Double

ice_cp_si = ErrorReturn

g_tt = ice_g_si(2, 0, t_si, p_si)
If g_tt = ErrorReturn Then Exit Function
If g_tt >= 0 Then Exit Function

ice_cp_si = -t_si * g_tt

End Function

'==========================================================================
Public Function ice_density_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   1/(dg/dP)_T density in kg/m^3,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value: ice_density_si(270,1E5) = 917.181167191815

Dim g_p As Double

ice_density_si = ErrorReturn

g_p = ice_g_si(0, 1, t_si, p_si)
If g_p = ErrorReturn Then Exit Function
If g_p <= 0 Then Exit Function

ice_density_si = 1# / g_p

End Function

'==========================================================================
Public Function ice_enthalpy_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   h(T,P) = g - T*(dg/dT)_P enthalpy in J/kg,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value: ice_enthalpy_si(270,1E5) = -339929.55549867

Dim g As Double
Dim g_t As Double

ice_enthalpy_si = ErrorReturn

g = ice_g_si(0, 0, t_si, p_si)
If g = ErrorReturn Then Exit Function
g_t = ice_g_si(1, 0, t_si, p_si)
If g_t = ErrorReturn Then Exit Function

ice_enthalpy_si = g - t_si * g_t

End Function

'==========================================================================
Public Function ice_entropy_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   eta(T,P) = -(dg/dT)_P entropy in J/(kg*K),
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value: ice_entropy_si(270,1E5) = -1244.97335506441

Dim g_t As Double

ice_entropy_si = ErrorReturn

g_t = ice_g_si(1, 0, t_si, p_si)
If g_t = ErrorReturn Then Exit Function

ice_entropy_si = -g_t

End Function

'==========================================================================
Public Function ice_expansion_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   alpha(T,P) = (d2g/dTdP)/(dg/dP)_T cubic thermal expansion coefficient in 1/K,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value: ice_expansion_si(270,1E5) = 1.58309329594484E-04

Dim g_p As Double
Dim g_tp As Double

ice_expansion_si = ErrorReturn

g_p = ice_g_si(0, 1, t_si, p_si)
If g_p = ErrorReturn Then Exit Function
If g_p <= 0 Then Exit Function
g_tp = ice_g_si(1, 1, t_si, p_si)
If g_tp = ErrorReturn Then Exit Function

ice_expansion_si = g_tp / g_p

End Function

'==========================================================================
Public Function ice_helmholtz_energy_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   f(T,P) = g - P*(dg/dP)_T  Helmholtz energy (free energy) in J/kg,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value: ice_helmholtz_energy_si(270,1E5) = -3895.77934490183

Dim g As Double
Dim g_p As Double

ice_helmholtz_energy_si = ErrorReturn

g = ice_g_si(0, 0, t_si, p_si)
If g = ErrorReturn Then Exit Function
g_p = ice_g_si(0, 1, t_si, p_si)
If g_p = ErrorReturn Then Exit Function
If g_p <= 0 Then Exit Function

ice_helmholtz_energy_si = g - p_si * g_p

End Function

'==========================================================================
Public Function ice_internal_energy_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   u(T,P) = g - P*(dg/dP)_T - T*(dg/dT)_P internal energy in J/kg,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value: ice_internal_energy_si(270,1E5) = -340038.585212293

Dim g As Double
Dim g_t As Double
Dim g_p As Double

ice_internal_energy_si = ErrorReturn

g = ice_g_si(0, 0, t_si, p_si)
If g = ErrorReturn Then Exit Function
g_t = ice_g_si(1, 0, t_si, p_si)
If g_t = ErrorReturn Then Exit Function
g_p = ice_g_si(0, 1, t_si, p_si)
If g_p = ErrorReturn Then Exit Function
If g_p <= 0 Then Exit Function

ice_internal_energy_si = g - p_si * g_p - t_si * g_t

End Function

'==========================================================================
Public Function ice_kappa_s_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   -(1/v)(dv/dP)_eta isentropic compressibility in 1/Pa
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value: ice_kappa_s_si(270,1E5) = 1.13667916416195E-10

Dim g_p As Double, g_tt As Double, g_tp As Double, g_pp As Double

g_p = ice_g_si(0, 1, t_si, p_si)
If g_p = ErrorReturn Then Exit Function
If g_p <= 0 Then Exit Function
g_tt = ice_g_si(2, 0, t_si, p_si)
If g_tt = ErrorReturn Then Exit Function
If g_tt >= 0 Then Exit Function
g_tp = ice_g_si(1, 1, t_si, p_si)
If g_tp = ErrorReturn Then Exit Function
g_pp = ice_g_si(0, 2, t_si, p_si)
If g_pp = ErrorReturn Then Exit Function
If g_pp >= 0 Then Exit Function

ice_kappa_s_si = (g_tp ^ 2 - g_tt * g_pp) / (g_p * g_tt)

End Function

'==========================================================================
Public Function ice_kappa_t_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   -(1/v)(dv/dP)_T isothermal compressibility in 1/Pa
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value: ice_kappa_t_si(270,1E5) = 1.17226047281283E-10

Dim g_p As Double
Dim g_pp As Double

ice_kappa_t_si = ErrorReturn

g_p = ice_g_si(0, 1, t_si, p_si)
If g_p = ErrorReturn Then Exit Function
If g_p <= 0 Then Exit Function
g_pp = ice_g_si(0, 2, t_si, p_si)
If g_pp = ErrorReturn Then Exit Function
If g_pp >= 0 Then Exit Function

ice_kappa_t_si = -g_pp / g_p

End Function

'==========================================================================
Public Function ice_lapserate_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   (dT/dP)_eta adiabatic lapse rate in K/Pa
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value: ice_lapserate_si(270,1E5) = 2.24758128545083E-08

Dim g_tt As Double
Dim g_tp As Double

ice_lapserate_si = ErrorReturn

g_tt = ice_g_si(2, 0, t_si, p_si)
If g_tt = ErrorReturn Then Exit Function
If g_tt >= 0 Then Exit Function
g_tp = ice_g_si(1, 1, t_si, p_si)
If g_tp = ErrorReturn Then Exit Function

ice_lapserate_si = -g_tp / g_tt

End Function

'==========================================================================
Public Function ice_p_coefficient_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   (dP/dT)_v  isochoric pressure coefficient in Pa/K
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value: ice_p_coefficient_si(270,1E5) = 1350462.06253651

Dim g_tp As Double
Dim g_pp As Double

ice_p_coefficient_si = ErrorReturn

g_tp = ice_g_si(1, 1, t_si, p_si)
If g_tp = ErrorReturn Then Exit Function
g_pp = ice_g_si(0, 2, t_si, p_si)
If g_pp = ErrorReturn Then Exit Function
If g_pp >= 0 Then Exit Function

ice_p_coefficient_si = -g_tp / g_pp

End Function

'==========================================================================
Public Function ice_specific_volume_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   v(T,P) = (dg/dP)_T specific_volume in m^3/kg,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value: ice_specific_volume_si(270,1E5) = 1.09029713623728E-03

ice_specific_volume_si = ice_g_si(0, 1, t_si, p_si)

End Function

'==========================================================================
Public Function Ice_2_example_call(ByVal t_si As Double, ByVal p_si As Double) As String

Dim CRLF As String, TB As String
Dim txt As String

CRLF = Chr(13) + Chr(10)
TB = Chr(9)

txt = " Implementation of thermodynamic properties of ice Ih in Visual Basic" + CRLF
txt = txt + " for Publication in Ocean Science, 2009" + CRLF
txt = txt + " R. Feistel, IOW, Version " + Version + CRLF
txt = txt + " Compiled on " + CStr(Now) + CRLF + CRLF

If p_si < 0 Then
  txt = txt + "incorrect: negative pressure"
  Ice_2_example_call = txt
  Exit Function
End If

If t_si < 0 Then
  txt = txt + "incorrect: negative temperature"
  Ice_2_example_call = txt
  Exit Function
End If

If p_si > 200000000# Then
  txt = txt + "Warning: pressure > 200 MPa is outside validity" + CRLF + CRLF
End If

If t_si > 273.16 Then
  txt = txt + "Warning: temperature > 273.16 K is outside validity" + CRLF + CRLF
End If

txt = txt + " Absolute temperature:       " + TB + Str(t_si) + TB + "K" + CRLF
txt = txt + " Absolute pressure:          " + TB + Str(p_si) + TB + "Pa" + CRLF + CRLF
txt = txt + " Chemical potential:         " + TB + Str(ice_chempot_si(t_si, p_si)) + TB + "J/kg" + CRLF
txt = txt + " Heat capacity cp:           " + TB + Str(ice_cp_si(t_si, p_si)) + TB + "J/(kg K)" + CRLF
txt = txt + " Density:                    " + TB + Str(ice_density_si(t_si, p_si)) + TB + "kg/m3" + CRLF
txt = txt + " Enthalpy:                   " + TB + Str(ice_enthalpy_si(t_si, p_si)) + TB + "J/kg" + CRLF
txt = txt + " Entropy:                    " + TB + Str(ice_entropy_si(t_si, p_si)) + TB + "J/(kg K)" + CRLF
txt = txt + " Thermal expansion:          " + TB + Str(ice_expansion_si(t_si, p_si)) + TB + "1/K" + CRLF
txt = txt + " Helmholtz energy:           " + TB + Str(ice_helmholtz_energy_si(t_si, p_si)) + TB + "J/kg" + CRLF
txt = txt + " Internal energy:            " + TB + Str(ice_internal_energy_si(t_si, p_si)) + TB + "J/kg" + CRLF
txt = txt + " Adiabatic compressibility:  " + TB + Str(ice_kappa_s_si(t_si, p_si)) + TB + "1/Pa" + CRLF
txt = txt + " Isothermal compressibility: " + TB + Str(ice_kappa_t_si(t_si, p_si)) + TB + "1/Pa" + CRLF
txt = txt + " Adiabatic lapse rate:       " + TB + Str(ice_lapserate_si(t_si, p_si)) + TB + "K/Pa" + CRLF
txt = txt + " Pressure coefficient:       " + TB + Str(ice_p_coefficient_si(t_si, p_si)) + TB + "Pa/K" + CRLF
txt = txt + " Specific volume:            " + TB + Str(ice_specific_volume_si(t_si, p_si)) + TB + "m3/kg" + CRLF + CRLF

txt = txt + "Note: Absolute entropy is defined by IAPWS-95 rather than" + CRLF
txt = txt + "      by Pauling 's residual entropy" + CRLF + CRLF

txt = txt + "Note: Phase boundaries of ice Ih are ignored" + CRLF + CRLF

Ice_2_example_call = txt

End Function

