Attribute VB_Name = "Flu_2_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules
'     Constants_0_Mdl, file Constants_0.bas
'     Flu_1_Mdl,     file Flu_1.bas

'#########################################################################

'=========================================================================
'This module implements thermodynamic properties of fluid water
'as functions of absolute temperature in K and density in kg/m^3,
'computed from the Gibbs potential flu_f_si(drv_t, drv_d, t_si, d_si)

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98

Private Const Version = "11 Nov 2009"

'==========================================================================
Public Function Flu_2_example_call(ByVal t_si As Double, ByVal d_si As Double) As String

Dim CRLF As String, TB As String
Dim txt As String

CRLF = Chr(13) + Chr(10)
TB = Chr(9)

txt = " Implementation of thermodynamic properties of fluid water in Visual Basic" + CRLF
txt = txt + " for Publication in Ocean Science, 2009" + CRLF
txt = txt + " R. Feistel, IOW, Version " + Version + CRLF
txt = txt + " Compiled on " + CStr(Now) + CRLF + CRLF

If d_si <= 0 Then
  txt = txt + "incorrect: negative density"
  Flu_2_example_call = txt
  Exit Function
End If

If t_si <= 0 Then
  txt = txt + "incorrect: negative temperature"
  Flu_2_example_call = txt
  Exit Function
End If

If d_si > 1250# Then
  txt = txt + "Warning: density > 1250 kg m-3 is outside validity" + CRLF + CRLF
End If

If d_si < 0.00000000000001 Then
  txt = txt + "Warning: density < 1E-14 kg m-3 is outside validity" + CRLF + CRLF
End If

If t_si > 1300# Then
  txt = txt + "Warning: temperature > 1300 K is outside validity" + CRLF + CRLF
End If

If t_si < 130# Then
  txt = txt + "Warning: temperature < 130 K is outside validity" + CRLF + CRLF
End If

txt = txt + " Absolute temperature:       " + TB + Str(t_si) + TB + "K" + CRLF
txt = txt + " Density:                    " + TB + Str(d_si) + TB + "kg/m3" + CRLF + CRLF
txt = txt + " Heat capacity cp:           " + TB + Str(flu_cp_si(t_si, d_si)) + TB + "J/(kg K)" + CRLF
txt = txt + " Heat capacity cv:           " + TB + Str(flu_cv_si(t_si, d_si)) + TB + "J/(kg K)" + CRLF
txt = txt + " Enthalpy:                   " + TB + Str(flu_enthalpy_si(t_si, d_si)) + TB + "J/kg" + CRLF
txt = txt + " Entropy:                    " + TB + Str(flu_entropy_si(t_si, d_si)) + TB + "J/(kg K)" + CRLF
txt = txt + " Thermal expansion:          " + TB + Str(flu_expansion_si(t_si, d_si)) + TB + "1/K" + CRLF
txt = txt + " Gibbs energy:               " + TB + Str(flu_gibbs_energy_si(t_si, d_si)) + TB + "J/kg" + CRLF
txt = txt + " Internal energy:            " + TB + Str(flu_internal_energy_si(t_si, d_si)) + TB + "J/kg" + CRLF
txt = txt + " Adiabatic compressibility:  " + TB + Str(flu_kappa_s_si(t_si, d_si)) + TB + "1/Pa" + CRLF
txt = txt + " Isothermal compressibility: " + TB + Str(flu_kappa_t_si(t_si, d_si)) + TB + "1/Pa" + CRLF
txt = txt + " Adiabatic lapse rate:       " + TB + Str(flu_lapserate_si(t_si, d_si)) + TB + "K/Pa" + CRLF
txt = txt + " Absolute pressure:          " + TB + Str(flu_pressure_si(t_si, d_si)) + TB + "Pa" + CRLF
txt = txt + " Sound speed:                " + TB + Str(flu_soundspeed_si(t_si, d_si)) + TB + "m/s" + CRLF + CRLF

txt = txt + "Note: Absolute energy and entropy are defined by IAPWS-95" + CRLF + CRLF

txt = txt + "Note: Phase boundaries of fluid water are ignored" + CRLF + CRLF

Flu_2_example_call = txt

End Function

'==========================================================================
Public Function flu_cp_si(ByVal t_si As Double, ByVal d_si As Double) As Double

'returns   cp(T,D) = T * (deta/dT)_P  isobaric heat capacity in J/(kg K),
't_si      absolute temperature in K,
'd_si      density in kg m-3

'check value: flu_cp_si(300,1000) = 4159.37519963368

Dim f_d As Double
Dim f_tt As Double
Dim f_td As Double
Dim f_dd As Double
Dim x As Double

flu_cp_si = ErrorReturn

f_d = flu_f_si(0, 1, t_si, d_si)
If f_d = ErrorReturn Then Exit Function
f_tt = flu_f_si(2, 0, t_si, d_si)
If f_tt = ErrorReturn Then Exit Function
f_td = flu_f_si(1, 1, t_si, d_si)
If f_td = ErrorReturn Then Exit Function
f_dd = flu_f_si(0, 2, t_si, d_si)
If f_dd = ErrorReturn Then Exit Function

x = 2 * f_d + d_si * f_dd
If x = 0 Then Exit Function

flu_cp_si = t_si * (f_td ^ 2 * d_si / x - f_tt)

End Function

'==========================================================================
Public Function flu_cv_si(ByVal t_si As Double, ByVal d_si As Double) As Double

'returns   cv(T,D) = T * (deta/dT)_D  isochoric heat capacity in J/(kg K),
't_si      absolute temperature in K,
'd_si      density in kg m-3

'check value: flu_cv_si(300,1000) = 4105.20614776059

Dim f_tt As Double

flu_cv_si = ErrorReturn

f_tt = flu_f_si(2, 0, t_si, d_si)
If f_tt = ErrorReturn Then Exit Function

flu_cv_si = -t_si * f_tt

End Function

'==========================================================================
Public Function flu_enthalpy_si(ByVal t_si As Double, ByVal d_si As Double) As Double

'returns   h(T,D) = f - T*(df/dT)_D + D*(df/dD)_T enthalpy in J/kg,
't_si      absolute temperature in K,
'd_si      density in kg m-3

'check value: flu_enthalpy_si(300,1000) = 119752.511434171

Dim f As Double
Dim f_t As Double
Dim f_d As Double

flu_enthalpy_si = ErrorReturn

f = flu_f_si(0, 0, t_si, d_si)
If f = ErrorReturn Then Exit Function
f_t = flu_f_si(1, 0, t_si, d_si)
If f_t = ErrorReturn Then Exit Function
f_d = flu_f_si(0, 1, t_si, d_si)
If f_d = ErrorReturn Then Exit Function

flu_enthalpy_si = f - t_si * f_t + d_si * f_d

End Function

'==========================================================================
Public Function flu_entropy_si(ByVal t_si As Double, ByVal d_si As Double) As Double

'returns   s(T,D) = - (df/dT)_D entropy in J/kg K,
't_si      absolute temperature in K,
'd_si      density in kg m-3

'check value: flu_entropy_si(300,1000) = 390.904170767491

Dim f_t As Double

flu_entropy_si = ErrorReturn

f_t = flu_f_si(1, 0, t_si, d_si)
If f_t = ErrorReturn Then Exit Function

flu_entropy_si = -f_t

End Function

'==========================================================================
Public Function flu_expansion_si(ByVal t_si As Double, ByVal d_si As Double) As Double

'returns   alpha(T,D) = -(1/D) * (dD/dT)_P  thermal expansion in 1/K,
't_si      absolute temperature in K,
'd_si      density in kg m-3

'check value: flu_expansion_si(300,1000) = 2.82413312530867E-04

Dim f_d As Double
Dim f_td As Double
Dim f_dd As Double
Dim x As Double

flu_expansion_si = ErrorReturn

f_d = flu_f_si(0, 1, t_si, d_si)
If f_d = ErrorReturn Then Exit Function
f_td = flu_f_si(1, 1, t_si, d_si)
If f_td = ErrorReturn Then Exit Function
f_dd = flu_f_si(0, 2, t_si, d_si)
If f_dd = ErrorReturn Then Exit Function

x = 2 * f_d + d_si * f_dd
If x = 0 Then Exit Function

flu_expansion_si = f_td / x

End Function

'==========================================================================
Public Function flu_gibbs_energy_si(ByVal t_si As Double, ByVal d_si As Double) As Double

'returns   g(T,D) = f + D*(df/dD)_T   Gibbs energy in J/kg,
't_si      absolute temperature in K,
'd_si      density in kg m-3

'check value: flu_gibbs_energy_si(300,1000) = 2481.26020392421

Dim f As Double
Dim f_d As Double

flu_gibbs_energy_si = ErrorReturn

f = flu_f_si(0, 0, t_si, d_si)
If f = ErrorReturn Then Exit Function
f_d = flu_f_si(0, 1, t_si, d_si)
If f_d = ErrorReturn Then Exit Function

flu_gibbs_energy_si = f + d_si * f_d

End Function

'==========================================================================
Public Function flu_internal_energy_si(ByVal t_si As Double, ByVal d_si As Double) As Double

'returns   u(T,D) = f - T*(df/dT)_D internal energy in J/kg,
't_si      absolute temperature in K,
'd_si      density in kg m-3

'check value: flu_internal_energy_si(300,1000) = 111919.510078207

Dim f As Double
Dim f_t As Double

flu_internal_energy_si = ErrorReturn

f = flu_f_si(0, 0, t_si, d_si)
If f = ErrorReturn Then Exit Function
f_t = flu_f_si(1, 0, t_si, d_si)
If f_t = ErrorReturn Then Exit Function

flu_internal_energy_si = f - t_si * f_t

End Function

'==========================================================================
Public Function flu_kappa_s_si(ByVal t_si As Double, ByVal d_si As Double) As Double

'returns   kappa_s(T,D) = (1/D) * (dD/dP)_eta  isentropic compressibility in 1/Pa,
't_si      absolute temperature in K,
'd_si      density in kg m-3

'check value: flu_kappa_s_si(300,1000) = 4.35960581170918E-10

Dim f_d As Double
Dim f_tt As Double
Dim f_td As Double
Dim f_dd As Double
Dim x As Double

flu_kappa_s_si = ErrorReturn

f_d = flu_f_si(0, 1, t_si, d_si)
If f_d = ErrorReturn Then Exit Function
f_tt = flu_f_si(2, 0, t_si, d_si)
If f_tt = ErrorReturn Then Exit Function
f_td = flu_f_si(1, 1, t_si, d_si)
If f_td = ErrorReturn Then Exit Function
f_dd = flu_f_si(0, 2, t_si, d_si)
If f_dd = ErrorReturn Then Exit Function

x = d_si ^ 2 * (f_tt * (2 * f_d + d_si * f_dd) - d_si * f_td ^ 2)
If x = 0 Then Exit Function

flu_kappa_s_si = f_tt / x

End Function

'==========================================================================
Public Function flu_kappa_t_si(ByVal t_si As Double, ByVal d_si As Double) As Double

'returns   kappa_t(T,D) = (1/D) * (dD/dP)_T  isothermal compressibility in 1/Pa,
't_si      absolute temperature in K,
'd_si      density in kg m-3

'check value: flu_kappa_t_si(300,1000) = 4.41713172024108E-10

Dim f_d As Double
Dim f_dd As Double
Dim x As Double

flu_kappa_t_si = ErrorReturn

f_d = flu_f_si(0, 1, t_si, d_si)
If f_d = ErrorReturn Then Exit Function
f_dd = flu_f_si(0, 2, t_si, d_si)
If f_dd = ErrorReturn Then Exit Function

x = d_si ^ 2 * (2 * f_d + d_si * f_dd)
If x = 0 Then Exit Function

flu_kappa_t_si = 1 / x

End Function

'==========================================================================
Public Function flu_lapserate_si(ByVal t_si As Double, ByVal d_si As Double) As Double

'returns   gamma(T,D) = (dT/dP)_eta  adiabatic lapse rate in K/Pa,
't_si      absolute temperature in K,
'd_si      density in kg m-3

'check value: flu_lapserate_si(300,1000) = 2.0369403983|2526E-08

Dim f_d As Double
Dim f_tt As Double
Dim f_td As Double
Dim f_dd As Double
Dim x As Double

flu_lapserate_si = ErrorReturn

f_d = flu_f_si(0, 1, t_si, d_si)
If f_d = ErrorReturn Then Exit Function
f_tt = flu_f_si(2, 0, t_si, d_si)
If f_tt = ErrorReturn Then Exit Function
f_td = flu_f_si(1, 1, t_si, d_si)
If f_td = ErrorReturn Then Exit Function
f_dd = flu_f_si(0, 2, t_si, d_si)
If f_dd = ErrorReturn Then Exit Function

x = d_si * (f_tt * (2 * f_d + d_si * f_dd) - d_si * f_td ^ 2)
If x = 0 Then Exit Function

flu_lapserate_si = -f_td / x

End Function

'==========================================================================
Public Function flu_pressure_si(ByVal t_si As Double, ByVal d_si As Double) As Double

'returns   P(T,D) = D^2*(df/dD)_T pressure in Pa,
't_si      absolute temperature in K,
'd_si      density in kg m-3

'check value: flu_pressure_si(300,1000) = 7833001.3559|6477

Dim f_d As Double

flu_pressure_si = ErrorReturn

f_d = flu_f_si(0, 1, t_si, d_si)
If f_d = ErrorReturn Then Exit Function

flu_pressure_si = d_si ^ 2 * f_d

End Function

'==========================================================================
Public Function flu_soundspeed_si(ByVal t_si As Double, ByVal d_si As Double) As Double

'returns   c(T,D) = sqr[ (dP/dD)_eta ] sound speed in m/s,
't_si      absolute temperature in K,
'd_si      density in kg m-3

'check value: flu_soundspeed_si(300,1000) = 1514.52479779946

Dim f_d As Double
Dim f_tt As Double
Dim f_td As Double
Dim f_dd As Double
Dim x As Double

flu_soundspeed_si = ErrorReturn

f_d = flu_f_si(0, 1, t_si, d_si)
If f_d = ErrorReturn Then Exit Function
f_tt = flu_f_si(2, 0, t_si, d_si)
If f_tt = ErrorReturn Then Exit Function
f_td = flu_f_si(1, 1, t_si, d_si)
If f_td = ErrorReturn Then Exit Function
f_dd = flu_f_si(0, 2, t_si, d_si)
If f_dd = ErrorReturn Then Exit Function

If f_tt = 0 Then Exit Function
x = 2 * d_si * f_d + d_si ^ 2 * (f_dd - f_td ^ 2 / f_tt)
If x < 0 Then Exit Function

flu_soundspeed_si = Sqr(x)

End Function
