Attribute VB_Name = "Air_3a_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules:
'     Constants_0_Mdl,  file Constants_0.bas
'     Air_1_Mdl,        file Air_1.bas
'     Air_2_Mdl,        file Air_2.bas
'     Maths_0_Mdl,      file Maths_0.bas
'     Convert_Mdl,      file Convert.bas

'#########################################################################

'=========================================================================
'This module implements the Gibbs function of humid air, depending on
'dry-air fraction, temperature and pressure, as well as their partial derivatives,
'computed numerically from the Helmholtz function of humid air.

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009

'Feistel, R., Kretzschmar, H.-J., Span, R., Hagen, E., Wright, D.G., Herrmann, S.:
'Thermodynamic Properties of Sea Air.
'Ocean Science Discussion 6(2009)21932325.
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98

'Control parameters of the density iteration
Private ctrl_initialized As Integer
Private ctrl_mode_air As Integer
Private ctrl_loop_maximum As Long
Private ctrl_density_air As Double
Private ctrl_eps_exit_air As Double
Private ctrl_method_air As Long
Private ctrl_density2_air As Double

Private Const Version = "28 May 2010"

'==========================================================================
Public Function air_g_si(ByVal drv_a As Integer, _
                         ByVal drv_t As Integer, _
                         ByVal drv_p As Integer, _
                         ByVal a_si As Double, _
                         ByVal t_si As Double, _
                         ByVal p_si As Double) As Double

'this implements the Gibbs function of humid air computed from its Helmholtz function
'and its first and second derivatives with respect to the air fraction a_si,
'the absolute temperature t_si and the absolute pressure p_si

'note: this Gibbs function is not designed for liquid or critical air states
'note: the accuracy of this functions depends on the iteration settings of this module

'Output: air_g_si = specific Gibbs energy in J/kg or its derivative

'Input:  drv_a = order of air-fraction derivative, 0 <= drv_a <= 2
'        drv_t = order of temperature derivative, 0 <= drv_t + drv_a <= 2
'        drv_p = order of pressure derivative, 0 <= drv_p + drv_t + drv_a <= 2
'        a_si = mass fraction of dry air in humid air, in kg/kg
'        t_si = absolute temperature, in K
'        p_si = absolute pressure, in Pa

'Check values with default settings, v. 1.0:
'air_g_si( 0, 0, 0, 0.9, 300, 1E5) = 4574.43347101731
'air_g_si( 1, 0, 0, 0.9, 300, 1E5) =-210115.722120426
'air_g_si( 0, 1, 0, 0.9, 300, 1E5) =-911.203092171182
'air_g_si( 0, 0, 1, 0.9, 300, 1E5) = 0.911688234199384
'air_g_si( 2, 0, 0, 0.9, 300, 1E5) = 1415842.71443159
'air_g_si( 1, 1, 0, 0.9, 300, 1E5) = 7566.48354961423
'air_g_si( 1, 0, 1, 0.9, 300, 1E5) =-0.483164035113658
'air_g_si( 0, 2, 0, 0.9, 300, 1E5) =-4.15491953968641
'air_g_si( 0, 1, 1, 0.9, 300, 1E5) = 3.15167187428014E-03
'air_g_si( 0, 0, 2, 0.9, 300, 1E5) =-9.14645118670962E-06

'Check values with default settings, v. 1.1:
'air_g_si( 0, 0, 0, 0.9, 300, 1E5) = 4577.93065688699
'air_g_si( 1, 0, 0, 0.9, 300, 1E5) =-210141.953243099
'air_g_si( 0, 1, 0, 0.9, 300, 1E5) =-911.170080461387
'air_g_si( 0, 0, 1, 0.9, 300, 1E5) = 0.911504137472816
'air_g_si( 2, 0, 0, 0.9, 300, 1E5) = 1415779.23409976
'air_g_si( 1, 1, 0, 0.9, 300, 1E5) = 7566.34779196021
'air_g_si( 1, 0, 1, 0.9, 300, 1E5) =-0.483353002175824
'air_g_si( 0, 2, 0, 0.9, 300, 1E5) =-4.15449972147975
'air_g_si( 0, 1, 1, 0.9, 300, 1E5) = 3.15111222847295E-03
'air_g_si( 0, 0, 2, 0.9, 300, 1E5) =-9.14462130186205E-06

Dim g As Double, d As Double

air_g_si = ErrorReturn

If a_si < 0 Or a_si > 1 Then Exit Function
If p_si <= 0 Then Exit Function

If check_limits = 1 Then
  If (t_si < dry_air_tmin Or t_si > dry_air_tmax) Then Exit Function
  If a_si <> 1 And (t_si < mix_air_tmin Or t_si > mix_air_tmax) Then Exit Function
Else
  If t_si <= 0 Then Exit Function
End If

d = air_density_si(a_si, t_si, p_si)  'numerical inverse function to p = -df/dv

If d = ErrorReturn Then Exit Function
If d <= 0 Then Exit Function

If check_limits = 1 Then
  If d <= dry_air_dmin Or d > dry_air_dmax Then Exit Function
End If

g = air_a_t_p_derivative_si(drv_a, drv_t, drv_p, a_si, t_si, d)
If g = ErrorReturn Then Exit Function

air_g_si = g

End Function

'==========================================================================
Public Function air_density_si(ByVal a_si As Double, _
                               ByVal t_si As Double, _
                               ByVal p_si As Double) As Double

'this function returns humid-air density as a function of air fraction, temperature and pressure
'from numerical iteration of the Helmholtz function derivative, p = - (df/dv)_a,t

'air_density_si: density of humid air in kg/m3

'a_si:  mass fraction of dry air in kg/kg
't_si:  absolute temperature in K
'p_si:  absolute pressure in Pa

'Check value with default settings: air_density_si(0.9, 300, 1E5) = 1.09686619009421   v 1.0
'Check value with default settings: air_density_si(0.9, 300, 1E5) = 1.09708772444253   v 1.1

Dim d As Double, d2 As Double, eps As Double
Dim maxit As Long

air_density_si = ErrorReturn

If a_si < 0 Or a_si > 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

init_it_ctrl_airdensity

'consider the control settings for the iteration
Select Case ctrl_mode_air
  Case 1: d = ctrl_density_air
  Case 0: d = aux_density_ideal_si(a_si, t_si, p_si)
  Case -1: d = aux_density_ideal_si(a_si, t_si, p_si)
  Case Else: Exit Function
End Select

If d = ErrorReturn Then Exit Function
If d <= 0 Then Exit Function

Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: air_density_si = d
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

eps = ctrl_eps_exit_air
If eps = 0 Then Exit Function

If ctrl_method_air > 1 Then
  'specify the second point for Secant or Brent method
  Select Case ctrl_density2_air
    Case 0:  d2 = DensityIteration_Newton(a_si, t_si, p_si, d, 1, 1E+99)
             If d2 = ErrorReturn Then Exit Function
             If ctrl_method_air = 2 Then 'Brent
               d2 = d + 2 * (d2 - d)
             End If
    Case -1: d2 = DensityIteration_Newton(a_si, t_si, p_si, d, 1, 1E+99)
             If d2 = ErrorReturn Then Exit Function
    Case -2: d2 = DensityIteration_Newton(a_si, t_si, p_si, d, 1, 1E+99)
             If d2 = ErrorReturn Then Exit Function
             d2 = d + 0.5 * (d2 - d)
    Case -3: d2 = DensityIteration_Newton(a_si, t_si, p_si, d, 1, 1E+99)
             If d2 = ErrorReturn Then Exit Function
             d2 = d + 2 * (d2 - d)
    Case Is > 0: d2 = ctrl_density2_air
    Case Else: Exit Function
  End Select
End If

'run the iteration
Select Case ctrl_method_air
  Case 0: d = DensityIteration_Newton(a_si, t_si, p_si, d, maxit, eps)
  Case 1: d = DensityIteration_Newton(a_si, t_si, p_si, d, maxit, eps)
  Case 2: d = DensityIteration_Brent(a_si, t_si, p_si, d, d2, maxit, eps)
  Case 3: d = DensityIteration_Secant(a_si, t_si, p_si, d, d2, maxit, eps)
  Case Else: Exit Function
End Select


If d = ErrorReturn Then Exit Function
If d <= 0 Then Exit Function

air_density_si = d

End Function

'==========================================================================
Private Function DensityIteration_Newton(ByVal a As Double, _
                                         ByVal t As Double, _
                                         ByVal p As Double, _
                                         ByVal d As Double, _
                                         ByVal maxit As Integer, _
                                         ByVal eps As Double) As Double

'The function returns the density of humid air as a function of temperature and pressure,
'computed from the Helmholtz function by Newton iteration
'http://en.wikipedia.org/wiki/Newton%27s_method

'output: DensityIteration_Newton: density of humid air in kg/m3
'
'        The value ErrorReturn is returned if
'        - the maximum number of iterations is exceeded without meeting the exit criterion
'        - the function call to air_f_si has returned an error
'        - density has taken a zero or negative value during the iteration

'input:  a: mass fraction of dry air in kg/kg
'        t: absolute temperature in K
'        p: absolute pressure in Pa
'        d: initial guess for density in kg/m3
'    maxit: maximum number of iteration steps to be done
'      eps: required accuracy of density
'           eps > 0: absolute density tolerance in kg/m3
'           eps < 0: relative density tolerance

Dim dd As Double, fd As Double, fdd As Double
Dim it As Long

DensityIteration_Newton = ErrorReturn

If a < 0 Or a > 1 Then Exit Function
If p <= 0 Then Exit Function

If check_limits = 1 Then
  If t < dry_air_tmin Or t > dry_air_tmax Then Exit Function
  If a <> 1 And (t < mix_air_tmin Or t > mix_air_tmax) Then Exit Function
Else
  If t <= 0 Then Exit Function
End If

check_limits = check_limits - 1

For it = 1 To maxit

  fd = air_f_si(0, 0, 1, a, t, d)
  If fd = ErrorReturn Then Exit For
  
  fdd = air_f_si(0, 0, 2, a, t, d)
  If fdd = ErrorReturn Then Exit For
  
  dd = d * (2 * fd + d * fdd)
  If dd <= 0 Then Exit For
  
  dd = (p - d ^ 2 * fd) / dd
  d = d + dd

  If d <= 0 Then Exit For

  If eps > 0 Then                'absolute limit
    If Abs(dd) < eps Then
      DensityIteration_Newton = d
      Exit For
    End If
  Else                           'relative limit
    If Abs(dd) < -eps * d Then
      DensityIteration_Newton = d
      Exit For
    End If
  End If
  
Next it

check_limits = check_limits + 1

End Function

'==========================================================================
Private Function DensityIteration_Brent(ByVal a_si As Double, _
                                        ByVal t As Double, _
                                        ByVal p As Double, _
                                        ByVal d1 As Double, _
                                        ByVal d2 As Double, _
                                        ByVal maxit As Integer, _
                                        ByVal eps As Double) As Double

'The function returns the density of humid air as a function of temperature and pressure,
'computed from the Helmholtz function by Brent iteration
'http://en.wikipedia.org/wiki/Brent's_method

'output: DensityIteration_Brent: density of humid air in kg/m3
'
'        The value ErrorReturn is returned if
'        - the maximum number of iterations is exceeded without meeting the exit criterion
'        - the function call to air_f_si has returned an error
'        - density has taken a zero or negative value during the iteration

'input: a_si: mass fraction of dry air in kg/kg
'          t: absolute temperature in K
'          p: absolute pressure in Pa
'         d1: initial guess for density in kg/m3
'         d2: counterpoint density
'      maxit: maximum number of iteration steps to be done
'        eps: required accuracy of density
'             eps > 0: absolute density tolerance in kg/m3
'             eps < 0: relative density tolerance

Dim a As Double, b As Double, c As Double, d As Double, S As Double
Dim fa As Double, fb As Double, fc As Double, fs As Double
Dim mflag As Boolean
Dim it As Long

DensityIteration_Brent = ErrorReturn

If d1 < d2 Then
  a = d1
  b = d2
Else
  b = d1
  a = d2
End If

If a_si < 0 Or a_si > 1 Then Exit Function
If p <= 0 Then Exit Function

If check_limits = 1 Then
  If t < dry_air_tmin Or t > dry_air_tmax Then Exit Function
  If a <> 1 And (t < mix_air_tmin Or t > mix_air_tmax) Then Exit Function
  If a = 0 Then a = 0.000001
  If b > dry_air_dmax Then b = dry_air_dmax
Else
  If t <= 0 Then Exit Function
End If

check_limits = check_limits - 1

fa = a ^ 2 * air_f_si(0, 0, 1, a_si, t, a) - p
fb = b ^ 2 * air_f_si(0, 0, 1, a_si, t, b) - p
If fa * fb > 0 Then GoTo ExitFunction

If Abs(fa) < Abs(fb) Then
  Swap a, b
  Swap fa, fb
End If

c = a
fc = fa
mflag = True

For it = 1 To maxit

  If fb = 0 Then
    DensityIteration_Brent = b
    Exit For
  End If
  If eps > 0 Then                'absolute limit
    If Abs(a - b) < eps Then
      DensityIteration_Brent = b
      Exit For
    End If
  Else                           'relative limit
    If Abs(a - b) < -eps * b Then
      DensityIteration_Brent = b
      Exit For
    End If
  End If

  If fa = fb Then Exit For

  If fa <> fc And fb <> fc Then
    S = a * fb * fc / ((fa - fb) * (fa - fc)) + _
        b * fa * fc / ((fb - fa) * (fb - fc)) + _
        c * fa * fb / ((fc - fa) * (fc - fb))
  Else
    S = b - (b - a) * fb / (fb - fa)
  End If

  If ((3 * a + b) / 4 - S) * (b - S) > 0 Or _
     (mflag = True And Abs(S - b) >= 0.5 * Abs(b - c)) Or _
     (mflag = False And Abs(S - b) >= 0.5 * (c - d)) Then
    S = 0.5 * (a + b)
  Else
    mflag = False
  End If

  fs = S ^ 2 * air_f_si(0, 0, 1, a_si, t, S) - p
  d = c
  c = b
  fc = fb

  If fa * fs < 0 Then
    b = S
    fb = fs
  Else
    a = S
    fa = fs
  End If

  If Abs(fa) < Abs(fb) Then
    Swap a, b
    Swap fa, fb
  End If

Next it

ExitFunction:
check_limits = check_limits + 1

End Function

'==========================================================================
Private Function DensityIteration_Secant(ByVal a As Double, _
                                         ByVal t As Double, _
                                         ByVal p As Double, _
                                         ByVal d As Double, _
                                         ByVal d2 As Double, _
                                         ByVal maxit As Integer, _
                                         ByVal eps As Double) As Double

'The function returns the density of humid air as a function of temperature and pressure,
'computed from the Helmholtz function by secant iteration
'http://en.wikipedia.org/wiki/Secant_method

'output: DensityIteration_Secant: density of humid air in kg/m3
'
'        The value ErrorReturn is returned if
'        - the maximum number of iterations is exceeded without meeting the exit criterion
'        - the function call to air_f_si has returned an error
'        - density has taken a zero or negative value during the iteration

'input:  a: mass fraction of dry air in kg/kg
'        t: absolute temperature in K
'        p: absolute pressure in Pa
'        d: initial guess for density in kg/m3
'       d2: counterpoint density
'    maxit: maximum number of iteration steps to be done
'      eps: required accuracy of density
'           eps > 0: absolute density tolerance in kg/m3
'           eps < 0: relative density tolerance

Dim dd As Double, d1 As Double, p1 As Double, p2 As Double
Dim it As Long

DensityIteration_Secant = ErrorReturn

If a < 0 Or a > 1 Then Exit Function
If p <= 0 Then Exit Function

If check_limits = 1 Then
  If t < dry_air_tmin Or t > dry_air_tmax Then Exit Function
  If a <> 1 And (t < mix_air_tmin Or t > mix_air_tmax) Then Exit Function
Else
  If t <= 0 Then Exit Function
End If

check_limits = check_limits - 1

p2 = d2 ^ 2 * air_f_si(0, 0, 1, a, t, d2)
If p2 = ErrorReturn Then GoTo ExitFunction

For it = 1 To maxit
  d1 = d2
  p1 = p2
  d2 = d
  p2 = d ^ 2 * air_f_si(0, 0, 1, a, t, d)

  If p2 = ErrorReturn Then Exit For
  If p2 = p1 Then Exit For
  
  dd = -(d2 - d1) * (p2 - p) / (p2 - p1)
  d = d + dd

  If d <= 0 Then Exit For

  If eps > 0 Then                'absolute limit
    If Abs(dd) < eps Then
      DensityIteration_Secant = d
      Exit For
    End If
  Else                           'relative limit
    If Abs(dd) < -eps * d Then
      DensityIteration_Secant = d
      Exit For
    End If
  End If
  
Next it

ExitFunction:
check_limits = check_limits + 1

If check_limits = 1 Then
  If d <= dry_air_dmin Or d > dry_air_dmax Then
    DensityIteration_Secant = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Function air_a_t_p_derivative_si(ByVal drv_a As Integer, _
                                         ByVal drv_t As Integer, _
                                         ByVal drv_p As Integer, _
                                         ByVal a_si As Double, _
                                         ByVal t_si As Double, _
                                         ByVal d_si As Double) As Double

'this function computes a-t-p derivatives of g from a-t-d derivatives of f
'at given air fraction, temperature and density

Dim g As Double, gt As Double, gp As Double
Dim gtt As Double, gtp As Double, gpp As Double

Dim d As Double, n As Double

Dim f As Double, ft As Double, fd As Double
Dim ftt As Double, ftd As Double, fdd As Double
Dim fat As Double, fad As Double, faa As Double

air_a_t_p_derivative_si = ErrorReturn

If drv_a < 0 Then Exit Function
If drv_t < 0 Then Exit Function
If drv_p < 0 Then Exit Function
If drv_a + drv_t + drv_p > 2 Then Exit Function

If a_si < 0 Or a_si > 1 Then Exit Function
If t_si <= 0 Then Exit Function
If d_si <= 0 Then Exit Function

If drv_a + drv_t + drv_p = 2 Then 'compute common denominator of each 2nd derivative
  fd = air_f_si(0, 0, 1, a_si, t_si, d_si)
  If fd = ErrorReturn Then Exit Function
  fdd = air_f_si(0, 0, 2, a_si, t_si, d_si)
  If fdd = ErrorReturn Then Exit Function
  n = 2# * fd + d_si * fdd
  If n = 0 Then Exit Function
  n = 1# / n
End If

g = ErrorReturn

Select Case drv_a
  Case 0:
    Select Case drv_t
      Case 0:
        Select Case drv_p
          Case 0: f = air_f_si(0, 0, 0, a_si, t_si, d_si)
                  If f = ErrorReturn Then Exit Function
                  fd = air_f_si(0, 0, 1, a_si, t_si, d_si)
                  If fd = ErrorReturn Then Exit Function
                  g = f + d_si * fd                          'g
          Case 1: g = 1# / d_si                              'g_p
          Case 2: g = -n / d_si ^ 3                          'g_pp
        End Select
        
      Case 1:
        Select Case drv_p
          Case 0: ft = air_f_si(0, 1, 0, a_si, t_si, d_si)   'g_t
                  If ft = ErrorReturn Then Exit Function
                  g = ft
          Case 1: ftd = air_f_si(0, 1, 1, a_si, t_si, d_si)
                  If ftd = ErrorReturn Then Exit Function
                  g = ftd * n / d_si                         'g_tp
        End Select
        
      Case 2:
        Select Case drv_p
          Case 0: ftt = air_f_si(0, 2, 0, a_si, t_si, d_si)
                  If ftt = ErrorReturn Then Exit Function
                  ftd = air_f_si(0, 1, 1, a_si, t_si, d_si)
                  If ftd = ErrorReturn Then Exit Function
                  g = ftt - d_si * n * ftd ^ 2                'g_tt
        End Select
    End Select

  Case 1:
    Select Case drv_t
      Case 0:
        Select Case drv_p
          Case 0: g = air_f_si(1, 0, 0, a_si, t_si, d_si)     'g_a
          Case 1: fad = air_f_si(1, 0, 1, a_si, t_si, d_si)
                  If fad = ErrorReturn Then Exit Function
                  g = fad * n / d_si                          'g_ap
        End Select
    
      Case 1:
        Select Case drv_p
          Case 0: fat = air_f_si(1, 1, 0, a_si, t_si, d_si)
                  If fat = ErrorReturn Then Exit Function
                  fad = air_f_si(1, 0, 1, a_si, t_si, d_si)
                  If fad = ErrorReturn Then Exit Function
                  ftd = air_f_si(0, 1, 1, a_si, t_si, d_si)
                  If ftd = ErrorReturn Then Exit Function
                  g = fat - d_si * fad * ftd * n               'g_at
        End Select
    End Select

  Case 2:
    Select Case drv_t
      Case 0:
        Select Case drv_p
          Case 0: faa = air_f_si(2, 0, 0, a_si, t_si, d_si)
                  If faa = ErrorReturn Then Exit Function
                  fad = air_f_si(1, 0, 1, a_si, t_si, d_si)
                  If fad = ErrorReturn Then Exit Function
                  g = faa - d_si * fad ^ 2 * n                 'g_aa
        End Select
    End Select
End Select

air_a_t_p_derivative_si = g

End Function

'==========================================================================
Public Sub set_it_ctrl_airdensity(ByVal key As String, ByVal value As Double)

'this sub sets control parameters for the iteration used to compute
'humid-air density from air fraction, temperature and pressure

'key              value
'it_steps         0           set iteration number to default (100)
'it_steps         n > 0       set iteration number to n
'it_steps        -1           do not iterate, use initial value

'init_air_dens    0           use default density to start (ideal gas)
'init_air_dens   -1           use ideal-gas density to start
'init_air_dens    d > 0       use value d as density to start

'tol_air_dens     0           use default exit accuracy for density (0.1 ppm)
'tol_air_dens     eps         use eps as exit accuracy for density (eps < 0 means relative error)

'method_air       0           use default iteration method (now: Newton method) for density
'method_air       1           use Newton method for density
'method_air       2           use Brent method for density
'method_air       3           use secant method for density

'dens2_air        0           use default counterpoint for Brent/Secant method for density
'                             Brent: 2 * Newton step, Secant: 1 * Newton step
'dens2_air       -1           use Newton step as the first counterpoint for density
'dens2_air       -2           use 0.5 * Newton step as the first counterpoint for density
'dens2_air       -3           use 2 * Newton step as the first counterpoint for density
'dens2_air        d > 0       use d as the first counterpoint for density

init_it_ctrl_airdensity

Select Case LCase(Trim(key))

  Case "it_steps":   'iteration steps
    Select Case value
      Case 0:      ctrl_loop_maximum = 100  'default = 100
      Case Is < 0: ctrl_loop_maximum = -1
      Case Else:   ctrl_loop_maximum = value
    End Select

  Case "init_air_dens":   'start density
    Select Case CLng(value)
      Case 0:       ctrl_mode_air = 0    'default = ideal gas
      Case Is < -1: 'Exit Sub
      Case Is < 0:  ctrl_mode_air = value
      Case Else:    ctrl_mode_air = 1
                    ctrl_density_air = value
    End Select

  Case "tol_air_dens":      'required density tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_air = -0.0000001   'default = 0.1 ppm relative
      Case Else:   ctrl_eps_exit_air = value
    End Select

  Case "method_air":
    Select Case value
      Case 0 To 3:   ctrl_method_air = value
    End Select

  Case "dens2_air":
    Select Case value
      Case Is >= -3: ctrl_density2_air = value
    End Select
  
End Select

End Sub

'==========================================================================
Public Function get_it_ctrl_airdensity(ByVal key As String) As Double

'this function returns control parameters as set for the Newton etc. iteration
'method used to compute humid-air density from pressure

init_it_ctrl_airdensity

Select Case LCase(Trim(key))

  Case "it_steps":         'max. iteration steps
    get_it_ctrl_airdensity = ctrl_loop_maximum

  Case "init_air_dens":    'initial humid air density
    If ctrl_mode_air = 1 Then
      get_it_ctrl_airdensity = ctrl_density_air
    Else
      get_it_ctrl_airdensity = ctrl_mode_air
    End If

  Case "tol_air_dens":      'required humid air density tolerance
    get_it_ctrl_airdensity = ctrl_eps_exit_air

  Case "method_air":        'selected iteration method for humid air
    get_it_ctrl_airdensity = ctrl_method_air

  Case "dens2_air":         'counterpoint value for humid air (irrelevant for Newton)
    get_it_ctrl_airdensity = ctrl_density2_air

  Case Else:
    get_it_ctrl_airdensity = ErrorReturn
    
End Select

End Function

'==========================================================================
Private Function aux_density_ideal_si(ByVal a_si As Double, _
                                      ByVal t_si As Double, _
                                      ByVal p_si As Double) As Double

'This function returns the density of ideal-gas humid air as a function of
'air fraction, temperature and pressure

'output:  aux_density_ideal_si: density in kg/m^3

'input:   a_si: mass fraction of dry air in kg/kg
'         t_si: absolute temperature in K
'         p_si: absolute pressure in Pa

Dim ma As Double

Const R = Gas_constant_molar_si        'molar gas constant R in J/(mol K)

aux_density_ideal_si = ErrorReturn

If a_si < 0 Or a_si > 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

ma = air_molar_mass_si(a_si)

aux_density_ideal_si = ma * p_si / (R * t_si)

End Function

'==========================================================================
Private Sub init_it_ctrl_airdensity()

If ctrl_initialized = -1 Then Exit Sub

ctrl_initialized = -1

'Set default values and modes for density iteration
ctrl_mode_air = 0
ctrl_loop_maximum = 100
ctrl_density_air = 1
ctrl_eps_exit_air = -0.0000001 'relative, 0.1 ppm

'Set default values for alternative iteration methods
ctrl_method_air = 0 'default = Newton

'Set default counterpoint values for alternative iteration methods
ctrl_density2_air = 0  'default = .5 * Newton step

End Sub

'==========================================================================
Private Sub Swap(ByRef a As Double, ByRef b As Double)
Dim c As Double
c = a
a = b
b = c
End Sub






