Attribute VB_Name = "Sea_3b_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules:
'     Constants_0_Mdl, file Constants_0.bas
'     Sal_2_Mdl,     file Sal_2.bas
'     Flu_3a_Mdl,    file Flu_3a.bas
'     Sea_3a_Mdl,    file Sea_3a.bas

'#########################################################################

'=========================================================================
'This module implements the enthalpy of seawater depending on salinity, entropy
'and pressure, as well as its partial derivatives, as defined in:

'Feistel, R.
'A Gibbs Function for Seawater Thermodynamics
'for -6 C to 80 C and Salinity up to 120 g kg-1
'Deep-Sea Research I, 55(2008)1639-1671

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98

'Control parameters of the temperature iteration
Private ctrl_initialized As Integer

Private ctrl_mode_pottemp As Integer
Private ctrl_loop_maximum As Long
Private ctrl_init_pottemp As Double
Private ctrl_eps_exit_pottemp As Double


Private Const Version = "05 Aug 2009"

'==========================================================================
Public Function sea_h_si(ByVal drv_s As Long, _
                         ByVal drv_eta As Long, _
                         ByVal drv_p As Long, _
                         ByVal sa_si As Double, _
                         ByVal eta_si As Double, _
                         ByVal p_si As Double) As Double

'this function implements enthalpy as a thermodynamic potential, depending
'on salinity, entropy and pressure

'returns sea_h_si as the S-eta-P derivative

'(d/dS)^drv_s (d/deta)^drv_eta (d/dP)^drv_p h(S,eta,P)

'of the specific enthalpy of seawater, h(S,eta,P), in J/kg

'sa_si     absolute salinity in kg/kg
'eta_si    specific entropy in J/(kg K)
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a
'      and on the iteration settings for temperature by set_it_ctrl_pottemp of this module

'Check values with default settings:
'sea_h_si( 0, 0, 0, 0.035, 500, 1E5) = 145481.970750042
'sea_h_si( 1, 0, 0, 0.035, 500, 1E5) = 86860.798048569
'sea_h_si( 0, 1, 0, 0.035, 500, 1E5) = 309.557955853377
'sea_h_si( 0, 0, 1, 0.035, 500, 1E5) = 9.81092930969333E-04
'sea_h_si( 2, 0, 0, 0.035, 500, 1E5) = 2393730.16716348
'sea_h_si( 1, 1, 0, 0.035, 500, 1E5) = 72.5298236487807
'sea_h_si( 1, 0, 1, 0.035, 500, 1E5) =-6.84629317366511E-04
'sea_h_si( 0, 2, 0, 0.035, 500, 1E5) = 7.72873234085026E-02
'sea_h_si( 0, 1, 1, 0.035, 500, 1E5) = 2.86305358701992E-08
'sea_h_si( 0, 0, 2, 0.035, 500, 1E5) =-3.96880481108341E-13

Dim t As Double

sea_h_si = ErrorReturn

If drv_s < 0 Or drv_s > 2 Then Exit Function
If drv_eta < 0 Or drv_eta > 2 Then Exit Function
If drv_p < 0 Or drv_p > 2 Then Exit Function

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If p_si < 0 Then Exit Function

'compute temperature from entropy.
'if p_si = in-situ pressure, this is in-situ temperature
'if p_si = reference pressure, this is potential temperature
t = sea_temperature_si(sa_si, eta_si, p_si)
If t = ErrorReturn Then Exit Function

sea_h_si = sea_s_eta_p_derivatives_si(drv_s, drv_eta, drv_p, sa_si, t, p_si)

End Function

'==========================================================================
Public Function sea_pottemp_si(ByVal sa_si As Double, _
                               ByVal t_si As Double, _
                               ByVal p_si As Double, _
                               ByVal pr_si As Double) As Double
                               
'this function computes potential temperature of seawater

'returns   273.15 K + theta(S,T,P,Pr) potential temperature of seawater in K,
'sa_si     absolute salinity in kg/kg
't_si      absolute in-situ temperature in K
'p_si      absolute in-situ pressure in Pa
'pr_si     absolute reference pressure in Pa

'Check value with default settings: sea_pottemp_si(0.035,300,1e7,1e5) = 299.771869405324

Dim S As Double

sea_pottemp_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si < 0 Then Exit Function
If p_si < 0 Then Exit Function
If pr_si < 0 Then Exit Function

If p_si = pr_si Then
  sea_pottemp_si = t_si
  Exit Function
End If

S = sea_entropy_si(sa_si, t_si, p_si)  'specific entropy in-situ
If S = ErrorReturn Then Exit Function

sea_pottemp_si = sea_temperature_si(sa_si, S, pr_si)

End Function

'==========================================================================
Public Function sea_potdensity_si(ByVal sa_si As Double, _
                                  ByVal t_si As Double, _
                                  ByVal p_si As Double, _
                                  ByVal pr_si As Double) As Double
                               
'this function computes potential density of seawater

'returns   rho_theta(S,T,P,Pr) potential density of seawater in kg/m,
'sa_si     absolute salinity in kg/kg
't_si      absolute in-situ temperature in K
'p_si      absolute in-situ pressure in Pa
'pr_si     absolute reference pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a
'      and on the iteration settings for temperature by set_it_ctrl_pottemp of this module

'Check value with default settings: sea_potdensity_si(0.035,300,1e7,1e5) = 1022.71520130298

Dim S As Double, v As Double

sea_potdensity_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si < 0 Then Exit Function
If p_si < 0 Then Exit Function
If pr_si < 0 Then Exit Function

S = sea_entropy_si(sa_si, t_si, p_si)  'specific entropy in-situ
If S = ErrorReturn Then Exit Function

v = sea_h_si(0, 0, 1, sa_si, S, pr_si) 'specific volume at reference pressure
If v = ErrorReturn Then Exit Function
If v <= 0 Then Exit Function

sea_potdensity_si = 1 / v

End Function

'==========================================================================
Public Function sea_potenthalpy_si(ByVal sa_si As Double, _
                                   ByVal t_si As Double, _
                                   ByVal p_si As Double, _
                                   ByVal pr_si As Double) As Double
                               
'this function computes potential enthalpy of seawater

'returns   h_theta(S,T,P,Pr) potential enthalpy of seawater in J/kg,
'sa_si     absolute salinity in kg/kg
't_si      absolute in-situ temperature in K
'p_si      absolute in-situ pressure in Pa
'pr_si     absolute reference pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a
'      and on the iteration settings for temperature by set_it_ctrl_pottemp of this module

'Check value with default settings: sea_potenthalpy_si(0.035,300,1e7,1e5) = 106307.996083199

Dim S As Double

sea_potenthalpy_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si < 0 Then Exit Function
If p_si < 0 Then Exit Function
If pr_si < 0 Then Exit Function

S = sea_entropy_si(sa_si, t_si, p_si)  'specific entropy in-situ
If S = ErrorReturn Then Exit Function

sea_potenthalpy_si = sea_h_si(0, 0, 0, sa_si, S, pr_si)

End Function

'==========================================================================
Public Function sea_h_expansion_t_si(ByVal sa_si As Double, _
                                     ByVal eta_si As Double, _
                                     ByVal p_si As Double) As Double
                               
'this function computes thermal expansion coefficient of seawater as function of entropy

'returns   sea_h_expansion_t_si(S,eta,P) = (1/v)*(dv/dt)_s_p thermal expansion coefficient in 1/K,
'                                          at constant salinity and pressure
'sa_si     absolute salinity in kg/kg
'eta_si    entropy in J/(kg K)
'p_si      absolute in-situ pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a
'      and on the iteration settings for temperature by set_it_ctrl_pottemp of this module

'Check value with default settings: sea_h_expansion_t_si(0.035, 500, 1E5) = 3.77581809091213E-04

Dim hp As Double, hep As Double, hee As Double

sea_h_expansion_t_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function

hp = sea_h_si(0, 0, 1, sa_si, eta_si, p_si)
If hp = ErrorReturn Or hp = 0 Then Exit Function
hep = sea_h_si(0, 1, 1, sa_si, eta_si, p_si)
If hep = ErrorReturn Then Exit Function
hee = sea_h_si(0, 2, 0, sa_si, eta_si, p_si)
If hee = ErrorReturn Or hee = 0 Then Exit Function

sea_h_expansion_t_si = hep / (hp * hee)

End Function

'==========================================================================
Public Function sea_h_expansion_theta_si(ByVal sa_si As Double, _
                                         ByVal eta_si As Double, _
                                         ByVal p_si As Double, _
                                         ByVal pref_si As Double) As Double
                               
'this function computes thermal expansion coefficient wrt potential temperature of seawater
'as function of entropy

'returns   sea_h_expansion_theta_si(S,eta,P) = (1/v)*(dv/dtheta)_s_p thermal expansion coefficient in 1/K,
'                                          at constant salinity and pressure
'sa_si     absolute salinity in kg/kg
'eta_si    entropy in J/(kg K)
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a
'      and on the iteration settings for temperature by set_it_ctrl_pottemp of this module

'Check value with default settings: sea_h_expansion_theta_si(0.035, 500, 1E7, 1E5) = 3.84755380181319E-04

Dim hp As Double, hep As Double, hee As Double

sea_h_expansion_theta_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function
If pref_si <= 0 Then Exit Function

hp = sea_h_si(0, 0, 1, sa_si, eta_si, p_si)
If hp = ErrorReturn Or hp = 0 Then Exit Function
hep = sea_h_si(0, 1, 1, sa_si, eta_si, p_si)
If hep = ErrorReturn Then Exit Function
hee = sea_h_si(0, 2, 0, sa_si, eta_si, pref_si)
If hee = ErrorReturn Or hee = 0 Then Exit Function

sea_h_expansion_theta_si = hep / (hp * hee)

End Function

'==========================================================================
Public Function sea_h_expansion_h_si(ByVal sa_si As Double, _
                                     ByVal eta_si As Double, _
                                     ByVal p_si As Double, _
                                     ByVal pref_si As Double) As Double
                               
'this function computes the thermal expansion coefficient wrt potential enthapy of seawater
'as function of entropy

'returns   sea_h_expansion_h_si(S,eta,P) = (1/v)*(dv/dhpot)_s_p thermal expansion coefficient in (kg K) / J,
'                                          at constant salinity and pressure. hpot is potential enthalpy.
'sa_si     absolute salinity in kg/kg
'eta_si    entropy in J/(kg K)
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a
'      and on the iteration settings for temperature by set_it_ctrl_pottemp of this module

'Check value with default settings: sea_h_expansion_h_si(0.035, 500, 1E7, 1E5) = 9.60618615640423E-08

Dim hp As Double, hep As Double, he As Double

sea_h_expansion_h_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function
If pref_si <= 0 Then Exit Function

hp = sea_h_si(0, 0, 1, sa_si, eta_si, p_si)
If hp = ErrorReturn Or hp = 0 Then Exit Function
hep = sea_h_si(0, 1, 1, sa_si, eta_si, p_si)
If hep = ErrorReturn Then Exit Function
he = sea_h_si(0, 1, 0, sa_si, eta_si, pref_si)
If he = ErrorReturn Or he = 0 Then Exit Function

sea_h_expansion_h_si = hep / (hp * he)

End Function

'==========================================================================
Public Function sea_h_contraction_t_si(ByVal sa_si As Double, _
                                       ByVal eta_si As Double, _
                                       ByVal p_si As Double) As Double
                               
'this function computes the adiabatic haline contraction coefficient of seawater

'returns   sea_h_contraction_t_si(S,eta,P) = - (1/v)*(dv/ds)_t_p haline contraction coefficient in kg/kg,
'                                                (kg seawater / kg salt)
'                                               at constant temperature and pressure
'sa_si     absolute salinity in kg/kg
'eta_si    entropy in J/(kg K)
'p_si      absolute in-situ pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a
'      and on the iteration settings for temperature by set_it_ctrl_pottemp of this module

'Check value with default settings: sea_h_contraction_t_si(0.035, 500, 1E5) = 0.725209049048546

Dim hp As Double, hep As Double, hse As Double, hee As Double, hsp As Double

sea_h_contraction_t_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function

hp = sea_h_si(0, 0, 1, sa_si, eta_si, p_si)
If hp = ErrorReturn Or hp = 0 Then Exit Function
hse = sea_h_si(1, 1, 0, sa_si, eta_si, p_si)
If hse = ErrorReturn Then Exit Function
hep = sea_h_si(0, 1, 1, sa_si, eta_si, p_si)
If hep = ErrorReturn Then Exit Function
hsp = sea_h_si(1, 0, 1, sa_si, eta_si, p_si)
If hsp = ErrorReturn Then Exit Function
hee = sea_h_si(0, 2, 0, sa_si, eta_si, p_si)
If hee = ErrorReturn Or hee = 0 Then Exit Function

sea_h_contraction_t_si = (hse * hep - hsp * hee) / (hp * hee)

End Function

'==========================================================================
Public Function sea_h_contraction_theta_si(ByVal sa_si As Double, _
                                           ByVal eta_si As Double, _
                                           ByVal p_si As Double, _
                                           ByVal pref_si As Double) As Double
                               
'this function computes the haline contraction coefficient of seawater at constant potential temperature

'returns   sea_h_contraction_theta_si(S,eta,P) = - (1/v)*(dv/ds)_theta_p haline contraction coefficient in kg/kg,
'                                                (kg seawater / kg salt)
'                                                at constant potential temperature and pressure
'sa_si     absolute salinity in kg/kg
'eta_si    entropy in J/(kg K)
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a
'      and on the iteration settings for temperature by set_it_ctrl_pottemp of this module

'Check value with default settings: sea_h_contraction_theta_si(0.035, 500, 1E7, 1E5) = 0.717342103504727

Dim hp As Double, hep As Double, hse As Double, hee As Double, hsp As Double

sea_h_contraction_theta_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function
If pref_si <= 0 Then Exit Function

hp = sea_h_si(0, 0, 1, sa_si, eta_si, p_si)
If hp = ErrorReturn Or hp = 0 Then Exit Function
hse = sea_h_si(1, 1, 0, sa_si, eta_si, pref_si)
If hse = ErrorReturn Then Exit Function
hep = sea_h_si(0, 1, 1, sa_si, eta_si, p_si)
If hep = ErrorReturn Then Exit Function
hsp = sea_h_si(1, 0, 1, sa_si, eta_si, p_si)
If hsp = ErrorReturn Then Exit Function
hee = sea_h_si(0, 2, 0, sa_si, eta_si, pref_si)
If hee = ErrorReturn Or hee = 0 Then Exit Function

sea_h_contraction_theta_si = (hse * hep - hsp * hee) / (hp * hee)

End Function

'==========================================================================
Public Function sea_h_contraction_h_si(ByVal sa_si As Double, _
                                       ByVal eta_si As Double, _
                                       ByVal p_si As Double, _
                                       ByVal pref_si As Double) As Double
                               
'this function computes the haline contraction coefficient of seawater at constant potential enthalpy

'returns   sea_h_contraction_h_si(S,eta,P) = - (1/v)*(dv/ds)_h_p haline contraction coefficient in kg/kg,
'                                                (kg seawater / kg salt)
'                                               at constant potential enthalpy and pressure
'sa_si     absolute salinity in kg/kg
'eta_si    entropy in J/(kg K)
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a
'      and on the iteration settings for temperature by set_it_ctrl_pottemp of this module

'Check value with default settings: sea_h_contraction_h_si(0.035, 500, 1E7, 1E5) = 0.69777987358974

Dim hp As Double, hep As Double, hs As Double, he As Double, hsp As Double

sea_h_contraction_h_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function
If pref_si <= 0 Then Exit Function

hp = sea_h_si(0, 0, 1, sa_si, eta_si, p_si)
If hp = ErrorReturn Or hp = 0 Then Exit Function
hs = sea_h_si(1, 0, 0, sa_si, eta_si, pref_si)
If hs = ErrorReturn Then Exit Function
hep = sea_h_si(0, 1, 1, sa_si, eta_si, p_si)
If hep = ErrorReturn Then Exit Function
hsp = sea_h_si(1, 0, 1, sa_si, eta_si, p_si)
If hsp = ErrorReturn Then Exit Function
he = sea_h_si(0, 1, 0, sa_si, eta_si, pref_si)
If he = ErrorReturn Or he = 0 Then Exit Function

sea_h_contraction_h_si = (hs * hep - hsp * he) / (hp * he)

End Function

'=========================================================================
Public Function sea_temperature_si(ByVal sa_si As Double, _
                                   ByVal eta_si As Double, _
                                   ByVal p_si As Double) As Double
                                 
'returns   t(S,eta,P) (potential) temperature of seawater in K,
'sa_si     absolute salinity in kg/kg
'eta_si    entropy in J/(kg K),
'p_si      absolute (reference) pressure in Pa

'this is the inverse function to sea_entropy_si(sa_si, t_si, p_si) in Sea_3a

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a and on those made in this module

'Check value with default settings: sea_temperature_si(0.035, 500, 1E5) = 309.557955853377

Dim t As Double, eps As Double, maxit As Long

sea_temperature_si = ErrorReturn

init_it_ctrl_pottemp

Select Case ctrl_mode_pottemp
  Case 0:    t = 273.15 + eta_si / 4000#
  Case Else: t = ctrl_init_pottemp
End Select

Select Case ctrl_loop_maximum
  Case 0:      maxit = 100
  Case -1:     sea_temperature_si = t
               Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else:   Exit Function
End Select

eps = ctrl_eps_exit_pottemp
If eps <= 0 Then Exit Function

'run the iteration
sea_temperature_si = pottemp_iteration(sa_si, eta_si, p_si, t, maxit, eps)

End Function

'==========================================================================
Private Function sea_s_eta_p_derivatives_si(ByVal drv_s As Integer, _
                                            ByVal drv_eta As Integer, _
                                            ByVal drv_p As Integer, _
                                            ByVal sa_si As Double, _
                                            ByVal t_si As Double, _
                                            ByVal p_si As Double) As Double
                                              
'this function computes seawater s-eta-p derivatives of h from s-t-p derivatives of g

Dim g As Double, gt As Double, gp As Double, gs As Double
Dim gss As Double, gst As Double, gsp As Double
Dim gtt As Double, gpp As Double, gtp As Double

Dim h As Double

'in one case we do not at all need to compute the Gibbs function:
If drv_s = 0 And drv_eta = 1 And drv_p = 0 Then
  sea_s_eta_p_derivatives_si = t_si
  Exit Function
End If

sea_s_eta_p_derivatives_si = ErrorReturn

Select Case drv_s

  Case 0:
    Select Case drv_eta
    
      Case 0:
        Select Case drv_p
        
          Case 0: g = sea_g_si(0, 0, 0, sa_si, t_si, p_si)
                  If g = ErrorReturn Then Exit Function
                  gt = sea_g_si(0, 1, 0, sa_si, t_si, p_si)
                  If gt = ErrorReturn Then Exit Function
                  h = g - t_si * gt                          'h
          
          Case 1: gp = sea_g_si(0, 0, 1, sa_si, t_si, p_si)
                  If gp = ErrorReturn Then Exit Function
                  h = gp                                      'dh/dp
                  
          Case 2: gtt = sea_g_si(0, 2, 0, sa_si, t_si, p_si)
                  If gtt = ErrorReturn Then Exit Function
                  If gtt = 0 Then Exit Function
                  gtp = sea_g_si(0, 1, 1, sa_si, t_si, p_si)
                  If gtp = ErrorReturn Then Exit Function
                  gpp = sea_g_si(0, 0, 2, sa_si, t_si, p_si)
                  If gpp = ErrorReturn Then Exit Function
                  h = (gtt * gpp - gtp ^ 2) / gtt              'd2h/dp2
                  
          Case Else: Exit Function
        End Select
        
      Case 1:
        Select Case drv_p
          Case 0: h = t_si                                     'dh/deta, has already been handled initially
          
          Case 1: gtt = sea_g_si(0, 2, 0, sa_si, t_si, p_si)
                  If gtt = ErrorReturn Then Exit Function
                  If gtt = 0 Then Exit Function
                  gtp = sea_g_si(0, 1, 1, sa_si, t_si, p_si)
                  If gtp = ErrorReturn Then Exit Function
                  h = -gtp / gtt                               'd2h/deta dp
                  
          Case Else: Exit Function
        End Select

      Case 2:
        Select Case drv_p
          Case 0: gtt = sea_g_si(0, 2, 0, sa_si, t_si, p_si)
                  If gtt = ErrorReturn Then Exit Function
                  If gtt = 0 Then Exit Function
                  h = -1# / gtt                                'd2h/deta2
                  
          Case Else: Exit Function
        End Select

      Case Else: Exit Function
    End Select
    
  Case 1:
    Select Case drv_eta
    
      Case 0:
        Select Case drv_p
        
          Case 0: gs = sea_g_si(1, 0, 0, sa_si, t_si, p_si)
                  If gs = ErrorReturn Then Exit Function
                  h = gs                                        'dh/ds
          
          Case 1: gtt = sea_g_si(0, 2, 0, sa_si, t_si, p_si)
                  If gtt = ErrorReturn Then Exit Function
                  If gtt = 0 Then Exit Function
                  gst = sea_g_si(1, 1, 0, sa_si, t_si, p_si)
                  If gst = ErrorReturn Then Exit Function
                  gsp = sea_g_si(1, 0, 1, sa_si, t_si, p_si)
                  If gsp = ErrorReturn Then Exit Function
                  gtp = sea_g_si(0, 1, 1, sa_si, t_si, p_si)
                  If gtp = ErrorReturn Then Exit Function
                  h = (gtt * gsp - gst * gtp) / gtt             'd2h/dsdp
          
          Case Else: Exit Function
        End Select

      Case 1:
        Select Case drv_p
        
          Case 0: gtt = sea_g_si(0, 2, 0, sa_si, t_si, p_si)
                  If gtt = ErrorReturn Then Exit Function
                  If gtt = 0 Then Exit Function
                  gst = sea_g_si(1, 1, 0, sa_si, t_si, p_si)
                  If gst = ErrorReturn Then Exit Function
                  h = -gst / gtt                                'd2h/dsdeta
          
          Case Else: Exit Function
        End Select

      Case Else: Exit Function
    End Select

  
  Case 2:
    Select Case drv_eta
    
      Case 0:
        Select Case drv_p
        
          Case 0: gtt = sea_g_si(0, 2, 0, sa_si, t_si, p_si)
                  If gtt = ErrorReturn Then Exit Function
                  If gtt = 0 Then Exit Function
                  gst = sea_g_si(1, 1, 0, sa_si, t_si, p_si)
                  If gst = ErrorReturn Then Exit Function
                  gss = sea_g_si(2, 0, 0, sa_si, t_si, p_si)
                  If gss = ErrorReturn Then Exit Function
                  h = (gtt * gss - gst ^ 2) / gtt               'd2h/ds2
          
          Case Else: Exit Function
        End Select

      Case Else: Exit Function
    End Select

  Case Else: Exit Function
End Select

sea_s_eta_p_derivatives_si = h

End Function

'==========================================================================
Private Sub init_it_ctrl_pottemp()

If ctrl_initialized = -1 Then Exit Sub

ctrl_initialized = -1

'Set default values and modes for temperature iteration
ctrl_loop_maximum = 100
ctrl_mode_pottemp = 0           'default: theta = t0 + 273*eta/4000
ctrl_init_pottemp = 273.15
ctrl_eps_exit_pottemp = 0.0001  'default = 0.1 mK

End Sub

'=========================================================================
Private Function pottemp_iteration(ByVal sa_si As Double, _
                                   ByVal eta_si As Double, _
                                   ByVal p_si As Double, _
                                   ByVal t_si As Double, _
                                   ByVal maxit As Long, _
                                   ByVal eps As Double) As Double

'returns   theta =  potential temperature of seawater in K,
'          i.e. the temperature that solves eta_si = sea_entropy_si(sa_si, t_si, p_si)

'sa_si     absolute salinity in kg/kg
'eta_si    entropy in J/(kg K)
'p_si      absolute (reference) pressure in Pa
't_si      absolute (potential) temperature in K, initial value
'maxit     max. number of iterations
'eps       required tolerance in K

Dim i As Long
Dim d As Double, S As Double, theta As Double, cp As Double, dt As Double

pottemp_iteration = ErrorReturn

If eps <= 0 Then Exit Function
If maxit <= 0 Then Exit Function

If check_limits <> 1 Then
  If sa_si < 0 Or sa_si > 1 Or _
     t_si <= 0 Or p_si <= 0 Then Exit Function
Else
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Then Exit Function
  'SAL_LIMITS
  If t_si < sal_tmin Or t_si > sal_tmax Or _
     sa_si > sal_smax Or _
     p_si < sal_pmin Or p_si > sal_pmax Then Exit Function
End If

check_limits = check_limits - 1

theta = t_si
For i = 1 To maxit

  'get entropy and its first derivative for Newton iteration
  S = sea_entropy_si(sa_si, theta, p_si)
  cp = sea_cp_si(sa_si, theta, p_si)
    
  If S = ErrorReturn Then Exit For
  If cp = ErrorReturn Then Exit For
  If cp <= 0 Then Exit For
  
  'next temperature improvement step
  dt = theta * (eta_si - S) / cp
  theta = theta + dt
  If theta <= 0 Then Exit For
  
  If Abs(dt) < eps Then
    pottemp_iteration = theta
    Exit For
  End If
  
Next i

check_limits = check_limits + 1

End Function

'==========================================================================
Public Sub set_it_ctrl_pottemp(ByVal key As String, ByVal value As Double)

'this sub sets control parameters for the Newton iteration used to compute
'potential temperature from reference pressure

'key             value
'it_steps        0           set iteration number to default (100)
'it_steps        n > 0       set iteration number to n
'it_steps       -1           do not iterate, use initial value

'init_theta      0           use default potential temperature theta = t0 + 273*eta/4000 to start
'init_theta      t > 0       use value t as potential temperature to start

'tol_theta       0           use default exit accuracy for potential temperature (0.1 mK)
'tol_theta       eps > 0     use eps as exit accuracy for potential temperature

init_it_ctrl_pottemp

Select Case LCase(Trim(key))

  Case "it_steps":   'iteration steps
    Select Case value
      Case 0:      ctrl_loop_maximum = 100     'default = 100
      Case Is < 0: ctrl_loop_maximum = -1
      Case Else:   ctrl_loop_maximum = value
    End Select

  Case "init_theta":   'start theta
    Select Case CLng(value)
      Case 0:      ctrl_mode_pottemp = 0        'default: theta = t0 + eta/4000
      Case Is > 0: ctrl_mode_pottemp = 1
                   ctrl_init_pottemp = value
    End Select

  Case "tol_theta":      'required theta tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_pottemp = 0.0001 'default = 0.1 mK
      Case Is > 0: ctrl_eps_exit_pottemp = value
    End Select

End Select

End Sub







