Attribute VB_Name = "Sal_2_mdl"
Option Explicit

'#########################################################################

'This module requires the library modules
'     Constants_0_Mdl, file Constants_0.bas
'     Sal_1_Mdl,     file Sal_1.bas

'#########################################################################

'=========================================================================
'This module implements the saline part of the Gibbs function of seawater,
'sal_g_si(drv_s, drv_t, drv_p, sa_si, t_si, p_si) as well as its 1st and 2n
'partial derivatives with respect to salinity, temperature and pressure,
'as well as thermodynamic properties of sea salt in seawater as functions
'of absolute salinity in kg/kg, absolute temperature in K and absolute pressure in Pa,
'computed from the salinity expansion terms of the Gibbs potential,
'as defined in IAPWS-08:

'Release on the IAPWS Formulation 2008 for the Thermodynamic Properties of Seawater
'The International Association for the Properties of Water and Steam
'Berlin, Germany, September 2008

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

Private Const maxs = 7      'number of salinity expansion terms gi

'Private Const ErrorReturn = 9.99999999E+98

Private Const Version = "05 Nov 2009"

'==========================================================================
Public Function sal_g_si(ByVal drv_s As Integer, _
                         ByVal drv_t As Integer, _
                         ByVal drv_p As Integer, _
                         ByVal sa_si As Double, _
                         ByVal t_si As Double, _
                         ByVal p_si As Double) As Double

'returns the S-T-P derivative (d/dS)^drv_s (d/dT)^drv_t (d/dP)^drv_p gS(S,T,P)
'of the saline Gibbs function gS,
'
' gS(S,T,P) = g1(T,P)*S*ln(S) + sum(i>1) gi(T,P)*S^(i/2)
'
' sa_si = S = absolute salinity in kg/kg, i.e. the mass fraction of salt in seawater
' t_si  = T = absolute temperature in K, ITS-90
' p_si  = P = absolute pressure in Pa
' sal_g_si = gS = specific Gibbs energy in J/kg, relative to pure water

'check values:
'sal_g_si(0,0,0,0.035,300,1E6)= 127.03364030915
'sal_g_si(1,0,0,0.035,300,1E6)= 77949.2100394602
'sal_g_si(0,1,0,0.035,300,1E6)= 18.636040707255
'sal_g_si(0,0,1,0.035,300,1E6)= -2.55600080318783E-05
'sal_g_si(2,0,0,0.035,300,1E6)= 2248200.54659641
'sal_g_si(1,1,0,0.035,300,1E6)= 790.563810558295
'sal_g_si(1,0,1,0.035,300,1E6)= -7.15686520587585E-04
'sal_g_si(0,2,0,0.035,300,1E6)= 0.597842170749115
'sal_g_si(0,1,1,0.035,300,1E6)= 2.85865076268776E-08
'sal_g_si(0,0,2,0.035,300,1E6)= 4.09543164904626E-14

Dim g As Double, sqr_s As Double
Dim gi(maxs) As Double
Dim i As Integer

sal_g_si = ErrorReturn

If drv_s < 0 Or drv_s > 3 Then Exit Function
If drv_t < 0 Then Exit Function
If drv_p < 0 Then Exit Function

If check_limits <> 1 Then
  If sa_si < 0 Or sa_si >= 1 Then Exit Function
  If t_si <= 0 Then Exit Function
  If p_si <= 0 Then Exit Function
Else
  'SAL_LIMITS
  If t_si < sal_tmin Or t_si > sal_tmax Or _
     sa_si < sal_smin Or sa_si > sal_smax Or _
     p_si < sal_pmin Or p_si > sal_pmax Then Exit Function
End If

'compute the limit S = 0:
If sa_si = 0 Then

  If drv_s = 0 Then         'if no S-derivative taken, each term of gS vanishes
    sal_g_si = 0
    Exit Function
  End If

  If (drv_p > 0 Or drv_t > 1) And drv_s = 1 Then          'if the log term is gone, the 1st S-derivative
    sal_g_si = sal_g_term_si(2, drv_t, drv_p, t_si, p_si) 'returns the linear S term
    Exit Function
  End If

  Exit Function            'all other cases have a singularity at S = 0
End If


'compute the cases S > 0:
For i = 1 To maxs
  gi(i) = sal_g_term_si(i, drv_t, drv_p, t_si, p_si)  'get all coefficients of the salinity expansion
  If gi(i) = ErrorReturn Then Exit Function
Next i

g = 0
If gi(1) <> 0 Then    'take the required S-derivative of the log term
  Select Case drv_s
    Case 0: g = gi(1) * sa_si * Log(sa_si)
    Case 1: g = gi(1) * (Log(sa_si) + 1)
    Case 2: g = gi(1) / sa_si
    Case 3: g = -gi(1) / sa_si ^ 2
  End Select
End If

sqr_s = Sqr(sa_si)

For i = 2 To maxs    'add the S-derivatives of each root(S) term
  If gi(i) <> 0 Then
    Select Case drv_s
      Case 0: g = g + gi(i) * sqr_s ^ i
      Case 1: g = g + gi(i) * (i / 2) * sqr_s ^ (i - 2)
      Case 2: g = g + gi(i) * (i / 2) * (i / 2 - 1) * sqr_s ^ (i - 4)
      Case 3: g = g + gi(i) * (i / 2) * (i / 2 - 1) * (i / 2 - 2) * sqr_s ^ (i - 6)
    End Select
  End If
Next i

sal_g_si = g

End Function

'==========================================================================
Public Function sal_act_coeff_si(ByVal sa_si As Double, _
                                 ByVal t_si As Double, _
                                 ByVal p_si As Double) As Double

'returns the mean activity coefficient, ln(gamma), of seawater

'sa_si:   absolute salinity in kg/kg
't_si:    absolute temperature in K
'p_si:    absolute pressure in Pa

'check value: sal_act_coeff_si(0.035,300,1E6) = -0.527003008913238

Dim lng As Double, sqr_s As Double
Dim gi(maxs) As Double
Dim i As Integer

sal_act_coeff_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

If sa_si = 0 Then
  sal_act_coeff_si = 0
  Exit Function
End If

'compute the cases S > 0:
For i = 1 To maxs
  If i <> 2 Then
    gi(i) = sal_g_term_si(i, 0, 0, t_si, p_si)  'get all coefficients of the salinity expansion
    If gi(i) = ErrorReturn Then Exit Function
  End If
Next i

sqr_s = Sqr(sa_si)
lng = 0

For i = 3 To maxs
  lng = lng + gi(i) * (sa_si * (1# - 0.5 * i) + 0.5 * i) * sqr_s ^ (i - 2)
Next i

sal_act_coeff_si = lng / gi(1) + Log(1# - sa_si) - sa_si

End Function

'==========================================================================
Public Function sal_act_potential_si(ByVal sa_si As Double, _
                                     ByVal t_si As Double, _
                                     ByVal p_si As Double) As Double

'returns the activity potential, psi, of seawater

'sa_si:   absolute salinity in kg/kg
't_si:    absolute temperature in K
'p_si:    absolute pressure in Pa

'check value: sal_act_potential_si(0.035,300,1E6) = -0.429940465498403

Dim psi As Double, sqr_s As Double
Dim gi(maxs) As Double
Dim i As Integer

sal_act_potential_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

If sa_si = 0 Then
  sal_act_potential_si = 0
  Exit Function
End If

'compute the cases S > 0:
For i = 1 To maxs
  If i <> 2 Then
    gi(i) = sal_g_term_si(i, 0, 0, t_si, p_si)  'get all coefficients of the salinity expansion
    If gi(i) = ErrorReturn Then Exit Function
  End If
Next i

sqr_s = Sqr(sa_si)
psi = 0

For i = 3 To maxs
  psi = psi + gi(i) * sqr_s ^ (i - 2)
Next i

sal_act_potential_si = psi / gi(1) + Log(1# - sa_si)

End Function

'==========================================================================
Public Function sal_activity_w_si(ByVal sa_si As Double, _
                                  ByVal t_si As Double, _
                                  ByVal p_si As Double) As Double

'returns the activity of water in seawater, a_w
'for seawater with Reference Composition, see Millero et al. Deep-Sea Research I, 55(2008) 50-72

'sa_si:   absolute salinity in kg/kg
't_si:    absolute temperature in K
'p_si:    absolute pressure in Pa

'check value: sal_activity_w_si(0.035,300,1E6) = 0.981388410188134

Dim phi As Double

If sa_si = 0 Then
  sal_activity_w_si = 1#
  Exit Function
End If

phi = sal_osm_coeff_si(sa_si, t_si, p_si)

If phi = ErrorReturn Then
  sal_activity_w_si = ErrorReturn
  Exit Function
End If

sal_activity_w_si = Exp(-phi * sa_si / (1# - sa_si) * Molar_mass_H2O_si / Molar_mass_seasalt_si)

End Function

'==========================================================================
Public Function sal_dilution_si(ByVal sa_si As Double, _
                                ByVal t_si As Double, _
                                ByVal p_si As Double) As Double

'returns the dilution coefficient of seawater, D = S * (d2g/dS2)_T_P, in J/kg

'sa_si:   absolute salinity in kg/kg
't_si:    absolute temperature in K
'p_si:    absolute pressure in Pa

'check value: sal_dilution_si(0.035,300,1E6) = 78687.0191308744

Dim d As Double, sqr_s As Double
Dim gi(maxs) As Double
Dim i As Integer

sal_dilution_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

If sa_si = 0 Then
  sal_dilution_si = sal_g_term_si(1, 0, 0, t_si, p_si)
End If

For i = 1 To maxs
  If i <> 2 Then
    gi(i) = sal_g_term_si(i, 0, 0, t_si, p_si)  'get all coefficients of the salinity expansion
    If gi(i) = ErrorReturn Then Exit Function
  End If
Next i

sqr_s = Sqr(sa_si)
d = gi(1)

For i = 3 To maxs
  d = d + gi(i) * 0.25 * i * (i - 2) * sqr_s ^ (i - 2)
Next i

sal_dilution_si = d

End Function

'==========================================================================
Public Function sal_mixenthalpy_si(ByVal sa1_si As Double, _
                                   ByVal sa2_si As Double, _
                                   ByVal w1 As Double, _
                                   ByVal t_si As Double, _
                                   ByVal p_si As Double) As Double

'returns the mixing enthalpy of seawater, h_mix, in J/kg

'sa1_si:  salinity of component 1, in kg/kg
'sa2_si:  salinity of component 2, in kg/kg
'w1:      mass fraction of component 1, w1 = m1 / (m1 + m2)
't_si:    absolute temperature in K
'p_si:    absolute pressure in Pa

'check value: sal_mixenthalpy_si(0.01,0.035,0.6,300,1E6) = 16.1539911284051

Dim h As Double, h1 As Double, h2 As Double
Dim S As Double, s1 As Double, s2 As Double
Dim s12 As Double, w2 As Double

sal_mixenthalpy_si = ErrorReturn

If sa1_si < 0 Or sa1_si >= 1 Then Exit Function
If sa2_si < 0 Or sa2_si >= 1 Then Exit Function
If w1 < 0 Or w1 > 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

w2 = 1 - w1
s12 = w1 * sa1_si + w2 * sa2_si
h = sal_g_si(0, 0, 0, s12, t_si, p_si)
If h = ErrorReturn Then Exit Function
S = -sal_g_si(0, 1, 0, s12, t_si, p_si)
If S = ErrorReturn Then Exit Function
h = h + t_si * S

h1 = sal_g_si(0, 0, 0, sa1_si, t_si, p_si)
If h1 = ErrorReturn Then Exit Function
s1 = -sal_g_si(0, 1, 0, sa1_si, t_si, p_si)
If s1 = ErrorReturn Then Exit Function
h1 = h1 + t_si * s1

h2 = sal_g_si(0, 0, 0, sa2_si, t_si, p_si)
If h2 = ErrorReturn Then Exit Function
s2 = -sal_g_si(0, 1, 0, sa2_si, t_si, p_si)
If s2 = ErrorReturn Then Exit Function
h2 = h2 + t_si * s2

sal_mixenthalpy_si = h - w1 * h1 - w2 * h2

End Function

'==========================================================================
Public Function sal_mixentropy_si(ByVal sa1_si As Double, _
                                  ByVal sa2_si As Double, _
                                  ByVal w1 As Double, _
                                  ByVal t_si As Double, _
                                  ByVal p_si As Double) As Double

'returns the mixing entropy of seawater, eta_mix, in J/(kg K)

'sa1_si:  salinity of component 1, in kg/kg
'sa2_si:  salinity of component 2, in kg/kg
'w1:      mass fraction of component 1, w1 = m1 / (m1 + m2)
't_si:    absolute temperature in K
'p_si:    absolute pressure in Pa

'check value: sal_mixentropy_si(0.01,0.035,0.6,300,1E6) = 0.96682942261|6511

Dim S As Double, s1 As Double, s2 As Double
Dim s12 As Double, w2 As Double

sal_mixentropy_si = ErrorReturn

If sa1_si < 0 Or sa1_si >= 1 Then Exit Function
If sa2_si < 0 Or sa2_si >= 1 Then Exit Function
If w1 < 0 Or w1 > 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

w2 = 1 - w1
s12 = w1 * sa1_si + w2 * sa2_si
S = -sal_g_si(0, 1, 0, s12, t_si, p_si)
If S = ErrorReturn Then Exit Function

s1 = -sal_g_si(0, 1, 0, sa1_si, t_si, p_si)
If s1 = ErrorReturn Then Exit Function

s2 = -sal_g_si(0, 1, 0, sa2_si, t_si, p_si)
If s2 = ErrorReturn Then Exit Function

sal_mixentropy_si = S - w1 * s1 - w2 * s2

End Function

'==========================================================================
Public Function sal_mixvolume_si(ByVal sa1_si As Double, _
                                 ByVal sa2_si As Double, _
                                 ByVal w1 As Double, _
                                 ByVal t_si As Double, _
                                 ByVal p_si As Double) As Double

'returns the mixing volume of seawater, v_mix, in m^3/kg

'sa1_si:  salinity of component 1, in kg/kg
'sa2_si:  salinity of component 2, in kg/kg
'w1:      mass fraction of component 1, w1 = m1 / (m1 + m2)
't_si:    absolute temperature in K
'p_si:    absolute pressure in Pa

'check value: sal_mixvolume_si(0.01,0.035,0.6,300,1E6) = -5.94174956891818E-08

Dim v As Double, v1 As Double, v2 As Double
Dim s12 As Double, w2 As Double

sal_mixvolume_si = ErrorReturn

If sa1_si < 0 Or sa1_si >= 1 Then Exit Function
If sa2_si < 0 Or sa2_si >= 1 Then Exit Function
If w1 < 0 Or w1 > 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

w2 = 1 - w1
s12 = w1 * sa1_si + w2 * sa2_si
v = sal_g_si(0, 0, 1, s12, t_si, p_si)
If v = ErrorReturn Then Exit Function

v1 = sal_g_si(0, 0, 1, sa1_si, t_si, p_si)
If v1 = ErrorReturn Then Exit Function

v2 = sal_g_si(0, 0, 1, sa2_si, t_si, p_si)
If v2 = ErrorReturn Then Exit Function

sal_mixvolume_si = v - w1 * v1 - w2 * v2

End Function

'==========================================================================
Public Function sal_osm_coeff_si(ByVal sa_si As Double, _
                                 ByVal t_si As Double, _
                                 ByVal p_si As Double) As Double

'returns the osmotic coefficient of seawater, phi

'sa_si:   absolute salinity in kg/kg
't_si:    absolute temperature in K
'p_si:    absolute pressure in Pa

'check value: sal_osm_coeff_si(0.035,300,1E6) = 0.902937456585165

Dim phi As Double, sqr_s As Double
Dim gi(maxs) As Double
Dim i As Integer

sal_osm_coeff_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

If sa_si = 0 Then
  sal_osm_coeff_si = 1#
  Exit Function
End If

'compute the cases S > 0:
For i = 1 To maxs
  If i <> 2 Then
    gi(i) = sal_g_term_si(i, 0, 0, t_si, p_si)  'get all coefficients of the salinity expansion
    If gi(i) = ErrorReturn Then Exit Function
  End If
Next i

sqr_s = Sqr(sa_si)
phi = 0

For i = 3 To maxs
  phi = phi + gi(i) * (0.5 * i - 1#) * sqr_s ^ (i - 2)
Next i

sal_osm_coeff_si = phi * (1# - sa_si) / gi(1) + 1# - sa_si

End Function

'==========================================================================
Public Function sal_chempot_h2o_si(ByVal sa_si As Double, _
                                   ByVal t_si As Double, _
                                   ByVal p_si As Double) As Double

'returns the saline part of chemical potential of water in seawater, _WS, in J/kg

'sa_si:   absolute salinity in kg/kg
't_si:    absolute temperature in K
'p_si:    absolute pressure in Pa

'check value: sal_chempot_h2o_si(0.035,300,1E6) = -2601.18871107196

Dim g As Double, sqr_s As Double
Dim gi(maxs) As Double
Dim i As Integer

sal_chempot_h2o_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

For i = 1 To maxs
  If i <> 2 Then
    gi(i) = sal_g_term_si(i, 0, 0, t_si, p_si)  'get all coefficients of the salinity expansion
    If gi(i) = ErrorReturn Then Exit Function
  End If
Next i

sqr_s = Sqr(sa_si)
g = gi(1)

For i = 3 To maxs
  g = g + gi(i) * (0.5 * i - 1#) * sqr_s ^ (i - 2)
Next i

sal_chempot_h2o_si = -g * sa_si

End Function

'==========================================================================
Public Function sal_chempot_rel_si(ByVal sa_si As Double, _
                                   ByVal t_si As Double, _
                                   ByVal p_si As Double) As Double

'returns the relative chemical potential of seawater, , in J/kg

'sa_si:   absolute salinity in kg/kg
't_si:    absolute temperature in K
'p_si:    absolute pressure in Pa

'check value: sal_chempot_rel_si(0.035,300,1E6) = 77949.2100394602

sal_chempot_rel_si = ErrorReturn

If sa_si <= 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

sal_chempot_rel_si = sal_g_si(1, 0, 0, sa_si, t_si, p_si)

End Function

'==========================================================================
Public Function sal_chem_coeff_si(ByVal sa_si As Double, _
                                  ByVal t_si As Double, _
                                  ByVal p_si As Double) As Double

'returns the chemical coefficient of seawater, DS, in J/kg

'sa_si:   absolute salinity in kg/kg
't_si:    absolute temperature in K
'p_si:    absolute pressure in Pa

'check value: sal_chem_coeff_si(0.035,300,1E6) = 2754.04566958061

Dim d As Double

sal_chem_coeff_si = ErrorReturn

d = sal_dilution_si(sa_si, t_si, p_si)
If d = ErrorReturn Then Exit Function

sal_chem_coeff_si = sa_si * d

End Function

'==========================================================================
Public Function sal_saltenthalpy_si(ByVal sa_si As Double, _
                                    ByVal t_si As Double, _
                                    ByVal p_si As Double) As Double

'returns the specific enthalpy of seasalt in seawater, h_S, in J/kg

'sa_si:   absolute salinity in kg/kg
't_si:    absolute temperature in K
'p_si:    absolute pressure in Pa

'check value: sal_saltenthalpy_si(0.035,300,1E6) = -156107.95919621

Dim h As Double, sqr_s As Double
Dim gi(maxs) As Double
Dim gti(maxs) As Double
Dim i As Integer

sal_saltenthalpy_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

For i = 2 To maxs
  gi(i) = sal_g_term_si(i, 0, 0, t_si, p_si)  'get all coefficients of the salinity expansion
  If gi(i) = ErrorReturn Then Exit Function
  gti(i) = sal_g_term_si(i, 1, 0, t_si, p_si)  'get all t-derivatives of the salinity expansion
  If gti(i) = ErrorReturn Then Exit Function
Next i

sqr_s = Sqr(sa_si)
h = 0

For i = 2 To maxs
  h = h + (gi(i) - t_si * gti(i)) * sqr_s ^ (i - 2)
Next i

sal_saltenthalpy_si = h

End Function

'==========================================================================
Public Function sal_saltentropy_si(ByVal sa_si As Double, _
                                   ByVal t_si As Double, _
                                   ByVal p_si As Double) As Double

'returns the specific entropy of seasalt in seawater, eta_S, in J/(kg K)

'sa_si:   absolute salinity in kg/kg
't_si:    absolute temperature in K
'p_si:    absolute pressure in Pa

'check value: sal_saltentropy_si(0.035,300,1E6) = -532.458305921571

Dim eta As Double, sqr_s As Double
Dim gi(maxs) As Double
Dim i As Integer

sal_saltentropy_si = ErrorReturn

If sa_si <= 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

For i = 1 To maxs
  gi(i) = sal_g_term_si(i, 1, 0, t_si, p_si)  'get all coefficients of the salinity expansion
  If gi(i) = ErrorReturn Then Exit Function
Next i

sqr_s = Sqr(sa_si)
eta = -gi(1) * Log(sa_si)

For i = 2 To maxs
  eta = eta - gi(i) * sqr_s ^ (i - 2)
Next i

sal_saltentropy_si = eta

End Function

'==========================================================================
Public Function sal_saltvolume_si(ByVal sa_si As Double, _
                                  ByVal t_si As Double, _
                                  ByVal p_si As Double) As Double

'returns the specific volume of seasalt in seawater, v_S, in m^3/kg

'sa_si:   absolute salinity in kg/kg
't_si:    absolute temperature in K
'p_si:    absolute pressure in Pa

'check value: sal_saltvolume_si(0.035,300,1E6) = -7.30285943767952E-04

Dim v As Double, sqr_s As Double
Dim gi(maxs) As Double
Dim i As Integer

sal_saltvolume_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

If sa_si = 0 Then
  sal_saltvolume_si = sal_g_term_si(2, 0, 1, t_si, p_si)
  Exit Function
End If

For i = 2 To maxs
  gi(i) = sal_g_term_si(i, 0, 1, t_si, p_si)  'get all coefficients of the salinity expansion
  If gi(i) = ErrorReturn Then Exit Function
Next i

sqr_s = Sqr(sa_si)
v = gi(2)

For i = 3 To maxs
  v = v + gi(i) * sqr_s ^ (i - 2)
Next i

sal_saltvolume_si = v

End Function

'==========================================================================
Public Function sal_molality_si(ByVal sa_si As Double) As Double

'function returns the molality of seawater in mol/kg as a function of absolute salinity in kg/kg
'for seawater with Reference Composition, see Millero et al. Deep-Sea Research I, 55(2008) 50-72

'check value: sal_molality_si(0.035) = 1.15493681892608

sal_molality_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function

sal_molality_si = sa_si / ((1 - sa_si) * Molar_mass_seasalt_si)

End Function


