Attribute VB_Name = "Sal_1_Mdl"
Option Explicit

'#########################################################################

'This module requires the library module
'     Constants_0_Mdl, file Constants_0.bas

'#########################################################################

'=========================================================================
'This module implements the saline part of the Gibbs potential of seawater
'in the form of 7 coefficients of its power expansion in salinity,
'and their first and second partial derivatives with respect to temperature
'and pressure, as defined in IAPWS-08:

'Release on the IAPWS Formulation 2008 for the Thermodynamic Properties of Seawater
'The International Association for the Properties of Water and Steam
'Berlin, Germany, September 2008

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science 2009, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009

'If IsExtension2010 = True is specified, the Gibbs function is optionally extended
'(although not endorsed by IAPWS) as described in
'R. Feistel
'Extended Equation of State for Seawater at Elevated Temperature and Salinity
'Reference updated: Desalination, 250, 14-18, 2010
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98

Private Const Version = "06 Dec 2009"

Private Const maxs = 7, maxt = 6, maxp = 5   'Max. powers of polynomials

Private gi(maxs, maxt, maxp) As Double       'Array of coeffs of the Gibbs function terms

'==========================================================================
Public Function sal_g_term_si(ByVal term As Integer, _
                              ByVal drv_t As Integer, _
                              ByVal drv_p As Integer, _
                              ByVal t_si As Double, _
                              ByVal p_si As Double) As Double

'returns the T-P derivative of the i-th term  (d/dT)^drv_t (d/dP)^drv_p gi(T, P)
'of the salinity expansion of the saline Gibbs function gS,
'
' gS(S,T,P) = g1(T,P)*S*ln(S) + sum(i>1) gi(T,P)*S^(i/2)
'
' S = absolute salinity in kg/kg, i.e. the mass fraction of salt in seawater
' t_si = T = absolute temperature in K, ITS-90
' p_si = P = absolute pressure in Pa
' gS = specific Gibbs energy in J/kg, relative to pure water

'check values with default settings:

'sal_g_term_si(1,0,0,300,1E6)= 79427.9694845756
'sal_g_term_si(1,1,0,300,1E6)= 264.759898281919
'sal_g_term_si(1,0,1,300,1E6)= 0
'sal_g_term_si(1,2,0,300,1E6)= 0
'sal_g_term_si(1,1,1,300,1E6)= 0
'sal_g_term_si(1,0,2,300,1E6)= 0

'sal_g_term_si(2,0,0,300,1E6)= 301223.934546274
'sal_g_term_si(2,1,0,300,1E6)= 1558.10730392553
'sal_g_term_si(2,0,1,300,1E6)= -7.60101106780255E-04
'sal_g_term_si(2,2,0,300,1E6)= 19.0397747693888
'sal_g_term_si(2,1,1,300,1E6)= 9.35496617544689E-07
'sal_g_term_si(2,0,2,300,1E6)= 1.28077803425625E-12

'sal_g_term_si(3,0,0,300,1E6)= -345570.781497815
'sal_g_term_si(3,1,0,300,1E6)= -1749.19911586842
'sal_g_term_si(3,0,1,300,1E6)= 2.01608449880727E-04
'sal_g_term_si(3,2,0,300,1E6)= -9.3208882361981
'sal_g_term_si(3,1,1,300,1E6)= 2.17412289836757E-08
'sal_g_term_si(3,0,2,300,1E6)= -4.78063827317764E-13

'sal_g_term_si(4,0,0,300,1E6)= 1468073.64558789
'sal_g_term_si(4,1,0,300,1E6)= 7741.24404962481
'sal_g_term_si(4,0,1,300,1E6)= -4.33952339915708E-04
'sal_g_term_si(4,2,0,300,1E6)= -6.13689642369677
'sal_g_term_si(4,1,1,300,1E6)= -3.50876195079922E-06
'sal_g_term_si(4,0,2,300,1E6)= -6.06204383304926E-13

'sal_g_term_si(5,0,0,300,1E6)= -3776969.31546023
'sal_g_term_si(5,1,0,300,1E6)= -15135.6522247701
'sal_g_term_si(5,0,1,300,1E6)= 1.11272425476394E-03
'sal_g_term_si(5,2,0,300,1E6)= 0
'sal_g_term_si(5,1,1,300,1E6)= 0
'sal_g_term_si(5,0,2,300,1E6)= 0

'sal_g_term_si(6,0,0,300,1E6)= 6151235.69234474
'sal_g_term_si(6,1,0,300,1E6)= 14157.050993291
'sal_g_term_si(6,0,1,300,1E6)= 0
'sal_g_term_si(6,2,0,300,1E6)= 0
'sal_g_term_si(6,1,1,300,1E6)= 0
'sal_g_term_si(6,0,2,300,1E6)= 0

'sal_g_term_si(7,0,0,300,1E6)= -3734033.38866189
'sal_g_term_si(7,1,0,300,1E6)= 0
'sal_g_term_si(7,0,1,300,1E6)= 0
'sal_g_term_si(7,2,0,300,1E6)= 0
'sal_g_term_si(7,1,1,300,1E6)= 0
'sal_g_term_si(7,0,2,300,1E6)= 0


'check values with Public Const IsExtension2010 = True:

'sal_g_term_si(1,0,0,300,1E6)= 79427.9694845756
'sal_g_term_si(1,1,0,300,1E6)= 264.759898281919
'sal_g_term_si(1,0,1,300,1E6)= 0
'sal_g_term_si(1,2,0,300,1E6)= 0
'sal_g_term_si(1,1,1,300,1E6)= 0
'sal_g_term_si(1,0,2,300,1E6)= 0

'sal_g_term_si(2,0,0,300,1E6)= 301223.934546274
'sal_g_term_si(2,1,0,300,1E6)= 1558.10730392553
'sal_g_term_si(2,0,1,300,1E6)= -7.60101106780255E-04
'sal_g_term_si(2,2,0,300,1E6)= 19.0397747693888
'sal_g_term_si(2,1,1,300,1E6)= 9.35496617544689E-07
'sal_g_term_si(2,0,2,300,1E6)= 1.28077803425625E-12

'sal_g_term_si(3,0,0,300,1E6)= -345570.781497815
'sal_g_term_si(3,1,0,300,1E6)= -1749.19911586842
'sal_g_term_si(3,0,1,300,1E6)= 2.01608449880727E-04
'sal_g_term_si(3,2,0,300,1E6)= -9.3208882361981
'sal_g_term_si(3,1,1,300,1E6)= 2.17412289836757E-08
'sal_g_term_si(3,0,2,300,1E6)= -4.78063827317764E-13

'sal_g_term_si(4,0,0,300,1E6)= 1468073.64558789
'sal_g_term_si(4,1,0,300,1E6)= 7741.24404962481
'sal_g_term_si(4,0,1,300,1E6)= -4.33952339915708E-04
'sal_g_term_si(4,2,0,300,1E6)= -6.13689642369677
'sal_g_term_si(4,1,1,300,1E6)= -3.50876195079922E-06
'sal_g_term_si(4,0,2,300,1E6)= -6.06204383304926E-13

'sal_g_term_si(5,0,0,300,1E6)= -3776969.31546023
'sal_g_term_si(5,1,0,300,1E6)= -15135.6522247701
'sal_g_term_si(5,0,1,300,1E6)= 1.11272425476394E-03
'sal_g_term_si(5,2,0,300,1E6)= 0
'sal_g_term_si(5,1,1,300,1E6)= 0
'sal_g_term_si(5,0,2,300,1E6)= 0

'sal_g_term_si(6,0,0,300,1E6)= 6151235.69234474
'sal_g_term_si(6,1,0,300,1E6)= 14157.050993291
'sal_g_term_si(6,0,1,300,1E6)= 0
'sal_g_term_si(6,2,0,300,1E6)= 0
'sal_g_term_si(6,1,1,300,1E6)= 0
'sal_g_term_si(6,0,2,300,1E6)= 0

'sal_g_term_si(7,0,0,300,1E6)= -3734033.38866189
'sal_g_term_si(7,1,0,300,1E6)= 0
'sal_g_term_si(7,0,1,300,1E6)= 0
'sal_g_term_si(7,2,0,300,1E6)= 0
'sal_g_term_si(7,1,1,300,1E6)= 0
'sal_g_term_si(7,0,2,300,1E6)= 0

sal_g_term_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

If drv_t < 0 Then Exit Function
If drv_p < 0 Then Exit Function

Dim g As Double

If term < 1 Or term > maxs Then
  sal_g_term_si = 0
  Exit Function
End If

InitIAPWS08

If check_limits = 1 Then
  'SAL_LIMITS
  If t_si < sal_tmin Or t_si > sal_tmax Or _
     p_si < sal_pmin Or p_si > sal_pmax Then Exit Function
End If

sal_g_term_si = poly_gtp(term, drv_t, drv_p, t_si, p_si)

End Function

'==========================================================================
Private Sub InitIAPWS08()

Const ups = SO_salinity_si / 35#
Const su = 40# * ups
Const gi100 = 5812.81456626732 * 0.5 / su

If gi(1, 0, 0) = gi100 Then Exit Sub

Dim gc(maxs, maxt, maxp) As Double       'Local array of coeffs of the Gibbs function
Dim i As Integer, j As Integer, k As Integer
Dim ln_su As Double, sqr_su As Double

'=======================================================================
'Coefficients from the IAPWS Release 2008
'=======================================================================

'Logarithmic terms from Reference Composition 2008, computed 01 Aug 2007
'Deep-Sea Research I 55(2008)50-72.
gc(1, 0, 0) = 5812.81456626732
gc(1, 1, 0) = 851.226734946706

'Seawater Reference State defined by WG127, computed 03 March 2008
gc(2, 0, 0) = 1416.27648484197  'computed from a quadruple-precision implementation
gc(2, 1, 0) = 168.072408311545

'Thermal and colligative properties at 101325 Pa, computed 01 Aug 2007
gc(3, 0, 0) = -2432.14662381794
gc(4, 0, 0) = 2025.80115603697
gc(5, 0, 0) = -1091.66841042967
gc(6, 0, 0) = 374.60123787784
gc(7, 0, 0) = -48.5891069025409
gc(3, 1, 0) = -493.407510141682
gc(4, 1, 0) = 543.835333000098
gc(5, 1, 0) = -196.028306689776
gc(6, 1, 0) = 36.7571622995805
gc(2, 2, 0) = 880.031352997204
gc(3, 2, 0) = -43.0664675978042
gc(4, 2, 0) = -68.5572509204491
gc(2, 3, 0) = -225.267649263401
gc(3, 3, 0) = -10.0227370861875
gc(4, 3, 0) = 49.3667694856254
gc(2, 4, 0) = 91.4260447751259
gc(3, 4, 0) = 0.875600661808945
gc(4, 4, 0) = -17.1397577419788
gc(2, 5, 0) = -21.6603240875311
gc(4, 5, 0) = 2.49697009569508
gc(2, 6, 0) = 2.13016970847183

'coefficients of the pressure part of the 2003 Gibbs function
'Progress in Oceanography 58(2003)43-114
gc(2, 0, 1) = -3310.49154044839
gc(3, 0, 1) = 199.459603073901
gc(4, 0, 1) = -54.7919133532887
gc(5, 0, 1) = 36.0284195611086
gc(2, 1, 1) = 729.116529735046
gc(3, 1, 1) = -175.292041186547
gc(4, 1, 1) = -22.6683558512829
gc(2, 2, 1) = -860.764303783977
gc(3, 2, 1) = 383.058066002476
gc(2, 3, 1) = 694.244814133268
gc(3, 3, 1) = -460.319931801257
gc(2, 4, 1) = -297.728741987187
gc(3, 4, 1) = 234.565187611355

gc(2, 0, 2) = 384.794152978599
gc(3, 0, 2) = -52.2940909281335
gc(4, 0, 2) = -4.08193978912261
gc(2, 1, 2) = -343.956902961561
gc(3, 1, 2) = 83.1923927801819
gc(2, 2, 2) = 337.409530269367
gc(3, 2, 2) = -54.1917262517112
gc(2, 3, 2) = -204.889641964903
gc(2, 4, 2) = 74.726141138756

gc(2, 0, 3) = -96.5324320107458
gc(3, 0, 3) = 68.0444942726459
gc(4, 0, 3) = -30.1755111971161
gc(2, 1, 3) = 124.687671116248
gc(3, 1, 3) = -29.483064349429
gc(2, 2, 3) = -178.314556207638
gc(3, 2, 3) = 25.6398487389914
gc(2, 3, 3) = 113.561697840594
gc(2, 4, 3) = -36.4872919001588

gc(2, 0, 4) = 15.8408172766824
gc(3, 0, 4) = -3.41251932441282
gc(2, 1, 4) = -31.656964386073
gc(2, 2, 4) = 44.2040358308
gc(2, 3, 4) = -11.1282734326413

gc(2, 0, 5) = -2.62480156590992
gc(2, 1, 5) = 7.04658803315449
gc(2, 2, 5) = -7.92001547211682

If IsExtension2010 Then
  'R. Feistel
  'Extended Equation of State for Seawater at Elevated Temperature and Salinity
  'Desalination, 250, 14-18, 2010
  '(note that the coefficients given in that paper are incorrect and
  ' must be multiplied by 100, as given below)
  '
  'to cover the measurements of
  'F. Millero and F. Huang
  'Ocean Sci. Discuss. 6 (2009) 153-169.
  'http://www.ocean-sci-discuss.net/6/153/2009/
  
  gc(2, 1, 1) = gc(2, 1, 1) - 34.7600838235511
  gc(3, 1, 1) = gc(3, 1, 1) + 93.5178208024272
  gc(4, 1, 1) = gc(4, 1, 1) - 60.3483495593212
  gc(2, 2, 1) = gc(2, 2, 1) + 22.8081199116236
  gc(4, 2, 1) = gc(4, 2, 1) - 24.2869748847311
  gc(2, 3, 1) = gc(2, 3, 1) - 56.0725782144008
  gc(3, 3, 1) = gc(3, 3, 1) - 14.3523729429211
  gc(4, 3, 1) = gc(4, 3, 1) + 92.6367388049097
  gc(4, 4, 1) = gc(4, 4, 1) - 41.6658900599273
  gc(2, 5, 1) = gc(2, 5, 1) + 64.5288813326254
  gc(3, 5, 1) = gc(3, 5, 1) - 40.3505133068118
  gc(2, 6, 1) = gc(2, 6, 1) - 4.32746069361075
  gc(4, 6, 1) = gc(4, 6, 1) + 2.05954716712622
  
End If

'=======================================================================
'Modified coefficients as required for computing each of the terms g1 to g7
'of the salinity expansion of gS, stored into the global array gi()
'=======================================================================
ln_su = Log(su)
sqr_su = Sqr(su)

For j = 0 To maxt
  For k = 0 To maxp
    gi(1, j, k) = gc(1, j, k) * 0.5 / su
    gi(2, j, k) = (gc(2, j, k) - 0.5 * gc(1, j, k) * ln_su) / su
    For i = 3 To maxs
      gi(i, j, k) = gc(i, j, k) / sqr_su ^ i
    Next i
  Next k
Next j

End Sub

'==========================================================================
Private Function poly_gyz(ByVal term As Integer, _
                          ByVal drv_y As Integer, _
                          ByVal drv_z As Integer, _
                          ByVal y As Double, _
                          ByVal z As Double) As Double

'returns the value of the polynomial derivative
'(d/dy)^drv_y (d/dz)^drv_z sum(j,k) gi(term,j,k)*y^j*z^k

Dim g As Double
Dim yj As Double, zk As Double

Dim j As Integer, jmax As Integer
Dim k As Integer, kmax As Integer

Dim c As Double, L As Integer

g = 0
If y = 0 Then jmax = drv_y Else jmax = maxt
If z = 0 Then kmax = drv_z Else kmax = maxp

yj = 1#
For j = drv_y To jmax   'loop over powers of y

  zk = 1#
  For k = drv_z To kmax    'loop over powers of z

    If gi(term, j, k) <> 0 Then
      c = gi(term, j, k) * yj * zk

      For L = 1 To drv_y            'factors from y-derivatives
        c = c * CDbl(j - L + 1)
      Next L

      For L = 1 To drv_z            'factors from z-derivatives
        c = c * CDbl(k - L + 1)
      Next L

      g = g + c
    End If

    If k < kmax Then zk = zk * z
  Next k

  If j < jmax Then yj = yj * y
Next j

poly_gyz = g

End Function

'==========================================================================
Private Function poly_gtp(ByVal term As Integer, _
                          ByVal drv_t As Integer, _
                          ByVal drv_p As Integer, _
                          ByVal t_si As Double, _
                          ByVal p_si As Double) As Double

'returns the T-P derivative of the i-th term  (d/dT)^drv_t (d/dP)^drv_p gi(T, P)
'of the salinity expansion of the saline Gibbs function gS,
'
' gS(S,T,P) = g1(T,P)*S*ln(S) + sum(i>1) gi(T,P)*S^(i/2)
'
' S = absolute salinity in kg/kg, i.e. the mass fraction of salt in seawater
' T = t_si = absolute temperature in K, ITS-90
' P = p_si = absolute pressure in Pa

Const tu = 40#
Const pu = 100000000#

Dim y As Double, z As Double

'reduced input values of T, P
y = (t_si - Celsius_temperature_si) / tu
z = (p_si - Sealevel_pressure_si) / pu

poly_gtp = poly_gyz(term, drv_t, drv_p, y, z) / (tu ^ drv_t * pu ^ drv_p)

End Function


