Attribute VB_Name = "Liq_F03_5_Mdl"
Option Explicit


'#########################################################################

'This module requires the library module
'     Constants_0_Mdl, file Constants_0.bas
'     Maths_0_Mdl,     file Maths_0.bas

'#########################################################################


'=========================================================================
'This module implements the Gibbs potentials of liquid water and its
'first and second partial derivatives with respect to temperature and
'density, as defined consistently with IAPWS-95 for the oceanographic
'range -6 - 40 C, 0 - 100 MPa, in the article:

'R. Feistel:
'A new extended Gibbs thermodynamic potential of seawater.
'Progress in Oceanography, 58 (2003) 43-115

'and in the IAPWS-09 document:
'Supplementary Release on a Computationally Efficient Thermodynamic Formulation
'for Liquid Water for Oceanographic Use.
'The International Association for the Properties of Water and Steam
'Doorwerth, The Netherlands, September 2009

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98

Private Const Version = "20 Nov 2009"

'Coefficients of the Gibbs function
Private Const maxt = 7, maxp = 6
Private gc(maxt, maxp) As Double

'==========================================================================
Public Function fit_liq_g_f03_si(ByVal drv_t As Integer, _
                                 ByVal drv_p As Integer, _
                                 ByVal t_si As Double, _
                                 ByVal p_si As Double) As Double

'this function implements the Gibbs function of pure water as defined in

'R. Feistel:
'A new extended Gibbs thermodynamic potential of seawater.
'Progress in Oceanography, 58 (2003) 43-115

'and in the IAPWS-09 document:
'Supplementary Release on a Computationally Efficient Thermodynamic Formulation
'for Liquid Water for Oceanographic Use.
'The International Association for the Properties of Water and Steam
'Doorwerth, The Netherlands, September 2009


Const T0 = Celsius_temperature_si     'in K
Const tu = 40#                        'in K
Const P0 = Sealevel_pressure_si       'in Pa
Const pu = 100000000# 'in Pa

Dim y As Double, z As Double, g As Double

fit_liq_g_f03_si = ErrorReturn

If drv_t < 0 Then Exit Function
If drv_p < 0 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

InitF03

y = (t_si - T0) / tu
z = (p_si - P0) / pu

g = poly_gyz(drv_t, drv_p, y, z)
If g = ErrorReturn Then Exit Function

fit_liq_g_f03_si = g / (tu ^ drv_t * pu ^ drv_p)

End Function

'==========================================================================
Public Function fit_liq_cp_f03_si(ByVal t_si As Double, _
                                  ByVal p_si As Double) As Double

'This function returns the heat capacity cp of liquid water computed from the
'Gibbs function 2003 of Seawater as a function of temperature
'and pressure, published in

'R. Feistel:
'A new extended Gibbs thermodynamic potential of seawater.
'Progress in Oceanography, 58/1 (2003) 43-115

'output:  fit_liq_density_f03_si: cp in J/(kg K)

'input:   t_si: absolute temperature in K
'         p_si: absolute pressure in Pa

Dim gtt As Double

fit_liq_cp_f03_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

gtt = fit_liq_g_f03_si(2, 0, t_si, p_si)

If gtt = ErrorReturn Then Exit Function
If gtt >= 0 Then Exit Function

fit_liq_cp_f03_si = -t_si * gtt

End Function

'==========================================================================
Public Function fit_liq_density_f03_si(ByVal t_si As Double, _
                                       ByVal p_si As Double) As Double

'This function returns the density of liquid water computed from the
'Gibbs function 2003 of Seawater as a function of temperature
'and pressure, published in

'R. Feistel:
'A new extended Gibbs thermodynamic potential of seawater.
'Progress in Oceanography, 58/1 (2003) 43-115

'output:  fit_liq_density_f03_si: density in kg/m^3

'input:   t_si: absolute temperature in K
'         p_si: absolute pressure in Pa

Dim d As Double

fit_liq_density_f03_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = fit_liq_g_f03_si(0, 1, t_si, p_si)

If d = ErrorReturn Then Exit Function
If d <= 0 Then Exit Function

fit_liq_density_f03_si = 1# / d

End Function

'==========================================================================
Public Function fit_liq_expansion_f03_si(ByVal t_si As Double, _
                                         ByVal p_si As Double) As Double

'This function returns the thermal expansion of liquid water computed from the
'Gibbs function 2003 of Seawater as a function of temperature
'and pressure, published in

'R. Feistel:
'A new extended Gibbs thermodynamic potential of seawater.
'Progress in Oceanography, 58/1 (2003) 43-115

'output:  fit_liq_expansion_f03_si: alpha in 1/K

'input:   t_si: absolute temperature in K
'         p_si: absolute pressure in Pa

Dim gp As Double, gtp As Double

fit_liq_expansion_f03_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

gp = fit_liq_g_f03_si(0, 1, t_si, p_si)
If gp = ErrorReturn Then Exit Function
If gp <= 0 Then Exit Function

gtp = fit_liq_g_f03_si(1, 1, t_si, p_si)
If gtp = ErrorReturn Then Exit Function

fit_liq_expansion_f03_si = gtp / gp

End Function

'==========================================================================
Public Function fit_liq_kappa_t_f03_si(ByVal t_si As Double, _
                                       ByVal p_si As Double) As Double

'This function returns the isothermal compressibility of liquid water computed from the
'Gibbs function 2003 of Seawater as a function of temperature
'and pressure, published in

'R. Feistel:
'A new extended Gibbs thermodynamic potential of seawater.
'Progress in Oceanography, 58/1 (2003) 43-115

'output:  fit_liq_kappa_t_f03_si: kappa_t in 1/Pa

'input:   t_si: absolute temperature in K
'         p_si: absolute pressure in Pa

Dim gp As Double, gpp As Double

fit_liq_kappa_t_f03_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

gp = fit_liq_g_f03_si(0, 1, t_si, p_si)
If gp = ErrorReturn Then Exit Function
If gp <= 0 Then Exit Function

gpp = fit_liq_g_f03_si(0, 2, t_si, p_si)
If gpp = ErrorReturn Then Exit Function
If gpp >= 0 Then Exit Function

fit_liq_kappa_t_f03_si = -gpp / gp

End Function

'==========================================================================
Public Function fit_liq_soundspeed_f03_si(ByVal t_si As Double, _
                                          ByVal p_si As Double) As Double

'This function returns the sound speed of liquid water computed from the
'Gibbs function 2003 of Seawater as a function of temperature
'and pressure, published in

'R. Feistel:
'A new extended Gibbs thermodynamic potential of seawater.
'Progress in Oceanography, 58/1 (2003) 43-115

'output:  fit_liq_soundspeed_f03_si: speed in m/s

'input:   t_si: absolute temperature in K
'         p_si: absolute pressure in Pa

Dim gp As Double, gtt As Double, gtp As Double, gpp As Double, c As Double

fit_liq_soundspeed_f03_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

gp = fit_liq_g_f03_si(0, 1, t_si, p_si)
If gp = ErrorReturn Then Exit Function
If gp <= 0 Then Exit Function

gtt = fit_liq_g_f03_si(2, 0, t_si, p_si)
If gtt = ErrorReturn Then Exit Function
If gtt >= 0 Then Exit Function

gtp = fit_liq_g_f03_si(1, 1, t_si, p_si)
If gtp = ErrorReturn Then Exit Function

gpp = fit_liq_g_f03_si(0, 2, t_si, p_si)
If gpp = ErrorReturn Then Exit Function
If gpp >= 0 Then Exit Function

c = gtp * gtp - gtt * gpp
If c = 0 Then Exit Function
c = gtt / c
If c < 0 Then Exit Function

fit_liq_soundspeed_f03_si = gp * Sqr(c)

End Function

'==========================================================================
Private Sub InitF03()
' g(T,P) = Sum gc(j,k) * (t/40C)^j * (p/100 MPa)^k
'          j,k

Const Tt = TP_temperature_si
Const T0 = Celsius_temperature_si
Const tu = 40#

Const Pt = TP_pressure_IAPWS95_si 'IAPWS-95 triple point pressure
Const P0 = Sealevel_pressure_si
Const pu = 100000000#

Const yt = (Tt - T0) / tu
Const zt = (Pt - P0) / pu

Dim g As Double, gt As Double, gp As Double

If gc(2, 0) = -12357.785933039 Then Exit Sub

gc(2, 0) = -12357.785933039
gc(3, 0) = 736.741204151612
gc(4, 0) = -148.185936433658
gc(5, 0) = 58.0259125842571
gc(6, 0) = -18.9843846514172
gc(7, 0) = 3.05081646487967
gc(0, 1) = 100015.695367145
gc(1, 1) = -270.983805184062
gc(2, 1) = 1455.0364540468
gc(3, 1) = -672.50778314507
gc(4, 1) = 397.968445406972
gc(5, 1) = -194.618310617595
gc(6, 1) = 63.5113936641785
gc(7, 1) = -9.63108119393062
gc(0, 2) = -2544.5765420363
gc(1, 2) = 776.153611613101
gc(2, 2) = -756.558385769359
gc(3, 2) = 499.360390819152
gc(4, 2) = -301.815380621876
gc(5, 2) = 120.520654902025
gc(6, 2) = -22.2897317140459
gc(0, 3) = 284.517778446287
gc(1, 3) = -196.51255088122
gc(2, 3) = 273.479662323528
gc(3, 3) = -239.545330654412
gc(4, 3) = 152.196371733841
gc(5, 3) = -55.2723052340152
gc(6, 3) = 8.17060541818112
gc(0, 4) = -33.3146754253611
gc(1, 4) = 28.9796526294175
gc(2, 4) = -55.5604063817218
gc(3, 4) = 48.8012518593872
gc(4, 4) = -26.3748377232802
gc(5, 4) = 6.48190668077221
gc(0, 5) = 4.20263108803084
gc(1, 5) = -2.13290083518327
gc(2, 5) = 4.34420671917197
gc(3, 5) = -1.66307106208905
gc(0, 6) = -0.546428511471039

'IAPWS-95 reference state condition
'energy = 0 and entropy = 0 at the triple point:
'gc(0, 0) = 101.342743139672
'gc(1, 0) = 5.90578348518236

'quadruple precision values (D.G.Wright 21 July 2008)
'gc(0, 0) = 1.013427431396741480431228220832E2
'gc(1, 0) = 5.905783479094018366702121889468E0
gc(0, 0) = 101.342743139674
gc(1, 0) = 5.90578347909402

'RF: dynamical adjustment:
'
'g = poly_gyz(0, 0, yt, zt)
'gt = poly_gyz(1, 0, yt, zt) / tu
'gp = poly_gyz(0, 1, yt, zt) / pu
'
'gc(0, 0) = gc(0, 0) - g + pt * gp
'gc(1, 0) = gc(1, 0) - gt * tu

End Sub

'==========================================================================
Private Function poly_gyz(ByVal drv_y As Integer, _
                          ByVal drv_z As Integer, _
                          ByVal y As Double, _
                          ByVal z As Double) As Double

'returns the value of the polynomial derivative
'(d/dy)^drv_y (d/dz)^drv_z sum(j,k) gc(j,k)*y^j*z^k

Dim g As Double
Dim yj As Double, zk As Double

Dim j As Integer, jmax As Integer
Dim k As Integer, kmax As Integer

Dim c As Double, L As Integer

g = 0
If y = 0 Then jmax = drv_y Else jmax = maxt
If z = 0 Then kmax = drv_z Else kmax = maxp

yj = 1#
For j = drv_y To jmax   'loop over powers of y

  zk = 1#
  For k = drv_z To kmax    'loop over powers of z

    If gc(j, k) <> 0 Then
      c = gc(j, k) * yj * zk

      For L = 1 To drv_y            'factors from y-derivatives
        c = c * CDbl(j - L + 1)
      Next L

      For L = 1 To drv_z            'factors from z-derivatives
        c = c * CDbl(k - L + 1)
      Next L

      g = g + c
    End If

    If k < kmax Then zk = zk * z
  Next k

  If j < jmax Then yj = yj * y
Next j

poly_gyz = g

End Function

'==========================================================================
Public Function chk_IAPWS09_Table6() As String

'TABLE 6  Numerical check values for the Gibbs function g and its derivatives, Table 4,
'The numerical functions evaluated here at given points (T, p) are defined in Tables 3 and 4.
'
'Quantity       Value              Value             Value          Unit
'T              273.15             273.15            313.15         K
'p              101325             1E8               101325         Pa
'g             0.101342743E3     0.977303868E5    -0.116198898E5    J kg-1
'(dg/dT)_p     0.147644587       0.851506346E1    -0.572365181E3    J kg-1 K-1
'(dg/dp)_T     0.100015695E-2    0.956683354E-3    0.100784471E-2   m3 kg-1
'(d2g/dT2)_p  -0.154472324E2    -0.142970174E2    -0.133463968E2    J kg-1 K-2
' d2g/dTdp    -0.677459513E-7    0.199088060E-6    0.388499694E-6   m3 kg-1 K-1
'(d2g/dp2)_T  -0.508915308E-12  -0.371527164E-12  -0.445841077E-12  m3 kg-1 Pa-1
'h             0.610136242E2     0.954044973E5     0.167616267E6    J kg-1
'f             0.183980891E-2    0.206205140E4    -0.117220097E5    J kg-1
'u            -0.403272791E2    -0.263838183E3     0.167514147E6    J kg-1
's            -0.147644587      -0.851506346E1     0.572365181E3    J kg-1 K-1
'rho           0.999843071E3     0.104527793E4     0.992216354E3    kg m-3
'cp            0.421941153E4     0.390523030E4     0.417942416E4    J kg-1 K-1
'w             0.140240099E4     0.157543089E4     0.152891242E4    m s-1

Dim CRLF As String
Dim txt As String, row As String
Dim t As Double, p As Double, q(3) As Double
Dim i As Integer, j As Integer

CRLF = Chr(13) + Chr(10)

txt = " Implementation of IAPWS-09 in Visual Basic" + CRLF
txt = txt + " for Publication in Ocean Science, 2009" + CRLF
txt = txt + " R. Feistel, IOW, Version " + Version + CRLF
txt = txt + " Compiled on " + CStr(Now) + CRLF + CRLF

txt = txt + " Function values as given in Table 6 of IAPWS-09:" + CRLF
txt = txt + " Numerical check values for the Gibbs function g and its derivatives, Table 4, " + CRLF
txt = txt + " The numerical functions evaluated here at given points (T, p) are defined in Tables 3 and 4." + CRLF + CRLF

txt = txt + "Quantity      Value             Value             Value            Unit" + CRLF

For i = 1 To 15
  Select Case i
    Case 1:  row = "T             0.27315E+3        0.27315E+3        0.31315E+3       K"
    Case 2:  row = "p             0.101325E+6       0.1E+9            0.101325E+6      Pa"
    Case 3:  row = "g             0.101342743E+3    0.977303868E+5   -0.116198898E+5   J kg-1"
    Case 4:  row = "(dg/dT)_p     0.147644587       0.851506346E+1   -0.572365181E+3   J kg-1 K-1"
    Case 5:  row = "(dg/dp)_T     0.100015695E-2    0.956683354E-3    0.100784471E-2   m3 kg-1"
    Case 6:  row = "(d2g/dT2)_p  -0.154472324E+2   -0.142970174E+2   -0.133463968E+2   J kg-1 K-2"
    Case 7:  row = " d2g/dTdp    -0.677459513E-7    0.199088060E-6    0.388499694E-6   m3 kg-1 K-1"
    Case 8:  row = "(d2g/dp2)_T  -0.508915308E-12  -0.371527164E-12  -0.445841077E-12  m3 kg-1 Pa-1"
    Case 9:  row = "h             0.610136242E+2    0.954044973E+5    0.167616267E+6   J kg-1"
    Case 10: row = "f             0.183980891E-2    0.206205140E+4   -0.117220097E+5   J kg-1"
    Case 11: row = "u            -0.403272791E+2   -0.263838183E+3    0.167514147E+6   J kg-1"
    Case 12: row = "s            -0.147644587      -0.851506346E+1    0.572365181E+3   J kg-1 K-1"
    Case 13: row = "rho           0.999843071E+3    0.104527793E+4    0.992216354E+3   kg m-3"
    Case 14: row = "cp            0.421941153E+4    0.390523030E+4    0.417942416E+4   J kg-1 K-1"
    Case 15: row = "w             0.140240099E+4    0.157543089E+4    0.152891242E+4   m s-1"
  End Select

  txt = txt + row + CRLF
  
  If i > 2 Then
    txt = txt + "this code:   "
    
    For j = 1 To 3
      t = Choose(j, 273.15, 273.15, 313.15)
      p = Choose(j, 101325, 100000000#, 101325)
      q(j) = 0
      Select Case i
        Case 1: q(j) = t
        Case 2: q(j) = p
        Case 3: q(j) = fit_liq_g_f03_si(0, 0, t, p)
        Case 4: q(j) = fit_liq_g_f03_si(1, 0, t, p)
        Case 5: q(j) = fit_liq_g_f03_si(0, 1, t, p)
        Case 6: q(j) = fit_liq_g_f03_si(2, 0, t, p)
        Case 7: q(j) = fit_liq_g_f03_si(1, 1, t, p)
        Case 8: q(j) = fit_liq_g_f03_si(0, 2, t, p)
        Case 9: q(j) = fit_liq_g_f03_si(0, 0, t, p) - t * fit_liq_g_f03_si(1, 0, t, p)
        Case 10: q(j) = fit_liq_g_f03_si(0, 0, t, p) - p * fit_liq_g_f03_si(0, 1, t, p)
        Case 11: q(j) = fit_liq_g_f03_si(0, 0, t, p) - p * fit_liq_g_f03_si(0, 1, t, p) _
                                                     - t * fit_liq_g_f03_si(1, 0, t, p)
        Case 12: q(j) = -fit_liq_g_f03_si(1, 0, t, p)
        Case 13: q(j) = fit_liq_density_f03_si(t, p)
        Case 14: q(j) = fit_liq_cp_f03_si(t, p)
        Case 15: q(j) = fit_liq_soundspeed_f03_si(t, p)
      End Select
      
      txt = txt + Left(EFormat(q(j), 9) + Space(18), 18)
    Next j
    
    txt = txt + CRLF + CRLF
  End If
Next i

chk_IAPWS09_Table6 = txt

' Implementation of IAPWS-09 in Visual Basic
' for Publication in Ocean Science, 2009
' R. Feistel, IOW, Version 20 Nov 2009
' Compiled on 20.11.2009 18:31:31
'
' Function values as given in Table 6 of IAPWS-09:
'
'Quantity       Value              Value             Value          Unit
'T              273.15             273.15            313.15         K
'p              101325             1E8               101325         Pa
'g             0.101342743E+3    0.977303868E+5   -0.116198898E+5   J kg-1
'this code:    0.101342743E+3    0.977303868E+5   -0.116198898E+5
'
'(dg/dT)_p     0.147644587       0.851506346E+1   -0.572365181E+3   J kg-1 K-1
'this code:    0.147644587       0.851506346E+1   -0.572365181E+3
'
'(dg/dp)_T     0.100015695E-2    0.956683354E-3    0.100784471E-2   m3 kg-1
'this code:    0.100015695E-2    0.956683354E-3    0.100784471E-2
'
'(d2g/dT2)_p  -0.154472324E+2   -0.142970174E+2   -0.133463968E+2   J kg-1 K-2
'this code:   -0.154472324E+2   -0.142970174E+2   -0.133463968E+2
'
' d2g/dTdp    -0.677459513E-7    0.199088060E-6    0.388499694E-6   m3 kg-1 K-1
'this code:   -0.677459513E-7    0.199088060E-6    0.388499694E-6
'
'(d2g/dp2)_T  -0.508915308E-12  -0.371527164E-12  -0.445841077E-12  m3 kg-1 Pa-1
'this code:   -0.508915308E-12  -0.371527164E-12  -0.445841077E-12
'
'h             0.610136242E+2    0.954044973E+5    0.167616267E+6   J kg-1
'this code:    0.610136242E+2    0.954044973E+5    0.167616267E+6
'
'f             0.183980891E-2    0.206205140E+4   -0.117220097E+5   J kg-1
'this code:    0.183980891E-2    0.206205140E+4   -0.117220097E+5
'
'u            -0.403272791E+2   -0.263838183E+3    0.167514147E+6   J kg-1
'this code:   -0.403272791E+2   -0.263838183E+3    0.167514147E+6
'
's            -0.147644587      -0.851506346E+1    0.572365181E+3   J kg-1 K-1
'this code:   -0.147644587      -0.851506346E+1    0.572365181E+3
'
'rho           0.999843071E+3    0.104527793E+4    0.992216354E+3   kg m-3
'this code:    0.999843071E+3    0.104527793E+4    0.992216354E+3
'
'cp            0.421941153E+4    0.390523030E+4    0.417942416E+4   J kg-1 K-1
'this code:    0.421941153E+4    0.390523030E+4    0.417942416E+4
'
'w             0.140240099E+4    0.157543089E+4    0.152891242E+4   m s-1
'this code:    0.140240099E+4    0.157543089E+4    0.152891242E+4

End Function

