Attribute VB_Name = "Liq_Air_4c_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules:
'     Constants_0_Mdl,  file Constants_0.bas
'     Air_3a_Mdl,       file Air_3a.bas
'     Liq_Air_4a_Mdl,   file Liq_Air_4a.bas
'     Liq_Air_4b_Mdl,   file Liq_Air_4b.bas
'     Ice_Liq_4_Mdl,    file Ice_Liq_4.bas

'#########################################################################

'=========================================================================
'This module implements the enthalpy of wet air, as well as its partial
'derivatives, depending on air fraction, entropy and pressure.

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009

'Feistel, R., Kretzschmar, H.-J., Span, R., Hagen, E., Wright, D.G., Herrmann, S.:
'Thermodynamic Properties of Sea Air.
'Ocean Science Discussion 6(2009)21932325.
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98

'Control parameters of the temperature iteration
Private ctrl_initialized As Integer

Private ctrl_mode_pottemp As Integer
Private ctrl_loop_maximum As Long
Private ctrl_init_pottemp As Double
Private ctrl_eps_exit_pottemp As Double


Private Const Version = "07 Jun 2010"

'==========================================================================
Public Function liq_air_h_si(ByVal drv_wa As Long, _
                             ByVal drv_eta As Long, _
                             ByVal drv_p As Long, _
                             ByVal wa_si As Double, _
                             ByVal eta_si As Double, _
                             ByVal p_si As Double) As Double

'this function implements enthalpy of wet air as a thermodynamic potential,
'depending on air mass fraction, entropy and pressure

'returns liq_air_h_si as the wA-eta-P derivative

'(d/dwA)^drv_wA (d/deta)^drv_eta (d/dP)^drv_p h(wA,eta,P)

'of the specific enthalpy of wet air, h(wA,eta,P), in J/kg

'wa_si     dry-air mass fraction of wet air in kg/kg
'eta_si    specific entropy in J/(kg K),
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a and Air_3a, and on the iteration settings
'      for temperature by set_it_ctrl_liq_air_pottemp of this module

'check values with default settings:  v. 1.0
'liq_air_h_si( 0, 0, 0, 0.5, 100, 1E5) = 26898.2871261607
'liq_air_h_si( 1, 0, 0, 0.5, 100, 1E5) =-1682.2814754|215
'liq_air_h_si( 0, 1, 0, 0.5, 100, 1E5) = 280.392597112172
'liq_air_h_si( 0, 0, 1, 0.5, 100, 1E5) = 0.406967545474776
'liq_air_h_si( 2, 0, 0, 0.5, 100, 1E5) = 35.665043619|41
'liq_air_h_si( 1, 1, 0, 0.5, 100, 1E5) = 1.78333840654336
'liq_air_h_si( 1, 0, 1, 0.5, 100, 1E5) = 0.811934874970271
'liq_air_h_si( 0, 2, 0, 0.5, 100, 1E5) = 8.91712318142741E-02
'liq_air_h_si( 0, 1, 1, 0.5, 100, 1E5) = 1.55091234445055E-04
'liq_air_h_si( 0, 0, 2, 0.5, 100, 1E5) =-3.83855233558049E-06

'check values with default settings:  v. 1.1
'liq_air_h_si( 0, 0, 0, 0.5, 100, 1E5) = 26898.5215491787
'liq_air_h_si( 1, 0, 0, 0.5, 100, 1E5) =-1681.7936611|2978
'liq_air_h_si( 0, 1, 0, 0.5, 100, 1E5) = 280.393544898538
'liq_air_h_si( 0, 0, 1, 0.5, 100, 1E5) = 0.406872930019273
'liq_air_h_si( 2, 0, 0, 0.5, 100, 1E5) = 35.768970891|4729
'liq_air_h_si( 1, 1, 0, 0.5, 100, 1E5) = 1.7859992519|5513
'liq_air_h_si( 1, 0, 1, 0.5, 100, 1E5) = 0.811745643965305
'liq_air_h_si( 0, 2, 0, 0.5, 100, 1E5) = 8.91776656829876E-02
'liq_air_h_si( 0, 1, 1, 0.5, 100, 1E5) = 1.55067379031342E-04
'liq_air_h_si( 0, 0, 2, 0.5, 100, 1E5) =-3.83770118469693E-06

Dim t As Double

liq_air_h_si = ErrorReturn

If drv_wa < 0 Then Exit Function
If drv_eta < 0 Then Exit Function
If drv_p < 0 Then Exit Function
If drv_wa + drv_eta + drv_p > 2 Then Exit Function

If wa_si <= 0 Or wa_si >= 1 Then Exit Function
If p_si < 0 Then Exit Function

'compute temperature from entropy.
'if p_si = in-situ pressure, this is in-situ temperature
'if p_si = reference pressure, this is potential temperature
t = liq_air_h_temperature_si(wa_si, eta_si, p_si)
If t = ErrorReturn Then Exit Function

liq_air_h_si = liq_air_a_eta_p_derivatives_si(drv_wa, drv_eta, drv_p, wa_si, t, p_si)

End Function

'==========================================================================
Public Function liq_air_pottemp_si(ByVal a_si As Double, _
                                   ByVal t_si As Double, _
                                   ByVal p_si As Double, _
                                   ByVal pr_si As Double) As Double
                               
'this function computes potential temperature of wet air

'returns   273.15 K + theta(A,T,P,Pr) potential temperature of wet air in K,
'a_si      absolute air mass fraction in kg/kg
't_si      absolute in-situ temperature in K,
'p_si      absolute in-situ pressure in Pa
'pr_si     absolute reference pressure in Pa

'check value with default settings:
'liq_air_pottemp_si(0.5, 300, 1e4, 1E5) = 348.223146077119  v. 1.0
'liq_air_pottemp_si(0.5, 300, 1e4, 1E5) = 348.222379216949  v. 1.1

Dim S As Double

liq_air_pottemp_si = ErrorReturn

If a_si < 0 Or a_si >= 1 Then Exit Function
If t_si < 0 Then Exit Function
If p_si < 0 Then Exit Function
If pr_si < 0 Then Exit Function

If p_si = pr_si Then
  liq_air_pottemp_si = t_si
  Exit Function
End If

S = liq_air_g_entropy_si(a_si, t_si, p_si)  'specific entropy in-situ
If S = ErrorReturn Then Exit Function

liq_air_pottemp_si = liq_air_h_temperature_si(a_si, S, pr_si)

End Function

'==========================================================================
Public Function liq_air_potdensity_si(ByVal a_si As Double, _
                                      ByVal t_si As Double, _
                                      ByVal p_si As Double, _
                                      ByVal pr_si As Double) As Double
                               
'this function computes potential density of wet air

'returns   rho_theta(A,T,P,Pr) potential density of wet air in kg/m,
'a_si      absolute air mass fraction in kg/kg
't_si      absolute in-situ temperature in K,
'p_si      absolute in-situ pressure in Pa
'pr_si     absolute reference pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a
'      and on the iteration settings for temperature by set_it_ctrl_liq_air_pottemp of this module

'check value with default settings:
'liq_air_potdensity_si(0.5, 300, 1e4, 1E5) = 1.22518908856086  v. 1.0
'liq_air_potdensity_si(0.5, 300, 1e4, 1E5) = 1.22550664944784  v. 1.1

Dim S As Double, v As Double

liq_air_potdensity_si = ErrorReturn

If a_si < 0 Or a_si >= 1 Then Exit Function
If t_si < 0 Then Exit Function
If p_si < 0 Then Exit Function
If pr_si < 0 Then Exit Function

S = liq_air_g_entropy_si(a_si, t_si, p_si)  'specific entropy in-situ
If S = ErrorReturn Then Exit Function

v = liq_air_h_si(0, 0, 1, a_si, S, pr_si) 'specific volume at reference pressure
If v = ErrorReturn Then Exit Function
If v <= 0 Then Exit Function

liq_air_potdensity_si = 1 / v

End Function

'==========================================================================
Public Function liq_air_potenthalpy_si(ByVal a_si As Double, _
                                       ByVal t_si As Double, _
                                       ByVal p_si As Double, _
                                       ByVal pr_si As Double) As Double
                               
'this function computes potential enthalpy of wet air

'returns   h_theta(A,T,P,Pr) potential enthalpy of wet air in J/kg,
'a_si      absolute air mass fraction in kg/kg
't_si      absolute in-situ temperature in K,
'p_si      absolute in-situ pressure in Pa
'pr_si     absolute reference pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a
'      and on the iteration settings for temperature by set_it_ctrl_liq_air_pottemp of this module

'check value with default settings:
'liq_air_potenthalpy_si(0.5, 300, 1e4, 1E5) = 655299.468706051  v. 1.0
'liq_air_potenthalpy_si(0.5, 300, 1e4, 1E5) = 655155.797981886  v. 1.1

Dim S As Double

liq_air_potenthalpy_si = ErrorReturn

If a_si < 0 Or a_si >= 1 Then Exit Function
If t_si < 0 Then Exit Function
If p_si < 0 Then Exit Function
If pr_si < 0 Then Exit Function

S = liq_air_g_entropy_si(a_si, t_si, p_si)  'specific entropy in-situ
If S = ErrorReturn Then Exit Function

liq_air_potenthalpy_si = liq_air_h_si(0, 0, 0, a_si, S, pr_si)

End Function

'=========================================================================
Public Function liq_air_h_temperature_si(ByVal wa_si As Double, _
                                         ByVal eta_si As Double, _
                                         ByVal p_si As Double) As Double
                                 
'returns   t(A,eta,P) (potential) temperature of wet air in K,
'wa_si     dry-air mass fraction in kg/kg
'eta_si    specific entropy in J/(kg K),
'p_si      absolute (reference) pressure in Pa

'this is the inverse function for temperature of
'    eta_si = liq_air_g_entropy_si(wa_si, t_si, p_si)
'in liq_air_4b

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a, Air_3a and on those made in this module

'check value with default settings:
'liq_air_h_temperature_si(0.5, 100, 1E5) = 280.392597112172  v. 1.0
'liq_air_h_temperature_si(0.5, 100, 1E5) = 280.393544898538  v. 1.1

Dim t1 As Double, t2 As Double, eps As Double, maxit As Long

liq_air_h_temperature_si = ErrorReturn

init_it_ctrl_pottemp

Select Case ctrl_mode_pottemp
  Case 0:    t1 = aux_temperature_min_si(wa_si, p_si)
  Case Else: t1 = ctrl_init_pottemp
End Select
t2 = aux_temperature_max_si(wa_si, p_si)

Select Case ctrl_loop_maximum
  Case 0:      maxit = 100
  Case -1:     liq_air_h_temperature_si = t1
               Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else:   Exit Function
End Select

eps = ctrl_eps_exit_pottemp
If eps <= 0 Then Exit Function

'run the iteration
liq_air_h_temperature_si = temperatureiteration_Brent(wa_si, eta_si, p_si, t1, t2, maxit, eps)

End Function

'=========================================================================
Public Function liq_air_h_lapserate_si(ByVal wa_si As Double, _
                                       ByVal eta_si As Double, _
                                       ByVal p_si As Double) As Double
                                 
'returns   gamma(A,eta,P) = h_eta_p  moist-adiabatic lapse rate in K/Pa,
'wa_si     dry-air mass fraction of wet air in kg/kg
'eta_si    specific entropy in J/(kg K),
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a, Air_3a and on those made in this module

'check value with default settings:
'liq_air_h_lapserate_si(0.5, 100, 1E5) = 1.55091234445055E-04  v. 1.0
'liq_air_h_lapserate_si(0.5, 100, 1E5) = 1.55067379031342E-04  v. 1.1

liq_air_h_lapserate_si = liq_air_h_si(0, 1, 1, wa_si, eta_si, p_si)

End Function

'=========================================================================
Public Function liq_air_h_cp_si(ByVal wa_si As Double, _
                                ByVal eta_si As Double, _
                                ByVal p_si As Double) As Double
                                 
'returns   cp(A,eta,P) = t/h_eta_eta  heat capacity of wet air in J/(kg K),
'wa_si     dry-air mass fraction in kg/kg
'eta_si    specific entropy in J/(kg K),
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a, Air_3a and on those made in this module

'check value with default settings:
'liq_air_h_cp_si(0.5, 100, 1E5) = 3144.42888594579  v. 1.0
'liq_air_h_cp_si(0.5, 100, 1E5) = 3144.21265404382  v. 1.1

Dim t As Double

liq_air_h_cp_si = ErrorReturn

t = liq_air_h_temperature_si(wa_si, eta_si, p_si)
If t = ErrorReturn Then Exit Function
If t <= 0 Then Exit Function

liq_air_h_cp_si = liq_air_g_cp_si(wa_si, t, p_si)

End Function

'=========================================================================
Public Function liq_air_h_kappa_s_si(ByVal wa_si As Double, _
                                     ByVal eta_si As Double, _
                                     ByVal p_si As Double) As Double
                                 
'returns   kappa(A,eta,P) = - h_pp/h_p adiabatic compressibility of wet air in 1/Pa,
'wa_si     dry-air mass fraction in kg/kg
'eta_si    specific entropy in J/(kg K),
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a, Air_3a and on those made in this module

'check value with default settings:
'liq_air_h_kappa_s_si(0.5, 100, 1E5) = 9.43208464228361E-06  v. 1.0
'liq_air_h_kappa_s_si(0.5, 100, 1E5) = 9.43218607469204E-06  v. 1.1

Dim d As Double, c As Double

liq_air_h_kappa_s_si = ErrorReturn

d = liq_air_h_density_si(wa_si, eta_si, p_si)
If d = ErrorReturn Then Exit Function
If d <= 0 Then Exit Function

c = liq_air_h_si(0, 0, 2, wa_si, eta_si, p_si)
If c = ErrorReturn Then Exit Function
If c >= 0 Then Exit Function

liq_air_h_kappa_s_si = -d * c

End Function

'=========================================================================
Public Function liq_air_h_density_si(ByVal wa_si As Double, _
                                     ByVal eta_si As Double, _
                                     ByVal p_si As Double) As Double
                                 
'returns   d(A,eta,P) = 1/h_p density of wet air in kg/m3,
'wa_si     dry-air mass fraction in kg/kg
'eta_si    specific entropy in J/(kg K),
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a, Air_3a and on those made in this module

'check value with default settings:
'liq_air_h_density_si(0.5, 100, 1E5) = 2.4571983960868  v. 1.0
'liq_air_h_density_si(0.5, 100, 1E5) = 2.45776980039599  v. 1.1

Dim t As Double

liq_air_h_density_si = ErrorReturn

t = liq_air_h_temperature_si(wa_si, eta_si, p_si)
If t = ErrorReturn Then Exit Function
If t <= 0 Then Exit Function

liq_air_h_density_si = liq_air_g_density_si(wa_si, t, p_si)

End Function

'==========================================================================
Private Function temperatureiteration_Brent(ByVal wa_si As Double, _
                                            ByVal eta_si As Double, _
                                            ByVal p_si As Double, _
                                            ByVal t1 As Double, _
                                            ByVal t2 As Double, _
                                            ByVal maxit As Integer, _
                                            ByVal eps As Double) As Double

'The function computes the temperature as a function of air fraction, entropy and pressure,
'computed by Brent iteration
'http://en.wikipedia.org/wiki/Brent's_method

'output: temperatureiteration_Brent: temperature in K
'
'        The value ErrorReturn is returned if
'        - the maximum number of iterations is exceeded without meeting the exit criterion
'        - the function call to liq_air_g_si has returned an error
'        - temperature has taken a zero or negative value during the iteration

'input: wa_si: mass fraction of dry air in wet air in kg/kg
'      eta_si: specific entropy in J/(kg K)
'        p_si: absolute pressure in Pa
'          t1: initial guess for temperature in K
'          t2: counterpoint temperature
'       maxit: maximum number of iteration steps to be done
'         eps: required accuracy of temperature
'           eps > 0: absolute temperature uncertainty in K
'           eps < 0: relative temperature uncertainty

Dim a As Double, b As Double, c As Double, d As Double, S As Double
Dim fa As Double, fb As Double, fc As Double, fs As Double
Dim mflag As Boolean
Dim it As Long

temperatureiteration_Brent = ErrorReturn

If wa_si < 0 Or wa_si >= 1 Then Exit Function
If t1 <= 0 Then Exit Function
If t2 <= 0 Then Exit Function
If p_si <= 0 Then Exit Function
If eps = 0 Then Exit Function
If maxit <= 0 Then Exit Function

a = t1
fa = liq_air_g_si(0, 1, 0, wa_si, a, p_si)
If fa = ErrorReturn Then 'permit excursions to the humid-air region
  fa = air_g_si(0, 1, 0, wa_si, a, p_si)
  If fa = ErrorReturn Then Exit Function
End If
fa = -fa - eta_si

b = t2
fb = liq_air_g_si(0, 1, 0, wa_si, b, p_si)
If fb = ErrorReturn Then 'permit excursions to the humid-air region
  fb = air_g_si(0, 1, 0, wa_si, b, p_si)
  If fb = ErrorReturn Then Exit Function
End If
fb = -fb - eta_si

If fa * fb > 0 Then Exit Function

If Abs(fa) < Abs(fb) Then
  Swap a, b
  Swap fa, fb
End If

c = a
fc = fa
mflag = True

check_limits = check_limits - 1

For it = 1 To maxit

  If fb = 0 Then
    temperatureiteration_Brent = b
    Exit For
  End If

  If eps > 0 Then                'absolute limit
    If Abs(a - b) < eps Then
      temperatureiteration_Brent = b
      Exit For
    End If
  Else                           'relative limit
    If Abs(a - b) < -eps * b Then
      temperatureiteration_Brent = b
      Exit For
    End If
  End If

  If fa = fb Then Exit For

  If fa <> fc And fb <> fc Then
    S = a * fb * fc / ((fa - fb) * (fa - fc)) + _
        b * fa * fc / ((fb - fa) * (fb - fc)) + _
        c * fa * fb / ((fc - fa) * (fc - fb))
  Else
    S = b - (b - a) * fb / (fb - fa)
  End If

  If ((3 * a + b) / 4 - S) * (b - S) > 0 Or _
     (mflag = True And Abs(S - b) >= 0.5 * Abs(b - c)) Or _
     (mflag = False And Abs(S - b) >= 0.5 * (c - d)) Then
    S = 0.5 * (a + b)
  Else
    mflag = False
  End If

  fs = liq_air_g_si(0, 1, 0, wa_si, S, p_si)
  If fs = ErrorReturn Then 'permit excursions to the humid-air region
    fs = air_g_si(0, 1, 0, wa_si, S, p_si)
    If fs = ErrorReturn Then Exit Function
  End If
  fs = -fs - eta_si

  d = c
  c = b
  fc = fb

  If fa * fs < 0 Then
    b = S
    fb = fs
  Else
    a = S
    fa = fs
  End If

  If Abs(fa) < Abs(fb) Then
    Swap a, b
    Swap fa, fb
  End If

Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'FLU_LIMITS
  If b < flu_tmin Or b > flu_tmax Then
    temperatureiteration_Brent = ErrorReturn
    Exit Function
  End If
  'AIR_LIMITS
  If b < dry_air_tmin Or b > dry_air_tmax Then
    temperatureiteration_Brent = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Sub Swap(ByRef a As Double, ByRef b As Double)
Dim c As Double
c = a
a = b
b = c
End Sub

'==========================================================================
Private Function liq_air_a_eta_p_derivatives_si(ByVal drv_wa As Integer, _
                                                ByVal drv_eta As Integer, _
                                                ByVal drv_p As Integer, _
                                                ByVal wa_si As Double, _
                                                ByVal t_si As Double, _
                                                ByVal p_si As Double) As Double
                                              
'this function computes the wet-air partial wa-eta-p derivatives of h from wa-t-p derivatives of g

Dim g As Double, gt As Double, gp As Double, ga As Double
Dim gaa As Double, gat As Double, gap As Double
Dim gtt As Double, gpp As Double, gtp As Double

Dim h As Double, d As Double
Dim ctrl0_air As Double, ctrl1_air As Double
Dim ctrl0_liq As Double, ctrl1_liq As Double

'in one case we do not at all need to compute the Gibbs function:
If drv_wa = 0 And drv_eta = 1 And drv_p = 0 Then
  liq_air_a_eta_p_derivatives_si = t_si
  Exit Function
End If

liq_air_a_eta_p_derivatives_si = ErrorReturn

Select Case drv_wa

  Case 0:
    Select Case drv_eta
    
      Case 0:
        Select Case drv_p
        
          Case 0: g = liq_air_g_si(0, 0, 0, wa_si, t_si, p_si)
                  If g = ErrorReturn Then Exit Function
                  gt = liq_air_g_si(0, 1, 0, wa_si, t_si, p_si)
                  If gt = ErrorReturn Then Exit Function
                  h = g - t_si * gt                          'h
          
          Case 1: gp = liq_air_g_si(0, 0, 1, wa_si, t_si, p_si)
                  If gp = ErrorReturn Then Exit Function
                  h = gp                                      'dh/dp
                  
          Case 2: gtt = liq_air_g_si(0, 2, 0, wa_si, t_si, p_si)
                  If gtt = ErrorReturn Then Exit Function
                  If gtt = 0 Then Exit Function
                  gtp = liq_air_g_si(0, 1, 1, wa_si, t_si, p_si)
                  If gtp = ErrorReturn Then Exit Function
                  gpp = liq_air_g_si(0, 0, 2, wa_si, t_si, p_si)
                  If gpp = ErrorReturn Then Exit Function
                  h = (gtt * gpp - gtp ^ 2) / gtt              'd2h/dp2
                  
          Case Else: Exit Function
        End Select
        
      Case 1:
        Select Case drv_p
          Case 0: h = t_si                                     'dh/deta, has already been handled initially
          
          Case 1: gtt = liq_air_g_si(0, 2, 0, wa_si, t_si, p_si)
                  If gtt = ErrorReturn Then Exit Function
                  If gtt = 0 Then Exit Function
                  gtp = liq_air_g_si(0, 1, 1, wa_si, t_si, p_si)
                  If gtp = ErrorReturn Then Exit Function
                  h = -gtp / gtt                               'd2h/detadp
                  
          Case Else: Exit Function
        End Select

      Case 2:
        Select Case drv_p
          Case 0: gtt = liq_air_g_si(0, 2, 0, wa_si, t_si, p_si)
                  If gtt = ErrorReturn Then Exit Function
                  If gtt = 0 Then Exit Function
                  h = -1# / gtt                                'd2h/deta2
                  
          Case Else: Exit Function
        End Select

      Case Else: Exit Function
    End Select
    
  Case 1:
    Select Case drv_eta
    
      Case 0:
        Select Case drv_p
        
          Case 0: ga = liq_air_g_si(1, 0, 0, wa_si, t_si, p_si)
                  If ga = ErrorReturn Then Exit Function
                  h = ga                                        'dh/da
          
          Case 1: gtt = liq_air_g_si(0, 2, 0, wa_si, t_si, p_si)
                  If gtt = ErrorReturn Then Exit Function
                  If gtt = 0 Then Exit Function
                  gat = liq_air_g_si(1, 1, 0, wa_si, t_si, p_si)
                  If gat = ErrorReturn Then Exit Function
                  gap = liq_air_g_si(1, 0, 1, wa_si, t_si, p_si)
                  If gap = ErrorReturn Then Exit Function
                  h = (gtt * gap - gat * gtp) / gtt             'd2h/dadp
          
          Case Else: Exit Function
        End Select

      Case 1:
        Select Case drv_p
        
          Case 0: gtt = liq_air_g_si(0, 2, 0, wa_si, t_si, p_si)
                  If gtt = ErrorReturn Then Exit Function
                  If gtt = 0 Then Exit Function
                  gat = liq_air_g_si(1, 1, 0, wa_si, t_si, p_si)
                  If gat = ErrorReturn Then Exit Function
                  h = -gat / gtt                                'd2h/dadeta
          
          Case Else: Exit Function
        End Select

      Case Else: Exit Function
    End Select

  
  Case 2:
    Select Case drv_eta
    
      Case 0:
        Select Case drv_p
        
          Case 0: gtt = liq_air_g_si(0, 2, 0, wa_si, t_si, p_si)
                  If gtt = ErrorReturn Then Exit Function
                  If gtt = 0 Then Exit Function
                  gat = liq_air_g_si(1, 1, 0, wa_si, t_si, p_si)
                  If gat = ErrorReturn Then Exit Function
                  gaa = liq_air_g_si(2, 0, 0, wa_si, t_si, p_si)
                  If gaa = ErrorReturn Then Exit Function
                  h = (gtt * gaa - gat ^ 2) / gtt               'd2h/da2
          
          Case Else: Exit Function
        End Select

      Case Else: Exit Function
    End Select

  Case Else: Exit Function
End Select

liq_air_a_eta_p_derivatives_si = h

End Function

'==========================================================================
Private Sub init_it_ctrl_pottemp()

If ctrl_initialized = -1 Then Exit Sub

ctrl_initialized = -1

'Set default values and modes for density iteration
ctrl_loop_maximum = 100
ctrl_mode_pottemp = 0           'default: theta = aux_temperature_min_si to start
ctrl_init_pottemp = 273.15
ctrl_eps_exit_pottemp = 0.0001  'default = 0.1 mK

End Sub

'==========================================================================
Public Sub set_it_ctrl_liq_air_pottemp(ByVal key As String, ByVal value As Double)

'this sub sets control parameters for the Newton iteration used to compute
'potential temperature from reference pressure

'key             value
'it_steps        0           set iteration number to default (100)
'it_steps        n > 0       set iteration number to n
'it_steps       -1           do not iterate, use initial value

'init_theta      0           use default potential temperature theta = aux_temperature_min_si to start
'init_theta      t > 0       use value t as potential temperature to start

'tol_theta       0           use default exit accuracy for potential temperature (0.1 mK)
'tol_theta       eps > 0     use eps as exit accuracy for potential temperature

init_it_ctrl_pottemp

Select Case LCase(Trim(key))

  Case "it_steps":   'iteration steps
    Select Case value
      Case 0:      ctrl_loop_maximum = 100     'default = 100
      Case Is < 0: ctrl_loop_maximum = -1
      Case Else:   ctrl_loop_maximum = value
    End Select

  Case "init_theta":   'start theta
    Select Case CLng(value)
      Case 0:      ctrl_mode_pottemp = 0        'default: theta = aux_temperature_min_si
      Case Is > 0: ctrl_mode_pottemp = 1
                   ctrl_init_pottemp = value
    End Select

  Case "tol_theta":      'required theta tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_pottemp = 0.0001 'default = 0.1 mK
      Case Is > 0: ctrl_eps_exit_pottemp = value
    End Select

End Select

End Sub

'==========================================================================
Public Function aux_temperature_max_si(ByVal wa_si As Double, _
                                       ByVal p_si As Double) As Double

'this function estimates the temperature of wet air as a function of
'the dry-air fraction wa_si and the pressure p_si
'in the asymptotic limit of a vanishing liquid fraction, wW << 1.

aux_temperature_max_si = ErrorReturn
If wa_si <= 0 Or wa_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function

'maximum entropy at the saturation point:
set_liq_air_eq_at_a_p wa_si, p_si
aux_temperature_max_si = liq_air_temperature_si

End Function

'==========================================================================
Public Function aux_temperature_min_si(ByVal wa_si As Double, _
                                       ByVal p_si As Double) As Double

'this function estimates the temperature of wet air as a function of
'the dry-air fraction wa_si and the pressure p_si
'in the asymptotic limit of a freezing liquid fraction.

aux_temperature_min_si = ErrorReturn
If wa_si < 0 Or wa_si > 1 Then Exit Function
If p_si <= 0 Then Exit Function

'minimum entropy at the freezing point:
aux_temperature_min_si = ice_liq_meltingtemperature_si(p_si)

End Function


