Attribute VB_Name = "Liq_Air_4b_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules:
'     Constants_0_Mdl, file Constants_0.bas
'     Air_3a_Mdl,      file Air_3a.bas
'     Flu_3a_Mdl,      file Flu_3a.bas
'     Liq_Air_4a_Mdl,  file Liq_Air_4a.bas
'
'#########################################################################

'=========================================================================
'This module implements the Gibbs function of wet air, i.e., of the composite
'system of liquid water and humid air at mutual equilibrium ("cloudy air")
'Therefore, the humid-air properties computed here refer to saturated air.

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009

'Feistel, R., Kretzschmar, H.-J., Span, R., Hagen, E., Wright, D.G., Herrmann, S.:
'Thermodynamic Properties of Sea Air.
'Ocean Science Discussion 6(2009)21932325.
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98
'Private Const IsOK = -1

Private Const Version = "07 Jun 2010"

'==========================================================================
Public Function liq_air_g_si(ByVal drv_wa As Integer, _
                             ByVal drv_t As Integer, _
                             ByVal drv_p As Integer, _
                             ByVal wa_si As Double, _
                             ByVal t_si As Double, _
                             ByVal p_si As Double) As Double

'this implements the Gibbs function of wet air computed from its Helmholtz function
'and its first and second derivatives with respect to the air fraction wa_si,
'the absolute temperature t_si and the absolute pressure p_si

'note: this Gibbs function is not designed for liquid air or critical air states
'note: the accuracy of this functions depends on the iteration settings
'in Air_3a to compute the Gibbs function of humid air and on the settings
'in Flu_3a to compute the Gibbs function of liquid water

'Output: liq_air_g_si = specific Gibbs energy in J/kg or its derivative

' g(wa,T,P) = (1-w) * gl(T,P) + w * gh(A(T,P),T,P)

'  w = mass fraction of humid air in wet air, w(wa, T, P) = wa/A(T,P)
'     (1-w) = liquid mass fraction of wet air
'  A = saturated air mass fraction of humid air
'     (1-A) = saturated specific humidity of the humid-air component
' gl = Gibbs function of liquid water
' gh = Gibbs function of humid air
'
'Input:
'drv_wa = order of the partial wa-derivative
' drv_t = order of the partial t-derivative
' drv_p = order of the partial p-derivative
' wa_si = absolute air fraction in kg/kg, i.e. the mass fraction of dry air in wet air
' t_si  = T = absolute temperature in K, ITS-90
' p_si  = P = absolute pressure in Pa

'check values with default settings: v. 1.0
'liq_air_g_si( 0, 0, 0, 0.5, 300, 1E5) =-5397.43216341774
'liq_air_g_si( 1, 0, 0, 0.5, 300, 1E5) =-264.7634152|91866
'liq_air_g_si( 0, 1, 0, 0.5, 300, 1E5) =-343.818233767934
'liq_air_g_si( 0, 0, 1, 0.5, 300, 1E5) = 0.446835172414366
'liq_air_g_si( 2, 0, 0, 0.5, 300, 1E5) = 0
'liq_air_g_si( 1, 1, 0, 0.5, 300, 1E5) = 98.4884000932715
'liq_air_g_si( 1, 0, 1, 0.5, 300, 1E5) = 0.891663433709969
'liq_air_g_si( 0, 2, 0, 0.5, 300, 1E5) =-14.1012539815792
'liq_air_g_si( 0, 1, 1, 0.5, 300, 1E5) = 2.43241086431687E-03
'liq_air_g_si( 0, 0, 2, 0.5, 300, 1E5) =-4.6246973798681E-06

'check values with default settings: v. 1.1
'liq_air_g_si( 0, 0, 0, 0.5, 300, 1E5) =-5396.77820137313
'liq_air_g_si( 1, 0, 0, 0.5, 300, 1E5) =-263.45549120|2659
'liq_air_g_si( 0, 1, 0, 0.5, 300, 1E5) =-343.78339387248
'liq_air_g_si( 0, 0, 1, 0.5, 300, 1E5) = 0.446729465554996
'liq_air_g_si( 2, 0, 0, 0.5, 300, 1E5) = 0
'liq_air_g_si( 1, 1, 0, 0.5, 300, 1E5) = 98.5580798841792
'liq_air_g_si( 1, 0, 1, 0.5, 300, 1E5) = 0.89145201999123
'liq_air_g_si( 0, 2, 0, 0.5, 300, 1E5) =-14.0995955396963
'liq_air_g_si( 0, 1, 1, 0.5, 300, 1E5) = 2.43183979422311E-03
'liq_air_g_si( 0, 0, 2, 0.5, 300, 1E5) =-4.62360294022528E-06

Dim g As Double, a As Double, w As Double, d As Double
Dim gh As Double, gl As Double
Dim gh_p As Double, gl_p As Double
Dim gh_t As Double, gl_t As Double
Dim gh_pp As Double, gl_pp As Double, gc_pp As Double
Dim gh_tp As Double, gl_tp As Double, gc_tp As Double
Dim gh_tt As Double, gl_tt As Double, gc_tt As Double
Dim gh_a As Double, gh_ap As Double, gh_at As Double

Dim a_p As Double, a_t As Double

liq_air_g_si = ErrorReturn

If wa_si < 0 Or wa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

If drv_wa < 0 Then Exit Function
If drv_t < 0 Then Exit Function
If drv_p < 0 Then Exit Function
If drv_wa + drv_t + drv_p > 2 Then Exit Function

If drv_wa > 1 Then    'g is linear in wa_si
  liq_air_g_si = 0
  Exit Function
End If

a = liq_air_massfraction_air_si(t_si, p_si)      'saturated air mass fraction
If a = ErrorReturn Then Exit Function
If a <= 0 Or a >= 1 Then Exit Function

If a < wa_si Then   'no liquid fraction at this t and p
  'instead, invoke the Gibbs function of unsaturated humid air
  'liq_air_g_si = air_g_si(drv_wa, drv_t, drv_p, wa_si, t_si, p_si)
  Exit Function
End If

w = wa_si / a               'gaseous mass fraction of wet air

If drv_t + drv_p > 1 Then
  d = a ^ 2 * air_g_si(2, 0, 0, a, t_si, p_si) 'air coefficient DA = A^2 * g_aa
End If

Select Case drv_wa

  Case 0:

    Select Case drv_t

      Case 0:     'd/dt

        Select Case drv_p

          Case 0: gh = air_g_si(0, 0, 0, a, t_si, p_si)
                  If gh = ErrorReturn Then Exit Function
                  gl = liq_g_si(0, 0, t_si, p_si)
                  If gl = ErrorReturn Then Exit Function
                  g = (1 - w) * gl + w * gh                      'g

          Case 1: gh_p = air_g_si(0, 0, 1, a, t_si, p_si)
                  If gh_p = ErrorReturn Then Exit Function
                  gl_p = liq_g_si(0, 1, t_si, p_si)
                  If gl_p = ErrorReturn Then Exit Function
                  g = (1 - w) * gl_p + w * gh_p                  'v = g_p

          Case 2: a_p = liq_air_a_si(0, 1, t_si, p_si)
                  If a_p = ErrorReturn Then Exit Function
                  gc_pp = -d * a_p ^ 2 / a                      'latent condensation derivative
                  gh_pp = air_g_si(0, 0, 2, a, t_si, p_si)
                  If gh_pp = ErrorReturn Then Exit Function
                  gl_pp = liq_g_si(0, 2, t_si, p_si)
                  If gl_pp = ErrorReturn Then Exit Function
                  g = (1 - w) * gl_pp + w * (gh_pp + gc_pp)      'g_pp

          Case Else: Exit Function
        End Select

      Case 1:     'd/dt

        Select Case drv_p

          Case 0: gh_t = air_g_si(0, 1, 0, a, t_si, p_si)
                  If gh_t = ErrorReturn Then Exit Function
                  gl_t = liq_g_si(1, 0, t_si, p_si)
                  If gl_t = ErrorReturn Then Exit Function
                  g = (1 - w) * gl_t + w * gh_t                  '-eta = g_t

          Case 1: a_t = liq_air_a_si(1, 0, t_si, p_si)
                  If a_t = ErrorReturn Then Exit Function
                  a_p = liq_air_a_si(0, 1, t_si, p_si)
                  If a_p = ErrorReturn Then Exit Function
                  gc_tp = -d * a_t * a_p / a                     'latent derivative
                  gh_tp = air_g_si(0, 1, 1, a, t_si, p_si)
                  If gh_tp = ErrorReturn Then Exit Function
                  gl_tp = liq_g_si(1, 1, t_si, p_si)
                  If gl_tp = ErrorReturn Then Exit Function
                  g = (1 - w) * gl_tp + w * (gh_tp + gc_tp)                 'g_tp

          Case Else: Exit Function
        End Select

      Case 2:     'd2/dt2

        Select Case drv_p

          Case 0: a_t = liq_air_a_si(1, 0, t_si, p_si)
                  If a_t = ErrorReturn Then Exit Function
                  gc_tt = -d * a_t ^ 2 / a                      'latent derivative
                  gh_tt = air_g_si(0, 2, 0, a, t_si, p_si)
                  If gh_tt = ErrorReturn Then Exit Function
                  gl_tt = liq_g_si(2, 0, t_si, p_si)
                  If gl_tt = ErrorReturn Then Exit Function
                  g = (1 - w) * gl_tt + w * (gh_tt + gc_tt)                 'g_tt

          Case Else: Exit Function
        End Select

       Case Else: Exit Function
    End Select

  Case 1:       'g_w, g_wp, g_wt
    gh = air_g_si(0, drv_t, drv_p, a, t_si, p_si)
    If gh = ErrorReturn Then Exit Function
    gl = liq_g_si(drv_t, drv_p, t_si, p_si)
    If gl = ErrorReturn Then Exit Function
    g = (gh - gl) / a
  
  Case 2: g = 0
  
  Case Else: Exit Function
End Select


liq_air_g_si = g

End Function

'=========================================================================
Private Function liq_air_a_si(ByVal drv_t As Integer, _
                              ByVal drv_p As Integer, _
                              ByVal t_si As Double, _
                              ByVal p_si As Double) As Double

'this function returns the air fraction a of saturated humid air in kg/kg,
'and its partial derivatives (d/dt)^drv_t (d/dp)^drv_p a(t,p)
'as a function of absolute temperature, t_si, in K, and absolute pressure, p_si, in Pa

Dim d As Double  'DA coefficient
Dim a As Double  'air fraction

Dim gh_p As Double, gl_p As Double, gh_ap As Double
Dim gh_t As Double, gl_t As Double, gh_at As Double

liq_air_a_si = ErrorReturn

If drv_t < 0 Or drv_t > 1 Then Exit Function
If drv_p < 0 Or drv_p > 1 Then Exit Function

If t_si < 0 Then Exit Function
If p_si < 0 Then Exit Function

a = liq_air_massfraction_air_si(t_si, p_si)
If a = ErrorReturn Then Exit Function
If a < 0 Or a >= 1 Then Exit Function

If drv_t > 0 Or drv_p > 0 Then
  d = a ^ 2 * air_g_si(2, 0, 0, a, t_si, p_si) 'air coefficient DA = A^2 * g_aa
  If d = ErrorReturn Then Exit Function
  If d <= 0 Then Exit Function
End If

Select Case drv_t

  Case 0:

    Select Case drv_p

      Case 0: liq_air_a_si = a

      Case 1: gh_p = air_g_si(0, 0, 1, a, t_si, p_si)
              If gh_p = ErrorReturn Then Exit Function
              gh_ap = air_g_si(1, 0, 1, a, t_si, p_si)
              If gh_ap = ErrorReturn Then Exit Function
              gl_p = liq_g_si(0, 1, t_si, p_si)
              If gl_p = ErrorReturn Then Exit Function
              liq_air_a_si = a * (gh_p - gl_p - a * gh_ap) / d
    End Select

  Case 1:

    Select Case drv_p

      Case 0: gh_t = air_g_si(0, 1, 0, a, t_si, p_si)
              If gh_t = ErrorReturn Then Exit Function
              gh_at = air_g_si(1, 1, 0, a, t_si, p_si)
              If gh_at = ErrorReturn Then Exit Function
              gl_t = liq_g_si(1, 0, t_si, p_si)
              If gl_t = ErrorReturn Then Exit Function
              liq_air_a_si = a * (gh_t - gl_t - a * gh_at) / d
    End Select

End Select

End Function

'=========================================================================
Public Function liq_air_liquidfraction_si(ByVal wa_si As Double, _
                                          ByVal t_si As Double, _
                                          ByVal p_si As Double) As Double

'returns   wW = 1 - wA/A(T,P) mass fraction of liquid water in wet air (0 < wW < 1)
'wa_si     dry-air mass fraction of wet air in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value with default settings:
'liq_air_liquidfraction_si(0.5, 300, 1E5) = 0.488543697764709  v. 1.0
'liq_air_liquidfraction_si(0.5, 300, 1E5) = 0.488546404734467  v. 1.1

Dim a As Double

liq_air_liquidfraction_si = ErrorReturn

If wa_si <= 0 Or wa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

a = liq_air_massfraction_air_si(t_si, p_si)
If a = ErrorReturn Then Exit Function
If a < wa_si Then Exit Function

liq_air_liquidfraction_si = 1 - wa_si / a

End Function

'=========================================================================
Public Function liq_air_vapourfraction_si(ByVal wa_si As Double, _
                                          ByVal t_si As Double, _
                                          ByVal p_si As Double) As Double

'returns   wV = wa*(1/A(T,P)-1) mass fraction of vapour in wet air (0 < wV < 1)
'wa_si     dry-air mass fraction of wet air in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value with default settings:
'liq_air_vapourfraction_si(0.5, 300, 1E5) = 1.14563022352915E-02  v. 1.0
'liq_air_vapourfraction_si(0.5, 300, 1E5) = 1.14535952655325E-02  v. 1.1

Dim a As Double

liq_air_vapourfraction_si = ErrorReturn

If wa_si <= 0 Or wa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

a = liq_air_massfraction_air_si(t_si, p_si)
If a = ErrorReturn Then Exit Function
If a < wa_si Then Exit Function

liq_air_vapourfraction_si = wa_si * (1 / a - 1)

End Function

'=========================================================================
Public Function liq_air_g_cp_si(ByVal wa_si As Double, _
                                ByVal t_si As Double, _
                                ByVal p_si As Double) As Double

'returns   cp(wa,T,P) = -T * (d2g/dT2) heat capacity of wet air in J/(kg K),
'wa_si     mass fraction of dry air in wet air in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value with default settings:
'liq_air_g_cp_si(0.5, 300, 1E5) = 4230.37619447375  v. 1.0
'liq_air_g_cp_si(0.5, 300, 1E5) = 4229.87866190888  v. 1.1

Dim g_tt As Double

liq_air_g_cp_si = ErrorReturn

If wa_si <= 0 Or wa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g_tt = liq_air_g_si(0, 2, 0, wa_si, t_si, p_si)
If g_tt = ErrorReturn Then Exit Function

liq_air_g_cp_si = -t_si * g_tt

End Function

'=========================================================================
Public Function liq_air_g_density_si(ByVal wa_si As Double, _
                                     ByVal t_si As Double, _
                                     ByVal p_si As Double) As Double

'returns   d(wa,T,P) = 1/(dg/dp)  density of wet air in kg/m3,
'wa_si     mass fraction of dry air in wet air in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value with default settings:
'liq_air_g_density_si(0.5, 300, 1E5) = 2.23796169535344  v. 1.0
'liq_air_g_density_si(0.5, 300, 1E5) = 2.23849125053268  v. 1.1

Dim g_p As Double

liq_air_g_density_si = ErrorReturn

If wa_si <= 0 Or wa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g_p = liq_air_g_si(0, 0, 1, wa_si, t_si, p_si)
If g_p = ErrorReturn Then Exit Function
If g_p <= 0 Then Exit Function

liq_air_g_density_si = 1# / g_p

End Function

'=========================================================================
Public Function liq_air_g_enthalpy_si(ByVal wa_si As Double, _
                                      ByVal t_si As Double, _
                                      ByVal p_si As Double) As Double

'returns   h(wa,T,P) = g - T * (dg/dT)  enthalpy of wet air in J/kg,
'wa_si     mass fraction of dry air in wet air in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value with default settings:
'liq_air_g_enthalpy_si(0.5, 300, 1E5) = 97748.0379669623  v. 1.0
'liq_air_g_enthalpy_si(0.5, 300, 1E5) = 97738.2399603708  v. 1.1

Dim g As Double, g_t As Double

liq_air_g_enthalpy_si = ErrorReturn

If wa_si <= 0 Or wa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g = liq_air_g_si(0, 0, 0, wa_si, t_si, p_si)
If g = ErrorReturn Then Exit Function
g_t = liq_air_g_si(0, 1, 0, wa_si, t_si, p_si)
If g_t = ErrorReturn Then Exit Function

liq_air_g_enthalpy_si = g - t_si * g_t

End Function

'=========================================================================
Public Function liq_air_g_entropy_si(ByVal wa_si As Double, _
                                     ByVal t_si As Double, _
                                     ByVal p_si As Double) As Double

'returns   eta(wa,T,P) = -(dg/dT)  entropy of wet air in J/(kg K),
'wa_si     mass fraction of dry air in wet air in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value with default settings:
'liq_air_g_entropy_si(0.5, 300, 1E5) = 343.818233767934  v. 1.0
'liq_air_g_entropy_si(0.5, 300, 1E5) = 343.78339387248  v. 1.1

Dim g_t As Double

liq_air_g_entropy_si = ErrorReturn

If wa_si <= 0 Or wa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g_t = liq_air_g_si(0, 1, 0, wa_si, t_si, p_si)
If g_t = ErrorReturn Then Exit Function

liq_air_g_entropy_si = -g_t

End Function

'=========================================================================
Public Function liq_air_g_expansion_si(ByVal wa_si As Double, _
                                       ByVal t_si As Double, _
                                       ByVal p_si As Double) As Double

'returns   alpha(wa,T,P) = (d2g/dTdP)/(dg/dP)  thermal expansion of wet air in 1/K,
'wa_si     mass fraction of dry air in wet air in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value with default settings:
'liq_air_g_expansion_si(0.5, 300, 1E5) = 5.44364234170271E-03  v. 1.0
'liq_air_g_expansion_si(0.5, 300, 1E5) = 5.44365210206562E-03  v. 1.1

Dim g_p As Double, g_tp As Double

liq_air_g_expansion_si = ErrorReturn

If wa_si <= 0 Or wa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g_p = liq_air_g_si(0, 0, 1, wa_si, t_si, p_si)
If g_p = ErrorReturn Then Exit Function
If g_p <= 0 Then Exit Function
g_tp = liq_air_g_si(0, 1, 1, wa_si, t_si, p_si)
If g_tp = ErrorReturn Then Exit Function

liq_air_g_expansion_si = g_tp / g_p

End Function

'=========================================================================
Public Function liq_air_g_kappa_t_si(ByVal wa_si As Double, _
                                     ByVal t_si As Double, _
                                     ByVal p_si As Double) As Double

'returns   kappa_t(wa,T,P) = - (d2g/dP2)/(dg/dP)  isothermal compressibility of wet air in 1/Pa,
'wa_si     mass fraction of dry air in wet air in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value with default settings:
'liq_air_g_kappa_t_si(0.5, 300, 1E5) = 1.03498955887462E-05  v. 1.0
'liq_air_g_kappa_t_si(0.5, 300, 1E5) = 1.03498947276315E-05  v. 1.1

Dim g_p As Double, g_pp As Double

liq_air_g_kappa_t_si = ErrorReturn

If wa_si <= 0 Or wa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g_p = liq_air_g_si(0, 0, 1, wa_si, t_si, p_si)
If g_p = ErrorReturn Then Exit Function
If g_p <= 0 Then Exit Function
g_pp = liq_air_g_si(0, 0, 2, wa_si, t_si, p_si)
If g_pp = ErrorReturn Then Exit Function

liq_air_g_kappa_t_si = -g_pp / g_p

End Function

'=========================================================================
Public Function liq_air_g_lapserate_si(ByVal wa_si As Double, _
                                       ByVal t_si As Double, _
                                       ByVal p_si As Double) As Double

'returns   gamma(wa,T,P) = -(d2g/dTdP)/(d2g/dT2) "moist" adiabatic lapse rate of wet air in K/Pa,
'wa_si     mass fraction of dry air in wet air in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value with default settings:
'liq_air_g_lapserate_si(0.5, 300, 1E5) = 1.724960679025E-04  v. 1.0
'liq_air_g_lapserate_si(0.5, 300, 1E5) = 1.72475854883671E-04  v. 1.1

Dim g_tp As Double, g_tt As Double

liq_air_g_lapserate_si = ErrorReturn

If wa_si <= 0 Or wa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g_tp = liq_air_g_si(0, 1, 1, wa_si, t_si, p_si)
If g_tp = ErrorReturn Then Exit Function
If g_tp <= 0 Then Exit Function
g_tt = liq_air_g_si(0, 2, 0, wa_si, t_si, p_si)
If g_tt = ErrorReturn Then Exit Function
If g_tt >= 0 Then Exit Function

liq_air_g_lapserate_si = -g_tp / g_tt

End Function


