Attribute VB_Name = "Flu_3a_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules:
'     Constants_0_Mdl,  file Constants_0.bas
'     Flu_1_Mdl,        file Flu_1.bas
'     Maths_0_Mdl,      file Maths_0.bas
'     Convert_0_Mdl,    file Convert_0.bas

'#########################################################################

'=========================================================================
'This module implements the Gibbs functions of liquid water and vapour
'depending on temperature and pressure, as well as their partial derivatives,
'computed numerically from the Helmholtz function of fluid water, IAPWS-95.

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98

'Control parameters of the density iteration
Private ctrl_initialized As Integer
Private ctrl_mode_liquid As Integer
Private ctrl_mode_vapour As Integer

Private ctrl_loop_maximum As Long

Private ctrl_density_liquid As Double
Private ctrl_density_vapour As Double

Private ctrl_eps_exit_liquid As Double
Private ctrl_eps_exit_vapour As Double

Private ctrl_method_liquid As Long
Private ctrl_method_vapour As Long

Private ctrl_density2_liquid As Double
Private ctrl_density2_vapour As Double

'Coefficients of IF97
'Coefficients of region 1
Private i1i(34) As Integer
Private j1i(34) As Integer
Private n1i(34) As Double

'Coefficients of region 2, ideal part
Private j0i(9) As Integer
Private n0i(9) As Double

'Coefficients of region 2, residual part
Private iri(43) As Integer
Private jri(43) As Integer
Private nri(43) As Double

'Coefficients of the F03 Gibbs function
Private Const maxt = 7, maxp = 6
Private gc(maxt, maxp) As Double

Private Const Version = "22 Dec 2009"

'==========================================================================
Public Function liq_g_si(ByVal drv_t As Integer, _
                         ByVal drv_p As Integer, _
                         ByVal t_si As Double, _
                         ByVal p_si As Double) As Double

'this implements the Gibbs function of liquid water computed from IAPWS-95
'and its first and second derivatives with respect to the absolute temperature t_si
'and the absolute pressure p_si

'note: the accuracy of this functions depends on the iteration settings of this module

'Output: liq_g_si = specific Gibbs energy in J/kg or its derivative

'Input:  drv_t = order of temperature derivative, 0 <= drv_t <= 2
'        drv_p = order of pressure derivative, 0 <= drv_p + drv_t <= 2
'        t_si = absolute temperature, in K
'        p_si = absolute pressure, in Pa

'Check values with default settings:
'liq_g_si( 0, 0, 300, 1E5) =-5265.0504557718
'liq_g_si( 1, 0, 300, 1E5) =-393.062433814569
'liq_g_si( 0, 1, 300, 1E5) = 1.00345555938138E-03
'liq_g_si( 2, 0, 300, 1E5) =-13.9354650734086
'liq_g_si( 1, 1, 300, 1E5) = 2.75753316815429E-07
'liq_g_si( 0, 2, 300, 1E5) =-4.52072086722098E-13

Dim g As Double, d As Double

liq_g_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = liq_density_si(t_si, p_si)  'numerical inverse function of IAPWS-95

If d = ErrorReturn Then Exit Function
If d <= 0 Then Exit Function

g = flu_t_p_derivative_si(drv_t, drv_p, t_si, d)
If g = ErrorReturn Then Exit Function

liq_g_si = g

End Function

'==========================================================================
Public Function vap_g_si(ByVal drv_t As Integer, _
                         ByVal drv_p As Integer, _
                         ByVal t_si As Double, _
                         ByVal p_si As Double) As Double

'this implements the Gibbs function of water vapour computed from IAPWS-95
'and its first and second derivatives with respect to the absolute temperature t_si
'and the absolute pressure p_si

'note: the accuracy of this functions depends on the iteration settings of this module

'Output: liq_g_si = specific Gibbs energy in J/kg or its derivative

'Input:  drv_t = order of temperature derivative, 0 <= drv_t <= 2
'        drv_p = order of pressure derivative, 0 <= drv_p + drv_t <= 2
'        t_si = absolute temperature, in K
'        p_si = absolute pressure, in Pa

'Check values with default settings:
'vap_g_si( 0, 0, 300, 1E3) =-180090.341338025
'vap_g_si( 1, 0, 300, 1E3) =-9103.67940087138
'vap_g_si( 0, 1, 300, 1E3) = 138.38847806943
'vap_g_si( 2, 0, 300, 1E3) =-6.24707163426883
'vap_g_si( 1, 1, 300, 1E3) = 0.462704658817776
'vap_g_si( 0, 2, 300, 1E3) =-0.138455798863587

Dim g As Double, d As Double

vap_g_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = vap_density_si(t_si, p_si)  'numerical inverse function of IAPWS-95

If d = ErrorReturn Then Exit Function
If d <= 0 Then Exit Function

g = flu_t_p_derivative_si(drv_t, drv_p, t_si, d)
If g = ErrorReturn Then Exit Function

vap_g_si = g

End Function

'==========================================================================
Public Function liq_density_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'this function returns the density of liquid water as a function of temperature and pressure

'Output: liq_density_si = density in kg/m3

'Input:  t_si = absolute temperature, in K
'        p_si = absolute pressure, in Pa

'Check value with default settings: liq_density_si(300, 1E5) = 996.556340388894

Dim d As Double, d2 As Double, eps As Double
Dim maxit As Long

liq_density_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

'avoid numerical problems at the very critical point
If t_si = CP_temperature_si And p_si = CP_pressure_si Then
  liq_density_si = CP_density_si
  Exit Function
End If

init_it_ctrl_density

'consider the control settings for the iteration
Select Case ctrl_mode_liquid
  Case 1:  d = ctrl_density_liquid
  Case 0:
           If t_si <= 623.15 And p_si <= 100000000# Then
             'IF-97 at subcritical temperature, liquid
             d = aux_liq_density_if97_si(t_si, p_si)
           ElseIf t_si <= CP_temperature_si And p_si <= 16529000# Then
             'IF-97 at subcritical temperature, superheated liquid
             d = aux_liq_density_if97_si(t_si, p_si)
           ElseIf t_si <= 1073.15 And p_si <= 16529000# Then
             'IF-97 at subcritical pressure, fluid
             d = aux_vap_density_if97_si(t_si, p_si)
           ElseIf t_si <= 650 And p_si <= 35000000 Then
              'cubic EOF in the critical region
             d = aux_liq_density_critical_si(t_si, p_si)
           ElseIf t_si <= 650 Then
             'dense fluid
             d = 1000
           Else
            'ideal gas anywhere else
             d = aux_density_ideal_si(t_si, p_si)
           End If
  Case -1: d = aux_liq_density_if97_si(t_si, p_si)
  Case -2: d = aux_density_EOS80_si(t_si, p_si)
  Case -3: d = aux_liq_density_f03_si(t_si, p_si)
  Case Else: Exit Function
End Select

If d = ErrorReturn Then Exit Function
If d <= 0 Then Exit Function

Select Case ctrl_loop_maximum
  Case 0:      maxit = 100
  Case -1:     liq_density_si = d
               Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else:   Exit Function
End Select

eps = ctrl_eps_exit_liquid
If eps = 0 Then Exit Function

If ctrl_method_liquid > 1 Then
  'specify the second point for Secant or Brent method
  Select Case ctrl_density2_liquid
    Case 0:  d2 = DensityIteration_Newton(t_si, p_si, d, 1, 1E+99)
             If d2 = ErrorReturn Then Exit Function
             If ctrl_method_liquid = 2 Then 'Brent
               d2 = d + 2 * (d2 - d)
             End If
    Case -1: d2 = DensityIteration_Newton(t_si, p_si, d, 1, 1E+99)
             If d2 = ErrorReturn Then Exit Function
    Case -2: d2 = DensityIteration_Newton(t_si, p_si, d, 1, 1E+99)
             If d2 = ErrorReturn Then Exit Function
             d2 = d + 0.5 * (d2 - d)
    Case -3: d2 = DensityIteration_Newton(t_si, p_si, d, 1, 1E+99)
             If d2 = ErrorReturn Then Exit Function
             d2 = d + 2 * (d2 - d)
    Case Is > 0: d2 = ctrl_density2_liquid
    Case Else: Exit Function
  End Select
End If

'run the iteration
Select Case ctrl_method_liquid
  Case 0: d = DensityIteration_Newton(t_si, p_si, d, maxit, eps)
  Case 1: d = DensityIteration_Newton(t_si, p_si, d, maxit, eps)
  Case 2: d = DensityIteration_Brent(t_si, p_si, d, d2, maxit, eps)
  Case 3: d = DensityIteration_Secant(t_si, p_si, d, d2, maxit, eps)
  Case Else: Exit Function
End Select

If d = ErrorReturn Then Exit Function
If d <= 0 Then Exit Function

'avoid accidental vapour density
If t_si < CP_temperature_si And p_si < CP_pressure_si Then
 If d < CP_density_si Then Exit Function
End If


liq_density_si = d

End Function

'==========================================================================
Public Function vap_density_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'this function returns the density of water vapour as a function of temperature and pressure

'Output: vap_density_si = density in kg/m3

'Input:  t_si = absolute temperature, in K
'        p_si = absolute pressure, in Pa

'Check value with default settings: vap_density_si(300, 1E3) = 7.2260351002509E-03


Dim d As Double, d2 As Double, eps As Double
Dim maxit As Long

vap_density_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

'avoid numerical problems at the very critical point
If t_si = CP_temperature_si And p_si = CP_pressure_si Then
  vap_density_si = CP_density_si
  Exit Function
End If

init_it_ctrl_density

'consider the control settings for the iteration
Select Case ctrl_mode_liquid
  Case 1: d = ctrl_density_vapour
  Case 0:
           If t_si <= 273.15 Then
             d = aux_density_ideal_si(t_si, p_si)
           ElseIf t_si <= 623.15 And p_si <= CP_pressure_si Then
             'IF-97 at subcritical pressure, vapour
             d = aux_vap_density_if97_si(t_si, p_si)
           ElseIf t_si <= 1073.15 And p_si <= 16529000 Then
             'IF-97 at subcritical pressure, fluid
             d = aux_vap_density_if97_si(t_si, p_si)
           ElseIf t_si <= 623.15 And p_si <= 100000000# Then
             'IF-97 at subcritical temperature, liquid
             d = aux_liq_density_if97_si(t_si, p_si)
           ElseIf t_si <= 650 And p_si <= CP_pressure_si Then
              'cubic EOF in the critical region, vapour
             d = aux_vap_density_critical_si(t_si, p_si)
           ElseIf t_si <= 650 And p_si <= 35000000 Then
              'cubic EOF in the critical region, liquid since p > pc
             d = aux_liq_density_critical_si(t_si, p_si)
           ElseIf t_si <= 650 Then
             'dense fluid
             d = 1000
           Else
            'ideal gas anywhere else
             d = aux_density_ideal_si(t_si, p_si)
           End If
  Case -1: d = aux_vap_density_if97_si(t_si, p_si)
  Case -2: d = aux_density_ideal_si(t_si, p_si)
  Case Else: Exit Function
End Select

If d = ErrorReturn Then Exit Function
If d <= 0 Then Exit Function

Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: vap_density_si = d
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

eps = ctrl_eps_exit_vapour
If eps = 0 Then Exit Function

If ctrl_method_vapour > 1 Then
  'specify the second point for Secant or Brent method
  Select Case ctrl_density2_vapour
    Case 0:  d2 = DensityIteration_Newton(t_si, p_si, d, 1, 1E+99)
             If d2 = ErrorReturn Then Exit Function
             If ctrl_method_vapour = 2 Then 'Brent
               d2 = d + 2 * (d2 - d)
             End If
    Case -1: d2 = DensityIteration_Newton(t_si, p_si, d, 1, 1E+99)
             If d2 = ErrorReturn Then Exit Function
    Case -2: d2 = DensityIteration_Newton(t_si, p_si, d, 1, 1E+99)
             If d2 = ErrorReturn Then Exit Function
             d2 = d + 0.5 * (d2 - d)
    Case -3: d2 = DensityIteration_Newton(t_si, p_si, d, 1, 1E+99)
             If d2 = ErrorReturn Then Exit Function
             d2 = d + 2 * (d2 - d)
    Case Is > 0: d2 = ctrl_density2_vapour
    Case Else: Exit Function
  End Select
End If

'run the iteration
Select Case ctrl_method_vapour
  Case 0: d = DensityIteration_Newton(t_si, p_si, d, maxit, eps)
  Case 1: d = DensityIteration_Newton(t_si, p_si, d, maxit, eps)
  Case 2: d = DensityIteration_Brent(t_si, p_si, d, d2, maxit, eps)
  Case 3: d = DensityIteration_Secant(t_si, p_si, d, d2, maxit, eps)
  Case Else: Exit Function
End Select


If d = ErrorReturn Then Exit Function
If d <= 0 Then Exit Function

'avoid accidental liquid density
If t_si < CP_temperature_si And p_si < CP_pressure_si Then
 If d > CP_density_si Then Exit Function
End If

vap_density_si = d

End Function

'==========================================================================
Private Function DensityIteration_Newton(ByVal t As Double, _
                                         ByVal p As Double, _
                                         ByVal d As Double, _
                                         ByVal maxit As Integer, _
                                         ByVal eps As Double) As Double

'The function returns the density as a function of temperature and pressure,
'computed from the Helmholtz function by Newton iteration
'http://en.wikipedia.org/wiki/Newton%27s_method

'output: DensityIteration_Newton: density in kg/m3
'
'        The value ErrorReturn is returned if
'        - the maximum number of iterations is exceeded without meeting the exit criterion
'        - the function call to flu_f_si has returned an error
'        - density has taken a zero or negative value during the iteration

'input:  t: absolute temperature in K
'        p: absolute pressure in Pa
'        d: initial guess for density in kg/m3
'    maxit: maximum number of iteration steps to be done
'      eps: required accuracy of density
'           eps > 0: absolute density uncertainty in kg/m3
'           eps < 0: relative density uncertainty

Dim dd As Double, fd As Double, fdd As Double
Dim it As Long

DensityIteration_Newton = ErrorReturn

If p <= 0 Then Exit Function

If check_limits = 1 Then
  If t < flu_tmin Or t > flu_tmax Then Exit Function
Else
  If t <= 0 Then Exit Function
End If

check_limits = check_limits - 1

For it = 1 To maxit

  fd = flu_f_si(0, 1, t, d)
  If fd = ErrorReturn Then Exit For
  
  fdd = flu_f_si(0, 2, t, d)
  If fdd = ErrorReturn Then Exit For
  
  dd = d * (2 * fd + d * fdd)
  If dd = 0 Then Exit For
  
  dd = (p - d ^ 2 * fd) / dd
  d = d + dd

  If d <= 0 Then Exit For

  If eps > 0 Then                'absolute limit
    If Abs(dd) < eps Then
      DensityIteration_Newton = d
      Exit For
    End If
  Else                           'relative limit
    If Abs(dd) < -eps * d Then
      DensityIteration_Newton = d
      Exit For
    End If
  End If
  
Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  If d <= flu_dmin Or d > flu_dmax Then
    DensityIteration_Newton = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Function DensityIteration_Brent(ByVal t As Double, _
                                        ByVal p As Double, _
                                        ByVal d1 As Double, _
                                        ByVal d2 As Double, _
                                        ByVal maxit As Integer, _
                                        ByVal eps As Double) As Double

'The function returns the density as a function of temperature and pressure,
'computed from the Helmholtz function by Brent iteration
'http://en.wikipedia.org/wiki/Brent's_method

'output: DensityIteration_Brent: density in kg/m^3
'
'        The value ErrorReturn is returned if
'        - the maximum number of iterations is exceeded without meeting the exit criterion
'        - the function call to flu_f_si has returned an error
'        - density has taken a zero or negative value during the iteration

'input:  t: absolute temperature in K
'        p: absolute pressure in Pa
'       d1: initial guess for density in kg/m^3
'       d2: counterpoint density
'    maxit: maximum number of iteration steps to be done
'      eps: required accuracy of density
'           eps > 0: absolute density uncertainty in kg/m^3
'           eps < 0: relative density uncertainty

Dim a As Double, b As Double, c As Double, d As Double, S As Double
Dim fa As Double, fb As Double, fc As Double, fs As Double
Dim mflag As Boolean
Dim it As Long

DensityIteration_Brent = ErrorReturn

If p <= 0 Then Exit Function

If check_limits = 1 Then
  If t < flu_tmin Or t > flu_tmax Then Exit Function
Else
  If t <= 0 Then Exit Function
End If

check_limits = check_limits - 1

a = d1
fa = a ^ 2 * flu_f_si(0, 1, t, a) - p
b = d2
fb = b ^ 2 * flu_f_si(0, 1, t, b) - p
If fa * fb > 0 Then GoTo ExitFunction

If Abs(fa) < Abs(fb) Then
  Swap a, b
  Swap fa, fb
End If

c = a
fc = fa
mflag = True

For it = 1 To maxit

  If fb = 0 Then
    DensityIteration_Brent = b
    Exit For
  End If
  If eps > 0 Then                'absolute limit
    If Abs(a - b) < eps Then
      DensityIteration_Brent = b
      Exit For
    End If
  Else                           'relative limit
    If Abs(a - b) < -eps * b Then
      DensityIteration_Brent = b
      Exit For
    End If
  End If

  If fa = fb Then Exit For

  If fa <> fc And fb <> fc Then
    S = a * fb * fc / ((fa - fb) * (fa - fc)) + _
        b * fa * fc / ((fb - fa) * (fb - fc)) + _
        c * fa * fb / ((fc - fa) * (fc - fb))
  Else
    S = b - (b - a) * fb / (fb - fa)
  End If

  If ((3 * a + b) / 4 - S) * (b - S) > 0 Or _
     (mflag = True And Abs(S - b) >= 0.5 * Abs(b - c)) Or _
     (mflag = False And Abs(S - b) >= 0.5 * (c - d)) Then
    S = 0.5 * (a + b)
  Else
    mflag = False
  End If

  fs = S ^ 2 * flu_f_si(0, 1, t, S) - p
  d = c
  c = b
  fc = fb

  If fa * fs < 0 Then
    b = S
    fb = fs
  Else
    a = S
    fa = fs
  End If

  If Abs(fa) < Abs(fb) Then
    Swap a, b
    Swap fa, fb
  End If

Next it

ExitFunction:
check_limits = check_limits + 1

If check_limits = 1 Then
  If d <= flu_dmin Or d > flu_dmax Then
    DensityIteration_Brent = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Function DensityIteration_Secant(ByVal t As Double, _
                                         ByVal p As Double, _
                                         ByVal d As Double, _
                                         ByVal d2 As Double, _
                                         ByVal maxit As Integer, _
                                         ByVal eps As Double) As Double

'The function returns the density as a function of temperature and pressure,
'computed from the Helmholtz function by secant iteration
'http://en.wikipedia.org/wiki/Secant_method

'output: DensityIteration_Secant: density in kg/m^3
'
'        The value ErrorReturn is returned if
'        - the maximum number of iterations is exceeded without meeting the exit criterion
'        - the function call to flu_f_si has returned an error
'        - density has taken a zero or negative value during the iteration

'input:  t: absolute temperature in K
'        p: absolute pressure in Pa
'        d: initial guess for density in kg/m^3
'       d2: counterpoint density
'    maxit: maximum number of iteration steps to be done
'      eps: required accuracy of density
'           eps > 0: absolute density uncertainty in kg/m^3
'           eps < 0: relative density uncertainty

Dim dd As Double, d1 As Double, p1 As Double, p2 As Double
Dim it As Long

DensityIteration_Secant = ErrorReturn

If p <= 0 Then Exit Function

If check_limits = 1 Then
  If t < flu_tmin Or t > flu_tmax Then Exit Function
Else
  If t <= 0 Then Exit Function
End If

check_limits = check_limits - 1

p2 = d2 ^ 2 * flu_f_si(0, 1, t, d2)
If p2 = ErrorReturn Then GoTo ExitFunction

For it = 1 To maxit
  d1 = d2
  p1 = p2
  d2 = d
  p2 = d ^ 2 * flu_f_si(0, 1, t, d)

  If p2 = ErrorReturn Then Exit For
  If p2 = p1 Then Exit For
  
  dd = -(d2 - d1) * (p2 - p) / (p2 - p1)
  d = d + dd

  If d <= 0 Then Exit For

  If eps > 0 Then                'absolute limit
    If Abs(dd) < eps Then
      DensityIteration_Secant = d
      Exit For
    End If
  Else                           'relative limit
    If Abs(dd) < -eps * d Then
      DensityIteration_Secant = d
      Exit For
    End If
  End If
  
Next it

ExitFunction:
check_limits = check_limits + 1

If check_limits = 1 Then
  If d <= flu_dmin Or d > flu_dmax Then
    DensityIteration_Secant = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Function flu_t_p_derivative_si(ByVal drv_t As Integer, _
                                       ByVal drv_p As Integer, _
                                       ByVal t_si As Double, _
                                       ByVal d_si As Double) As Double

'this function computes t-p derivatives of g from t-d derivatives of f

Dim g As Double, gt As Double, gp As Double
Dim gtt As Double, gtp As Double, gpp As Double

Dim d As Double

Dim f As Double, ft As Double, fd As Double
Dim ftt As Double, ftd As Double, fdd As Double

flu_t_p_derivative_si = ErrorReturn

If t_si <= 0 Then Exit Function
If d_si <= 0 Then Exit Function

Select Case drv_t

  Case 0:
    Select Case drv_p

      Case 0: f = flu_f_si(0, 0, t_si, d_si)              'g
              If f = ErrorReturn Then Exit Function
              fd = flu_f_si(0, 1, t_si, d_si)
              If fd = ErrorReturn Then Exit Function
              g = f + d_si * fd

      Case 1: g = 1# / d_si                               'g_p

      Case 2: fd = flu_f_si(0, 1, t_si, d_si)             'g_pp
              If fd = ErrorReturn Then Exit Function
              fdd = flu_f_si(0, 2, t_si, d_si)
              If fdd = ErrorReturn Then Exit Function
              g = d_si ^ 3 * (2# * fd + d_si * fdd)
              If g = 0 Then Exit Function
              g = -1# / g

      Case Else: Exit Function
    End Select

  Case 1:
    Select Case drv_p

      Case 0: ft = flu_f_si(1, 0, t_si, d_si)             'g_t
              If ft = ErrorReturn Then Exit Function
              g = ft

      Case 1: fd = flu_f_si(0, 1, t_si, d_si)             'g_tp
              If fd = ErrorReturn Then Exit Function
              ftd = flu_f_si(1, 1, t_si, d_si)
              If ftd = ErrorReturn Then Exit Function
              fdd = flu_f_si(0, 2, t_si, d_si)
              If fdd = ErrorReturn Then Exit Function
              g = d_si * (2# * fd + d_si * fdd)
              If g = 0 Then Exit Function
              g = ftd / g

      Case Else: Exit Function
    End Select

  Case 2:
    Select Case drv_p

      Case 0: fd = flu_f_si(0, 1, t_si, d_si)             'g_tt
              If fd = ErrorReturn Then Exit Function
              ftt = flu_f_si(2, 0, t_si, d_si)
              If ftt = ErrorReturn Then Exit Function
              ftd = flu_f_si(1, 1, t_si, d_si)
              If ftd = ErrorReturn Then Exit Function
              fdd = flu_f_si(0, 2, t_si, d_si)
              If fdd = ErrorReturn Then Exit Function
              g = 2# * fd + d_si * fdd
              If g = 0 Then Exit Function
              g = ftt - d_si * ftd ^ 2 / g

      Case Else: Exit Function
    End Select

  Case Else: Exit Function
End Select

flu_t_p_derivative_si = g

End Function

'==========================================================================
Public Sub set_it_ctrl_density(ByVal key As String, ByVal value As Double)

'this sub sets control parameters for the iteration used to compute
'IAPWS-95 density from pressure

'key              value
'it_steps         0           set iteration number to default (100)
'it_steps         n > 0       set iteration number to n
'it_steps        -1           do not iterate, use initial value

'init_liq_dens    0           use default liquid density (IF97) to start in the subcritical region (T <= 623.16 K or P <=16.529 MPa)
'                             use default cubic EOS to start in the critical region (623.16 K < T < 650 K and 16.529 MPa < P < 35 MPa)
'                             use default 1000 kg/m to start in the remaining supercritical region
'init_liq_dens   -1           use IF97 liquid density to start
'init_liq_dens   -2           use EOS80 liquid density to start
'init_liq_dens   -3           use F03 liquid density to start
'init_liq_dens    d > 0       use value d as liquid density to start

'init_vap_dens    0           use default vapour density (IF97) to start in the subcritical region (T <= 623.16 K or P <=16.529 MPa)
'                             use default cubic EOS to start in the critical region (623.16 K < T < 650 K and 16.529 MPa < P < 35 MPa)
'                             use default 1000 kg/m to start in the remaining supercritical region
'init_vap_dens   -1           use IF97 vapour density to start
'init_vap_dens   -2           use ideal-gas vapour density to start
'init_vap_dens    d > 0       use value d as vapour density to start

'tol_liq_dens     0           use default exit accuracy for liquid density (0.1 ppm)
'tol_liq_dens     eps         use eps as exit accuracy for liquid density (eps < 0 means relative error)

'tol_vap_dens     0           use default exit accuracy for vapour density (0.1 ppm)
'tol_vap_dens     eps         use eps as exit accuracy for vapour density (eps < 0 means relative error)

'method_liq       0           use default iteration method (now: Newton method) for liquid
'method_liq       1           use Newton method for liquid
'method_liq       2           use Brent method for liquid
'method_liq       3           use secant method for liquid

'dens2_liq        0           use default counterpoint for Brent/Secant method for liquid
'                         Brent: 2 * Newton step, Secant: 1 * Newton step
'dens2_liq       -1           use Newton step as the first counterpoint for liquid
'dens2_liq       -2           use 0.5 * Newton step as the first counterpoint for liquid
'dens2_liq       -3           use 2 * Newton step as the first counterpoint for liquid
'dens2_liq        d > 0       use d as the first counterpoint density for liquid

'method_vap       0           use default iteration method (now: Newton method) for vapour
'method_vap       1           use Newton method for vapour
'method_vap       2           use Brent method for vapour
'method_vap       3           use secant method for vapour

'dens2_vap        0           use default counterpoint for Brent/Secant method for vapour
'                         Brent: 2 * Newton step, Secant: 1 * Newton step
'dens2_vap       -1           use Newton step as the first counterpoint for vapour
'dens2_vap       -2           use 0.5 * Newton step as the first counterpoint for vapour
'dens2_vap       -3           use 2 * Newton step as the first counterpoint for vapour
'dens2_vap        d > 0       use d as the first counterpoint density for vapour

init_it_ctrl_density

Select Case LCase(Trim(key))

  Case "it_steps":   'iteration steps
    Select Case value
      Case 0:      ctrl_loop_maximum = 100  'default = 100
      Case Is < 0: ctrl_loop_maximum = -1
      Case Else:   ctrl_loop_maximum = value
    End Select

  Case "init_liq_dens":   'start liquid density
    Select Case CLng(value)
      Case 0:       ctrl_mode_liquid = 0    'default = IF97
      Case Is < -3: 'Exit Sub
      Case Is < 0:  ctrl_mode_liquid = value
      Case Else:    ctrl_mode_liquid = 1
                    ctrl_density_liquid = value
    End Select

  Case "init_vap_dens":   'start vapour density
    Select Case CLng(value)
      Case 0:       ctrl_mode_vapour = 0    'default = IF97
      Case Is < -2: 'Exit Sub
      Case Is < 0:  ctrl_mode_vapour = value
      Case Else:    ctrl_mode_vapour = 1
                    ctrl_density_vapour = value
    End Select

  Case "tol_liq_dens":      'required liquid density tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_liquid = -0.0000001   'default = 0.1 ppm relative
      Case Else:   ctrl_eps_exit_liquid = value
    End Select

  Case "tol_vap_dens":      'required vapour density tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_vapour = -0.0000001   'default = 0.1 ppm relative
      Case Else:   ctrl_eps_exit_vapour = value
    End Select

  Case "method_liq":
    Select Case value
      Case 0 To 3:   ctrl_method_liquid = value
    End Select

  Case "dens2_liq":
    Select Case value
      Case Is >= -3: ctrl_density2_liquid = value
    End Select

  Case "method_vap":
    Select Case value
      Case 0 To 3:   ctrl_method_vapour = value
    End Select

  Case "dens2_vap":
    Select Case value
      Case Is >= -3: ctrl_density2_vapour = value
    End Select
  
End Select

End Sub

'==========================================================================
Public Function get_it_ctrl_density(ByVal key As String) As Double

'this function returns control parameters as set for the Newton iteration
'used to compute IAPWS-95 density from pressure

init_it_ctrl_density

Select Case LCase(Trim(key))

  Case "it_steps":   'iteration steps
    get_it_ctrl_density = ctrl_loop_maximum

  Case "init_liq_dens":   'start liquid density
    If ctrl_mode_liquid = 1 Then
      get_it_ctrl_density = ctrl_density_liquid
    Else
      get_it_ctrl_density = ctrl_mode_liquid
    End If

  Case "init_vap_dens":   'start vapour density
    If ctrl_mode_vapour = 1 Then
      get_it_ctrl_density = ctrl_density_vapour
    Else
      get_it_ctrl_density = ctrl_mode_vapour
    End If

  Case "tol_liq_dens":      'required liquid density tolerance
    get_it_ctrl_density = ctrl_eps_exit_liquid

  Case "tol_vap_dens":      'required vapour density tolerance
    get_it_ctrl_density = ctrl_eps_exit_vapour

  Case "method_liq":      'selected iteration method for liquid
    get_it_ctrl_density = ctrl_method_liquid

  Case "dens2_liq":      'counterpoint value for liquid (irrelevant for Newton)
    get_it_ctrl_density = ctrl_density2_liquid

  Case "method_vap":      'selected iteration method for vapour
    get_it_ctrl_density = ctrl_method_vapour

  Case "dens2_vap":      'counterpoint value for vapour (irrelevant for Newton)
    get_it_ctrl_density = ctrl_density2_vapour

  Case Else:
    get_it_ctrl_density = ErrorReturn
    
End Select

End Function

'==========================================================================
Private Function aux_density_EOS80_si(ByVal t_si As Double, _
                                      ByVal p_si As Double) As Double

'This function returns the density of liquid water computed from the
'International Equation of State of Seawater 1980, EOS-80, as a function of temperature
'and pressure

'output:  aux_density_EOS80_si: density in kg/m^3

'input:   t_si: absolute temperature in K
'         p_si: absolute pressure in Pa

Dim AW As Double, BW As Double, KW As Double, RW As Double
Dim t As Double, p As Double

aux_density_EOS80_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

'R.C.Millard Jr.: International Oceanographic Tables Vol.4
'UNESCO technical Papers in Marine Science 40
'UNESCO 1987

t = cnv_temperature("DEGC(T68)", t_si, "K(T90)")  't68 in C
p = 0.1 * cnv_pressure("DBAR", p_si, "PA")        'sea pressure in bar

BW = 0.0000850935 + t * (-0.00000612293 + t * 0.000000052787)
AW = 3.239908 + t * (0.00143713 + t * (0.000116092 - t * 0.000000577905))
KW = 0.01360477 - t * 0.00005155288
KW = 19652.21 + t * (148.4206 + t * (-2.327105 + t * KW))
KW = KW + p * (AW + p * BW)
If KW = 0 Then Exit Function

KW = 1# - p / KW
If KW = 0 Then Exit Function

RW = 0.0001001685 + t * (-0.000001120083 + t * 0.000000006536332)
RW = 999.842594 + t * (0.06793952 + t * (-0.00909529 + t * RW))

aux_density_EOS80_si = RW / KW

End Function

'==========================================================================
Private Function aux_density_ideal_si(ByVal t_si As Double, _
                                      ByVal p_si As Double) As Double

'This function returns the density of ideal-gas vapour as a function of temperature
'and pressure

'output:  aux_density_ideal_si: density in kg/m^3

'input:   t_si: absolute temperature in K
'         p_si: absolute pressure in Pa


Const R = Gas_constant_H2O_si         'specific gas constant of water in J/(kg K)

aux_density_ideal_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

aux_density_ideal_si = p_si / (R * t_si)

End Function

'==========================================================================
Private Sub init_it_ctrl_density()

If ctrl_initialized = -1 Then Exit Sub

ctrl_initialized = -1

'Set default values and modes for density iteration
ctrl_mode_liquid = 0
ctrl_mode_vapour = 0
ctrl_loop_maximum = 100
ctrl_density_liquid = 1000
ctrl_density_vapour = 1
ctrl_eps_exit_liquid = -0.0000001 'relative, 0.1 ppm
ctrl_eps_exit_vapour = -0.0000001 'relative, 0.1 ppm

'Set default values for alternative iteration methods
ctrl_method_liquid = 0 'default = Newton
ctrl_method_vapour = 0 'default = Newton

'Set default counterpoint values for alternative iteration methods
ctrl_density2_liquid = 0  'default = .5 * Newton step
ctrl_density2_vapour = 0  'default = .5 * Newton step

End Sub

'==========================================================================
Private Sub aux_density_critical_si(ByVal t_si As Double, _
                                    ByVal p_si As Double, _
                                    ByRef d_liq As Double, _
                                    ByRef d_vap As Double)
                                      
'returns approximate liquid and/or vapour density from a cubic equation of state at given T, P
'usable in the range 620 K < T < 650 K, 10 MPa < P < 35 MPa
'supercritical range:  return valid d_liq = d_vap
'2-phase region:       return valid d_liq <> d_vap          (including the metastable region)
'liquid region:        return valid d_liq,  invalid d_vap
'vapour region:        return valid d_vap,  invalid d_liq

'the critical properties of IAPWS-95
Const dc = CP_density_si
Const pc = CP_pressure_si
Const Tc = CP_temperature_si

Const a10 = -7.60041479494879
Const a20 = 118.661872386874
Const a11 = -17.463827264079
Const a21 = 186.040087842884
Const a12 = 0.69701967809328
Const a22 = 25.5059905941023
Const a03 = -0.602044738250314
Const a13 = 30.8633119943879
Const a23 = 14.4873846518829

Dim pr As Double, tr As Double
Dim R As Double, S As Double, t As Double
Dim d1 As Double, d2 As Double, d3 As Double
Dim p1 As Double, p2 As Double
Dim a0 As Double, a1 As Double, a2 As Double, a3 As Double

d_liq = ErrorReturn
d_vap = ErrorReturn
If p_si <= 0 Then Exit Sub
If t_si <= 0 Then Exit Sub

'reduced P and T
pr = pc / p_si - 1#
tr = t_si / Tc - 1#

'cubic eq. pr = a0 + a1*dr + a2*dr^2 + a3*dr^3
'with reduced density dr = d/dc - 1
a0 = (a10 + a20 * tr) * tr
a1 = (a11 + a21 * tr) * tr
a2 = (a12 + a22 * tr) * tr
a3 = a03 + (a13 + a23 * tr) * tr  'a3 < 0 holds for T < 659.6 K

If a3 >= 0 Then Exit Sub

If tr < 0 Then

  'get the pressure range of 2-phase solutions at the given temperature.
  'solutions of dP/dV = 0, i.e. the spinodal curve:
  R = a2 * a2 - 3# * a1 * a3
  If R < 0 Then Exit Sub
  R = Sqr(R)
  d1 = -(a2 + R) / (3# * a3)
  d2 = -(a2 - R) / (3# * a3)
  'the pressure range is
  p1 = a0 + d1 * (a1 + d1 * (a2 + d1 * a3)) 'highest pressure for subcooled vapour (lowest reduced pressure)
  p2 = a0 + d2 * (a1 + d2 * (a2 + d2 * a3)) 'lowest pressure for superheated liquid (highest reduced pressure)

Else

  'one fluid state
  p1 = pr
  p2 = pr
  
End If

'Coefficients of the cubic eq.   dr^3 + r * dr^2 + s * dr + t = 0
R = a2 / a3
S = a1 / a3
t = (a0 - pr) / a3

Select Case get_CubicRoots(R, S, t, d1, d2, d3)

  Case 1:
          If d1 <= -1 Then Exit Sub
          If pr >= p1 Then
            'get the vapour density
            d_vap = (d1 + 1) * dc
          End If
          If pr <= p2 Then
            'get the liquid density
            d_liq = (d1 + 1) * dc
          End If
  
  Case 3: Sort3up d1, d2, d3           '3 solutions, min is vapour, max is liquid
          If d1 <= -1 Then Exit Sub
          If d3 <= -1 Then Exit Sub
          d_vap = (d1 + 1) * dc
          d_liq = (d3 + 1) * dc
          
End Select

End Sub

'==========================================================================
Private Function aux_liq_density_critical_si(ByVal t_si As Double, _
                                             ByVal p_si As Double) As Double

'returns the approximate liquid density from the cubic equation

Dim d_liq As Double, d_vap As Double

aux_density_critical_si t_si, p_si, d_liq, d_vap

aux_liq_density_critical_si = d_liq

End Function

'==========================================================================
Private Function aux_vap_density_critical_si(ByVal t_si As Double, _
                                             ByVal p_si As Double) As Double

'returns the approximate vapour density from the cubic equation

Dim d_liq As Double, d_vap As Double

aux_density_critical_si t_si, p_si, d_liq, d_vap

aux_vap_density_critical_si = d_vap

End Function

'==========================================================================
Private Sub Swap(ByRef a As Double, ByRef b As Double)
Dim c As Double
c = a
a = b
b = c
End Sub

'==========================================================================
Private Sub Sort3up(ByRef d1 As Double, _
                    ByRef d2 As Double, _
                    ByRef d3 As Double)

'sort d1, d2, d3 increasing

If d2 > d3 Then Swap d2, d3
If d1 > d2 Then Swap d1, d2
If d2 > d3 Then Swap d2, d3

End Sub

'==========================================================================
Private Function aux_liq_density_f03_si(ByVal t_si As Double, _
                                        ByVal p_si As Double) As Double

'This function returns the density of liquid water computed from the
'Gibbs function 2003 of Seawater as a function of temperature
'and pressure, published in

'R. Feistel:
'A new extended Gibbs thermodynamic potential of seawater.
'Progress in Oceanography, 58/1 (2003) 43-115

'output:  aux_liq_density_f03_si: density in kg/m^3

'input:   t_si: absolute temperature in K
'         p_si: absolute pressure in Pa

Dim d As Double

aux_liq_density_f03_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = aux_liq_g_f03_si(0, 1, t_si, p_si)

If d = ErrorReturn Then Exit Function
If d <= 0 Then Exit Function

aux_liq_density_f03_si = 1# / d

End Function

'==========================================================================
Private Function aux_liq_g_f03_si(ByVal drv_t As Integer, _
                                  ByVal drv_p As Integer, _
                                  ByVal t_si As Double, _
                                  ByVal p_si As Double) As Double

'this function implements the Gibbs function of pure water as defined in
'R. Feistel:
'A new extended Gibbs thermodynamic potential of seawater.
'Progress in Oceanography, 58 (2003) 43-115

Const T0 = Celsius_temperature_si     'in K
Const tu = 40#                        'in K
Const P0 = Sealevel_pressure_si       'in Pa
Const pu = 100000000# 'in Pa

Dim y As Double, z As Double, g As Double

aux_liq_g_f03_si = ErrorReturn

If drv_t < 0 Then Exit Function
If drv_p < 0 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

InitF03

y = (t_si - T0) / tu
z = (p_si - P0) / pu

g = polyf03_gyz(drv_t, drv_p, y, z)
If g = ErrorReturn Then Exit Function

aux_liq_g_f03_si = g / (tu ^ drv_t * pu ^ drv_p)

End Function

'==========================================================================
Private Sub InitF03()
' g(T,P) = Sum gc(j,k) * (t/40C)^j * (p/100 MPa)^k
'          j,k

Const Tt = TP_temperature_si
Const T0 = Celsius_temperature_si
Const tu = 40#

Const Pt = TP_pressure_IAPWS95_si 'IAPWS-95 triple point pressure
Const P0 = Sealevel_pressure_si
Const pu = 100000000#

Const yt = (Tt - T0) / tu
Const zt = (Pt - P0) / pu

Dim g As Double, gt As Double, gp As Double

If gc(2, 0) = -12357.785933039 Then Exit Sub

gc(2, 0) = -12357.785933039
gc(3, 0) = 736.741204151612
gc(4, 0) = -148.185936433658
gc(5, 0) = 58.0259125842571
gc(6, 0) = -18.9843846514172
gc(7, 0) = 3.05081646487967
gc(0, 1) = 100015.695367145
gc(1, 1) = -270.983805184062
gc(2, 1) = 1455.0364540468
gc(3, 1) = -672.50778314507
gc(4, 1) = 397.968445406972
gc(5, 1) = -194.618310617595
gc(6, 1) = 63.5113936641785
gc(7, 1) = -9.63108119393062
gc(0, 2) = -2544.5765420363
gc(1, 2) = 776.153611613101
gc(2, 2) = -756.558385769359
gc(3, 2) = 499.360390819152
gc(4, 2) = -301.815380621876
gc(5, 2) = 120.520654902025
gc(6, 2) = -22.2897317140459
gc(0, 3) = 284.517778446287
gc(1, 3) = -196.51255088122
gc(2, 3) = 273.479662323528
gc(3, 3) = -239.545330654412
gc(4, 3) = 152.196371733841
gc(5, 3) = -55.2723052340152
gc(6, 3) = 8.17060541818112
gc(0, 4) = -33.3146754253611
gc(1, 4) = 28.9796526294175
gc(2, 4) = -55.5604063817218
gc(3, 4) = 48.8012518593872
gc(4, 4) = -26.3748377232802
gc(5, 4) = 6.48190668077221
gc(0, 5) = 4.20263108803084
gc(1, 5) = -2.13290083518327
gc(2, 5) = 4.34420671917197
gc(3, 5) = -1.66307106208905
gc(0, 6) = -0.546428511471039

'IAPWS-95 reference state condition
'energy = 0 and entropy = 0 at the triple point:
'gc(0, 0) = 101.342743139672
'gc(1, 0) = 5.90578348518236

'quadruple precision values (D.G.Wright 21 July 2008)
'gc(0, 0) = 1.013427431396741480431228220832E2
'gc(1, 0) = 5.905783479094018366702121889468E0
gc(0, 0) = 101.342743139674
gc(1, 0) = 5.90578347909402

'RF: dynamical adjustment:
'
'g = polyf03_gyz(0, 0, yt, zt)
'gt = polyf03_gyz(1, 0, yt, zt) / tu
'gp = polyf03_gyz(0, 1, yt, zt) / pu
'
'gc(0, 0) = gc(0, 0) - g + pt * gp
'gc(1, 0) = gc(1, 0) - gt * tu

End Sub

'==========================================================================
Private Function polyf03_gyz(ByVal drv_y As Integer, _
                             ByVal drv_z As Integer, _
                             ByVal y As Double, _
                             ByVal z As Double) As Double

'returns the value of the polynomial derivative
'(d/dy)^drv_y (d/dz)^drv_z sum(j,k) gc(j,k)*y^j*z^k

Dim g As Double
Dim yj As Double, zk As Double

Dim j As Integer, jmax As Integer
Dim k As Integer, kmax As Integer

Dim c As Double, L As Integer

g = 0
If y = 0 Then jmax = drv_y Else jmax = maxt
If z = 0 Then kmax = drv_z Else kmax = maxp

yj = 1#
For j = drv_y To jmax   'loop over powers of y

  zk = 1#
  For k = drv_z To kmax    'loop over powers of z

    If gc(j, k) <> 0 Then
      c = gc(j, k) * yj * zk

      For L = 1 To drv_y            'factors from y-derivatives
        c = c * CDbl(j - L + 1)
      Next L

      For L = 1 To drv_z            'factors from z-derivatives
        c = c * CDbl(k - L + 1)
      Next L

      g = g + c
    End If

    If k < kmax Then zk = zk * z
  Next k

  If j < jmax Then yj = yj * y
Next j

polyf03_gyz = g

End Function

'=========================================================================
Private Function aux_liq_g_if97_si(ByVal drv_t As Integer, _
                                  ByVal drv_p As Integer, _
                                  ByVal t_si As Double, _
                                  ByVal p_si As Double) As Double

'This function returns the Gibbs function g(t,p) and its 1st and 2nd derivatives
'with respect to temperature and pressure, as defined for the region 1 (liquid) in IAPWS-IF97
'
'output: aux_liq_g_if97_si: specific Gibbs energy in J/kg or its t-p derivative
'
'input:  drv_t: order of the temperature derivative (0-2)
'        drv_p: order of the pressure derivative (0-2)
'        t_si:  absolute temperature in K
'        p_si:  absolute pressure in Pa

Const tu = 1386#
Const pu# = 16530000#
Const R = 461.526         'J kg-1 K-1  specific gas constant
                          'note this deviates from Gas_Constant_H2O_si = 461.51805
                          
Dim g As Double, gt As Double, gtt As Double
Dim gp As Double, gtp As Double, gpp As Double
Dim PI As Double, tau As Double, RT As Double

aux_liq_g_if97_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

InitIF97_1

PI = p_si / pu
tau = tu / t_si
RT = R * t_si

Select Case drv_t

  Case 0:
    Select Case drv_p

      Case 0: g = gamma_1(0, 0, tau, PI)
              If g = ErrorReturn Then Exit Function
              aux_liq_g_if97_si = RT * g

      Case 1: gp = gamma_1(0, 1, tau, PI)
              If gp = ErrorReturn Then Exit Function
              aux_liq_g_if97_si = RT * gp / pu

      Case 2: gpp = gamma_1(0, 2, tau, PI)
              If gpp = ErrorReturn Then Exit Function
              aux_liq_g_if97_si = RT * gpp / pu ^ 2

      Case Else: Exit Function
    End Select

  Case 1:
    Select Case drv_p

      Case 0: g = gamma_1(0, 0, tau, PI)
              If g = ErrorReturn Then Exit Function
              gt = gamma_1(1, 0, tau, PI)
              If gt = ErrorReturn Then Exit Function
              aux_liq_g_if97_si = R * (g - tau * gt)

      Case 1: gp = gamma_1(0, 1, tau, PI)
              If gp = ErrorReturn Then Exit Function
              gtp = gamma_1(1, 1, tau, PI)
              If gtp = ErrorReturn Then Exit Function
              aux_liq_g_if97_si = R * (gp - tau * gtp) / pu

      Case Else: Exit Function
    End Select

  Case 2:
    Select Case drv_p

      Case 0: gtt = gamma_1(2, 0, tau, PI)
              If gtt = ErrorReturn Then Exit Function
              aux_liq_g_if97_si = R * tau ^ 2 * gtt / t_si

      Case Else: Exit Function
    End Select

  Case Else: Exit Function
End Select

End Function

'=========================================================================
Private Function aux_vap_g_if97_si(ByVal drv_t As Integer, _
                                  ByVal drv_p As Integer, _
                                  ByVal t_si As Double, _
                                  ByVal p_si As Double) As Double

'This function returns the Gibbs function g(t,p) and its 1st and 2nd derivatives
'with respect to temperature and pressure, as defined for the region 2 (vapour) in IAPWS-IF97
'
'output: aux_vap_g_if97_si: specific Gibbs energy in J/kg or its t-p derivative
'
'input:  drv_t: order of the temperature derivative (0-2)
'        drv_p: order of the pressure derivative (0-2)
'        t_si:  absolute temperature in K
'        p_si:  absolute pressure in Pa

Const tu = 540#
Const pu = 1000000#
Const R = 461.526         'J kg-1 K-1  specific gas constant
                          'note this deviates from Gas_Constant_H2O_si = 461.51805
                          
Dim g As Double, gt As Double, gtt As Double
Dim gp As Double, gtp As Double, gpp As Double
Dim PI As Double, tau As Double, RT As Double

aux_vap_g_if97_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

InitIF97_2

PI = p_si / pu
tau = tu / t_si
RT = R * t_si

Select Case drv_t

  Case 0:
    Select Case drv_p

      Case 0: g = gamma_2(0, 0, tau, PI)
              If g = ErrorReturn Then Exit Function
              aux_vap_g_if97_si = RT * g

      Case 1: gp = gamma_2(0, 1, tau, PI)
              If gp = ErrorReturn Then Exit Function
              aux_vap_g_if97_si = RT * gp / pu

      Case 2: gpp = gamma_2(0, 2, tau, PI)
              If gpp = ErrorReturn Then Exit Function
              aux_vap_g_if97_si = RT * gpp / pu ^ 2

      Case Else: Exit Function
    End Select

  Case 1:
    Select Case drv_p

      Case 0: g = gamma_2(0, 0, tau, PI)
              If g = ErrorReturn Then Exit Function
              gt = gamma_2(1, 0, tau, PI)
              If gt = ErrorReturn Then Exit Function
              aux_vap_g_if97_si = R * (g - tau * gt)

      Case 1: gp = gamma_2(0, 1, tau, PI)
              If gp = ErrorReturn Then Exit Function
              gtp = gamma_2(1, 1, tau, PI)
              If gtp = ErrorReturn Then Exit Function
              aux_vap_g_if97_si = R * (gp - tau * gtp) / pu

      Case Else: Exit Function
    End Select

  Case 2:
    Select Case drv_p

      Case 0: gtt = gamma_2(2, 0, tau, PI)
              If gtt = ErrorReturn Then Exit Function
              aux_vap_g_if97_si = R * tau ^ 2 * gtt / t_si

      Case Else: Exit Function
    End Select

  Case Else: Exit Function
End Select

End Function

'=========================================================================
Private Function aux_liq_density_if97_si(ByVal t_si As Double, _
                                         ByVal p_si As Double) As Double

'This function returns the density of liquid water as a function of temperature
'and pressure, in the region 1 (liquid) in IAPWS-IF97

'output:  aux_liq_density_if97_si: density in kg/m^3

'input:   t_si: absolute temperature in K
'         p_si: absolute pressure in Pa

Dim d As Double

aux_liq_density_if97_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = aux_liq_g_if97_si(0, 1, t_si, p_si)

If d = ErrorReturn Then Exit Function
If d <= 0 Then Exit Function

aux_liq_density_if97_si = 1# / d

End Function

'=========================================================================
Private Function aux_vap_density_if97_si(ByVal t_si As Double, _
                                         ByVal p_si As Double) As Double

'This function returns the density of water vapour as a function of temperature
'and pressure, in the region 2 (vapour) in IAPWS-IF97

'output:  aux_vap_density_if97_si: density in kg/m^3

'input:   t_si: absolute temperature in K
'         p_si: absolute pressure in Pa

Dim d As Double

aux_vap_density_if97_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = aux_vap_g_if97_si(0, 1, t_si, p_si)

If d = ErrorReturn Then Exit Function
If d <= 0 Then Exit Function

aux_vap_density_if97_si = 1# / d

End Function

'=========================================================================
Private Sub InitIF97_1()

Dim i As Integer

If n1i(1) = 0.14632971213167 Then Exit Sub

'Table 2. Numerical values of the coefficients and exponents of the dimensionless Gibbs free energy
'for region 1, Eq. (7)
'i      Ii          Ji           ni                             i       Ii           Ji            ni
i = 1: i1i(i) = 0:  j1i(i) = -2: n1i(i) = 0.14632971213167:     i = 18: i1i(i) = 2:  j1i(i) = 3:   n1i(i) = -4.4141845330846E-06
i = 2: i1i(i) = 0:  j1i(i) = -1: n1i(i) = -0.84548187169114:    i = 19: i1i(i) = 2:  j1i(i) = 17:  n1i(i) = -7.2694996297594E-16
i = 3: i1i(i) = 0:  j1i(i) = 0:  n1i(i) = -3.756360367204:      i = 20: i1i(i) = 3:  j1i(i) = -4:  n1i(i) = -3.1679644845054E-05
i = 4: i1i(i) = 0:  j1i(i) = 1:  n1i(i) = 3.3855169168385:      i = 21: i1i(i) = 3:  j1i(i) = 0:   n1i(i) = -2.8270797985312E-06
i = 5: i1i(i) = 0:  j1i(i) = 2:  n1i(i) = -0.95791963387872:    i = 22: i1i(i) = 3:  j1i(i) = 6:   n1i(i) = -8.5205128120103E-10
i = 6: i1i(i) = 0:  j1i(i) = 3:  n1i(i) = 0.15772038513228:     i = 23: i1i(i) = 4:  j1i(i) = -5:  n1i(i) = -2.2425281908E-06
i = 7: i1i(i) = 0:  j1i(i) = 4:  n1i(i) = -0.016616417199501:   i = 24: i1i(i) = 4:  j1i(i) = -2:  n1i(i) = -6.5171222895601E-07
i = 8: i1i(i) = 0:  j1i(i) = 5:  n1i(i) = 8.1214629983568E-04:  i = 25: i1i(i) = 4:  j1i(i) = 10:  n1i(i) = -1.4341729937924E-13
i = 9: i1i(i) = 1:  j1i(i) = -9: n1i(i) = 2.8319080123804E-04:  i = 26: i1i(i) = 5:  j1i(i) = -8:  n1i(i) = -4.0516996860117E-07
i = 10: i1i(i) = 1: j1i(i) = -7: n1i(i) = -6.0706301565874E-04: i = 27: i1i(i) = 8:  j1i(i) = -11: n1i(i) = -1.2734301741641E-09
i = 11: i1i(i) = 1: j1i(i) = -1: n1i(i) = -0.018990068218419:   i = 28: i1i(i) = 8:  j1i(i) = -6:  n1i(i) = -1.7424871230634E-10
i = 12: i1i(i) = 1: j1i(i) = 0:  n1i(i) = -0.032529748770505:   i = 29: i1i(i) = 21: j1i(i) = -29: n1i(i) = -6.8762131295531E-19
i = 13: i1i(i) = 1: j1i(i) = 1:  n1i(i) = -0.021841717175414:   i = 30: i1i(i) = 23: j1i(i) = -31: n1i(i) = 1.4478307828521E-20
i = 14: i1i(i) = 1: j1i(i) = 3:  n1i(i) = -5.283835796993E-05:  i = 31: i1i(i) = 29: j1i(i) = -38: n1i(i) = 2.6335781662795E-23
i = 15: i1i(i) = 2: j1i(i) = -3: n1i(i) = -4.7184321073267E-04: i = 32: i1i(i) = 30: j1i(i) = -39: n1i(i) = -1.1947622640071E-23
i = 16: i1i(i) = 2: j1i(i) = 0:  n1i(i) = -3.0001780793026E-04: i = 33: i1i(i) = 31: j1i(i) = -40: n1i(i) = 1.8228094581404E-24
i = 17: i1i(i) = 2: j1i(i) = 1:  n1i(i) = 4.7661393906987E-05:  i = 34: i1i(i) = 32: j1i(i) = -41: n1i(i) = -9.3537087292458E-26

End Sub

'=========================================================================
Private Sub InitIF97_2()

Dim i As Integer

If n0i(i) = -9.6927686500217 Then Exit Sub

'Table 10. Numerical values of the coefficients and exponents of the ideal-gas part gamma_0 of the
'dimensionless Gibbs free energy for region 2, Eq. (16)
'i     Ji0          ni0
i = 1: j0i(i) = 0:  n0i(i) = -9.6927686500217
i = 2: j0i(i) = 1:  n0i(i) = 10.086655968018
i = 3: j0i(i) = -5: n0i(i) = -0.005608791128302
i = 4: j0i(i) = -4: n0i(i) = 0.071452738081455
i = 5: j0i(i) = -3: n0i(i) = -0.40710498223928
i = 6: j0i(i) = -2: n0i(i) = 1.4240819171444
i = 7: j0i(i) = -1: n0i(i) = -4.383951131945
i = 8: j0i(i) = 2:  n0i(i) = -0.28408632460772
i = 9: j0i(i) = 3:  n0i(i) = 0.021268463753307
'If Eq. (16) is incorporated into Eq. (18), instead of the-values for n0i(1) and n0i(1)
'given above, the following values
'n0i(1) = -0.96937268393049E1
'n0i(2) = 0.10087275970006E2
'should be used


'Table 11. Numerical values of the coefficients and exponents of the
'residual part gamma_r of the dimensionless Gibbs free energy for
'region 2, Eq. (17)
'i      Ii           Ji           ni
i = 1:  iri(i) = 1:  jri(i) = 0:  nri(i) = -1.7731742473213E-03
i = 2:  iri(i) = 1:  jri(i) = 1:  nri(i) = -0.017834862292358
i = 3:  iri(i) = 1:  jri(i) = 2:  nri(i) = -0.045996013696365
i = 4:  iri(i) = 1:  jri(i) = 3:  nri(i) = -0.057581259083432
i = 5:  iri(i) = 1:  jri(i) = 6:  nri(i) = -0.05032527872793
i = 6:  iri(i) = 2:  jri(i) = 1:  nri(i) = -3.3032641670203E-05
i = 7:  iri(i) = 2:  jri(i) = 2:  nri(i) = -1.8948987516315E-04
i = 8:  iri(i) = 2:  jri(i) = 4:  nri(i) = -3.9392777243355E-03
i = 9:  iri(i) = 2:  jri(i) = 7:  nri(i) = -0.043797295650573
i = 10: iri(i) = 2:  jri(i) = 36: nri(i) = -2.6674547914087E-05
i = 11: iri(i) = 3:  jri(i) = 0:  nri(i) = 2.0481737692309E-08
i = 12: iri(i) = 3:  jri(i) = 1:  nri(i) = 4.3870667284435E-07
i = 13: iri(i) = 3:  jri(i) = 3:  nri(i) = -3.227767723857E-05
i = 14: iri(i) = 3:  jri(i) = 6:  nri(i) = -1.5033924542148E-03
i = 15: iri(i) = 3:  jri(i) = 35: nri(i) = -0.040668253562649
i = 16: iri(i) = 4:  jri(i) = 1:  nri(i) = -7.8847309559367E-10
i = 17: iri(i) = 4:  jri(i) = 2:  nri(i) = 1.2790717852285E-08
i = 18: iri(i) = 4:  jri(i) = 3:  nri(i) = 4.8225372718507E-07
i = 19: iri(i) = 5:  jri(i) = 7:  nri(i) = 2.2922076337661E-06
i = 20: iri(i) = 6:  jri(i) = 3:  nri(i) = -1.6714766451061E-11
i = 21: iri(i) = 6:  jri(i) = 16: nri(i) = -2.1171472321355E-03
i = 22: iri(i) = 6:  jri(i) = 35: nri(i) = -23.895741934104
i = 23: iri(i) = 7:  jri(i) = 0:  nri(i) = -5.905956432427E-18
i = 24: iri(i) = 7:  jri(i) = 11: nri(i) = -1.2621808899101E-06
i = 25: iri(i) = 7:  jri(i) = 25: nri(i) = -0.038946842435739
i = 26: iri(i) = 8:  jri(i) = 8:  nri(i) = 1.1256211360459E-11
i = 27: iri(i) = 8:  jri(i) = 36: nri(i) = -8.2311340897998
i = 28: iri(i) = 9:  jri(i) = 13: nri(i) = 1.9809712802088E-08
i = 29: iri(i) = 10: jri(i) = 4:  nri(i) = 1.0406965210174E-19
i = 30: iri(i) = 10: jri(i) = 10: nri(i) = -1.0234747095929E-13
i = 31: iri(i) = 10: jri(i) = 14: nri(i) = -1.0018179379511E-09
i = 32: iri(i) = 16: jri(i) = 29: nri(i) = -8.0882908646985E-11
i = 33: iri(i) = 16: jri(i) = 50: nri(i) = 0.10693031879409
i = 34: iri(i) = 18: jri(i) = 57: nri(i) = -0.33662250574171
i = 35: iri(i) = 20: jri(i) = 20: nri(i) = 8.9185845355421E-25
i = 36: iri(i) = 20: jri(i) = 35: nri(i) = 3.0629316876232E-13
i = 37: iri(i) = 20: jri(i) = 48: nri(i) = -4.2002467698208E-06
i = 38: iri(i) = 21: jri(i) = 21: nri(i) = -5.9056029685639E-26
i = 39: iri(i) = 22: jri(i) = 53: nri(i) = 3.7826947613457E-06
i = 40: iri(i) = 23: jri(i) = 39: nri(i) = -1.2768608934681E-15
i = 41: iri(i) = 24: jri(i) = 26: nri(i) = 7.3087610595061E-29
i = 42: iri(i) = 24: jri(i) = 40: nri(i) = 5.5414715350778E-17
i = 43: iri(i) = 24: jri(i) = 58: nri(i) = -9.436970724121E-07

End Sub

'=========================================================================
Private Function gamma_0(ByVal drv_t As Integer, _
                         ByVal drv_p As Integer, _
                         ByVal tau As Double, _
                         ByVal PI As Double) As Double

'this function implements the derivatives of gamma_0 as given in Table 13 of IF-97

Dim g As Double, pwrt As Double
Dim i As Integer, k As Integer

gamma_0 = ErrorReturn

If PI <= 0 Then Exit Function
If tau <= 0 Then Exit Function
If drv_t < 0 Then Exit Function
If drv_p < 0 Then Exit Function

g = 0
If drv_t = 0 Then
  If drv_p = 0 Then
    g = Log(PI)
  Else
    g = 1# / PI
    For k = 2 To drv_p
      g = (1 - k) * g / PI
    Next k
  End If
End If

If drv_p = 0 Then
  For i = 1 To 9
    pwrt = tau ^ (j0i(i) - drv_t)
    For k = 0 To drv_t - 1
      pwrt = pwrt * (j0i(i) - k)
    Next k
    g = g + n0i(i) * pwrt
  Next i
End If

gamma_0 = g

End Function

'=========================================================================
Private Function gamma_1(ByVal drv_t As Integer, _
                         ByVal drv_p As Integer, _
                         ByVal tau As Double, _
                         ByVal PI As Double) As Double

'this function implements the derivatives of gamma as given in Table 4 of IF-97

Dim i As Integer, k As Integer
Dim g As Double, pp As Double, Tt As Double
Dim pwrt As Double, pwrp As Double

gamma_1 = ErrorReturn

If PI <= 0 Then Exit Function
If tau <= 0 Then Exit Function
If drv_t < 0 Then Exit Function
If drv_p < 0 Then Exit Function

pp = 7.1 - PI
Tt = tau - 1.222

g = 0
For i = 1 To 34

  If Tt = 0 Then
    Select Case j1i(i) - drv_t
      Case 0: pwrt = 1
      Case Is > 0: pwrt = 0
      Case Else: Exit Function
    End Select
  Else
    pwrt = Tt ^ (j1i(i) - drv_t)
  End If
  For k = 0 To drv_t - 1
    pwrt = pwrt * (j1i(i) - k)
  Next k

  If pp = 0 Then
    Select Case i1i(i) - drv_p
      Case 0: pwrp = 1
      Case Is > 0: pwrp = 0
      Case Else: Exit Function
    End Select
  Else
    pwrp = pp ^ (i1i(i) - drv_p)
  End If
  For k = 0 To drv_p - 1
    pwrp = -pwrp * (i1i(i) - k)
  Next k

  g = g + n1i(i) * pwrp * pwrt

Next i

gamma_1 = g

End Function

'=========================================================================
Private Function gamma_2(ByVal drv_t As Integer, _
                         ByVal drv_p As Integer, _
                         ByVal tau As Double, _
                         ByVal PI As Double) As Double

'this function implements the derivatives of gamma as given in Eq. 15 of IF-97

Dim g0 As Double, gr As Double

InitIF97_2

gamma_2 = ErrorReturn

g0 = gamma_0(drv_t, drv_p, tau, PI)
If g0 = ErrorReturn Then Exit Function

gr = gamma_r(drv_t, drv_p, tau, PI)
If gr = ErrorReturn Then Exit Function

gamma_2 = g0 + gr

End Function

'=========================================================================
Private Function gamma_r(ByVal drv_t As Integer, _
                         ByVal drv_p As Integer, _
                         ByVal tau As Double, _
                         ByVal PI As Double) As Double
                         
'this function implements the derivatives of gamma_r as given in Table 14 of IF-97

Dim i As Integer, k As Integer
Dim g As Double, Tt As Double
Dim pwrt As Double, pwrp As Double

gamma_r = ErrorReturn

If PI <= 0 Then Exit Function
If tau <= 0 Then Exit Function
If drv_t < 0 Then Exit Function
If drv_p < 0 Then Exit Function

Tt = tau - 0.5

g = 0
For i = 1 To 43

  If Tt = 0 Then
    Select Case jri(i) - drv_t
      Case 0: pwrt = 1
      Case Is > 0: pwrt = 0
      Case Else: Exit Function
    End Select
  Else
    pwrt = Tt ^ (jri(i) - drv_t)
  End If
  For k = 0 To drv_t - 1
    pwrt = pwrt * (jri(i) - k)
  Next k

  pwrp = PI ^ (iri(i) - drv_p)
  For k = 0 To drv_p - 1
    pwrp = pwrp * (iri(i) - k)
  Next k

  g = g + nri(i) * pwrp * pwrt

Next i

gamma_r = g

End Function
