Attribute VB_Name = "Flu_1_Mdl"
Option Explicit

'#########################################################################

'This module requires the library module
'     Constants_0_Mdl, file Constants_0.bas
'     Maths_0_Mdl,     file Maths_0.bas

'#########################################################################

'=========================================================================
'This module implements the Helmholtz potential of fluid water and its
'first and second partial derivatives with respect to temperature and
'density as defined in IAPWS-95:

'Release on the IAPWS Formulation 1995 for the Thermodynamic Properties of
'Ordinary Water Substance for General and Scientific Use
'The International Association for the Properties of Water and Steam
'Fredericia, Denmark, September 1996,
'revised Doorwerth, The Netherlands, September 2009

'with an extension to 50 - 130 K of the ideal-gas part, as described in

'Feistel, R., Kretzschmar, H.-J., Span, R., Hagen, E., Wright, D.G., Herrmann, S.:
'Thermodynamic Properties of Sea Air.
'Ocean Science Discussion 6(2009)21932325.

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98

'Arrays defined in Table 1 of IAPWS-95
'Numerical values of the coefficients and parameters of the ideal-gas
'part of the dimensionless Helmholtz function and energy, Eq.(5)

Private n0(8) As Double
Private gam0(8) As Double

'Arrays defined in Table 2
'Numerical values of the coefficients and parameters of the
'residual part of the dimensionless Helmholtz function and energy, Eq.(6)

Private ai(56) As Double
Private bi(56) As Double
Private ci(56) As Double
Private di(56) As Double
Private ni(56) As Double
Private ti(56) As Double

Private aa(56) As Double
Private bb(56) As Double
Private cc(56) As Double
Private dd(56) As Double

Private alf(56) As Double
Private bet(56) As Double
Private gam(56) As Double
Private eps(56) As Double

'cp extension below 130 K
Private ee As Double

'Critical Point
Private Const rhoc = CP_density_si   'kg/m3
Private Const Tc = CP_temperature_si  'K

Private Const Version = "07 Jun 2010"

'==========================================================================
Public Function flu_f_si(ByVal drv_t As Integer, _
                         ByVal drv_d As Integer, _
                         ByVal t_si As Double, _
                         ByVal d_si As Double) As Double
                         
'return value:
'flu_f_si:          derivative of the Helmholtz function in the basic SI unit

'input parameters:
'drv_t:             order of the partial temperature derivative
'drv_d:             order of the partial density derivative
't_si:              absolute temperature ITS-90 in K
'd_si:              density in kg/m^3

'Check values:
'flu_f_si(0,0,300,1000) = -5351.74115204056
'flu_f_si(1,0,300,1000) = -390.904170767491
'flu_f_si(0,1,300,1000) = 7.83300135596|477
'flu_f_si(2,0,300,1000) = -13.6840204925353
'flu_f_si(1,1,300,1000) = 0.639359046588391
'flu_f_si(0,2,300,1000) = 2.24824656167368

Dim del As Double, tau As Double
Dim RT As Double
Dim f As Double, ft As Double, fd As Double, fdt As Double, ftt As Double

'Const R = Gas_constant_H2O_si  'specific gas constant of H2O in J/(kg K)
Const R = Gas_constant_H2O_IAPWS95 'value used in the IAPWS-95 Release

flu_f_si = ErrorReturn

'exclude illegal calls:
If check_limits = 1 Then
  If t_si < flu_tmin Or t_si > flu_tmax Then Exit Function
  If d_si <= flu_dmin Or d_si > flu_dmax Then Exit Function
Else
  If t_si <= 0 Then Exit Function
  If d_si <= 0 Then Exit Function
End If

'reduce input values:
tau = Tc / t_si
del = d_si / rhoc

RT = R * t_si

InitIAPWS95

'expressions from Table 4 of the IAPWS Advisory Note #3, revised, Berlin 2008:
Select Case drv_t

  Case 0:
   Select Case drv_d
   
     Case 0: flu_f_si = RT * (phi0_(del, tau) + phir_(del, tau))
     
     Case 1: flu_f_si = RT / rhoc * (phi0_d(del, tau) + phir_d(del, tau))
     
     Case 2: flu_f_si = RT / rhoc ^ 2 * (phi0_dd(del, tau) + phir_dd(del, tau))
     
     Case Else: Exit Function
    End Select
    
  Case 1:
   Select Case drv_d
   
     Case 0: f = phi0_(del, tau) + phir_(del, tau)
             ft = phi0_t(del, tau) + phir_t(del, tau)
             flu_f_si = R * (f - tau * ft)
             
     Case 1: fd = phi0_d(del, tau) + phir_d(del, tau)
             fdt = phi0_dt(del, tau) + phir_dt(del, tau)
             flu_f_si = R / rhoc * (fd - tau * fdt)
             
     Case Else: Exit Function
    End Select

  Case 2:
   Select Case drv_d
   
     Case 0: ftt = phi0_tt(del, tau) + phir_tt(del, tau)
             flu_f_si = R / t_si * tau ^ 2 * ftt
     
     Case Else: Exit Function
    End Select

  Case Else: Exit Function
End Select

End Function

'==========================================================================
Private Function delta_(ByVal i As Integer, ByVal del As Double, ByVal tau As Double) As Double

'Function delta as defined in Table 5 of IAPWS-95:
'The residual part phir of the dimensionless Helmholtz free energy and its derivatives

'away from and at the critical density
delta_ = theta_(i, del, tau) ^ 2 + bb(i) * Abs(del - 1#) ^ (2# * ai(i))


End Function

'==========================================================================
Private Function delta_d(ByVal i As Integer, ByVal del As Double, ByVal tau As Double) As Double

'Function delta_d as defined in Table 5 of IAPWS-95:
'The residual part phir of the dimensionless Helmholtz free energy and its derivatives

If del = 1 Then  'at the critical density
  delta_d = 0
Else
  delta_d = (del - 1#) * (aa(i) * theta_(i, del, tau) * 2# / bet(i) * Abs(del - 1#) ^ (1# / bet(i) - 2#) _
            + 2# * bb(i) * ai(i) * Abs(del - 1#) ^ (2# * ai(i) - 2#))
End If

End Function

'==========================================================================
Private Function delta_dd(ByVal i As Integer, ByVal del As Double, ByVal tau As Double) As Double

'Function delta_dd as defined in Table 5 of IAPWS-95:
'The residual part phir of the dimensionless Helmholtz free energy and its derivatives

If del = 1 Then  'at the critical density
  delta_dd = 0
Else
  delta_dd = delta_d(i, del, tau) / (del - 1) + _
             4 * bb(i) * ai(i) * (ai(i) - 1) * Abs(del - 1) ^ (2 * ai(i) - 2) + _
             2 * (aa(i) / bet(i)) ^ 2 * Abs(del - 1) ^ (2 / bet(i) - 2) + _
             2 * theta_(i, del, tau) * aa(i) / bet(i) * (1 / bet(i) - 2) * Abs(del - 1) ^ (1 / bet(i) - 2)
End If

End Function

'==========================================================================
Private Function deltab_d(ByVal i As Integer, ByVal del As Double, ByVal tau As Double) As Double

'Function delta^bi_d as defined in Table 5 of IAPWS-95:
'The residual part phir of the dimensionless Helmholtz free energy and its derivatives

Dim delta As Double

delta = delta_(i, del, tau)

If delta = 0 Then
  deltab_d = 0
Else
  deltab_d = bi(i) * delta ^ (bi(i) - 1#) * delta_d(i, del, tau)
End If

End Function

'==========================================================================
Private Function deltab_dt(ByVal i As Integer, ByVal del As Double, ByVal tau As Double) As Double

'Function delta^bi_dt as defined in Table 5 of IAPWS-95:
'The residual part phir of the dimensionless Helmholtz free energy and its derivatives

Dim delta As Double

delta = delta_(i, del, tau)

If delta = 0 Then
  deltab_dt = 0
Else
  deltab_dt = -2 * bi(i) * delta ^ (bi(i) - 2) * _
               (aa(i) / bet(i) * delta * (del - 1#) * Abs(del - 1#) ^ (1# / bet(i) - 2#) + _
               theta_(i, del, tau) * (bi(i) - 1) * delta_d(i, del, tau))
End If

End Function

'==========================================================================
Private Function deltab_t(ByVal i As Integer, ByVal del As Double, ByVal tau As Double) As Double

'Function delta^bi_t as defined in Table 5 of IAPWS-95:
'The residual part phir of the dimensionless Helmholtz free energy and its derivatives

Dim delta As Double

delta = delta_(i, del, tau)
If delta = 0 Then
  deltab_t = 0
Else
  deltab_t = -2 * theta_(i, del, tau) * bi(i) * delta ^ (bi(i) - 1)
End If

End Function

'==========================================================================
Private Function deltab_tt(ByVal i As Integer, ByVal del As Double, ByVal tau As Double) As Double

'Function delta^bi_tt as defined in Table 5 of IAPWS-95:
'The residual part phir of the dimensionless Helmholtz free energy and its derivatives

Dim delta As Double

delta = delta_(i, del, tau)
If delta = 0 Then
  deltab_tt = 0
Else
  deltab_tt = 2 * bi(i) * delta ^ (bi(i) - 2) * (delta + 2# * theta_(i, del, tau) ^ 2 * (bi(i) - 1))
End If
End Function

'==========================================================================
Private Function deltab_dd(ByVal i As Integer, ByVal del As Double, ByVal tau As Double) As Double

'Function delta^bi_dd as defined in Table 5 of IAPWS-95:
'The residual part phir of the dimensionless Helmholtz free energy and its derivatives

Dim delta As Double

delta = delta_(i, del, tau)
If delta = 0 Then
  deltab_dd = 0
Else
  deltab_dd = bi(i) * (delta ^ (bi(i) - 1) * delta_dd(i, del, tau) + _
            (bi(i) - 1) * delta ^ (bi(i) - 2) * delta_d(i, del, tau) ^ 2)
End If
End Function

'==========================================================================
Private Function theta_(ByVal i As Integer, ByVal del As Double, ByVal tau As Double) As Double

'Function theta as defined in Table 5 of IAPWS-95:
'The residual part phir of the dimensionless Helmholtz free energy and its derivatives

If del = 1 Then  'at the critical density
  theta_ = 1# - tau
Else
  theta_ = 1# - tau + aa(i) * Abs(del - 1#) ^ (1# / bet(i))
End If

End Function

'==========================================================================
Private Function psi_(ByVal i As Integer, ByVal del As Double, ByVal tau As Double) As Double

'Function psi as defined in Table 5 of IAPWS-95:
'The residual part phir of the dimensionless Helmholtz free energy and its derivatives

psi_ = Exp(-cc(i) * (del - 1#) ^ 2 - dd(i) * (tau - 1#) ^ 2)

End Function

'==========================================================================
Private Function psi_t(ByVal i As Integer, ByVal del As Double, ByVal tau As Double) As Double

'Function psi_t as defined in Table 5 of IAPWS-95:
'The residual part phir of the dimensionless Helmholtz free energy and its derivatives

psi_t = -2 * dd(i) * (tau - 1#) * psi_(i, del, tau)

End Function

'==========================================================================
Private Function psi_tt(ByVal i As Integer, ByVal del As Double, ByVal tau As Double) As Double

'Function psi_tt as defined in Table 5 of IAPWS-95:
'The residual part phir of the dimensionless Helmholtz free energy and its derivatives

psi_tt = 2 * dd(i) * (2 * dd(i) * (tau - 1#) ^ 2 - 1) * psi_(i, del, tau)

End Function

'==========================================================================
Private Function psi_d(ByVal i As Integer, ByVal del As Double, ByVal tau As Double) As Double

'Function psi_d as defined in Table 5 of IAPWS-95:
'The residual part phir of the dimensionless Helmholtz free energy and its derivatives

psi_d = -2# * cc(i) * (del - 1#) * psi_(i, del, tau)

End Function

'==========================================================================
Private Function psi_dt(ByVal i As Integer, ByVal del As Double, ByVal tau As Double) As Double

'Function psi_dt as defined in Table 5 of IAPWS-95:
'The residual part phir of the dimensionless Helmholtz free energy and its derivatives

psi_dt = 4# * cc(i) * dd(i) * (del - 1#) * (tau - 1#) * psi_(i, del, tau)

End Function

'==========================================================================
Private Function psi_dd(ByVal i As Integer, ByVal del As Double, ByVal tau As Double) As Double

'Function psi_dd as defined in Table 5 of IAPWS-95:
'The residual part phir of the dimensionless Helmholtz free energy and its derivatives

psi_dd = 2# * cc(i) * (2 * cc(i) * (del - 1#) ^ 2 - 1#) * psi_(i, del, tau)

End Function

'==========================================================================
Private Sub InitIAPWS95()

Dim i As Integer, phi As Double, phi_t As Double

If di(1) = 1 Then Exit Sub

'Coefficients defined in Table 1 of IAPWS-95
'Numerical values of the coefficients and parameters of the ideal-gas
'part of the dimensionless Helmholtz free energy, Eq.(5)
'-------------------------------------------------------------------------------

'adjustment to the IAPWS-95 reference state conditions at the triple point
'original values in the IAPWS-95 Release
'i = 1: n0(i) = -8.32044648201
'i = 2: n0(i) = 6.6832105268

'improved values obtained from a quadruple precision implementation
i = 1: n0(i) = -8.32044648374969
i = 2: n0(i) = 6.68321052759323

i = 3: n0(i) = 3.00632
i = 4: n0(i) = 0.012436: gam0(i) = 1.28728967
i = 5: n0(i) = 0.97315:  gam0(i) = 3.53734222
i = 6: n0(i) = 1.2795:   gam0(i) = 7.74073708
i = 7: n0(i) = 0.96956:  gam0(i) = 9.24437796
i = 8: n0(i) = 0.24873:  gam0(i) = 27.5075105
'-------------------------------------------------------------------------------

'Coefficients defined in Table 2 of IAPWS-95
'Numerical values of the coefficients and parameters of the
'residual part of the dimensionless Helmholtz free energy, Eq.(6)
'-------------------------------------------------------------------------------
i = 1:             di(i) = 1:  ti(i) = -0.5:  ni(i) = 0.012533547935523
i = 2:             di(i) = 1:  ti(i) = 0.875: ni(i) = 7.8957634722828
i = 3:             di(i) = 1:  ti(i) = 1:     ni(i) = -8.7803203303561
i = 4:             di(i) = 2:  ti(i) = 0.5:   ni(i) = 0.31802509345418
i = 5:             di(i) = 2:  ti(i) = 0.75:  ni(i) = -0.26145533859358
i = 6:             di(i) = 3:  ti(i) = 0.375: ni(i) = -7.8199751687981E-03
i = 7:             di(i) = 4:  ti(i) = 1:     ni(i) = 8.8089493102134E-03
i = 8:  ci(i) = 1: di(i) = 1:  ti(i) = 4:     ni(i) = -0.66856572307965
i = 9:  ci(i) = 1: di(i) = 1:  ti(i) = 6:     ni(i) = 0.20433810950965
i = 10: ci(i) = 1: di(i) = 1:  ti(i) = 12:    ni(i) = -6.6212605039687E-05
i = 11: ci(i) = 1: di(i) = 2:  ti(i) = 1:     ni(i) = -0.19232721156002
i = 12: ci(i) = 1: di(i) = 2:  ti(i) = 5:     ni(i) = -0.25709043003438
i = 13: ci(i) = 1: di(i) = 3:  ti(i) = 4:     ni(i) = 0.16074868486251
i = 14: ci(i) = 1: di(i) = 4:  ti(i) = 2:     ni(i) = -0.040092828925807
i = 15: ci(i) = 1: di(i) = 4:  ti(i) = 13:    ni(i) = 3.9343422603254E-07
i = 16: ci(i) = 1: di(i) = 5:  ti(i) = 9:     ni(i) = -7.5941377088144E-06
i = 17: ci(i) = 1: di(i) = 7:  ti(i) = 3:     ni(i) = 5.6250979351888E-04
i = 18: ci(i) = 1: di(i) = 9:  ti(i) = 4:     ni(i) = -1.5608652257135E-05
i = 19: ci(i) = 1: di(i) = 10: ti(i) = 11:    ni(i) = 1.1537996422951E-09
i = 20: ci(i) = 1: di(i) = 11: ti(i) = 4:     ni(i) = 3.6582165144204E-07
i = 21: ci(i) = 1: di(i) = 13: ti(i) = 13:    ni(i) = -1.3251180074668E-12
i = 22: ci(i) = 1: di(i) = 15: ti(i) = 1:     ni(i) = -6.2639586912454E-10
i = 23: ci(i) = 2: di(i) = 1:  ti(i) = 7:     ni(i) = -0.10793600908932
i = 24: ci(i) = 2: di(i) = 2:  ti(i) = 1:     ni(i) = 0.017611491008752
i = 25: ci(i) = 2: di(i) = 2:  ti(i) = 9:     ni(i) = 0.22132295167546
i = 26: ci(i) = 2: di(i) = 2:  ti(i) = 10:    ni(i) = -0.40247669763528
i = 27: ci(i) = 2: di(i) = 3:  ti(i) = 10:    ni(i) = 0.58083399985759
i = 28: ci(i) = 2: di(i) = 4:  ti(i) = 3:     ni(i) = 4.9969146990806E-03
i = 29: ci(i) = 2: di(i) = 4:  ti(i) = 7:     ni(i) = -0.031358700712549
i = 30: ci(i) = 2: di(i) = 4:  ti(i) = 10:    ni(i) = -0.74315929710341
i = 31: ci(i) = 2: di(i) = 5:  ti(i) = 10:    ni(i) = 0.4780732991548
i = 32: ci(i) = 2: di(i) = 6:  ti(i) = 6:     ni(i) = 0.020527940895948
i = 33: ci(i) = 2: di(i) = 6:  ti(i) = 10:    ni(i) = -0.13636435110343
i = 34: ci(i) = 2: di(i) = 7:  ti(i) = 10:    ni(i) = 0.014180634400617
i = 35: ci(i) = 2: di(i) = 9:  ti(i) = 1:     ni(i) = 8.3326504880713E-03
i = 36: ci(i) = 2: di(i) = 9:  ti(i) = 2:     ni(i) = -0.029052336009585
i = 37: ci(i) = 2: di(i) = 9:  ti(i) = 3:     ni(i) = 0.038615085574206
i = 38: ci(i) = 2: di(i) = 9:  ti(i) = 4:     ni(i) = -0.020393486513704
i = 39: ci(i) = 2: di(i) = 9:  ti(i) = 8:     ni(i) = -1.6554050063734E-03
i = 40: ci(i) = 2: di(i) = 10: ti(i) = 6:     ni(i) = 1.9955571979541E-03
i = 41: ci(i) = 2: di(i) = 10: ti(i) = 9:     ni(i) = 1.5870308324157E-04
i = 42: ci(i) = 2: di(i) = 12: ti(i) = 8:     ni(i) = -1.638856834253E-05
i = 43: ci(i) = 3: di(i) = 3:  ti(i) = 16:    ni(i) = 0.043613615723811
i = 44: ci(i) = 3: di(i) = 4:  ti(i) = 22:    ni(i) = 0.034994005463765
i = 45: ci(i) = 3: di(i) = 4:  ti(i) = 23:    ni(i) = -0.076788197844621
i = 46: ci(i) = 3: di(i) = 5:  ti(i) = 23:    ni(i) = 0.022446277332006
i = 47: ci(i) = 4: di(i) = 14: ti(i) = 10:    ni(i) = -6.2689710414685E-05
i = 48: ci(i) = 6: di(i) = 3:  ti(i) = 50:    ni(i) = -5.5711118565645E-10
i = 49: ci(i) = 6: di(i) = 6:  ti(i) = 44:    ni(i) = -0.19905718354408
i = 50: ci(i) = 6: di(i) = 6:  ti(i) = 46:    ni(i) = 0.31777497330738
i = 51: ci(i) = 6: di(i) = 6:  ti(i) = 50:    ni(i) = -0.11841182425981
'------------------------------------------------------------------------------------------------------------------------------
i = 52: di(i) = 3:  ti(i) = 0: ni(i) = -31.306260323435: alf(i) = 20: bet(i) = 150: gam(i) = 1.21: eps(i) = 1
i = 53: di(i) = 3:  ti(i) = 1: ni(i) = 31.546140237781:  alf(i) = 20: bet(i) = 150: gam(i) = 1.21: eps(i) = 1
i = 54: di(i) = 3:  ti(i) = 4: ni(i) = -2521.3154341695: alf(i) = 20: bet(i) = 250: gam(i) = 1.25: eps(i) = 1
'------------------------------------------------------------------------------------------------------------------------------
i = 55: ai(i) = 3.5: bi(i) = 0.85: bb(i) = 0.2: ni(i) = -0.14874640856724: cc(i) = 28: dd(i) = 700: aa(i) = 0.32: bet(i) = 0.3
i = 56: ai(i) = 3.5: bi(i) = 0.95: bb(i) = 0.2: ni(i) = 0.31806110878444:  cc(i) = 32: dd(i) = 800: aa(i) = 0.32: bet(i) = 0.3
'------------------------------------------------------------------------------------------------------------------------------

'cp extension below 130 K
ee = 0.278296458178592

End Sub

'==========================================================================
Private Function phi0_(ByVal del As Double, ByVal tau As Double) As Double

'Function phi0 as defined in row #1 of Table 4 of IAPWS-95:
'The ideal-gas part phi0 of the dimensionless Helmholtz free energy and its derivatives

Const e = Tc / 130#

Dim phi As Double
Dim i As Integer

phi = Log(del) + n0(1) + n0(2) * tau + n0(3) * Log(tau)
For i = 4 To 8
  phi = phi + n0(i) * Log(1# - Exp(-gam0(i) * tau))
Next i

If tau > e Then  'extension below 130 K
  phi = phi + ee * (-0.5 / tau - 3# / e ^ 2 * (tau + e) * Log(tau / e) _
                    + 4.5 * tau / e ^ 2 + 0.5 * tau ^ 2 / e ^ 3 - 4.5 / e)
End If

phi0_ = phi
 
End Function

'==========================================================================
Private Function phi0_t(ByVal del As Double, ByVal tau As Double) As Double

'Function phi0_t as defined in row #4 of Table 4 of IAPWS-95:
'The ideal-gas part phi0 of the dimensionless Helmholtz free energy and its derivatives

Const e = Tc / 130#

Dim phi As Double
Dim i As Integer

phi = n0(2) + n0(3) / tau
For i = 4 To 8
  phi = phi + n0(i) * gam0(i) * (1# / (1# - Exp(-gam0(i) * tau)) - 1#)
Next i

If tau > e Then  'extension below 130 K
  phi = phi + ee * (0.5 / tau ^ 2 - 3# / e ^ 2 * Log(tau / e) _
                    - 3# / (e * tau) + tau / e ^ 3 + 1.5 / e ^ 2)
End If

phi0_t = phi

End Function

'==========================================================================
Private Function phi0_tt(ByVal del As Double, ByVal tau As Double) As Double

'Function phi0_tt as defined in row #5 of Table 4 of IAPWS-95:
'The ideal-gas part phi0 of the dimensionless Helmholtz free energy and its derivatives

Const e = Tc / 130#

Dim phi As Double
Dim i As Integer

phi = -n0(3) / tau ^ 2
For i = 4 To 8
  phi = phi - n0(i) * gam0(i) ^ 2 * Exp(-gam0(i) * tau) / (1# - Exp(-gam0(i) * tau)) ^ 2
Next i

If tau > e Then  'extension below 130 K
  phi = phi + ee * (-1# / tau + 1# / e) ^ 3
End If

phi0_tt = phi

End Function

'==========================================================================
Private Function phi0_d(ByVal del As Double, ByVal tau As Double) As Double

'Function phi0_d as defined in row #2 of Table 4 of IAPWS-95:
'The ideal-gas part phi0 of the dimensionless Helmholtz free energy and its derivatives

phi0_d = 1# / del

End Function

'==========================================================================
Private Function phi0_dd(ByVal del As Double, ByVal tau As Double) As Double

'Function phi0_dd as defined in row #3 of Table 4 of IAPWS-95:
'The ideal-gas part phi0 of the dimensionless Helmholtz free energy and its derivatives

phi0_dd = -1# / del ^ 2

End Function

'==========================================================================
Private Function phi0_dt(ByVal del As Double, ByVal tau As Double) As Double

'Function phi0_dt as defined in row #6 of Table 4 of IAPWS-95:
'The ideal-gas part phi0 of the dimensionless Helmholtz free energy and its derivatives

phi0_dt = 0

End Function

'==========================================================================
Private Function phir_(ByVal del As Double, ByVal tau As Double) As Double

'Function phir as defined in Table 5 of IAPWS-95:
'The residual part phir of the dimensionless Helmholtz free energy and its derivatives

Dim phi As Double
Dim i As Integer

phi = 0

For i = 1 To 7
  phi = phi + ni(i) * del ^ di(i) * tau ^ ti(i)
Next i

For i = 8 To 51
  phi = phi + ni(i) * del ^ di(i) * tau ^ ti(i) * Exp(-del ^ ci(i))
Next i

For i = 52 To 54
  phi = phi + ni(i) * del ^ di(i) * tau ^ ti(i) * _
              Exp(-alf(i) * (del - eps(i)) ^ 2 - bet(i) * (tau - gam(i)) ^ 2)
Next i

For i = 55 To 56
  phi = phi + ni(i) * delta_(i, del, tau) ^ bi(i) * del * psi_(i, del, tau)
Next i

phir_ = phi

End Function

'==========================================================================
Private Function phir_d(ByVal del As Double, ByVal tau As Double) As Double

'Function phir_d as defined in Table 5 of IAPWS-95:
'The residual part phir of the dimensionless Helmholtz free energy and its derivatives

Dim phi As Double, psi As Double
Dim i As Integer

phi = 0

For i = 1 To 7
  phi = phi + ni(i) * di(i) * del ^ (di(i) - 1) * tau ^ ti(i)
Next i

For i = 8 To 51
  phi = phi + ni(i) * Exp(-del ^ ci(i)) * (del ^ (di(i) - 1) * tau ^ ti(i) * (di(i) - ci(i) * del ^ ci(i)))
Next i

For i = 52 To 54
  phi = phi + ni(i) * del ^ di(i) * tau ^ ti(i) * _
              Exp(-alf(i) * (del - eps(i)) ^ 2 - bet(i) * (tau - gam(i)) ^ 2) * _
              (di(i) / del - 2# * alf(i) * (del - eps(i)))
Next i

For i = 55 To 56
  psi = psi_(i, del, tau)
  phi = phi + ni(i) * (delta_(i, del, tau) ^ bi(i) * _
              (psi + del * psi_d(i, del, tau)) + deltab_d(i, del, tau) * del * psi)
Next i

phir_d = phi

End Function

'==========================================================================
Private Function phir_dd(ByVal del As Double, ByVal tau As Double) As Double

'Function phir_dd as defined in Table 5 of IAPWS-95:
'The residual part phir of the dimensionless Helmholtz free energy and its derivatives

Dim phi As Double, psi As Double, psid As Double, tmp As Double
Dim i As Integer

phi = 0

For i = 1 To 7
  phi = phi + ni(i) * di(i) * (di(i) - 1) * del ^ (di(i) - 2) * tau ^ ti(i)
Next i

For i = 8 To 51
  tmp = (del ^ (di(i) - 2) * tau ^ ti(i) * _
        ((di(i) - ci(i) * del ^ ci(i)) * (di(i) - 1 - ci(i) * del ^ ci(i)) - ci(i) ^ 2 * del ^ ci(i)))
  phi = phi + ni(i) * Exp(-del ^ ci(i)) * tmp
Next i

For i = 52 To 54
  phi = phi + ni(i) * tau ^ ti(i) * _
              Exp(-alf(i) * (del - eps(i)) ^ 2 - bet(i) * (tau - gam(i)) ^ 2) * _
              (-2 * alf(i) * del ^ di(i) + 4 * alf(i) ^ 2 * del ^ di(i) * (del - eps(i)) ^ 2 _
              - 4 * di(i) * alf(i) * del ^ (di(i) - 1) * (del - eps(i)) + _
              di(i) * (di(i) - 1) * del ^ (di(i) - 2))
Next i

For i = 55 To 56
  psi = psi_(i, del, tau)
  psid = psi_d(i, del, tau)
  phi = phi + ni(i) * (delta_(i, del, tau) ^ bi(i) * (2 * psid + del * psi_dd(i, del, tau)) + _
              2 * deltab_d(i, del, tau) * (psi + del * psid) + deltab_dd(i, del, tau) * del * psi)
Next i

phir_dd = phi

End Function

'==========================================================================
Private Function phir_t(ByVal del As Double, ByVal tau As Double) As Double

'Function phir_t as defined in Table 5 of IAPWS-95:
'The residual part phir of the dimensionless Helmholtz free energy and its derivatives

Const fastmode = False
'flu_entropy_si(647, 358) = 4320.923066755  correct

'Const fastmode = True
'flu_entropy_si(647, 358) = 4320.92306292204  less correct

If fastmode Then GoTo JackettCode

Dim phi As Double
Dim i As Integer

phi = 0

For i = 1 To 7
  phi = phi + ni(i) * ti(i) * del ^ di(i) * tau ^ (ti(i) - 1)
Next i

For i = 8 To 51
  phi = phi + ni(i) * ti(i) * del ^ di(i) * tau ^ (ti(i) - 1) * Exp(-del ^ ci(i))
Next i

For i = 52 To 54
  phi = phi + ni(i) * del ^ di(i) * tau ^ ti(i) * _
              Exp(-alf(i) * (del - eps(i)) ^ 2 - bet(i) * (tau - gam(i)) ^ 2) * _
              (ti(i) / tau - 2 * bet(i) * (tau - gam(i)))
Next i

For i = 55 To 56
  phi = phi + ni(i) * del * (deltab_t(i, del, tau) * psi_(i, del, tau) + _
                             delta_(i, del, tau) ^ bi(i) * psi_t(i, del, tau))
Next i

phir_t = phi

Exit Function


'======================================================================================
'fast version adopted from David Jackett's Fortran code
'======================================================================================

JackettCode:

'!    First derivative (with respect to tau) of the residual part of the
'!    dimensionless Helmholtz free energy of fluid water, as in
'!
'!    The International Association for the Properties of Water and Steam
'!    Fredericia, Denmark, September 1996
'!
'!    delta               : dimensionless density
'!    tau                 : dimensionless temperature
'!
'!    DphirDtau           : first derivative of phir (wrt tau)
'!
'!
'!    David Jackett       : 10/04/08

Dim delta#, delta2#, delta3#, delta4#, delta6#, delta13#, delta14#

Dim delta1_2#, delta1_delta#, delta_tau#, delta_tau2#, delta_tau3#, delta_tau4#, delta2_tau#

Dim tau2#, tau3#, tau4#, tau5#, tau6#, tau7#

Dim tau8#, tau9#, tau10#, tau11#, tau13#, tau12#, tau15#, tau16#, tau21#, tau22#, tau23#, tau43#, tau44#, tau49#, tau50#

Dim taup375#, taup5#, taup75#, taup875#

Dim tau1_2#, tau2_2#, tau3_2#, tau_1#

Dim cd1#, cd2#, cd3#, cd4#, cd6#, cdt1#, cdt2#, cdt3#, cdt4#

Dim edt1#, edt2#, edt3#, edt4#, the_rest#

delta = del
delta2 = delta * delta: delta3 = delta * delta2: delta4 = delta * delta3: delta6 = delta3 * delta3:
delta13 = delta3 * delta4 * delta6: delta14 = delta * delta13

delta1_2 = (-1# + delta) * (-1# + delta):

tau2 = tau * tau:    tau3 = tau * tau2: tau4 = tau * tau3: tau5 = tau * tau4: tau6 = tau * tau5:
tau7 = tau * tau6:   tau8 = tau * tau7: tau9 = tau * tau8: tau10 = tau * tau9: tau11 = tau2 * tau9:
tau12 = tau * tau11: tau13 = tau4 * tau9: tau15 = tau4 * tau11:
tau16 = tau8 * tau8: tau21 = tau5 * tau16: tau22 = tau9 * tau13: tau23 = tau * tau22:
tau43 = tau21 * tau22: tau44 = tau22 * tau22: tau49 = tau5 * tau44: tau50 = tau44 * tau6

taup375 = tau ^ 0.375: taup5 = Sqr(tau): taup75 = taup375 * taup375: taup875 = taup375 * taup5

tau1_2 = (-1.21 + tau) * (-1.21 + tau): tau2_2 = (-1.25 + tau) * (-1.25 + tau):
tau3_2 = (-1# + tau) * (-1# + tau):


cd1 = delta * (1.26917960652461E-08 * tau10 + _
                     delta * (1.46328660576816E-06 * tau3 + _
                       delta2 * (-6.2639586912454E-10 * delta2 - 1.72265340970684E-11 * tau12)))
cd1 = delta * (-6.83472393793296E-05 * tau8 + _
                 delta2 * (1.68752938055664E-03 * tau2 + _
                  delta2 * (-6.243460902854E-05 * tau3 + cd1)))
cd1 = delta * (0.64299473945004 * tau3 + delta * _
              (tau * (-0.080185657851614 + 5.11464493842302E-06 * tau11) + cd1))
cd1 = delta * (tau3 * (-2.6742628923186 + tau2 * (1.2260286570579 - 7.94551260476244E-04 * tau6)) + _
        delta * (-0.19232721156002 - 1.2854521501719 * tau4 + cd1))


cd2 = delta * (-1.3110854674024E-04 * delta2 * tau7 + _
                       tau5 * (1.19733431877246E-02 + 1.42832774917413E-03 * tau3))
cd2 = cd2 + tau * (-0.05810467201917 + _
                       tau * (tau * (-0.081573946054816 - 1.32432400509872E-02 * tau4) + 0.115845256722618))
cd2 = delta * (tau5 * (0.123167645375688 - 1.3636435110343 * tau4) + _
                  delta * (0.14180634400617 * tau9 + _
                    delta2 * (8.3326504880713E-03 + cd2)))
cd2 = (tau2 * (1.49907440972418E-02 + tau4 * (-0.219510904987843 - 7.4315929710341 * tau3)) + _
               delta * (4.780732991548 * tau9 + cd2))
cd2 = delta * (-0.75555206362524 * tau6 + delta * (0.017611491008752 + _
            (1.99190656507914 - 4.0247669763528 * tau) * tau8 + _
            delta * (5.8083399985759 * tau9 + delta * cd2)))


cd3 = delta3 * (0.697817851580976 * tau15 + delta * _
           ((0.76986812020283 - 1.76612855042628 * tau) * tau21 + 0.516264378636138 * delta * tau22))


cd4 = -6.2689710414685E-04 * delta14 * tau9


cd6 = delta3 * (-2.78555592828225E-08 * tau49 + _
         delta3 * tau43 * (-8.75851607593952 + tau2 * (14.6176487721395 - 5.9205912129905 * tau4)))



cdt1 = delta3 * (-11332.6263571691 + (20843.127003345 - 9463.8420713343 * tau) * tau)

edt1 = Exp(-20# * delta1_2 - 150# * tau1_2)


cdt2 = delta3 * tau3 * (-10085.261736678 + tau * (-1575822.14635594 + 1260657.71708475 * tau))

edt2 = Exp(-20# * delta1_2 - 250# * tau2_2)


delta_tau = (1# + 0.32 * delta1_2 ^ 1.66666666666667 - tau): delta_tau2 = delta_tau * delta_tau: delta2_tau = delta * delta_tau:

delta_tau3 = (0.2 * delta1_2 ^ 3.5 + delta_tau2): delta_tau4 = delta * tau_1:

If delta_tau3 = 0 Then
  cdt3 = 0
Else
  cdt3 = delta_tau3 ^ 0.85 * ((0.252868894564308 * delta2_tau) / delta_tau3 + 208.244971994136 * delta_tau4)
End If

edt3 = Exp(-28# * delta1_2 - 700# * tau3_2):

If delta_tau3 = 0 Then
  cdt4 = 0
Else
  cdt4 = delta_tau3 ^ 0.95 * ((-0.604316106690436 * delta2_tau) / delta_tau3 - 508.897774055104 * delta_tau4)
End If

edt4 = Exp(-32# * delta1_2 - 800# * tau3_2):
        
        
the_rest = -8.7803203303561 * delta + 8.8089493102134E-03 * delta4 - (6.2667739677615E-03 * delta * taup5) / tau2 + _
                  (-2.93249068829929E-03 * delta3 * taup375 + delta2 * (0.15901254672709 * taup5 - _
                         0.196091503945185 * taup75) + 6.90879303824745 * delta * taup875) / tau
      
      
phir_t = cd1 * Exp(-delta) + cd2 * Exp(-delta2) + cd3 * Exp(-delta3) + cd4 * Exp(-delta4) + cd6 * Exp(-delta6) + _
              cdt1 * edt1 + cdt2 * edt2 + cdt3 * edt3 + cdt4 * edt4 + the_rest
       


End Function

'==========================================================================
Private Function phir_dt(ByVal del As Double, ByVal tau As Double) As Double

'Function phir_dt as defined in Table 5 of IAPWS-95:
'The residual part phir of the dimensionless Helmholtz free energy and its derivatives

Dim phi As Double, psi As Double, psit As Double
Dim i As Integer

phi = 0

For i = 1 To 7
  phi = phi + ni(i) * di(i) * ti(i) * del ^ (di(i) - 1) * tau ^ (ti(i) - 1)
Next i

For i = 8 To 51
  phi = phi + ni(i) * ti(i) * Exp(-del ^ ci(i)) * del ^ (di(i) - 1) * tau ^ (ti(i) - 1) * _
             (di(i) - ci(i) * del ^ ci(i))
Next i

For i = 52 To 54
  phi = phi + ni(i) * del ^ di(i) * tau ^ ti(i) * _
              Exp(-alf(i) * (del - eps(i)) ^ 2 - bet(i) * (tau - gam(i)) ^ 2) * _
              (di(i) / del - 2# * alf(i) * (del - eps(i))) * _
              (ti(i) / tau - 2 * bet(i) * (tau - gam(i)))
Next i

For i = 55 To 56
  psi = psi_(i, del, tau)
  psit = psi_t(i, del, tau)
  phi = phi + ni(i) * (delta_(i, del, tau) ^ bi(i) * (psit + del * psi_dt(i, del, tau)) + _
              del * deltab_d(i, del, tau) * psit + _
              deltab_t(i, del, tau) * (psi + del * psi_d(i, del, tau)) + _
              deltab_dt(i, del, tau) * del * psi)
Next i

phir_dt = phi

End Function

'==========================================================================
Private Function phir_tt(ByVal del As Double, ByVal tau As Double) As Double

'Function phir_tt as defined in Table 5 of IAPWS-95:
'The residual part phir of the dimensionless Helmholtz free energy and its derivatives

Dim phi As Double
Dim i As Integer

phi = 0

For i = 1 To 7
  phi = phi + ni(i) * ti(i) * (ti(i) - 1) * del ^ di(i) * tau ^ (ti(i) - 2)
Next i

For i = 8 To 51
  phi = phi + ni(i) * ti(i) * (ti(i) - 1) * del ^ di(i) * tau ^ (ti(i) - 2) * Exp(-del ^ ci(i))
Next i

For i = 52 To 54
  phi = phi + ni(i) * del ^ di(i) * tau ^ ti(i) * _
              Exp(-alf(i) * (del - eps(i)) ^ 2 - bet(i) * (tau - gam(i)) ^ 2) * _
              ((ti(i) / tau - 2 * bet(i) * (tau - gam(i))) ^ 2 - ti(i) / tau ^ 2 - 2 * bet(i))
Next i

For i = 55 To 56
  phi = phi + ni(i) * del * (deltab_tt(i, del, tau) * psi_(i, del, tau) + _
                         2 * deltab_t(i, del, tau) * psi_t(i, del, tau) + _
                             delta_(i, del, tau) ^ bi(i) * psi_tt(i, del, tau))
Next i

phir_tt = phi

End Function

'==========================================================================
Public Function chk_IAPWS95_Table6() As String

'Function values as given in Table 6 of IAPWS-95, revision of Doorwerth 2009:
'Values for the ideal-gas part phi0, Eq. (5), and for the residual part phir , Eq. (6), of the
'dimensionless Helmholtz free energy, together with the corresponding derivatives
'for T = 500 K and rho = 838.025 kg m3

'phi0_ =     0.204 797 733E1  phir_ =    0.342 693 206E1
'phi0_d =    0.384 236 747    phir_d =   0.364 366 650
'phi0_dd =  0.147 637 878    phir_dd =   0.856 063 701
'phi0_t =    0.904 611 106E1  phir_t =   0.581 403 435E1
'phi0_tt =  0.193 249 185E1  phir_tt =  0.223 440 737E1
'phi0_dt =   0                phir_dt =  0.112 176 915E1

Dim CRLF As String, TB As String
Dim txt As String
Dim tau As Double, del As Double

CRLF = Chr(13) + Chr(10)
TB = Chr(9)

txt = "Implementation of IAPWS-95 in Visual Basic" + CRLF
txt = txt + "for Publication in Ocean Science, 2009" + CRLF
txt = txt + "R. Feistel, IOW, Version " + Version + CRLF
txt = txt + "Compiled on " + CStr(Now) + CRLF + CRLF

txt = txt + "Function values as given in Table 6 of IAPWS-95:" + CRLF
txt = txt + "Values for the ideal-gas part, phi0, Eq.(5)," + CRLF
txt = txt + "for T = 500 K and rho = 838.025 kg/m^3" + CRLF + CRLF

del = 838.025 / rhoc
tau = Tc / 500#

InitIAPWS95

txt = txt + "Call:    " + TB + " Table Value" + CRLF
txt = txt + "phi0_    " + TB + EFormat(Val(" 0.204797733E1"), 9) + CRLF
txt = txt + "this code" + TB + EFormat(phi0_(del, tau), 9) + CRLF + CRLF
txt = txt + "phi0_d   " + TB + EFormat(Val(" 0.384236747"), 9) + CRLF
txt = txt + "this code" + TB + EFormat(phi0_d(del, tau), 9) + CRLF + CRLF
txt = txt + "phi0_dd  " + TB + EFormat(Val("-0.147637878"), 9) + CRLF
txt = txt + "this code" + TB + EFormat(phi0_dd(del, tau), 9) + CRLF + CRLF
txt = txt + "phi0_t   " + TB + EFormat(Val(" 0.904611106E1"), 9) + CRLF
txt = txt + "this code" + TB + EFormat(phi0_t(del, tau), 9) + CRLF + CRLF
txt = txt + "phi0_tt  " + TB + EFormat(Val("-0.193249185E1"), 9) + CRLF
txt = txt + "this code" + TB + EFormat(phi0_tt(del, tau), 9) + CRLF + CRLF
txt = txt + "phi0_dt  " + TB + EFormat(Val(" 0.0"), 9) + CRLF
txt = txt + "this code" + TB + EFormat(phi0_dt(del, tau), 9) + CRLF + CRLF

txt = txt + "Values for the residual part, phir, Eq.(5)," + CRLF + CRLF

txt = txt + "Call:    " + TB + " Table Value" + CRLF
txt = txt + "phir_    " + TB + EFormat(Val("-0.342693206E1"), 9) + CRLF
txt = txt + "this code" + TB + EFormat(phir_(del, tau), 9) + CRLF + CRLF
txt = txt + "phir_d   " + TB + EFormat(Val("-0.364366650"), 9) + CRLF
txt = txt + "this code" + TB + EFormat(phir_d(del, tau), 9) + CRLF + CRLF
txt = txt + "phir_dd  " + TB + EFormat(Val(" 0.856063701"), 9) + CRLF
txt = txt + "this code" + TB + EFormat(phir_dd(del, tau), 9) + CRLF + CRLF
txt = txt + "phir_t   " + TB + EFormat(Val("-0.581403435E1"), 9) + CRLF
txt = txt + "this code" + TB + EFormat(phir_t(del, tau), 9) + CRLF + CRLF
txt = txt + "phir_tt  " + TB + EFormat(Val("-0.223440737E1"), 9) + CRLF
txt = txt + "this code" + TB + EFormat(phir_tt(del, tau), 9) + CRLF + CRLF
txt = txt + "phir_dt  " + TB + EFormat(Val("-0.112176915E1"), 9) + CRLF
txt = txt + "this code" + TB + EFormat(phir_dt(del, tau), 9) + CRLF + CRLF

chk_IAPWS95_Table6 = txt

End Function

'==========================================================================
Public Function chk_IAPWS95_Table7() As String

'Function values as given in Table 7 of IAPWS-95, revision of Doorwerth 2009:
'Thermodynamic property values in the single-phase region for selected values of T and rho

'T   rho             p                cv              w               s
'K   kg m3          MPa              kJ kg1 K1     m s1           kJ kg1 K1
'300 0.996 556 0E3   0.992 418 352E1 0.413 018 112E1 0.150 151 914E4 0.393 062 643
'    0.100 530 8E4   0.200 022 515E2  0.406 798 347E1 0.153 492 501E4 0.387 405 401
'    0.118 820 2E4   0.700 004 704E3  0.346 135 580E1 0.244 357 992E4 0.132 609 616
'500 0.435 000 0     0.999 679 423E1 0.150 817 541E1 0.548 314 253E3 0.794 488 271E1
'    0.453 200 0E1   0.999 938 125    0.166 991 025E1 0.535 739 001E3 0.682 502 725E1
'    0.838 025 0E3   0.100 003 858E2  0.322 106 219E1 0.127 128 441E4 0.256 690 919E1
'    0.108 456 4E4   0.700 000 405E3  0.307 437 693E1 0.241 200 877E4 0.203 237 509E1
'647 0.358 000 0E3   0.220 384 756E2  0.618 315 728E1 0.252 145 078E3 0.432 092 307E1
'900 0.241 000 0     0.100 062 559    0.175 890 657E1 0.724 027 147E3 0.916 653 194E1
'    0.526 150 0E2   0.200 000 690E2  0.193 510 526E1 0.698 445 674E3 0.659 070 225E1
'    0.870 769 0E3   0.700 000 006E3  0.266 422 350E1 0.201 933 608E4 0.417 223 802E1

Dim CRLF As String
Dim txt As String
Dim d As Double, t As Double

Dim i As Integer, row As String
Dim ft As Double, fd As Double, ftt As Double, ftd As Double, fdd As Double
Dim p As Double, cv As Double, c As Double, eta As Double

CRLF = Chr(13) + Chr(10)

txt = "Implementation of IAPWS-95 in Visual Basic" + CRLF
txt = txt + "for Publication in Ocean Science, 2009" + CRLF
txt = txt + "R. Feistel, IOW, Version " + Version + CRLF
txt = txt + "Compiled on " + CStr(Now) + CRLF + CRLF

txt = txt + "Function values as given in Table 7 of IAPWS-95:" + CRLF
txt = txt + "Thermodynamic property values in the single-phase region for selected values of T and rho" + CRLF + CRLF

txt = txt + "T   rho           P               cv              c               eta" + CRLF
txt = txt + "K   kg/m^3        MPa             kJ/(kg K)       m/s             kJ/(kg K)" + CRLF

For i = 1 To 11
  Select Case i
    Case 1:  row = "300 0.9965560E+3  0.992418352E-1  0.413018112E+1  0.150151914E+4  0.393062643"
    Case 2:  row = "    0.1005308E+4  0.200022515E+2  0.406798347E+1  0.153492501E+4  0.387405401"
    Case 3:  row = "    0.1188202E+4  0.700004704E+3  0.346135580E+1  0.244357992E+4  0.132609616"
    Case 4:  row = "500 0.4350000     0.999679423E-1  0.150817541E+1  0.548314253E+3  0.794488271E+1"
    Case 5:  row = "    0.4532000E+1  0.999938125     0.166991025E+1  0.535739001E+3  0.682502725E+1"
    Case 6:  row = "    0.8380250E+3  0.100003858E+2  0.322106219E+1  0.127128441E+4  0.256690918E+1"
    Case 7:  row = "    0.1084564E+4  0.700000405E+3  0.307437693E+1  0.241200877E+4  0.203237509E+1"
    Case 8:  row = "647 0.3580000E+3  0.220384756E+2  0.618315728E+1  0.252145078E+3  0.432092307E+1"
    Case 9:  row = "900 0.2410000     0.100062559     0.175890657E+1  0.724027147E+3  0.916653194E+1"
    Case 10: row = "    0.5261500E+2  0.200000690E+2  0.193510526E+1  0.698445674E+3  0.659070225E+1"
    Case 11: row = "    0.8707690E+3  0.700000006E+3  0.266422350E+1  0.201933608E+4  0.417223802E+1"
  End Select

  txt = txt + row + CRLF
  txt = txt + "this code:       "
  
  'get temperature and density input
  If Left(row, 3) <> "   " Then t = Val(Left(row, 3))
  d = Val(Mid(row, 5, 12))
  
  'compute pressure
  fd = flu_f_si(0, 1, t, d)
  p = d ^ 2 * fd
  txt = txt + Left(EFormat(0.000001 * p, 9) + Space(16), 16)
  
  'compute cv
  ftt = flu_f_si(2, 0, t, d)
  cv = -t * ftt
  txt = txt + Left(EFormat(0.001 * cv, 9) + Space(16), 16)
  
  'compute sound speed
  fdd = flu_f_si(0, 2, t, d)
  ftd = flu_f_si(1, 1, t, d)
  c = Sqr(d ^ 2 * (ftt * fdd - ftd ^ 2) / ftt + 2 * d * fd)
  txt = txt + Left(EFormat(c, 9) + Space(16), 16)
  
  'compute entropy
  ft = flu_f_si(1, 0, t, d)
  eta = -ft
  txt = txt + EFormat(0.001 * eta, 9) + CRLF + CRLF
  
Next i

chk_IAPWS95_Table7 = txt

End Function






