Attribute VB_Name = "Air_1_Mdl"
Option Explicit

'#########################################################################

'This module requires the library module
'     Constants_0_Mdl, file Constants_0.bas

'#########################################################################


'=========================================================================
'This module implements the Helmholtz potential of dry air and its
'first and second partial derivatives with respect to temperature and
'density as defined in:

'Lemmon, E.W., Jacobsen, R.T., Penoncello, S.G., Friend, D.G.:
'Thermodynamic Properties of Air and Mixtures of Nitrogen, Argon and Oxygen
'From 60 to 2000 K at Pressures to 2000 MPa.
'J. Phys. Chem. Ref. Data 29(2000)331-362.

'This module implements the cross-virial coefficients of humid air
'and their first and second derivatives with respect to temperature.

'The 2nd mixed virial coefficient is defined in:

'Harvey, A.H., Huang, P.H.:
'First-Principles Calculation of the Air-Water Second Virial Coefficient.
'Int. J. Thermophys. 28(2007)556-565.

'The 3rd mixed virial coefficients are defined in:

'Hyland, R.W., Wexler, A.:
'Formulations for the thermodynamic properties of dry air from 173.15 K to 473.15 K,
'and of saturated moist air from 173.15 K to 372.15 K, at pressures up to 5 MPa.
'ASHRAE Transact. 89(1983)520-535.

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009

'Feistel, R., Kretzschmar, H.-J., Span, R., Hagen, E., Wright, D.G., Herrmann, S.:
'Thermodynamic Properties of Sea Air.
'Ocean Science Discussion 6(2009)21932325.
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98

'Arrays defined in Tables 12 and 13 of Lemmon et al. (2000)
'Coefficients and Exponents for the Fundamental Equation (3) for Air

Private nk0(13) As Double
Private nk(19) As Double
Private ik(19) As Double
Private jk(19) As Double
Private lk(19) As Double

Private Const ma = Molar_mass_air_L2000    'molar mass of air in kg/mol used by Lemmon et al. 2000

Private Const Version = "27 May 2010"

'==========================================================================
Public Function dry_f_si(drv_t As Integer, _
                         drv_d As Integer, _
                         t_si As Double, _
                         d_si As Double) As Double
         
'This function implements the Helmholtz function of dry air and
'its 1st and 2nd derivatives with respect to temperature and density
'as published in:

'Lemmon, E.W., Jacobsen, R.T., Penoncello, S.G., Friend, D.G.
'Thermodynamic Properties of Air and Mixtures of Nitrogen, Argon and Oxygen
'From 60 to 2000 K at Pressures to 2000 MPa.
'J. Phys. Chem. Ref. Data 29(2000)331-362.

'Here, in contrast to the original article, the coefficients of absolute
'energy and absolute entropy are adjusted to the reference state condition
'of zero entropy and zero enthalpy at the standard ocean state, 0C and 101325 Pa.

'Input parameters:
'drv_t:     order of the temperature derivative, drv_t < 3
'drv_d:     order of the density derivative, drv_d + drv_t < 3
't_si:      absolute its-90 temperature in K
'd_si:      density in kg/m3

'Output:
'dry_f_si:  value of the Helmholtz function or its derivative in J/kg or its derivatives

'check values, v. 1.0:
'dry_f_si( 0, 0, 300,0.001) =-696384.49528633
'dry_f_si( 1, 0, 300,0.001) =-2124.98673213572
'dry_f_si( 0, 1, 300,0.001) = 86135114.6607569
'dry_f_si( 2, 0, 300,0.001) =-2.39299064929384
'dry_f_si( 1, 1, 300,0.001) = 287117.623821587
'dry_f_si( 0, 2, 300,0.001) =-86135137748.1857

'check values, v. 1.1:
'dry_f_si( 0, 0, 300,0.001) =-696239.965189878
'dry_f_si( 1, 0, 300,0.001) =-2124.55145455549
'dry_f_si( 0, 1, 300,0.001) = 86114714.9595876
'dry_f_si( 2, 0, 300,0.001) =-2.39242390805757
'dry_f_si( 1, 1, 300,0.001) = 287049.624545385
'dry_f_si( 0, 2, 300,0.001) =-86114738036.0821

Const ra = Gas_constant_air_L2000   'specific gas constant of air in J kg-1 K-1 used by Lemmon et al. 2000

Const tr = 132.6312                 'reducing temperature in K
Const dr = 10447.7 * ma             'reducing density in kg m-3

Dim f As Double, tau As Double, del As Double
Dim a As Double, a_t As Double, a_tt As Double
Dim a_d As Double, a_td As Double, a_dd As Double

dry_f_si = ErrorReturn

Init_IAPWS10

If check_limits = 1 Then
  If t_si < dry_air_tmin Or t_si > dry_air_tmax Then Exit Function
  If d_si <= dry_air_dmin Or d_si > dry_air_dmax Then Exit Function
Else
  If t_si <= 0 Then Exit Function
  If d_si <= 0 Then Exit Function
End If

tau = tr / t_si  'reduced temperature
del = d_si / dr  'reduced density

'Helmholtz function and its derivatives
Select Case drv_t
  Case 0:
    Select Case drv_d
      Case 0: a = Log(del) + alpha_ideal(tau) + alpha_res(0, 0, tau, del)
              f = ra * t_si * a
      Case 1: a_d = 1 / del + alpha_res(0, 1, tau, del)
              f = ra * t_si * a_d / dr
      Case 2: a_dd = -1 / del ^ 2 + alpha_res(0, 2, tau, del)
              f = ra * t_si * a_dd / dr ^ 2
      Case Else: Exit Function
    End Select
    
  Case 1:
    Select Case drv_d
      Case 0: a = Log(del) + alpha_ideal(tau) + alpha_res(0, 0, tau, del)
              a_t = alpha_ideal_t(tau) + alpha_res(1, 0, tau, del)
              f = ra * (a - tau * a_t)
      Case 1: a_d = 1 / del + alpha_res(0, 1, tau, del)
              a_td = alpha_res(1, 1, tau, del)
              f = ra * (a_d - tau * a_td) / dr
      Case Else: Exit Function
    End Select

  Case 2:
    Select Case drv_d
      Case 0: a_tt = alpha_ideal_tt(tau) + alpha_res(2, 0, tau, del)
              f = ra * tau ^ 2 * a_tt / t_si
      Case Else: Exit Function
    End Select
    
  Case Else: Exit Function
  
End Select

dry_f_si = f

End Function

'==========================================================================
Public Function air_baw_m3mol(drv_t As Integer, t_si As Double) As Double

'this function implements the 2nd virial coefficient Baw as defined in

'Harvey, A.H., Huang, P.H.: First-Principles Calculation of the Air-Water Second
'Virial Coefficient. Int. J. Thermophys. 28(2007)556-565.

'Output:  air_baw_m3mol = molar virial coefficient Baw in m3/(mol K^drv_t)

'Input:   drv_t order of the temperature derivative
'         t_si absolute temperature in K

Const c1 = 66.5687, d1 = -0.237
Const c2 = -238.834, d2 = -1.048
Const c3 = -176.755, d3 = -3.183
Const tr = 100 'K

Dim t As Double, b As Double

air_baw_m3mol = ErrorReturn

If t_si <= 0 Then Exit Function

t = t_si / tr

Select Case drv_t
  Case 0: b = c1 * t ^ d1 + c2 * t ^ d2 + c3 * t ^ d3
  Case 1: b = c1 * d1 * t ^ (d1 - 1) + c2 * d2 * t ^ (d2 - 1) + c3 * d3 * t ^ (d3 - 1)
          b = b / tr
  Case 2: b = c1 * d1 * (d1 - 1) * t ^ (d1 - 2) + _
              c2 * d2 * (d2 - 1) * t ^ (d2 - 2) + _
              c3 * d3 * (d3 - 1) * t ^ (d3 - 2)
          b = b / tr ^ 2
  Case Else: Exit Function
End Select

air_baw_m3mol = b * 0.000001 'from cm3/mol to m3/mol

End Function

'==========================================================================
Public Function air_caaw_m6mol2(drv_t As Integer, t_si As Double) As Double

'this function implements the 3rd virial coefficient Caaw as defined in

'Hyland, R.W., Wexler, A.:
'Formulations for the thermodynamic properties of dry air from 173.15 K to 473.15 K,
'and of saturated moist air from 173.15 K to 372.15 K, at pressures up to 5 MPa.
'ASHRAE Transact. 89(1983)520-535.

'Output:  air_caaw_m6mol2 = molar virial coefficient Caaw in m6/(mol2 K^drv_t)

'Input:   drv_t order of the temperature derivative
'         t_si absolute temperature in K

Const c1 = 482.737
Const c2 = 105678#
Const c3 = -65639400#
Const c4 = 29444200000#
Const c5 = -3193170000000#
Const tr = 1 'K

Dim t As Double, c As Double

air_caaw_m6mol2 = ErrorReturn

If t_si <= 0 Then Exit Function

t = t_si / tr

Select Case drv_t
  Case 0: c = c1 + c2 / t + c3 / t ^ 2 + c4 / t ^ 3 + c5 / t ^ 4
  Case 1: c = -c2 / t ^ 2 - 2 * c3 / t ^ 3 - 3 * c4 / t ^ 4 - 4 * c5 / t ^ 5
  Case 2: c = 2 * c2 / t ^ 3 + 6 * c3 / t ^ 4 + 12 * c4 / t ^ 5 + 20 * c5 / t ^ 6
  Case Else: Exit Function
End Select

air_caaw_m6mol2 = c * 0.000000000001 'from cm6/mol2 to m6/mol2

End Function

'==========================================================================
Public Function air_caww_m6mol2(drv_t As Integer, t_si As Double) As Double

'this function implements the 3rd virial coefficient Caww as defined in

'Hyland, R.W., Wexler, A.:
'Formulations for the thermodynamic properties of dry air from 173.15 K to 473.15 K,
'and of saturated moist air from 173.15 K to 372.15 K, at pressures up to 5 MPa.
'ASHRAE Transact. 89(1983)520-535.

'Output:  air_caww_m6mol2 = molar virial coefficient Caww in m6/(mol2 K^drv_t)

'Input:   drv_t order of the temperature derivative
'         t_si absolute temperature in K

Const c1 = -10.728876
Const c2 = 3478.02
Const c3 = -383383#
Const c4 = 33406000#
Const tr = 1 'K

Dim t As Double, c As Double

air_caww_m6mol2 = ErrorReturn

If t_si <= 0 Then Exit Function

t = t_si / tr

c = -1000000# * Exp(c1 + c2 / t + c3 / t ^ 2 + c4 / t ^ 3)
Select Case drv_t
  Case 0:
  Case 1: c = c * (-c2 / t ^ 2 - 2 * c3 / t ^ 3 - 3 * c4 / t ^ 4)
  Case 2: c = c * ((-c2 / t ^ 2 - 2 * c3 / t ^ 3 - 3 * c4 / t ^ 4) ^ 2 + _
                   2 * c2 / t ^ 3 + 6 * c3 / t ^ 4 + 12 * c4 / t ^ 5)
  Case Else: Exit Function
End Select

air_caww_m6mol2 = c * 0.000000000001 'from cm6/mol2 to m6/mol2

End Function

'==========================================================================
Private Sub Init_IAPWS10()

Dim k As Integer

If nk0(1) = 0.00000006057194 Then Exit Sub

'Arrays defined in Table 12 of Lemmon et al. (2000)
'Coefficients for the ideal gas expressions
'-------------------------------------------------------------------------------
k = 1:  nk0(k) = 0.00000006057194
k = 2:  nk0(k) = -0.0000210274769
k = 3:  nk0(k) = -0.000158860716
k = 4:  nk0(k) = 9.7450251743948   '-13.841928076 'original Lemmon et al. 2000
k = 5:  nk0(k) = 10.0986147428912  ' 17.275266575 'original Lemmon et al. 2000
k = 6:  nk0(k) = -0.00019536342
k = 7:  nk0(k) = 2.490888032
k = 8:  nk0(k) = 0.791309509
k = 9:  nk0(k) = 0.212236768
k = 10: nk0(k) = -0.197938904

k = 11: nk0(k) = 25.36365
k = 12: nk0(k) = 16.90741
k = 13: nk0(k) = 87.31279


'Arrays defined in Table 13 of Lemmon et al. (2000)
'Coefficients and exponents for the equation of state for air
'-------------------------------------------------------------------------------
k = 1:  ik(k) = 1:  jk(k) = 0:    lk(k) = 0: nk(k) = 0.118160747229
k = 2:  ik(k) = 1:  jk(k) = 0.33: lk(k) = 0: nk(k) = 0.713116392079
k = 3:  ik(k) = 1:  jk(k) = 1.01: lk(k) = 0: nk(k) = -1.61824192067
k = 4:  ik(k) = 2:  jk(k) = 0:    lk(k) = 0: nk(k) = 0.0714140178971
k = 5:  ik(k) = 3:  jk(k) = 0:    lk(k) = 0: nk(k) = -0.0865421396646
k = 6:  ik(k) = 3:  jk(k) = 0.15: lk(k) = 0: nk(k) = 0.134211176704
k = 7:  ik(k) = 4:  jk(k) = 0:    lk(k) = 0: nk(k) = 0.0112626704218
k = 8:  ik(k) = 4:  jk(k) = 0.2:  lk(k) = 0: nk(k) = -0.0420533228842
k = 9:  ik(k) = 4:  jk(k) = 0.35: lk(k) = 0: nk(k) = 0.0349008431982
k = 10: ik(k) = 6:  jk(k) = 1.35: lk(k) = 0: nk(k) = 0.000164957183186

k = 11: ik(k) = 1:  jk(k) = 1.6:  lk(k) = 1: nk(k) = -0.101365037912
k = 12: ik(k) = 3:  jk(k) = 0.8:  lk(k) = 1: nk(k) = -0.17381369097
k = 13: ik(k) = 5:  jk(k) = 0.95: lk(k) = 1: nk(k) = -0.0472103183731
k = 14: ik(k) = 6:  jk(k) = 1.25: lk(k) = 1: nk(k) = -0.0122523554253
k = 15: ik(k) = 1:  jk(k) = 3.6:  lk(k) = 2: nk(k) = -0.146629609713
k = 16: ik(k) = 3:  jk(k) = 6:    lk(k) = 2: nk(k) = -0.0316055879821
k = 17: ik(k) = 11: jk(k) = 3.25: lk(k) = 2: nk(k) = 0.000233594806142
k = 18: ik(k) = 1:  jk(k) = 3.5:  lk(k) = 3: nk(k) = 0.0148287891978
k = 19: ik(k) = 3:  jk(k) = 15:   lk(k) = 3: nk(k) = -0.00938782884667

'dynamic adjustment to the reference state conditions:
'vanishing entropy s = 0 and enthalpy h = f + Ts + pv = 0 at 273.15 K, 101325 Pa
'related density is 1.29275937468394 kg m-3, the reducing temperature 132.6312 K
'nk0(4) = 0
'nk0(4) = -dry_f_si(1, 0, 273.15, 1.29275937468394) * Molar_mass_air_L2000 / (Gas_constant_molar_L2000)
'Debug.Print nk0(4); -dry_f_si(1, 0, 273.15, 1.29275937468394)
'
'nk0(5) = 0
'nk0(5) = -(dry_f_si(0, 0, 273.15, 1.29275937468394) + 101325 / 1.29275937468394) * Molar_mass_air_L2000 / (132.6312 * Gas_constant_molar_L2000)
'Debug.Print nk0(5); dry_f_si(0, 0, 273.15, 1.29275937468394) + 101325 / 1.29275937468394
'
'output:
' 9.7450251743948 -1.05459846463211E-12
' 10.0986147428912  2.75164779850456E-10


End Sub

'==========================================================================
Public Sub dry_init_clear()

'clears the initialization of dry_f_si()

nk0(1) = 0

End Sub

'==========================================================================
Public Sub dry_init_Lemmon2000()

'sets original adjustable coefficients of Lemmon et al (2000)

Init_IAPWS10

nk0(4) = -13.841928076 'original Lemmon et al. 2000
nk0(5) = 17.275266575  'original Lemmon et al. 2000

End Sub

'==========================================================================
Private Function alpha_ideal(ByVal tau As Double) As Double

'temperature part of the ideal-gas dimensionless Helmholtz function of air

Dim a As Double, k As Integer

alpha_ideal = ErrorReturn

If tau <= 0 Then Exit Function

a = 0
For k = 1 To 5
  a = a + nk0(k) * tau ^ (k - 4)
Next k
a = a + nk0(6) * tau ^ 1.5 + nk0(7) * Log(tau)
a = a + nk0(8) * Log(1 - Exp(-nk0(11) * tau))
a = a + nk0(9) * Log(1 - Exp(-nk0(12) * tau))
a = a + nk0(10) * Log(2 / 3 + Exp(nk0(13) * tau))

alpha_ideal = a

End Function

'==========================================================================
Private Function alpha_ideal_t(ByVal tau As Double) As Double

'temperature derivative of the ideal-gas dimensionless Helmholtz function of air

Dim a As Double, k As Integer

alpha_ideal_t = ErrorReturn

If tau <= 0 Then Exit Function

a = 0
For k = 1 To 5
  a = a + nk0(k) * (k - 4) * tau ^ (k - 5)
Next k
a = a + 1.5 * nk0(6) * Sqr(tau) + nk0(7) / tau
a = a + nk0(8) * nk0(11) / (Exp(nk0(11) * tau) - 1)
a = a + nk0(9) * nk0(12) / (Exp(nk0(12) * tau) - 1)
a = a + nk0(10) * nk0(13) / (2 * Exp(-nk0(13) * tau) / 3 + 1)

alpha_ideal_t = a

End Function

'==========================================================================
Private Function alpha_ideal_tt(ByVal tau As Double) As Double

'2nd temperature derivative of the ideal-gas dimensionless Helmholtz function of air

Dim a As Double, k As Integer, e As Double

alpha_ideal_tt = ErrorReturn

If tau <= 0 Then Exit Function

a = 0
For k = 1 To 3
  a = a + nk0(k) * (k - 4) * (k - 5) * tau ^ (k - 6)
Next k
a = a + 0.75 * nk0(6) / Sqr(tau) - nk0(7) / tau ^ 2
e = Exp(nk0(11) * tau)
a = a - nk0(8) * nk0(11) ^ 2 * e / (e - 1) ^ 2
e = Exp(nk0(12) * tau)
a = a - nk0(9) * nk0(12) ^ 2 * e / (e - 1) ^ 2
e = Exp(-nk0(13) * tau)
a = a + nk0(10) * nk0(13) ^ 2 * e / (2 * e / 3 + 1) ^ 2 * 2 / 3
alpha_ideal_tt = a

End Function

'==========================================================================
Private Function alpha_res(ByVal drv_tau As Integer, _
                           ByVal drv_del As Integer, _
                           ByVal tau As Double, _
                           ByVal del As Double) As Double

'This implements equation (28) and its derivatives with respect to tau and del
'of
'Lemmon, E.W., Jacobsen, R.T., Penoncello, S.G., Friend, D.G.
'Thermodynamic Properties of Air and Mixtures of Nitrogen, Argon and Oxygen
'From 60 to 2000 K at Pressures to 2000 MPa.
'J. Phys. Chem. Ref. Data 29(2000)331-362.

Dim a As Double, a_tau As Double, a_del As Double, k As Integer
Dim i As Double, j As Double, L As Double

alpha_res = ErrorReturn

a = 0#

For k = 1 To 19

  i = ik(k)
  j = jk(k)
  L = lk(k)
  
  Select Case drv_tau
    Case 0: a_tau = tau ^ j
    Case 1: a_tau = j * tau ^ (j - 1)
    Case 2: a_tau = j * (j - 1) * tau ^ (j - 2)
    Case Else: Exit Function
  End Select
  
  If k < 11 Then
  
    Select Case drv_del
      Case 0: a_del = del ^ i
      Case 1: a_del = i * del ^ (i - 1)
      Case 2: a_del = i * (i - 1) * del ^ (i - 2)
      Case Else: Exit Function
    End Select
  
  Else
  
    Select Case drv_del
      Case 0: a_del = del ^ i * Exp(-del ^ L)
      Case 1: a_del = (i - L * del ^ L) * del ^ (i - 1) * Exp(-del ^ L)
      Case 2: a_del = (i * (i - 1) - (L + 2 * i - 1) * L * del ^ L _
                    + (L * del ^ L) ^ 2) * del ^ (i - 2) * Exp(-del ^ L)
      Case Else: Exit Function
    End Select

  End If
  a = a + nk(k) * a_tau * a_del

Next k

alpha_res = a

End Function



