Attribute VB_Name = "Sea_Air_4_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules:
'     Constants_0_Mdl, file Constants_0.bas
'     Convert_0_Mdl,   file Convert_0.bas
'     Air_1_Mdl,       file Air_1.bas
'     Flu_1_Mdl,       file Flu_1.bas
'     Sal_1_Mdl,       file Sal_1.bas
'     Air_2_Mdl,       file Air_2.bas
'     Flu_2_Mdl,       file Flu_2.bas
'     Sal_2_Mdl,       file Sal_2.bas
'     Air_3a_Mdl,      file Air_3a.bas
'     Air_3b_Mdl,      file Air_3b.bas
'     Flu_3a_Mdl,      file Flu_3a.bas
'     Sea_3a_Mdl,      file Sea_3a.bas
'     Liq_Air_4a_Mdl,  file Liq_Air_4a.bas
'     Liq_Vap_4_Mdl,   file Liq_Vap_4.bas
'     Maths_0_Mdl,     file Maths_0.bas
'
'#########################################################################

'=========================================================================
'This module implements the phase equilibrium properties of seawater with humid air

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009

'Feistel, R., Kretzschmar, H.-J., Span, R., Hagen, E., Wright, D.G., Herrmann, S.:
'Thermodynamic Properties of Sea Air.
'Ocean Science Discussion 6(2009)21932325.
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98
'Private Const IsOK = -1

'Control parameters of the vapour pressure iteration
Private ctrl_initialized As Integer

'select initial guessing method
Private ctrl_mode_humidity As Integer
Private ctrl_mode_temperature As Integer
Private ctrl_mode_pressure As Integer

'store customized initial values here
Private ctrl_loop_maximum As Long
Private ctrl_init_t As Double
Private ctrl_init_a As Double
Private ctrl_init_p As Double

'set tolerance of the iteration loop for
Private ctrl_eps_exit_p_vap As Double   'vapour partial pressure
Private ctrl_eps_exit_temp As Double    'temperature
Private ctrl_eps_exit_press As Double    'pressure

'Properties of the current liquid-vapour equilibrium state
Private equi_sea_air_done As Integer

'storage of the iteration results = the equilibrium state (saturated air)
Private equi_sea_air_s As Double        'salinity
Private equi_sea_air_a As Double        'dry air mass fraction of humid air
Private equi_sea_air_t As Double        'temperature
Private equi_sea_air_p As Double        'pressure
Private equi_sea_air_d_liq As Double    'liquid water density
Private equi_sea_air_d_air As Double    'humid air density

Private Const Version = "28 May 2010"

'==========================================================================
Public Function sea_air_massfraction_air_si(ByVal sa_si As Double, _
                                            ByVal t_si As Double, _
                                            ByVal p_si As Double) As Double

'returns the air mass fraction in kg/kg of sea air

'input: sa_si  absolute salinity in kg/kg
'        t_si  absolute temperature in K
'        p_si  absolute pressure of humid air in Pa
'output: a_si  mass fraction of dry air in sea air, in kg/kg

'check value with default settings:
'sea_air_massfraction_air_si(0.035, 300, 1E5) = 0.97802440543935  v. 1.0
'sea_air_massfraction_air_si(0.035, 300, 1E5) = 0.978029483888009  v. 1.1

sea_air_massfraction_air_si = ErrorReturn

If set_sea_air_eq_at_s_t_p(sa_si, t_si, p_si) = ErrorReturn Then Exit Function

sea_air_massfraction_air_si = equi_sea_air_a

End Function

'==========================================================================
Public Function sea_air_vapourpressure_si(ByVal sa_si As Double, _
                                          ByVal t_si As Double, _
                                          ByVal p_si As Double) As Double

'returns the partial pressure of vapour in humid air at equilibrium with seawater,
'from absolute salinity, temperature and total pressure

'input:  sa_si absolute salinity in kg/kg
'        t_si  absolute temperature in K
'        p_si  absolute (total) pressure in Pa
'output: vapour pressure in Pa

'check value with default settings:
'sea_air_vapourpressure_si(0.035,300,1E5) = 3485.92802425416  v. 1.0
'sea_air_vapourpressure_si(0.035,300,1E5) = 3485.92986681407  v. 1.1

Dim a_sat As Double, xv As Double

sea_air_vapourpressure_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

'get the air fraction at equilibrium with seawater
a_sat = sea_air_massfraction_air_si(sa_si, t_si, p_si)
If a_sat = ErrorReturn Then Exit Function
If a_sat <= 0 Or a_sat > 1 Then Exit Function

'convert air mass fraction to vapour mole fraction
xv = air_molfraction_vap_si(a_sat)
If xv = ErrorReturn Then Exit Function
If xv <= 0 Or xv > 1 Then Exit Function

'compute the partial pressure
sea_air_vapourpressure_si = xv * p_si

End Function

'==========================================================================
Public Function sea_air_chempot_evap_si(ByVal a_si As Double, _
                                        ByVal sa_si As Double, _
                                        ByVal t_si As Double, _
                                        ByVal p_si As Double) As Double

'this function returns the chemical potential difference x_sa = ln(phi(a_si)/phi(sa_si))
'computed from the relative fugacity of humid air, phi(A), and of sea air, phi(SA)
'as a measure for the deviation from air-sea equilibrium (Onsager force)

'input:  a_si:  mass fraction of dry air in humid air, in kg/kg
'        sa_si: absolute salinity of seawater
'        t_si:  absolute temperature in K
'        p_si:  absolute pressure in Pa
'output: chemical potential difference /(R_W T), unitless

'check value with default settings:
'sea_air_chempot_evap_si(0.9,0.035,300,1E5) = 1.45564492466977  v. 1.0
'sea_air_chempot_evap_si(0.9,0.035,300,1E5) = 1.45584069070883  v. 1.1

Dim mu_v As Double, mu_w As Double

sea_air_chempot_evap_si = ErrorReturn

If a_si < 0 Or a_si > 1 Then Exit Function
If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

'chemical potential of vapour in humid air
mu_v = air_g_chempot_vap_si(a_si, t_si, p_si)
If mu_v = ErrorReturn Then Exit Function

'chemical potential of water in seawater
mu_w = sea_chempot_h2o_si(sa_si, t_si, p_si)
If mu_w = ErrorReturn Then Exit Function

sea_air_chempot_evap_si = (mu_v - mu_w) / (Gas_constant_H2O_si * t_si)

End Function

'==========================================================================
Public Function sea_air_entropy_air_si(ByVal sa_si As Double, _
                                       ByVal t_si As Double, _
                                       ByVal p_si As Double) As Double

'returns the humid-air entropy in J/(kg K) of sea air

'input: sa_si  absolute salinity in kg/kg of the seawater fraction
'        t_si  absolute temperature in K
'        p_si  absolute pressure of humid air in Pa
'output: eta_si  entropy of humid air, in J/(kg K)

'check value with default settings:
'sea_air_entropy_air_si(0.035, 300, 1E5) = 293.218508134414  v. 1.0
'sea_air_entropy_air_si(0.035, 300, 1E5) = 293.15067295733  v. 1.1

sea_air_entropy_air_si = ErrorReturn

If set_sea_air_eq_at_s_t_p(sa_si, t_si, p_si) = ErrorReturn Then Exit Function

sea_air_entropy_air_si = air_g_entropy_si(equi_sea_air_a, t_si, p_si)

End Function

'==========================================================================
Public Function sea_air_condense_temp_si(ByVal sa_si As Double, _
                                         ByVal a_si As Double, _
                                         ByVal p_si As Double) As Double

'returns the seawater condensation temperature of unsaturated humid air

'input: sa_si  absolute salinity in kg/kg
'        a_si  mass fraction of dry air in humid air, in kg/kg
'        p_si  absolute pressure of humid air in Pa
'output: t_si  absolute condensation temperature in K

'check value with default settings:
'sea_air_condense_temp_si(0.035, 0.99, 1E5) = 287.363869940874  v. 1.0
'sea_air_condense_temp_si(0.035, 0.99, 1E5) = 287.367456467663  v. 1.1

sea_air_condense_temp_si = ErrorReturn

If sa_si = 0 Then
  sea_air_condense_temp_si = liq_air_dewpoint_si(a_si, p_si)
  Exit Function
End If

If set_sea_air_eq_at_s_a_p(sa_si, a_si, p_si) = ErrorReturn Then Exit Function

sea_air_condense_temp_si = equi_sea_air_t

End Function

'==========================================================================
Public Function sea_air_enthalpy_evap_si()

'returns the latent heat in J/kg of sea air
'after setting the equilibrium state by calling set_sea_air_eq_at_s_t_p etc

'check values with default settings:  v. 1.0
'set_sea_air_eq_at_s_a_p 0.035, 0.99, 1E5
'sea_air_enthalpy_evap_si = 2464747.68964351
'
'set_sea_air_eq_at_s_t_p 0.035, 300, 1E5
'sea_air_enthalpy_evap_si = 2434550.22477297

'check values with default settings:  v. 1.1
'set_sea_air_eq_at_s_a_p 0.035, 0.99, 1E5
'sea_air_enthalpy_evap_si = 2464738.79136465
'
'set_sea_air_eq_at_s_t_p 0.035, 300, 1E5
'sea_air_enthalpy_evap_si = 2434549.89769804

Dim a As Double, t As Double, dh As Double, dl As Double
Dim hh As Double, hh_a As Double, hl As Double
Dim fh_a As Double, fh_d As Double
Dim fh_at As Double, fh_ad As Double
Dim fh_td As Double, fh_dd As Double
Dim p_a As Double, p_d As Double
Dim h_a As Double, h_d As Double
Dim S As Double, p As Double, hs As Double, hs_s As Double
Dim gs As Double, gs_s As Double, gs_t As Double, gs_st As Double

sea_air_enthalpy_evap_si = ErrorReturn

If equi_sea_air_done <> IsOK Then Exit Function

'get the equilibrium properties
S = equi_sea_air_s
If S = ErrorReturn Then Exit Function
If S < 0 Or S >= 1 Then Exit Function
a = equi_sea_air_a
If a = ErrorReturn Then Exit Function
If a < 0 Or a > 1 Then Exit Function
t = equi_sea_air_t
If t = ErrorReturn Then Exit Function
If t <= 0 Then Exit Function
p = equi_sea_air_p
If p = ErrorReturn Then Exit Function
If p <= 0 Then Exit Function
dh = equi_sea_air_d_air
If dh = ErrorReturn Then Exit Function
If dh <= 0 Then Exit Function
dl = equi_sea_air_d_liq
If dl = ErrorReturn Then Exit Function
If dl <= 0 Then Exit Function

'enthalpy of humid air
hh = air_f_enthalpy_si(a, t, dh)
If hh = ErrorReturn Then Exit Function

'derivative hh_a = (dh/da)_t_p of enthalpy of humid air
fh_a = air_f_si(1, 0, 0, a, t, dh)
If fh_a = ErrorReturn Then Exit Function
fh_d = air_f_si(0, 0, 1, a, t, dh)
If fh_d = ErrorReturn Then Exit Function
fh_at = air_f_si(1, 1, 0, a, t, dh)
If fh_at = ErrorReturn Then Exit Function
fh_ad = air_f_si(1, 0, 1, a, t, dh)
If fh_ad = ErrorReturn Then Exit Function
fh_td = air_f_si(0, 1, 1, a, t, dh)
If fh_td = ErrorReturn Then Exit Function
fh_dd = air_f_si(0, 0, 2, a, t, dh)
If fh_dd = ErrorReturn Then Exit Function

p_a = dh * fh_ad                            'dp/da at const t,d
p_d = dh * (2 * fh_d + dh * fh_dd)          'dp/dd at const a,t
If p_d = 0 Then Exit Function
h_a = fh_a + dh * fh_ad - t * fh_at         'dh/da at const t,d
h_d = 2 * fh_d + dh * fh_dd - t * fh_td     'dh/dd at const a,t

hh_a = h_a - h_d * p_a / p_d                'dh/da at const t,p

'enthalpy of liquid water
hl = flu_enthalpy_si(t, dl)
If hl = ErrorReturn Then Exit Function

If S = 0 Then 'treat the pure-water case separately
  'compute the latent heat of water
  sea_air_enthalpy_evap_si = hh - a * hh_a - hl
  Exit Function
End If

'saline enthalpy
gs = sal_g_si(0, 0, 0, S, t, p)
If gs = ErrorReturn Then Exit Function
gs_t = sal_g_si(0, 1, 0, S, t, p)
If gs_t = ErrorReturn Then Exit Function
hs = gs - t * gs_t
gs_s = sal_g_si(1, 0, 0, S, t, p)
If gs_s = ErrorReturn Then Exit Function
gs_st = sal_g_si(1, 1, 0, S, t, p)
If gs_st = ErrorReturn Then Exit Function
hs_s = gs_s - t * gs_st

'compute the latent heat of seawater
sea_air_enthalpy_evap_si = hh - a * hh_a - hl - hs + S * hs_s

End Function

'==========================================================================
Public Function sea_air_density_air_si()

'returns the humid air density in kg/m3 of sea air
'after setting the equilibrium state by calling set_sea_air_eq_at_s_t_p etc

'check values with default settings:  v. 1.0
'set_sea_air_eq_at_s_a_p 0.035, 0.99, 1E5
'sea_air_density_air_si = 1.20527399511464
'
'set_sea_air_eq_at_s_t_p 0.035, 300, 1E5
'sea_air_density_air_si = 1.14616388271098

'check values with default settings:  v. 1.1
'set_sea_air_eq_at_s_a_p 0.035, 0.99, 1E5
'sea_air_density_air_si = 1.20553988597722
'
'set_sea_air_eq_at_s_t_p 0.035, 300, 1E5
'sea_air_density_air_si = 1.14642944447943

sea_air_density_air_si = ErrorReturn

If equi_sea_air_done <> IsOK Then Exit Function

sea_air_density_air_si = equi_sea_air_d_air

End Function

'==========================================================================
Public Function sea_air_density_vap_si()

'returns the vapour density in kg/m3 of sea air
'after setting the equilibrium state by calling set_sea_air_eq_at_s_t_p etc

'check values with default settings:  v. 1.0
'set_sea_air_eq_at_s_a_p 0.035, 0.99, 1E5
'sea_air_density_vap_si = 1.20527399511464E-02
'
'set_sea_air_eq_at_s_t_p 0.035, 300, 1E5
'sea_air_density_vap_si = 0.025187632786517

'check values with default settings:  v. 1.1
'set_sea_air_eq_at_s_a_p 0.035, 0.99, 1E5
'sea_air_density_vap_si = 1.20553988597722E-02
'
'set_sea_air_eq_at_s_t_p 0.035, 300, 1E5
'sea_air_density_vap_si = 2.51876465811967E-02

sea_air_density_vap_si = ErrorReturn

If equi_sea_air_done <> IsOK Then Exit Function

sea_air_density_vap_si = equi_sea_air_d_air * (1 - equi_sea_air_a)

End Function

'==========================================================================
Public Function set_sea_air_eq_at_s_t_p(ByVal sa_si As Double, _
                                                 ByVal t_si As Double, _
                                                 ByVal p_si As Double) As Double
                                             
'this function computes the equilibrium state between humid air and
'seawater, in particular, the specific humidity of sea air.

'input: sa_si  absolute salinity in kg/kg
'        t_si  absolute temperature in K
'        p_si  absolute pressure of humid air in Pa

Dim a_si As Double, dl As Double, dh As Double

Dim maxit As Long, eps As Double

If equi_sea_air_done = IsOK And _
   sa_si = equi_sea_air_s And _
   t_si = equi_sea_air_t And _
   p_si = equi_sea_air_p Then
  'the requested state has already been computed earlier
  set_sea_air_eq_at_s_t_p = IsOK
  Exit Function
End If

clear_sea_air_state 'waste any previous state

set_sea_air_eq_at_s_t_p = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

init_it_ctrl_sea_air

'set initial air fraction guess: use pure water
Select Case ctrl_mode_humidity
  Case 0:  a_si = aux_airfraction_si(t_si, p_si)
           If a_si = ErrorReturn Then Exit Function
           If a_si < 0 Or a_si > 1 Then Exit Function
  Case 1:  a_si = ctrl_init_a
  Case Else: Exit Function
End Select
If a_si = ErrorReturn Then Exit Function
If a_si < 0 Or a_si > 1 Then Exit Function

'liquid density can be computed from t, p
dl = liq_density_si(t_si, p_si)
If dl = ErrorReturn Then Exit Function
If dl <= 0 Then Exit Function

dh = air_density_si(a_si, t_si, p_si)
If dh = ErrorReturn Then Exit Function
If dh <= 0 Then Exit Function

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: set_sea_air_eq_at_s_t_p = IsOK
           set_sea_air_state sa_si, a_si, t_si, p_si, dl, dh
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit for the vapour partial pressure
eps = ctrl_eps_exit_p_vap
If eps = 0 Then Exit Function

'run iteration loop
If sea_air_iteration_at_s_t_p(sa_si, t_si, p_si, dl, maxit, eps, a_si, dh) = ErrorReturn Then
  Exit Function
End If

set_sea_air_state sa_si, a_si, t_si, p_si, dl, dh

set_sea_air_eq_at_s_t_p = IsOK

End Function

'==========================================================================
Public Function set_sea_air_eq_at_s_a_p(ByVal sa_si As Double, _
                                                 ByVal a_si As Double, _
                                                 ByVal p_si As Double) As Double

                                             
'this function computes the equilibrium state between humid air and seawater,
'in particular, the sea-air temperature at which humid air condensates
'at the sea surface (already before the dewpoint is reached)

'input: sa_si  absolute salinity in kg/kg
'        a_si  mass fraction of dry air in humid air in kg/kg
'        p_si  absolute pressure of humid air in Pa

Dim t_si As Double, dl As Double, dh As Double

Dim maxit As Long, eps As Double

If equi_sea_air_done = IsOK And _
   sa_si = equi_sea_air_s And _
   a_si = equi_sea_air_a And _
   p_si = equi_sea_air_p Then
  'the requested state has already been computed earlier
  set_sea_air_eq_at_s_a_p = IsOK
  Exit Function
End If

clear_sea_air_state 'waste any previous state

set_sea_air_eq_at_s_a_p = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If a_si < 0 Or a_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function

init_it_ctrl_sea_air

'set initial temperature guess
Select Case ctrl_mode_temperature
  Case 0:  t_si = aux_temperature_si(a_si, p_si)
  Case 1:  t_si = ctrl_init_t
  Case Else: Exit Function
End Select
If t_si = ErrorReturn Then Exit Function
If t_si <= 0 Then Exit Function

'set initial liquid density
dl = liq_density_si(t_si, p_si)
If dl = ErrorReturn Then Exit Function
If dl <= 0 Then Exit Function

'set initial humid air density
dh = air_density_si(a_si, t_si, p_si)
If dh = ErrorReturn Then Exit Function
If dh <= 0 Then Exit Function

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: set_sea_air_eq_at_s_a_p = IsOK
           set_sea_air_state sa_si, a_si, t_si, p_si, dl, dh
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit for the vapour partial pressure
eps = ctrl_eps_exit_p_vap
If eps = 0 Then Exit Function

'run iteration loop
If sea_air_iteration_at_s_a_p(sa_si, a_si, p_si, maxit, eps, t_si, dl, dh) = ErrorReturn Then
  Exit Function
End If

set_sea_air_state sa_si, a_si, t_si, p_si, dl, dh

set_sea_air_eq_at_s_a_p = IsOK

End Function

'==========================================================================
Private Sub set_sea_air_state(ByVal S As Double, _
                              ByVal a As Double, _
                              ByVal t As Double, _
                              ByVal p As Double, _
                              ByVal dl As Double, _
                              ByVal dh As Double)

'stores the actual properties as the current equilibrium state descriptor
equi_sea_air_done = IsOK

equi_sea_air_s = S
equi_sea_air_a = a
equi_sea_air_t = t
equi_sea_air_p = p
equi_sea_air_d_liq = dl
equi_sea_air_d_air = dh

End Sub

'==========================================================================
Private Sub clear_sea_air_state()

'clears the current equilibrium state descriptor

equi_sea_air_done = 0

End Sub

'==========================================================================
Private Sub init_it_ctrl_sea_air()

'triple point values
Const t_triple = TP_temperature_si
Const p_triple = TP_pressure_IAPWS95_si

If ctrl_initialized = IsOK Then Exit Sub

ctrl_initialized = IsOK

'Set default values and modes for iteration
ctrl_mode_pressure = 0
ctrl_mode_temperature = 0
ctrl_mode_humidity = 0

ctrl_loop_maximum = 100
ctrl_init_t = t_triple
ctrl_init_p = p_triple
ctrl_init_a = 1 - 0.001

ctrl_eps_exit_p_vap = -0.0000001 'relative, 0.1 ppm
ctrl_eps_exit_temp = -0.0000001 'relative, 0.1 ppm
ctrl_eps_exit_press = -0.0000001 'relative, 0.1 ppm

End Sub

'==========================================================================
Public Sub set_it_ctrl_sea_air(ByVal key As String, ByVal value As Double)

'this sub sets control parameters for the iteration used to compute
'saturated humid air properties

'key              value
'it_steps         0           set iteration number to default (100)
'it_steps         n > 0       set iteration number to n
'it_steps        -1           do not iterate, use initial values as the saturation state

'init_air         0           use default air fraction to start ( = aux_airfraction_si(t, p))
'init_air         a > 0       use value a as air fraction to start

'init_temp        0           use default temperature to start ( = aux_temperature_si(a, p))
'init_temp        t > 0       use value t as temperature to start

'init_press       0           use default pressure to start ( = aux_pressure_si(a, t))
'init_press       p > 0       use value p as pressure to start

'tol_vap_press    0           use default exit accuracy for vapour pressure (0.1 ppm)
'tol_vap_press    eps         use eps as exit accuracy for vapour pressure (eps < 0 means relative error)

'tol_temp         0           use default exit accuracy for temperature (0.1 ppm)
'tol_temp         eps         use eps as exit accuracy for temperature (eps < 0 means relative error)

'tol_press        0           use default exit accuracy for pressure (0.1 ppm)
'tol_press        eps         use eps as exit accuracy for pressure (eps < 0 means relative error)


init_it_ctrl_sea_air

clear_sea_air_state

Select Case LCase(Trim(key))

  Case "it_steps":   'iteration steps
    Select Case value
      Case 0:      ctrl_loop_maximum = 100  'default = 100
      Case Is < 0: ctrl_loop_maximum = -1
      Case Else:   ctrl_loop_maximum = value
    End Select

  Case "init_air":   'start air fraction
    Select Case CLng(value)
      Case 0:       ctrl_mode_humidity = 0    'default = aux polynomial
      Case Else:    ctrl_mode_humidity = 1
                    ctrl_init_a = value
    End Select
    
  Case "init_temp":   'start temperature
    Select Case CLng(value)
      Case 0:       ctrl_mode_temperature = 0    'default = aux polynomial
      Case Else:    ctrl_mode_temperature = 1
                    ctrl_init_t = value
    End Select
    
  Case "init_press":   'start pressure
    Select Case CLng(value)
      Case 0:       ctrl_mode_pressure = 0    'default = aux polynomial
      Case Else:    ctrl_mode_pressure = 1
                    ctrl_init_p = value
    End Select

  Case "tol_vap_press":      'required vapour pressure tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_p_vap = -0.0000001   'default = 0.1 ppm relative
      Case Else:   ctrl_eps_exit_p_vap = value
    End Select

  Case "tol_temp":      'required temperature tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_temp = -0.0000001   'default = 0.1 ppm relative
      Case Else:   ctrl_eps_exit_temp = value
    End Select

  Case "tol_press":      'required total pressure tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_press = -0.0000001   'default = 0.1 ppm relative
      Case Else:   ctrl_eps_exit_press = value
    End Select

End Select

End Sub

'==========================================================================
Function sea_air_iteration_at_s_t_p(ByVal sa_si, _
                                    ByVal t_si, _
                                    ByVal p_si, _
                                    ByVal d_liq_si, _
                                    ByVal maxit, _
                                    ByVal eps, _
                                    ByRef a_si, _
                                    ByRef d_air_si) As Double

'this function returns the seawater-humid-air phase equilibrium from equal pressures,
'temperatures and chemical potential of water of the two phases at given salinity, sa_si,
'pressure, p_si, and  temperature, t_si, from initial guesses for the humid-air density,
'd_air_si, and the air fraction, a_si

'The iteration limit eps refers to the error in vapour partial pressure

'output:    sea_air_iteration_at_s_t_p = IsOK if successful
'           sea_air_iteration_at_s_t_p = ErrorReturn is returned if
'           - the maximum number of iterations is exceeded without meeting the exit criterion
'           - function calls to sal_g_si, flu_f_si or air_f_si have returned an error
'           - density or air fraction have taken an invalid value during the iteration
'     a_si: mass fraction of dry air in humid air, kg/kg
' d_air_si: humid-air density at the equilibrium

'input: sa_si: absolute salinity in kg/kg
'        p_si: absolute pressure in Pa
'        t_si: absolute temperature in K
'    d_liq_si: liquid pure-water density at t_si, p_si
'       maxit: maximum number of iteration steps to be done
'         eps: required accuracy of vapour pressure
'              eps > 0: absolute vapour pressure uncertainty in Pa
'              eps < 0: relative vapour pressure uncertainty
'        a_si: initial guess of the mass fraction of dry air in humid air, kg/kg
'    d_air_si: initial guess of humid-air density

Dim ah As Double, fh_a As Double, fh_aa As Double, fh_ad As Double
Dim gl As Double, gss As Double
Dim fh As Double, fh_d As Double, fh_dd As Double
Dim dl As Double, dh As Double
Dim pv As Double, pv_old As Double

Dim it As Long

Dim a(2, 2) As Double, b(2) As Double, x(2) As Double

sea_air_iteration_at_s_t_p = ErrorReturn

'check for invalid input values
If a_si < 0 Or a_si > 1 Or _
   d_air_si <= 0 Then
  a_si = ErrorReturn
  d_air_si = ErrorReturn
  Exit Function
End If

If check_limits <> 1 Then
  If d_liq_si <= 0 Or _
     sa_si < 0 Or sa_si >= 1 Or _
     t_si <= 0 Or _
     p_si <= 0 Then
    a_si = ErrorReturn
    d_air_si = ErrorReturn
    Exit Function
  End If
Else
  'AIR_LIMITS
  If t_si < dry_air_tmin Or t_si > dry_air_tmax Then
    a_si = ErrorReturn
    d_air_si = ErrorReturn
    Exit Function
  End If
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Or _
     d_liq_si <= flu_dmin Or d_liq_si > flu_dmax Then
    a_si = ErrorReturn
    d_air_si = ErrorReturn
    Exit Function
  End If
  'SAL_LIMITS
  If t_si < sal_tmin Or t_si > sal_tmax Or _
     sa_si < sal_smin Or sa_si > sal_smax Or _
     p_si < sal_pmin Or p_si > sal_pmax Then
    a_si = ErrorReturn
    d_air_si = ErrorReturn
    Exit Function
  End If
End If

check_limits = check_limits - 1

ah = a_si
dh = d_air_si

'chemical potential of water
gl = flu_gibbs_energy_si(t_si, d_liq_si)
If gl = ErrorReturn Then
  check_limits = check_limits + 1
  Exit Function
End If
gss = sal_chempot_h2o_si(sa_si, t_si, p_si)
If gss = ErrorReturn Then
  check_limits = check_limits + 1
  Exit Function
End If

For it = 0 To maxit

  'Derivatives of the Helmholtz function for 2D-Newton iteration
  'humid air:
  fh = air_f_si(0, 0, 0, ah, t_si, dh)
  If fh = ErrorReturn Then Exit For
  fh_a = air_f_si(1, 0, 0, ah, t_si, dh)
  If fh_a = ErrorReturn Then Exit For
  fh_d = air_f_si(0, 0, 1, ah, t_si, dh)
  If fh_d = ErrorReturn Then Exit For
  fh_aa = air_f_si(2, 0, 0, ah, t_si, dh)
  If fh_aa = ErrorReturn Then Exit For
  fh_ad = air_f_si(1, 0, 1, ah, t_si, dh)
  If fh_ad = ErrorReturn Then Exit For
  fh_dd = air_f_si(0, 0, 2, ah, t_si, dh)
  If fh_dd = ErrorReturn Then Exit For
  
  'vapour partial pressure for error checking
  pv_old = pv
  pv = air_molfraction_vap_si(ah)
  If pv = ErrorReturn Then Exit For
  pv = p_si * pv
  
  If it > 0 Then
    'check absolute or relative error limit
    If (eps > 0 And Abs(pv - pv_old) < eps) Or _
       (eps < 0 And Abs(pv - pv_old) < -eps * pv) Then
      sea_air_iteration_at_s_t_p = IsOK
      d_air_si = dh
      a_si = ah
      Exit For
    End If
  End If
  
  If it = maxit Then Exit For
  
  'coefficient matrix
  a(1, 1) = -ah * fh_aa
  a(1, 2) = fh_d - ah * fh_ad - p_si / dh ^ 2
  a(2, 1) = dh * fh_ad
  a(2, 2) = 2 * fh_d + dh * fh_dd

  'right-hand sides, must vanish at equilibrium
  b(1) = gl - fh - p_si / dh + ah * fh_a + gss
  b(2) = p_si / dh - dh * fh_d
  
  'solve equations
  If matrix_solve(a(), b(), x(), 2) <> 0 Then Exit For 'matrix singular

  'update air fraction & density
  ah = ah + x(1)
  If ah < 0 Or ah > 1 Then Exit For
  dh = dh + x(2)
  If dh <= 0 Then Exit For

Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'AIR_LIMITS
  If a_si < 0 Or a_si > 1 Or _
     d_air_si <= dry_air_dmin Or d_air_si > dry_air_dmax Then
    a_si = ErrorReturn
    d_air_si = ErrorReturn
    sea_air_iteration_at_s_t_p = ErrorReturn
  End If
End If

End Function

'==========================================================================
Function sea_air_iteration_at_s_a_p(ByVal sa_si, _
                                    ByVal a_si, _
                                    ByVal p_si, _
                                    ByVal maxit, _
                                    ByVal eps, _
                                    ByRef t_si, _
                                    ByRef d_liq_si, _
                                    ByRef d_air_si) As Double

'this function returns the seawater-humid-air phase equilibrium from equal pressures,
'temperatures and chemical potential of water of the two phases at given pressure, p_si,
'salinity, sa_si, and air fraction, a_si, from initial guesses for the liquid density,
'd_liq_si, the humid-air density, d_air_si, and the temperature, t_si

'The iteration limit eps refers to the error in the temperature

'output:    sea_air_iteration_at_s_a_p = IsOK if successful
'           sea_air_iteration_at_s_a_p = ErrorReturn is returned if
'           - the maximum number of iterations is exceeded without meeting the exit criterion
'           - function calls to sal_g_si, flu_f_si or air_f_si have returned an error
'           - densities or temperature have taken a zero or negative value during the iteration
'     t_si: absolute temperature in K
' d_liq_si: liquid density at t_si, p_si
' d_air_si: humid-air density at the saturation equilibrium

'input: sa_si: absolute salinity in kg/kg
'        p_si: absolute pressure in Pa
'        a_si: mass fraction of dry air in humid air, kg/kg
'       maxit: maximum number of iteration steps to be done
'         eps: required accuracy of temperature
'              eps > 0: absolute temperature uncertainty in K
'              eps < 0: relative temperature uncertainty
'        t_si: initial guess of absolute temperature in K
'    d_liq_si: initial guess of liquid density at p_si, t_si
'    d_air_si: initial guess of humid-air density


Dim fh As Double, fh_a As Double, fh_t As Double, fh_d As Double
Dim fh_ad As Double, fh_at As Double, fh_td As Double, fh_dd As Double
Dim fl As Double, fl_d As Double, fl_t As Double, fl_td As Double, fl_dd As Double
Dim dl As Double, dh As Double, t As Double
Dim gss As Double, gs_t As Double, gs_st As Double

Dim it As Long

Dim a(3, 3) As Double, b(3) As Double, x(3) As Double

sea_air_iteration_at_s_a_p = ErrorReturn

'check for invalid input values
If d_air_si <= 0 Or _
   d_liq_si <= 0 Or _
   a_si < 0 Or a_si >= 1 Or _
   t_si <= 0 Then
  t_si = ErrorReturn
  d_liq_si = ErrorReturn
  d_air_si = ErrorReturn
  Exit Function
End If

If check_limits <> 1 Then
  If sa_si < 0 Or sa_si >= 1 Or _
     p_si <= 0 Then
    t_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_air_si = ErrorReturn
    Exit Function
  End If
Else
  'SAL_LIMITS
  If sa_si < sal_smin Or sa_si > sal_smax Or _
     p_si < sal_pmin Or p_si > sal_pmax Then
    t_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_air_si = ErrorReturn
    Exit Function
  End If
End If

check_limits = check_limits - 1

t = t_si
dh = d_air_si
dl = d_liq_si

For it = 1 To maxit

  'Derivatives of the Helmholtz function for 3D-Newton iteration
  'humid air:
  fh = air_f_si(0, 0, 0, a_si, t, dh)
  If fh = ErrorReturn Then Exit For
  fh_a = air_f_si(1, 0, 0, a_si, t, dh)
  If fh_a = ErrorReturn Then Exit For
  fh_t = air_f_si(0, 1, 0, a_si, t, dh)
  If fh_t = ErrorReturn Then Exit For
  fh_d = air_f_si(0, 0, 1, a_si, t, dh)
  If fh_d = ErrorReturn Then Exit For
  fh_ad = air_f_si(1, 0, 1, a_si, t, dh)
  If fh_ad = ErrorReturn Then Exit For
  fh_at = air_f_si(1, 1, 0, a_si, t, dh)
  If fh_at = ErrorReturn Then Exit For
  fh_td = air_f_si(0, 1, 1, a_si, t, dh)
  If fh_td = ErrorReturn Then Exit For
  fh_dd = air_f_si(0, 0, 2, a_si, t, dh)
  If fh_dd = ErrorReturn Then Exit For

  'liquid water:
  fl = flu_f_si(0, 0, t, dl)
  If fl = ErrorReturn Then Exit For
  fl_t = flu_f_si(1, 0, t, dl)
  If fl_t = ErrorReturn Then Exit For
  fl_d = flu_f_si(0, 1, t, dl)
  If fl_d = ErrorReturn Then Exit For
  fl_td = flu_f_si(0, 2, t, dl)
  If fl_td = ErrorReturn Then Exit For
  fl_dd = flu_f_si(0, 2, t, dl)
  If fl_dd = ErrorReturn Then Exit For
  
  'seawater
  gss = sal_chempot_h2o_si(sa_si, t, p_si)
  If gss = ErrorReturn Then Exit For
  gs_t = sal_g_si(0, 1, 0, sa_si, t, p_si)
  If gs_t = ErrorReturn Then Exit For
  gs_st = sal_g_si(1, 1, 0, sa_si, t, p_si)
  If gs_st = ErrorReturn Then Exit For
  
  'coefficient matrix
  a(1, 1) = fh_t - a_si * fh_at - fl_t - gs_t + sa_si * gs_st
  a(1, 2) = fh_d - a_si * fh_ad - p_si / dh ^ 2
  a(1, 3) = -fl_d + p_si / dl ^ 2
  a(2, 1) = dh * fh_td
  a(2, 2) = 2 * fh_d + dh * fh_dd
  a(2, 3) = 0
  a(3, 1) = dl * fl_td
  a(3, 2) = 0
  a(3, 3) = 2 * fl_d + dl * fl_dd

  'right-hand sides, must vanish at equilibrium
  b(1) = fl - fh + a_si * fh_a + p_si * (1 / dl - 1 / dh) + gss
  b(2) = p_si / dh - dh * fh_d
  b(3) = p_si / dl - dl * fl_d
  
  'solve equations
  If matrix_solve(a(), b(), x(), 3) <> 0 Then Exit For 'matrix singular

  'update temperature & densities
  t = t + x(1)
  If t <= 0 Then Exit For
  dh = dh + x(2)
  If dh <= 0 Then Exit For
  dl = dl + x(3)
  If dl <= 0 Then Exit For

  'check absolute or relative error limit
  If (eps > 0 And Abs(x(1)) < eps) Or _
     (eps < 0 And Abs(x(1)) < -eps * t) Then
    sea_air_iteration_at_s_a_p = IsOK
    d_air_si = dh
    d_liq_si = dl
    t_si = t
    Exit For
  End If

Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'AIR_LIMITS
  If t_si < dry_air_tmin Or t_si > dry_air_tmax Or _
     d_air_si <= dry_air_dmin Or d_air_si > dry_air_dmax Then
    a_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_air_si = ErrorReturn
    sea_air_iteration_at_s_a_p = ErrorReturn
    Exit Function
  End If
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Or _
     d_liq_si <= flu_dmin Or d_liq_si > flu_dmax Then
    a_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_air_si = ErrorReturn
    sea_air_iteration_at_s_a_p = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Function aux_airfraction_si(ByVal t_si As Double, _
                                    ByVal p_si As Double) As Double

'function returns an estimate of the air mass fraction of saturated
'humid air at given absolute temperature, t_si, in K and absolute
'pressure, p_si, in Pa

Dim dh As Double, dv As Double, pv As Double

aux_airfraction_si = ErrorReturn
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

'as an estimate, add the densities of dry air and vapour

'vapour pressure
pv = liq_vap_vapourpressure_si(t_si)
If pv = ErrorReturn Then Exit Function
If pv > p_si Then Exit Function

If pv = p_si Then
  aux_airfraction_si = 0
  Exit Function
End If

'vapour density
dv = liq_vap_density_vap_si
If dv = ErrorReturn Then Exit Function
If dv <= 0 Then Exit Function

'dry air density at remaining partial pressure
dh = air_density_si(1, t_si, p_si - pv)
If dh = ErrorReturn Then Exit Function
If dh <= 0 Then Exit Function

'air fraction of the ideal mixture
aux_airfraction_si = dh / (dh + dv)
           
End Function

'==========================================================================
Private Function aux_temperature_si(ByVal a_si As Double, _
                                    ByVal p_si As Double) As Double

'function returns an estimate of the temperature in K of saturated
'humid air at given air mass fraction, a_si, in kg/kg and absolute
'pressure, p_si, in Pa

Dim pv As Double, t As Double

aux_temperature_si = ErrorReturn
If a_si < 0 Or a_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function

'as an estimate, ignore the air present

'partial vapour pressure
pv = p_si * air_molfraction_vap_si(a_si)
'related saturation temperature
t = liq_vap_boilingtemperature_si(pv)
If t = ErrorReturn Then Exit Function
If t <= 0 Then Exit Function

aux_temperature_si = t

End Function
