Attribute VB_Name = "Sea_3d_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules:
'     Constants_0_Mdl, file Constants_0.bas
'     Sal_2_Mdl,     file Sal_2.bas
'     Flu_3a_Mdl,    file Flu_3a.bas

'#########################################################################

'=========================================================================
'This module computes the properties of seawater as functions of temperature,
'pressure and density, in particular, the absolute salinity

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98

'Control parameters of the salinity iteration
Private ctrl_initialized As Integer

Private ctrl_mode_salinity As Integer
Private ctrl_loop_maximum As Long
Private ctrl_init_salinity As Double
Private ctrl_eps_exit_salinity As Double

Private Const Version = "19 Aug 2009"

'=========================================================================
Public Function sea_sa_si(ByVal t_si As Double, _
                          ByVal p_si As Double, _
                          ByVal d_si As Double) As Double

'This function computes the absolute salinity sa_si in kg/kg from
'the absolute temperature t_si in K, the absolute pressure p_si in Pa
'and the seawater density d_si in kg/m3

'check value with default settings:
'sea_sa_si(273.15, 101325, 1028) = 3.50315257708621E-02

Dim S As Double, maxit As Integer, eps As Double

sea_sa_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function
If d_si <= 0 Then Exit Function

init_it_ctrl_salinity

Select Case ctrl_mode_salinity
  Case 0:    S = SO_salinity_si
  Case Else: S = ctrl_init_salinity
End Select

Select Case ctrl_loop_maximum
  Case 0:      maxit = 100
  Case -1:     sea_sa_si = S
               Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else:   Exit Function
End Select

eps = ctrl_eps_exit_salinity
If eps = 0 Then Exit Function

'run the iteration
sea_sa_si = salinity_iteration(t_si, p_si, d_si, S, maxit, eps)

End Function

'=========================================================================
Private Function salinity_iteration(ByVal t_si As Double, _
                                    ByVal p_si As Double, _
                                    ByVal d_si As Double, _
                                    ByVal S As Double, _
                                    ByVal maxit As Integer, _
                                    ByVal eps As Double) As Double

Dim ds As Double, dv As Double
Dim gs_p As Double, gs_sp As Double
Dim it As Integer

'this function calculates the absolute salinity s from given values
'of the absolute temperature, t_si, in K, absolute pressure, p_si,
'in Pa and the seawater density, d_si, in kg/m3

'output: salinity_iteration is the result for the absolute salinity in kg/kg

'input: t_si:  absolute temperature in K
'       p_si:  absolute pressure in Pa
'       d_si:  seawater density in kg/m3
'       s:     initial salinity estimate in kg/kg
'       maxit: maxiumum number of iteration steps
'       eps:   salinity tolerance, eps < 0 means relative error

salinity_iteration = ErrorReturn

If eps <= 0 Then Exit Function
If maxit <= 0 Then Exit Function

If check_limits = 1 Then
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Or _
     d_si < flu_dmin Or d_si > flu_dmax Then Exit Function
  'SAL_LIMITS
  If t_si < sal_tmin Or t_si > sal_tmax Or _
     p_si < sal_pmin Or p_si > sal_pmax Then Exit Function
Else
  If t_si <= 0 Or _
     p_si <= 0 Or _
     d_si <= 0 Then Exit Function
End If

check_limits = check_limits - 1

'pure water density
dv = liq_density_si(t_si, p_si)
If dv = ErrorReturn Or dv = 0 Then Exit Function

'saline part of the given specific volume
dv = 1 / d_si - 1 / dv

For it = 1 To maxit
  'derivatives for Newton iteration
  gs_p = sal_g_si(0, 0, 1, S, t_si, p_si)
  If gs_p = ErrorReturn Then Exit For
  gs_sp = sal_g_si(1, 0, 1, S, t_si, p_si)
  If gs_sp = ErrorReturn Then Exit For
  If gs_sp = 0 Then Exit For
  
  'compute salinity correction
  ds = (dv - gs_p) / gs_sp
  'update salinity estimate
  S = S + ds
  If S < 0 Or S >= 1 Then Exit For

  'check iteration tolerance
  If (eps > 0 And Abs(ds) < eps) Or _
     (eps < 0 And Abs(ds) < eps * S) Then
    salinity_iteration = S
    Exit For
  End If
  
Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  If S < sal_smin Or S > sal_smax Then
    salinity_iteration = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Sub init_it_ctrl_salinity()

If ctrl_initialized = -1 Then Exit Sub

ctrl_initialized = -1

'Set default values and modes for salinity iteration
ctrl_loop_maximum = 100
ctrl_mode_salinity = 0           'default: sa = SO_salinity_si = 36.16504 g/kg
ctrl_init_salinity = SO_salinity_si
ctrl_eps_exit_salinity = 0.0000001  'default = 0.0001 g/kg

End Sub

'==========================================================================
Public Sub set_it_ctrl_salinity(ByVal key As String, ByVal value As Double)

'this sub sets control parameters for the Newton iteration used to compute
'potential temperature from reference pressure

'key             value
'it_steps        0           set iteration number to default (100)
'it_steps        n > 0       set iteration number to n
'it_steps       -1           do not iterate, use initial value

'init_sa         0           use default sa = 35.16504 g/kg to start
'init_sa         s > 0       use value s as salinity to start

'tol_sa          0           use default exit accuracy for salinity (0.1 mg/kg)
'tol_sa          eps > 0     use eps as exit accuracy for salinity

init_it_ctrl_salinity

Select Case LCase(Trim(key))

  Case "it_steps":   'iteration steps
    Select Case value
      Case 0:      ctrl_loop_maximum = 100     'default = 100
      Case Is < 0: ctrl_loop_maximum = -1
      Case Else:   ctrl_loop_maximum = value
    End Select

  Case "init_sa":   'start salinity
    Select Case CLng(value)
      Case 0:      ctrl_mode_salinity = 0        'default: sa = 35.16504 g/kg
      Case Is > 0: ctrl_mode_salinity = 1
                   ctrl_init_salinity = value
    End Select

  Case "tol_sa":      'required salinity tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_salinity = 0.0000001 'default = 0.1 mg/kg
      Case Is > 0: ctrl_eps_exit_salinity = value
    End Select

End Select

End Sub
