Attribute VB_Name = "Sea_3c_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules:
'     Constants_0_Mdl, file Constants_0.bas
'     Sea_3a_Mdl,      file Sea_3a.bas
'     Sea_3b_Mdl,      file Sea_3b.bas

'#########################################################################

'=========================================================================
'This module implements the entropy of seawater depending on salinity, enthalpy
'and pressure, as defined in:

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science 2009, as

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98

'Control parameters of the entropy iteration
Private ctrl_initialized As Integer

Private ctrl_mode_entropy As Integer
Private ctrl_loop_maximum As Long
Private ctrl_init_entropy As Double
Private ctrl_eps_exit_entropy As Double


Private Const Version = "20 Jul 2009"

'=========================================================================
Public Function sea_eta_entropy_si(ByVal sa_si As Double, _
                                   ByVal h_si As Double, _
                                   ByVal p_si As Double) As Double

'this function computes specific entropy of seawater from enthalpy
'(the thermodynamic potential in terms of salinity, enthalpy and pressure is entropy)

'returns   entropy of seawater in J/(kg K),

'either from
'sa_si     absolute salinity in kg/kg
'h_si      in-situ enthalpy in J/kg,
'p_si      absolute in-situ pressure in Pa

'or from
'sa_si     absolute salinity in kg/kg
'h_si      potential enthalpy in J/kg,
'p_si      absolute reference pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      entropy computed in Sea_3b and on those made in this module

'Check value with default settings: sea_eta_entropy_si(0.035, 1E5, 1E5) = 350.310622663165

sea_eta_entropy_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function


Dim eta As Double, eps As Double, maxit As Long

sea_eta_entropy_si = ErrorReturn

init_it_ctrl_entropy

Select Case ctrl_mode_entropy
  Case 0:    eta = h_si / 273.15
  Case Else: eta = ctrl_init_entropy
End Select

Select Case ctrl_loop_maximum
  Case 0:      maxit = 100
  Case -1:     sea_eta_entropy_si = eta
               Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else:   Exit Function
End Select

eps = ctrl_eps_exit_entropy
If eps <= 0 Then Exit Function

sea_eta_entropy_si = entropy_iteration(sa_si, h_si, p_si, eta, maxit, eps)

End Function

'=========================================================================
Public Function sea_eta_temperature_si(ByVal sa_si As Double, _
                                       ByVal h_si As Double, _
                                       ByVal p_si As Double, _
                                       ByVal pref_si As Double, _
                                       ByVal key As String) As Double
                                       
'this function computes in-situ temperature of seawater from either in-situ enthalpy or potential enthalpy

'returns   temperature of seawater in K,

'either from
'sa_si     absolute salinity in kg/kg
'h_si      in-situ enthalpy in J/kg,
'p_si      absolute in-situ pressure in Pa
'pref_si   not required
'key       "h"

'or from
'sa_si     absolute salinity in kg/kg
'h_si      potential enthalpy in J/kg,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "hpot"

'note: the accuracy of this function depends on the iteration settings for
'      entropy computed in Sea_3b and on those made in this module

'Check values with default settings:
'sea_eta_temperature_si(0.035,1E5,1E7,1E5,"h") = 295.985682129013
'sea_eta_temperature_si(0.035,1E5,1E7,1E5,"hpot") = 298.413424847945

sea_eta_temperature_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function
If key <> "h" And key <> "hpot" Then Exit Function
If key = "hpot" And pref_si <= 0 Then Exit Function

Dim eta As Double

If key = "hpot" Then
  eta = sea_eta_entropy_si(sa_si, h_si, pref_si)  'entropy from pot. enthalpy
Else
  eta = sea_eta_entropy_si(sa_si, h_si, p_si)     'entropy from in-situ enthalpy
End If
If eta = ErrorReturn Then Exit Function

sea_eta_temperature_si = sea_h_si(0, 1, 0, sa_si, eta, p_si)

End Function

'=========================================================================
Public Function sea_eta_pottemp_si(ByVal sa_si As Double, _
                                   ByVal h_si As Double, _
                                   ByVal p_si As Double, _
                                   ByVal pref_si As Double, _
                                   ByVal key As String) As Double

'this function computes  potential temperature of seawater from either in-situ enthalpy or potential enthalpy

'returns   absolute potential temperature of seawater in K,

'either from
'sa_si     absolute salinity in kg/kg
'h_si      in-situ enthalpy in J/kg,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "h"

'or from
'sa_si     absolute salinity in kg/kg
'h_si      potential enthalpy in J/kg,
'p_si      not required
'pref_si   absolute reference pressure in Pa
'key       "hpot"

'note: the accuracy of this function depends on the iteration settings for
'      entropy computed in Sea_3b and on those made in this module

'Check values with default settings:
'sea_eta_pottemp_si(0.035,1E5,1E7,1E5,"h") = 295.782199115252
'sea_eta_pottemp_si(0.035,1E5,1E7,1E5,"hpot") = 298.194955181952

sea_eta_pottemp_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function
If key <> "h" And key <> "hpot" Then Exit Function
If key = "hpot" And pref_si <= 0 Then Exit Function

Dim eta As Double

If key = "hpot" Then
  eta = sea_eta_entropy_si(sa_si, h_si, pref_si)  'entropy from pot. enthalpy
Else
  eta = sea_eta_entropy_si(sa_si, h_si, p_si)     'entropy from in-situ enthalpy
End If
If eta = ErrorReturn Then Exit Function

sea_eta_pottemp_si = sea_h_si(0, 1, 0, sa_si, eta, pref_si)

End Function

'=========================================================================
Public Function sea_eta_density_si(ByVal sa_si As Double, _
                                   ByVal h_si As Double, _
                                   ByVal p_si As Double, _
                                   ByVal pref_si As Double, _
                                   ByVal key As String) As Double

'this function computes in-situ density of seawater from either in-situ enthalpy or potential enthalpy

'returns   density of seawater in kg/m3,

'either from
'sa_si     absolute salinity in kg/kg
'h_si      in-situ enthalpy in J/kg,
'p_si      absolute in-situ pressure in Pa
'pref_si   not required
'key       "h"

'or from
'sa_si     absolute salinity in kg/kg
'h_si      potential enthalpy in J/kg,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "hpot"

'note: the accuracy of this function depends on the iteration settings for
'      entropy computed in Sea_3b and on those made in this module

'Check values with default settings:
'sea_eta_density_si(0.035,1E5,1E7,1E5,"h") = 1028.10986556355
'sea_eta_density_si(0.035,1E5,1E7,1E5,"hpot") = 1027.36529797843

sea_eta_density_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function
If key <> "h" And key <> "hpot" Then Exit Function
If key = "hpot" And pref_si <= 0 Then Exit Function

Dim eta As Double, v As Double

If key = "hpot" Then
  eta = sea_eta_entropy_si(sa_si, h_si, pref_si)  'entropy from pot. enthalpy
Else
  eta = sea_eta_entropy_si(sa_si, h_si, p_si)     'entropy from in-situ enthalpy
End If
If eta = ErrorReturn Then Exit Function

v = sea_h_si(0, 0, 1, sa_si, eta, p_si)      'specific volume
If v = ErrorReturn Or v <= 0 Then Exit Function

sea_eta_density_si = 1 / v

End Function

'=========================================================================
Public Function sea_eta_potdensity_si(ByVal sa_si As Double, _
                                      ByVal h_si As Double, _
                                      ByVal p_si As Double, _
                                      ByVal pref_si As Double, _
                                      ByVal key As String) As Double

'this function computes potential density of seawater from either in-situ enthalpy or potential enthalpy

'returns   potential density of seawater in kg/m3,

'either from
'sa_si     absolute salinity in kg/kg
'h_si      in-situ enthalpy in J/kg,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "h"

'or from
'sa_si     absolute salinity in kg/kg
'h_si      potential enthalpy in J/kg,
'p_si      not required
'pref_si   absolute reference pressure in Pa
'key       "hpot"

'note: the accuracy of this function depends on the iteration settings for
'      entropy computed in Sea_3b and on those made in this module

'Check values with default settings:
'sea_eta_potdensity_si(0.035,1E5,1E7,1E5,"h") = 1023.91737474051
'sea_eta_potdensity_si(0.035,1E5,1E7,1E5,"hpot") = 1023.20527737089

sea_eta_potdensity_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function
If pref_si <= 0 Then Exit Function

Dim eta As Double, v As Double

If key = "hpot" Then
  eta = sea_eta_entropy_si(sa_si, h_si, pref_si)  'entropy from pot. enthalpy
Else
  eta = sea_eta_entropy_si(sa_si, h_si, p_si)     'entropy from in-situ enthalpy
End If
If eta = ErrorReturn Then Exit Function

v = sea_h_si(0, 0, 1, sa_si, eta, pref_si)      'pot. specific volume
If v = ErrorReturn Or v <= 0 Then Exit Function

sea_eta_potdensity_si = 1 / v

End Function

'=========================================================================
Public Function sea_eta_contraction_t_si(ByVal sa_si As Double, _
                                         ByVal x_si As Double, _
                                         ByVal p_si As Double, _
                                         ByVal pref_si As Double, _
                                         ByVal key As String) As Double

'this function computes the haline contraction coefficient of seawater at constant temperature
'from either in-situ enthalpy (key="h") or potential enthalpy (key = "hpot")
'or from in-situ temperature (key="t") or potential temperature (key = "tpot")

'returns   haline contraction -(1/v)*(dv/ds)_t_p of seawater in kg/kg,

'either from
'sa_si     absolute salinity in kg/kg
'x_si      in-situ enthalpy in J/kg,
'p_si      absolute in-situ pressure in Pa
'pref_si   not required
'key       "h"

'or from
'sa_si     absolute salinity in kg/kg
'x_si      potential enthalpy in J/kg,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "hpot"

'or from
'sa_si     absolute salinity in kg/kg
'x_si      in-situ temperature in K,
'p_si      absolute in-situ pressure in Pa
'pref_si   not required
'key       "t"

'or from
'sa_si     absolute salinity in kg/kg
'x_si      potential temperature in K,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "tpot"

'note: the accuracy of this function depends on the iteration settings for
'      entropy computed in Sea_3b and on those made in this module

'Check values with default settings:
'sea_eta_contraction_t_si(0.035,1E5,1E7,1E5,"h") = 0.728755239643685
'sea_eta_contraction_t_si(0.035,1E5,1E7,1E5,"hpot") = 0.72634931742843
'sea_eta_contraction_t_si(0.035,300,1E7,1E5,"t") = 0.724913833446307
'sea_eta_contraction_t_si(0.035,300,1E7,1E5,"tpot") = 0.724714253917745

Dim eta As Double

sea_eta_contraction_t_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function

eta = sea_eta_entropy_x_si(sa_si, x_si, p_si, pref_si, key)  'entropy
If eta = ErrorReturn Then Exit Function

sea_eta_contraction_t_si = sea_h_contraction_t_si(sa_si, eta, p_si)

End Function

'=========================================================================
Public Function sea_eta_contraction_theta_si(ByVal sa_si As Double, _
                                             ByVal x_si As Double, _
                                             ByVal p_si As Double, _
                                             ByVal pref_si As Double, _
                                             ByVal key As String) As Double

'this function computes the haline contraction coefficient of seawater at constant potential temperature
'from either in-situ enthalpy (key="h") or potential enthalpy (key = "hpot")
'or from in-situ temperature (key="t") or potential temperature (key = "tpot")

'returns   haline contraction - (1/v)*(dv/ds)_t_p of seawater in kg/kg,

'either from
'sa_si     absolute salinity in kg/kg
'x_si      in-situ enthalpy in J/kg,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "h"

'or from
'sa_si     absolute salinity in kg/kg
'x_si      potential enthalpy in J/kg,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "hpot"

'or from
'sa_si     absolute salinity in kg/kg
'x_si      in-situ temperature in K,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "t"

'or from
'sa_si     absolute salinity in kg/kg
'x_si      potential temperature in K,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "tpot"

'note: the accuracy of this function depends on the iteration settings for
'      entropy computed in Sea_3b and on those made in this module

'Check values with default settings:
'sea_eta_contraction_theta_si(0.035,1E5,1E7,1E5,"h") = 0.728499505688196
'sea_eta_contraction_theta_si(0.035,1E5,1E7,1E5,"hpot") = 0.72609973270266
'sea_eta_contraction_theta_si(0.035,300,1E7,1E5,"t") = 0.724667977117403
'sea_eta_contraction_theta_si(0.035,300,1E7,1E5,"tpot") = 0.724468894946259

Dim eta As Double

sea_eta_contraction_theta_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function
If pref_si <= 0 Then Exit Function

eta = sea_eta_entropy_x_si(sa_si, x_si, p_si, pref_si, key)  'entropy
If eta = ErrorReturn Then Exit Function

sea_eta_contraction_theta_si = sea_h_contraction_theta_si(sa_si, eta, p_si, pref_si)

End Function

'=========================================================================
Public Function sea_eta_contraction_h_si(ByVal sa_si As Double, _
                                         ByVal x_si As Double, _
                                         ByVal p_si As Double, _
                                         ByVal pref_si As Double, _
                                         ByVal key As String) As Double

'this function computes the haline contraction coefficient of seawater at constant potential enthalpy
'from either in-situ enthalpy (key="h") or potential enthalpy (key = "hpot")
'or from in-situ temperature (key="t") or potential temperature (key = "tpot")

'returns   haline contraction -(1/v)*(dv/ds)_h_p of seawater in kg/kg,

'either from
'sa_si     absolute salinity in kg/kg
'x_si      in-situ enthalpy in J/kg,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "h"

'or from
'sa_si     absolute salinity in kg/kg
'x_si      potential enthalpy in J/kg,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "hpot"

'or from
'sa_si     absolute salinity in kg/kg
'x_si      in-situ temperature in K,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "t"

'or from
'sa_si     absolute salinity in kg/kg
'x_si      potential temperature in K,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "tpot"

'note: the accuracy of this function depends on the iteration settings for
'      entropy computed in Sea_3b and on those made in this module

'Check values with default settings:
'sea_eta_contraction_h_si(0.035,1E5,1E7,1E5,"h") = 0.718452125956615
'sea_eta_contraction_h_si(0.035,1E5,1E7,1E5,"hpot") = 0.714531922616011
'sea_eta_contraction_h_si(0.035,300,1E7,1E5,"t") = 0.712069013012519
'sea_eta_contraction_h_si(0.035,300,1E7,1E5,"tpot") = 0.711718411190048

Dim eta As Double

sea_eta_contraction_h_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function
If pref_si <= 0 Then Exit Function

eta = sea_eta_entropy_x_si(sa_si, x_si, p_si, pref_si, key)  'entropy
If eta = ErrorReturn Then Exit Function

sea_eta_contraction_h_si = sea_h_contraction_h_si(sa_si, eta, p_si, pref_si)

End Function

'=========================================================================
Public Function sea_eta_expansion_t_si(ByVal sa_si As Double, _
                                       ByVal x_si As Double, _
                                       ByVal p_si As Double, _
                                       ByVal pref_si As Double, _
                                       ByVal key As String) As Double

'this function computes the thermal expansion coefficient of seawater wrt temperature
'from either in-situ enthalpy (key="h") or potential enthalpy (key = "hpot")
'or from in-situ temperature (key="t") or potential temperature (key = "tpot")

'returns   thermal expansion  (1/v)*(dv/dt)_s_p of seawater in 1/K
'          at constant salinity and pressure.

'either from
'sa_si     absolute salinity in kg/kg
'x_si      in-situ enthalpy in J/kg,
'p_si      absolute in-situ pressure in Pa
'pref_si   not used
'key       "h"

'or from
'sa_si     absolute salinity in kg/kg
'x_si      potential enthalpy in J/kg,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "hpot"

'or from
'sa_si     absolute salinity in kg/kg
'x_si      in-situ temperature in K,
'p_si      absolute in-situ pressure in Pa
'pref_si   not used
'key       "t"

'or from
'sa_si     absolute salinity in kg/kg
'x_si      potential temperature in K,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "tpot"

'note: the accuracy of this function depends on the iteration settings for
'      entropy computed in Sea_3b and on those made in this module

'Check values with default settings:
'sea_eta_expansion_t_si(0.035,1E5,1E7,1E5,"h") = 2.89480851145219E-04
'sea_eta_expansion_t_si(0.035,1E5,1E7,1E5,"hpot") = 3.07242256460933E-04
'sea_eta_expansion_t_si(0.035,300,1E7,1E5,"t") = 3.18513471409967E-04
'sea_eta_expansion_t_si(0.035,300,1E7,1E5,"tpot") = 3.20122324739611E-04

Dim eta As Double

sea_eta_expansion_t_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function

eta = sea_eta_entropy_x_si(sa_si, x_si, p_si, pref_si, key)  'entropy
If eta = ErrorReturn Then Exit Function

sea_eta_expansion_t_si = sea_h_expansion_t_si(sa_si, eta, p_si)

End Function

'=========================================================================
Public Function sea_eta_expansion_theta_si(ByVal sa_si As Double, _
                                           ByVal x_si As Double, _
                                           ByVal p_si As Double, _
                                           ByVal pref_si As Double, _
                                           ByVal key As String) As Double

'this function computes the thermal expansion coefficient of seawater wrt potential temperature
'from either in-situ enthalpy (key="h") or potential enthalpy (key = "hpot")
'or from in-situ temperature (key="t") or potential temperature (key = "tpot")

'returns   thermal expansion  (1/v)*(dv/dtheta)_s_p of seawater in 1/K
'          at constant salinity and pressure. theta is potential temperature

'either from
'sa_si     absolute salinity in kg/kg
'x_si      in-situ enthalpy in J/kg,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "h"

'or from
'sa_si     absolute salinity in kg/kg
'x_si      potential enthalpy in J/kg,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "hpot"

'or from
'sa_si     absolute salinity in kg/kg
'x_si      in-situ temperature in K,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "t"

'or from
'sa_si     absolute salinity in kg/kg
'x_si      potential temperature in K,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "tpot"

'note: the accuracy of this function depends on the iteration settings for
'      entropy computed in Sea_3b and on those made in this module

'Check values with default settings:
'sea_eta_expansion_theta_si(0.035,1E5,1E7,1E5,"h") = 2.91293979901893E-04
'sea_eta_expansion_theta_si(0.035,1E5,1E7,1E5,"hpot") = 3.0913484855363E-04
'sea_eta_expansion_theta_si(0.035,300,1E7,1E5,"t") = 3.20454167782952E-04
'sea_eta_expansion_theta_si(0.035,300,1E7,1E5,"tpot") = 3.2206971083886E-04

Dim eta As Double
sea_eta_expansion_theta_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function
If pref_si <= 0 Then Exit Function

eta = sea_eta_entropy_x_si(sa_si, x_si, p_si, pref_si, key)  'entropy
If eta = ErrorReturn Then Exit Function

sea_eta_expansion_theta_si = sea_h_expansion_theta_si(sa_si, eta, p_si, pref_si)

End Function

'=========================================================================
Public Function sea_eta_expansion_h_si(ByVal sa_si As Double, _
                                       ByVal x_si As Double, _
                                       ByVal p_si As Double, _
                                       ByVal pref_si As Double, _
                                       ByVal key As String) As Double

'this function computes the thermal expansion coefficient of seawater wrt potential enthalpy
'from either in-situ enthalpy (key="h") or potential enthalpy (key = "hpot")
'or from in-situ temperature (key="t") or potential temperature (key = "tpot")

'returns   thermal expansion  (1/v)*(dv/dhpot)_s_p of seawater in kg / J
'          at constant salinity and pressure. hpot is potential enthalpy.

'either from
'sa_si     absolute salinity in kg/kg
'x_si      in-situ enthalpy in J/kg,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "h"

'or from
'sa_si     absolute salinity in kg/kg
'x_si      potential enthalpy in J/kg,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "hpot"

'or from
'sa_si     absolute salinity in kg/kg
'x_si      in-situ temperature in K,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "t"

'or from
'sa_si     absolute salinity in kg/kg
'x_si      potential temperature in K,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "tpot"

'note: the accuracy of this function depends on the iteration settings for
'      entropy computed in Sea_3b and on those made in this module

'Check values with default settings:
'sea_eta_expansion_h_si(0.035,1E5,1E7,1E5,"h") = 7.28514646020664E-08
'sea_eta_expansion_h_si(0.035,1E5,1E7,1E5,"hpot") = 7.7287677224533E-08
'sea_eta_expansion_h_si(0.035,300,1E7,1E5,"t") = 8.01009066332972E-08
'sea_eta_expansion_h_si(0.035,300,1E7,1E5,"tpot") = 8.05023387610723E-08

Dim eta As Double

sea_eta_expansion_h_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function
If pref_si <= 0 Then Exit Function

eta = sea_eta_entropy_x_si(sa_si, x_si, p_si, pref_si, key)  'entropy
If eta = ErrorReturn Then Exit Function

sea_eta_expansion_h_si = sea_h_expansion_h_si(sa_si, eta, p_si, pref_si)

End Function

'==========================================================================
Public Sub set_it_ctrl_entropy(ByVal key As String, ByVal value As Double)

'this sub sets control parameters for the Newton iteration used to compute
'entropy from enthalpy

'key             value
'it_steps        0           set iteration number to default (100)
'it_steps        n > 0       set iteration number to n
'it_steps       -1           do not iterate, use initial value

'mode_eta        0           use default entropy eta = h/273.15 to start
'mode_eta        1           use value set by init_eta as entropy to start

'init_eta        eta         use value eta as entropy to start if mode_eta = 1

'tol_eta         0           use default exit accuracy for entropy (1E-4 J/(kg K))
'tol_eta         eps > 0     use eps as exit accuracy for entropy

init_it_ctrl_entropy

Select Case LCase(Trim(key))

  Case "it_steps":   'iteration steps
    Select Case value
      Case 0:      ctrl_loop_maximum = 100     'default = 100
      Case Is < 0: ctrl_loop_maximum = -1
      Case Else:   ctrl_loop_maximum = value
    End Select

  Case "mode_eta":   'start eta
    Select Case CLng(value)
      Case 0:      ctrl_mode_entropy = 0        'default: eta = h/t0
      Case 1:      ctrl_mode_entropy = 1        'default: eta = init_eta
    End Select

  Case "init_eta":   'start eta if ctrl_mode_entropy = 1
                   ctrl_init_entropy = value

  Case "tol_eta":      'required eta tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_entropy = 0.0001 'default = 1E-4 J/(kg K)
      Case Is > 0: ctrl_eps_exit_entropy = value
    End Select

End Select

End Sub

'==========================================================================
Private Sub init_it_ctrl_entropy()

If ctrl_initialized = -1 Then Exit Sub

ctrl_initialized = -1

'Set default values and modes for entropy iteration
ctrl_loop_maximum = 100
ctrl_mode_entropy = 0           'default: eta = h/t0
ctrl_eps_exit_entropy = 0.0001  'default = 1E-4 J/kgK
ctrl_init_entropy = 0

End Sub

'=========================================================================
Private Function entropy_iteration(ByVal sa_si As Double, _
                                   ByVal h_si As Double, _
                                   ByVal p_si As Double, _
                                   ByVal eta_si As Double, _
                                   ByVal maxit As Long, _
                                   ByVal eps As Double) As Double

'returns   eta =  entropy of seawater in J/(kg K),
'          i.e. the entropy that solves h_si = sea_enthalpy_si(sa_si, eta_si, p_si)
'          for eta_si at given h_si

'sa_si     absolute salinity in kg/kg
'h_si      enthalpy in J/kg
'p_si      absolute pressure in Pa
'eta_si    entropy in J/(kg K), initial value
'maxit     max. number of iterations
'eps       required tolerance in J/(kg K)

Dim i As Long
Dim h As Double, eta As Double, t As Double, de As Double

entropy_iteration = ErrorReturn

If eps <= 0 Then Exit Function
If maxit <= 0 Then Exit Function

If check_limits <> 1 Then
  If sa_si < 0 Or sa_si >= 1 Or _
     p_si <= 0 Then Exit Function
Else
  'SAL_LIMITS
  If sa_si < sal_smin Or sa_si > sal_smax Or _
     p_si < sal_pmin Or p_si > sal_pmax Then Exit Function
End If

check_limits = check_limits - 1

eta = eta_si
For i = 1 To maxit

  'get enthalpy and its first derivative for Newton iteration
  h = sea_h_si(0, 0, 0, sa_si, eta, p_si)
  If h = ErrorReturn Then Exit For
  t = sea_h_si(0, 1, 0, sa_si, eta, p_si)
  If t = ErrorReturn Then Exit For
  If t <= 0 Then Exit For
  
  'next entropy improvement step
  de = (h_si - h) / t
  eta = eta + de

  If Abs(de) < eps Then
    entropy_iteration = eta
    Exit For
  End If
  
Next i

check_limits = check_limits + 1

End Function

'=========================================================================
Private Function sea_eta_entropy_x_si(ByVal sa_si As Double, _
                                      ByVal x_si As Double, _
                                      ByVal p_si As Double, _
                                      ByVal pref_si As Double, _
                                      ByVal key As String) As Double

'this function computes the entropy of seawater
'from either in-situ enthalpy (key="h") or potential enthalpy (key = "hpot")
'or from in-situ temperature (key="t") or potential temperature (key = "tpot")

'returns   entropy of seawater in J/(kg K),

'either from
'sa_si     absolute salinity in kg/kg
'x_si      in-situ enthalpy in J/kg,
'p_si      absolute in-situ pressure in Pa
'pref_si   not required
'key       "h"

'or from
'sa_si     absolute salinity in kg/kg
'x_si      potential enthalpy in J/kg,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "hpot"

'or from
'sa_si     absolute salinity in kg/kg
'x_si      in-situ temperature in K,
'p_si      absolute in-situ pressure in Pa
'pref_si   not required
'key       "t"

'or from
'sa_si     absolute salinity in kg/kg
'x_si      potential temperature in K,
'p_si      absolute in-situ pressure in Pa
'pref_si   absolute reference pressure in Pa
'key       "tpot"

sea_eta_entropy_x_si = ErrorReturn

If sa_si < 0 Or sa_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function

Select Case key
  Case "hpot":
    If pref_si <= 0 Then Exit Function
    sea_eta_entropy_x_si = sea_eta_entropy_si(sa_si, x_si, pref_si)  'entropy from pot. enthalpy

  Case "h":
    sea_eta_entropy_x_si = sea_eta_entropy_si(sa_si, x_si, p_si)     'entropy from in-situ enthalpy

  Case "tpot":
    If pref_si <= 0 Then Exit Function
    If x_si <= 0 Then Exit Function
    sea_eta_entropy_x_si = sea_entropy_si(sa_si, x_si, pref_si)  'entropy from pot. temperature

  Case "t":
    If x_si <= 0 Then Exit Function
    sea_eta_entropy_x_si = sea_entropy_si(sa_si, x_si, p_si)     'entropy from in-situ temperature

End Select

End Function
