Attribute VB_Name = "OS2008_5_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules:
'     Constants_0
'     Maths_0
'     Flu_1
'     Ice_1
'     Ice_2
'     Flu_2
'     Sal_2
'     Flu_3a
'     Flu_3b
'     Sea_3a
'     Liq_Vap_4

'#########################################################################

'=========================================================================
'This module produces comparison tables with the check values published in
'R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner
'Mutually consistent thermodynamic potentials for fluid water, ice and seawater:
'a new standard for oceanography.
'Ocean Sci., 4, 275-291, 2008
'www.ocean-sci.net/4/275/2008/
'with the implementation in this library

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98

Private Const Version = "26 Nov 2009"

'==========================================================================
Public Function chk_OS2008_Table2() As String

'Comparison of the library function with check values
'computed in quadruple precision (128 bit floating point)
'as published in Table 2 of

'R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner
'Mutually consistent thermodynamic potentials for fluid water, ice and seawater:
'a new standard for oceanography.
'Ocean Sci., 4, 275-291, 2008
'www.ocean-sci.net/4/275/2008/

'Table 2: Numerical results for properties at the IAPWS-95 triple point obtained from quadruple-precision
'implementations. Here, g is the specific Gibbs energy and rho the density of liquid water (superscript W)
'and vapor (Vap).  The underlined numbers indicate the digits that, based on our tests, can reasonably be
'expected to be reproduced using double-precision code.
'
'Property   Value                             Unit
'T          273.16                            K
'p_W        611.6547|710078944264442598E-6    MPa
'p_Vap      611.6547710078|944264442598E-6    MPa
'g_W          0.6117817|034563826236673       J kg-1
'g_Vap        0.6117817|034563826236673       J kg-1
'rho_w      999.7925200316|20646603898354735  kg m-3
'rho_Vap      4.854575724778|588417176210E-3  kg m-3

Dim CRLF As String
Dim txt As String
Dim t As Double, x As Double

Dim i As Integer, j As Integer, row As String
Dim lrow(5) As Integer

CRLF = Chr(13) + Chr(10)

txt = "Properties of water and vapour at the IAPWS-95 triple point" + CRLF
txt = txt + "defined by T = 273.16 K and liquid.vapour equilibrium." + CRLF
txt = txt + "Computed from thermodynamic potentials of the seawater library" + CRLF
txt = txt + "for publication in Ocean Science, 2009" + CRLF
txt = txt + "R. Feistel, IOW, Version " + Version + CRLF
txt = txt + "Compiled on " + CStr(Now) + CRLF + CRLF

txt = txt + "Compared with Table 2 of the article" + CRLF
txt = txt + "R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner" + CRLF
txt = txt + "Mutually consistent thermodynamic potentials for fluid water, ice and seawater:" + CRLF
txt = txt + "A new standard for oceanography." + CRLF
txt = txt + "Ocean Sci., 4, 275-291, 2008" + CRLF
txt = txt + "www.ocean-sci.net/4/275/2008/:" + CRLF + CRLF

txt = txt + "Table 2: Numerical results for properties at the IAPWS-95 triple point obtained from quadruple-precision" + CRLF
txt = txt + "implementations. Here, g is the specific Gibbs energy and rho the density of liquid water (superscript W)" + CRLF
txt = txt + "and vapor (Vap).  The underlined numbers indicate the digits that, based on our tests, can reasonably be" + CRLF
txt = txt + "expected to be reproduced using double-precision code." + CRLF

'set_it_ctrl_liq_vap "tol_vap_pres", -0.000000000000001  'this appears unnecessary

t = 273.16
set_liq_vap_eq_at_t t

For i = 0 To 7
  txt = txt + CRLF
  For j = 1 To 3
    Select Case i
      Case 0:  row = Choose(j, "Property ", " Value                                ", "Unit")
      Case 1:  row = Choose(j, "T        ", " 273.16                               ", "K")
      Case 2:  row = Choose(j, "p_W      ", " 0.6116547|710078944264442598E-3      ", "MPa")
      Case 3:  row = Choose(j, "p_Vap    ", " 0.6116547710078|944264442598E-3      ", "MPa")
      Case 4:  row = Choose(j, "g_W      ", " 0.6117817|034563826236673            ", "J kg-1")
      Case 5:  row = Choose(j, "g_Vap    ", " 0.6117817|034563826236673            ", "J kg-1")
      Case 6:  row = Choose(j, "rho_W    ", " 0.9997925200316|20646603898354735E+3 ", "kg m-3")
      Case 7:  row = Choose(j, "rho_Vap  ", " 0.4854575724778|588417176210E-2      ", "kg m-3")
    End Select
    
    lrow(j) = Len(row)
    If j > 1 Then txt = txt + "    "
    txt = txt + row
  Next j
  
  If i > 1 Then
    txt = txt + CRLF + "this code:   "

    Select Case i

      Case 2: x = liq_vap_pressure_liq_si
              
      Case 3: x = liq_vap_pressure_vap_si
        
      Case 4: x = flu_gibbs_energy_si(t, liq_vap_density_liq_si)
        
      Case 5: x = flu_gibbs_energy_si(t, liq_vap_density_vap_si)
        
      Case 6: x = liq_vap_density_liq_si
        
      Case 7: x = liq_vap_density_vap_si
        
    End Select
    
    txt = txt + Left(EFormat(x) + Space(lrow(2)), lrow(2))

  End If
  txt = txt + CRLF
Next i

txt = txt + CRLF + "Note: This computation was done with default iteration settings" + CRLF + CRLF

'set_it_ctrl_liq_vap "tol_vap_pres", 0

chk_OS2008_Table2 = txt

'output:

'Properties of water and vapour at the IAPWS-95 triple point
'defined by T = 273.16 K and liquid.vapour equilibrium.
'Computed from thermodynamic potentials of the seawater library
'for publication in Ocean Science, 2009
'R. Feistel, IOW, Version 19 Nov 2009
'Compiled on 19.11.2009 20:37:57
'
'Compared with Table 2 of the article
'R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner
'Mutually consistent thermodynamic potentials for fluid water, ice and seawater:
'A new standard for oceanography.
'Ocean Sci., 4, 275-291, 2008
'www.ocean-sci.net/4/275/2008/:
'
'Table 2: Numerical results for properties at the IAPWS-95 triple point obtained from quadruple-precision
'implementations. Here, g is the specific Gibbs energy and rho the density of liquid water (superscript W)
'and vapor (Vap).  The underlined numbers indicate the digits that, based on our tests, can reasonably be
'expected to be reproduced using double-precision code.
'
'Property      Value                                    Unit
'
'T             273.16                                   K
'
'p_W           0.6116547|710078944264442598E-3          MPa
'this code:    0.611654779024747E+3
'
'p_Vap         0.6116547710078|944264442598E-3          MPa
'this code:    0.611654771007896E+3
'
'g_W           0.6117817|034563826236673                J kg-1
'this code:    0.611781715281918
'
'g_Vap         0.6117817|034563826236673                J kg-1
'this code:    0.611781705055009
'
'rho_W         0.9997925200316|20646603898354735E+3     kg m-3
'this code:    0.999792520031619E+3
'
'rho_Vap       0.4854575724778|588417176210E-2          kg m-3
'this code:    0.48545757247786E-2

'Note: This computation was done with default iteration settings

End Function

'==========================================================================
Public Function chk_OS2008_Table3() As String

'Comparison of the library function with check values
'computed in quadruple precision (128 bit floating point)
'as published in Table 3 of

'R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner
'Mutually consistent thermodynamic potentials for fluid water, ice and seawater:
'a new standard for oceanography.
'Ocean Sci., 4, 275-291, 2008
'www.ocean-sci.net/4/275/2008/

'Table 3: Quadruple-precision results for the properties of water, vapor and ice at the quadruple-precision
'estimate of the IAPWS-95 triple point given in Table 2 (T = 273.16 K, p = 611.6547710078944264442598E-6 MPa),
'computed with the coefficients given in Table 1 of this paper. The underlined numbers indicate the digits that,
'based on our tests, can reasonably be expected to be reproduced using double-precision code.

'Property      Property of water            Property of vapor          Property of ice             Unit
'g             6.117817|0345638262367E-1    6.1178170|345638262367E-1  6.11781703|45638262367E-1   J kg-1
'(dg/dT)_p    -2.E-27 |  5.E-11            -9.15549340929856|14572E+3  1.22069433939648|07369E+3   J kg-1 K-1
'(dg/dp)_T     1.00020752302525|00703E-3    2.05991224917108|24907E+2  1.09085812736668|70118E-3   m3 kg-1
'(d2g/dT2)_p  -1.5448497277676|153768E+1   -6.89834540986158|31310    -7.67602985875190|86996      J kg-1 K-2
'(d2g/dTdp)   -6.797921518|0919329145E-8    7.5819292780093|068933E-1  1.74387964700076|13398E-7   m3 kg-1 K-1
'(d2g/dp2)_T  -5.09062317110|21898838E-13  -3.36992433127456|08812E-1 -1.28495941571693|02559E-13  m3 kg-1 Pa-1
'h             6.117817|0345638262367E-1    2.50091519146569|85040E+6 -3.3344425396783|922172E+5   J kg-1
'f             4.E-26 |  5.E-9             -1.2599490372460|606626E+5 -5.5446874|640188852815E-2   J kg-1
'u             7.E-25 |  2.E-9              2.37491967595938|89814E+6 -3.3344492119641|731829E+5   J kg-1
's             2.E-27 | -5.E-11             9.15549340929856|14572E+3 -1.22069433939648|07369E+3   J kg-1 K-1
'rho           9.997925200316|2064660E+2    4.8545757247785|884172E-3  9.16709492199487|95011E+2   kg m-3
'cp            4.21991151637|00181633E+3    1.88435203215779|00481E+3  2.09678431621667|13804E+3   J kg-1 K-1
'alpha        -6.79651108555|03138742E-5    3.68070498200120|30606E-3  1.59863102565909|02275E-4   K-1
'kappa-T       5.08956696876|76183977E-10   1.63595528529462|01284E-3  1.17793449347881|78705E-10  Pa-1
'kappa-s       5.08657624752|61953085E-10   1.2314112503314|651356E-3  1.14161597778763|31107E-10  Pa-1

Dim CRLF As String, TB As String
Dim txt As String
Dim t As Double, x As Double, p As Double

Dim i As Integer, j As Integer, row As String
Dim tol_liq As Double, tol_vap As Double
Dim lrow(5) As Integer

CRLF = Chr(13) + Chr(10)
TB = Chr(9)

txt = "Properties of water, vapor and ice at the IAPWS-95 triple point" + CRLF
txt = txt + "(T = 273.16 K, p = 611.6547710078944264442598E-6 MPa)" + CRLF
txt = txt + "Computed from thermodynamic potentials of the seawater library" + CRLF
txt = txt + "for publication in Ocean Science, 2009" + CRLF
txt = txt + "R. Feistel, IOW, Version " + Version + CRLF
txt = txt + "Compiled on " + CStr(Now) + CRLF + CRLF

txt = txt + "Compared with Table 3 of the article" + CRLF
txt = txt + "R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner" + CRLF
txt = txt + "Mutually consistent thermodynamic potentials for fluid water, ice and seawater:" + CRLF
txt = txt + "A new standard for oceanography." + CRLF
txt = txt + "Ocean Sci., 4, 275-291, 2008" + CRLF
txt = txt + "www.ocean-sci.net/4/275/2008/:" + CRLF + CRLF

txt = txt + "Table 3: Quadruple-precision results for the properties of water, vapor and ice at the quadruple-precision" + CRLF
txt = txt + "estimate of the IAPWS-95 triple point given in Table 2 (T = 273.16 K, p = 611.6547710078944264442598E-6 MPa)," + CRLF
txt = txt + "computed with the coefficients given in Table 1 of this paper. The separated numbers indicate the digits that," + CRLF
txt = txt + "based on our tests, can reasonably be expected to be reproduced using double-precision code." + CRLF

t = 273.16
p = 611.654771007894

tol_liq = get_it_ctrl_density("tol_liq_dens")
tol_vap = get_it_ctrl_density("tol_vap_dens")
set_it_ctrl_density "tol_liq_dens", -0.000000000000001
set_it_ctrl_density "tol_vap_dens", -0.000000000000001

For i = 0 To 15
  txt = txt + CRLF
  For j = 1 To 5
    Select Case i
      Case 0:  row = Choose(j, "Property     ", " Property of water           ", " Property of vapor         ", " Property of ice             ", "Unit")
      Case 1:  row = Choose(j, "g            ", " 6.117817|0345638262367E-1   ", " 6.1178170|345638262367E-1 ", " 6.11781703|45638262367E-1   ", "J kg-1")
      Case 2:  row = Choose(j, "(dg/dT)_p    ", "-2.E-27 |  5.E-11            ", "-9.15549340929856|14572E+3 ", " 1.22069433939648|07369E+3   ", "J kg-1 K-1")
      Case 3:  row = Choose(j, "(dg/dp)_T    ", " 1.00020752302525|00703E-3   ", " 2.05991224917108|24907E+2 ", " 1.09085812736668|70118E-3   ", "m3 kg-1")
      Case 4:  row = Choose(j, "(d2g/dT2)_p  ", "-1.5448497277676|153768E+1   ", "-6.89834540986158|31310    ", "-7.67602985875190|86996      ", "J kg-1 K-2")
      Case 5:  row = Choose(j, "(d2g/dTdp)   ", "-6.797921518|0919329145E-8   ", " 7.5819292780093|068933E-1 ", " 1.74387964700076|13398E-7   ", "m3 kg-1 K-1")
      Case 6:  row = Choose(j, "(d2g/dp2)_T  ", "-5.09062317110|21898838E-13  ", "-3.36992433127456|08812E-1 ", "-1.28495941571693|02559E-13  ", "m3 kg-1 Pa-1")
      Case 7:  row = Choose(j, "h            ", " 6.117817|0345638262367E-1   ", " 2.50091519146569|85040E+6 ", "-3.3344425396783|922172E+5   ", "J kg-1")
      Case 8:  row = Choose(j, "f            ", " 4.E-26 |  5.E-9             ", "-1.2599490372460|606626E+5 ", "-5.5446874|640188852815E-2   ", "J kg-1")
      Case 9:  row = Choose(j, "u            ", " 7.E-25 |  2.E-9             ", " 2.37491967595938|89814E+6 ", "-3.3344492119641|731829E+5   ", "J kg-1")
      Case 10: row = Choose(j, "s            ", " 2.E-27 | -5.E-11            ", " 9.15549340929856|14572E+3 ", "-1.22069433939648|07369E+3   ", "J kg-1 K-1")
      Case 11: row = Choose(j, "rho          ", " 9.997925200316|2064660E+2   ", " 4.8545757247785|884172E-3 ", " 9.16709492199487|95011E+2   ", "kg m-3")
      Case 12: row = Choose(j, "cp           ", " 4.21991151637|00181633E+3   ", " 1.88435203215779|00481E+3 ", " 2.09678431621667|13804E+3   ", "J kg-1 K-1")
      Case 13: row = Choose(j, "alpha        ", "-6.79651108555|03138742E-5   ", " 3.68070498200120|30606E-3 ", " 1.59863102565909|02275E-4   ", "K-1")
      Case 14: row = Choose(j, "kappa-T      ", " 5.08956696876|76183977E-10  ", " 1.63595528529462|01284E-3 ", " 1.17793449347881|78705E-10  ", "Pa-1")
      Case 15: row = Choose(j, "kappa-s      ", " 5.08657624752|61953085E-10  ", " 1.2314112503314|651356E-3 ", " 1.14161597778763|31107E-10  ", "Pa-1")
    End Select
    
    lrow(j) = Len(row)
    If j > 1 Then txt = txt + TB
    txt = txt + row
  Next j
  
  If i > 0 Then
    txt = txt + CRLF + "this code:   "
    For j = 2 To 4
      Select Case i
        Case 1
          Select Case j
            Case 2: x = liq_g_si(0, 0, t, p)
            Case 3: x = vap_g_si(0, 0, t, p)
            Case 4: x = ice_g_si(0, 0, t, p)
          End Select
          
        Case 2
          Select Case j
            Case 2: x = liq_g_si(1, 0, t, p)
            Case 3: x = vap_g_si(1, 0, t, p)
            Case 4: x = ice_g_si(1, 0, t, p)
          End Select
          
        Case 3
          Select Case j
            Case 2: x = liq_g_si(0, 1, t, p)
            Case 3: x = vap_g_si(0, 1, t, p)
            Case 4: x = ice_g_si(0, 1, t, p)
          End Select
          
        Case 4
          Select Case j
            Case 2: x = liq_g_si(2, 0, t, p)
            Case 3: x = vap_g_si(2, 0, t, p)
            Case 4: x = ice_g_si(2, 0, t, p)
          End Select
          
        Case 5
          Select Case j
            Case 2: x = liq_g_si(1, 1, t, p)
            Case 3: x = vap_g_si(1, 1, t, p)
            Case 4: x = ice_g_si(1, 1, t, p)
          End Select
          
        Case 6
          Select Case j
            Case 2: x = liq_g_si(0, 2, t, p)
            Case 3: x = vap_g_si(0, 2, t, p)
            Case 4: x = ice_g_si(0, 2, t, p)
          End Select
          
        Case 7
          Select Case j
            Case 2: x = liq_enthalpy_si(t, p)
            Case 3: x = vap_enthalpy_si(t, p)
            Case 4: x = ice_enthalpy_si(t, p)
          End Select
          
        Case 8
          Select Case j
            Case 2: x = liq_g_si(0, 0, t, p) - p * liq_g_si(0, 1, t, p)
            Case 3: x = vap_g_si(0, 0, t, p) - p * vap_g_si(0, 1, t, p)
            Case 4: x = ice_g_si(0, 0, t, p) - p * ice_g_si(0, 1, t, p)
          End Select
          
        Case 9
          Select Case j
            Case 2: x = liq_internal_energy_si(t, p)
            Case 3: x = vap_internal_energy_si(t, p)
            Case 4: x = ice_internal_energy_si(t, p)
          End Select
          
        Case 10
          Select Case j
            Case 2: x = liq_entropy_si(t, p)
            Case 3: x = vap_entropy_si(t, p)
            Case 4: x = ice_entropy_si(t, p)
          End Select
          
        Case 11
          Select Case j
            Case 2: x = liq_density_si(t, p)
            Case 3: x = vap_density_si(t, p)
            Case 4: x = ice_density_si(t, p)
          End Select
          
        Case 12
          Select Case j
            Case 2: x = liq_cp_si(t, p)
            Case 3: x = vap_cp_si(t, p)
            Case 4: x = ice_cp_si(t, p)
          End Select
          
        Case 13
          Select Case j
            Case 2: x = liq_expansion_si(t, p)
            Case 3: x = vap_expansion_si(t, p)
            Case 4: x = ice_expansion_si(t, p)
          End Select
          
        Case 14
          Select Case j
            Case 2: x = liq_kappa_t_si(t, p)
            Case 3: x = vap_kappa_t_si(t, p)
            Case 4: x = ice_kappa_t_si(t, p)
          End Select
          
        Case 15
          Select Case j
            Case 2: x = liq_kappa_s_si(t, p)
            Case 3: x = vap_kappa_s_si(t, p)
            Case 4: x = ice_kappa_s_si(t, p)
          End Select
      End Select
      
      txt = txt + TB + Left(EFormat(x) + Space(lrow(j)), lrow(j))
    Next j
    txt = txt + TB + row + CRLF
  End If
Next i

txt = txt + CRLF + CRLF + "Note: For this computation, the iteration accuracy was increased as"
txt = txt + CRLF + "set_it_ctrl_density " + Chr(34) + "tol_liq_dens" + Chr(34) + ", -1E-15"
txt = txt + CRLF + "set_it_ctrl_density " + Chr(34) + "tol_vap_dens" + Chr(34) + ", -1E-15"

txt = txt + CRLF + CRLF + "The default iteration accuracy is as low as 0.1 ppm, i.e. equivalent to"
txt = txt + CRLF + "set_it_ctrl_density " + Chr(34) + "tol_liq_dens" + Chr(34) + ", -1E-7"
txt = txt + CRLF + "set_it_ctrl_density " + Chr(34) + "tol_vap_dens" + Chr(34) + ", -1E-7"

chk_OS2008_Table3 = txt

set_it_ctrl_density "tol_liq_dens", tol_liq
set_it_ctrl_density "tol_vap_dens", tol_vap


'Properties of water, vapor and ice at the IAPWS-95 triple point
'(T = 273.16 K, p = 611.6547710078944264442598E-6 MPa)
'Computed from thermodynamic potentials of the seawater library
'for publication in Ocean Science, 2009
'R. Feistel, IOW, Version 10 Apr 2009
'Compiled on 11.04.2009 10:21:10
'
'Compared with Table 3 of the article
'R.Feistel , d.g.Wright, k.Miyagawa, a.h.Harvey, j.Hruby, d.R.Jackett, t.j.McDougall, w.Wagner
'Mutually consistent thermodynamic potentials for fluid water, ice and seawater:
'A new standard for oceanography.
'Ocean Sci., 4, 275-291, 2008
'www.ocean-sci.net/4/275/2008/:
'
'Table 3: Quadruple-precision results for the properties of water, vapor and ice at the quadruple-precision
'estimate of the IAPWS-95 triple point given in Table 2 (T = 273.16 K, p = 611.6547710078944264442598E-6 MPa),
'computed with the coefficients given in Table 1 of this paper. The separated numbers indicate the digits that,
'based on our tests, can reasonably be expected to be reproduced using double-precision code.
'
'Property       Property of water             Property of vapor           Property of ice              Unit
'g              6.117817|0345638262367E-1     6.1178170|345638262367E-1   6.11781703|45638262367E-1    J kg-1
'this code:     0.611781702747821             0.611781704694508           0.611781703075394            J kg-1
'(dg/dT)_p     -2.E-27 |  5.E-11             -9.15549340929856|14572E+3   1.22069433939648|07369E+3    J kg-1 K-1
'this code:     0.323231705790626E-10        -0.915549340929856E+4        0.122069433939648E+4         J kg-1 K-1
'(dg/dp)_T      1.00020752302525|00703E-3     2.05991224917108|24907E+2   1.09085812736668|70118E-3    m3 kg-1
'this code:     0.100020752302525E-2          0.205991224917108E+3        0.109085812736669E-2         m3 kg-1
'(d2g/dT2)_p   -1.5448497277676|153768E+1    -6.89834540986158|31310     -7.67602985875190|86996       J kg-1 K-2
'this code:    -0.154484972776758E+2         -0.689834540986158E+1       -0.767602985875191E+1         J kg-1 K-2
'(d2g/dTdp)    -6.797921518|0919329145E-8     7.5819292780093|068933E-1   1.74387964700076|13398E-7    m3 kg-1 K-1
'this code:    -0.679792151810124E-7          0.758192927800931           0.174387964700076E-6         m3 kg-1 K-1
'(d2g/dp2)_T   -5.09062317110|21898838E-13   -3.36992433127456|08812E-1  -1.28495941571693|02559E-13   m3 kg-1 Pa-1
'this code:    -0.509062317110255E-12        -0.336992433127457          -0.128495941571693E-12        m3 kg-1 Pa-1
'h              6.117817|0345638262367E-1     2.50091519146569|85040E+6  -3.3344425396783|922172E+5    J kg-1
'this code:     0.611781693918423             0.25009151914657E+7        -0.33344425396784E+6          J kg-1
'f              4.E-26 |  5.E-9              -1.2599490372460|606626E+5  -5.5446874|640188852815E-2    J kg-1
'this code:    -0.708563455467919E-9         -0.125994903724605E+6       -0.55446875021177E-1          J kg-1
'u              7.E-25 |  2.E-9               2.37491967595938|89814E+6  -3.3344492119641|731829E+5    J kg-1
'this code:    -0.636603269475984E-8          0.237491967595939E+7       -0.333444921196418E+6         J kg-1
's              2.E-27 | -5.E-11              9.15549340929856|14572E+3  -1.22069433939648|07369E+3    J kg-1 K-1
'this code:    -0.323231705790626E-10         0.915549340929856E+4       -0.122069433939648E+4         J kg-1 K-1
'rho            9.997925200316|2064660E+2     4.8545757247785|884172E-3   9.16709492199487|95011E+2    kg m-3
'this code:     0.999792520031618E+3          0.485457572477859E-2        0.916709492199488E+3         kg m-3
'cp             4.21991151637|00181633E+3     1.88435203215779|00481E+3   2.09678431621667|13804E+3    J kg-1 K-1
'this code:     0.421991151636992E+4          0.188435203215779E+4        0.209678431621667E+4         J kg-1 K-1
'alpha         -6.79651108555|03138742E-5     3.68070498200120|30606E-3   1.59863102565909|02275E-4    K-1
'this code:    -0.679651108555959E-4          0.36807049820012E-2         0.159863102565909E-3         K-1
'kappa-T        5.08956696876|76183977E-10    1.63595528529462|01284E-3   1.17793449347881|78705E-10   Pa-1
'this code:     0.508956696876796E-9          0.163595528529462E-2        0.117793449347882E-9         Pa-1
'kappa-s        5.08657624752|61953085E-10    1.2314112503314|651356E-3   1.14161597778763|31107E-10   Pa-1
'this code:     0.508657624752653E-9          0.123141125033147E-2        0.114161597778763E-9         Pa-1
'
'Note: For this computation, the iteration accuracy was increased as
'set_it_ctrl_density "tol_liq_dens", -1E-15
'set_it_ctrl_density "tol_vap_dens", -1E-15
'
'The default iteration accuracy is as low as 0.1 ppm, i.e. equivalent to
'set_it_ctrl_density "tol_liq_dens", -1E-7
'set_it_ctrl_density "tol_vap_dens", -1E-7

End Function

'==========================================================================
Public Function chk_OS2008_TableA1() As String

'Comparison of the library function with check values
'computed in quadruple precision (128 bit floating point)
'as published in Table A1 of

'R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner
'Mutually consistent thermodynamic potentials for fluid water, ice and seawater:
'a new standard for oceanography.
'Ocean Sci., 4, 275291, 2008
'www.ocean-sci.net/4/275/2008/

'Quadruple-precision values corresponding to results published in Table 6 of IAPWS-95 with
'the coefficients given in Table 1 of this paper. The ideal-gas part  and the residual part
'of the dimensionless Helmholtz free energy together with the corresponding derivatives
'are shown for T = 500 K and   = 838.025 kg m-3.
'The underlined numbers indicate the digits that, based on our tests, can reasonably be expected
'to be reproduced using double-precision code.

'since phi_0 and phi_r are Private functions inaccessible from here, just call chk_IAPWS95_Table6
chk_OS2008_TableA1 = chk_IAPWS95_Table6

End Function

'==========================================================================
Public Function chk_OS2008_TableA2() As String

'Comparison of the library function with check values
'computed in quadruple precision (128 bit floating point)
'as published in Table A2 of

'R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner
'Mutually consistent thermodynamic potentials for fluid water, ice and seawater:
'a new standard for oceanography.
'Ocean Sci., 4, 275291, 2008
'www.ocean-sci.net/4/275/2008/

'Table A2: Quadruple-precision results for water properties in the single-phase region at the
'selected values of T and rho published in Table 7 of IAPWS-95 with the coefficients given in
'Table 1 of this paper.  The underlined numbers indicate the digits that, based on our tests,
'can reasonably be expected to be reproduced using double-precision code.

'T     rho        p                            cv                           w                            s
'K     kg m-3     MPa                          J kg-1 K-1                   m s-1                        J kg-1 K-1
'300   996.5560   9.92418351|81379651094E-2    4.13018111585|84960765E+3    1.501519138081|3064465E+3    0.3930626428807|7403467E+3
'300   1005.308   2.00022515281|32528836E+1    4.067983470885|8382510E+3    1.534925010962|1962916E+3    0.3874054009992|1296375E+3
'300   1188.202   7.0000470354978|172066E+2    3.4613558020377|671029E+3    2.44357991674018|94386E+3    0.1326096164207|5693598E+3
'500   0.435000   9.99679423176022|31789E-2    1.50817541391104|36746E+3    5.48314252654327|71044E+2    7.94488271364642|13826E+3
'500   4.532000   9.99938124839912|78912E-1    1.66991024524550|69490E+3    5.35739001345214|77951E+2    6.82502725276895|84303E+3
'500   838.0250   1.00003858009|22118506E+1    3.22106218674058|33518E+3    1.2712844091476|324779E+3    2.56690918542225|39144E+3
'500   1084.564   7.0000040549458|516645E+2    3.07437693004544|36204E+3    2.41200876574467|58352E+3    2.03237509190663|89535E+3
'647   358.0000   2.203847557065|2149984E+1    6.18315727666|83092166E+3    2.521450782700|0715938E+2    4.32092306675500|33099E+3
'900   0.241000   1.00062558682661|88154E-1    1.75890657044481|38520E+3    7.24027146529189|38252E+2    9.16653193855238|42681E+3
'900   52.61500   2.00000690372146|14551E+1    1.93510525512622|41493E+3    6.9844567383679|534276E+2    6.59070224851012|77853E+3
'900   870.7690   7.000000057556|5402666E+2    2.66422349779369|96739E+3    2.0193360824868|338741E+3    4.17223801584632|58117E+3


Dim CRLF As String, TB As String
Dim txt As String
Dim t As Double, x As Double, d As Double

Dim i As Integer, j As Integer, row As String
Dim lrow(6) As Integer

CRLF = Chr(13) + Chr(10)
TB = Chr(9)

txt = "Properties of fluid water" + CRLF
txt = txt + "Computed from thermodynamic potentials of the seawater library" + CRLF
txt = txt + "for publication in Ocean Science, 2009" + CRLF
txt = txt + "R. Feistel, IOW, Version " + Version + CRLF
txt = txt + "Compiled on " + CStr(Now) + CRLF + CRLF

txt = txt + "Compared with Table A2 of the article" + CRLF
txt = txt + "R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner" + CRLF
txt = txt + "Mutually consistent thermodynamic potentials for fluid water, ice and seawater:" + CRLF
txt = txt + "A new standard for oceanography." + CRLF
txt = txt + "Ocean Sci., 4, 275-291, 2008" + CRLF
txt = txt + "www.ocean-sci.net/4/275/2008/:" + CRLF + CRLF

txt = txt + "Table A2: Quadruple-precision results for water properties in the single-phase region at the" + CRLF
txt = txt + "selected values of T and rho published in Table 7 of IAPWS-95 with the coefficients given in" + CRLF
txt = txt + "Table 1 of this paper.  The underlined numbers indicate the digits that, based on our tests," + CRLF
txt = txt + "can reasonably be expected to be reproduced using double-precision code." + CRLF

For i = 0 To 12

  txt = txt + CRLF
  For j = 1 To 6
  
    Select Case i
      Case 0:  row = Choose(j, "T     ", "rho        ", "p                            ", "cv                           ", "w                            ", "s")
      Case 1:  row = Choose(j, "K     ", "kg m-3     ", "MPa                          ", "J kg-1 K-1                   ", "m s-1                        ", "J kg-1 K-1")
      Case 2:  row = Choose(j, "300   ", "996.5560   ", "0.992418351|81379651094E-1   ", "0.413018111585|84960765E+4   ", "0.1501519138081|3064465E+4   ", "0.3930626428807|7403467E+3")
      Case 3:  row = Choose(j, "300   ", "1005.308   ", "0.200022515281|32528836E+2   ", "0.4067983470885|8382510E+4   ", "0.1534925010962|1962916E+4   ", "0.3874054009992|1296375E+3")
      Case 4:  row = Choose(j, "300   ", "1188.202   ", "0.70000470354978|172066E+3   ", "0.34613558020377|671029E+4   ", "0.244357991674018|94386E+4   ", "0.1326096164207|5693598E+3")
      Case 5:  row = Choose(j, "500   ", "0.435000   ", "0.999679423176022|31789E-1   ", "0.150817541391104|36746E+4   ", "0.548314252654327|71044E+3   ", "0.794488271364642|13826E+4")
      Case 6:  row = Choose(j, "500   ", "4.532000   ", "0.999938124839912|78912      ", "0.166991024524550|69490E+4   ", "0.535739001345214|77951E+3   ", "0.682502725276895|84303E+4")
      Case 7:  row = Choose(j, "500   ", "838.0250   ", "0.100003858009|22118506E+2   ", "0.322106218674058|33518E+4   ", "0.12712844091476|324779E+4   ", "0.256690918542225|39144E+4")
      Case 8:  row = Choose(j, "500   ", "1084.564   ", "0.70000040549458|516645E+3   ", "0.307437693004544|36204E+4   ", "0.241200876574467|58352E+4   ", "0.203237509190663|89535E+4")
      Case 9:  row = Choose(j, "647   ", "358.0000   ", "0.2203847557065|2149984E+2   ", "0.618315727666|83092166E+4   ", "0.2521450782700|0715938E+3   ", "0.432092306675500|33099E+4")
      Case 10: row = Choose(j, "900   ", "0.241000   ", "0.100062558682661|88154      ", "0.175890657044481|38520E+4   ", "0.724027146529189|38252E+3   ", "0.916653193855238|42681E+4")
      Case 11: row = Choose(j, "900   ", "52.61500   ", "0.200000690372146|14551E+2   ", "0.193510525512622|41493E+4   ", "0.69844567383679|534276E+3   ", "0.659070224851012|77853E+4")
      Case 12: row = Choose(j, "900   ", "870.7690   ", "0.7000000057556|5402666E+3   ", "0.266422349779369|96739E+4   ", "0.20193360824868|338741E+4   ", "0.417223801584632|58117E+4")
    End Select
    
    lrow(j) = Len(row)
    If j > 1 Then txt = txt + TB
    txt = txt + row
    
    Select Case j
      Case 1: t = Val(row)
      Case 2: d = Val(row)
    End Select

  Next j
  
  If i > 1 Then
  
    txt = txt + CRLF + Left("this code" + Space(lrow(1)) + Space(lrow(2)), lrow(1) + lrow(2) + 2)
    For j = 3 To 6
    
      Select Case j
        Case 3: x = flu_pressure_si(t, d) * 0.000001
        Case 4: x = flu_cv_si(t, d)
        Case 5: x = flu_soundspeed_si(t, d)
        Case 6: x = flu_entropy_si(t, d)
      End Select
      txt = txt + TB + Left(Trim(EFormat(x)) + Space(lrow(j)), lrow(j))
    Next j
    txt = txt + CRLF
  End If
Next i

chk_OS2008_TableA2 = txt

'Properties of fluid water
'Computed from thermodynamic potentials of the seawater library
'for publication in Ocean Science, 2009
'R. Feistel, IOW, Version 11 Apr 2009
'Compiled on 11.04.2009 17:54:49
'
'Compared with Table A2 of the article
'R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner
'Mutually consistent thermodynamic potentials for fluid water, ice and seawater:
'A new standard for oceanography.
'Ocean Sci., 4, 275-291, 2008
'www.ocean-sci.net/4/275/2008/:
'
'Table A2: Quadruple-precision results for water properties in the single-phase region at the
'selected values of T and rho published in Table 7 of IAPWS-95 with the coefficients given in
'Table 1 of this paper.  The underlined numbers indicate the digits that, based on our tests,
'can reasonably be expected to be reproduced using double-precision code.
'
'T       rho         p                             cv                            w                             s
'K       kg m-3      MPa                           J kg-1 K-1                    m s-1                         J kg-1 K-1
'300     996.5560    9.92418351|81379651094E-2     4.13018111585|84960765E+3     1.501519138081|3064465E+3     0.3930626428807|7403467E+3
'this code:          0.992418351852896E-1          0.413018111585852E+4          0.150151913808129E+4          0.393062642880778E+3
'300     1005.308    2.00022515281|32528836E+1     4.067983470885|8382510E+3     1.534925010962|1962916E+3     0.3874054009992|1296375E+3
'this code:          0.200022515281346E+2          0.406798347088583E+4          0.153492501096219E+4          0.387405400999218E+3
'300     1188.202    7.0000470354978|172066E+2     3.4613558020377|671029E+3     2.44357991674018|94386E+3     0.1326096164207|5693598E+3
'this code:          0.700004703549783E+3          0.346135580203776E+4          0.244357991674019E+4          0.132609616420763E+3
'500     0.435000    9.99679423176022|31789E-2     1.50817541391104|36746E+3     5.48314252654327|71044E+2     7.94488271364642|13826E+3
'this code:          0.999679423176022E-1          0.150817541391104E+4          0.548314252654328E+3          0.794488271364642E+4
'500     4.532000    9.99938124839912|78912E-1     1.66991024524550|69490E+3     5.35739001345214|77951E+2     6.82502725276895|84303E+3
'this code:          0.999938124839913             0.166991024524551E+4          0.535739001345215E+3          0.682502725276896E+4
'500     838.0250    1.00003858009|22118506E+1     3.22106218674058|33518E+3     1.2712844091476|324779E+3     2.56690918542225|39144E+3
'this code:          0.100003858009223E+2          0.322106218674058E+4          0.127128440914763E+4          0.256690918542225E+4
'500     1084.564    7.0000040549458|516645E+2     3.07437693004544|36204E+3     2.41200876574467|58352E+3     2.03237509190663|89535E+3
'this code:          0.700000405494585E+3          0.307437693004544E+4          0.241200876574468E+4          0.203237509190663E+4
'647     358.0000    2.203847557065|2149984E+1     6.18315727666|83092166E+3     2.521450782700|0715938E+2     4.32092306675500|33099E+3
'this code:          0.220384755706523E+2          0.618315727666818E+4          0.252145078270012E+3          0.432092306292204E+4
'900     0.241000    1.00062558682661|88154E-1     1.75890657044481|38520E+3     7.24027146529189|38252E+2     9.16653193855238|42681E+3
'this code:          0.100062558682662             0.175890657044481E+4          0.724027146529189E+3          0.916653193855238E+4
'900     52.61500    2.00000690372146|14551E+1     1.93510525512622|41493E+3     6.9844567383679|534276E+2     6.59070224851012|77853E+3
'this code:          0.200000690372146E+2          0.193510525512622E+4          0.698445673836795E+3          0.659070224851013E+4
'900     870.7690    7.000000057556|5402666E+2     2.66422349779369|96739E+3     2.0193360824868|338741E+3     4.17223801584632|58117E+3
'this code:          0.700000005755655E+3          0.26642234977937E+4           0.201933608248683E+4          0.417223801584632E+4

End Function

'==========================================================================
Public Function chk_OS2008_TableA3() As String

'Comparison of the library function with check values
'computed in quadruple precision (128 bit floating point)
'as published in Table A2 of

'R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner
'Mutually consistent thermodynamic potentials for fluid water, ice and seawater:
'a new standard for oceanography.
'Ocean Sci., 4, 275-291, 2008
'www.ocean-sci.net/4/275/2008/

'Table 6: Quadruple-precision results for property values in the two-phase region at the selected values
'of temperature published in Table 8 of IAPWS-95 with the coefficients given in Table 1 of this paper.
'The underlined numbers indicate the digits that, based on our tests, can reasonably be expected to be
'reproduced using double-precision code.

'Property   T = 275 K                  T = 450 K                  T = 625 K                Unit
'p_W      6.9845116|670084935279E-4  9.32203563628|20145516E-1  1.6908269318578|409807E+1  MPa
'p_Vap    6.98451166700|84935279E-4  9.322035636282|0145516E-1  1.6908269318578|409807E+1  MPa
'rho_W    9.99887406119|84984069E+2  8.9034124976167|258553E+2  5.670903851463|5254862E+2  kg m-3
'rho_Vap  5.506649185041|2278079E-3  4.812003601256|7123262     1.182902804511|5688596E+2  kg m-3
'h_W      7.759722015|5398177939E+3  7.4916158501216|908622E+5  1.6862697594697|419575E+6  J kg-1
'h_Vap    2.5042899500405|145942E+6  2.77441077988962|10429E+6  2.5507162456234|704801E+6  J kg-1
's_W      2.83094669595|19726149E+1  2.10865844688447|30194E+3  3.8019468301114|322634E+3  J kg-1 K-1
's_Vap    9.1066012052321|552768E+3  6.6092122132788|107010E+3  5.1850612079573|978994E+3  J kg-1 K-1

Dim CRLF As String, TB As String
Dim txt As String
Dim d As Double, t As Double

Dim i As Integer, j As Integer, row As String
Dim p As Double

CRLF = Chr(13) + Chr(10)
TB = Chr(9)

txt = "Properties of fluid water at the triple point" + CRLF
txt = txt + "Computed from thermodynamic potentials of the seawater library" + CRLF
txt = txt + "for publication in Ocean Science, 2009" + CRLF
txt = txt + "R. Feistel, IOW, Version " + Version + CRLF
txt = txt + "Compiled on " + CStr(Now) + CRLF + CRLF

txt = txt + "Compared with Table A3 of the article" + CRLF
txt = txt + "R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner" + CRLF
txt = txt + "Mutually consistent thermodynamic potentials for fluid water, ice and seawater:" + CRLF
txt = txt + "A new standard for oceanography." + CRLF
txt = txt + "Ocean Sci., 4, 275-291, 2008" + CRLF
txt = txt + "www.ocean-sci.net/4/275/2008/:" + CRLF + CRLF

txt = txt + "Table A3: Quadruple-precision results for property values in the two-phase region at the selected values" + CRLF
txt = txt + "of temperature published in Table 8 of IAPWS-95 with the coefficients given in Table 1 of this paper." + CRLF
txt = txt + "The underlined numbers indicate the digits that, based on our tests, can reasonably be expected to be" + CRLF
txt = txt + "reproduced using double-precision code." + CRLF + CRLF

txt = txt + "Property   T = 275 K                   T = 450 K                   T = 625 K                   Unit" + CRLF

For i = 1 To 8

  Select Case i
    Case 1:  row = "p_W        0.69845116|670084935279E-3  0.932203563628|20145516     0.16908269318578|409807E+2  MPa"
    Case 2:  row = "p_Vap      0.698451166700|84935279E-3  0.9322035636282|0145516     0.16908269318578|409807E+2  MPa"
    Case 3:  row = "rho_W      0.999887406119|84984069E+3  0.89034124976167|258553E+3  0.5670903851463|5254862E+3  kg m-3"
    Case 4:  row = "rho_Vap    0.5506649185041|2278079E-2  0.4812003601256|7123262E+1  0.1182902804511|5688596E+3  kg m-3"
    Case 5:  row = "h_W        0.7759722015|5398177939E+4  0.74916158501216|908622E+6  0.16862697594697|419575E+7  J kg-1"
    Case 6:  row = "h_Vap      0.25042899500405|145942E+7  0.277441077988962|10429E+7  0.25507162456234|704801E+7  J kg-1"
    Case 7:  row = "s_W        0.283094669595|19726149E+2  0.210865844688447|30194E+4  0.38019468301114|322634E+4  J kg-1 K-1"
    Case 8:  row = "s_Vap      0.91066012052321|552768E+4  0.66092122132788|107010E+4  0.51850612079573|978994E+4  J kg-1 K-1"
  End Select

  txt = txt + row + CRLF
  txt = txt + "this code:"
  
  For j = 1 To 3
  
    t = Choose(j, 275, 450, 625)
    set_liq_vap_eq_at_t t

    Select Case i
      Case 1: 'compute pressure in MPa
               p = 0.000001 * liq_vap_pressure_liq_si
      Case 2: 'compute pressure in MPa
               p = 0.000001 * liq_vap_pressure_vap_si
      Case 3: 'compute liquid density in kg m-3
               p = liq_vap_density_liq_si
      Case 4: 'compute vapour density in kg m-3
               p = liq_vap_density_vap_si
      Case 5: 'compute liquid enthalpy in J kg-1
               p = liq_vap_enthalpy_liq_si
      Case 6: 'compute vapour enthalpy in J kg-1
               p = liq_vap_enthalpy_vap_si
      Case 7: 'compute liquid entropy in J kg-1 K-1
               p = liq_vap_entropy_liq_si
      Case 8: 'compute vapour entropy in J kg-1 K-1
               p = liq_vap_entropy_vap_si
    End Select
    txt = txt + Left(EFormat(p) + Space(28), 28)
    
  Next j
  txt = txt + CRLF + CRLF
  
Next i

txt = txt + "Note: This computation was done with default iteration settings" + CRLF

chk_OS2008_TableA3 = txt

'output:

'Properties of fluid water at the triple point
'Computed from thermodynamic potentials of the seawater library
'for publication in Ocean Science, 2009
'R. Feistel, IOW, Version 20 Nov 2009
'Compiled on 20.11.2009 12:00:55
'
'Compared with Table A3 of the article
'R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner
'Mutually consistent thermodynamic potentials for fluid water, ice and seawater:
'A new standard for oceanography.
'Ocean Sci., 4, 275-291, 2008
'www.ocean-sci.net/4/275/2008/:
'
'Table A3: Quadruple-precision results for property values in the two-phase region at the selected values
'of temperature published in Table 8 of IAPWS-95 with the coefficients given in Table 1 of this paper.
'The underlined numbers indicate the digits that, based on our tests, can reasonably be expected to be
'reproduced using double-precision code.
'
'Property   T = 275 K                  T = 450 K                  T = 625 K                Unit
'p_W        0.69845116|670084935279E-3  0.932203563628|20145516     0.16908269318578|409807E+2  MPa
'this code: 0.698451178501321E-3        0.932203563628228           0.169082693185786E+2
'
'p_Vap      0.698451166700|84935279E-3  0.9322035636282|0145516     0.16908269318578|409807E+2  MPa
'this code: 0.698451166700846E-3        0.932203563628204           0.169082693185785E+2
'
'rho_W      0.999887406119|84984069E+3  0.89034124976167|258553E+3  0.5670903851463|5254862E+3  kg m-3
'this code: 0.999887406119852E+3        0.890341249761673E+3        0.567090385146352E+3
'
'rho_Vap    0.5506649185041|2278079E-2  0.4812003601256|7123262E+1  0.1182902804511|5688596E+3  kg m-3
'this code: 0.55066491850412E-2         0.481200360125673E+1        0.11829028045116E+3
'
'h_W        0.7759722015|5398177939E+4  0.74916158501216|908622E+6  0.16862697594697|419575E+7  J kg-1
'this code: 0.775972201555673E+4        0.74916158501217E+6         0.168626975946974E+7
'
'h_Vap      0.25042899500405|145942E+7  0.277441077988962|10429E+7  0.25507162456234|704801E+7  J kg-1
'this code: 0.250428995004052E+7        0.277441077988962E+7        0.255071624562346E+7
'
's_W        0.283094669595|19726149E+2  0.210865844688447|30194E+4  0.38019468301114|322634E+4  J kg-1 K-1
'this code: 0.283094669595258E+2        0.210865844688447E+4        0.380194683011143E+4
'
's_Vap      0.91066012052321|552768E+4  0.66092122132788|107010E+4  0.51850612079573|978994E+4  J kg-1 K-1
'this code: 0.910660120523216E+4        0.660921221327881E+4        0.518506120795738E+4
'
'Note: This computation was done with default iteration settings

End Function

'==========================================================================
Public Function chk_OS2008_TableA4() As String

'Comparison of the library function with check values
'computed in quadruple precision (128 bit floating point)
'as published in Table A4 of

'R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner
'Mutually consistent thermodynamic potentials for fluid water, ice and seawater:
'a new standard for oceanography.
'Ocean Sci., 4, 275291, 2008
'www.ocean-sci.net/4/275/2008/

'Table A4: Quadruple-precision results for the properties of pure ice at the triple point,
'the normal-pressure melting point and at T = 100 K, p = 100 MPa.
'Results correspond to Table 6 of IAPWS-06 with the corrected coefficient g00 of ice given in Table 1.
'The underlined numbers indicate the digits that, based on our tests, can reasonably be expected
'to be reproduced using double-precision code.

'Property     Value at Tt, pt             Value at Tf0, p0            Value at T, p               Unit
'T            273.16                      273.152519                  100 K
'p            611.657                     101325                      1.0E+8 Pa
'g            6.11784134|97053682445E-1   1.0134274068|780095492E+2  -2.22296513087615|83787E+5   J kg-1
'(dg/dp)_T    1.09085812736640|05954E-3   1.09084388214311|03431E-3   1.061933892596491|4671E-3   m3 kg-1
'(dg/dT)_p    1.22069433939686|94463E+3   1.22076932549695|58410E+3   2.61195122588784|94194E+3   J kg-1 K-1
'(d2g/dp2)_T -1.28495941571494|44477E-13 -1.28485364928455|47078E-13 -9.41807981760913|98970E-14  m3 kg-1 Pa-1
' d2g/dpdT    1.74387964699598|04034E-7   1.74362219972159|07057E-7   2.74505162488107|67755E-8   m3 kg-1 K-1
'(d2g/dT2)_p -7.6760298587506|742565     -7.67598233364797|66851     -8.66333195516833|78537      J kg-1 K-2
'h           -3.33444253965513|88743E+5  -3.33354873636736|61384E+5  -4.83491635676400|77981E+5   J kg-1
'f           -5.5446874|640013664515E-2  -9.18701567|03497005930     -3.2848990234726|498458E+5   J kg-1
'u           -3.3344492119652|349798E+5  -3.33465403393094|76449E+5  -5.8968502493604|992651E+5   J kg-1
's           -1.22069433939686|94463E+3  -1.22076932549695|58410E+3  -2.61195122588784|94194E+3   J kg-1 K-1
'cp           2.09678431621633|41799E+3   2.09671391023544|32908E+3   8.66333195516833|78537E+2   J kg-1 K-1
'rho          9.16709492199728|64196E+2   9.16721463419096|00300E+2   9.41678203296572|99139E+2   kg m-3
'alpha        1.59863102565512|72353E-4   1.59841589457879|99191E-4   2.58495528207435|06386E-5   K-1
'beta         1.35714764658593|92367E+6   1.35705899321101|05876E+6   2.91466166993892|77480E+5   Pa K-1
'kappa-T      1.17793449347730|67405E-10  1.17785291765150|29239E-10  8.86880048114989|07193E-11  Pa-1
'kappa-s      1.14161597778630|57425E-10  1.14154442556498|04160E-10  8.86060982686811|64748E-11  Pa-1

Dim CRLF As String, TB As String
Dim txt As String
Dim t As Double, x As Double, p As Double

Dim i As Integer, j As Integer, row As String
Dim lrow(5) As Integer

CRLF = Chr(13) + Chr(10)
TB = Chr(9)

txt = "Properties of ice Ih" + CRLF
txt = txt + "Computed from thermodynamic potentials of the seawater library" + CRLF
txt = txt + "for publication in Ocean Science, 2009" + CRLF
txt = txt + "R. Feistel, IOW, Version " + Version + CRLF
txt = txt + "Compiled on " + CStr(Now) + CRLF + CRLF

txt = txt + "Compared with Table A5 of the article" + CRLF
txt = txt + "R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner" + CRLF
txt = txt + "Mutually consistent thermodynamic potentials for fluid water, ice and seawater:" + CRLF
txt = txt + "A new standard for oceanography." + CRLF
txt = txt + "Ocean Sci., 4, 275-291, 2008" + CRLF
txt = txt + "www.ocean-sci.net/4/275/2008/:" + CRLF + CRLF

txt = txt + "Table A4: Quadruple-precision results for the properties of pure ice at the triple point," + CRLF
txt = txt + "the normal-pressure melting point and at T = 100 K, p = 100 MPa." + CRLF
txt = txt + "Results correspond to Table 6 of IAPWS-06 with the corrected coefficient g00 of ice given in Table 1." + CRLF
txt = txt + "The separated numbers indicate the digits that, based on our tests, can reasonably be expected" + CRLF
txt = txt + "to be reproduced using double-precision code." + CRLF

For i = 0 To 18
  txt = txt + CRLF
  For j = 1 To 5
    Select Case i
      Case 0:  row = Choose(j, "Property    ", " Value at Tt, pt             ", " Value at Tf0, p0            ", " Value at T, p                ", "Unit")
      Case 1:  row = Choose(j, "T           ", " 273.16                      ", " 273.152519                  ", " 100                          ", "K")
      Case 2:  row = Choose(j, "p           ", " 611.657                     ", " 101325                      ", " 1.0E+8                       ", "Pa")
      Case 3:  row = Choose(j, "g           ", " 0.611784134|97053682445     ", " 0.10134274068|780095492E+3  ", "-0.222296513087615|83787E+6   ", "J kg-1")
      Case 4:  row = Choose(j, "(dg/dp)_T   ", " 0.109085812736640|05954E-2  ", " 0.109084388214311|03431E-2  ", " 0.1061933892596491|4671E-2   ", "m3 kg-1")
      Case 5:  row = Choose(j, "(dg/dT)_p   ", " 0.122069433939686|94463E+4  ", " 0.122076932549695|58410E+4  ", " 0.261195122588784|94194E+4   ", "J kg-1 K-1")
      Case 6:  row = Choose(j, "(d2g/dp2)_T ", "-0.128495941571494|44477E-12 ", "-0.128485364928455|47078E-12 ", "-0.941807981760913|98970E-13  ", "m3 kg-1 Pa-1")
      Case 7:  row = Choose(j, " d2g/dpdT   ", " 0.174387964699598|04034E-6  ", " 0.174362219972159|07057E-6  ", " 0.274505162488107|67755E-7   ", "m3 kg-1 K-1")
      Case 8:  row = Choose(j, "(d2g/dT2)_p ", "-0.76760298587506|742565E+1  ", "-0.767598233364797|66851E+1  ", "-0.866333195516833|78537E+1   ", "J kg-1 K-2")
      Case 9:  row = Choose(j, "h           ", "-0.333444253965513|88743E+6  ", "-0.333354873636736|61384E+6  ", "-0.483491635676400|77981E+6   ", "J kg-1")
      Case 10: row = Choose(j, "f           ", "-0.55446874|640013664515E-1  ", "-0.918701567|03497005930E+1  ", "-0.32848990234726|498458E+6   ", "J kg-1")
      Case 11: row = Choose(j, "u           ", "-0.33344492119652|349798E+6  ", "-0.333465403393094|76449E+6  ", "-0.58968502493604|992651E+6   ", "J kg-1")
      Case 12: row = Choose(j, "s           ", "-0.122069433939686|94463E+4  ", "-0.122076932549695|58410E+4  ", "-0.261195122588784|94194E+4   ", "J kg-1 K-1")
      Case 13: row = Choose(j, "cp          ", " 0.209678431621633|41799E+4  ", " 0.209671391023544|32908E+4  ", " 0.866333195516833|78537E+3   ", "J kg-1 K-1")
      Case 14: row = Choose(j, "rho         ", " 0.916709492199728|64196E+3  ", " 0.916721463419096|00300E+3  ", " 0.941678203296572|99139E+3   ", "kg m-3")
      Case 15: row = Choose(j, "alpha       ", " 0.159863102565512|72353E-3  ", " 0.159841589457879|99191E-3  ", " 0.258495528207435|06386E-4   ", "K-1")
      Case 16: row = Choose(j, "beta        ", " 0.135714764658593|92367E+7  ", " 0.135705899321101|05876E+7  ", " 0.291466166993892|77480E+6   ", "Pa K-1")
      Case 17: row = Choose(j, "kappa-T     ", " 0.117793449347730|67405E-9  ", " 0.117785291765150|29239E-9  ", " 0.886880048114989|07193E-10  ", "Pa-1")
      Case 18: row = Choose(j, "kappa-s     ", " 0.114161597778630|57425E-9  ", " 0.114154442556498|04160E-9  ", " 0.886060982686811|64748E-10  ", "Pa-1")
    End Select
    
    lrow(j) = Len(row)
    If j > 1 Then txt = txt + TB
    txt = txt + row
  Next j
  
  If i > 2 Then
  
    txt = txt + CRLF + "this code:  "
    For j = 2 To 4
    
      Select Case j
        Case 2: t = 273.16:     p = 611.657
        Case 3: t = 273.152519: p = 101325
        Case 4: t = 100:        p = 100000000#
      End Select

      Select Case i
        Case 3: x = ice_g_si(0, 0, t, p)
        Case 4: x = ice_g_si(0, 1, t, p)
        Case 5: x = ice_g_si(1, 0, t, p)
        Case 6: x = ice_g_si(0, 2, t, p)
        Case 7: x = ice_g_si(1, 1, t, p)
        Case 8: x = ice_g_si(2, 0, t, p)
        Case 9: x = ice_enthalpy_si(t, p)
        Case 10: x = ice_helmholtz_energy_si(t, p)
        Case 11: x = ice_internal_energy_si(t, p)
        Case 12: x = ice_entropy_si(t, p)
        Case 13: x = ice_cp_si(t, p)
        Case 14: x = ice_density_si(t, p)
        Case 15: x = ice_expansion_si(t, p)
        Case 16: x = ice_p_coefficient_si(t, p)
        Case 17: x = ice_kappa_t_si(t, p)
        Case 18: x = ice_kappa_s_si(t, p)
      End Select
      
      txt = txt + TB + Left(EFormat(x) + Space(lrow(j)), lrow(j))
    Next j
    txt = txt + TB + row + CRLF
  End If
Next i

chk_OS2008_TableA4 = txt


'Properties of ice Ih
'Computed from thermodynamic potentials of the seawater library
'for publication in Ocean Science, 2009
'R. Feistel, IOW, Version 11 Apr 2009
'Compiled on 11.04.2009 16:06:00
'
'Compared with Table A5 of the article
'R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner
'Mutually consistent thermodynamic potentials for fluid water, ice and seawater:
'A new standard for oceanography.
'Ocean Sci., 4, 275-291, 2008
'www.ocean-sci.net/4/275/2008/:
'
'Table A4: Quadruple-precision results for the properties of pure ice at the triple point,
'the normal-pressure melting point and at T = 100 K, p = 100 MPa.
'Results correspond to Table 6 of IAPWS-06 with the corrected coefficient g00 of ice given in Table 1.
'The separated numbers indicate the digits that, based on our tests, can reasonably be expected
'to be reproduced using double-precision code.
'
'Property       Value at Tt, pt               Value at Tf0, p0              Value at T, p                Unit
'T              273.16                        273.152519                    100                          K
'p              611.657                       101325                        1.0E+8                       Pa
'g              6.11784134|97053682445E-1     1.0134274068|780095492E+2    -2.22296513087615|83787E+5    J kg-1
'this code:     0.611784134525806             0.101342740687309E+3         -0.222296513087616E+6         J kg-1
'(dg/dp)_T      1.09085812736640|05954E-3     1.09084388214311|03431E-3     1.061933892596491|4671E-3    m3 kg-1
'this code:     0.10908581273664E-2           0.109084388214311E-2          0.106193389259649E-2         m3 kg-1
'(dg/dT)_p      1.22069433939686|94463E+3     1.22076932549695|58410E+3     2.61195122588784|94194E+3    J kg-1 K-1
'this code:     0.122069433939687E+4          0.122076932549696E+4          0.261195122588785E+4         J kg-1 K-1
'(d2g/dp2)_T   -1.28495941571494|44477E-13   -1.28485364928455|47078E-13   -9.41807981760913|98970E-14   m3 kg-1 Pa-1
'this code:    -0.128495941571494E-12        -0.128485364928455E-12        -0.941807981760913E-13        m3 kg-1 Pa-1
' d2g/dpdT      1.74387964699598|04034E-7     1.74362219972159|07057E-7     2.74505162488107|67755E-8    m3 kg-1 K-1
'this code:     0.174387964699598E-6          0.174362219972159E-6          0.274505162488106E-7         m3 kg-1 K-1
'(d2g/dT2)_p   -7.6760298587506|742565       -7.67598233364797|66851       -8.66333195516833|78537       J kg-1 K-2
'this code:    -0.767602985875067E+1         -0.767598233364798E+1         -0.866333195516834E+1         J kg-1 K-2
'h             -3.33444253965513|88743E+5    -3.33354873636736|61384E+5    -4.83491635676400|77981E+5    J kg-1
'this code:    -0.333444253965514E+6         -0.333354873636737E+6         -0.483491635676401E+6         J kg-1
'f             -5.5446874|640013664515E-2    -9.18701567|03497005930       -3.2848990234726|498458E+5    J kg-1
'this code:    -0.554468750847448E-1         -0.918701567084181E+1         -0.328489902347266E+6         J kg-1
'u             -3.3344492119652|349798E+5    -3.33465403393094|76449E+5    -5.8968502493604|992651E+5    J kg-1
'this code:    -0.333444921196524E+6         -0.333465403393095E+6         -0.589685024936051E+6         J kg-1
's             -1.22069433939686|94463E+3    -1.22076932549695|58410E+3    -2.61195122588784|94194E+3    J kg-1 K-1
'this code:    -0.122069433939687E+4         -0.122076932549696E+4         -0.261195122588785E+4         J kg-1 K-1
'cp             2.09678431621633|41799E+3     2.09671391023544|32908E+3     8.66333195516833|78537E+2    J kg-1 K-1
'this code:     0.209678431621633E+4          0.209671391023544E+4          0.866333195516834E+3         J kg-1 K-1
'rho            9.16709492199728|64196E+2     9.16721463419096|00300E+2     9.41678203296572|99139E+2    kg m-3
'this code:     0.916709492199729E+3          0.916721463419096E+3          0.941678203296573E+3         kg m-3
'alpha          1.59863102565512|72353E-4     1.59841589457879|99191E-4     2.58495528207435|06386E-5    K-1
'this code:     0.159863102565513E-3          0.15984158945788E-3           0.258495528207433E-4         K-1
'beta           1.35714764658593|92367E+6     1.35705899321101|05876E+6     2.91466166993892|77480E+5    Pa K-1
'this code:     0.135714764658594E+7          0.135705899321101E+7          0.291466166993891E+6         Pa K-1
'kappa-T        1.17793449347730|67405E-10    1.17785291765150|29239E-10    8.86880048114989|07193E-11   Pa-1
'this code:     0.117793449347731E-9          0.11778529176515E-9           0.886880048114989E-10        Pa-1
'kappa-s        1.14161597778630|57425E-10    1.14154442556498|04160E-10    8.86060982686811|64748E-11   Pa-1
'this code:     0.11416159777863E-9           0.114154442556498E-9          0.886060982686811E-10        Pa-1

End Function

'==========================================================================
Public Function chk_OS2008_TableA5() As String

'Comparison of the library function with check values
'computed in quadruple precision (128 bit floating point)
'as published in Table A5 of

'R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner
'Mutually consistent thermodynamic potentials for fluid water, ice and seawater:
'a new standard for oceanography.
'Ocean Sci., 4, 275291, 2008
'www.ocean-sci.net/4/275/2008/

'Table A5: Quadruple-precision results for the water part, saline part and total properties published in
'Table 8a of IAPWS-08 with the coefficients given in Table 1 of this paper.
'Properties at SA = Sn = 0.035 165 04 kg kg-1, T = T0 = 273.15 K, p = p0 = 0.101325 MPa.
'The underlined numbers indicate the digits that, based on our tests, can reasonably
'be expected to be reproduced using double-precision code.

'Property        Water part                   Saline part                  Property of seawater        Unit
'g               1.01342741|72939062882E+2   -1.0134274172939|062882E+2   -2.E-29 |  4.E-9             J kg-1
'(dg/dS)_T,p     0.0                          6.39974067312299|04527E+4    6.39974067312299|04527E+4   J kg-1
'(dg/dT)_S,p     1.47643376|34625266531E-1   -1.47643376346|25266531E-1    7.E-32 | -6.E-11            J kg-1 K-1
'(dg/dp)_S,T     1.00015693912169|26347E-3   -2.74957224268432|87457E-5    9.7266121669484|934729E-4   m3 kg-1
'(d2g/dSdp)_T    0.0                         -7.59615411515308|89445E-4   -7.59615411515308|89445E-4   m3 kg-1
'(d2g/dT 2)_S,p -1.54473542319|77289339E+1    8.5286115117592|251026E-1   -1.459449308080|1366829E+1   J kg-1 K-2
'(d2g/dTdp)_S   -6.777003178|6558265755E-8    1.1928678741395|764132E-7    5.1516755627|399375563E-8   m3 kg-1 K-1
'(d2g/dp2)_S,T  -5.08928894643|49017238E-13   5.8153517233288|224927E-14  -4.507753774102|0194745E-13  m3 kg-1 Pa-1
'h               6.10139534|80411713295E+1   -6.10139534804|11713295E+1   -4.E-29 |   2.E-8            J kg-1
'f               1.8398|728851226087838E-3   -9.8556737654490|732723E+1   -9.855489778|1605610114E+1   J kg-1
'u              -4.03269483|76093792920E+1   -5.82279494055|11817194E+1   -9.85548977|81605610114E+1   J kg-1
's              -1.47643376|34625266530E-1    1.47643376346|25266531E-1   -7.E-32 |   6.E-11           J kg-1 K-1
'rho             9.9984308550433|049647E+2    -                            1.02810719995400|78127E+3   kg m-3
'cp              4.21944480846|45965831E+3   -2.3295902344370|323368E+2    3.98648578502|08933494E+3   J kg-1 K-1
'w               1.40238253108|82262606E+3    -                            1.44900246362|14836206E+3   m s-1
'W              1.01342741|72939062882E+2   -2.35181410932935|94707E+3   -2.250471367|5999688419E+3   J kg-1

Dim CRLF As String, TB As String
Dim txt As String
Dim S As Double, t As Double, x As Double, p As Double

Dim i As Integer, j As Integer, row As String
Dim tol_liq As Double
Dim lrow(5) As Integer

CRLF = Chr(13) + Chr(10)
TB = Chr(9)

txt = "Properties of water and seawater" + CRLF
txt = txt + "Computed from thermodynamic potentials of the seawater library" + CRLF
txt = txt + "for publication in Ocean Science, 2009" + CRLF
txt = txt + "R. Feistel, IOW, Version " + Version + CRLF
txt = txt + "Compiled on " + CStr(Now) + CRLF + CRLF

txt = txt + "Compared with Table A5 of the article" + CRLF
txt = txt + "R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner" + CRLF
txt = txt + "Mutually consistent thermodynamic potentials for fluid water, ice and seawater:" + CRLF
txt = txt + "A new standard for oceanography." + CRLF
txt = txt + "Ocean Sci., 4, 275-291, 2008" + CRLF
txt = txt + "www.ocean-sci.net/4/275/2008/:" + CRLF + CRLF

txt = txt + "Table A5: Quadruple-precision results for the water part, saline part and total properties " + CRLF
txt = txt + "published in Table 8a of IAPWS-08 with the coefficients given in Table 1 of this paper.  " + CRLF
txt = txt + "Properties at SA = Sn = 0.035 165 04 kg kg-1, T = T0 = 273.15 K, p = p0 = 0.101325 MPa. g0, g200 and g210 " + CRLF
txt = txt + "are as given in Table 1.  The separated numbers indicate the digits that, based on our tests, can " + CRLF
txt = txt + "reasonably be expected to be reproduced using double-precision code." + CRLF + CRLF

S = 0.03516504
t = 273.15
p = 101325

tol_liq = get_it_ctrl_density("tol_liq_dens")
set_it_ctrl_density "tol_liq_dens", -0.000000000000001

For i = 0 To 16
  txt = txt + CRLF
  For j = 1 To 5
    Select Case i
      Case 0:  row = Choose(j, "Property       ", " Water part                  ", " Saline part                 ", " Property of seawater        ", "Unit")
      Case 1:  row = Choose(j, "g              ", " 0.101342741|72939062882E+3  ", "-0.10134274172939|062882E+3  ", "-2.E-29 |  4.E-9             ", "J kg-1")
      Case 2:  row = Choose(j, "(dg/dS)_T,p    ", " 0.0                         ", " 0.639974067312299|04527E+5  ", " 0.639974067312299|04527E+5  ", "J kg-1")
      Case 3:  row = Choose(j, "(dg/dT)_S,p    ", " 0.147643376|34625266531     ", "-0.147643376346|25266531     ", " 7.E-32 | -6.E-11            ", "J kg-1 K-1")
      Case 4:  row = Choose(j, "(dg/dp)_S,T    ", " 0.100015693912169|26347E-2  ", "-0.274957224268432|87457E-4  ", " 0.97266121669484|934729E-3  ", "m3 kg-1")
      Case 5:  row = Choose(j, "(d2g/dSdp)_T   ", " 0.0                         ", "-0.759615411515308|89445E-3  ", "-0.759615411515308|89445E-3  ", "m3 kg-1")
      Case 6:  row = Choose(j, "(d2g/dT 2)_S,p ", "-0.154473542319|77289339E+2  ", " 0.85286115117592|251026     ", "-0.1459449308080|1366829E+2  ", "J kg-1 K-2")
      Case 7:  row = Choose(j, "(d2g/dTdp)_S   ", "-0.6777003178|6558265755E-7  ", " 0.11928678741395|764132E-6  ", " 0.51516755627|399375563E-7  ", "m3 kg-1 K-1")
      Case 8:  row = Choose(j, "(d2g/dp2)_S,T  ", "-0.508928894643|49017238E-12 ", " 0.58153517233288|224927E-13 ", "-0.4507753774102|0194745E-12 ", "m3 kg-1 Pa-1")
      Case 9:  row = Choose(j, "h              ", " 0.610139534|80411713295E+2  ", "-0.610139534804|11713295E+2  ", "-4.E-29 |   2.E-8            ", "J kg-1")
      Case 10: row = Choose(j, "f              ", " 0.18398|728851226087838E-2  ", "-0.98556737654490|732723E+2  ", "-0.9855489778|1605610114E+2  ", "J kg-1")
      Case 11: row = Choose(j, "u              ", "-0.403269483|76093792920E+2  ", "-0.582279494055|11817194E+2  ", "-0.985548977|81605610114E+2  ", "J kg-1")
      Case 12: row = Choose(j, "s              ", "-0.147643376|34625266530     ", " 0.147643376346|25266531     ", "-7.E-32 |   6.E-11           ", "J kg-1 K-1")
      Case 13: row = Choose(j, "rho            ", " 0.99984308550433|049647E+3  ", " -                           ", " 0.102810719995400|78127E+4  ", "kg m-3")
      Case 14: row = Choose(j, "cp             ", " 0.421944480846|45965831E+4  ", "-0.23295902344370|323368E+3  ", " 0.398648578502|08933494E+4  ", "J kg-1 K-1")
      Case 15: row = Choose(j, "w              ", " 0.140238253108|82262606E+4  ", " -                           ", " 0.144900246362|14836206E+4  ", "m s-1")
      Case 16: row = Choose(j, "W             ", " 0.101342741|72939062882E+3  ", "-0.235181410932935|94707E+4  ", "-0.2250471367|5999688419E+4  ", "J kg-1")
    End Select
    
    lrow(j) = Len(row)
    If j > 1 Then txt = txt + TB
    txt = txt + row
  Next j
  
  If i > 0 Then
    txt = txt + CRLF + "this code:    "
    For j = 2 To 4
      Select Case i
        Case 1
          Select Case j
            Case 2: x = liq_g_si(0, 0, t, p)
            Case 3: x = sal_g_si(0, 0, 0, S, t, p)
            Case 4: x = sea_g_si(0, 0, 0, S, t, p)
          End Select
          
        Case 2
          Select Case j
            Case 2: x = 0
            Case 3: x = sal_g_si(1, 0, 0, S, t, p)
            Case 4: x = sea_g_si(1, 0, 0, S, t, p)
          End Select
          
        Case 3
          Select Case j
            Case 2: x = liq_g_si(1, 0, t, p)
            Case 3: x = sal_g_si(0, 1, 0, S, t, p)
            Case 4: x = sea_g_si(0, 1, 0, S, t, p)
          End Select
          
        Case 4
          Select Case j
            Case 2: x = liq_g_si(0, 1, t, p)
            Case 3: x = sal_g_si(0, 0, 1, S, t, p)
            Case 4: x = sea_g_si(0, 0, 1, S, t, p)
          End Select
          
        Case 5
          Select Case j
            Case 2: x = 0
            Case 3: x = sal_g_si(1, 0, 1, S, t, p)
            Case 4: x = sea_g_si(1, 0, 1, S, t, p)
          End Select
          
        Case 6
          Select Case j
            Case 2: x = liq_g_si(2, 0, t, p)
            Case 3: x = sal_g_si(0, 2, 0, S, t, p)
            Case 4: x = sea_g_si(0, 2, 0, S, t, p)
          End Select
          
        Case 7
          Select Case j
            Case 2: x = liq_g_si(1, 1, t, p)
            Case 3: x = sal_g_si(0, 1, 1, S, t, p)
            Case 4: x = sea_g_si(0, 1, 1, S, t, p)
          End Select
          
        Case 8
          Select Case j
            Case 2: x = liq_g_si(0, 2, t, p)
            Case 3: x = sal_g_si(0, 0, 2, S, t, p)
            Case 4: x = sea_g_si(0, 0, 2, S, t, p)
          End Select

        Case 9
          Select Case j
            Case 2: x = liq_enthalpy_si(t, p)
            Case 3: x = sal_g_si(0, 0, 0, S, t, p) - t * sal_g_si(0, 1, 0, S, t, p)
            Case 4: x = sea_enthalpy_si(S, t, p)
          End Select
          
        Case 10
          Select Case j
            Case 2: x = liq_g_si(0, 0, t, p) - p * liq_g_si(0, 1, t, p)
            Case 3: x = sal_g_si(0, 0, 0, S, t, p) - p * sal_g_si(0, 0, 1, S, t, p)
            Case 4: x = sea_g_si(0, 0, 0, S, t, p) - p * sea_g_si(0, 0, 1, S, t, p)
          End Select
          
        Case 11
          Select Case j
            Case 2: x = liq_internal_energy_si(t, p)
            Case 3: x = sal_g_si(0, 0, 0, S, t, p) - p * sal_g_si(0, 0, 1, S, t, p) - t * sal_g_si(0, 1, 0, S, t, p)
            Case 4: x = sea_internal_energy_si(S, t, p)
          End Select
          
        Case 12
          Select Case j
            Case 2: x = liq_entropy_si(t, p)
            Case 3: x = -sal_g_si(0, 1, 0, S, t, p)
            Case 4: x = sea_entropy_si(S, t, p)
          End Select
          
        Case 13
          Select Case j
            Case 2: x = liq_density_si(t, p)
            Case 3: x = 0
            Case 4: x = sea_density_si(S, t, p)
          End Select
          
        Case 14
          Select Case j
            Case 2: x = liq_cp_si(t, p)
            Case 3: x = -t * sal_g_si(0, 2, 0, S, t, p)
            Case 4: x = sea_cp_si(S, t, p)
          End Select
          
        Case 15
          Select Case j
            Case 2: x = liq_soundspeed_si(t, p)
            Case 3: x = 0
            Case 4: x = sea_soundspeed_si(S, t, p)
          End Select
          
        Case 16
          Select Case j
            Case 2: x = liq_gibbs_energy_si(t, p)
            Case 3: x = sal_chempot_h2o_si(S, t, p)
            Case 4: x = sea_chempot_h2o_si(S, t, p)
          End Select

      End Select
      
      txt = txt + TB + Left(EFormat(x) + Space(lrow(j)), lrow(j))
    Next j
    txt = txt + TB + row + CRLF
  End If
Next i

txt = txt + CRLF + CRLF + "Note: For this computation, the iteration accuracy was increased as"
txt = txt + CRLF + "set_it_ctrl_density " + Chr(34) + "tol_liq_dens" + Chr(34) + ", -1E-15"

txt = txt + CRLF + CRLF + "The default iteration accuracy is as low as 0.1 ppm, i.e. equivalent to"
txt = txt + CRLF + "set_it_ctrl_density " + Chr(34) + "tol_liq_dens" + Chr(34) + ", -1E-7"

chk_OS2008_TableA5 = txt

set_it_ctrl_density "tol_liq_dens", tol_liq


'Properties of water and seawater
'Computed from thermodynamic potentials of the seawater library
'for publication in Ocean Science, 2009
'R. Feistel, IOW, Version 11 Apr 2009
'Compiled on 11.04.2009 12:40:38
'
'Compared with Table A5 of the article
'R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner
'Mutually consistent thermodynamic potentials for fluid water, ice and seawater:
'A new standard for oceanography.
'Ocean Sci., 4, 275-291, 2008
'www.ocean-sci.net/4/275/2008/:
'
'Table A5: Quadruple-precision results for the water part, saline part and total properties
'published in Table 8a of IAPWS-08 with the coefficients given in Table 1 of this paper.
'Properties at SA = Sn = 0.035 165 04 kg kg-1, T = T0 = 273.15 K, p = p0 = 0.101325 MPa. g0, g200 and g210
'are as given in Table 1.  The separated numbers indicate the digits that, based on our tests, can
'reasonably be expected to be reproduced using double-precision code.
'
'
'Property         Water part                    Saline part                   Property of seawater         Unit
'g                1.01342741|72939062882E+2    -1.0134274172939|062882E+2    -2.E-29 |  4.E-9              J kg-1
'this code:       0.101342741730827E+3         -0.10134274172939E+3           0.143698741794651E-8         J kg-1
'(dg/dS)_T,p      0.0                           6.39974067312299|04527E+4     6.39974067312299|04527E+4    J kg-1
'this code:       0.0                           0.639974067312299E+5          0.639974067312299E+5         J kg-1
'(dg/dT)_S,p      1.47643376|34625266531E-1    -1.47643376346|25266531E-1     7.E-32 | -6.E-11             J kg-1 K-1
'this code:       0.147643376311825            -0.147643376346251            -0.344255735029719E-10        J kg-1 K-1
'(dg/dp)_S,T      1.00015693912169|26347E-3    -2.74957224268432|87457E-5     9.7266121669484|934729E-4    m3 kg-1
'this code:       0.100015693912169E-2         -0.274957224268433E-4          0.972661216694849E-3         m3 kg-1
'(d2g/dSdp)_T     0.0                          -7.59615411515308|89445E-4    -7.59615411515308|89445E-4    m3 kg-1
'this code:       0.0                          -0.759615411515309E-3         -0.759615411515309E-3         m3 kg-1
'(d2g/dT 2)_S,p  -1.54473542319|77289339E+1     8.5286115117592|251026E-1    -1.459449308080|1366829E+1    J kg-1 K-2
'this code:      -0.154473542319782E+2          0.852861151175923            -0.145944930808023E+2         J kg-1 K-2
'(d2g/dTdp)_S    -6.777003178|6558265755E-8     1.1928678741395|764132E-7     5.1516755627|399375563E-8    m3 kg-1 K-1
'this code:      -0.677700317864069E-7          0.119286787413958E-6          0.515167556275508E-7         m3 kg-1 K-1
'(d2g/dp2)_S,T   -5.08928894643|49017238E-13    5.8153517233288|224927E-14   -4.507753774102|0194745E-13   m3 kg-1 Pa-1
'this code:      -0.508928894643498E-12         0.581535172332882E-13        -0.45077537741021E-12         m3 kg-1 Pa-1
'h                6.10139534|80411713295E+1    -6.10139534804|11713295E+1    -4.E-29 |   2.E-8             J kg-1
'this code:       0.610139534912514E+2         -0.610139534804111E+2          0.108403328202833E-7         J kg-1
'f                1.8398|728851226087838E-3    -9.8556737654490|732723E+1    -9.855489778|1605610114E+1    J kg-1
'this code:       0.183987432106544E-2         -0.985567376544896E+2         -0.985548977801686E+2         J kg-1
'u               -4.03269483|76093792920E+1    -5.82279494055|11817194E+1    -9.85548977|81605610114E+1    J kg-1
'this code:      -0.403269483655095E+2         -0.582279494055112E+2         -0.985548977707652E+2         J kg-1
's               -1.47643376|34625266530E-1     1.47643376346|25266531E-1    -7.E-32 |   6.E-11            J kg-1 K-1
'this code:      -0.147643376311825             0.147643376346251             0.344255735029719E-10        J kg-1 K-1
'rho              9.9984308550433|049647E+2     -                             1.02810719995400|78127E+3    kg m-3
'this code:       0.999843085504331E+3          0.0                           0.102810719995401E+4         kg m-3
'cp               4.21944480846|45965831E+3    -2.3295902344370|323368E+2     3.98648578502|08933494E+3    J kg-1 K-1
'this code:       0.421944480846485E+4         -0.232959023443703E+3          0.398648578502115E+4         J kg-1 K-1
'w                1.40238253108|82262606E+3     -                             1.44900246362|14836206E+3    m s-1
'this code:       0.140238253108821E+4          0.0                           0.144900246362147E+4         m s-1
'W               1.01342741|72939062882E+2    -2.35181410932935|94707E+3    -2.250471367|5999688419E+3    J kg-1
'this code:       0.101342741730827E+3         -0.235181410932936E+4         -0.225047136759853E+4         J kg-1
'
'Note: For this computation, the iteration accuracy was increased as
'set_it_ctrl_density "tol_liq_dens", -1E-15
'
'The default iteration accuracy is as low as 0.1 ppm, i.e. equivalent to
'set_it_ctrl_density "tol_liq_dens", -1E-7


End Function

'==========================================================================
Public Function chk_OS2008_TableA6() As String

'Comparison of the library function with check values
'computed in quadruple precision (128 bit floating point)
'as published in Table A6 of

'R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner
'Mutually consistent thermodynamic potentials for fluid water, ice and seawater:
'a new standard for oceanography.
'Ocean Sci., 4, 275291, 2008
'www.ocean-sci.net/4/275/2008/

'Table A6: Quadruple-precision results for the water part, saline part and total properties published in
'Table 8b of IAPWS-08 with the coefficients given in Table 1 of this paper.
'Properties at SA = 0.1 kg kg-1 = 100 g kg-1, T = 353 K, p = p0 = 0.101325 MPa.
'This point is located in the regions with restricted validity.
'The separated numbers indicate the digits that, based on our tests, can reasonably
'be expected to be reproduced using double-precision code.

'Property        Water part                   Saline part                  Property of seawater        Unit
'g              -4.46114968995|69225658E+4    1.5087174003705|274197E+4   -2.95243228958|63951462E+4   J kg-1
'(dg/dS)_T,p     0.0                          2.51957275851412|84916E+5    2.51957275851412|84916E+5   J kg-1
'(dg/dT)_S,p    -1.0737599318874|549194E+3    1.56230907404290|99119E+2   -9.175290244831|6392819E+2   J kg-1 K-1
'(dg/dp)_S,T     1.02892955635611|00700E-3   -5.7922728577125|827635E-5    9.7100682777898|424241E-4   m3 kg-1
'(d2g/dSdp)_T    0.0                         -3.0595780244233|918005E-4   -3.0595780244233|918005E-4   m3 kg-1
'(d2g/dT2)_S,p  -1.1888500004754|679683E+1    1.27922649315507|38238E+0   -1.0609273511599|605860E+1   J kg-1 K-2
'(d2g/dTdp)_S    6.59051552339|36161134E-7    8.0306159575153|091195E-7    1.462113148090|8925233E-6   m3 kg-1 K-1
'(d2g/dp2)_S,T  -4.746728193610|6728585E-13   2.1308615424374|026844E-13  -2.61586665117|32701741E-13  m3 kg-1 Pa-1
'h               3.344257590567|0236088E+5   -4.006233631000|9445692E+4    2.94363422746|6929152E+5    J kg-1
'f              -4.47157531868|67008511E+4    1.5093043024178|351471E+4   -2.96227101626|8865704E+4    J kg-1
'u               3.343215027694|0457803E+5   -4.0056467289536|368418E+4    2.942650354798|6820961E+5   J kg-1
's               1.0737599318874|549194E+3   -1.56230907404290|99119E+2    9.175290244831|6392819E+2   J kg-1 K-1
'rho             9.7188383191307|839206E+2    -                            1.02985887574790|05489E+3   kg m-3
'cp              4.1966405016784|019283E+3   -4.51566952083741|05981E+2    3.745073549594|6608685E+3   J kg-1 K-1
'w               1.5544629665347|162190E+3    -                            3.9612783529|250024954E+3   m s-1
'W             -4.46114968995|69225658E+4   -1.01085535814360|10719E+4   -5.47200504810|05236377E+4   J kg-1

Dim CRLF As String, TB As String
Dim txt As String
Dim S As Double, t As Double, x As Double, p As Double

Dim i As Integer, j As Integer, row As String
Dim tol_liq As Double
Dim lrow(5) As Integer

CRLF = Chr(13) + Chr(10)
TB = Chr(9)

txt = "Properties of water and seawater" + CRLF
txt = txt + "Computed from thermodynamic potentials of the seawater library" + CRLF
txt = txt + "for publication in Ocean Science, 2009" + CRLF
txt = txt + "R. Feistel, IOW, Version " + Version + CRLF
txt = txt + "Compiled on " + CStr(Now) + CRLF + CRLF

txt = txt + "Compared with Table A6 of the article" + CRLF
txt = txt + "R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner" + CRLF
txt = txt + "Mutually consistent thermodynamic potentials for fluid water, ice and seawater:" + CRLF
txt = txt + "A new standard for oceanography." + CRLF
txt = txt + "Ocean Sci., 4, 275-291, 2008" + CRLF
txt = txt + "www.ocean-sci.net/4/275/2008/:" + CRLF + CRLF

txt = txt + "Table A6: Quadruple-precision results for the water part, saline part and total properties " + CRLF
txt = txt + "published in Table 8b of IAPWS-08 with the coefficients given in Table 1 of this paper.  " + CRLF
txt = txt + "Properties at SA = 0.1 kg kg-1 = 100 g kg-1, T = 353 K, p = p0 = 0.101325 MPa. " + CRLF
txt = txt + "This point is located in the regions with restricted validity" + CRLF
txt = txt + "The separated numbers indicate the digits that, based on our tests, can " + CRLF
txt = txt + "reasonably be expected to be reproduced using double-precision code." + CRLF

S = 0.1
t = 353
p = 101325

tol_liq = get_it_ctrl_density("tol_liq_dens")
set_it_ctrl_density "tol_liq_dens", -0.000000000000001

For i = 0 To 16
  txt = txt + CRLF
  For j = 1 To 5
    Select Case i
      Case 0:  row = Choose(j, "Property       ", " Water part                  ", " Saline part                 ", " Property of seawater        ", "Unit")
      Case 1:  row = Choose(j, "g              ", "-0.446114968995|69225658E+5  ", " 0.15087174003705|274197E+5  ", "-0.295243228958|63951462E+5  ", "J kg-1")
      Case 2:  row = Choose(j, "(dg/dS)_T,p    ", " 0.0                         ", " 0.251957275851412|84916E+6  ", " 0.251957275851412|84916E+6  ", "J kg-1")
      Case 3:  row = Choose(j, "(dg/dT)_S,p    ", "-0.10737599318874|549194E+4  ", " 0.156230907404290|99119E+3  ", "-0.9175290244831|6392819E+3  ", "J kg-1 K-1")
      Case 4:  row = Choose(j, "(dg/dp)_S,T    ", " 0.102892955635611|00700E-2  ", "-0.57922728577125|827635E-4  ", " 0.97100682777898|424241E-3  ", "m3 kg-1")
      Case 5:  row = Choose(j, "(d2g/dSdp)_T   ", " 0.0                         ", "-0.30595780244233|918005E-3  ", "-0.30595780244233|918005E-3  ", "m3 kg-1")
      Case 6:  row = Choose(j, "(d2g/dT2)_S,p  ", "-0.11888500004754|679683E+2  ", " 0.127922649315507|38238E+1  ", "-0.10609273511599|605860E+2  ", "J kg-1 K-2")
      Case 7:  row = Choose(j, "(d2g/dTdp)_S   ", " 0.659051552339|36161134E-6  ", " 0.80306159575153|091195E-6  ", " 0.1462113148090|8925233E-5  ", "m3 kg-1 K-1")
      Case 8:  row = Choose(j, "(d2g/dp2)_S,T  ", "-0.4746728193610|6728585E-12 ", " 0.21308615424374|026844E-12 ", "-0.261586665117|32701741E-12 ", "m3 kg-1 Pa-1")
      Case 9:  row = Choose(j, "h              ", " 0.3344257590567|0236088E+6  ", "-0.4006233631000|9445692E+5  ", " 0.294363422746|6929152E+6   ", "J kg-1")
      Case 10: row = Choose(j, "f              ", "-0.447157531868|67008511E+5  ", " 0.15093043024178|351471E+5  ", "-0.296227101626|8865704E+5   ", "J kg-1")
      Case 11: row = Choose(j, "u              ", " 0.3343215027694|0457803E+6  ", "-0.40056467289536|368418E+5  ", " 0.2942650354798|6820961E+6  ", "J kg-1")
      Case 12: row = Choose(j, "s              ", " 0.10737599318874|549194E+4  ", "-0.156230907404290|99119E+3  ", " 0.9175290244831|6392819E+3  ", "J kg-1 K-1")
      Case 13: row = Choose(j, "rho            ", " 0.97188383191307|839206E+3  ", " -                           ", " 0.102985887574790|05489E+4  ", "kg m-3")
      Case 14: row = Choose(j, "cp             ", " 0.41966405016784|019283E+4  ", "-0.451566952083741|05981E+3  ", " 0.3745073549594|6608685E+4  ", "J kg-1 K-1")
      Case 15: row = Choose(j, "w              ", " 0.15544629665347|162190E+4  ", " -                           ", " 0.39612783529|250024954E+4  ", "m s-1")
      Case 16: row = Choose(j, "W             ", "-0.446114968995|69225658E+5  ", "-0.101085535814360|10719E+5  ", "-0.547200504810|05236377E+5  ", "J kg-1")
    End Select
    
    lrow(j) = Len(row)
    If j > 1 Then txt = txt + TB
    txt = txt + row
  Next j
  
  If i > 0 Then
    txt = txt + CRLF + "this code:    "
    For j = 2 To 4
      Select Case i
        Case 1
          Select Case j
            Case 2: x = liq_g_si(0, 0, t, p)
            Case 3: x = sal_g_si(0, 0, 0, S, t, p)
            Case 4: x = sea_g_si(0, 0, 0, S, t, p)
          End Select
          
        Case 2
          Select Case j
            Case 2: x = 0
            Case 3: x = sal_g_si(1, 0, 0, S, t, p)
            Case 4: x = sea_g_si(1, 0, 0, S, t, p)
          End Select
          
        Case 3
          Select Case j
            Case 2: x = liq_g_si(1, 0, t, p)
            Case 3: x = sal_g_si(0, 1, 0, S, t, p)
            Case 4: x = sea_g_si(0, 1, 0, S, t, p)
          End Select
          
        Case 4
          Select Case j
            Case 2: x = liq_g_si(0, 1, t, p)
            Case 3: x = sal_g_si(0, 0, 1, S, t, p)
            Case 4: x = sea_g_si(0, 0, 1, S, t, p)
          End Select
          
        Case 5
          Select Case j
            Case 2: x = 0
            Case 3: x = sal_g_si(1, 0, 1, S, t, p)
            Case 4: x = sea_g_si(1, 0, 1, S, t, p)
          End Select
          
        Case 6
          Select Case j
            Case 2: x = liq_g_si(2, 0, t, p)
            Case 3: x = sal_g_si(0, 2, 0, S, t, p)
            Case 4: x = sea_g_si(0, 2, 0, S, t, p)
          End Select
          
        Case 7
          Select Case j
            Case 2: x = liq_g_si(1, 1, t, p)
            Case 3: x = sal_g_si(0, 1, 1, S, t, p)
            Case 4: x = sea_g_si(0, 1, 1, S, t, p)
          End Select
          
        Case 8
          Select Case j
            Case 2: x = liq_g_si(0, 2, t, p)
            Case 3: x = sal_g_si(0, 0, 2, S, t, p)
            Case 4: x = sea_g_si(0, 0, 2, S, t, p)
          End Select

        Case 9
          Select Case j
            Case 2: x = liq_enthalpy_si(t, p)
            Case 3: x = sal_g_si(0, 0, 0, S, t, p) - t * sal_g_si(0, 1, 0, S, t, p)
            Case 4: x = sea_enthalpy_si(S, t, p)
          End Select
          
        Case 10
          Select Case j
            Case 2: x = liq_g_si(0, 0, t, p) - p * liq_g_si(0, 1, t, p)
            Case 3: x = sal_g_si(0, 0, 0, S, t, p) - p * sal_g_si(0, 0, 1, S, t, p)
            Case 4: x = sea_g_si(0, 0, 0, S, t, p) - p * sea_g_si(0, 0, 1, S, t, p)
          End Select
          
        Case 11
          Select Case j
            Case 2: x = liq_internal_energy_si(t, p)
            Case 3: x = sal_g_si(0, 0, 0, S, t, p) - p * sal_g_si(0, 0, 1, S, t, p) - t * sal_g_si(0, 1, 0, S, t, p)
            Case 4: x = sea_internal_energy_si(S, t, p)
          End Select
          
        Case 12
          Select Case j
            Case 2: x = liq_entropy_si(t, p)
            Case 3: x = -sal_g_si(0, 1, 0, S, t, p)
            Case 4: x = sea_entropy_si(S, t, p)
          End Select
          
        Case 13
          Select Case j
            Case 2: x = liq_density_si(t, p)
            Case 3: x = 0
            Case 4: x = sea_density_si(S, t, p)
          End Select
          
        Case 14
          Select Case j
            Case 2: x = liq_cp_si(t, p)
            Case 3: x = -t * sal_g_si(0, 2, 0, S, t, p)
            Case 4: x = sea_cp_si(S, t, p)
          End Select
          
        Case 15
          Select Case j
            Case 2: x = liq_soundspeed_si(t, p)
            Case 3: x = 0
            Case 4: x = sea_soundspeed_si(S, t, p)
          End Select
          
        Case 16
          Select Case j
            Case 2: x = liq_gibbs_energy_si(t, p)
            Case 3: x = sal_chempot_h2o_si(S, t, p)
            Case 4: x = sea_chempot_h2o_si(S, t, p)
          End Select

      End Select
      
      txt = txt + TB + Left(EFormat(x) + Space(lrow(j)), lrow(j))
    Next j
    txt = txt + TB + row + CRLF
  End If
Next i

txt = txt + CRLF + CRLF + "Note: For this computation, the iteration accuracy was increased as"
txt = txt + CRLF + "set_it_ctrl_density " + Chr(34) + "tol_liq_dens" + Chr(34) + ", -1E-15"

txt = txt + CRLF + CRLF + "The default iteration accuracy is as low as 0.1 ppm, i.e. equivalent to"
txt = txt + CRLF + "set_it_ctrl_density " + Chr(34) + "tol_liq_dens" + Chr(34) + ", -1E-7"

chk_OS2008_TableA6 = txt

set_it_ctrl_density "tol_liq_dens", tol_liq


'Properties of water and seawater
'Computed from thermodynamic potentials of the seawater library
'for publication in Ocean Science, 2009
'R. Feistel, IOW, Version 11 Apr 2009
'Compiled on 11.04.2009 12:45:27
'
'Compared with Table A6 of the article
'R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner
'Mutually consistent thermodynamic potentials for fluid water, ice and seawater:
'A new standard for oceanography.
'Ocean Sci., 4, 275-291, 2008
'www.ocean-sci.net/4/275/2008/:
'
'Table A6: Quadruple-precision results for the water part, saline part and total properties
'published in Table 8b of IAPWS-08 with the coefficients given in Table 1 of this paper.
'Properties at SA = 0.1 kg kg-1 = 100 g kg-1, T = 353 K, p = p0 = 0.101325 MPa.
'This point is located in the regions with restricted validity
'The separated numbers indicate the digits that, based on our tests, can
'reasonably be expected to be reproduced using double-precision code.
'
'Property         Water part                    Saline part                   Property of seawater         Unit
'g               -4.46114968995|69225658E+4     1.5087174003705|274197E+4    -2.95243228958|63951462E+4    J kg-1
'this code:      -0.446114968995688E+5          0.150871740037053E+5         -0.295243228958636E+5         J kg-1
'(dg/dS)_T,p      0.0                           2.51957275851412|84916E+5     2.51957275851412|84916E+5    J kg-1
'this code:       0.0                           0.251957275851413E+6          0.251957275851413E+6         J kg-1
'(dg/dT)_S,p     -1.0737599318874|549194E+3     1.56230907404290|99119E+2    -9.175290244831|6392819E+2    J kg-1 K-1
'this code:      -0.107375993188745E+4          0.156230907404291E+3         -0.917529024483159E+3         J kg-1 K-1
'(dg/dp)_S,T      1.02892955635611|00700E-3    -5.7922728577125|827635E-5     9.7100682777898|424241E-4    m3 kg-1
'this code:       0.102892955635611E-2         -0.579227285771257E-4          0.971006827778985E-3         m3 kg-1
'(d2g/dSdp)_T     0.0                          -3.0595780244233|918005E-4    -3.0595780244233|918005E-4    m3 kg-1
'this code:       0.0                          -0.305957802442338E-3         -0.305957802442338E-3         m3 kg-1
'(d2g/dT2)_S,p   -1.1888500004754|679683E+1     1.27922649315507|38238E+0    -1.0609273511599|605860E+1    J kg-1 K-2
'this code:      -0.118885000047548E+2          0.127922649315507E+1         -0.106092735115997E+2         J kg-1 K-2
'(d2g/dTdp)_S     6.59051552339|36161134E-7     8.0306159575153|091195E-7     1.462113148090|8925233E-6    m3 kg-1 K-1
'this code:       0.659051552339389E-6          0.803061595751535E-6          0.146211314809092E-5         m3 kg-1 K-1
'(d2g/dp2)_S,T   -4.746728193610|6728585E-13    2.1308615424374|026844E-13   -2.61586665117|32701741E-13   m3 kg-1 Pa-1
'this code:      -0.474672819361079E-12         0.213086154243741E-12        -0.261586665117338E-12        m3 kg-1 Pa-1
'h                3.344257590567|0236088E+5    -4.006233631000|9445692E+4     2.94363422746|6929152E+5     J kg-1
'this code:       0.334425759056701E+6         -0.400623363100094E+5          0.294363422746692E+6         J kg-1
'f               -4.47157531868|67008511E+4     1.5093043024178|351471E+4    -2.96227101626|8865704E+4     J kg-1
'this code:      -0.447157531868666E+5          0.150930430241784E+5         -0.296227101626883E+5         J kg-1
'u                3.343215027694|0457803E+5    -4.0056467289536|368418E+4     2.942650354798|6820961E+5    J kg-1
'this code:       0.334321502769405E+6         -0.400564672895364E+5          0.294265035479867E+6         J kg-1
's                1.0737599318874|549194E+3    -1.56230907404290|99119E+2     9.175290244831|6392819E+2    J kg-1 K-1
'this code:       0.107375993188745E+4         -0.156230907404291E+3          0.917529024483159E+3         J kg-1 K-1
'rho              9.7188383191307|839206E+2     -                             1.02985887574790|05489E+3    kg m-3
'this code:       0.971883831913078E+3          0.0                           0.10298588757479E+4          kg m-3
'cp               4.1966405016784|019283E+3    -4.51566952083741|05981E+2     3.745073549594|6608685E+3    J kg-1 K-1
'this code:       0.419664050167843E+4         -0.451566952083741E+3          0.374507354959469E+4         J kg-1 K-1
'w                1.5544629665347|162190E+3     -                             3.9612783529|250024954E+3    m s-1
'this code:       0.15544629665347E+4           0.0                           0.396127835292488E+4         m s-1
'W              -4.46114968995|69225658E+4    -1.01085535814360|10719E+4    -5.47200504810|05236377E+4    J kg-1
'this code:      -0.446114968995688E+5         -0.10108553581436E+5          -0.547200504810049E+5         J kg-1
'
'Note: For this computation, the iteration accuracy was increased as
'set_it_ctrl_density "tol_liq_dens", -1E-15
'
'The default iteration accuracy is as low as 0.1 ppm, i.e. equivalent to
'set_it_ctrl_density "tol_liq_dens", -1E-7


End Function

'==========================================================================
Public Function chk_OS2008_TableA7() As String

'Comparison of the library function with check values
'computed in quadruple precision (128 bit floating point)
'as published in Table A7 of

'R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner
'Mutually consistent thermodynamic potentials for fluid water, ice and seawater:
'a new standard for oceanography.
'Ocean Sci., 4, 275291, 2008
'www.ocean-sci.net/4/275/2008/

'Table A7: Quadruple-precision results for the water part, saline part and total properties published in
'Table 8c of IAPWS-08 with the coefficients given in Table 1 of this paper.
'Properties at SA = 0.035 165 04 kg kg-1, T = T0 = 273.15 K, p = 100 MPa.
'The separated numbers indicate the digits that, based on our tests, can reasonably
'be expected to be reproduced using double-precision code.

'Property        Water part                   Saline part                  Property of seawater        Unit
'g               9.773038621953|7338734E+4   -2.60093050730636|60852E+3    9.512945571223|0972649E+4   J kg-1
'(dg/dS)_T,p     0.0                         -5.4586158064879|659787E+3   -5.4586158064879|659787E+3   J kg-1
'(dg/dT)_S,p     8.5146650206|262343669E+0    7.5404568488116|539426E+0    1.60551218694|37888309E+1   J kg-1 K-1
'(dg/dp)_S,T     9.5668332915350|911569E-4   -2.2912384179113|101721E-5    9.3377094497439|601397E-4   m3 kg-1
'(d2g/dSdp)_T    0.0                         -6.4075761854574|757172E-4   -6.40757618545747|57172E-4   m3 kg-1
'(d2g/dT2)_S,p  -1.429698733875|9055994E+1    4.88076973942251|22581E-1   -1.380891036481|6804768E+1   J kg-1 K-2
'(d2g/dTdp)_S    1.99079570803|15389517E-7    4.6628441224121|312517E-8    2.45708012027|27520769E-7   m3 kg-1 K-1
'(d2g/dp2)_S,T  -3.715308894234|1756981E-13   3.57345735845326|66554E-14  -3.35796315838|88490325E-13  m3 kg-1 Pa-1
'h               9.54046054691|53282817E+4   -4.6606062955592|693597E+3    9.07439991735|94013457E+4   J kg-1
'f               2.06205330418|64271652E+3   -3.0969208939505|591314E+2    1.75236121479|13712521E+3   J kg-1
'u              -2.637274461|9762875211E+2   -2.3693678776479|591876E+3   -2.6330953238|455879397E+3   J kg-1
's              -8.5146650206|262343669E+0   -7.5404568488116|539426E+0   -1.6055121869|437888309E+1   J kg-1 K-1
'rho             1.04527796139692|14514E+3    -                            1.0709264465574|263352E+3   kg m-3
'cp              3.905222091582|0361447E+3   -1.33318225432325|92233E+2    3.77190386614|97102223E+3   J kg-1 K-1
'w               1.575422398485|9303496E+3    -                            1.621989976498|7563752E+3   m s-1
'W              9.77303862195|37338734E+4   -2.4089780641265|845021E+3    9.532140815541|0754232E+4   J kg-1

Dim CRLF As String, TB As String
Dim txt As String
Dim S As Double, t As Double, x As Double, p As Double

Dim i As Integer, j As Integer, row As String
Dim tol_liq As Double
Dim lrow(5) As Integer

CRLF = Chr(13) + Chr(10)
TB = Chr(9)

txt = "Properties of water and seawater" + CRLF
txt = txt + "Computed from thermodynamic potentials of the seawater library" + CRLF
txt = txt + "for publication in Ocean Science, 2009" + CRLF
txt = txt + "R. Feistel, IOW, Version " + Version + CRLF
txt = txt + "Compiled on " + CStr(Now) + CRLF + CRLF

txt = txt + "Compared with Table A7 of the article" + CRLF
txt = txt + "R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner" + CRLF
txt = txt + "Mutually consistent thermodynamic potentials for fluid water, ice and seawater:" + CRLF
txt = txt + "A new standard for oceanography." + CRLF
txt = txt + "Ocean Sci., 4, 275-291, 2008" + CRLF
txt = txt + "www.ocean-sci.net/4/275/2008/:" + CRLF + CRLF

txt = txt + "Table A7: Quadruple-precision results for the water part, saline part and total properties " + CRLF
txt = txt + "published in Table 8c of IAPWS-08 with the coefficients given in Table 1 of this paper.  " + CRLF
txt = txt + "Properties at SA = 0.035 165 04 kg kg-1, T = T0 = 273.15 K, p = 100 MPa. " + CRLF
txt = txt + "The separated numbers indicate the digits that, based on our tests, can " + CRLF
txt = txt + "reasonably be expected to be reproduced using double-precision code." + CRLF

S = 0.03516504
t = 273.15
p = 100000000#

tol_liq = get_it_ctrl_density("tol_liq_dens")
set_it_ctrl_density "tol_liq_dens", -0.000000000000001

For i = 0 To 16
  txt = txt + CRLF
  For j = 1 To 5
    Select Case i
      Case 0:  row = Choose(j, "Property       ", " Water part                  ", " Saline part                 ", " Property of seawater        ", "Unit")
      Case 1:  row = Choose(j, "g              ", " 0.9773038621953|7338734E+5  ", "-0.260093050730636|60852E+4  ", " 0.9512945571223|0972649E+5  ", "J kg-1")
      Case 2:  row = Choose(j, "(dg/dS)_T,p    ", " 0.0                         ", "-0.54586158064879|659787E+4  ", "-0.54586158064879|659787E+4  ", "J kg-1")
      Case 3:  row = Choose(j, "(dg/dT)_S,p    ", " 0.85146650206|262343669E+1  ", " 0.75404568488116|539426E+1  ", " 0.160551218694|37888309E+2  ", "J kg-1 K-1")
      Case 4:  row = Choose(j, "(dg/dp)_S,T    ", " 0.95668332915350|911569E-3  ", "-0.22912384179113|101721E-6  ", " 0.93377094497439|601397E-3  ", "m3 kg-1")
      Case 5:  row = Choose(j, "(d2g/dSdp)_T   ", " 0.0                         ", "-0.64075761854574|757172E-5  ", "-0.640757618545747|57172E-3  ", "m3 kg-1")
      Case 6:  row = Choose(j, "(d2g/dT2)_S,p  ", "-0.1429698733875|9055994E+2  ", " 0.488076973942251|22581     ", "-0.1380891036481|6804768E+2  ", "J kg-1 K-2")
      Case 7:  row = Choose(j, "(d2g/dTdp)_S   ", " 0.199079570803|15389517E-6  ", " 0.46628441224121|312517E-7  ", " 0.245708012027|27520769E-6  ", "m3 kg-1 K-1")
      Case 8:  row = Choose(j, "(d2g/dp2)_S,T  ", "-0.3715308894234|1756981E-12 ", " 0.357345735845326|66554E-13 ", "-0.335796315838|88490325E-12 ", "m3 kg-1 Pa-1")
      Case 9:  row = Choose(j, "h              ", " 0.954046054691|53282817E+5  ", "-0.46606062955592|693597E+4  ", " 0.907439991735|94013457E+5  ", "J kg-1")
      Case 10: row = Choose(j, "f              ", " 0.206205330418|64271652E+4  ", "-0.30969208939505|591314E+3  ", " 0.175236121479|13712521E+4  ", "J kg-1")
      Case 11: row = Choose(j, "u              ", "-0.2637274461|9762875211E+3  ", "-0.23693678776479|591876E+4  ", "-0.26330953238|455879397E+4  ", "J kg-1")
      Case 12: row = Choose(j, "s              ", "-0.85146650206|262343669E+1  ", "-0.75404568488116|539426E+1  ", "-0.16055121869|437888309E+2  ", "J kg-1 K-1")
      Case 13: row = Choose(j, "rho            ", " 0.104527796139692|14514E+4  ", " -                           ", " 0.10709264465574|263352E+4  ", "kg m-3")
      Case 14: row = Choose(j, "cp             ", " 0.3905222091582|0361447E+4  ", "-0.133318225432325|92233E+3  ", " 0.377190386614|97102223E+4  ", "J kg-1 K-1")
      Case 15: row = Choose(j, "w              ", " 0.1575422398485|9303496E+4  ", " -                           ", " 0.1621989976498|7563752E+4  ", "m s-1")
      Case 16: row = Choose(j, "W             ", " 0.977303862195|37338734E+5  ", "-0.24089780641265|845021E+4  ", " 0.9532140815541|0754232E+5  ", "J kg-1")
    End Select
    
    lrow(j) = Len(row)
    If j > 1 Then txt = txt + TB
    txt = txt + row
  Next j
  
  If i > 0 Then
    txt = txt + CRLF + "this code:    "
    For j = 2 To 4
      Select Case i
        Case 1
          Select Case j
            Case 2: x = liq_g_si(0, 0, t, p)
            Case 3: x = sal_g_si(0, 0, 0, S, t, p)
            Case 4: x = sea_g_si(0, 0, 0, S, t, p)
          End Select
          
        Case 2
          Select Case j
            Case 2: x = 0
            Case 3: x = sal_g_si(1, 0, 0, S, t, p)
            Case 4: x = sea_g_si(1, 0, 0, S, t, p)
          End Select
          
        Case 3
          Select Case j
            Case 2: x = liq_g_si(1, 0, t, p)
            Case 3: x = sal_g_si(0, 1, 0, S, t, p)
            Case 4: x = sea_g_si(0, 1, 0, S, t, p)
          End Select
          
        Case 4
          Select Case j
            Case 2: x = liq_g_si(0, 1, t, p)
            Case 3: x = sal_g_si(0, 0, 1, S, t, p)
            Case 4: x = sea_g_si(0, 0, 1, S, t, p)
          End Select
          
        Case 5
          Select Case j
            Case 2: x = 0
            Case 3: x = sal_g_si(1, 0, 1, S, t, p)
            Case 4: x = sea_g_si(1, 0, 1, S, t, p)
          End Select
          
        Case 6
          Select Case j
            Case 2: x = liq_g_si(2, 0, t, p)
            Case 3: x = sal_g_si(0, 2, 0, S, t, p)
            Case 4: x = sea_g_si(0, 2, 0, S, t, p)
          End Select
          
        Case 7
          Select Case j
            Case 2: x = liq_g_si(1, 1, t, p)
            Case 3: x = sal_g_si(0, 1, 1, S, t, p)
            Case 4: x = sea_g_si(0, 1, 1, S, t, p)
          End Select
          
        Case 8
          Select Case j
            Case 2: x = liq_g_si(0, 2, t, p)
            Case 3: x = sal_g_si(0, 0, 2, S, t, p)
            Case 4: x = sea_g_si(0, 0, 2, S, t, p)
          End Select

        Case 9
          Select Case j
            Case 2: x = liq_enthalpy_si(t, p)
            Case 3: x = sal_g_si(0, 0, 0, S, t, p) - t * sal_g_si(0, 1, 0, S, t, p)
            Case 4: x = sea_enthalpy_si(S, t, p)
          End Select
          
        Case 10
          Select Case j
            Case 2: x = liq_g_si(0, 0, t, p) - p * liq_g_si(0, 1, t, p)
            Case 3: x = sal_g_si(0, 0, 0, S, t, p) - p * sal_g_si(0, 0, 1, S, t, p)
            Case 4: x = sea_g_si(0, 0, 0, S, t, p) - p * sea_g_si(0, 0, 1, S, t, p)
          End Select
          
        Case 11
          Select Case j
            Case 2: x = liq_internal_energy_si(t, p)
            Case 3: x = sal_g_si(0, 0, 0, S, t, p) - p * sal_g_si(0, 0, 1, S, t, p) - t * sal_g_si(0, 1, 0, S, t, p)
            Case 4: x = sea_internal_energy_si(S, t, p)
          End Select
          
        Case 12
          Select Case j
            Case 2: x = liq_entropy_si(t, p)
            Case 3: x = -sal_g_si(0, 1, 0, S, t, p)
            Case 4: x = sea_entropy_si(S, t, p)
          End Select
          
        Case 13
          Select Case j
            Case 2: x = liq_density_si(t, p)
            Case 3: x = 0
            Case 4: x = sea_density_si(S, t, p)
          End Select
          
        Case 14
          Select Case j
            Case 2: x = liq_cp_si(t, p)
            Case 3: x = -t * sal_g_si(0, 2, 0, S, t, p)
            Case 4: x = sea_cp_si(S, t, p)
          End Select
          
        Case 15
          Select Case j
            Case 2: x = liq_soundspeed_si(t, p)
            Case 3: x = 0
            Case 4: x = sea_soundspeed_si(S, t, p)
          End Select
          
        Case 16
          Select Case j
            Case 2: x = liq_gibbs_energy_si(t, p)
            Case 3: x = sal_chempot_h2o_si(S, t, p)
            Case 4: x = sea_chempot_h2o_si(S, t, p)
          End Select

      End Select
      
      txt = txt + TB + Left(EFormat(x) + Space(lrow(j)), lrow(j))
    Next j
    txt = txt + TB + row + CRLF
  End If
Next i

txt = txt + CRLF + CRLF + "Note: For this computation, the iteration accuracy was increased as"
txt = txt + CRLF + "set_it_ctrl_density " + Chr(34) + "tol_liq_dens" + Chr(34) + ", -1E-15"

txt = txt + CRLF + CRLF + "The default iteration accuracy is as low as 0.1 ppm, i.e. equivalent to"
txt = txt + CRLF + "set_it_ctrl_density " + Chr(34) + "tol_liq_dens" + Chr(34) + ", -1E-7"

chk_OS2008_TableA7 = txt

set_it_ctrl_density "tol_liq_dens", tol_liq


'Properties of water and seawater
'Computed from thermodynamic potentials of the seawater library
'for publication in Ocean Science, 2009
'R. Feistel, IOW, Version 11 Apr 2009
'Compiled on 11.04.2009 13:42:35
'
'Compared with Table A7 of the article
'R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner
'Mutually consistent thermodynamic potentials for fluid water, ice and seawater:
'A new standard for oceanography.
'Ocean Sci., 4, 275-291, 2008
'www.ocean-sci.net/4/275/2008/:
'
'Table A7: Quadruple-precision results for the water part, saline part and total properties
'published in Table 8c of IAPWS-08 with the coefficients given in Table 1 of this paper.
'Properties at SA = 0.035 165 04 kg kg-1, T = T0 = 273.15 K, p = 100 MPa.
'The separated numbers indicate the digits that, based on our tests, can
'reasonably be expected to be reproduced using double-precision code.
'
'Property         Water part                    Saline part                   Property of seawater         Unit
'g                9.773038621953|7338734E+4    -2.60093050730636|60852E+3     9.512945571223|0972649E+4    J kg-1
'this code:       0.977303862195402E+5         -0.260093050730637E+4          0.951294557122338E+5         J kg-1
'(dg/dS)_T,p      0.0                          -5.4586158064879|659787E+3    -5.4586158064879|659787E+3    J kg-1
'this code:       0.0                          -0.5458615806488E+4           -0.5458615806488E+4           J kg-1
'(dg/dT)_S,p      8.5146650206|262343669E+0     7.5404568488116|539426E+0     1.60551218694|37888309E+1    J kg-1 K-1
'this code:       0.851466502060921E+1          0.754045684881166E+1          0.160551218694209E+2         J kg-1 K-1
'(dg/dp)_S,T      9.5668332915350|911569E-4    -2.2912384179113|101721E-5     9.3377094497439|601397E-4    m3 kg-1
'this code:       0.95668332915351E-3          -0.229123841791131E-4          0.933770944974397E-3         m3 kg-1
'(d2g/dSdp)_T     0.0                          -6.4075761854574|757172E-4    -6.40757618545747|57172E-4    m3 kg-1
'this code:       0.0                          -0.640757618545748E-3         -0.640757618545748E-3         m3 kg-1
'(d2g/dT2)_S,p   -1.429698733875|9055994E+1     4.88076973942251|22581E-1    -1.380891036481|6804768E+1    J kg-1 K-2
'this code:      -0.142969873387593E+2          0.488076973942251            -0.138089103648171E+2         J kg-1 K-2
'(d2g/dTdp)_S     1.99079570803|15389517E-7     4.6628441224121|312517E-8     2.45708012027|27520769E-7    m3 kg-1 K-1
'this code:       0.199079570803143E-6          0.466284412241213E-7          0.245708012027264E-6         m3 kg-1 K-1
'(d2g/dp2)_S,T   -3.715308894234|1756981E-13    3.57345735845326|66554E-14   -3.35796315838|88490325E-13   m3 kg-1 Pa-1
'this code:      -0.371530889423415E-12         0.357345735845327E-13        -0.335796315838882E-12        m3 kg-1 Pa-1
'h                9.54046054691|53282817E+4    -4.6606062955592|693597E+3     9.07439991735|94013457E+4    J kg-1
'this code:       0.954046054691608E+5         -0.466060629555927E+4          0.907439991736015E+5         J kg-1
'f                2.06205330418|64271652E+3    -3.0969208939505|591314E+2     1.75236121479|13712521E+3    J kg-1
'this code:       0.206205330418918E+4         -0.309692089395056E+3          0.175236121479412E+4         J kg-1
'u               -2.637274461|9762875211E+2    -2.3693678776479|591876E+3    -2.6330953238|455879397E+3    J kg-1
'this code:      -0.263727446190609E+3         -0.236936787764796E+4         -0.263309532383819E+4         J kg-1
's               -8.5146650206|262343669E+0    -7.5404568488116|539426E+0    -1.6055121869|437888309E+1    J kg-1 K-1
'this code:      -0.851466502060921E+1         -0.754045684881166E+1         -0.160551218694209E+2         J kg-1 K-1
'rho              1.04527796139692|14514E+3     -                             1.0709264465574|263352E+3    kg m-3
'this code:       0.104527796139692E+4          0.0                           0.107092644655742E+4         kg m-3
'cp               3.905222091582|0361447E+3    -1.33318225432325|92233E+2     3.77190386614|97102223E+3    J kg-1 K-1
'this code:       0.390522209158211E+4         -0.133318225432326E+3          0.377190386614978E+4         J kg-1 K-1
'w                1.575422398485|9303496E+3     -                             1.621989976498|7563752E+3    m s-1
'this code:       0.157542239848594E+4          0.0                           0.162198997649876E+4         m s-1
'W               9.77303862195|37338734E+4    -2.4089780641265|845021E+3     9.532140815541|0754232E+4    J kg-1
'this code:       0.977303862195402E+5         -0.240897806412658E+4          0.953214081554136E+5         J kg-1
'
'Note: For this computation, the iteration accuracy was increased as
'set_it_ctrl_density "tol_liq_dens", -1E-15
'
'The default iteration accuracy is as low as 0.1 ppm, i.e. equivalent to
'set_it_ctrl_density "tol_liq_dens", -1E-7


End Function

'==========================================================================
Public Function chk_OS2008_TableA8() As String

'Comparison of the library function with check values
'computed in quadruple precision (128 bit floating point)
'as published in Table A8 of

'R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner
'Mutually consistent thermodynamic potentials for fluid water, ice and seawater:
'a new standard for oceanography.
'Ocean Sci., 4, 275291, 2008
'www.ocean-sci.net/4/275/2008/

'Table A8: Quadruple-precision results for the properties of water, ice and seawater at the standard ocean state,
'computed with all coefficients as given in Table 1 of this paper.  T = 273.15 K, p = 0.101 325 MPa and
'SA = 0.035 165 04 for seawater.  The underlined numbers indicate the digits that, based on our tests,
'can reasonably be expected to be reproduced using double-precision code.

'Property         Property of water           Property of ice             Property of seawater       Unit
'g                1.0134274172939062882E+2    9.8267598403431717064E+1   -2.E-29 b,     4.E-9 b      J kg-1
'(dg/dS)_T,p      0.0                         0.0                         6.3997406731229904527E+4   J kg-1
'(dg/dT)_S,p      1.4764337634625266531E-1    1.2207886612999530642E+3    7.E-32 b,    -6.E-11 b     J kg-1 K-1
'(dg/dp)_S,T      1.0001569391216926347E-3    1.0908434429264352467E-3    9.7266121669484934729E-4   m3 kg-1
'(d2g/dSdp)_T     0.0                         0.0                        -7.5961541151530889445E-4   m3 kg-1
'(d2g/dT2)_S,p   -1.5447354231977289339E+1   -7.6759851115667509185E+0   -1.4594493080801366829E+1   J kg-1 K-2
'(d2g/dTdp)_S    -6.7770031786558265755E-8    1.7436082496084962410E-7    5.1516755627399375563E-8   m3 kg-1 K-1
'(d2g/dp2)_S,T   -5.0892889464349017238E-13  -1.2848482463976179327E-13  -4.5077537741020194745E-13  m3 kg-1 Pa-1
'h                6.1013953480411713295E+1   -3.3336015523567874778E+5   -4.E-29 b,    2.E-8 b       J kg-1
'f                1.8398728851226087838E-3   -1.2262113451089334305E+1   -9.8554897781605610114E+1   J kg-1
'u               -4.0326948376093792920E+1   -3.3347068494753326883E+5   -9.8554897781605610114E+1   J kg-1
's               -1.4764337634625266531E-1   -1.2207886612999530642E+3   -7.E-32 b,     6.E-11 b     J kg-1 K-1
'rho              9.9984308550433049647E+2    9.1672183252738167257E+2    1.0281071999540078127E+3   kg m-3
'cp               4.2194448084645965831E+3    2.0966953332244580134E+3    3.9864857850208933494E+3   J kg-1 K-1
'alpha           -6.7759397686198971741E-5    1.5984037497909609918E-4    5.2964747378800447017E-5   K-1
'kappa_T          5.0884903632265555122E-10   1.1778484389572171289E-10   4.6344541107741383004E-10  Pa-1
'kappa_s          5.0855176492808069170E-10   1.1415405263722770087E-10   4.6325845206948706884E-10  Pa-1
'W               1.0134274172939062882E+2    9.8267598403431717064E+1   -2.2504713675999688419E+3   J kg-1

Dim CRLF As String, TB As String
Dim txt As String
Dim S As Double, t As Double, x As Double, p As Double

Dim i As Integer, j As Integer, row As String, tol As Double
Dim lrow(5) As Integer

CRLF = Chr(13) + Chr(10)
TB = Chr(9)

txt = "Properties of water and seawater" + CRLF
txt = txt + "Computed from thermodynamic potentials of the seawater library" + CRLF
txt = txt + "for publication in Ocean Science, 2009" + CRLF
txt = txt + "R. Feistel, IOW, Version " + Version + CRLF
txt = txt + "Compiled on " + CStr(Now) + CRLF + CRLF

txt = txt + "Compared with Table A8 of the article" + CRLF
txt = txt + "R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner" + CRLF
txt = txt + "Mutually consistent thermodynamic potentials for fluid water, ice and seawater:" + CRLF
txt = txt + "A new standard for oceanography." + CRLF
txt = txt + "Ocean Sci., 4, 275-291, 2008" + CRLF
txt = txt + "www.ocean-sci.net/4/275/2008/:" + CRLF + CRLF

txt = txt + "Table A8: Quadruple-precision results for the properties of water, ice and seawater at the standard ocean state," + CRLF
txt = txt + "computed with all coefficients as given in Table 1 of this paper.  T = 273.15 K, p = 0.101 325 MPa and" + CRLF
txt = txt + "SA = 0.035 165 04 for seawater. The underlined numbers indicate the digits that, based on our tests," + CRLF
txt = txt + "can reasonably be expected to be reproduced using double-precision code." + CRLF

S = 0.03516504
t = 273.15
p = 101325

tol = get_it_ctrl_density("tol_liq_dens")
set_it_ctrl_density "tol_liq_dens", -0.000000000000001

For i = 0 To 16
  txt = txt + CRLF
  For j = 1 To 5
    Select Case i
      Case 0:  row = Choose(j, "Property       ", " Property of water          ", " Property of ice            ", " Property of seawater        ", "Unit")
      Case 1:  row = Choose(j, "g              ", " 0.1013427417|2939062882E+3 ", " 0.98267598403|431717064E+2 ", "-2.E-29,  4.E-9              ", "J kg-1")
      Case 2:  row = Choose(j, "(dg/dS)_T,p    ", " 0.0                        ", " 0.0                        ", " 0.639974067312299|04527E+5  ", "J kg-1")
      Case 3:  row = Choose(j, "(dg/dT)_S,p    ", " 0.1476433763|4625266531    ", " 0.122078866129995|30642E+4 ", " 7.E-32, -6.E-11             ", "J kg-1 K-1")
      Case 4:  row = Choose(j, "(dg/dp)_S,T    ", " 0.100015693912169|26347E-2 ", " 0.109084344292643|52467E-2 ", " 0.97266121669484|934729E-3  ", "m3 kg-1")
      Case 5:  row = Choose(j, "(d2g/dSdp)_T   ", " 0.0                        ", " 0.0                        ", "-0.759615411515308|89445E-3  ", "m3 kg-1")
      Case 6:  row = Choose(j, "(d2g/dT2)_S,p  ", "-0.1544735423197|7289339E+2 ", "-0.767598511156675|09185E+1 ", "-0.1459449308080|1366829E+2  ", "J kg-1 K-2")
      Case 7:  row = Choose(j, "(d2g/dTdp)_S   ", "-0.6777003178|6558265755E-7 ", " 0.1743608249|6084962410E-6 ", " 0.515167556273|99375563E-7  ", "m3 kg-1 K-1")
      Case 8:  row = Choose(j, "(d2g/dp2)_S,T  ", "-0.5089288946434|9017238E-12", "-0.128484824639761|79327E-12", "-0.4507753774102|0194745E-12 ", "m3 kg-1 Pa-1")
      Case 9:  row = Choose(j, "h              ", " 0.610139534|80411713295E+2 ", "-0.333360155235678|74778E+6 ", "-4.E-29,  2.E-8              ", "J kg-1")
      Case 10: row = Choose(j, "f              ", " 0.18398|728851226087838E-2 ", "-0.1226211345|1089334305E+2 ", "-0.9855489778|1605610114E+2  ", "J kg-1")
      Case 11: row = Choose(j, "u              ", "-0.403269483|76093792920E+2 ", "-0.3334706849475|3326883E+6 ", "-0.985548977|81605610114E+2  ", "J kg-1")
      Case 12: row = Choose(j, "s              ", "-0.147643376|34625266531    ", "-0.122078866129995|30642E+4 ", "-7.E-32,  6.E-11             ", "J kg-1 K-1")
      Case 13: row = Choose(j, "rho            ", " 0.99984308550433|049647E+3 ", " 0.916721832527381|67257E+3 ", " 0.102810719995400|78127E+4  ", "kg m-3")
      Case 14: row = Choose(j, "cp             ", " 0.4219444808464|5965831E+4 ", " 0.209669533322445|80134E+4 ", " 0.398648578502|08933494E+4  ", "J kg-1 K-1")
      Case 15: row = Choose(j, "alpha          ", "-0.67759397686|198971741E-4 ", " 0.159840374979096|09918E-3 ", " 0.52964747378|800447017E-4  ", "K-1")
      Case 16: row = Choose(j, "kappa_t        ", " 0.508849036322|65555122E-9 ", " 0.117784843895721|71289E-9 ", " 0.4634454110774|1383004E-9  ", "Pa-1")
      Case 17: row = Choose(j, "kappa_s        ", " 0.508551764928|08069170E-9 ", " 0.114154052637227|70087E-9 ", " 0.463258452069|48706884E-9  ", "Pa-1")
      Case 18: row = Choose(j, "W             ", " 0.1013427417|2939062882E+3 ", " 0.98267598403|431717064E+2 ", "-0.2250471367|5999688419E+4  ", "J kg-1")
    End Select
    
    lrow(j) = Len(row)
    If j > 1 Then txt = txt + TB
    txt = txt + row
  Next j
  
  If i > 0 Then
    txt = txt + CRLF + "this code:    "
    For j = 2 To 4
      Select Case i
        Case 1
          Select Case j
            Case 2: x = liq_g_si(0, 0, t, p)
            Case 3: x = ice_g_si(0, 0, t, p)
            Case 4: x = sea_g_si(0, 0, 0, S, t, p)
          End Select
          
        Case 2
          Select Case j
            Case 2: x = 0
            Case 3: x = 0
            Case 4: x = sea_g_si(1, 0, 0, S, t, p)
          End Select
          
        Case 3
          Select Case j
            Case 2: x = liq_g_si(1, 0, t, p)
            Case 3: x = ice_g_si(1, 0, t, p)
            Case 4: x = sea_g_si(0, 1, 0, S, t, p)
          End Select
          
        Case 4
          Select Case j
            Case 2: x = liq_g_si(0, 1, t, p)
            Case 3: x = ice_g_si(0, 1, t, p)
            Case 4: x = sea_g_si(0, 0, 1, S, t, p)
          End Select
          
        Case 5
          Select Case j
            Case 2: x = 0
            Case 3: x = 0
            Case 4: x = sea_g_si(1, 0, 1, S, t, p)
          End Select
          
        Case 6
          Select Case j
            Case 2: x = liq_g_si(2, 0, t, p)
            Case 3: x = ice_g_si(2, 0, t, p)
            Case 4: x = sea_g_si(0, 2, 0, S, t, p)
          End Select
          
        Case 7
          Select Case j
            Case 2: x = liq_g_si(1, 1, t, p)
            Case 3: x = ice_g_si(1, 1, t, p)
            Case 4: x = sea_g_si(0, 1, 1, S, t, p)
          End Select
          
        Case 8
          Select Case j
            Case 2: x = liq_g_si(0, 2, t, p)
            Case 3: x = ice_g_si(0, 2, t, p)
            Case 4: x = sea_g_si(0, 0, 2, S, t, p)
          End Select

        Case 9
          Select Case j
            Case 2: x = liq_enthalpy_si(t, p)
            Case 3: x = ice_enthalpy_si(t, p)
            Case 4: x = sea_enthalpy_si(S, t, p)
          End Select
          
        Case 10
          Select Case j
            Case 2: x = liq_g_si(0, 0, t, p) - p * liq_g_si(0, 1, t, p)
            Case 3: x = ice_g_si(0, 0, t, p) - p * ice_g_si(0, 1, t, p)
            Case 4: x = sea_g_si(0, 0, 0, S, t, p) - p * sea_g_si(0, 0, 1, S, t, p)
          End Select
          
        Case 11
          Select Case j
            Case 2: x = liq_internal_energy_si(t, p)
            Case 3: x = ice_internal_energy_si(t, p)
            Case 4: x = sea_internal_energy_si(S, t, p)
          End Select
          
        Case 12
          Select Case j
            Case 2: x = liq_entropy_si(t, p)
            Case 3: x = ice_entropy_si(t, p)
            Case 4: x = sea_entropy_si(S, t, p)
          End Select
          
        Case 13
          Select Case j
            Case 2: x = liq_density_si(t, p)
            Case 3: x = ice_density_si(t, p)
            Case 4: x = sea_density_si(S, t, p)
          End Select
          
        Case 14
          Select Case j
            Case 2: x = liq_cp_si(t, p)
            Case 3: x = ice_cp_si(t, p)
            Case 4: x = sea_cp_si(S, t, p)
          End Select
          
        Case 15
          Select Case j
            Case 2: x = liq_expansion_si(t, p)
            Case 3: x = ice_expansion_si(t, p)
            Case 4: x = sea_g_expansion_t_si(S, t, p)
          End Select
  
        Case 16
          Select Case j
            Case 2: x = liq_kappa_t_si(t, p)
            Case 3: x = ice_kappa_t_si(t, p)
            Case 4: x = sea_kappa_t_si(S, t, p)
          End Select
  
        Case 17
          Select Case j
            Case 2: x = liq_kappa_s_si(t, p)
            Case 3: x = ice_kappa_s_si(t, p)
            Case 4: x = sea_kappa_s_si(S, t, p)
          End Select
  
        Case 18
          Select Case j
            Case 2: x = liq_gibbs_energy_si(t, p)
            Case 3: x = ice_g_si(0, 0, t, p)
            Case 4: x = sea_chempot_h2o_si(S, t, p)
          End Select

      End Select
      
      txt = txt + TB + Left(EFormat(x) + Space(lrow(j)), lrow(j))
    Next j
    txt = txt + TB + row + CRLF
  End If
Next i

txt = txt + CRLF + CRLF + "Note: For this computation, the iteration accuracy was increased as"
txt = txt + CRLF + "set_it_ctrl_density " + Chr(34) + "tol_liq_dens" + Chr(34) + ", -1E-15"

txt = txt + CRLF + CRLF + "The default iteration accuracy is as low as 0.1 ppm, i.e. equivalent to"
txt = txt + CRLF + "set_it_ctrl_density " + Chr(34) + "tol_liq_dens" + Chr(34) + ", -1E-7"

chk_OS2008_TableA8 = txt

set_it_ctrl_density "tol_liq_dens", tol

'Properties of water and seawater
'Computed from thermodynamic potentials of the seawater library
'for publication in Ocean Science, 2009
'R. Feistel, IOW, Version 20 Nov 2009
'Compiled on 20.11.2009 17:43:09
'
'Compared with Table A8 of the article
'R. Feistel, D.G. Wright, K. Miyagawa, A.H. Harvey, J. Hruby, D.R. Jackett, T.J. McDougall, W.Wagner
'Mutually consistent thermodynamic potentials for fluid water, ice and seawater:
'A new standard for oceanography.
'Ocean Sci., 4, 275-291, 2008
'www.ocean-sci.net/4/275/2008/:
'
'Table A8: Quadruple-precision results for the properties of water, ice and seawater at the standard ocean state,
'computed with all coefficients as given in Table 1 of this paper.  T = 273.15 K, p = 0.101 325 MPa and
'SA = 0.035 165 04 for seawater. The underlined numbers indicate the digits that, based on our tests,
'can reasonably be expected to be reproduced using double-precision code.
'
'Property         Property of water             Property of ice               Property of seawater         Unit
'g                0.1013427417|2939062882E+3    0.98267598403|431717064E+2   -2.E-29,  4.E-9               J kg-1
'this code:       0.101342741730827E+3          0.982675984029192E+2          0.143698741794651E-8         J kg-1
'
'(dg/dS)_T,p      0.0                           0.0                           0.639974067312299|04527E+5   J kg-1
'this code:       0.0                           0.0                           0.639974067312299E+5         J kg-1
'
'(dg/dT)_S,p      0.1476433763|4625266531       0.122078866129995|30642E+4    7.E-32, -6.E-11              J kg-1 K-1
'this code:       0.147643376383685             0.122078866129995E+4          0.374345832110379E-10        J kg-1 K-1
'
'(dg/dp)_S,T      0.100015693912169|26347E-2    0.109084344292643|52467E-2    0.97266121669484|934729E-3   m3 kg-1
'this code:       0.100015693912169E-2          0.109084344292644E-2          0.972661216694849E-3         m3 kg-1
'
'(d2g/dSdp)_T     0.0                           0.0                          -0.759615411515308|89445E-3   m3 kg-1
'this code:       0.0                           0.0                          -0.759615411515309E-3         m3 kg-1
'
'(d2g/dT2)_S,p   -0.1544735423197|7289339E+2   -0.767598511156675|09185E+1   -0.1459449308080|1366829E+2   J kg-1 K-2
'this code:      -0.154473542319782E+2         -0.767598511156675E+1         -0.145944930808023E+2         J kg-1 K-2
'
'(d2g/dTdp)_S    -0.6777003178|6558265755E-7    0.1743608249|6084962410E-6    0.515167556273|99375563E-7   m3 kg-1 K-1
'this code:      -0.677700317864069E-7          0.174360824960849E-6          0.515167556275508E-7         m3 kg-1 K-1
'
'(d2g/dp2)_S,T   -0.5089288946434|9017238E-12  -0.128484824639761|79327E-12  -0.4507753774102|0194745E-12  m3 kg-1 Pa-1
'this code:      -0.508928894643498E-12        -0.128484824639762E-12        -0.45077537741021E-12         m3 kg-1 Pa-1
'
'h                0.610139534|80411713295E+2   -0.333360155235678|74778E+6   -4.E-29,  2.E-8               J kg-1
'this code:       0.610139534716228E+2         -0.333360155235679E+6         -0.878826898614848E-8         J kg-1
'
'f                0.18398|728851226087838E-2   -0.1226211345|1089334305E+2   -0.9855489778|1605610114E+2   J kg-1
'this code:       0.183987432106544E-2         -0.122621134516018E+2         -0.985548977801686E+2         J kg-1
'
'u               -0.403269483|76093792920E+2   -0.3334706849475|3326883E+6   -0.985548977|81605610114E+2   J kg-1
'this code:      -0.403269483851381E+2         -0.333470684947534E+6         -0.985548977903938E+2         J kg-1
'
's               -0.147643376|34625266531      -0.122078866129995|30642E+4   -7.E-32,  6.E-11              J kg-1 K-1
'this code:      -0.147643376383685            -0.122078866129995E+4         -0.374345832110379E-10        J kg-1 K-1
'
'rho              0.99984308550433|049647E+3    0.916721832527381|67257E+3    0.102810719995400|78127E+4   kg m-3
'this code:       0.999843085504331E+3          0.916721832527382E+3          0.102810719995401E+4         kg m-3
'
'cp               0.4219444808464|5965831E+4    0.209669533322445|80134E+4    0.398648578502|08933494E+4   J kg-1 K-1
'this code:       0.421944480846485E+4          0.209669533322446E+4          0.398648578502115E+4         J kg-1 K-1
'
'alpha           -0.67759397686|198971741E-4    0.159840374979096|09918E-3    0.52964747378|800447017E-4   K-1
'this code:      -0.677593976860477E-4          0.159840374979096E-3          0.529647473789561E-4         K-1
'
'kappa_t          0.508849036322|65555122E-9    0.117784843895721|71289E-9    0.4634454110774|1383004E-9   Pa-1
'this code:       0.508849036322664E-9          0.117784843895722E-9          0.463445411077422E-9         Pa-1
'
'
'Note: For this computation, the iteration accuracy was increased as
'set_it_ctrl_density "tol_liq_dens", -1E-15
'
'The default iteration accuracy is as low as 0.1 ppm, i.e. equivalent to
'set_it_ctrl_density "tol_liq_dens", -1E-7


End Function

'==========================================================================
Public Function chk_OS2008_Table(ByVal table As String) As String

Select Case table
  Case "2":  chk_OS2008_Table = chk_OS2008_Table2
  Case "3":  chk_OS2008_Table = chk_OS2008_Table3
  Case "A1": chk_OS2008_Table = chk_OS2008_TableA1
  Case "A2": chk_OS2008_Table = chk_OS2008_TableA2
  Case "A3": chk_OS2008_Table = chk_OS2008_TableA3
  Case "A4": chk_OS2008_Table = chk_OS2008_TableA4
  Case "A5": chk_OS2008_Table = chk_OS2008_TableA5
  Case "A6": chk_OS2008_Table = chk_OS2008_TableA6
  Case "A7": chk_OS2008_Table = chk_OS2008_TableA7
  Case "A8": chk_OS2008_Table = chk_OS2008_TableA8
  Case Else: chk_OS2008_Table = "Tables 2, 3, A1,...,A8 are available here"
End Select

End Function

