Attribute VB_Name = "Maths_0_Mdl"
Option Explicit

'#########################################################################

'This module requires the library module
'     Constants_0_Mdl, file Constants_0.bas

'#########################################################################

'=========================================================================
'This module implements mathematical functions

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science 2008, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98

'Private Const PI = 3.14159265358979

Private Const Version = "08 Nov 2009"

'==========================================================================
Public Function arccos(ByVal x As Double) As Double
'Returns real function arccos(x)

Dim a As Double

a = arcsin(x)
If a = ErrorReturn Then
  arccos = a
Else
  arccos = 0.5 * PI - a
End If

End Function

'==========================================================================
Public Function arcsin(ByVal x As Double) As Double

'Returns real function arcsin(x) between -PI/2 and +PI/2

Select Case Abs(x)

  'For invalid arguments > 1, accept a certain rounding tolerance
  Case Is > 1.00000001: arcsin = ErrorReturn
  
  'compute regular values from intrinsic arctan
  Case Is < 1: arcsin = Atn(x / Sqr(1# - x * x))

  'values -PI/2 and +PI/2
  Case Else: arcsin = 0.5 * PI * Sgn(x)

End Select
End Function

'==========================================================================
Public Function get_CubicRoots(ByVal R As Double, _
                               ByVal S As Double, _
                               ByVal t As Double, _
                               ByRef x1 As Double, _
                               ByRef x2 As Double, _
                               ByRef x3 As Double) As Long

'input: r,s,t are the coefficients of the polynomial
'       x^3 + r * x^2 + s * x + t = 0
'output: x1, x2, x3 are the roots of the polynomial
'returns:
'         get_CubicRoots = 1:  x1 real solution, x2 real part, x3 imag part of the complex pair
'         get_CubicRoots = 3:  x1, x2, x3 real solutions

'requires the function arccos(x)

Dim p#, q#, vz#
Dim a#, phi#, u#, v#

p# = S - R ^ 2 / 3#
q# = 2# * R ^ 3 / 27# - R * S / 3# + t

a = (q / 2#) ^ 2 + (p / 3#) ^ 3
If a >= 0 Then
  u = -q / 2 + Sqr(a)
  If u >= 0 Then
    u = u ^ (1# / 3#)
  Else
    u = -(-u) ^ (1# / 3#)
  End If
  v = -q / 2 - Sqr(a)
  If v >= 0 Then
    v = v ^ (1# / 3#)
  Else
    v = -(-v) ^ (1# / 3#)
  End If
  If a = 0 Then
  '2 equal + 1 real solution
    get_CubicRoots = 3
    x1 = u + v - R / 3#
    x2 = -(u + v) / 2 - R / 3#
    x3 = x2
  Else
    get_CubicRoots = 1
    '2 complex solutions + 1 real solution
    x1 = u + v - R / 3#
    x2 = -(u + v) / 2 - R / 3#    'real part
    x3 = (u - v) * 0.5 * Sqr(3)   'imag part
  End If
  Exit Function
End If

'3 real solutions
get_CubicRoots = 3
a = Sqr(-p ^ 3 / 27#)
phi = arccos(-q / (2# * a))

If phi = ErrorReturn Then
  get_CubicRoots = 0
  Exit Function
End If

a = 2# * a ^ (1# / 3#)
x1 = a * Cos(phi / 3#) - R / 3#
x2 = a * Cos((phi + 2# * PI) / 3#) - R / 3#
x3 = a * Cos((phi + 4# * PI) / 3#) - R / 3#

End Function

'==========================================================================
Public Function matrix_solve(ByRef a() As Double, _
                             ByRef b() As Double, _
                             ByRef x() As Double, _
                             ByVal n As Long) As Long

'solves a system of linear equation by matrix inversion

Dim notok As Long, i As Long, j As Long

notok = matrix_invert(a(), n)
If notok <> 0 Then  'singular
  matrix_solve = notok
  Exit Function
End If

For i = 1 To n
  x(i) = 0
  For j = 1 To n
    x(i) = x(i) + a(i, j) * b(j)
  Next j
Next i

matrix_solve = 0

End Function

'==========================================================================
Private Function matrix_invert(ByRef a() As Double, ByVal n As Long) As Long

'inverts a matrix in place by Gauss elimination without pivoting

Dim i As Long, j As Long, k As Long

For i = 1 To n
  If a(i, i) = 0 Then  'matrix singular
    matrix_invert = i
    Exit Function
  End If
  a(i, i) = 1# / a(i, i)
  For j = 1 To n
    If j <> i Then
      a(j, i) = -a(j, i) * a(i, i)
      For k = 1 To n
        If k <> i Then a(j, k) = a(j, k) + a(j, i) * a(i, k)
      Next k
    End If
  Next j
  For k = 1 To n
    If k <> i Then a(i, k) = a(i, i) * a(i, k)
  Next k
Next i

matrix_invert = 0  'no error

End Function

'==========================================================================
Public Function EFormat(ByVal x As Double, _
                        Optional ByVal digits As Integer = 0, _
                        Optional ByVal Esymbol As String = "E") As String

'converts a number x into a Fortran E format string
'example: EFormat(273.15) = 0.27315E+3
'         EFormat(-1) = -0.1E+1

Dim i%, ex&, xe$, xx#, vz%, ee$, xa#, inc_ee%

If x = 0 Then
  EFormat$ = " 0.0"
  Exit Function
End If

xe$ = Str(x)
i = InStr(xe, "E")
If i <> 0 Then
  ex& = Val(Mid(xe, i + 1)) + 1
  xx# = 0.1 * Val(Left(xe, i - 1))
  xe$ = Str(xx) + "E" + IIf(ex > 0, "+", "") + Trim(Str(ex))
  xe = Left(xe, 1) + "0" + Mid(xe, 2)
Else
  vz% = Sgn(x)
  xa# = Abs(x)
  xe$ = Trim(Str(xa))
  i = InStr(xe, ".")
  If i = 0 Then
    ex& = Len(xe)
    Do While Right(xe, 1) = "0"
      xe = Left(xe, Len(xe) - 1)
    Loop
    xe = IIf(vz < 0, "-", " ") + "0." + xe + "E+" + Trim(Str(ex))
  Else
    If Left(xe, 2) <> ".0" Then
      ex& = i - 1
      xe = IIf(vz < 0, "-", " ") + "0." + Left(xe, i - 1) + Mid(xe, i + 1) + "E+" + Trim(Str(ex))
    Else
      Do While Left(xe, 2) = ".0"
        xe = "." + Mid(xe, 3)
        ex = ex - 1
      Loop
      xe = IIf(vz < 0, "-", " ") + "0." + Left(xe, i - 1) + Mid(xe, i + 1) + _
           "E" + IIf(Sgn(ex) < 0, "", "+") + Trim(Str(ex))
    End If
  End If
End If

If Right(xe, 3) = "E+0" Then xe = Left(xe, Len(xe) - 3)

If digits > 0 Then
  vz% = 1
  If Left(xe, 1) = "-" Then
    vz = -1
    xe = Mid(xe, 2)
  End If
  i = InStr(xe, "E")
  If i > 0 Then
    ee$ = Mid(xe, i)
    xe = Left(xe, i - 1)
  Else
    ee$ = ""
  End If
  If digits < 16 Then
    xe = Trim(Str(Round(Val(xe), digits)))
    If xe = "1" Then
      xe = ".1"
      inc_ee = 1
    Else
      inc_ee = 0
    End If
  End If
  If Len(xe) < digits + 1 Then
    xe = xe + String(digits + 1 - Len(xe), "0")
  End If
  If Left(xe, 1) = "." Then xe = "0" + xe
  If inc_ee = 1 Then
    If Len(ee) > 2 Then 'increment exponent by 1
      ee = Trim(Str(Val(Mid(ee, 2)) + 1))
      If ee = "0" Then
        ee = ""
      Else
        Select Case Left(ee, 1)
          Case "+", "-"
          Case Else: ee = "+" + ee
        End Select
        ee = "E" + ee
      End If
    ElseIf ee = "" Then
      ee = "E+1"
    End If
  End If
  xe = xe + ee$
  If vz < 0 Then xe = "-" + xe Else xe = " " + xe
End If

If Esymbol$ <> "E" Then
  i = InStr(xe, "E")
  If i <> 0 Then Mid(xe, i, 1) = Esymbol$
End If

EFormat$ = xe

End Function

'==========================================================================
Function TwoCols(ByVal x1 As Double, ByVal x2 As Double) As String

Const wid = 24

Dim col1 As String, col2 As String

col1 = EFormat(x1)
col2 = EFormat(x2)

If col1 = col2 Then
  TwoCols = Left(col1 + "," + Space(wid), wid) + col2
Else
  TwoCols = Left(col1 + "," + Space(wid), wid) + _
            Left(col2 + "," + Space(wid), wid) + _
            "Diff = " + EFormat(x1 - x2)
End If

End Function
