Attribute VB_Name = "Liq_Vap_4_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules:
'     Constants_0_Mdl, file Constants_0.bas
'     Flu_1_Mdl,       file Flu_1.bas
'     Flu_2_Mdl,       file Flu_2.bas
'     Flu_3a_Mdl,      file Flu_3a.bas
'     Maths_0_Mdl,     file Maths_0.bas

'#########################################################################

'=========================================================================
'This module implements the phase equilibrium properties of fluid water
'computed from IAPWS-95:

'Release on the IAPWS Formulation 1995 for the Thermodynamic Properties of
'Ordinary Water Substance for General and Scientific Use
'The International Association for the Properties of Water and Steam
'Fredericia, Denmark, September 1996

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98
'Private Const IsOK = -1

'Control parameters of the vapour pressure iteration
Private ctrl_initialized As Integer

Private ctrl_mode_liquid As Integer
Private ctrl_mode_vapour As Integer
Private ctrl_mode_temperature As Integer
Private ctrl_loop_maximum As Long
Private ctrl_init_d_liq As Double
Private ctrl_init_d_vap As Double
Private ctrl_init_t As Double
Private ctrl_eps_exit_p_vap As Double

'Properties of the current liquid-vapour equilibrium state
Private equi_liq_vap_done As Integer

Private equi_liq_vap_t As Double
Private equi_liq_vap_p_liq As Double
Private equi_liq_vap_p_vap As Double
Private equi_liq_vap_g_liq As Double
Private equi_liq_vap_g_vap As Double
Private equi_liq_vap_d_liq As Double
Private equi_liq_vap_d_vap As Double

Private Const Version = "20 Nov 2009"

'==========================================================================
Public Function liq_vap_vapourpressure_si(ByVal t_si As Double) As Double

'returns the absolute pressure in Pa of water vapour at vapour-liquid equilibrium
'as a function of absolute temperature t_si in K

'note: the accuracy of this function depends on the iteration settings in ths module

'check value with default settings: liq_vap_vapourpressure_si(300) = 3536.80675|227417

liq_vap_vapourpressure_si = ErrorReturn

If set_liq_vap_eq_at_t(t_si) = ErrorReturn Then Exit Function

liq_vap_vapourpressure_si = equi_liq_vap_p_vap

End Function

'==========================================================================
Public Function liq_vap_boilingtemperature_si(ByVal p_si As Double) As Double

'returns the absolute temperature in K of the vapour-liquid equilibrium
'as a function of absolute pressure p_si in Pa

'note: the accuracy of this function depends on the iteration settings in ths module

'check value with default settings: liq_vap_boilingtemperature_si(1E4) = 318.956328923936

liq_vap_boilingtemperature_si = ErrorReturn

If set_liq_vap_eq_at_p(p_si) = ErrorReturn Then Exit Function

liq_vap_boilingtemperature_si = equi_liq_vap_t

End Function

'==========================================================================
Public Function liq_vap_pressure_vap_si() As Double

'returns the absolute pressure in Pa of water vapour at vapour-liquid equilibrium
'after execution of set_liq_vap_eq_at_t or set_liq_vap_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_liq_vap_eq_at_t 300
'liq_vap_pressure_vap_si = 3536.80675227417

'set_liq_vap_eq_at_p 1E4
'liq_vap_pressure_vap_si = 10000.00000|00

liq_vap_pressure_vap_si = ErrorReturn

If equi_liq_vap_done <> IsOK Then Exit Function

liq_vap_pressure_vap_si = equi_liq_vap_p_vap

End Function

'==========================================================================
Public Function liq_vap_pressure_liq_si() As Double

'returns the absolute pressure in Pa of liquid water at vapour-liquid equilibrium
'after execution of set_liq_vap_eq_at_t or set_liq_vap_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_liq_vap_eq_at_t 300
'liq_vap_pressure_liq_si = 3536.80674986979

'set_liq_vap_eq_at_p 1E4
'liq_vap_pressure_liq_si = 9999.99999|853165

liq_vap_pressure_liq_si = ErrorReturn

If equi_liq_vap_done <> IsOK Then Exit Function

liq_vap_pressure_liq_si = equi_liq_vap_p_liq

End Function

'==========================================================================
Public Function liq_vap_temperature_si() As Double

'returns the absolute temperature in K at vapour-liquid equilibrium
'after execution of set_liq_vap_eq_at_t or set_liq_vap_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_liq_vap_eq_at_t 300
'liq_vap_temperature_si = 300

'set_liq_vap_eq_at_p 1E4
'liq_vap_temperature_si = 318.956328923936

liq_vap_temperature_si = ErrorReturn

If equi_liq_vap_done <> IsOK Then Exit Function

liq_vap_temperature_si = equi_liq_vap_t

End Function

'==========================================================================
Public Function liq_vap_chempot_si() As Double

'returns the chemical potential in J/kg of water vapour at vapour-liquid equilibrium
'after execution of set_liq_vap_eq_at_t or set_liq_vap_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_liq_vap_eq_at_t 300
'liq_vap_chempot_si = -5361.84908682075

'set_liq_vap_eq_at_p 1E4
'liq_vap_chempot_si = -15259.1024273131

liq_vap_chempot_si = ErrorReturn

If equi_liq_vap_done <> IsOK Then Exit Function

liq_vap_chempot_si = equi_liq_vap_g_vap

End Function

'==========================================================================
Public Function liq_vap_density_liq_si() As Double

'returns the density in kg/m3 of liquid water at vapour-liquid equilibrium
'after execution of set_liq_vap_eq_at_t or set_liq_vap_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_liq_vap_eq_at_t 300
'liq_vap_density_liq_si = 996.513027468131

'set_liq_vap_eq_at_p 1E4
'liq_vap_density_liq_si = 989.833275364924

liq_vap_density_liq_si = ErrorReturn

If equi_liq_vap_done <> IsOK Then Exit Function

liq_vap_density_liq_si = equi_liq_vap_d_liq

End Function

'==========================================================================
Public Function liq_vap_density_vap_si() As Double

'returns the density in kg/m3 of water vapour at vapour-liquid equilibrium
'after execution of set_liq_vap_eq_at_t or set_liq_vap_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_liq_vap_eq_at_t 300
'liq_vap_density_vap_si = 2.55896736829199E-02

'set_liq_vap_eq_at_p 1E4
'liq_vap_density_vap_si = 6.81657223094408E-02

liq_vap_density_vap_si = ErrorReturn

If equi_liq_vap_done <> IsOK Then Exit Function

liq_vap_density_vap_si = equi_liq_vap_d_vap

End Function

'==========================================================================
Public Function liq_vap_entropy_liq_si() As Double

'returns the specific entropy in J/(kg K) of liquid water at vapour-liquid equilibrium
'after execution of set_liq_vap_eq_at_t or set_liq_vap_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_liq_vap_eq_at_t 300
'liq_vap_entropy_liq_si = 393.089029801268

'set_liq_vap_eq_at_p 1E4
'liq_vap_entropy_liq_si = 649.195605195818

liq_vap_entropy_liq_si = ErrorReturn

If equi_liq_vap_done <> IsOK Then Exit Function

liq_vap_entropy_liq_si = flu_entropy_si(equi_liq_vap_t, equi_liq_vap_d_liq)

End Function

'==========================================================================
Public Function liq_vap_entropy_vap_si() As Double

'returns the specific entropy in J/(kg K) of water vapour at vapour-liquid equilibrium
'after execution of set_liq_vap_eq_at_t or set_liq_vap_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_liq_vap_eq_at_t 300
'liq_vap_entropy_vap_si = 8517.38650060567

'set_liq_vap_eq_at_p 1E4
'liq_vap_entropy_vap_si = 8148.82019424941

liq_vap_entropy_vap_si = ErrorReturn

If equi_liq_vap_done <> IsOK Then Exit Function

liq_vap_entropy_vap_si = flu_entropy_si(equi_liq_vap_t, equi_liq_vap_d_vap)

End Function

'==========================================================================
Public Function liq_vap_enthalpy_liq_si() As Double

'returns the specific enthalpy in J/kg of liquid water at vapour-liquid equilibrium
'after execution of set_liq_vap_eq_at_t or set_liq_vap_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_liq_vap_eq_at_t 300
'liq_vap_enthalpy_liq_si = 112564.859853558

'set_liq_vap_eq_at_p 1E4
'liq_vap_enthalpy_liq_si = 191805.944559497

liq_vap_enthalpy_liq_si = ErrorReturn

If equi_liq_vap_done <> IsOK Then Exit Function

liq_vap_enthalpy_liq_si = flu_enthalpy_si(equi_liq_vap_t, equi_liq_vap_d_liq)

End Function

'==========================================================================
Public Function liq_vap_enthalpy_vap_si() As Double

'returns the specific enthalpy in J/kg of water vapour at vapour-liquid equilibrium
'after execution of set_liq_vap_eq_at_t or set_liq_vap_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_liq_vap_eq_at_t 300
'liq_vap_enthalpy_vap_si = 2549854.10109488

'set_liq_vap_eq_at_p 1E4
'liq_vap_enthalpy_vap_si = 2583858.67179171

liq_vap_enthalpy_vap_si = ErrorReturn

If equi_liq_vap_done <> IsOK Then Exit Function

liq_vap_enthalpy_vap_si = flu_enthalpy_si(equi_liq_vap_t, equi_liq_vap_d_vap)

End Function

'==========================================================================
Public Function liq_vap_volume_evap_si() As Double

'returns the specific evaporation volume in m3/kg at vapour-liquid equilibrium
'after execution of set_liq_vap_eq_at_t or set_liq_vap_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_liq_vap_eq_at_t 300
'liq_vap_volume_evap_si = 39.0772595686141

'set_liq_vap_eq_at_p 1E4
'liq_vap_volume_evap_si = 14.6691196140812

liq_vap_volume_evap_si = ErrorReturn

If equi_liq_vap_done <> IsOK Then Exit Function

If equi_liq_vap_d_liq <= 0 Then Exit Function
If equi_liq_vap_d_vap <= 0 Then Exit Function

liq_vap_volume_evap_si = 1# / equi_liq_vap_d_vap - 1# / equi_liq_vap_d_liq

End Function

'==========================================================================
Public Function liq_vap_entropy_evap_si() As Double

'returns the specific evaporation entropy in J/(kg K) of water
'after execution of set_liq_vap_eq_at_t or set_liq_vap_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_liq_vap_eq_at_t 300
'liq_vap_entropy_evap_si = 8124.29747080441

'set_liq_vap_eq_at_p 1E4
'liq_vap_entropy_evap_si = 7499.62458905359

Dim sl As Double, sv As Double

liq_vap_entropy_evap_si = ErrorReturn

If equi_liq_vap_done <> IsOK Then Exit Function

sl = flu_entropy_si(equi_liq_vap_t, equi_liq_vap_d_liq)
If sl = ErrorReturn Then Exit Function
sv = flu_entropy_si(equi_liq_vap_t, equi_liq_vap_d_vap)
If sv = ErrorReturn Then Exit Function

liq_vap_entropy_evap_si = sv - sl

End Function

'==========================================================================
Public Function liq_vap_enthalpy_evap_si() As Double

'returns the specific evaporation enthalpy in J/kg of water
'after execution of set_liq_vap_eq_at_t or set_liq_vap_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_liq_vap_eq_at_t 300
'liq_vap_enthalpy_evap_si = 2437289.24124132

'set_liq_vap_eq_at_p 1E4
'liq_vap_enthalpy_evap_si = 2392052.72723222

Dim hl As Double, hv As Double

liq_vap_enthalpy_evap_si = ErrorReturn

If equi_liq_vap_done <> IsOK Then Exit Function

hl = flu_enthalpy_si(equi_liq_vap_t, equi_liq_vap_d_liq)
If hl = ErrorReturn Then Exit Function
hv = flu_enthalpy_si(equi_liq_vap_t, equi_liq_vap_d_vap)
If hv = ErrorReturn Then Exit Function

liq_vap_enthalpy_evap_si = hv - hl

End Function

'==========================================================================
Public Function set_liq_vap_eq_at_p(ByVal p_si As Double) As Double

'this function computes the liquid-vapour equilibrium at given pressure in Pa
'by 3-dimensional Newton iteration

'critical point
Const dc = CP_density_si
Const pc = CP_pressure_si
Const Tc = CP_temperature_si
Const gc = CP_chempot_si

'triple point
Const dl_triple = TP_density_liq_IAPWS95_si
Const dv_triple = TP_density_vap_IAPWS95_si
Const t_triple = TP_temperature_si

Dim t_si As Double
Dim pl As Double, pv As Double
Dim gl As Double, gv As Double
Dim dl As Double, dv As Double

Dim fv_d As Double, eps As Double
Dim maxit As Long

If equi_liq_vap_done = IsOK And _
   p_si = equi_liq_vap_p_vap Then
  'the requested state has already been computed earlier
  set_liq_vap_eq_at_p = IsOK
  Exit Function
End If

clear_liq_vap_state 'waste any previous state

set_liq_vap_eq_at_p = ErrorReturn

If p_si <= 0 Then Exit Function
If p_si > pc Then Exit Function

If p_si = pc Then
  set_liq_vap_state Tc, pc, pc, gc, gc, dc, dc
  set_liq_vap_eq_at_p = IsOK
  Exit Function
End If

init_it_ctrl_liq_vap

'set initial temperature guess
Select Case ctrl_mode_temperature
  Case 0:  t_si = aux_boil_temperature(p_si)
  Case -1: t_si = t_triple
  Case 1:  t_si = ctrl_init_t
  Case Else: Exit Function
End Select

'set initial liquid density guess
Select Case ctrl_mode_liquid
  Case 0:  'default liquid density:
           If t_si < 350 Then
             dl = aux_liq_density_triplepoint(t_si)
           Else
             dl = aux_liq_density_critical(t_si)
           End If
  Case -1: dl = dl_triple
  Case 1:  dl = ctrl_init_d_liq
  Case Else: Exit Function
End Select

'set initial vapour density guess
Select Case ctrl_mode_vapour
  Case 0:
           If t_si < 550 Then
             dv = aux_vap_density_triplepoint(t_si)
           Else
             dv = aux_vap_density_critical(t_si)
           End If
  Case -1: dv = dv_triple
  Case 1:  dv = ctrl_init_d_vap
  Case Else: Exit Function
End Select

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: set_liq_vap_eq_at_p = IsOK
           pl = flu_pressure_si(t_si, dl)
           pv = flu_pressure_si(t_si, dv)
           gl = flu_gibbs_energy_si(t_si, dl)
           gv = flu_gibbs_energy_si(t_si, dv)
           set_liq_vap_state t_si, pl, pv, gl, gv, dl, dv
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit
eps = ctrl_eps_exit_p_vap
If eps = 0 Then Exit Function

'run iteration loop
If liq_vap_iteration_at_p(p_si, maxit, eps, t_si, dl, dv, gl, gv, pl, pv) = ErrorReturn Then
  Exit Function
End If

set_liq_vap_state t_si, pl, pv, gl, gv, dl, dv

set_liq_vap_eq_at_p = IsOK

End Function

'==========================================================================
Public Function set_liq_vap_eq_at_t(ByVal t_si As Double) As Double

'this function computes the liquid-vapour equilibrium at given temperature in K
'by 2-dimensional Newton iteration

'critical point
Const dc = CP_density_si
Const pc = CP_pressure_si
Const Tc = CP_temperature_si
Const gc = CP_chempot_si

'triple point densities
Const dl_triple = TP_density_liq_IAPWS95_si
Const dv_triple = TP_density_vap_IAPWS95_si

Dim pl As Double, pv As Double
Dim gl As Double, gv As Double
Dim dl As Double, dv As Double

Dim fv_d As Double, eps As Double
Dim maxit As Long

If equi_liq_vap_done = IsOK And _
   t_si = equi_liq_vap_t Then
  'the requested state has already been computed earlier
  set_liq_vap_eq_at_t = IsOK
  Exit Function
End If

clear_liq_vap_state 'waste any previous state

set_liq_vap_eq_at_t = ErrorReturn

If t_si <= 0 Then Exit Function
If t_si > Tc Then Exit Function

If t_si = Tc Then
  set_liq_vap_state t_si, pc, pc, gc, gc, dc, dc
  set_liq_vap_eq_at_t = IsOK
  Exit Function
End If

init_it_ctrl_liq_vap

'set initial liquid density guess
Select Case ctrl_mode_liquid
  Case 0:  'default liquid density:
           If t_si < 350 Then
             dl = aux_liq_density_triplepoint(t_si)
           Else
             dl = aux_liq_density_critical(t_si)
           End If
  Case -1: dl = dl_triple
  Case 1:  dl = ctrl_init_d_liq
  Case Else: Exit Function
End Select

'set initial vapour density guess
Select Case ctrl_mode_vapour
  Case 0:
           If t_si < 550 Then
             dv = aux_vap_density_triplepoint(t_si)
           Else
             dv = aux_vap_density_critical(t_si)
           End If
  Case -1: dv = dv_triple
  Case 1:  dv = ctrl_init_d_vap
  Case Else: Exit Function
End Select

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: set_liq_vap_eq_at_t = IsOK
           pl = flu_pressure_si(t_si, dl)
           pv = flu_pressure_si(t_si, dv)
           gl = flu_gibbs_energy_si(t_si, dl)
           gv = flu_gibbs_energy_si(t_si, dv)
           set_liq_vap_state t_si, pl, pv, gl, gv, dl, dv
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit
eps = ctrl_eps_exit_p_vap
If eps = 0 Then Exit Function

'run iteration loop
If liq_vap_iteration_at_t(t_si, maxit, eps, dl, dv, gl, gv, pl, pv) = ErrorReturn Then
  Exit Function
End If

set_liq_vap_state t_si, pl, pv, gl, gv, dl, dv

set_liq_vap_eq_at_t = IsOK

End Function

'==========================================================================
Private Sub set_liq_vap_state(ByVal t As Double, _
                              ByVal pl As Double, _
                              ByVal pv As Double, _
                              ByVal gl As Double, _
                              ByVal gv As Double, _
                              ByVal dl As Double, _
                              ByVal dv As Double)

'stores the actual properties as the current equilibrium state descriptor

equi_liq_vap_done = IsOK

equi_liq_vap_t = t
equi_liq_vap_p_liq = pl
equi_liq_vap_p_vap = pv
equi_liq_vap_g_liq = gl
equi_liq_vap_g_vap = gv
equi_liq_vap_d_liq = dl
equi_liq_vap_d_vap = dv

End Sub

'==========================================================================
Private Sub clear_liq_vap_state()

'clears the current equilibrium state descriptor

equi_liq_vap_done = 0

End Sub
'==========================================================================
Private Function liq_vap_iteration_at_p(ByVal p_si As Double, _
                                        ByVal maxit As Long, _
                                        ByVal eps As Double, _
                                        ByRef t_si As Double, _
                                        ByRef d_liq_si As Double, _
                                        ByRef d_vap_si As Double, _
                                        ByRef g_liq_si As Double, _
                                        ByRef g_vap_si As Double, _
                                        ByRef p_liq_si As Double, _
                                        ByRef p_vap_si As Double) As Double

'this function returns the liquid-vapour phase equilibrium from equal pressures,
'temperatures and chemical potentials of the two phases at given pressure, p_si,
'from initial guesses for the liquid density, d_liq_si, the vapour density, d_vap_si,
'and the temperature, t_si
'The iteration limit eps refers to the error in vapour pressure

'output:    liq_vap_iteration_at_p = IsOK if successful
'           liq_vap_iteration_at_p = ErrorReturn is returned if
'           - the maximum number of iterations is exceeded without meeting the exit criterion
'           - the function call to flu_f_si has returned an error
'           - densities or temperature have taken a zero or negative value during the iteration
'     t_si: absolute temperature in K
' d_liq_si: liquid density at the liquid-vapour equilibrium in kg/m3
' d_vap_si: vapour density at the liquid-vapour equilibrium in kg/m3
' g_liq_si: liquid chemical potential at the liquid-vapour equilibrium in J/kg
' g_vap_si: vapour chemical potential at the liquid-vapour equilibrium in J/kg
' p_liq_si: liquid pressure at the liquid-vapour equilibrium in Pa
' p_vap_si: vapour pressure at the liquid-vapour equilibrium in Pa

'input: p_si: absolute pressure in Pa
'      maxit: maximum number of iteration steps to be done
'        eps: required accuracy of vapour pressure
'             eps > 0: absolute vapour pressure tolerance in Pa
'             eps < 0: relative vapour pressure tolerance
'       t_si: initial guess of absolute temperature in K
'   d_liq_si: initial guess of liquid density in kg/m3
'   d_vap_si: initial guess of vapour density in kg/m3

'the critical properties of IAPWS-95
Const dc = CP_density_si
Const pc = CP_pressure_si
Const Tc = CP_temperature_si
Const gc = CP_chempot_si

Dim dl As Double, dv As Double
Dim pl As Double, pv As Double
Dim gl As Double, gv As Double
Dim t As Double
Dim fl As Double, fl_d As Double, fl_dd As Double
Dim fv As Double, fv_d As Double, fv_dd As Double
Dim fl_t As Double, fl_td As Double
Dim fv_t As Double, fv_td As Double

Dim it As Long

Dim a(3, 3) As Double, b(3) As Double, x(3) As Double
Dim det As Double, ddl As Double, ddv As Double, dt As Double, pv_old As Double

liq_vap_iteration_at_p = ErrorReturn

If p_si = pc Then 'equilibrium = critical point
  t_si = Tc
  d_liq_si = dc
  d_vap_si = dc
  g_liq_si = gc
  g_vap_si = gc
  p_liq_si = pc
  p_vap_si = pc
  liq_vap_iteration_at_p = IsOK
  Exit Function
End If

If d_liq_si < dc Or _
   d_vap_si <= 0 Or d_vap_si > dc Or _
   p_si <= 0 Or p_si > pc Then
  t_si = ErrorReturn
  d_liq_si = ErrorReturn
  d_vap_si = ErrorReturn
  g_liq_si = ErrorReturn
  g_vap_si = ErrorReturn
  p_liq_si = ErrorReturn
  p_vap_si = ErrorReturn
  Exit Function
End If

dl = d_liq_si
dv = d_vap_si
t = t_si

check_limits = check_limits - 1

For it = 0 To maxit

  'Derivatives of the Helmholtz function for 3D-Newton iteration
  fl = flu_f_si(0, 0, t, dl)
  If fl = ErrorReturn Then Exit For
  fv = flu_f_si(0, 0, t, dv)
  If fv = ErrorReturn Then Exit For
  
  fl_d = flu_f_si(0, 1, t, dl)
  If fl_d = ErrorReturn Then Exit For
  fv_d = flu_f_si(0, 1, t, dv)
  If fv_d = ErrorReturn Then Exit For
  fl_dd = flu_f_si(0, 2, t, dl)
  If fl_dd = ErrorReturn Then Exit For
  fv_dd = flu_f_si(0, 2, t, dv)
  If fv_dd = ErrorReturn Then Exit For
  
  fl_t = flu_f_si(1, 0, t, dl)
  If fl_t = ErrorReturn Then Exit For
  fv_t = flu_f_si(1, 0, t, dv)
  If fv_t = ErrorReturn Then Exit For
  fl_td = flu_f_si(1, 1, t, dl)
  If fl_td = ErrorReturn Then Exit For
  fv_td = flu_f_si(1, 1, t, dv)
  If fv_td = ErrorReturn Then Exit For

  pv_old = pv  'keep previous pv to check accuracy
  'liquid pressure and liquid + vapour chemical potentials
  pl = dl ^ 2 * fl_d     'pressure of liquid  'this often takes negative values, temporarily
  pv = dv ^ 2 * fv_d     'pressure of vapour
  gl = fl + dl * fl_d    'chem. pot. of liquid
  gv = fv + dv * fv_d    'chem. pot. of vapour

  If it > 0 Then
    'check absolute or relative error limit
    If (eps > 0 And Abs(pv - pv_old) < eps) Or _
       (eps < 0 And Abs(pv - pv_old) < -eps * pv) Then
      liq_vap_iteration_at_p = IsOK
      p_liq_si = pl
      p_vap_si = pv
      g_liq_si = gl
      g_vap_si = gv
      d_liq_si = dl
      d_vap_si = dv
      t_si = t
      Exit For
    End If
  End If
  
  If it = maxit Then Exit For
  
  'coefficient matrix
  a(1, 1) = dl * (2 * fl_d + dl * fl_dd)
  a(1, 2) = 0
  a(1, 3) = dl ^ 2 * fl_td
  a(2, 1) = 0
  a(2, 2) = dv * (2 * fv_d + dv * fv_dd)
  a(2, 3) = dv ^ 2 * fv_td
  a(3, 1) = (2 * fl_d + dl * fl_dd)
  a(3, 2) = -(2 * fv_d + dv * fv_dd)
  a(3, 3) = fl_t + dl * fl_td - fv_t - dv * fv_td

  'right-hand sides, must vanish in equilibrium
  b(1) = p_si - pl
  b(2) = p_si - pv
  b(3) = gv - gl

  'solve equations
  If matrix_solve(a(), b(), x(), 3) <> 0 Then Exit For 'matrix singular
  ddl = x(1)
  ddv = x(2)
  dt = x(3)
  
  'update densities & temperature
  dl = dl + ddl
  If dl <= 0 Then Exit For
  dv = dv + ddv
  If dv <= 0 Then Exit For
  t = t + dt
  If t <= 0 Then Exit For
  
Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Or _
     d_liq_si <= flu_dmin Or d_liq_si > flu_dmax Then
    t_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_vap_si = ErrorReturn
    g_liq_si = ErrorReturn
    g_vap_si = ErrorReturn
    p_liq_si = ErrorReturn
    p_vap_si = ErrorReturn
    liq_vap_iteration_at_p = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Function liq_vap_iteration_at_t(ByVal t_si As Double, _
                                        ByVal maxit As Long, _
                                        ByVal eps As Double, _
                                        ByRef d_liq_si As Double, _
                                        ByRef d_vap_si As Double, _
                                        ByRef g_liq_si As Double, _
                                        ByRef g_vap_si As Double, _
                                        ByRef p_liq_si As Double, _
                                        ByRef p_vap_si As Double) As Double

'this function returns the liquid-vapour phase equilibrium from equal pressures
'and chemical potentials of the two phases at given temperature, t_si,
'from initial guesses for the liquid density, d_liq_si, and the vapour density, d_vap_si.
'The iteration limit eps refers to the error in vapour pressure

'output:    liq_vap_iteration_at_t = IsOK if successful
'           liq_vap_iteration_at_t = ErrorReturn is returned if
'           - the maximum number of iterations is exceeded without meeting the exit criterion
'           - the function call to flu_f_si has returned an error
'           - densities have taken a zero or negative value during the iteration
' d_liq_si: liquid density at the liquid-vapour equilibrium in kg/m3
' d_vap_si: vapour density at the liquid-vapour equilibrium in kg/m3
' g_liq_si: liquid chemical potential at the liquid-vapour equilibrium in J/kg
' g_vap_si: vapour chemical potential at the liquid-vapour equilibrium in J/kg
' p_liq_si: liquid pressure at the liquid-vapour equilibrium in Pa
' p_vap_si: vapour pressure at the liquid-vapour equilibrium in Pa

'input: t_si: absolute temperature in K
'      maxit: maximum number of iteration steps to be done
'        eps: required accuracy of vapour pressure
'             eps > 0: absolute vapour pressure tolerance in Pa
'             eps < 0: relative vapour pressure tolerance
'   d_liq_si: initial guess of liquid density in kg/m3
'   d_vap_si: initial guess of vapour density in kg/m3

'the critical properties of IAPWS-95
Const dc = CP_density_si
Const pc = CP_pressure_si
Const Tc = CP_temperature_si
Const gc = CP_chempot_si

Dim dl As Double, dv As Double
Dim pl As Double, pv As Double
Dim gl As Double, gv As Double
Dim fv As Double, fv_d As Double, fv_dd As Double
Dim fl As Double, fl_d As Double, fl_dd As Double

Dim it As Long

Dim a(2, 2) As Double, b(2) As Double, x(2) As Double
Dim ddl As Double, ddv As Double, pv_old As Double

liq_vap_iteration_at_t = ErrorReturn

If t_si = Tc Then 'equlibrium = critical point
  d_liq_si = dc
  d_vap_si = dc
  g_liq_si = gc
  g_vap_si = gc
  p_liq_si = pc
  p_vap_si = pc
  liq_vap_iteration_at_t = IsOK
  Exit Function
End If

If t_si > Tc Or _
   d_liq_si < dc Or _
   d_vap_si <= 0 Or d_vap_si > dc Then  'no equilibrium in the supercritical range
   d_liq_si = ErrorReturn
   d_vap_si = ErrorReturn
   g_liq_si = ErrorReturn
   g_vap_si = ErrorReturn
   p_liq_si = ErrorReturn
   p_vap_si = ErrorReturn
   Exit Function
End If

If check_limits <> 1 Then
  If t_si <= 0 Then
     d_liq_si = ErrorReturn
     d_vap_si = ErrorReturn
     g_liq_si = ErrorReturn
     g_vap_si = ErrorReturn
     p_liq_si = ErrorReturn
     p_vap_si = ErrorReturn
     Exit Function
  End If
Else
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Then
    d_liq_si = ErrorReturn
    d_vap_si = ErrorReturn
    g_liq_si = ErrorReturn
    g_vap_si = ErrorReturn
    p_liq_si = ErrorReturn
    p_vap_si = ErrorReturn
    Exit Function
  End If
End If

If d_liq_si < dc Then Exit Function
If d_vap_si <= 0 Then Exit Function
If d_vap_si > dc Then Exit Function

check_limits = check_limits - 1

dl = d_liq_si
dv = d_vap_si

For it = 0 To maxit

  'Derivatives of the Helmholtz function for 2D-Newton iteration
  fl = flu_f_si(0, 0, t_si, dl)
  If fl = ErrorReturn Then Exit For
  fv = flu_f_si(0, 0, t_si, dv)
  If fv = ErrorReturn Then Exit For
  fl_d = flu_f_si(0, 1, t_si, dl)
  If fl_d = ErrorReturn Then Exit For
  fv_d = flu_f_si(0, 1, t_si, dv)
  If fv_d = ErrorReturn Then Exit For
  fl_dd = flu_f_si(0, 2, t_si, dl)
  If fl_dd = ErrorReturn Then Exit For
  fv_dd = flu_f_si(0, 2, t_si, dv)
  If fv_dd = ErrorReturn Then Exit For
  
  pv_old = pv  'keep previous pv to check accuracy
  'liquid + vapour pressure and chemical potentials
  pl = dl ^ 2 * fl_d     'pressure of liquid  'this often takes negative values temporarily
  pv = dv ^ 2 * fv_d     'pressure of vapour
  gl = fl + dl * fl_d    'chem. pot. of liquid
  gv = fv + dv * fv_d    'chem. pot. of vapour

  If it > 0 Then
    'check absolute or relative error limit
    If (eps > 0 And Abs(pv - pv_old) < eps) Or _
       (eps < 0 And Abs(pv - pv_old) < -eps * pv) Then
      liq_vap_iteration_at_t = IsOK
      p_liq_si = pl
      p_vap_si = pv
      g_liq_si = gl
      g_vap_si = gv
      d_liq_si = dl
      d_vap_si = dv
      Exit For
    End If
  End If
  
  If it = maxit Then Exit For
  
  'coefficient matrix
  a(2, 1) = 2 * fl_d + dl * fl_dd
  a(1, 1) = dl * a(2, 1)
  a(2, 2) = -2 * fv_d - dv * fv_dd
  a(1, 2) = dv * a(2, 2)
  'right-hand sides, must vanish in equilibrium
  b(1) = pv - pl
  b(2) = gv - gl

  'solve equations
  If matrix_solve(a(), b(), x(), 2) <> 0 Then Exit For 'matrix singular
  ddl = x(1)
  ddv = x(2)

  'update densities
  dl = dl + ddl
  If dl <= 0 Then Exit For
  dv = dv + ddv
  If dv <= 0 Then Exit For

Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'FLU_LIMITS
  If d_liq_si <= flu_dmin Or d_liq_si > flu_dmax Then
    d_liq_si = ErrorReturn
    d_vap_si = ErrorReturn
    g_liq_si = ErrorReturn
    g_vap_si = ErrorReturn
    p_liq_si = ErrorReturn
    p_vap_si = ErrorReturn
    liq_vap_iteration_at_t = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Sub init_it_ctrl_liq_vap()

'triple point
Const dl_triple = TP_density_liq_IAPWS95_si
Const dv_triple = TP_density_vap_IAPWS95_si
Const t_triple = TP_temperature_si

If ctrl_initialized = IsOK Then Exit Sub

ctrl_initialized = IsOK

'Set default values and modes for density iteration
ctrl_mode_liquid = 0
ctrl_mode_vapour = 0
ctrl_mode_temperature = 0
ctrl_loop_maximum = 100
ctrl_init_d_liq = dl_triple
ctrl_init_d_vap = dv_triple
ctrl_init_t = t_triple
ctrl_eps_exit_p_vap = -0.0000001 'relative, 0.1 ppm

End Sub

'==========================================================================
Public Sub set_it_ctrl_liq_vap(ByVal key As String, ByVal value As Double)

'this sub sets control parameters for the iteration used to compute
'IAPWS-95 vapour pressure from temperature

'key              value
'it_steps         0           set iteration number to default (100)
'it_steps         n > 0       set iteration number to n
'it_steps        -1           do not iterate, use initial vapour density to compute vapour pressure

'init_liq_dens    0           use default liquid density to start:
'                         T < 350 K: liquid density correlation function, d = aux_liq_density_triplepoint(T)
'                         T > 350 K: liquid density correlation function, d = aux_liq_density_critical(T)
'init_liq_dens   -1           use triple point liquid density to start
'init_liq_dens    d > 0       use value d as liquid density to start

'init_vap_dens    0           use default vapour density to start
'                          T < 550 K: vapour density correlation function, d = aux_vap_density_triplepoint(T)
'                          T > 550 K: vapour density correlation function, d = aux_vap_density_critical(T)
'init_vap_dens   -1           use triple point vapour density to start
'init_vap_dens    d > 0       use value d as vapour density to start

'init_temp        0           use default temperature to start ( = aux_boil_temperature(p))
'init_temp       -1           use triple point temperature to start
'init_temp        t > 0       use value t as temperature to start

'tol_vap_press    0           use default exit accuracy for vapour pressure (0.1 ppm)
'tol_vap_press    eps         use eps as exit accuracy for vapour pressure (eps < 0 means relative error)


init_it_ctrl_liq_vap

clear_liq_vap_state

Select Case LCase(Trim(key))

  Case "it_steps":   'iteration steps
    Select Case value
      Case 0:      ctrl_loop_maximum = 100  'default = 100
      Case Is < 0: ctrl_loop_maximum = -1
      Case Else:   ctrl_loop_maximum = value
    End Select

  Case "init_liq_dens":   'start liquid density
    Select Case CLng(value)
      Case 0:       ctrl_mode_liquid = 0    'default = aux polynomial
      Case Is < -1: 'ignore it
      Case Is < 0:  ctrl_mode_liquid = value
      Case Else:    ctrl_mode_liquid = 1
                    ctrl_init_d_liq = value
    End Select

  Case "init_vap_dens":   'start vapour density
    Select Case CLng(value)
      Case 0:       ctrl_mode_vapour = 0    'default = aux polynomial
      Case Is < -1: 'ignore it
      Case Is < 0:  ctrl_mode_vapour = value
      Case Else:    ctrl_mode_vapour = 1
                    ctrl_init_d_vap = value
    End Select

  Case "init_temp":   'start temperature
    Select Case CLng(value)
      Case 0:       ctrl_mode_temperature = 0    'default = aux polynomial
      Case Is < -1: 'ignore it
      Case Is < 0:  ctrl_mode_temperature = value
      Case Else:    ctrl_mode_temperature = 1
                    ctrl_init_t = value
    End Select

  Case "tol_vap_press":      'required vapour pressure tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_p_vap = -0.0000001   'default = 0.1 ppm relative
      Case Else:   ctrl_eps_exit_p_vap = value
    End Select

End Select

End Sub

'==========================================================================
Private Function aux_liq_density_critical(ByVal t_si As Double) As Double

'computes the liquid density on the saturation curve from a correlation polynomial
'as a first guess for liq_vap_iteration_at_t (300 K < T < Tc with rms = 1 K)

' T/Tc - 1 = a1 * (d/dc-1)^3 + a2 * (d/dc-1)^6 + a3 * (d/dc-1)^9

Const Tc = CP_temperature_si
Const dc = CP_density_si

Const a1 = -7.34017329598858E-02
Const a2 = 5.70516487711065E-03
Const a3 = -4.31313846955949E-04

Dim R As Double, S As Double, t As Double
Dim x1 As Double, x2 As Double, x3 As Double
Dim d As Double

aux_liq_density_critical = ErrorReturn

'solve       x^3 + r * x^2 + s * x + t = 0
R = a2 / a3
S = a1 / a3
t = (1 - t_si / Tc) / a3

If get_CubicRoots(R, S, t, x1, x2, x3) <> 1 Then Exit Function

If x1 <> 0 Then
  x1 = Sgn(x1) * Abs(x1) ^ (1# / 3#)
End If
d = (x1 + 1) * dc
If d < dc Then Exit Function

aux_liq_density_critical = d

End Function

'==========================================================================
Private Function aux_vap_density_critical(ByVal t_si As Double) As Double

'computes the vapour density on the saturation curve from a correlation polynomial
'as a first guess for liq_vap_iteration_at_t (550 K < T < Tc with rms = 0.4 K)

' T/Tc - 1 = a1 * (1-d/dc)^4 + a2 * (1-d/dc)^8 + a3 * (1-d/dc)^12

Const Tc = CP_temperature_si
Const dc = CP_density_si

Const a1 = -0.237216002118091
Const a2 = 0.186593118426901
Const a3 = -0.258472040504799

Dim R As Double, S As Double, t As Double
Dim x1 As Double, x2 As Double, x3 As Double
Dim d As Double

aux_vap_density_critical = ErrorReturn

'solve       x^3 + r * x^2 + s * x + t = 0
R = a2 / a3
S = a1 / a3
t = (1 - t_si / Tc) / a3

If get_CubicRoots(R, S, t, x1, x2, x3) <> 1 Then Exit Function

If x1 < 0 Then Exit Function
If x1 > 0 Then
  x1 = Abs(x1) ^ 0.25
End If
d = (1 - x1) * dc
If d <= 0 Then Exit Function

aux_vap_density_critical = d

End Function

'==========================================================================
Private Function aux_liq_density_triplepoint(ByVal t_si As Double) As Double

'computes the liquid density on the saturation curve from a correlation polynomial
'as a first guess for liq_vap_iteration_at_t (Tt < T < 350 K with rms = 0.002 kg/m3)

' (d/dt-1) = a1 * (T/Tt-1) + a2 * (T/Tt-1)^2 + a3 * (T/Tt-1)^3 + a4 * (T/Tt-1)^4 + a5 * (T/Tt-1)^5

Const Tt = TP_temperature_si
Const dt = TP_density_liq_IAPWS95_si

Const a1 = 1.80066818428501E-02
Const a2 = -0.648994409718973
Const a3 = 1.56594764908347
Const a4 = -3.18116999660964
Const a5 = 2.98590977093295

Dim tr As Double
Dim d As Double

tr = t_si / Tt - 1#
d = dt * (a1 * tr + a2 * tr ^ 2 + a3 * tr ^ 3 + a4 * tr ^ 4 + a5 * tr ^ 5 + 1)

aux_liq_density_triplepoint = d

End Function

'==========================================================================
Private Function aux_vap_density_triplepoint(ByVal t_si As Double) As Double

'computes the vapour density on the saturation curve from a correlation polynomial
'as a first guess for liq_vap_iteration_at_t (Tt < T < 550 K with rms = 0.01 in ln(d/dt))

' ln(d/dt) = a1 * (Tt/T-1) + a2 * (Tt/T-1)^2 + a3 * (Tt/T-1)^3

Const Tt = TP_temperature_si
Const dt = TP_density_vap_IAPWS95_si

Const a1 = -19.2235086866063
Const a2 = -6.15770193302955
Const a3 = -4.965736126494

Dim tr As Double
Dim d As Double

tr = Tt / t_si - 1#
d = dt * Exp(a1 * tr + a2 * tr ^ 2 + a3 * tr ^ 3)

aux_vap_density_triplepoint = d

End Function

'==========================================================================
Private Function aux_boil_temperature(ByVal p_si As Double) As Double

'computes the boiling temperature on the saturation curve from a correlation polynomial
'as a first guess for liq_vap_iteration_at_p (Tt < T < 640 K with rms = 0.01 in ln(p/pt))

' Clausius-Clapeyron type equation:
' ln(p/pt) = a1 * (Tt/T-1) + a2 * (Tt/T-1)^2

Const Tt = TP_temperature_si
Const Pt = TP_pressure_IAPWS95_si

Const a1 = -19.8731005709116
Const a2 = -3.08975437352998

Dim p As Double, q As Double
Dim tr As Double

p = a1 / a2
q = -Log(p_si / Pt) / a2

tr = -0.5 * p + Sqr(0.25 * p ^ 2 - q)

aux_boil_temperature = Tt / (tr + 1)

End Function

'==========================================================================
Public Function chk_IAPWS95_Table8() As String

'Function values as given in Table 8 of IAPWS-95, revision of Doorwerth 2009:
'Thermodynamic property values in the two-phase region for selected values of temperature

'                     T = 275 K         T = 450 K         T = 625 K
'P/MPa                0.698 451 167E3  0.932 203 564     0.169 082 693E+2
'rho_liq/(kg m3)     0.999 887 406E+3  0.890 341 250E+3  0.567 090 385E+3
'rho_vap/(kg m3)     0.550 664 919E2  0.481 200 360E+1  0.118 290 280E+3
'h_liq/(kJ kg1)      0.775 972 200E+1  0.749 161 585E+3  0.168 626 976E+4
'h_vap/(kJ kg 1)     0.250 428 995E+4  0.277 441 078E+4  0.255 071 625E+4
's_liq/(kJ kg1 K1)  0.283 094 669E1  0.210 865 845E+1  0.380 194 683E+1
's_vap/(kJ kg1 K1)  0.910 660 120E+1  0.660 921 221E+1  0.518 506 121E+1

Dim CRLF As String, TB As String
Dim txt As String
Dim d As Double, t As Double

Dim i As Integer, j As Integer, row As String
Dim p As Double

CRLF = Chr(13) + Chr(10)
TB = Chr(9)

txt = "Implementation of IAPWS-95 in Visual Basic" + CRLF
txt = txt + "for Publication in Ocean Science, 2009" + CRLF
txt = txt + "R. Feistel, IOW, Version " + Version + CRLF
txt = txt + "Compiled on " + CStr(Now) + CRLF + CRLF

txt = txt + "Function values as given in Table 8 of IAPWS-95:" + CRLF
txt = txt + "Thermodynamic property values in the single-phase region for selected values of T and rho" + CRLF + CRLF

txt = txt + "                     T = 275 K       T = 450 K       T = 625 K" + CRLF

For i = 1 To 7

  Select Case i
    Case 1:  row = "P/MPa                0.698451167E-3  0.932203564     0.169082693E+2"
    Case 2:  row = "rho_liq/(kg m-3)     0.999887406E+3  0.890341250E+3  0.567090385E+3"
    Case 3:  row = "rho_vap/(kg m-3)     0.550664919E-2  0.481200360E+1  0.118290280E+3"
    Case 4:  row = "h_liq/(kJ kg-1)      0.775972200E+1  0.749161585E+3  0.168626976E+4"
    Case 5:  row = "h_vap/(kJ kg-1)      0.250428995E+4  0.277441078E+4  0.255071625E+4"
    Case 6:  row = "s_liq/(kJ kg-1 K-1)  0.283094669E-1  0.210865845E+1  0.380194683E+1"
    Case 7:  row = "s_vap/(kJ kg-1 K-1)  0.910660120E+1  0.660921221E+1  0.518506121E+1"
  End Select

  txt = txt + row + CRLF
  txt = txt + "this code:          "
  
  For j = 1 To 3
  
    t = Choose(j, 275, 450, 625)
    set_liq_vap_eq_at_t t
    Select Case i
      Case 1: 'compute pressure in MPa
               p = 0.000001 * liq_vap_pressure_vap_si
      Case 2: 'compute liquid density in kg m3
               p = liq_vap_density_liq_si
      Case 3: 'compute vapour density in kg m3
               p = liq_vap_density_vap_si
      Case 4: 'compute liquid enthalpy in kJ kg1
               p = 0.001 * liq_vap_enthalpy_liq_si
      Case 5: 'compute vapour enthalpy in kJ kg1
               p = 0.001 * liq_vap_enthalpy_vap_si
      Case 6: 'compute liquid entropy in kJ kg1 K1
               p = 0.001 * liq_vap_entropy_liq_si
      Case 7: 'compute vapour entropy in kJ kg1 K1
               p = 0.001 * liq_vap_entropy_vap_si
    End Select
    txt = txt + Left(EFormat(p, 9) + Space(16), 16)
    
  Next j
  txt = txt + CRLF + CRLF
  
Next i

txt = txt + "Note: This computation was done with default iteration settings" + CRLF

chk_IAPWS95_Table8 = txt

End Function

