Attribute VB_Name = "Liq_Air_4a_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules:
'     Constants_0_Mdl, file Constants_0.bas
'     Convert_0_Mdl,   file Convert_0.bas
'     Air_1_Mdl,       file Air_1.bas
'     Flu_1_Mdl,       file Flu_1.bas
'     Air_2_Mdl,       file Air_2.bas
'     Flu_2_Mdl,       file Flu_2.bas
'     Air_3a_Mdl,      file Air_3a.bas
'     Air_3b_Mdl,      file Air_3b.bas
'     Flu_3a_Mdl,      file Flu_3a.bas
'     Liq_Vap_4_Mdl,   file Liq_Vap_4.bas
'     Maths_0_Mdl,     file Maths_0.bas
'
'#########################################################################

'=========================================================================
'This module implements the phase equilibrium properties of liquid water
'with humid air, commonly regarded as "saturated air" or "humidity 100%".
'Therefore, the air properties computed here refer to saturated air.

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009

'Feistel, R., Kretzschmar, H.-J., Span, R., Hagen, E., Wright, D.G., Herrmann, S.:
'Thermodynamic Properties of Sea Air.
'Ocean Science Discussion 6(2009)21932325.
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98
'Private Const IsOK = -1

'Control parameters of the vapour pressure iteration
Private ctrl_initialized As Integer

'select initial guessing method
Private ctrl_mode_humidity As Integer
Private ctrl_mode_temperature As Integer
Private ctrl_mode_pressure As Integer

'store customized initial values here
Private ctrl_loop_maximum As Long
Private ctrl_init_t As Double
Private ctrl_init_a As Double
Private ctrl_init_p As Double

'set tolerance of the iteration loop for
Private ctrl_eps_exit_p_vap As Double   'vapour partial pressure
Private ctrl_eps_exit_temp As Double    'temperature
Private ctrl_eps_exit_press As Double    'pressure

'Properties of the current liquid-vapour equilibrium state
Private equi_liq_air_done As Integer

'storage of the iteration results = the equilibrium state (saturated air)
Private equi_liq_air_a As Double        'dry air mass fraction of humid air
Private equi_liq_air_t As Double        'temperature
Private equi_liq_air_p As Double        'pressure
Private equi_liq_air_eta As Double      'humid air entropy
Private equi_liq_air_d_liq As Double    'liquid water density
Private equi_liq_air_d_air As Double    'humid air density

Private Const Version = "28 May 2010"

'==========================================================================
Public Function liq_air_massfraction_air_si(ByVal t_si As Double, _
                                            ByVal p_si As Double) As Double

'returns the air mass fraction in kg/kg of saturated humid air
'input:  t_si  absolute temperature in K
'        p_si  absolute pressure of humid air in Pa
'output: a_si  mass fraction of dry air in saturated humid air, in kg/kg

'check value at default settings:
'liq_air_massfraction_air_si(300, 1e5) = 0.977600623581678  v. 1.0
'liq_air_massfraction_air_si(300, 1e5) = 0.977605797727189  v. 1.1

liq_air_massfraction_air_si = ErrorReturn

If set_liq_air_eq_at_t_p(t_si, p_si) = ErrorReturn Then Exit Function

liq_air_massfraction_air_si = equi_liq_air_a

End Function

'==========================================================================
Public Function liq_air_dewpoint_si(ByVal a_si As Double, _
                                    ByVal p_si As Double) As Double

'returns the dewpoint temperature of humid air
'input:  a_si  mass fraction of dry air in humid air, in kg/kg
'        p_si  absolute pressure of humid air in Pa
'output: t_si  absolute dewpoint temperature in K

'check values with default settings:
'liq_air_dewpoint_si(0.99, 1E5) = 287.074721926982  v. 1.0
'liq_air_dewpoint_si(0.99, 1E5) = 287.078299794842  v. 1.1

liq_air_dewpoint_si = ErrorReturn

If set_liq_air_eq_at_a_p(a_si, p_si) = ErrorReturn Then Exit Function

liq_air_dewpoint_si = equi_liq_air_t

End Function

'==========================================================================
Public Function liq_air_ict_si(ByVal a_si As Double, _
                               ByVal t_si As Double, _
                               ByVal p_si As Double) As Double

'returns the isentropic condensation temperature ICT of humid air
'input:  a_si  mass fraction of dry air in humid air, in kg/kg
'        t_si  absolute temperature in K
'        p_si  absolute pressure of humid air in Pa
'output: ict_si  absolute isentropic condensation temperature in K

'check value with default settings:
'liq_air_ict_si(0.99, 300, 1E5) = 284.195916824567  v. 1.0
'liq_air_ict_si(0.99, 300, 1E5) = 284.200207629199  v. 1.1

Dim eta As Double

liq_air_ict_si = ErrorReturn

eta = air_g_entropy_si(a_si, t_si, p_si)
If eta = ErrorReturn Then Exit Function

If set_liq_air_eq_at_a_eta(a_si, eta) = ErrorReturn Then Exit Function

liq_air_ict_si = equi_liq_air_t

End Function

'==========================================================================
Public Function liq_air_icl_si(ByVal a_si As Double, _
                               ByVal t_si As Double, _
                               ByVal p_si As Double) As Double

'returns the isentropic condensation level ICL of humid air
'input:  a_si  mass fraction of dry air in humid air, in kg/kg
'        t_si  absolute temperature in K
'        p_si  absolute pressure of humid air in Pa
'output: icl_si  absolute isentropic condensation pressure in Pa

'check value with default settings:
'liq_air_icl_si(0.99, 300, 1E5) = 82719.2377442223  v. 1.0
'liq_air_icl_si(0.99, 300, 1E5) = 82723.6047630537  v. 1.1

Dim eta As Double

liq_air_icl_si = ErrorReturn

eta = air_g_entropy_si(a_si, t_si, p_si)
If eta = ErrorReturn Then Exit Function

If set_liq_air_eq_at_a_eta(a_si, eta) = ErrorReturn Then Exit Function

liq_air_icl_si = equi_liq_air_p

End Function


'==========================================================================
Public Function liq_air_condensationpressure_si(ByVal a_si As Double, _
                                                ByVal t_si As Double) As Double

'returns the condensation pressure of humid air
'input:  a_si  mass fraction of dry air in humid air, in kg/kg
'        t_si  absolute dewpoint temperature in K
'output: p_si  absolute pressure of humid air in Pa

'check value at default settings:
'liq_air_condensationpressure_si(0.9, 300) = 23385.9346276043  v. 1.0
'liq_air_condensationpressure_si(0.9, 300) = 23381.2332934951  v. 1.1

liq_air_condensationpressure_si = ErrorReturn

If set_liq_air_eq_at_a_t(a_si, t_si) = ErrorReturn Then Exit Function

liq_air_condensationpressure_si = equi_liq_air_p

End Function

'==========================================================================
Public Function liq_air_enthalpy_evap_si()

'returns the latent heat in J/kg of saturated air
'after setting the equilibrium state by calling set_liq_air_eq_at_t_p etc

'check values with default settings:  v. 1.0
'set_liq_air_eq_at_a_eta 0.99, 100
'liq_air_enthalpy_evap_si = 2458132.60499966
'
'set_liq_air_eq_at_a_p 0.99, 1E5
'liq_air_enthalpy_evap_si = 2465665.32463613
'
'set_liq_air_eq_at_a_t 0.99, 300
'liq_air_enthalpy_evap_si = 2433111.45416375
'
'set_liq_air_eq_at_t_p 300, 1E5
'liq_air_enthalpy_evap_si = 2434585.86629587

'check values with default settings:  v. 1.1
'set_liq_air_eq_at_a_eta 0.99, 100
'liq_air_enthalpy_evap_si = 2458121.74960753
'
'set_liq_air_eq_at_a_p 0.99, 1E5
'liq_air_enthalpy_evap_si = 2465656.38629966
'
'set_liq_air_eq_at_a_t 0.99, 300
'liq_air_enthalpy_evap_si = 2433111.29695544
'
'set_liq_air_eq_at_t_p 300, 1E5
'liq_air_enthalpy_evap_si = 2434585.53919211

Dim a As Double, t As Double, dh As Double, dl As Double
Dim hh As Double, hh_a As Double, hl As Double
Dim fh_a As Double, fh_d As Double
Dim fh_at As Double, fh_ad As Double
Dim fh_td As Double, fh_dd As Double
Dim p_a As Double, p_d As Double
Dim h_a As Double, h_d As Double

liq_air_enthalpy_evap_si = ErrorReturn

If equi_liq_air_done <> IsOK Then Exit Function

'get the equilibrium properties
a = equi_liq_air_a
If a = ErrorReturn Then Exit Function
If a < 0 Or a > 1 Then Exit Function
t = equi_liq_air_t
If t = ErrorReturn Then Exit Function
If t <= 0 Then Exit Function
dh = equi_liq_air_d_air
If dh = ErrorReturn Then Exit Function
If dh <= 0 Then Exit Function
dl = equi_liq_air_d_liq
If dl = ErrorReturn Then Exit Function
If dl <= 0 Then Exit Function

'enthalpy of humid air
hh = air_f_enthalpy_si(a, t, dh)
If hh = ErrorReturn Then Exit Function

'derivative hh_a = (dh/da)_t_p of enthalpy of humid air
fh_a = air_f_si(1, 0, 0, a, t, dh)
If fh_a = ErrorReturn Then Exit Function
fh_d = air_f_si(0, 0, 1, a, t, dh)
If fh_d = ErrorReturn Then Exit Function
fh_at = air_f_si(1, 1, 0, a, t, dh)
If fh_at = ErrorReturn Then Exit Function
fh_ad = air_f_si(1, 0, 1, a, t, dh)
If fh_ad = ErrorReturn Then Exit Function
fh_td = air_f_si(0, 1, 1, a, t, dh)
If fh_td = ErrorReturn Then Exit Function
fh_dd = air_f_si(0, 0, 2, a, t, dh)
If fh_dd = ErrorReturn Then Exit Function

p_a = dh * fh_ad                            'dp/da at const t,d
p_d = dh * (2 * fh_d + dh * fh_dd)          'dp/dd at const a,t
If p_d = 0 Then Exit Function
h_a = fh_a + dh * fh_ad - t * fh_at         'dh/da at const t,d
h_d = 2 * fh_d + dh * fh_dd - t * fh_td     'dh/dd at const a,t

hh_a = h_a - h_d * p_a / p_d                'dh/da at const t,p

'enthalpy of liquid water
hl = flu_enthalpy_si(t, dl)
If hl = ErrorReturn Then Exit Function

'compute the latent heat
liq_air_enthalpy_evap_si = hh - a * hh_a - hl

End Function

'==========================================================================
Public Function liq_air_entropy_air_si()

'returns the saturated humid-air entropy in J/kg
'after setting the equilibrium state by calling set_liq_air_eq_at_t_p etc

'check values with default settings:  v. 1.0
'set_liq_air_eq_at_a_eta 0.99, 100
'liq_air_entropy_air_si = 100
'
'set_liq_air_eq_at_a_p 0.99, 1E5
'liq_air_entropy_air_si = 145.864613600418
'
'set_liq_air_eq_at_a_t 0.99, 300
'liq_air_entropy_air_si = -42.0969943912465
'
'set_liq_air_eq_at_t_p 300, 1E5
'liq_air_entropy_air_si = 296.780112467825

'check values with default settings:  v. 1.1
'set_liq_air_eq_at_a_eta 0.99, 100
'liq_air_entropy_air_si = 100
'
'set_liq_air_eq_at_a_p 0.99, 1E5
'liq_air_entropy_air_si = 145.863545193528
'
'set_liq_air_eq_at_a_t 0.99, 300
'liq_air_entropy_air_si = -41.9991507402073
'
'set_liq_air_eq_at_t_p 300, 1E5
'liq_air_entropy_air_si = 296.711483506936

liq_air_entropy_air_si = ErrorReturn

If equi_liq_air_done <> IsOK Then Exit Function

liq_air_entropy_air_si = equi_liq_air_eta

End Function


'==========================================================================
Public Function liq_air_temperature_si()

'returns the temperature in K of saturated air
'after setting the equilibrium state by calling set_liq_air_eq_at_t_p etc

'check values with default settings:  v. 1.0
'set_liq_air_eq_at_a_eta 0.99, 100
'liq_air_temperature_si = 290.10306942911
'
'set_liq_air_eq_at_a_p 0.99, 1E5
'liq_air_temperature_si = 287.074721926982
'
'set_liq_air_eq_at_a_t 0.99, 300
'liq_air_temperature_si = 300
'
'set_liq_air_eq_at_t_p 300, 1E5
'liq_air_temperature_si = 300

'check values with default settings:  v. 1.1
'set_liq_air_eq_at_a_eta 0.99, 100
'liq_air_temperature_si = 290.107386673443
'
'set_liq_air_eq_at_a_p 0.99, 1E5
'liq_air_temperature_si = 287.078299794842
'
'set_liq_air_eq_at_a_t 0.99, 300
'liq_air_temperature_si = 300
'
'set_liq_air_eq_at_t_p 300, 1E5
'liq_air_temperature_si = 300

liq_air_temperature_si = ErrorReturn

If equi_liq_air_done <> IsOK Then Exit Function

liq_air_temperature_si = equi_liq_air_t

End Function

'==========================================================================
Public Function liq_air_pressure_si()

'returns the pressure in Pa of saturated air
'after setting the equilibrium state by calling set_liq_air_eq_at_a_t etc

'check values with default settings:  v. 1.0
'set_liq_air_eq_at_a_eta 0.99, 100
'liq_air_pressure_si = 121541.303926767
'
'set_liq_air_eq_at_a_p 0.99, 1E5
'liq_air_pressure_si = 100000
'
'set_liq_air_eq_at_a_t 0.99, 300
'liq_air_pressure_si = 223109.793431746
'
'set_liq_air_eq_at_t_p 300, 1E5
'liq_air_pressure_si = 100000

'check values with default settings:  v. 1.1
'set_liq_air_eq_at_a_eta 0.99, 100
'liq_air_pressure_si = 121546.373651585
'
'set_liq_air_eq_at_a_p 0.99, 1E5
'liq_air_pressure_si = 100000
'
'set_liq_air_eq_at_a_t 0.99, 300
'liq_air_pressure_si = 223057.741749908
'
'set_liq_air_eq_at_t_p 300, 1E5
'liq_air_pressure_si = 100000

liq_air_pressure_si = ErrorReturn

If equi_liq_air_done <> IsOK Then Exit Function

liq_air_pressure_si = equi_liq_air_p

End Function
'==========================================================================
Public Function liq_air_density_air_si()

'returns the saturated humid-air density in kg/m3
'after setting the equilibrium state by calling set_liq_air_eq_at_t_p etc

'check values with default settings:  v. 1.0
'set_liq_air_eq_at_a_eta 0.99, 100
'liq_air_density_air_si = 1.45116943913412
'
'set_liq_air_eq_at_a_p 0.99, 1E5
'liq_air_density_air_si = 1.20649187940702
'
'set_liq_air_eq_at_a_t 0.99, 300
'liq_air_density_air_si = 2.57657751338532
'
'set_liq_air_eq_at_t_p 300, 1E5
'liq_air_density_air_si = 1.14587677779855
 
'check values with default settings:  v. 1.1
'set_liq_air_eq_at_a_eta 0.99, 100
'liq_air_density_air_si = 1.45154665083435
'
'set_liq_air_eq_at_a_p 0.99, 1E5
'liq_air_density_air_si = 1.206758060224
'
'set_liq_air_eq_at_a_t 0.99, 300
'liq_air_density_air_si = 2.57657653270459
'
'set_liq_air_eq_at_t_p 300, 1E5
'liq_air_density_air_si = 1.14614215827163

liq_air_density_air_si = ErrorReturn

If equi_liq_air_done <> IsOK Then Exit Function

liq_air_density_air_si = equi_liq_air_d_air

End Function

'==========================================================================
Public Function liq_air_density_vap_si()

'returns the vapour density in kg/m3 of saturated air
'after setting the equilibrium state by calling set_liq_air_eq_at_t_p etc

'check values with default settings:  v. 1.0
'set_liq_air_eq_at_a_eta 0.99, 100
'liq_air_density_vap_si = 1.45116943913412E-02
'
'set_liq_air_eq_at_a_p 0.99, 1E5
'liq_air_density_vap_si = 1.20649187940703E-02
'
'set_liq_air_eq_at_a_t 0.99, 300
'liq_air_density_vap_si = 2.57657751338532E-02
'
'set_liq_air_eq_at_t_p 300, 1E5
'liq_air_density_vap_si = 2.56669252749238E-02

'check values with default settings:  v. 1.1
'set_liq_air_eq_at_a_eta 0.99, 100
'liq_air_density_vap_si = 1.45154665083435E-02
'
'set_liq_air_eq_at_a_p 0.99, 1E5
'liq_air_density_vap_si = 0.01206758060224
'
'set_liq_air_eq_at_a_t 0.99, 300
'liq_air_density_vap_si = 2.57657653270459E-02
'
'set_liq_air_eq_at_t_p 300, 1E5
'liq_air_density_vap_si = 2.56669393257306E-02

liq_air_density_vap_si = ErrorReturn

If equi_liq_air_done <> IsOK Then Exit Function

liq_air_density_vap_si = equi_liq_air_d_air * (1 - equi_liq_air_a)

End Function

'==========================================================================
Public Function liq_air_density_liq_si()

'returns the liquid-water density in kg/m3 of wet air
'after setting the equilibrium state by calling set_liq_air_eq_at_t_p etc

'check values with default settings:  v. 1.0
'set_liq_air_eq_at_a_eta 0.99, 100
'liq_air_density_liq_si = 998.795484566385
'
'set_liq_air_eq_at_a_p 0.99, 1E5
'liq_air_density_liq_si = 999.257178498591
'
'set_liq_air_eq_at_a_t 0.99, 300
'liq_air_density_liq_si = 996.611605024888
'
'set_liq_air_eq_at_t_p 300, 1E5
'liq_air_density_liq_si = 996.556340388894

'check values with default settings:  v. 1.1
'set_liq_air_eq_at_a_eta 0.99, 100
'liq_air_density_liq_si = 998.794738783831
'
'set_liq_air_eq_at_a_p 0.99, 1E5
'liq_air_density_liq_si = 999.256685197423
'
'set_liq_air_eq_at_a_t 0.99, 300
'liq_air_density_liq_si = 996.611581661662
'
'set_liq_air_eq_at_t_p 300, 1E5
'liq_air_density_liq_si = 996.556340388894

liq_air_density_liq_si = ErrorReturn

If equi_liq_air_done <> IsOK Then Exit Function

liq_air_density_liq_si = equi_liq_air_d_liq

End Function

'==========================================================================
Public Function liq_air_rh_cct_from_a_si(ByVal a_si As Double, _
                                         ByVal t_si As Double, _
                                         ByVal p_si As Double) As Double

'returns the relative humidity from air mass fraction, temperature and pressure

'input:  a_si  mass fraction of dry air in humid air, in kg/kg
'        t_si  absolute temperature in K
'        p_si  absolute pressure of humid air in Pa
'output: relative humidity, CCT/IUPAC definition

'check value with default settings:
'liq_air_rh_cct_from_a_si(0.99, 300, 1E5) = 0.449783278144408  v. 1.0
'liq_air_rh_cct_from_a_si(0.99, 300, 1E5) = 0.449887886958951  v. 1.1

Dim a_sat As Double

liq_air_rh_cct_from_a_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function
If a_si <= 0 Or a_si > 1 Then Exit Function

If a_si = 1 Then 'dry air
  liq_air_rh_cct_from_a_si = 0
  Exit Function
End If

a_sat = liq_air_massfraction_air_si(t_si, p_si) 'saturated air fraction
If a_sat = ErrorReturn Then Exit Function
If a_sat <= 0 Or a_sat > a_si Then Exit Function

If a_sat = 1 Then 'dry air
  liq_air_rh_cct_from_a_si = 0
  Exit Function
End If

liq_air_rh_cct_from_a_si = air_molfraction_vap_si(a_si) / air_molfraction_vap_si(a_sat)

End Function

'==========================================================================
Public Function liq_air_rh_wmo_from_a_si(ByVal a_si As Double, _
                                         ByVal t_si As Double, _
                                         ByVal p_si As Double) As Double

'returns the relative humidity from air mass fraction, temperature and pressure

'input:  a_si  mass fraction of dry air in humid air, in kg/kg
'        t_si  absolute temperature in K
'        p_si  absolute pressure of humid air in Pa
'output: relative humidity, WMO definition

'check value with default settings:
'liq_air_rh_wmo_from_a_si(0.99, 300, 1E5) = 0.440849494608027  v. 1.0
'liq_air_rh_wmo_from_a_si(0.99, 300, 1E5) = 0.440953686018889  v. 1.1

Dim a_sat As Double

liq_air_rh_wmo_from_a_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function
If a_si <= 0 Or a_si > 1 Then Exit Function

If a_si = 1 Then 'dry air
  liq_air_rh_wmo_from_a_si = 0
  Exit Function
End If

a_sat = liq_air_massfraction_air_si(t_si, p_si) 'saturated air fraction
If a_sat = ErrorReturn Then Exit Function
If a_sat <= 0 Or a_sat > a_si Then Exit Function

If a_sat = 1 Then 'dry air
  liq_air_rh_wmo_from_a_si = 0
  Exit Function
End If

liq_air_rh_wmo_from_a_si = (1 / a_si - 1) / (1 / a_sat - 1)

End Function

'==========================================================================
Public Function liq_air_a_from_rh_wmo_si(ByVal rh_si As Double, _
                                         ByVal t_si As Double, _
                                         ByVal p_si As Double) As Double

'returns the dry_air mass fraction of humid air from relative humidity,
'temperature and pressure

'input:  rh_si relative humidity, WMO definition, 0 < rh < 1
'        t_si  absolute dewpoint temperature in K
'        p_si  absolute pressure of humid air in Pa
'output: a_si  mass fraction of dry air in humid air, in kg/kg

'check value at default settings:
'liq_air_a_from_rh_wmo_si(0.8, 300, 1e5) = 0.981999860485227  v. 1.0
'liq_air_a_from_rh_wmo_si(0.8, 300, 1e5) = 0.982004037135254  v. 1.1

Dim a_sat As Double

liq_air_a_from_rh_wmo_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function
If rh_si < 0 Or rh_si > 1 Then Exit Function

If rh_si = 0 Then 'dry air
  liq_air_a_from_rh_wmo_si = 1
  Exit Function
End If

a_sat = liq_air_massfraction_air_si(t_si, p_si) 'saturated air fraction
If a_sat = ErrorReturn Then Exit Function
If a_sat <= 0 Or a_sat > 1 Then Exit Function

liq_air_a_from_rh_wmo_si = 1 / (1 + rh_si * (1 / a_sat - 1))

End Function

'==========================================================================
Public Function liq_air_a_from_rh_cct_si(ByVal rh_si As Double, _
                                         ByVal t_si As Double, _
                                         ByVal p_si As Double) As Double

'returns the dry-air mass fraction of humid air from relative humidity,
'temperature and pressure

'input:  rh_si relative humidity, CCT/IUPAC definition, 0 < rh < 1
'        t_si  absolute dewpoint temperature in K
'        p_si  absolute pressure of humid air in Pa
'output: a_si  mass fraction of dry air in humid air, in kg/kg

'check value at default settings:
'liq_air_a_from_rh_cct_si(0.8, 300, 1e5) = 0.982129130677262  v. 1.0
'liq_air_a_from_rh_cct_si(0.8, 300, 1e5) = 0.982133277948147  v. 1.1

Dim a_sat As Double

liq_air_a_from_rh_cct_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function
If rh_si < 0 Or rh_si > 1 Then Exit Function

If rh_si = 0 Then 'dry air
  liq_air_a_from_rh_cct_si = 1
  Exit Function
End If

a_sat = liq_air_massfraction_air_si(t_si, p_si) 'saturated air mass fraction
If a_sat = ErrorReturn Then Exit Function
If a_sat <= 0 Or a_sat > 1 Then Exit Function

a_sat = air_molfraction_vap_si(a_sat)          'saturated vapour mole fraction
If a_sat = ErrorReturn Then Exit Function
If a_sat <= 0 Or a_sat > 1 Then Exit Function

liq_air_a_from_rh_cct_si = air_massfraction_air_si(1 - rh_si * a_sat)

End Function

'==========================================================================
Public Function set_liq_air_eq_at_t_p(ByVal t_si As Double, _
                                               ByVal p_si As Double) As Double
                                             
'this function computes the saturation equilibrium state between humid air and
'liquid water, in particular, the saturated specific humidity of air.

'note: there is no check made whether the liquid water phase is undercooled, i.e. the
'result may apply to metastable water if the temperature is below the freezing point.

'input:  t_si  absolute temperature in K
'        p_si  absolute pressure of humid air in Pa

Dim a_si As Double, dl As Double, dh As Double, eta As Double

Dim maxit As Long, eps As Double

If equi_liq_air_done = IsOK And _
   t_si = equi_liq_air_t And _
   p_si = equi_liq_air_p Then
  'the requested state has already been computed earlier
  set_liq_air_eq_at_t_p = IsOK
  Exit Function
End If

clear_liq_air_state 'waste any previous state

set_liq_air_eq_at_t_p = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

init_it_ctrl_liq_air

'set initial air fraction guess
Select Case ctrl_mode_humidity
  Case 0:  a_si = aux_airfraction_si(t_si, p_si)
           If a_si = ErrorReturn Then Exit Function
           If a_si < 0 Or a_si > 1 Then Exit Function
  Case 1:  a_si = ctrl_init_a
  Case Else: Exit Function
End Select
If a_si = ErrorReturn Then Exit Function
If a_si < 0 Or a_si > 1 Then Exit Function

'liquid density can be computed without knowing a_si
dl = liq_density_si(t_si, p_si)
If dl = ErrorReturn Then Exit Function
If dl <= 0 Then Exit Function

dh = air_density_si(a_si, t_si, p_si)
If dh = ErrorReturn Then Exit Function
If dh <= 0 Then Exit Function

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: set_liq_air_eq_at_t_p = IsOK
           eta = air_f_entropy_si(a_si, t_si, dh)
           set_liq_air_state a_si, t_si, p_si, eta, dl, dh
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit for the vapour partial pressure
eps = ctrl_eps_exit_p_vap
If eps = 0 Then Exit Function

'run iteration loop
If liq_air_iteration_at_t_p(t_si, p_si, dl, maxit, eps, a_si, dh) = ErrorReturn Then
  Exit Function
End If

eta = air_f_entropy_si(a_si, t_si, dh)
set_liq_air_state a_si, t_si, p_si, eta, dl, dh

set_liq_air_eq_at_t_p = IsOK

End Function

'==========================================================================
Public Function set_liq_air_eq_at_a_p(ByVal a_si As Double, _
                                               ByVal p_si As Double) As Double
                                             
'this function computes the saturation equilibrium state between humid air and
'liquid water, in particular, the dewpoint temperature.

'note: there is no check made whether the liquid water phase is undercooled, i.e. the
'result may apply to metastable water if the temperature is below the freezing point.

'input:  a_si  mass fraction of dry air in humid air in kg/kg
'        p_si  absolute pressure of humid air in Pa

Dim t_si As Double, dl As Double, dh As Double, eta As Double

Dim maxit As Long, eps As Double

If equi_liq_air_done = IsOK And _
   a_si = equi_liq_air_a And _
   p_si = equi_liq_air_p Then
  'the requested state has already been computed earlier
  set_liq_air_eq_at_a_p = IsOK
  Exit Function
End If

clear_liq_air_state 'waste any previous state

set_liq_air_eq_at_a_p = ErrorReturn

If a_si < 0 Or a_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function

init_it_ctrl_liq_air

'set initial temperature guess
Select Case ctrl_mode_temperature
  Case 0:  t_si = aux_temperature_si(a_si, p_si)
  Case 1:  t_si = ctrl_init_t
  Case Else: Exit Function
End Select
If t_si = ErrorReturn Then Exit Function
If t_si <= 0 Then Exit Function

'set initial liquid density
dl = liq_density_si(t_si, p_si)
If dl = ErrorReturn Then Exit Function
If dl <= 0 Then Exit Function

'set initial humid air density
dh = air_density_si(a_si, t_si, p_si)
If dh = ErrorReturn Then Exit Function
If dh <= 0 Then Exit Function

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: set_liq_air_eq_at_a_p = IsOK
           eta = air_f_entropy_si(a_si, t_si, dh)
           set_liq_air_state a_si, t_si, p_si, eta, dl, dh
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit for the vapour partial pressure
eps = ctrl_eps_exit_p_vap
If eps = 0 Then Exit Function

'run iteration loop
If liq_air_iteration_at_a_p(a_si, p_si, maxit, eps, t_si, dl, dh) = ErrorReturn Then
  Exit Function
End If

eta = air_f_entropy_si(a_si, t_si, dh)
set_liq_air_state a_si, t_si, p_si, eta, dl, dh

set_liq_air_eq_at_a_p = IsOK

End Function

'==========================================================================
Public Function set_liq_air_eq_at_a_t(ByVal a_si As Double, _
                                               ByVal t_si As Double) As Double
                                             
'this function computes the saturation equilibrium state between humid air and
'liquid water, in particular, the condensation pressure.

'note: there is no check made whether the liquid water phase is undercooled, i.e. the
'result may apply to metastable water if the temperature is below the freezing point.

'input:  a_si  mass fraction of dry air in humid air in kg/kg
'        t_si  absolute temperature of humid air in K

Dim p_si As Double, dl As Double, dh As Double, eta As Double

Dim maxit As Long, eps As Double

If equi_liq_air_done = IsOK And _
   a_si = equi_liq_air_a And _
   t_si = equi_liq_air_t Then
  'the requested state has already been computed earlier
  set_liq_air_eq_at_a_t = IsOK
  Exit Function
End If

clear_liq_air_state 'waste any previous state

set_liq_air_eq_at_a_t = ErrorReturn

If a_si < 0 Or a_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function

init_it_ctrl_liq_air

'set initial pressure guess
Select Case ctrl_mode_pressure
  Case 0:  p_si = aux_pressure_si(a_si, t_si)
  Case 1:  p_si = ctrl_init_p
  Case Else: Exit Function
End Select
If p_si = ErrorReturn Then Exit Function
If p_si <= 0 Then Exit Function

'set initial liquid density
dl = liq_density_si(t_si, p_si)
If dl = ErrorReturn Then Exit Function
If dl <= 0 Then Exit Function

'set initial humid air density
dh = air_density_si(a_si, t_si, p_si)
If dh = ErrorReturn Then Exit Function
If dh <= 0 Then Exit Function

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: set_liq_air_eq_at_a_t = IsOK
           eta = air_f_entropy_si(a_si, t_si, dh)
           set_liq_air_state a_si, t_si, p_si, eta, dl, dh
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit for the vapour partial pressure
eps = ctrl_eps_exit_p_vap
If eps = 0 Then Exit Function

'run iteration loop
If liq_air_iteration_at_a_t(a_si, t_si, maxit, eps, p_si, dl, dh) = ErrorReturn Then
  Exit Function
End If

eta = air_f_entropy_si(a_si, t_si, dh)
set_liq_air_state a_si, t_si, p_si, eta, dl, dh

set_liq_air_eq_at_a_t = IsOK

End Function

'==========================================================================
Public Function set_liq_air_eq_at_a_eta(ByVal a_si As Double, _
                                                 ByVal eta_si As Double) As Double
                                             
'this function computes the saturation equilibrium state between humid air and
'liquid water, in particular, the isentropic condensation temperature and pressure.

'note: there is no check made whether the liquid water phase is undercooled, i.e. the
'result may apply to metastable water if the temperature is below the freezing point.

'input:  a_si  mass fraction of dry air in humid air in kg/kg
'        eta_si  specific entropy of humid air in J/(kg K)

Dim p_si As Double, t_si As Double, dl As Double, dh As Double

Dim maxit As Long, eps As Double

If equi_liq_air_done = IsOK And _
   a_si = equi_liq_air_a And _
   eta_si = equi_liq_air_eta Then
  'the requested state has already been computed earlier
  set_liq_air_eq_at_a_eta = IsOK
  Exit Function
End If

clear_liq_air_state 'waste any previous state

set_liq_air_eq_at_a_eta = ErrorReturn

If a_si < 0 Or a_si >= 1 Then Exit Function

init_it_ctrl_liq_air

'set initial temperature guess
Select Case ctrl_mode_temperature
  Case 0:  t_si = aux_ict_si(a_si, eta_si)
  Case 1:  t_si = ctrl_init_t
  Case Else: Exit Function
End Select
If t_si = ErrorReturn Then Exit Function
If t_si <= 0 Then Exit Function

'set initial pressure guess
Select Case ctrl_mode_pressure
  Case 0:  p_si = aux_pressure_si(a_si, t_si)
  Case 1:  p_si = ctrl_init_p
  Case Else: Exit Function
End Select
If p_si = ErrorReturn Then Exit Function
If p_si <= 0 Then Exit Function

'set initial liquid density
dl = liq_density_si(t_si, p_si)
If dl = ErrorReturn Then Exit Function
If dl <= 0 Then Exit Function

'set initial humid air density
dh = air_density_si(a_si, t_si, p_si)
If dh = ErrorReturn Then Exit Function
If dh <= 0 Then Exit Function

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: set_liq_air_eq_at_a_eta = IsOK
           set_liq_air_state a_si, t_si, p_si, eta_si, dl, dh
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit for the vapour partial pressure
eps = ctrl_eps_exit_p_vap
If eps = 0 Then Exit Function

'run iteration loop
If liq_air_iteration_at_a_eta(a_si, eta_si, maxit, eps, t_si, p_si, dl, dh) = ErrorReturn Then
  Exit Function
End If

set_liq_air_state a_si, t_si, p_si, eta_si, dl, dh

set_liq_air_eq_at_a_eta = IsOK

End Function

'==========================================================================
Private Sub set_liq_air_state(ByVal a As Double, _
                              ByVal t As Double, _
                              ByVal p As Double, _
                              ByVal eta As Double, _
                              ByVal dl As Double, _
                              ByVal dh As Double)

'stores the actual properties as the current equilibrium state descriptor
equi_liq_air_done = IsOK

equi_liq_air_a = a
equi_liq_air_t = t
equi_liq_air_p = p
equi_liq_air_eta = eta
equi_liq_air_d_liq = dl
equi_liq_air_d_air = dh

End Sub

'==========================================================================
Private Sub clear_liq_air_state()

'clears the current equilibrium state descriptor

equi_liq_air_done = 0

End Sub

'==========================================================================
Private Sub init_it_ctrl_liq_air()

'triple point values
Const t_triple = TP_temperature_si
Const p_triple = TP_pressure_IAPWS95_si

If ctrl_initialized = IsOK Then Exit Sub

ctrl_initialized = IsOK

'Set default values and modes for density iteration
ctrl_mode_pressure = 0
ctrl_mode_temperature = 0
ctrl_mode_humidity = 0

ctrl_loop_maximum = 100
ctrl_init_t = t_triple
ctrl_init_p = p_triple
ctrl_init_a = 1 - 0.001

ctrl_eps_exit_p_vap = -0.0000001 'relative, 0.1 ppm
ctrl_eps_exit_temp = -0.0000001 'relative, 0.1 ppm
ctrl_eps_exit_press = -0.0000001 'relative, 0.1 ppm

End Sub

'==========================================================================
Public Sub set_it_ctrl_liq_air(ByVal key As String, ByVal value As Double)

'this sub sets control parameters for the iteration used to compute
'saturated humid air properties

'key              value
'it_steps         0           set iteration number to default (100)
'it_steps         n > 0       set iteration number to n
'it_steps        -1           do not iterate, use initial values as the saturation state

'init_hum         0           use default air fraction to start ( = aux_airfraction_si(t, p))
'init_hum         a > 0       use value a as air fraction to start

'init_temp        0           use default temperature to start ( = aux_temperature_si(a, p))
'init_temp        t > 0       use value t as temperature to start

'init_press       0           use default pressure to start ( = aux_pressure_si(a, t))
'init_press       p > 0       use value p as pressure to start

'tol_vap_press    0           use default exit accuracy for vapour pressure (0.1 ppm)
'tol_vap_press    eps         use eps as exit accuracy for vapour pressure (eps < 0 means relative error)

'tol_temp         0           use default exit accuracy for temperature (0.1 ppm)
'tol_temp         eps         use eps as exit accuracy for temperature (eps < 0 means relative error)

'tol_press        0           use default exit accuracy for pressure (0.1 ppm)
'tol_press        eps         use eps as exit accuracy for pressure (eps < 0 means relative error)


init_it_ctrl_liq_air

clear_liq_air_state

Select Case LCase(Trim(key))

  Case "it_steps":   'iteration steps
    Select Case value
      Case 0:      ctrl_loop_maximum = 100  'default = 100
      Case Is < 0: ctrl_loop_maximum = -1
      Case Else:   ctrl_loop_maximum = value
    End Select

  Case "init_hum":   'start air fraction
    Select Case CLng(value)
      Case 0:       ctrl_mode_humidity = 0    'default = aux polynomial
      Case Else:    ctrl_mode_humidity = 1
                    ctrl_init_a = value
    End Select
    
  Case "init_temp":   'start temperature
    Select Case CLng(value)
      Case 0:       ctrl_mode_temperature = 0    'default = aux polynomial
      Case Else:    ctrl_mode_temperature = 1
                    ctrl_init_t = value
    End Select

  Case "init_press":   'start pressure
    Select Case CLng(value)
      Case 0:       ctrl_mode_pressure = 0    'default = aux polynomial
      Case Else:    ctrl_mode_pressure = 1
                    ctrl_init_p = value
    End Select

  Case "tol_vap_press":      'required vapour pressure tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_p_vap = -0.0000001   'default = 0.1 ppm relative
      Case Else:   ctrl_eps_exit_p_vap = value
    End Select

  Case "tol_temp":      'required temperature tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_temp = -0.0000001   'default = 0.1 ppm relative
      Case Else:   ctrl_eps_exit_temp = value
    End Select

  Case "tol_press":      'required total pressure tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_press = -0.0000001   'default = 0.1 ppm relative
      Case Else:   ctrl_eps_exit_press = value
    End Select

End Select

End Sub

'==========================================================================
Private Function liq_air_iteration_at_t_p(ByVal t_si, _
                                          ByVal p_si, _
                                          ByVal d_liq_si, _
                                          ByVal maxit, _
                                          ByVal eps, _
                                          ByRef a_si, _
                                          ByRef d_air_si) As Double

'this function returns the liquid-humid-air phase equilibrium from equal pressure,
'temperature and chemical potential of water of both phases at given pressure, p_si,
'and temperature, t_si, from initial guesses for the humid-air density, d_air_si,
'and the air fraction, a_si

'The iteration limit eps refers to the error in vapour partial pressure

'output:    liq_air_iteration_at_t_p = IsOK if successful
'           liq_air_iteration_at_t_p = ErrorReturn is returned if
'           - the maximum number of iterations is exceeded without meeting the exit criterion
'           - function calls to flu_f_si or air_f_si have returned an error
'           - density or air fraction have taken an invalid value during the iteration
'     a_si: mass fraction of dry air in humid air, kg/kg
' d_air_si: humid-air density in the saturation equilibrium

'input: p_si: absolute pressure in Pa
'       t_si: absolute temperature in K
'   d_liq_si: liquid density at t_si, p_si
'      maxit: maximum number of iteration steps to be done
'        eps: required accuracy of vapour pressure
'             eps > 0: absolute vapour pressure uncertainty in Pa
'             eps < 0: relative vapour pressure uncertainty
'       a_si: initial guess of the mass fraction of dry air in humid air, kg/kg
'   d_air_si: initial guess of humid-air density in kg/m3

Dim ah As Double, fh_a As Double, fh_aa As Double, fh_ad As Double
Dim gl As Double
Dim fh As Double, fh_d As Double, fh_dd As Double
Dim dl As Double, dh As Double
Dim pv As Double, pv_old As Double

Dim it As Long

Dim a(2, 2) As Double, b(2) As Double, x(2) As Double

liq_air_iteration_at_t_p = ErrorReturn

'check for invalid input values
If a_si < 0 Or a_si > 1 Or _
   d_air_si <= 0 Or _
   p_si <= 0 Then
  a_si = ErrorReturn
  d_air_si = ErrorReturn
  Exit Function
End If

If check_limits = 1 Then
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Or _
     d_liq_si <= flu_dmin Or d_liq_si > flu_dmax Then
    a_si = ErrorReturn
    d_air_si = ErrorReturn
    Exit Function
  End If
  'AIR_LIMITS
  If t_si < dry_air_tmin Or t_si > dry_air_tmax Then
    a_si = ErrorReturn
    d_air_si = ErrorReturn
    Exit Function
  End If
Else
  If t_si <= 0 Or _
     d_liq_si <= 0 Then
    a_si = ErrorReturn
    d_air_si = ErrorReturn
    Exit Function
  End If
End If

check_limits = check_limits - 1

ah = a_si
dh = d_air_si

'chemical potential of water
gl = flu_gibbs_energy_si(t_si, d_liq_si)
If gl = ErrorReturn Then
  check_limits = check_limits + 1
  Exit Function
End If

For it = 0 To maxit

  'Derivatives of the Helmholtz function for 2D-Newton iteration
  'humid air:
  fh = air_f_si(0, 0, 0, ah, t_si, dh)
  If fh = ErrorReturn Then Exit For
  fh_a = air_f_si(1, 0, 0, ah, t_si, dh)
  If fh_a = ErrorReturn Then Exit For
  fh_d = air_f_si(0, 0, 1, ah, t_si, dh)
  If fh_d = ErrorReturn Then Exit For
  fh_aa = air_f_si(2, 0, 0, ah, t_si, dh)
  If fh_aa = ErrorReturn Then Exit For
  fh_ad = air_f_si(1, 0, 1, ah, t_si, dh)
  If fh_ad = ErrorReturn Then Exit For
  fh_dd = air_f_si(0, 0, 2, ah, t_si, dh)
  If fh_dd = ErrorReturn Then Exit For
  
  'vapour partial pressure for error checking
  pv_old = pv
  pv = air_molfraction_vap_si(ah)
  If pv = ErrorReturn Then Exit For
  pv = p_si * pv
  
  If it > 0 Then
    'check absolute or relative error limit
    If (eps > 0 And Abs(pv - pv_old) < eps) Or _
       (eps < 0 And Abs(pv - pv_old) < -eps * pv) Then
      liq_air_iteration_at_t_p = IsOK
      d_air_si = dh
      a_si = ah
      Exit For
    End If
  End If
  
  If it = maxit Then Exit For
  
  'coefficient matrix
  a(1, 1) = -ah * fh_aa
  a(1, 2) = fh_d - ah * fh_ad - p_si / dh ^ 2
  a(2, 1) = dh * fh_ad
  a(2, 2) = 2 * fh_d + dh * fh_dd

  'right-hand sides, must vanish at equilibrium
  b(1) = gl - fh - p_si / dh + ah * fh_a
  b(2) = p_si / dh - dh * fh_d
  
  'solve equations
  If matrix_solve(a(), b(), x(), 2) <> 0 Then Exit For 'matrix singular

  'update air fraction & density
  ah = ah + x(1)
  If ah < 0 Or ah > 1 Then Exit For
  dh = dh + x(2)
  If dh <= 0 Then Exit For

Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'AIR_LIMITS
  If a_si < 0 Or a_si > 1 Or _
     d_air_si <= dry_air_dmin Or d_air_si > dry_air_dmax Then
    a_si = ErrorReturn
    d_air_si = ErrorReturn
    liq_air_iteration_at_t_p = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Function liq_air_iteration_at_a_t(ByVal a_si, _
                                          ByVal t_si, _
                                          ByVal maxit, _
                                          ByVal eps, _
                                          ByRef p_si, _
                                          ByRef d_liq_si, _
                                          ByRef d_air_si) As Double

'this function returns the liquid-humid-air phase equilibrium from equal pressure,
'temperature and chemical potential of water of both phases at given temperature, t_si,
'and air fraction, a_si, from initial guesses for the liquid density, d_liq_si,
'the humid-air density, d_air_si, and the pressure, p_si

'The iteration limit eps refers to the error in the pressure

'output:    liq_air_iteration_at_a_t = IsOK if successful
'           liq_air_iteration_at_a_t = ErrorReturn is returned if
'           - the maximum number of iterations is exceeded without meeting the exit criterion
'           - function calls to flu_f_si or air_f_si have returned an error
'           - densities or pressure have taken a zero or negative value during the iteration
'     p_si: absolute pressure in Pa
' d_liq_si: liquid density at t_si, p_si
' d_air_si: humid-air density in the saturation equilibrium

'input: t_si: absolute temperature in K
'       a_si: mass fraction of dry air in humid air, kg/kg
'      maxit: maximum number of iteration steps to be done
'        eps: required accuracy of pressure
'             eps > 0: absolute pressure uncertainty in Pa
'             eps < 0: relative pressure uncertainty
'       p_si: initial guess of absolute pressure in Pa
'   d_liq_si: initial guess of liquid density at p_si, t_si
'   d_air_si: initial guess of humid-air density


Dim fh As Double, fh_a As Double, fh_d As Double
Dim fh_ad As Double, fh_dd As Double
Dim fl As Double, fl_d As Double, fl_dd As Double
Dim dl As Double, dh As Double, p As Double

Dim it As Long

Dim a(3, 3) As Double, b(3) As Double, x(3) As Double

liq_air_iteration_at_a_t = ErrorReturn

'check for invalid input values
If d_air_si <= 0 Or _
   d_liq_si <= 0 Or _
   p_si <= 0 Then
  p_si = ErrorReturn
  d_liq_si = ErrorReturn
  d_air_si = ErrorReturn
  Exit Function
End If

If check_limits <> 1 Then
  If a_si < 0 Or a_si >= 1 Or _
     t_si <= 0 Then
    p_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_air_si = ErrorReturn
    Exit Function
  End If
Else
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Then
    p_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_air_si = ErrorReturn
    Exit Function
  End If
  'AIR_LIMITS
  If t_si < dry_air_tmin Or t_si > dry_air_tmax Or _
     a_si < 0 Or a_si > 1 Then
    p_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_air_si = ErrorReturn
    Exit Function
  End If
End If

check_limits = check_limits - 1

p = p_si
dh = d_air_si
dl = d_liq_si

For it = 1 To maxit

  'Derivatives of the Helmholtz function for 3D-Newton iteration
  'humid air:
  fh = air_f_si(0, 0, 0, a_si, t_si, dh)
  If fh = ErrorReturn Then Exit For
  fh_a = air_f_si(1, 0, 0, a_si, t_si, dh)
  If fh_a = ErrorReturn Then Exit For
  fh_d = air_f_si(0, 0, 1, a_si, t_si, dh)
  If fh_d = ErrorReturn Then Exit For
  fh_ad = air_f_si(1, 0, 1, a_si, t_si, dh)
  If fh_ad = ErrorReturn Then Exit For
  fh_dd = air_f_si(0, 0, 2, a_si, t_si, dh)
  If fh_dd = ErrorReturn Then Exit For

  'liquid water:
  fl = flu_f_si(0, 0, t_si, dl)
  If fl = ErrorReturn Then Exit For
  fl_d = flu_f_si(0, 1, t_si, dl)
  If fl_d = ErrorReturn Then Exit For
  fl_dd = flu_f_si(0, 2, t_si, dl)
  If fl_dd = ErrorReturn Then Exit For

  'coefficient matrix
  a(1, 1) = 1 / dh - 1 / dl
  a(1, 2) = fh_d - a_si * fh_ad - p / dh ^ 2
  a(1, 3) = -fl_d + p / dl ^ 2
  a(2, 1) = -1 / dh
  a(2, 2) = 2 * fh_d + dh * fh_dd
  a(2, 3) = 0
  a(3, 1) = -1 / dl
  a(3, 2) = 0
  a(3, 3) = 2 * fl_d + dl * fl_dd

  'right-hand sides, must vanish at equilibrium
  b(1) = fl - fh + a_si * fh_a + p * (1 / dl - 1 / dh)
  b(2) = p / dh - dh * fh_d
  b(3) = p / dl - dl * fl_d
  
  'solve equations
  If matrix_solve(a(), b(), x(), 3) <> 0 Then Exit For 'matrix singular

  'update pressure & densities
  p = p + x(1)
  If p <= 0 Then Exit For
  dh = dh + x(2)
  If dh <= 0 Then Exit For
  dl = dl + x(3)
  If dl <= 0 Then Exit For

  'check absolute or relative error limit
  If (eps > 0 And Abs(x(1)) < eps) Or _
     (eps < 0 And Abs(x(1)) < -eps * p) Then
    liq_air_iteration_at_a_t = IsOK
    d_air_si = dh
    d_liq_si = dl
    p_si = p
    Exit For
  End If

Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'FLU_LIMITS
  If d_liq_si <= flu_dmin Or d_liq_si > flu_dmax Or _
     p_si <= 0 Then
    p_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_air_si = ErrorReturn
    liq_air_iteration_at_a_t = ErrorReturn
    Exit Function
  End If
  'AIR_LIMITS
  If d_air_si <= dry_air_dmin Or d_air_si > dry_air_dmax Or _
     p_si <= 0 Then
    p_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_air_si = ErrorReturn
    liq_air_iteration_at_a_t = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Function liq_air_iteration_at_a_eta(ByVal a_si, _
                                            ByVal eta_si, _
                                            ByVal maxit, _
                                            ByVal eps, _
                                            ByRef t_si, _
                                            ByRef p_si, _
                                            ByRef d_liq_si, _
                                            ByRef d_air_si) As Double

'this function returns the liquid-humid-air phase equilibrium from equal pressure,
'temperature and chemical potential of water of both phases at given entropy, eta_si,
'and air fraction, a_si, from initial guesses for the liquid density, d_liq_si,
'the humid-air density, d_air_si, the temperature, t_si, and the pressure, p_si

'The iteration limit eps refers to the error in the pressure

'output:    liq_air_iteration_at_a_eta = IsOK if successful
'           liq_air_iteration_at_a_eta = ErrorReturn is returned if
'           - the maximum number of iterations is exceeded without meeting the exit criterion
'           - function calls to flu_f_si or air_f_si have returned an error
'           - densities, pressure or temperature have taken a zero or negative value during the iteration
'     t_si: absolute temperature in K
'     p_si: absolute pressure in Pa
' d_liq_si: liquid density at t_si, p_si
' d_air_si: humid-air density at the saturation equilibrium

'input: eta_si: specific entropy in J/(kg K)
'         a_si: mass fraction of dry air in humid air, kg/kg
'        maxit: maximum number of iteration steps to be done
'          eps: required accuracy of pressure
'               eps > 0: absolute pressure uncertainty in Pa
'               eps < 0: relative pressure uncertainty
'         t_si: initial guess of absolute temperature in K
'         p_si: initial guess of absolute pressure in Pa
'     d_liq_si: initial guess of liquid density at p_si, t_si
'     d_air_si: initial guess of humid-air density


Dim fh As Double, fh_a As Double, fh_d As Double
Dim fh_ad As Double, fh_dd As Double
Dim fh_t As Double, fh_at As Double, fh_td As Double, fh_tt As Double
Dim fl As Double, fl_d As Double, fl_dd As Double
Dim fl_t As Double, fl_td As Double
Dim dl As Double, dh As Double, p As Double, t As Double

Dim it As Long

Dim a(4, 4) As Double, b(4) As Double, x(4) As Double

liq_air_iteration_at_a_eta = ErrorReturn

'check for invalid input values
If d_air_si <= 0 Or _
   d_liq_si <= 0 Or _
   t_si <= 0 Or _
   p_si <= 0 Then
  t_si = ErrorReturn
  p_si = ErrorReturn
  d_liq_si = ErrorReturn
  d_air_si = ErrorReturn
  Exit Function
End If

If check_limits <> 1 Then
  If a_si < 0 Or a_si >= 1 Then
    t_si = ErrorReturn
    p_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_air_si = ErrorReturn
    Exit Function
  End If
Else
  'AIR_LIMITS
  If a_si < 0 Or a_si > 1 Then
    t_si = ErrorReturn
    p_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_air_si = ErrorReturn
    Exit Function
  End If
End If

check_limits = check_limits - 1

p = p_si
dh = d_air_si
dl = d_liq_si
t = t_si

For it = 1 To maxit

  'Derivatives of the Helmholtz function for 4D-Newton iteration
  'humid air:
  fh = air_f_si(0, 0, 0, a_si, t, dh)
  If fh = ErrorReturn Then Exit For
  fh_a = air_f_si(1, 0, 0, a_si, t, dh)
  If fh_a = ErrorReturn Then Exit For
  fh_t = air_f_si(0, 1, 0, a_si, t, dh)
  If fh_t = ErrorReturn Then Exit For
  fh_d = air_f_si(0, 0, 1, a_si, t, dh)
  If fh_d = ErrorReturn Then Exit For
  fh_ad = air_f_si(1, 0, 1, a_si, t, dh)
  If fh_ad = ErrorReturn Then Exit For
  fh_at = air_f_si(1, 1, 0, a_si, t, dh)
  If fh_at = ErrorReturn Then Exit For
  fh_tt = air_f_si(0, 2, 0, a_si, t, dh)
  If fh_tt = ErrorReturn Then Exit For
  fh_td = air_f_si(0, 1, 1, a_si, t, dh)
  If fh_td = ErrorReturn Then Exit For
  fh_dd = air_f_si(0, 0, 2, a_si, t, dh)
  If fh_dd = ErrorReturn Then Exit For

  'liquid water:
  fl = flu_f_si(0, 0, t, dl)
  If fl = ErrorReturn Then Exit For
  fl_t = flu_f_si(1, 0, t, dl)
  If fl_t = ErrorReturn Then Exit For
  fl_d = flu_f_si(0, 1, t, dl)
  If fl_d = ErrorReturn Then Exit For
  fl_td = flu_f_si(1, 1, t, dl)
  If fl_td = ErrorReturn Then Exit For
  fl_dd = flu_f_si(0, 2, t, dl)
  If fl_dd = ErrorReturn Then Exit For

  'coefficient matrix
  a(1, 1) = 1 / dh - 1 / dl
  a(1, 2) = fh_d - a_si * fh_ad - p / dh ^ 2
  a(1, 3) = -fl_d + p / dl ^ 2
  a(1, 4) = fh_t - a_si * fh_at - fl_t
  a(2, 1) = -1 / dh
  a(2, 2) = 2 * fh_d + dh * fh_dd
  a(2, 3) = 0
  a(2, 4) = dh * fh_td
  a(3, 1) = -1 / dl
  a(3, 2) = 0
  a(3, 3) = 2 * fl_d + dl * fl_dd
  a(3, 4) = dl * fl_td
  a(4, 1) = 0
  a(4, 2) = fh_td
  a(4, 3) = 0
  a(4, 4) = fh_tt

  'right-hand sides, must vanish at equilibrium
  b(1) = fl - fh + a_si * fh_a + p * (1 / dl - 1 / dh)
  b(2) = p / dh - dh * fh_d
  b(3) = p / dl - dl * fl_d
  b(4) = -eta_si - fh_t
  
  'solve equations
  If matrix_solve(a(), b(), x(), 4) <> 0 Then Exit For 'matrix singular

  'update pressure, densities and temperature
  p = p + x(1)
  If p <= 0 Then Exit For
  dh = dh + x(2)
  If dh <= 0 Then Exit For
  dl = dl + x(3)
  If dl <= 0 Then Exit For
  t = t + x(4)
  If t <= 0 Then Exit For

  'check absolute or relative error limit
  If (eps > 0 And Abs(x(1)) < eps) Or _
     (eps < 0 And Abs(x(1)) < -eps * p) Then
    liq_air_iteration_at_a_eta = IsOK
    d_air_si = dh
    d_liq_si = dl
    t_si = t
    p_si = p
    Exit For
  End If

Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Or _
     p_si <= 0 Or _
     d_liq_si <= flu_dmin Or d_liq_si > flu_dmax Then
    t_si = ErrorReturn
    p_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_air_si = ErrorReturn
    liq_air_iteration_at_a_eta = ErrorReturn
    Exit Function
  End If
  'AIR_LIMITS
  If t_si < dry_air_tmin Or t_si > dry_air_tmax Or _
     p_si <= 0 Or _
     d_air_si <= dry_air_dmin Or d_air_si > dry_air_dmax Then
    t_si = ErrorReturn
    p_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_air_si = ErrorReturn
    liq_air_iteration_at_a_eta = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Function liq_air_iteration_at_a_p(ByVal a_si, _
                                          ByVal p_si, _
                                          ByVal maxit, _
                                          ByVal eps, _
                                          ByRef t_si, _
                                          ByRef d_liq_si, _
                                          ByRef d_air_si) As Double

'this function returns the liquid-humid-air phase equilibrium from equal pressure,
'temperature and chemical potential of water of both phases at given pressure, p_si,
'and air fraction, a_si, from initial guesses for the liquid density, d_liq_si,
'the humid-air density, d_air_si, and the temperature, t_si

'The iteration limit eps refers to the error in the temperature

'output:    liq_air_iteration_at_a_p = IsOK if successful
'           liq_air_iteration_at_a_p = ErrorReturn is returned if
'           - the maximum number of iterations is exceeded without meeting the exit criterion
'           - function calls to flu_f_si or air_f_si have returned an error
'           - densities or temperature have taken a zero or negative value during the iteration
'     t_si: absolute temperature in K
' d_liq_si: liquid density at t_si, p_si
' d_air_si: humid-air density in the saturation equilibrium

'input: p_si: absolute pressure in Pa
'       a_si: mass fraction of dry air in humid air, kg/kg
'      maxit: maximum number of iteration steps to be done
'        eps: required accuracy of temperature
'             eps > 0: absolute temperature uncertainty in K
'             eps < 0: relative temperature uncertainty
'       t_si: initial guess of absolute temperature in K
'   d_liq_si: initial guess of liquid density at p_si, t_si
'   d_air_si: initial guess of humid-air density


Dim fh As Double, fh_a As Double, fh_t As Double, fh_d As Double
Dim fh_ad As Double, fh_at As Double, fh_td As Double, fh_dd As Double
Dim fl As Double, fl_d As Double, fl_t As Double, fl_td As Double, fl_dd As Double
Dim dl As Double, dh As Double, t As Double

Dim it As Long

Dim a(3, 3) As Double, b(3) As Double, x(3) As Double

liq_air_iteration_at_a_p = ErrorReturn

'check for invalid input values
If a_si < 0 Or a_si > 1 Or _
   d_air_si <= 0 Or _
   d_liq_si <= 0 Or _
   t_si <= 0 Or _
   p_si <= 0 Then
  t_si = ErrorReturn
  d_liq_si = ErrorReturn
  d_air_si = ErrorReturn
  Exit Function
End If

check_limits = check_limits - 1

t = t_si
dh = d_air_si
dl = d_liq_si

For it = 1 To maxit

  'Derivatives of the Helmholtz function for 3D-Newton iteration
  'humid air:
  fh = air_f_si(0, 0, 0, a_si, t, dh)
  If fh = ErrorReturn Then Exit For
  fh_a = air_f_si(1, 0, 0, a_si, t, dh)
  If fh_a = ErrorReturn Then Exit For
  fh_t = air_f_si(0, 1, 0, a_si, t, dh)
  If fh_t = ErrorReturn Then Exit For
  fh_d = air_f_si(0, 0, 1, a_si, t, dh)
  If fh_d = ErrorReturn Then Exit For
  fh_ad = air_f_si(1, 0, 1, a_si, t, dh)
  If fh_ad = ErrorReturn Then Exit For
  fh_at = air_f_si(1, 1, 0, a_si, t, dh)
  If fh_at = ErrorReturn Then Exit For
  fh_td = air_f_si(0, 1, 1, a_si, t, dh)
  If fh_td = ErrorReturn Then Exit For
  fh_dd = air_f_si(0, 0, 2, a_si, t, dh)
  If fh_dd = ErrorReturn Then Exit For

  'liquid water:
  fl = flu_f_si(0, 0, t, dl)
  If fl = ErrorReturn Then Exit For
  fl_t = flu_f_si(1, 0, t, dl)
  If fl_t = ErrorReturn Then Exit For
  fl_d = flu_f_si(0, 1, t, dl)
  If fl_d = ErrorReturn Then Exit For
  fl_td = flu_f_si(0, 2, t, dl)
  If fl_td = ErrorReturn Then Exit For
  fl_dd = flu_f_si(0, 2, t, dl)
  If fl_dd = ErrorReturn Then Exit For

  'coefficient matrix
  a(1, 1) = fh_t - a_si * fh_at - fl_t
  a(1, 2) = fh_d - a_si * fh_ad - p_si / dh ^ 2
  a(1, 3) = -fl_d + p_si / dl ^ 2
  a(2, 1) = dh * fh_td
  a(2, 2) = 2 * fh_d + dh * fh_dd
  a(2, 3) = 0
  a(3, 1) = dl * fl_td
  a(3, 2) = 0
  a(3, 3) = 2 * fl_d + dl * fl_dd

  'right-hand sides, must vanish at equilibrium
  b(1) = fl - fh + a_si * fh_a + p_si * (1 / dl - 1 / dh)
  b(2) = p_si / dh - dh * fh_d
  b(3) = p_si / dl - dl * fl_d
  
  'solve equations
  If matrix_solve(a(), b(), x(), 3) <> 0 Then Exit For 'matrix singular

  'update temperature & densities
  t = t + x(1)
  If t <= 0 Then Exit For
  dh = dh + x(2)
  If dh <= 0 Then Exit For
  dl = dl + x(3)
  If dl <= 0 Then Exit For

  'check absolute or relative error limit
  If (eps > 0 And Abs(x(1)) < eps) Or _
     (eps < 0 And Abs(x(1)) < -eps * t) Then
    liq_air_iteration_at_a_p = IsOK
    d_air_si = dh
    d_liq_si = dl
    t_si = t
    Exit For
  End If

Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Or _
     d_liq_si <= flu_dmin Or d_liq_si > flu_dmax Then
    t_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_air_si = ErrorReturn
    liq_air_iteration_at_a_p = ErrorReturn
    Exit Function
  End If
  'AIR_LIMITS
  If t_si < dry_air_tmin Or t_si > dry_air_tmax Or _
     d_air_si <= dry_air_dmin Or d_air_si > dry_air_dmax Then
    t_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_air_si = ErrorReturn
    liq_air_iteration_at_a_p = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Function aux_airfraction_si(ByVal t_si As Double, _
                                    ByVal p_si As Double) As Double

'function returns an estimate of the air mass fraction of saturated
'humid air at given absolute temperature, t_si, in K and absolute
'pressure, p_si, in Pa

Dim dh As Double, dv As Double, pv As Double

aux_airfraction_si = ErrorReturn
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

'as an estimate, add the densities of dry air and vapour

'vapour pressure
pv = liq_vap_vapourpressure_si(t_si)
If pv = ErrorReturn Then Exit Function
If pv > p_si Then Exit Function

If pv = p_si Then
  aux_airfraction_si = 0
  Exit Function
End If

'vapour density
dv = liq_vap_density_vap_si
If dv = ErrorReturn Then Exit Function
If dv <= 0 Then Exit Function

'dry air density at remaining partial pressure
dh = air_density_si(1, t_si, p_si - pv)
If dh = ErrorReturn Then Exit Function
If dh <= 0 Then Exit Function

'air fraction of the ideal mixture
aux_airfraction_si = dh / (dh + dv)
           
End Function

'==========================================================================
Private Function aux_temperature_si(ByVal a_si As Double, _
                                    ByVal p_si As Double) As Double

'function returns an estimate of the temperature in K of saturated
'humid air at given air mass fraction, a_si, in kg/kg and absolute
'pressure, p_si, in Pa

Dim pv As Double, t As Double

aux_temperature_si = ErrorReturn
If a_si < 0 Or a_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function

'as an estimate, ignore the air influence on the boiling temperature

'partial vapour pressure
pv = p_si * air_molfraction_vap_si(a_si)

'related saturation temperature
t = liq_vap_boilingtemperature_si(pv)
If t = ErrorReturn Then Exit Function
If t <= 0 Then Exit Function

aux_temperature_si = t

End Function

'==========================================================================
Private Function aux_ict_si(ByVal a_si As Double, _
                            ByVal eta_si As Double) As Double

'function returns an estimate of the isentropic condensation temperature
'in K of humid air at given air mass fraction, a_si, in kg/kg and entropy
'eta_si, in J/(kg K)

Const Tt = TP_temperature_si
Const Pt = TP_pressure_IAPWS95_si

Const ra = Gas_constant_air_si
Const RW = Gas_constant_H2O_si

'properties at the triple point
Const cpa = 1003.69  'heat capacity of air, in J/(kg K)
Const cpv = 1884.352 'heat capacity of vapour, in J/(kg K)
Const L = TP_enthalpy_vap_si - Pt / TP_density_liq_IAPWS95_si 'latent heat

Dim t As Double, etat As Double, rav As Double, xv As Double
Dim numer As Double, denom As Double

aux_ict_si = ErrorReturn
If a_si < 0 Or a_si >= 1 Then Exit Function

'to estimate t from eta and a, use the Clausius-Clapeyron equation Pvap(T)
'for the partial vapour pressure to eliminate Pvap from the ideal-gas
'equation eta(A,T,P) = const, and solve it for T

etat = air_g_entropy_si(a_si, Tt, Pt)  'entropy of humid air at the triple point
If etat = ErrorReturn Then Exit Function
rav = a_si * ra + (1 - a_si) * RW  'gas constant of humid air
xv = air_molfraction_vap_si(a_si)
If xv = ErrorReturn Then Exit Function
If xv <= 0 Then Exit Function

denom = a_si * (cpa - ra * L / (RW * Tt)) + (1 - a_si) * (cpv - L / Tt)
If denom = 0 Then Exit Function

numer = eta_si - etat - rav * Log(xv)

aux_ict_si = Tt * Exp(numer / denom)

End Function

'==========================================================================
Private Function aux_pressure_si(ByVal a_si As Double, _
                                 ByVal t_si As Double) As Double

'function returns an estimate of the pressure in Pa of saturated
'humid air at given air mass fraction, a_si, in kg/kg and absolute
'temperature, t_si, in K

Dim pv As Double, pa As Double, d As Double

aux_pressure_si = ErrorReturn
If a_si < 0 Or a_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function

'as an estimate, ignore the air-vapour interaction

'partial vapour pressure
pv = liq_vap_vapourpressure_si(t_si)
If pv = ErrorReturn Then Exit Function
If pv <= 0 Then Exit Function
d = liq_vap_density_vap_si
If d = ErrorReturn Then Exit Function
If d <= 0 Then Exit Function  'vapour density

'partial air pressure
d = a_si / (1 - a_si) * d 'air density
pa = air_f_pressure_si(1, t_si, d)
If pa = ErrorReturn Then Exit Function
If pa <= 0 Then Exit Function

aux_pressure_si = pa + pv

End Function


