Attribute VB_Name = "Ice_Air_4c_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules:
'     Constants_0_Mdl,  file Constants_0.bas
'     Convert_0_Mdl,    file Convert_0.bas
'     Ice_Liq_4_Mdl,    file Ice_Liq_4
'     Ice_Air_4b_Mdl,   file Ice_Air_4b

'#########################################################################

'=========================================================================
'This module implements the enthalpy of ice air, as well as its partial derivatives,
'depending on dry-air fraction, entropy and pressure.

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009

'Feistel, R., Kretzschmar, H.-J., Span, R., Hagen, E., Wright, D.G., Herrmann, S.:
'Thermodynamic Properties of Sea Air.
'Ocean Science Discussion 6(2009)21932325.
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98

'Control parameters of the temperature iteration
Private ctrl_initialized As Integer

Private ctrl_mode_pottemp As Integer
Private ctrl_loop_maximum As Long
Private ctrl_init_pottemp As Double
Private ctrl_eps_exit_pottemp As Double


Private Const Version = "28 May 2010"

'==========================================================================
Public Function ice_air_h_si(ByVal drv_wa As Long, _
                             ByVal drv_eta As Long, _
                             ByVal drv_p As Long, _
                             ByVal wa_si As Double, _
                             ByVal eta_si As Double, _
                             ByVal p_si As Double) As Double

'this function implements enthalpy of ice air as a thermodynamic potential,
'depending on air mass fraction, entropy and pressure

'returns ice_air_h_si as the wA-eta-P derivative

'(d/dwA)^drv_wA (d/deta)^drv_eta (d/dP)^drv_p h(wA,eta,P)

'of the specific enthalpy of ice air, h(wA,eta,P), in J/kg

'wa_si     dry-air mass fraction of ice air in kg/kg
'eta_si    specific entropy in J/(kg K)
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Air_3a, and on the iteration settings
'      for temperature by set_it_ctrl_ice_air_pottemp of this module

'Check values with default settings:  v. 1.0
'ice_air_h_si( 0, 0, 0, 0.5, -600, 1E5) =-164588.277175128
'ice_air_h_si( 1, 0, 0, 0.5, -600, 1E5) = 543.382787268341
'ice_air_h_si( 0, 1, 0, 0.5, -600, 1E5) = 271.449435547846
'ice_air_h_si( 0, 0, 1, 0.5, -600, 1E5) = 0.392073699416254
'ice_air_h_si( 2, 0, 0, 0.5, -600, 1E5) = 224938.714580441
'ice_air_h_si( 1, 1, 0, 0.5, -600, 1E5) =-177.440249222505
'ice_air_h_si( 1, 0, 1, 0.5, -600, 1E5) = 0.78196630302515
'ice_air_h_si( 0, 2, 0, 0.5, -600, 1E5) = 0.139971645622991
'ice_air_h_si( 0, 1, 1, 0.5, -600, 1E5) = 2.26942461987265E-04
'ice_air_h_si( 0, 0, 2, 0.5, -600, 1E5) =-3.57055653968235E-06

'Check values with default settings:  v. 1.1
'ice_air_h_si( 0, 0, 0, 0.5, -600, 1E5) =-164588.106002375
'ice_air_h_si( 1, 0, 0, 0.5, -600, 1E5) = 543.016638396322
'ice_air_h_si( 0, 1, 0, 0.5, -600, 1E5) = 271.449994437192
'ice_air_h_si( 0, 0, 1, 0.5, -600, 1E5) = 0.391981878510342
'ice_air_h_si( 2, 0, 0, 0.5, -600, 1E5) = 224958.5258639
'ice_air_h_si( 1, 1, 0, 0.5, -600, 1E5) =-177.457078495317
'ice_air_h_si( 1, 0, 1, 0.5, -600, 1E5) = 0.781782661019485
'ice_air_h_si( 0, 2, 0, 0.5, -600, 1E5) = 0.139985868893652
'ice_air_h_si( 0, 1, 1, 0.5, -600, 1E5) = 2.26912930198746E-04
'ice_air_h_si( 0, 0, 2, 0.5, -600, 1E5) =-3.56976697603203E-06

Dim t As Double

ice_air_h_si = ErrorReturn

If drv_wa < 0 Then Exit Function
If drv_eta < 0 Then Exit Function
If drv_p < 0 Then Exit Function
If drv_wa + drv_eta + drv_p > 2 Then Exit Function

If wa_si <= 0 Or wa_si >= 1 Then Exit Function
If p_si < 0 Then Exit Function

'compute temperature from entropy.
'if p_si = in-situ pressure, this is in-situ temperature
'if p_si = reference pressure, this is potential temperature
t = ice_air_h_temperature_si(wa_si, eta_si, p_si)
If t = ErrorReturn Then Exit Function

ice_air_h_si = ice_air_a_eta_p_derivatives_si(drv_wa, drv_eta, drv_p, wa_si, t, p_si)

End Function

'==========================================================================
Public Function ice_air_pottemp_si(ByVal a_si As Double, _
                                   ByVal t_si As Double, _
                                   ByVal p_si As Double, _
                                   ByVal pr_si As Double) As Double
                               
'this function computes potential temperature of ice air

'returns   theta(A,T,P,Pr) absolute potential temperature of ice air in K,
'a_si      absolute dry-air mass fraction of ice air in kg/kg
't_si      absolute in-situ temperature in K,
'p_si      absolute in-situ pressure in Pa
'pr_si     absolute reference pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Air_3a,
'      and on the iteration settings for temperature by set_it_ctrl_ice_air_pottemp of this module

'Check value with default settings: ice_air_pottemp_si(0.9, 230, 5e4, 1e5) = 266.106323310295  v. 1.0
'Check value with default settings: ice_air_pottemp_si(0.9, 230, 5e4, 1e5) = 266.105208870657  v. 1.1

Dim S As Double

ice_air_pottemp_si = ErrorReturn

If a_si < 0 Or a_si >= 1 Then Exit Function
If t_si < 0 Then Exit Function
If p_si < 0 Then Exit Function
If pr_si < 0 Then Exit Function

If p_si = pr_si Then
  ice_air_pottemp_si = t_si
  Exit Function
End If

S = ice_air_g_entropy_si(a_si, t_si, p_si)  'specific entropy in-situ
If S = ErrorReturn Then Exit Function

ice_air_pottemp_si = ice_air_h_temperature_si(a_si, S, pr_si)

End Function

'==========================================================================
Public Function ice_air_potdensity_si(ByVal a_si As Double, _
                                      ByVal t_si As Double, _
                                      ByVal p_si As Double, _
                                      ByVal pr_si As Double) As Double
                               
'this function computes potential density of ice air

'returns   rho_theta(A,T,P,Pr) potential density of ice air in kg/m,
'a_si      absolute dry-air mass fraction of ice air in kg/kg
't_si      absolute in-situ temperature in K,
'p_si      absolute in-situ pressure in Pa
'pr_si     absolute reference pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Air_3a,
'      and on the iteration settings for temperature by set_it_ctrl_ice_air_pottemp of this module

'Check value with default settings: ice_air_potdensity_si(0.9, 230, 5e4, 1e5) = 1.45013106602678  v. 1.0
'Check value with default settings: ice_air_potdensity_si(0.9, 230, 5e4, 1e5) = 1.45048110421887  v. 1.1

Dim S As Double, v As Double

ice_air_potdensity_si = ErrorReturn

If a_si < 0 Or a_si >= 1 Then Exit Function
If t_si < 0 Then Exit Function
If p_si < 0 Then Exit Function
If pr_si < 0 Then Exit Function

S = ice_air_g_entropy_si(a_si, t_si, p_si)  'specific entropy in-situ
If S = ErrorReturn Then Exit Function

v = ice_air_h_si(0, 0, 1, a_si, S, pr_si) 'specific volume at reference pressure
If v = ErrorReturn Then Exit Function
If v <= 0 Then Exit Function

ice_air_potdensity_si = 1 / v

End Function

'==========================================================================
Public Function ice_air_potenthalpy_si(ByVal a_si As Double, _
                                       ByVal t_si As Double, _
                                       ByVal p_si As Double, _
                                       ByVal pr_si As Double) As Double
                               
'this function computes potential enthalpy of ice air

'returns   h_theta(A,T,P,Pr) potential enthalpy of ice air in J/kg,
'a_si      absolute dry-air mass fraction of ice air in kg/kg
't_si      absolute in-situ temperature in K,
'p_si      absolute in-situ pressure in Pa
'pr_si     absolute reference pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Air_3a,
'      and on the iteration settings for temperature by set_it_ctrl_ice_air_pottemp of this module

'Check value with default settings: ice_air_potenthalpy_si(0.9, 230, 5e4, 1e5) = -35779.7348518847  v. 1.0
'Check value with default settings: ice_air_potenthalpy_si(0.9, 230, 5e4, 1e5) = -35781.2564451371  v. 1.1

Dim S As Double

ice_air_potenthalpy_si = ErrorReturn

If a_si < 0 Or a_si >= 1 Then Exit Function
If t_si < 0 Then Exit Function
If p_si < 0 Then Exit Function
If pr_si < 0 Then Exit Function

S = ice_air_g_entropy_si(a_si, t_si, p_si)  'specific entropy in-situ
If S = ErrorReturn Then Exit Function

ice_air_potenthalpy_si = ice_air_h_si(0, 0, 0, a_si, S, pr_si)

End Function

'=========================================================================
Public Function ice_air_h_temperature_si(ByVal wa_si As Double, _
                                         ByVal eta_si As Double, _
                                         ByVal p_si As Double) As Double
                                 
'returns   t(A,eta,P) (potential) temperature of ice air in K,
'wa_si     dry-air mass fraction of ice air in kg/kg
'eta_si    entropy in J/(kg K),
'p_si      absolute (reference) pressure in Pa

'this is the inverse function to
'    eta_si = ice_air_g_entropy_si(wa_si, t_si, p_si)
'in ice_air_4b

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Air_3a and on those made in this module

'Check value with default settings: ice_air_h_temperature_si(0.9, -100, 1e5) = 270.382806934047  v. 1.0
'Check value with default settings: ice_air_h_temperature_si(0.9, -100, 1e5) = 270.383680119296  v. 1.1

Dim t As Double, eps As Double, maxit As Long

ice_air_h_temperature_si = ErrorReturn

init_it_ctrl_pottemp

Select Case ctrl_mode_pottemp
  Case 0:    t = aux_ice_air_temperature_si(wa_si, eta_si, p_si)
  Case Else: t = ctrl_init_pottemp
End Select
If t <= 0 Then Exit Function
If t = ErrorReturn Then Exit Function

Select Case ctrl_loop_maximum
  Case 0:      maxit = 100
  Case -1:     ice_air_h_temperature_si = t
               Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else:   Exit Function
End Select

eps = ctrl_eps_exit_pottemp
If eps <= 0 Then Exit Function

'run the iteration
ice_air_h_temperature_si = pottemp_iteration(wa_si, eta_si, p_si, t, maxit, eps)

End Function

'=========================================================================
Public Function ice_air_h_lapserate_si(ByVal wa_si As Double, _
                                       ByVal eta_si As Double, _
                                       ByVal p_si As Double) As Double
                                 
'returns   gamma(A,eta,P) = (d2h/deta dp)_A  ice-adiabatic lapse rate in K/Pa,
'wa_si     dry-air mass fraction of ice air in kg/kg
'eta_si    specific entropy in J/(kg K),
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Air_3a and on those made in this module


'Check value with default settings: ice_air_h_lapserate_si(0.9, -100, 1e5) = 4.42476802728586E-04  v. 1.0
'Check value with default settings: ice_air_h_lapserate_si(0.9, -100, 1e5) = 4.42457786754781E-04  v. 1.1

ice_air_h_lapserate_si = ice_air_h_si(0, 1, 1, wa_si, eta_si, p_si)

End Function

'=========================================================================
Public Function ice_air_h_cp_si(ByVal wa_si As Double, _
                                ByVal eta_si As Double, _
                                ByVal p_si As Double) As Double
                                 
'returns   cp(A,eta,P) = t/(d2h/deta2)_A_P  heat capacity of ice air in J/(kg K),
'wa_si     dry-air mass fraction of ice air in kg/kg
'eta_si    specific entropy in J/(kg K),
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Air_3a and on those made in this module

'Check value with default settings: ice_air_h_cp_si(0.9, -100, 1e5) = 1766.84429536379  v. 1.0
'Check value with default settings: ice_air_h_cp_si(0.9, -100, 1e5) = 1766.52051487688  v. 1.1

Dim t As Double

ice_air_h_cp_si = ErrorReturn

t = ice_air_h_temperature_si(wa_si, eta_si, p_si)
If t = ErrorReturn Then Exit Function
If t <= 0 Then Exit Function

ice_air_h_cp_si = ice_air_g_cp_si(wa_si, t, p_si)

End Function

'=========================================================================
Public Function ice_air_h_kappa_s_si(ByVal wa_si As Double, _
                                     ByVal eta_si As Double, _
                                     ByVal p_si As Double) As Double
                                 
'returns   kappa(A,eta,P) = - (d2h/dp2)_A_eta/(dh/dp)_A_eta adiabatic compressibility of ice air in 1/Pa,
'wa_si     dry-air mass fraction of ice air in kg/kg
'eta_si    specific entropy in J/(kg K),
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Air_3a and on those made in this module

'Check value with default settings: ice_air_h_kappa_s_si(0.9, -100, 1e5) = 8.23024116522037E-06  v. 1.0
'Check value with default settings: ice_air_h_kappa_s_si(0.9, -100, 1e5) = 8.23031581046871E-06  v. 1.1

Dim d As Double, c As Double

ice_air_h_kappa_s_si = ErrorReturn

d = ice_air_h_density_si(wa_si, eta_si, p_si)
If d = ErrorReturn Then Exit Function
If d <= 0 Then Exit Function

c = ice_air_h_si(0, 0, 2, wa_si, eta_si, p_si)
If c = ErrorReturn Then Exit Function
If c >= 0 Then Exit Function

ice_air_h_kappa_s_si = -d * c

End Function

'=========================================================================
Public Function ice_air_h_density_si(ByVal wa_si As Double, _
                                     ByVal eta_si As Double, _
                                     ByVal p_si As Double) As Double
                                 
'returns   d(A,eta,P) = 1/(dh/dp)_A_eta  density of ice air in kg/m3,
'wa_si     dry-air mass fraction of ice air in kg/kg
'eta_si    specific entropy in J/(kg K),
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Air_3a and on those made in this module

'Check value with default settings: ice_air_h_density_si(0.9, -100, 1e5) = 1.42498657690712  v. 1.0
'Check value with default settings: ice_air_h_density_si(0.9, -100, 1e5) = 1.4253189599345  v. 1.1

Dim t As Double

ice_air_h_density_si = ErrorReturn

t = ice_air_h_temperature_si(wa_si, eta_si, p_si)
If t = ErrorReturn Then Exit Function
If t <= 0 Then Exit Function

ice_air_h_density_si = ice_air_g_density_si(wa_si, t, p_si)

End Function

'=========================================================================
Private Function pottemp_iteration(ByVal wa_si As Double, _
                                   ByVal eta_si As Double, _
                                   ByVal p_si As Double, _
                                   ByVal t_si As Double, _
                                   ByVal maxit As Long, _
                                   ByVal eps As Double) As Double

'returns   theta =  absolute potential temperature of ice air in K,
'          i.e. the temperature that solves the equation
'          eta_si = ice_air_g_entropy_si(wa_si, theta, p_si)

'wa_si     dry-air mass fraction of ice air in kg/kg
'eta_si    entropy in J/(kg K)
'p_si      absolute (reference) pressure in Pa
't_si      absolute (potential) temperature in K, initial value
'maxit     max. number of iterations
'eps       required tolerance in K

Dim i As Long
Dim d As Double, S As Double, theta As Double, cp As Double, dt As Double
Dim ctrl0 As Double, ctrl1 As Double
Dim theta_max As Double

pottemp_iteration = ErrorReturn

If wa_si < 0 Or wa_si > 1 Then Exit Function
If eps <= 0 Then Exit Function
If maxit <= 0 Then Exit Function

If check_limits = 1 Then
  'ICE_LIMITS
  If t_si <= ice_tmin Or t_si > ice_tmax Or _
     p_si <= ice_pmin Or p_si > ice_pmax Then Exit Function
  'AIR_LIMITS
  If t_si < dry_air_tmin Or t_si > dry_air_tmax Then Exit Function
Else
  If t_si <= 0 Then Exit Function
  If p_si <= 0 Then Exit Function
End If

check_limits = check_limits - 1

theta = t_si

For i = 1 To maxit

  'get entropy and its first derivative for Newton iteration
  S = ice_air_g_entropy_si(wa_si, theta, p_si)
  cp = ice_air_g_cp_si(wa_si, theta, p_si)
    
  If S = ErrorReturn Then Exit For
  If cp = ErrorReturn Then Exit For
  If cp <= 0 Then Exit For
  
  'next temperature improvement step
  dt = theta * (eta_si - S) / cp
  theta = theta + dt
  If theta <= 0 Then Exit For
    
  If Abs(dt) < eps Then
    pottemp_iteration = theta
    Exit For
  End If
  
Next i

check_limits = check_limits + 1

If check_limits = 1 Then
  'ICE_LIMITS
  If theta <= ice_tmin Or theta > ice_tmax Then
    pottemp_iteration = ErrorReturn
    Exit Function
  End If
  'AIR_LIMITS
  If theta < dry_air_tmin Or theta > dry_air_tmax Then
    pottemp_iteration = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Function ice_air_a_eta_p_derivatives_si(ByVal drv_wa As Integer, _
                                                ByVal drv_eta As Integer, _
                                                ByVal drv_p As Integer, _
                                                ByVal wa_si As Double, _
                                                ByVal t_si As Double, _
                                                ByVal p_si As Double) As Double
                                              
'this function computes ice-air wa-eta-p derivatives of h from wa-t-p derivatives of g

Dim g As Double, gt As Double, gp As Double, ga As Double
Dim gaa As Double, gat As Double, gap As Double
Dim gtt As Double, gpp As Double, gtp As Double

Dim h As Double, d As Double
Dim ctrl0_air As Double, ctrl1_air As Double
Dim ctrl0_liq As Double, ctrl1_liq As Double

'in one case we do not at all need to compute the Gibbs function:
If drv_wa = 0 And drv_eta = 1 And drv_p = 0 Then
  ice_air_a_eta_p_derivatives_si = t_si
  Exit Function
End If

ice_air_a_eta_p_derivatives_si = ErrorReturn

Select Case drv_wa

  Case 0:
    Select Case drv_eta
    
      Case 0:
        Select Case drv_p
        
          Case 0: g = ice_air_g_si(0, 0, 0, wa_si, t_si, p_si)
                  If g = ErrorReturn Then Exit Function
                  gt = ice_air_g_si(0, 1, 0, wa_si, t_si, p_si)
                  If gt = ErrorReturn Then Exit Function
                  h = g - t_si * gt                          'h
          
          Case 1: gp = ice_air_g_si(0, 0, 1, wa_si, t_si, p_si)
                  If gp = ErrorReturn Then Exit Function
                  h = gp                                      'dh/dp
                  
          Case 2: gtt = ice_air_g_si(0, 2, 0, wa_si, t_si, p_si)
                  If gtt = ErrorReturn Then Exit Function
                  If gtt = 0 Then Exit Function
                  gtp = ice_air_g_si(0, 1, 1, wa_si, t_si, p_si)
                  If gtp = ErrorReturn Then Exit Function
                  gpp = ice_air_g_si(0, 0, 2, wa_si, t_si, p_si)
                  If gpp = ErrorReturn Then Exit Function
                  h = (gtt * gpp - gtp ^ 2) / gtt              'd2h/dp2
                  
          Case Else: Exit Function
        End Select
        
      Case 1:
        Select Case drv_p
          Case 0: h = t_si                                     'dh/deta, has already been handled initially
          
          Case 1: gtt = ice_air_g_si(0, 2, 0, wa_si, t_si, p_si)
                  If gtt = ErrorReturn Then Exit Function
                  If gtt = 0 Then Exit Function
                  gtp = ice_air_g_si(0, 1, 1, wa_si, t_si, p_si)
                  If gtp = ErrorReturn Then Exit Function
                  h = -gtp / gtt                               'd2h/detadp
                  
          Case Else: Exit Function
        End Select

      Case 2:
        Select Case drv_p
          Case 0: gtt = ice_air_g_si(0, 2, 0, wa_si, t_si, p_si)
                  If gtt = ErrorReturn Then Exit Function
                  If gtt = 0 Then Exit Function
                  h = -1# / gtt                                'd2h/deta2
                  
          Case Else: Exit Function
        End Select

      Case Else: Exit Function
    End Select
    
  Case 1:
    Select Case drv_eta
    
      Case 0:
        Select Case drv_p
        
          Case 0: ga = ice_air_g_si(1, 0, 0, wa_si, t_si, p_si)
                  If ga = ErrorReturn Then Exit Function
                  h = ga                                        'dh/da
          
          Case 1: gtt = ice_air_g_si(0, 2, 0, wa_si, t_si, p_si)
                  If gtt = ErrorReturn Then Exit Function
                  If gtt = 0 Then Exit Function
                  gat = ice_air_g_si(1, 1, 0, wa_si, t_si, p_si)
                  If gat = ErrorReturn Then Exit Function
                  gap = ice_air_g_si(1, 0, 1, wa_si, t_si, p_si)
                  If gap = ErrorReturn Then Exit Function
                  h = (gtt * gap - gat * gtp) / gtt             'd2h/dadp
          
          Case Else: Exit Function
        End Select

      Case 1:
        Select Case drv_p
        
          Case 0: gtt = ice_air_g_si(0, 2, 0, wa_si, t_si, p_si)
                  If gtt = ErrorReturn Then Exit Function
                  If gtt = 0 Then Exit Function
                  gat = ice_air_g_si(1, 1, 0, wa_si, t_si, p_si)
                  If gat = ErrorReturn Then Exit Function
                  h = -gat / gtt                                'd2h/dadeta
          
          Case Else: Exit Function
        End Select

      Case Else: Exit Function
    End Select

  
  Case 2:
    Select Case drv_eta
    
      Case 0:
        Select Case drv_p
        
          Case 0: gtt = ice_air_g_si(0, 2, 0, wa_si, t_si, p_si)
                  If gtt = ErrorReturn Then Exit Function
                  If gtt = 0 Then Exit Function
                  gat = ice_air_g_si(1, 1, 0, wa_si, t_si, p_si)
                  If gat = ErrorReturn Then Exit Function
                  gaa = ice_air_g_si(2, 0, 0, wa_si, t_si, p_si)
                  If gaa = ErrorReturn Then Exit Function
                  h = (gtt * gaa - gat ^ 2) / gtt               'd2h/da2
          
          Case Else: Exit Function
        End Select

      Case Else: Exit Function
    End Select

  Case Else: Exit Function
End Select

ice_air_a_eta_p_derivatives_si = h

End Function

'==========================================================================
Private Sub init_it_ctrl_pottemp()

If ctrl_initialized = -1 Then Exit Sub

ctrl_initialized = -1

'Set default values and modes for temperature iteration
ctrl_loop_maximum = 100
ctrl_mode_pottemp = 0           'default: theta = aux_ice_air_temperature_si
ctrl_init_pottemp = 273.15
ctrl_eps_exit_pottemp = 0.0001  'default = 0.1 mK

End Sub

'==========================================================================
Public Sub set_it_ctrl_ice_air_pottemp(ByVal key As String, ByVal value As Double)

'this sub sets control parameters for the Newton iteration used to compute
'potential temperature from reference pressure

'key             value
'it_steps        0           set iteration number to default (100)
'it_steps        n > 0       set iteration number to n
'it_steps       -1           do not iterate, use initial value

'init_theta      0           use default potential temperature theta = aux_temperature_min_si to start
'init_theta      t > 0       use value t as potential temperature to start

'tol_theta       0           use default exit accuracy for potential temperature (0.1 mK)
'tol_theta       eps > 0     use eps as exit accuracy for potential temperature

init_it_ctrl_pottemp

Select Case LCase(Trim(key))

  Case "it_steps":   'iteration steps
    Select Case value
      Case 0:      ctrl_loop_maximum = 100     'default = 100
      Case Is < 0: ctrl_loop_maximum = -1
      Case Else:   ctrl_loop_maximum = value
    End Select

  Case "init_theta":   'start theta
    Select Case CLng(value)
      Case 0:      ctrl_mode_pottemp = 0        'default: aux_temperature_min_si
      Case Is > 0: ctrl_mode_pottemp = 1
                   ctrl_init_pottemp = value
    End Select

  Case "tol_theta":      'required theta tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_pottemp = 0.0001 'default = 0.1 mK
      Case Is > 0: ctrl_eps_exit_pottemp = value
    End Select

End Select

End Sub

'=========================================================================
Private Function aux_ice_air_temperature_si(ByVal wa_si As Double, _
                                           ByVal eta_si As Double, _
                                           ByVal p_si As Double) As Double

'this function estimates the temperature of ice air from its dry-air fraction wa_si in kg/kg,
'its specific entropy eta_si in J/(kg K), and its absolute pressure, p_si in Pa

Const Tt = TP_temperature_si         'triple-point temperature in K
Const Pt = TP_pressure_IAPWS95_si    'triple-point pressure in Pa

Const cpa = 1003.68997553091         'triple-point heat capacity of air in J/(kg K)
Const cpi = 2096.78431621667         'triple-point heat capacity of ice in J/(kg K)

Const sat = 1467.66694249983         'triple-point entropy of air in J/(kg K)
Const sit = -1220.69433939648        'triple-point entropy of ice in J/(kg K)

Const ra = Gas_constant_air_si

Dim numer As Double, denom As Double, t As Double, tf As Double

aux_ice_air_temperature_si = ErrorReturn
If p_si <= 0 Then Exit Function
If wa_si < 0 Or wa_si > 1 Then Exit Function

'assume constant heat capacity cp of ice and air, use ideal-gas equation,
'neglect the vapour fraction in humid air:

numer = eta_si - sit - wa_si * (sat - sit - ra * Log(p_si / Pt))

denom = cpi + wa_si * (cpa - cpi)
If denom = 0 Then Exit Function

t = Tt * Exp(numer / denom)

'if estimate is above the freezing point, rather take the freezing point
tf = ice_liq_meltingtemperature_si(p_si)
If tf <> ErrorReturn And tf > 0 Then
  If t > tf Then t = tf
End If

aux_ice_air_temperature_si = t

End Function

