Attribute VB_Name = "Ice_Air_4b_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules:
'     Constants_0_Mdl, file Constants_0.bas
'     Convert_0_Mdl,   file Convert_0.bas
'     Ice_1_Mdl,       file Ice_1.bas
'     Air_3a_Mdl,      file Air_3a.bas
'     Ice_Air_4a_Mdl,  file Ice_Air_4a.bas
'
'#########################################################################

'=========================================================================
'This module implements the Gibbs function of ice air, i.e., of the composite
'system of ice and humid air in mutual equilibrium ("icy air", e.g. cirrus clouds)
'Therefore, the air properties computed here refer to saturated air.

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009

'Feistel, R., Kretzschmar, H.-J., Span, R., Hagen, E., Wright, D.G., Herrmann, S.:
'Thermodynamic Properties of Sea Air.
'Ocean Science Discussion 6(2009)21932325.
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98
'Private Const IsOK = -1

Private Const Version = "28 May 2010"

'==========================================================================
Public Function ice_air_g_si(ByVal drv_wa As Integer, _
                             ByVal drv_t As Integer, _
                             ByVal drv_p As Integer, _
                             ByVal wa_si As Double, _
                             ByVal t_si As Double, _
                             ByVal p_si As Double) As Double

'this implements the Gibbs function of ice air computed from its Helmholtz function
'and its first and second derivatives with respect to the air fraction wa_si,
'the absolute temperature t_si and the absolute pressure p_si

'note: this Gibbs function is not designed for liquid or critical air states
'note: the accuracy of this functions depends on the iteration settings of this module

'Output: ice_air_g_si = specific Gibbs energy in J/kg or its derivative

' g(wa,T,P) = (1-w) * gi(T,P) + w * gh(A(T,P),T,P)

' w = mass fraction of humid air in ice air, w(wa, T, P) = wa/A(T,P)
'  (1-w) = solid mass fraction of ice air
' A = saturated air mass fraction of humid air
'  (1-A) = saturated specific humidity of the humid-air component
' gi = Gibbs function of ice
' gh = Gibbs function of humid air
'
' wa_si = absolute dry-air fraction in kg/kg, i.e. the mass fraction of dry air in ice air
' t_si  = T = absolute temperature in K, ITS-90
' p_si  = P = absolute pressure in Pa

'Check values with default settings:  v. 1.0
'ice_air_g_si( 0, 0, 0, 0.5, 270, 1E5) =-2595.73785824947
'ice_air_g_si( 1, 0, 0, 0.5, 270, 1E5) = 2382.02354605727
'ice_air_g_si( 0, 1, 0, 0.5, 270, 1E5) = 610.261631356964
'ice_air_g_si( 0, 0, 1, 0.5, 270, 1E5) = 0.389737675173871
'ice_air_g_si( 2, 0, 0, 0.5, 270, 1E5) = 0
'ice_air_g_si( 1, 1, 0, 0.5, 270, 1E5) =-1269.4234474149
'ice_air_g_si( 1, 0, 1, 0.5, 270, 1E5) = 0.777294756075268
'ice_air_g_si( 0, 2, 0, 0.5, 270, 1E5) =-7.00885819160146
'ice_air_g_si( 0, 1, 1, 0.5, 270, 1E5) = 1.60133877466081E-03
'ice_air_g_si( 0, 0, 2, 0.5, 270, 1E5) =-3.91271268873414E-06

'Check values with default settings:  v. 1.1
'ice_air_g_si( 0, 0, 0, 0.5, 270, 1E5) =-2595.57166633752
'ice_air_g_si( 1, 0, 0, 0.5, 270, 1E5) = 2382.35592988117
'ice_air_g_si( 0, 1, 0, 0.5, 270, 1E5) = 610.264515317715
'ice_air_g_si( 0, 0, 1, 0.5, 270, 1E5) = 0.389645501223942
'ice_air_g_si( 2, 0, 0, 0.5, 270, 1E5) = 0
'ice_air_g_si( 1, 1, 0, 0.5, 270, 1E5) =-1269.4176794934
'ice_air_g_si( 1, 0, 1, 0.5, 270, 1E5) = 0.777110408175409
'ice_air_g_si( 0, 2, 0, 0.5, 270, 1E5) =-7.00810930740276
'ice_air_g_si( 0, 1, 1, 0.5, 270, 1E5) = 1.60095965101151E-03
'ice_air_g_si( 0, 0, 2, 0.5, 270, 1E5) =-3.91178603884919E-06

Dim g As Double, a As Double, w As Double, d As Double
Dim gh As Double, gi As Double
Dim gh_p As Double, gi_p As Double
Dim gh_t As Double, gi_t As Double
Dim gh_pp As Double, gi_pp As Double, gc_pp As Double
Dim gh_tp As Double, gi_tp As Double, gc_tp As Double
Dim gh_tt As Double, gi_tt As Double, gc_tt As Double
Dim gh_a As Double, gh_ap As Double, gh_at As Double

Dim a_p As Double, a_t As Double

ice_air_g_si = ErrorReturn

If wa_si < 0 Or wa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

If drv_wa < 0 Then Exit Function
If drv_t < 0 Or drv_t > 2 Then Exit Function
If drv_p < 0 Or drv_p > 2 Then Exit Function

If drv_wa > 1 Then    'g is linear in wa_si
  ice_air_g_si = 0
  Exit Function
End If

a = ice_air_massfraction_air_si(t_si, p_si)      'saturated air mass fraction
If a = ErrorReturn Then Exit Function
If a <= 0 Or a >= 1 Then Exit Function

'note: a <= wa_si is necessary for validity, however, this test may fail
'due to insufficient accuracy of estimated initial iteration values
'if the difference between a and wa_si is very small
'Therefore, the following check is suppressed (22 Jan 2009):
'If a < wa_si Then Exit Function

w = wa_si / a               'gaseous mass fraction of ice air

If drv_t + drv_p > 1 Then
  d = a ^ 2 * air_g_si(2, 0, 0, a, t_si, p_si) 'air coefficient DA = A^2 * g_aa
End If

Select Case drv_wa

  Case 0:

    Select Case drv_t

      Case 0:

        Select Case drv_p

          Case 0: gh = air_g_si(0, 0, 0, a, t_si, p_si)
                  If gh = ErrorReturn Then Exit Function
                  gi = ice_g_si(0, 0, t_si, p_si)
                  If gi = ErrorReturn Then Exit Function
                  g = (1 - w) * gi + w * gh                      'g

          Case 1: gh_p = air_g_si(0, 0, 1, a, t_si, p_si)
                  If gh_p = ErrorReturn Then Exit Function
                  gi_p = ice_g_si(0, 1, t_si, p_si)
                  If gi_p = ErrorReturn Then Exit Function
                  g = (1 - w) * gi_p + w * gh_p                  'v = g_p

          Case 2: a_p = ice_air_a_si(0, 1, t_si, p_si)
                  If a_p = ErrorReturn Then Exit Function
                  gc_pp = -d * a_p ^ 2 / a                      'latent condensation derivative
                  gh_pp = air_g_si(0, 0, 2, a, t_si, p_si)
                  If gh_pp = ErrorReturn Then Exit Function
                  gi_pp = ice_g_si(0, 2, t_si, p_si)
                  If gi_pp = ErrorReturn Then Exit Function
                  g = (1 - w) * gi_pp + w * (gh_pp + gc_pp)      'g_pp

          Case Else: Exit Function
        End Select

      Case 1:     'd/dt

        Select Case drv_p

          Case 0: gh_t = air_g_si(0, 1, 0, a, t_si, p_si)
                  If gh_t = ErrorReturn Then Exit Function
                  gi_t = ice_g_si(1, 0, t_si, p_si)
                  If gi_t = ErrorReturn Then Exit Function
                  g = (1 - w) * gi_t + w * gh_t                  '-eta = g_t

          Case 1: a_t = ice_air_a_si(1, 0, t_si, p_si)
                  If a_t = ErrorReturn Then Exit Function
                  a_p = ice_air_a_si(0, 1, t_si, p_si)
                  If a_p = ErrorReturn Then Exit Function
                  gc_tp = -d * a_t * a_p / a                     'latent derivative
                  gh_tp = air_g_si(0, 1, 1, a, t_si, p_si)
                  If gh_tp = ErrorReturn Then Exit Function
                  gi_tp = ice_g_si(1, 1, t_si, p_si)
                  If gi_tp = ErrorReturn Then Exit Function
                  g = (1 - w) * gi_tp + w * (gh_tp + gc_tp)                 'g_tp

          Case Else: Exit Function
        End Select

      Case 2:     'd2/dt2

        Select Case drv_p

          Case 0: a_t = ice_air_a_si(1, 0, t_si, p_si)
                  If a_t = ErrorReturn Then Exit Function
                  gc_tt = -d * a_t ^ 2 / a                      'latent derivative
                  gh_tt = air_g_si(0, 2, 0, a, t_si, p_si)
                  If gh_tt = ErrorReturn Then Exit Function
                  gi_tt = ice_g_si(2, 0, t_si, p_si)
                  If gi_tt = ErrorReturn Then Exit Function
                  g = (1 - w) * gi_tt + w * (gh_tt + gc_tt)                 'g_tt

          Case Else: Exit Function
        End Select

       Case Else: Exit Function
    End Select

  Case 1:        'g_w, g_wp, g_wt
    gh = air_g_si(0, drv_t, drv_p, a, t_si, p_si)
    If gh = ErrorReturn Then Exit Function
    gi = ice_g_si(drv_t, drv_p, t_si, p_si)
    If gi = ErrorReturn Then Exit Function
    g = (gh - gi) / a
  
  Case 2: g = 0  'g_ww
  
  Case Else: Exit Function
End Select

ice_air_g_si = g

End Function

'=========================================================================
Private Function ice_air_a_si(ByVal drv_t As Integer, _
                              ByVal drv_p As Integer, _
                              ByVal t_si As Double, _
                              ByVal p_si As Double) As Double

'this function returns the air fraction a of saturated humid air in kg/kg,
'and its partial derivatives (d/dt)^drv_t (d/dp)^drv_p a(t,p)
'as a function of absolute temperature, t_si, in K, and absolute pressure, p_si, in Pa

Dim d As Double  'DA coefficient
Dim a As Double  'air fraction

Dim gh_p As Double, gi_p As Double, gh_ap As Double
Dim gh_t As Double, gi_t As Double, gh_at As Double

ice_air_a_si = ErrorReturn

If drv_t < 0 Or drv_t > 1 Then Exit Function
If drv_p < 0 Or drv_p > 1 Then Exit Function

If t_si < 0 Then Exit Function
If p_si < 0 Then Exit Function

a = ice_air_massfraction_air_si(t_si, p_si)
If a = ErrorReturn Then Exit Function
If a < 0 Or a >= 1 Then Exit Function

If drv_t > 0 Or drv_p > 0 Then
  d = a ^ 2 * air_g_si(2, 0, 0, a, t_si, p_si) 'air coefficient DA = A^2 * g_aa
  If d = ErrorReturn Then Exit Function
  If d <= 0 Then Exit Function
End If

Select Case drv_t

  Case 0:

    Select Case drv_p

      Case 0: ice_air_a_si = a

      Case 1: gh_p = air_g_si(0, 0, 1, a, t_si, p_si)
              If gh_p = ErrorReturn Then Exit Function
              gh_ap = air_g_si(1, 0, 1, a, t_si, p_si)
              If gh_ap = ErrorReturn Then Exit Function
              gi_p = ice_g_si(0, 1, t_si, p_si)
              If gi_p = ErrorReturn Then Exit Function
              ice_air_a_si = a * (gh_p - gi_p - a * gh_ap) / d
    End Select

  Case 1:

    Select Case drv_p

      Case 0: gh_t = air_g_si(0, 1, 0, a, t_si, p_si)
              If gh_t = ErrorReturn Then Exit Function
              gh_at = air_g_si(1, 1, 0, a, t_si, p_si)
              If gh_at = ErrorReturn Then Exit Function
              gi_t = ice_g_si(1, 0, t_si, p_si)
              If gi_t = ErrorReturn Then Exit Function
              ice_air_a_si = a * (gh_t - gi_t - a * gh_at) / d
    End Select

End Select

End Function

'=========================================================================
Public Function ice_air_solidfraction_si(ByVal wa_si As Double, _
                                         ByVal t_si As Double, _
                                         ByVal p_si As Double) As Double

'returns   1 - w(wa, T, P) = 1 - wa/A(T,P) mass fraction of solid water in ice air (0 < w < 1)
'wa_si     dry air mass fraction of ice air in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value with default settings: ice_air_solidfraction_si(0.5, 270, 1E5) = 0.498524741260245  v. 1.0
'check value with default settings: ice_air_solidfraction_si(0.5, 270, 1E5) = 0.498525089433976  v. 1.1

Dim a As Double

ice_air_solidfraction_si = ErrorReturn

If wa_si <= 0 Or wa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

a = ice_air_massfraction_air_si(t_si, p_si)
If a = ErrorReturn Then Exit Function
If a < wa_si Then Exit Function

ice_air_solidfraction_si = 1 - wa_si / a

End Function

'=========================================================================
Public Function ice_air_vapourfraction_si(ByVal wa_si As Double, _
                                          ByVal t_si As Double, _
                                          ByVal p_si As Double) As Double

'returns   wv = wa*/(1/A(T,P)-1) mass fraction of vapour in ice air (0 < w < 1)
'wa_si     dry-air mass fraction of ice air in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value with default settings: ice_air_vapourfraction_si(0.5, 270, 1E5) = 1.47525873975532E-03  v. 1.0
'check value with default settings: ice_air_vapourfraction_si(0.5, 270, 1E5) = 1.47491056602398E-03  v. 1.1

Dim a As Double

ice_air_vapourfraction_si = ErrorReturn

If wa_si <= 0 Or wa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

a = ice_air_massfraction_air_si(t_si, p_si)
If a = ErrorReturn Then Exit Function
If a < wa_si Then Exit Function

ice_air_vapourfraction_si = wa_si * (1 / a - 1)

End Function

'=========================================================================
Public Function ice_air_g_cp_si(ByVal wa_si As Double, _
                                ByVal t_si As Double, _
                                ByVal p_si As Double) As Double

'returns   cp(wa,T,P) = T * (d2g/dT2)_P heat capacity of ice air in J/(kg K),
'wa_si     mass fraction of dry air in ice air in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value with default settings: ice_air_g_cp_si(0.5, 270, 1E5) = 1892.39171173239  v. 1.0
'check value with default settings: ice_air_g_cp_si(0.5, 270, 1E5) = 1892.18951299875  v. 1.1

Dim g_tt As Double

ice_air_g_cp_si = ErrorReturn

If wa_si <= 0 Or wa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g_tt = ice_air_g_si(0, 2, 0, wa_si, t_si, p_si)
If g_tt = ErrorReturn Then Exit Function

ice_air_g_cp_si = -t_si * g_tt

End Function

'=========================================================================
Public Function ice_air_g_density_si(ByVal wa_si As Double, _
                                     ByVal t_si As Double, _
                                     ByVal p_si As Double) As Double

'returns   d(wa,T,P) = 1/(dg/dp)_T  density of ice air in kg/m3,
'wa_si     mass fraction of dry air in ice air in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value with default settings: ice_air_g_density_si(0.5, 270, 1E5) = 2.56582841151776  v. 1.0
'check value with default settings: ice_air_g_density_si(0.5, 270, 1E5) = 2.56643538000268  v. 1.1

Dim g_p As Double

ice_air_g_density_si = ErrorReturn

If wa_si <= 0 Or wa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g_p = ice_air_g_si(0, 0, 1, wa_si, t_si, p_si)
If g_p = ErrorReturn Then Exit Function
If g_p <= 0 Then Exit Function

ice_air_g_density_si = 1# / g_p

End Function

'=========================================================================
Public Function ice_air_g_enthalpy_si(ByVal wa_si As Double, _
                                      ByVal t_si As Double, _
                                      ByVal p_si As Double) As Double

'returns   h(wa,T,P) = g - T * (dg/dT)_P  enthalpy of ice air in J/kg,
'wa_si     mass fraction of dry air in ice air in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value with default settings: ice_air_g_enthalpy_si(0.5, 270, 1E5) = -167366.37832463  v. 1.0
'check value with default settings: ice_air_g_enthalpy_si(0.5, 270, 1E5) = -167366.99080212  v. 1.1

Dim g As Double, g_t As Double

ice_air_g_enthalpy_si = ErrorReturn

If wa_si <= 0 Or wa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g = ice_air_g_si(0, 0, 0, wa_si, t_si, p_si)
If g = ErrorReturn Then Exit Function
g_t = ice_air_g_si(0, 1, 0, wa_si, t_si, p_si)
If g_t = ErrorReturn Then Exit Function

ice_air_g_enthalpy_si = g - t_si * g_t

End Function

'=========================================================================
Public Function ice_air_g_entropy_si(ByVal wa_si As Double, _
                                     ByVal t_si As Double, _
                                     ByVal p_si As Double) As Double

'returns   eta(wa,T,P) = - (dg/dT)_P  entropy of ice air in J/(kg K),
'wa_si     mass fraction of dry air in ice air in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value with default settings: ice_air_g_entropy_si(0.5, 270, 1E5) = -610.261631356964  v. 1.0
'check value with default settings: ice_air_g_entropy_si(0.5, 270, 1E5) = -610.264515317715  v. 1.1

Dim g_t As Double

ice_air_g_entropy_si = ErrorReturn

If wa_si <= 0 Or wa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g_t = ice_air_g_si(0, 1, 0, wa_si, t_si, p_si)
If g_t = ErrorReturn Then Exit Function

ice_air_g_entropy_si = -g_t

End Function

'=========================================================================
Public Function ice_air_g_expansion_si(ByVal wa_si As Double, _
                                       ByVal t_si As Double, _
                                       ByVal p_si As Double) As Double

'returns   alpha(wa,T,P) = (d2g/dTdP)/(dg/dP)_T  thermal expansion of ice air in 1/K,
'wa_si     mass fraction of dry air in ice air in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value with default settings: ice_air_g_expansion_si(0.5, 270, 1E5) = 4.10876052448974E-03  v. 1.0
'check value with default settings: ice_air_g_expansion_si(0.5, 270, 1E5) = 4.10875949031269E-03  v. 1.1

Dim g_p As Double, g_tp As Double

ice_air_g_expansion_si = ErrorReturn

If wa_si <= 0 Or wa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g_p = ice_air_g_si(0, 0, 1, wa_si, t_si, p_si)
If g_p = ErrorReturn Then Exit Function
If g_p <= 0 Then Exit Function
g_tp = ice_air_g_si(0, 1, 1, wa_si, t_si, p_si)
If g_tp = ErrorReturn Then Exit Function

ice_air_g_expansion_si = g_tp / g_p

End Function

'=========================================================================
Public Function ice_air_g_kappa_t_si(ByVal wa_si As Double, _
                                     ByVal t_si As Double, _
                                     ByVal p_si As Double) As Double

'returns   kappa_t(wa,T,P) = - (d2g/dP2)_T/(dg/dP)_T  isothermal compressibility of ice air in 1/Pa,
'wa_si     mass fraction of dry air in ice air in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value with default settings: ice_air_g_kappa_t_si(0.5, 270, 1E5) = 1.00393493828601E-05  v. 1.0
'check value with default settings: ice_air_g_kappa_t_si(0.5, 270, 1E5) = 1.00393460891031E-05  v. 1.1

Dim g_p As Double, g_pp As Double

ice_air_g_kappa_t_si = ErrorReturn

If wa_si <= 0 Or wa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g_p = ice_air_g_si(0, 0, 1, wa_si, t_si, p_si)
If g_p = ErrorReturn Then Exit Function
If g_p <= 0 Then Exit Function
g_pp = ice_air_g_si(0, 0, 2, wa_si, t_si, p_si)
If g_pp = ErrorReturn Then Exit Function

ice_air_g_kappa_t_si = -g_pp / g_p

End Function

'=========================================================================
Public Function ice_air_g_lapserate_si(ByVal wa_si As Double, _
                                       ByVal t_si As Double, _
                                       ByVal p_si As Double) As Double

'returns   gamma(wa,T,P) = -(d2g/dTdP)/(d2g/dT2)_P "moist" adiabatic lapse rate of ice air in K/Pa,
'wa_si     mass fraction of dry air in ice air in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'check value with default settings: ice_air_g_lapserate_si(0.5, 270, 1E5) = 2.28473558871494E-04  v. 1.0
'check value with default settings: ice_air_g_lapserate_si(0.5, 270, 1E5) = 2.28443875628538E-04  v. 1.1

Dim g_tp As Double, g_tt As Double

ice_air_g_lapserate_si = ErrorReturn

If wa_si <= 0 Or wa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g_tp = ice_air_g_si(0, 1, 1, wa_si, t_si, p_si)
If g_tp = ErrorReturn Then Exit Function
If g_tp <= 0 Then Exit Function
g_tt = ice_air_g_si(0, 2, 0, wa_si, t_si, p_si)
If g_tt = ErrorReturn Then Exit Function
If g_tt >= 0 Then Exit Function

ice_air_g_lapserate_si = -g_tp / g_tt

End Function




