Attribute VB_Name = "Ice_1_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules
'     Constants_0_Mdl, file Constants_0.bas
'     Complex_0_Mdl,   file Complex_0.bas
'     Maths_0_Mdl,     file Maths_0.bas

'#########################################################################

'=========================================================================
'This module implements the Gibbs potential of hexagonal ice I and its
'first and second partial derivatives with respect to temperature and
'pressure as defined in IAPWS-06:

'Revised Release on the Equation of State 2006 for H2O Ice Ih
'The International Association for the Properties of Water and Steam
'Doorwerth, The Netherlands, September 2009

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science 2009, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

'Triple point:
Private Const Tt = TP_temperature_si
Private Const Pt = TP_pressure_exp_si

'Reference for sea pressure, Celsius temperature
Private Const P0 = Sealevel_pressure_si
Private Const T0 = Celsius_temperature_si

'auxiliary constants tau, psi
Private Const tau = 273.16
Private Const psi = 611.657

'Gibbs function coefficients
Private gcoeff(4) As Double
Private scoeff(0) As Double
Private rcoeff(2, 2) As CplxType
Private tcoeff(2, 0) As CplxType

'Private Const ErrorReturn = 9.99999999E+98

Private Const Version = "21 Nov 2009"

'==========================================================================
Public Function ice_g_si(ByVal drv_t As Integer, _
                         ByVal drv_p As Integer, _
                         ByVal t_si As Double, _
                         ByVal p_si As Double) As Double

'return value:
'ice_g_si:          derivative of the Gibbs function in the basic SI unit

'input parameters:
'drv_t:             order of the partial temperature derivative
'drv_p:             order of the partial pressure derivative
't_si:              absolute temperature ITS-90 in K
'p_si:              absolute pressure in Pa

'Check values:
'ice_g_si(0,0,270,1E5) = -3786.7496312781
'ice_g_si(1,0,270,1E5) = 1244.97335506441
'ice_g_si(0,1,270,1E5) = 1.09029713623728E-03
'ice_g_si(2,0,270,1E5) = -7.67955356337149
'ice_g_si(1,1,270,1E5) = 1.7260420869651E-07
'ice_g_si(0,2,270,1E5) = -1.27811223643199E-13

ice_g_si = ErrorReturn

InitIAPWS06

If check_limits = 1 Then
  If t_si <= ice_tmin Or t_si > ice_tmax _
  Or p_si <= ice_pmin Or p_si > ice_pmax Then Exit Function
Else
  If t_si <= 0 Or p_si <= 0 Then Exit Function
End If

Select Case drv_p
  Case 0:

    Select Case drv_t
      Case 0: ice_g_si = G_Ice(t_si, p_si)
      Case 1: ice_g_si = dGdT_Ice(t_si, p_si)
      Case 2: ice_g_si = d2GdT2_Ice(t_si, p_si)
      Case Else: Exit Function
    End Select

  Case 1:
    Select Case drv_t
      Case 0: ice_g_si = dGdP_Ice(t_si, p_si)
      Case 1: ice_g_si = d2GdTdP_Ice(t_si, p_si)
      Case 2: ice_g_si = d3GdT2dP_Ice(t_si, p_si)
      Case Else: Exit Function
    End Select

  Case 2:
    Select Case drv_t
      Case 0: ice_g_si = d2GdP2_Ice(t_si, p_si)
      Case 1: ice_g_si = d3GdTdP2_Ice(t_si, p_si)
      Case Else: Exit Function
    End Select

  Case 3: ice_g_si = d3GdP3_Ice(t_si, p_si)
  
  Case Else: Exit Function

End Select

End Function

'==========================================================================
Private Function d2GdP2_Ice(ByVal t As Double, ByVal p As Double) As Double

Dim dr1dp As CplxType
Dim d2r2dp2 As CplxType
Dim dt1dp As CplxType
Dim dt2dp As CplxType
Dim Tc As CplxType
Dim r2 As CplxType
Dim t2 As CplxType
Dim tk As CplxType
Dim vc As CplxType
Dim term_tlnt2 As CplxType

Dim y As Double, z As Double, dz As Double, d2gdz2 As Double

y = t / Tt  'T in K
Tc = Cplx_Num(y)

z = p / Pt  'P in Pa
dz = (p - P0) / Pt

r2 = ddz_rk(2, dz, 0)
t2 = ddz_tk(2, dz, 0)

d2r2dp2 = Cplx_Factor(ddz_rk(2, dz, 2), 1 / psi ^ 2)

tk = Cplx_Diff(t2, Tc)
vc = Cplx_Mult(tk, Cplx_Log(tk))
tk = Cplx_Sum(Tc, t2)
vc = Cplx_Sum(vc, Cplx_Mult(tk, Cplx_Log(tk)))
tk = Cplx_Factor(Cplx_Mult(t2, Cplx_Log(t2)), 2)
vc = Cplx_Diff(vc, tk)

term_tlnt2 = Cplx_Diff(vc, Cplx_Div(Cplx_Num(y ^ 2), t2))
vc = Cplx_Mult(d2r2dp2, term_tlnt2)
d2gdz2 = ddz_poly(gcoeff(), dz, 2) / (psi ^ 2 * tau) + vc.Re

d2GdP2_Ice = tau * d2gdz2

End Function

'==========================================================================
Private Function d2GdT2_Ice(ByVal t As Double, ByVal p As Double) As Double

Dim CPC As CplxType
Dim CP0 As CplxType
Dim Tc As CplxType
Dim r1 As CplxType
Dim r2 As CplxType
Dim t1 As CplxType
Dim t2 As CplxType

Dim y As Double, z As Double, dz As Double

y = t / Tt  'T in K
Tc = Cplx_Num(y)

z = p / Pt  'P in Pa
dz = (p - P0) / Pt

r1 = ddz_rk(1, dz, 0)
t1 = ddz_tk(1, dz, 0)
r2 = ddz_rk(2, dz, 0)
t2 = ddz_tk(2, dz, 0)

CPC = Cplx_Div(r1, Cplx_Diff(Tc, t1))
CPC = Cplx_Diff(CPC, Cplx_Div(r1, Cplx_Sum(Tc, t1)))
CPC = Cplx_Sum(CPC, Cplx_Div(r2, Cplx_Diff(Tc, t2)))
CPC = Cplx_Diff(CPC, Cplx_Div(r2, Cplx_Sum(Tc, t2)))
CP0 = Cplx_Sum(Cplx_Div(r1, t1), Cplx_Div(r2, t2))

d2GdT2_Ice = -(2 * CP0.Re + CPC.Re) / Tt

End Function

'==========================================================================
Private Function d2GdTdP_Ice(ByVal t As Double, ByVal p As Double) As Double

Dim dvdt As CplxType

Dim dr1dp As CplxType
Dim dr2dp As CplxType
Dim dt1dp As CplxType
Dim dt2dp As CplxType
Dim Tc As CplxType
Dim r1 As CplxType
Dim r2 As CplxType
Dim t1 As CplxType
Dim t2 As CplxType
Dim term_lnt1 As CplxType
Dim term_lnt2 As CplxType
Dim term_t1 As CplxType
Dim term_t2 As CplxType
Dim vc As CplxType

Dim y As Double, z As Double, dz As Double

y = t / Tt  'T in K
Tc = Cplx_Num(y)

z = p / Pt  'P in Pa
dz = (p - P0) / Pt

r1 = ddz_rk(1, dz, 0)
t1 = ddz_tk(1, dz, 0)
r2 = ddz_rk(2, dz, 0)
t2 = ddz_tk(2, dz, 0)

dr1dp = Cplx_Factor(ddz_rk(1, dz, 1), 1 / psi) 'dr1/dP
dr2dp = Cplx_Factor(ddz_rk(2, dz, 1), 1 / psi) 'dr2/dP
dt1dp = Cplx_Factor(ddz_tk(1, dz, 1), 1 / psi) 'dt1/dP
dt2dp = Cplx_Factor(ddz_tk(2, dz, 1), 1 / psi) 'dt2/dP

term_lnt1 = Cplx_Diff(Cplx_Log(Cplx_Sum(t1, Tc)), Cplx_Log(Cplx_Diff(t1, Tc)))
term_lnt1 = Cplx_Diff(term_lnt1, Cplx_Factor(Cplx_Div(Tc, t1), 2))

term_lnt2 = Cplx_Diff(Cplx_Log(Cplx_Sum(t2, Tc)), Cplx_Log(Cplx_Diff(t2, Tc)))
term_lnt2 = Cplx_Diff(term_lnt2, Cplx_Factor(Cplx_Div(Tc, t2), 2))

term_t1 = Cplx_Inv(Cplx_Diff(Tc, t1))
term_t1 = Cplx_Sum(term_t1, Cplx_Inv(Cplx_Sum(Tc, t1)))
term_t1 = Cplx_Sum(term_t1, Cplx_Div(Cplx_Num(2 * y), Cplx_Power(t1, 2)))

term_t2 = Cplx_Inv(Cplx_Diff(Tc, t2))
term_t2 = Cplx_Sum(term_t2, Cplx_Inv(Cplx_Sum(Tc, t2)))
term_t2 = Cplx_Sum(term_t2, Cplx_Div(Cplx_Num(2 * y), Cplx_Power(t2, 2)))

vc = Cplx_Mult(dr1dp, term_lnt1)
dvdt = vc
vc = Cplx_Mult(dr2dp, term_lnt2)
dvdt = Cplx_Sum(dvdt, vc)
vc = Cplx_Mult(dt1dp, term_t1)
vc = Cplx_Mult(vc, r1)
dvdt = Cplx_Sum(dvdt, vc)
vc = Cplx_Mult(dt2dp, term_t2)
vc = Cplx_Mult(vc, r2)

d2GdTdP_Ice = Cplx_Sum(dvdt, vc).Re

End Function

'==========================================================================
Private Function d3GdP3_Ice(ByVal t As Double, ByVal p As Double) As Double

Dim d3r2dp3 As CplxType
Dim Tc As CplxType
Dim r2 As CplxType
Dim t2 As CplxType
Dim tk As CplxType
Dim vc As CplxType
Dim term_tlnt2 As CplxType

Dim y As Double, z As Double, dz As Double, d3gdz3 As Double

y = t / Tt  'T in K
Tc = Cplx_Num(y)

z = p / Pt  'P in Pa
dz = (p - P0) / Pt

r2 = ddz_rk(2, dz, 0)
t2 = ddz_tk(2, dz, 0)

d3r2dp3 = Cplx_Factor(ddz_rk(2, dz, 3), 1 / psi ^ 3)

tk = Cplx_Diff(t2, Tc)
vc = Cplx_Mult(tk, Cplx_Log(tk))
tk = Cplx_Sum(Tc, t2)
vc = Cplx_Sum(vc, Cplx_Mult(tk, Cplx_Log(tk)))
tk = Cplx_Factor(Cplx_Mult(t2, Cplx_Log(t2)), 2)
vc = Cplx_Diff(vc, tk)
term_tlnt2 = Cplx_Diff(vc, Cplx_Div(Cplx_Num(y ^ 2), t2))

d3gdz3 = ddz_poly(gcoeff(), dz, 3) / (psi ^ 3 * tau)

vc = Cplx_Mult(d3r2dp3, term_tlnt2)
d3gdz3 = d3gdz3 + vc.Re

d3GdP3_Ice = tau * d3gdz3

End Function

'==========================================================================
Private Function d3GdT2dP_Ice(ByVal t As Double, ByVal p As Double) As Double

Dim CPC As CplxType
Dim CP0 As CplxType
Dim dr2dp As CplxType
Dim Tc As CplxType
Dim t2 As CplxType

Dim y As Double, z As Double, dz As Double

y = t / Tt  'T in K
Tc = Cplx_Num(y)

z = p / Pt  'P in Pa
dz = (p - P0) / Pt

dr2dp = Cplx_Factor(ddz_rk(2, dz, 1), 1 / psi)
t2 = ddz_tk(2, dz, 0)

CPC = Cplx_Div(dr2dp, Cplx_Diff(Tc, t2))
CPC = Cplx_Diff(CPC, Cplx_Div(dr2dp, Cplx_Sum(Tc, t2)))
CP0 = Cplx_Div(dr2dp, t2)

d3GdT2dP_Ice = -(2 * CP0.Re + CPC.Re) / Tt

End Function

'==========================================================================
Private Function d3GdTdP2_Ice(ByVal t As Double, ByVal p As Double) As Double

Dim d2r2dp2 As CplxType
Dim Tc As CplxType
Dim r2 As CplxType
Dim t2 As CplxType
Dim term_lnt2 As CplxType
Dim vc As CplxType

Dim y As Double, z As Double, dz As Double

y = t / Tt  'T in K
Tc = Cplx_Num(y)

z = p / Pt  'P in Pa
dz = (p - P0) / Pt

r2 = ddz_rk(2, dz, 0)
t2 = ddz_tk(2, dz, 0)

d2r2dp2 = Cplx_Factor(ddz_rk(2, dz, 2), 1 / psi ^ 2)

term_lnt2 = Cplx_Diff(Cplx_Log(Cplx_Sum(t2, Tc)), Cplx_Log(Cplx_Diff(t2, Tc)))
term_lnt2 = Cplx_Diff(term_lnt2, Cplx_Factor(Cplx_Div(Tc, t2), 2))
vc = Cplx_Mult(d2r2dp2, term_lnt2)

d3GdTdP2_Ice = vc.Re

End Function

'==========================================================================
Private Function ddz_cpoly(ByRef coeff() As CplxType, _
                           ByVal k As Integer, _
                           ByVal z As Double, _
                           Optional ByVal deriv As Integer = 0) As CplxType

'z-derivative (d/dz)^deriv of complex polynomial, poly = sum{ coeff(k, j)*z^j }

Dim polysum As CplxType
Dim cz As CplxType, zj As CplxType, cc As CplxType
Dim j As Integer, L As Integer

ddz_cpoly = Cplx_Num(0)
polysum = Cplx_Num(0)
cz = Cplx_Num(z)
zj = Cplx_Num(1)

For j = deriv To UBound(coeff, 2)
  cc = Cplx_Mult(coeff(k, j), zj)
  For L = 1 To deriv
    cc = Cplx_Factor(cc, j - L + 1)
  Next L
  polysum = Cplx_Sum(polysum, cc)
  If z = 0 Then Exit For
  If j < UBound(coeff, 2) Then zj = Cplx_Mult(zj, cz)
Next j

ddz_cpoly = polysum
End Function

'==========================================================================
Private Function ddz_poly(ByRef coeff() As Double, _
                          ByVal z As Double, _
                          Optional ByVal deriv As Integer = 0) As Double

'z-derivative (d/dz)^deriv of real polynomial, poly = sum{ coeff(j)*z^j }

Dim polysum As Double
Dim cz As Double, zj As Double, cc As Double
Dim j As Integer, L As Integer

ddz_poly = 0
polysum = 0
cz = z
zj = 1

For j = deriv To UBound(coeff)
  cc = coeff(j) * zj
  For L = 1 To deriv
    cc = cc * CDbl(j - L + 1)
  Next L
  polysum = polysum + cc
  If z = 0 Then Exit For
  If j < UBound(coeff) Then zj = zj * cz
Next j

ddz_poly = polysum
End Function

'==========================================================================
Private Function ddz_rk(ByVal k As Integer, ByVal z As Double, ByVal deriv As Integer) As CplxType

'pressure derivative (d/dp)^deriv of r(k)
'z = (P-P0)/Pt
ddz_rk = ddz_cpoly(rcoeff(), k, z, deriv)

End Function

'==========================================================================
Private Function ddz_tk(ByVal k As Integer, ByVal z As Double, ByVal deriv As Integer) As CplxType

'pressure derivative (d/dp)^deriv of t(k)
'z = (P-P0)/Pt
ddz_tk = ddz_cpoly(tcoeff(), k, z, deriv)

End Function

'==========================================================================
Private Function dGdP_Ice(ByVal t As Double, ByVal p As Double) As Double

Dim dr1dp As CplxType
Dim dr2dp As CplxType
Dim dt1dp As CplxType
Dim dt2dp As CplxType
Dim Tc As CplxType
Dim r1 As CplxType
Dim r2 As CplxType
Dim t1 As CplxType
Dim t2 As CplxType
Dim tk As CplxType
Dim vc As CplxType

Dim term_lnt1 As CplxType
Dim term_lnt2 As CplxType
Dim term_tlnt1 As CplxType
Dim term_tlnt2 As CplxType

Dim y As Double, z As Double, dz As Double, dgdz As Double

y = t / Tt  'T in K
Tc = Cplx_Num(y)

z = p / Pt  'P in Pa
dz = (p - P0) / Pt

r1 = ddz_rk(1, dz, 0)
t1 = ddz_tk(1, dz, 0)
r2 = ddz_rk(2, dz, 0)
t2 = ddz_tk(2, dz, 0)

dr1dp = Cplx_Factor(ddz_rk(1, dz, 1), 1 / psi) 'dr1/dP
dr2dp = Cplx_Factor(ddz_rk(2, dz, 1), 1 / psi) 'dr2/dP
dt1dp = Cplx_Factor(ddz_tk(1, dz, 1), 1 / psi) 'dt1/dP
dt2dp = Cplx_Factor(ddz_tk(2, dz, 1), 1 / psi) 'dt2/dP

tk = Cplx_Diff(t1, Tc)
vc = Cplx_Mult(tk, Cplx_Log(tk))
tk = Cplx_Sum(Tc, t1)
vc = Cplx_Sum(vc, Cplx_Mult(tk, Cplx_Log(tk)))
tk = Cplx_Factor(Cplx_Mult(t1, Cplx_Log(t1)), 2)
vc = Cplx_Diff(vc, tk)
term_tlnt1 = Cplx_Diff(vc, Cplx_Div(Cplx_Num(y ^ 2), t1))

tk = Cplx_Diff(t2, Tc)
vc = Cplx_Mult(tk, Cplx_Log(tk))
tk = Cplx_Sum(Tc, t2)
vc = Cplx_Sum(vc, Cplx_Mult(tk, Cplx_Log(tk)))
tk = Cplx_Factor(Cplx_Mult(t2, Cplx_Log(t2)), 2)
vc = Cplx_Diff(vc, tk)
term_tlnt2 = Cplx_Diff(vc, Cplx_Div(Cplx_Num(y ^ 2), t2))

vc = Cplx_Sum(Cplx_Log(Cplx_Diff(t1, Tc)), Cplx_Log(Cplx_Sum(Tc, t1)))
vc = Cplx_Diff(vc, Cplx_Factor(Cplx_Log(t1), 2))
term_lnt1 = Cplx_Sum(vc, Cplx_Power(Cplx_Div(Tc, t1), 2))

vc = Cplx_Sum(Cplx_Log(Cplx_Diff(t2, Tc)), Cplx_Log(Cplx_Sum(Tc, t2)))
vc = Cplx_Diff(vc, Cplx_Factor(Cplx_Log(t2), 2))
term_lnt2 = Cplx_Sum(vc, Cplx_Power(Cplx_Div(Tc, t2), 2))

dgdz = ddz_poly(gcoeff(), dz, 1) / (psi * tau)  'dg0/dp
vc = Cplx_Mult(dr1dp, term_tlnt1)
dgdz = dgdz + vc.Re

vc = Cplx_Mult(dr2dp, term_tlnt2)
dgdz = dgdz + vc.Re

vc = Cplx_Mult(dt1dp, term_lnt1)
vc = Cplx_Mult(r1, vc)
dgdz = dgdz + vc.Re

vc = Cplx_Mult(dt2dp, term_lnt2)
vc = Cplx_Mult(r2, vc)
dgdz = dgdz + vc.Re

dGdP_Ice = tau * dgdz

End Function

'==========================================================================
Private Function dGdT_Ice(ByVal t As Double, ByVal p As Double) As Double

Dim dsdr1 As CplxType
Dim dsdr2 As CplxType
Dim dsdt1 As CplxType
Dim dsdt2 As CplxType
Dim sc As CplxType
Dim Tc As CplxType
Dim r1 As CplxType
Dim r2 As CplxType
Dim t1 As CplxType
Dim t2 As CplxType

Dim y As Double, z As Double, dz As Double

y = t / Tt  'T in K
Tc = Cplx_Num(y)

z = p / Pt  'P in Pa
dz = (p - P0) / Pt

r1 = ddz_rk(1, dz, 0)
t1 = ddz_tk(1, dz, 0)
r2 = ddz_rk(2, dz, 0)
t2 = ddz_tk(2, dz, 0)

dsdr1 = Cplx_Diff(Cplx_Log(Cplx_Diff(t1, Tc)), Cplx_Log(Cplx_Sum(t1, Tc)))
dsdr1 = Cplx_Sum(dsdr1, Cplx_Div(Cplx_Num(2 * y), t1))
dsdr2 = Cplx_Diff(Cplx_Log(Cplx_Diff(t2, Tc)), Cplx_Log(Cplx_Sum(t2, Tc)))
dsdr2 = Cplx_Sum(dsdr2, Cplx_Div(Cplx_Num(2 * y), t2))
sc = Cplx_Mult(r1, dsdr1)
sc = Cplx_Sum(sc, Cplx_Mult(r2, dsdr2))
dGdT_Ice = -scoeff(0) - sc.Re

End Function

'==========================================================================
Private Function G_Ice(ByVal t As Double, ByVal p As Double) As Double

Dim Tc As CplxType
Dim r1 As CplxType
Dim t1 As CplxType
Dim r2 As CplxType
Dim t2 As CplxType
Dim tk As CplxType
Dim gc As CplxType

Dim term_tlnt1 As CplxType
Dim term_tlnt2 As CplxType

Dim y As Double, z As Double, dz As Double, g As Double

y = t / Tt  'T in K
Tc = Cplx_Num(y)

z = p / Pt  'P in Pa
dz = (p - P0) / Pt

r1 = ddz_rk(1, dz, 0)
t1 = ddz_tk(1, dz, 0)
r2 = ddz_rk(2, dz, 0)
t2 = ddz_tk(2, dz, 0)

tk = Cplx_Diff(t1, Tc)
gc = Cplx_Mult(tk, Cplx_Log(tk))
tk = Cplx_Sum(Tc, t1)
gc = Cplx_Sum(gc, Cplx_Mult(tk, Cplx_Log(tk)))
tk = Cplx_Factor(Cplx_Mult(t1, Cplx_Log(t1)), 2)
gc = Cplx_Diff(gc, tk)
term_tlnt1 = Cplx_Diff(gc, Cplx_Div(Cplx_Num(y ^ 2), t1))

tk = Cplx_Diff(t2, Tc)
gc = Cplx_Mult(tk, Cplx_Log(tk))
tk = Cplx_Sum(Tc, t2)
gc = Cplx_Sum(gc, Cplx_Mult(tk, Cplx_Log(tk)))
tk = Cplx_Factor(Cplx_Mult(t2, Cplx_Log(t2)), 2)
gc = Cplx_Diff(gc, tk)
term_tlnt2 = Cplx_Diff(gc, Cplx_Div(Cplx_Num(y ^ 2), t2))

gc = Cplx_Mult(r1, term_tlnt1)
g = tau * gc.Re
gc = Cplx_Mult(r2, term_tlnt2)
g = g + tau * gc.Re
g = g - scoeff(0) * tau * y

G_Ice = g + ddz_poly(gcoeff(), dz)

End Function

'==========================================================================
Private Sub InitIAPWS06()

If gcoeff(1) = 0.655022213658955 Then Exit Sub

rcoeff(1, 0) = Cplx_Num(44.7050716285388, 65.6876847463481)
rcoeff(1, 1) = Cplx_Num(0, 0)
rcoeff(1, 2) = Cplx_Num(0, 0)
tcoeff(1, 0) = Cplx_Num(3.68017112855051E-02, 5.10878114959572E-02)

rcoeff(2, 0) = Cplx_Num(-72.597457432922, -78.100842711287)
rcoeff(2, 1) = Cplx_Num(-5.57107698030123E-05, 4.64578634580806E-05)
rcoeff(2, 2) = Cplx_Num(2.34801409215913E-11, -2.85651142904972E-11)
tcoeff(2, 0) = Cplx_Num(0.337315741065416, 0.335449415919309)

scoeff(0) = -3327.33756492168

gcoeff(1) = 0.655022213658955
gcoeff(2) = -1.89369929326131E-08
gcoeff(3) = 3.39746123271053E-15
gcoeff(4) = -5.56464869058991E-22

'adjustment to the reference state condition:
'equilibrium with fluid water at the IAPWS-95 triple point

'the original value from the IAPWS-06 Release 2006 is
'gcoeff(0) = -632020.233449497
'the value obtained from a quadruple precision implementation is
gcoeff(0) = -632020.233335886
'as provided in the revised Release of 2009

End Sub

'==========================================================================
Public Function chk_IAPWS06_Table6() As String

'Function values as given in Table 6 of IAPWS-06:
'Revised Release on the Equation of State 2006 for H2O Ice Ih
'The International Association for the Properties of Water and Steam
'Doorwerth, The Netherlands, September 2009

'Properties at the triple point, the normal pressure melting point, and at T = 100 K, P = 100 MPa

'Quantity      Value at             Value at             Value at           Unit
'              T = 273.16 K         T = 273.152519 K     T = 100 K
'              P = 611.657 Pa       P = 101325 Pa        P = 100 MPa

' g            0.611784135xxx        0.10134274069xE+3    0.222296513088E+6   J kg-1
'(dg/dP)_T     0.109085812737E-2     0.109084388214E-2     0.106193389260E-2   m3 kg-1
'(dg/dT)_P     0.122069433940E+4     0.122076932550E+4     0.261195122589E+4   J kg-1 K-1
'(d2g/dP2)_T  -0.128495941571E-12   -0.128485364928E-12   -0.941807981761E-13  m3 kg-1 Pa-1
' d2g/dPdT     0.174387964700E-6     0.174362219972E-6     0.274505162488E-7   m3 kg-1 K-1
'(d2g/dT2)_P  -0.767602985875E+1    -0.767598233365E+1    -0.866333195517E+1   J kg-1 K-2
' h           -0.333444253966E+6    -0.333354873637E+6    -0.483491635676E+6   J kg-1
' f           -0.55446875xxxxE-1    -0.918701567xxxE-1    -0.328489902347E+6   J kg-1
' u           -0.333444921197E+6    -0.333465403393E+6    -0.589685024936E+6   J kg-1
' eta         -0.122069433940E+4    -0.122076932550E+4    -0.261195122589E+4   J kg-1 K-1
' cp           0.209678431622E+4     0.209671391024E+4     0.866333195517E+3   J kg-1 K-1
' rho          0.916709492200E+3     0.916721463419E+3     0.941678203297E+3   kg m-3
' alpha        0.159863102566E-3     0.159841589458E-3     0.258495528207E-4   K-1
' beta         0.135714764659E+7     0.135705899321E+7     0.291466166994E+6   Pa K-1
' kappa_T      0.117793449348E-9     0.117785291765E-9     0.886880048115E-10  Pa-1
' kappa_S      0.114161597779E-9     0.114154442556E-9     0.886060982687E-10  Pa-1


Dim CRLF As String
Dim txt As String, row As String
Dim t As Double, p As Double, q As Double
Dim i As Integer, j As Integer

CRLF = Chr(13) + Chr(10)

InitIAPWS06

txt = " Implementation of IAPWS-06 in Visual Basic" + CRLF
txt = txt + " for Publication in Ocean Science, 2009" + CRLF
txt = txt + " R. Feistel, IOW, Version " + Version + CRLF
txt = txt + " Compiled on " + CStr(Now) + CRLF + CRLF

txt = txt + " Function values as given in Table 6 of IAPWS-06:" + CRLF
txt = txt + " Properties at the triple point, the normal pressure melting point, and at T = 100 K, p = 100 MPa," + CRLF
txt = txt + " usable as numerical check values. The numerical functions evaluated here at given points (T, p)" + CRLF
txt = txt + " are defined in Eq. (1) and Tables 3 and 4, computed with the IAPWS-95 value of s0 from Table 2" + CRLF + CRLF

txt = txt + " Quantity     Value at              Value at              Value at            Unit" + CRLF
txt = txt + "              T = 273.16 K          T = 273.152519 K      T = 100 K" + CRLF
txt = txt + "              P = 611.657 Pa        P = 101325 Pa         P = 100 MPa" + CRLF + CRLF

For i = 1 To 16
  Select Case i
    Case 1:  row = " g            0.611784135xxx        0.10134274069xE+3    -0.222296513088E+6   J kg-1"
    Case 2:  row = "(dg/dP)_T     0.109085812737E-2     0.109084388214E-2     0.106193389260E-2   m3 kg-1"
    Case 3:  row = "(dg/dT)_P     0.122069433940E+4     0.122076932550E+4     0.261195122589E+4   J kg-1 K-1"
    Case 4:  row = "(d2g/dP2)_T  -0.128495941571E-12   -0.128485364928E-12   -0.941807981761E-13  m3 kg-1 Pa-1"
    Case 5:  row = " d2g/dPdT     0.174387964700E-6     0.174362219972E-6     0.274505162488E-7   m3 kg-1 K-1"
    Case 6:  row = "(d2g/dT2)_P  -0.767602985875E+1    -0.767598233365E+1    -0.866333195517E+1   J kg-1 K-2"
    Case 7:  row = " h           -0.333444253966E+6    -0.333354873637E+6    -0.483491635676E+6   J kg-1"
    Case 8:  row = " f           -0.55446875xxxxE-1    -0.918701567xxxE-1    -0.328489902347E+6   J kg-1"
    Case 9:  row = " u           -0.333444921197E+6    -0.333465403393E+6    -0.589685024936E+6   J kg-1"
    Case 10: row = " eta         -0.122069433940E+4    -0.122076932550E+4    -0.261195122589E+4   J kg-1 K-1"
    Case 11: row = " cp           0.209678431622E+4     0.209671391024E+4     0.866333195517E+3   J kg-1 K-1"
    Case 12: row = " rho          0.916709492200E+3     0.916721463419E+3     0.941678203297E+3   kg m-3"
    Case 13: row = " alpha        0.159863102566E-3     0.159841589458E-3     0.258495528207E-4   K-1"
    Case 14: row = " beta         0.135714764659E+7     0.135705899321E+7     0.291466166994E+6   Pa K-1"
    Case 15: row = " kappa_T      0.117793449348E-9     0.117785291765E-9     0.886880048115E-10  Pa-1"
    Case 16: row = " kappa_S      0.114161597779E-9     0.114154442556E-9     0.886060982687E-10  Pa-1"
  End Select

  txt = txt + row + CRLF
  txt = txt + "this code:   "
  
  For j = 1 To 3
    t = Choose(j, 273.16, 273.152519, 100)
    p = Choose(j, 611.657, 101325, 100000000#)
    Select Case i
      Case 1: q = ice_g_si(0, 0, t, p)
      Case 2: q = ice_g_si(0, 1, t, p)
      Case 3: q = ice_g_si(1, 0, t, p)
      Case 4: q = ice_g_si(0, 2, t, p)
      Case 5: q = ice_g_si(1, 1, t, p)
      Case 6: q = ice_g_si(2, 0, t, p)
      Case 7: q = ice_g_si(0, 0, t, p) - t * ice_g_si(1, 0, t, p)
      Case 8: q = ice_g_si(0, 0, t, p) - p * ice_g_si(0, 1, t, p)
      Case 9: q = ice_g_si(0, 0, t, p) - t * ice_g_si(1, 0, t, p) - p * ice_g_si(0, 1, t, p)
      Case 10: q = -ice_g_si(1, 0, t, p)
      Case 11: q = -t * ice_g_si(2, 0, t, p)
      Case 12: q = 1 / ice_g_si(0, 1, t, p)
      Case 13: q = ice_g_si(1, 1, t, p) / ice_g_si(0, 1, t, p)
      Case 14: q = -ice_g_si(1, 1, t, p) / ice_g_si(0, 2, t, p)
      Case 15: q = -ice_g_si(0, 2, t, p) / ice_g_si(0, 1, t, p)
      Case 16: q = (ice_g_si(1, 1, t, p) ^ 2 - ice_g_si(2, 0, t, p) * ice_g_si(0, 2, t, p)) / _
                   (ice_g_si(0, 1, t, p) * ice_g_si(2, 0, t, p))
    End Select
    txt = txt + Left(EFormat(q, 12) + Space(22), 22)
    
  Next j
  txt = txt + CRLF + CRLF
  
Next i

chk_IAPWS06_Table6 = txt

End Function

