Attribute VB_Name = "Air_3c_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules:
'     Constants_0_Mdl, file Constants_0.bas
'     Convert_0_Mdl,   file Convert_0.bas
'     Air_2_Mdl,       file Air_2.bas
'     Air_3a_Mdl,      file Air_3a.bas
'     Air_3b_Mdl,      file Air_3b.bas

'#########################################################################

'=========================================================================
'This module implements the enthalpy of humid air depending on air fraction, entropy
'and pressure, as well as its partial derivatives.

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009

'Feistel, R., Kretzschmar, H.-J., Span, R., Hagen, E., Wright, D.G., Herrmann, S.:
'Thermodynamic Properties of Sea Air.
'Ocean Science Discussion 6(2009)21932325.
'==========================================================================
'Private Const ErrorReturn = 9.99999999E+98

'Control parameters of the temperature iteration
Private ctrl_initialized As Integer

Private ctrl_mode_pottemp As Integer
Private ctrl_loop_maximum As Long
Private ctrl_init_pottemp As Double
Private ctrl_eps_exit_pottemp As Double


Private Const Version = "28 May 2010"

'==========================================================================
Public Function air_h_si(ByVal drv_a As Long, _
                         ByVal drv_eta As Long, _
                         ByVal drv_p As Long, _
                         ByVal a_si As Double, _
                         ByVal eta_si As Double, _
                         ByVal p_si As Double) As Double

'this function implements enthalpy as a thermodynamic potential of humid air, depending
'on the dry-air mass fraction, entropy and pressure

'returns air_h_si as the A-eta-P derivative

'(d/dA)^drv_a (d/deta)^drv_eta (d/dP)^drv_p h(a,eta,P)

'of the specific enthalpy of humid air, h(a,eta,P), in J/kg

'a_si      mass fraction of dry air in kg/kg,
'eta_si    specific entropy in J/(kg K),
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Air_3a
'      and on the iteration settings for temperature by set_it_ctrl_air_pottemp of this module

'Check values with default settings:  v. 1.0
'air_h_si( 0, 0, 0, 0.9, 900, 1E5) = 274589.199191971
'air_h_si( 1, 0, 0, 0.9, 900, 1E5) =-229734.926516548
'air_h_si( 0, 1, 0, 0.9, 900, 1E5) = 297.395859294205
'air_h_si( 0, 0, 1, 0.9, 900, 1E5) = 0.903422384671727
'air_h_si( 2, 0, 0, 0.9, 900, 1E5) = 13959323.1015864
'air_h_si( 1, 1, 0, 0.9, 900, 1E5) = 1676.3382283477
'air_h_si( 1, 0, 1, 0.9, 900, 1E5) =-0.467305898316057
'air_h_si( 0, 2, 0, 0.9, 900, 1E5) = 0.223618143111274
'air_h_si( 0, 1, 1, 0.9, 900, 1E5) = 7.15646251374936E-04
'air_h_si( 0, 0, 2, 0.9, 900, 1E5) =-6.78234488399621E-06

'Check values with default settings:  v. 1.1
'air_h_si( 0, 0, 0, 0.9, 900, 1E5) = 274592.611782659
'air_h_si( 1, 0, 0, 0.9, 900, 1E5) =-229706.905765403
'air_h_si( 0, 1, 0, 0.9, 900, 1E5) = 297.403043058409
'air_h_si( 0, 0, 1, 0.9, 900, 1E5) = 0.903262695635698
'air_h_si( 2, 0, 0, 0.9, 900, 1E5) = 13963273.0103549
'air_h_si( 1, 1, 0, 0.9, 900, 1E5) = 1676.8509855168
'air_h_si( 1, 0, 1, 0.9, 900, 1E5) =-0.467537679038088
'air_h_si( 0, 2, 0, 0.9, 900, 1E5) = 0.223684689765269
'air_h_si( 0, 1, 1, 0.9, 900, 1E5) = 7.15703143992457E-04
'air_h_si( 0, 0, 2, 0.9, 900, 1E5) =-6.78105152859357E-06

Dim t As Double

air_h_si = ErrorReturn

If drv_a < 0 Or drv_a > 2 Then Exit Function
If drv_eta < 0 Or drv_eta > 2 Then Exit Function
If drv_p < 0 Or drv_p > 2 Then Exit Function

If a_si < 0 Or a_si >= 1 Then Exit Function
If p_si < 0 Then Exit Function

'compute temperature from entropy.
'if p_si = in-situ pressure, this is in-situ temperature
'if p_si = reference pressure, this is potential temperature
t = air_temperature_si(a_si, eta_si, p_si)
If t = ErrorReturn Then Exit Function
If t <= 0 Then Exit Function

air_h_si = air_a_eta_p_derivatives_si(drv_a, drv_eta, drv_p, a_si, t, p_si)

End Function

'==========================================================================
Public Function air_pottemp_si(ByVal a_si As Double, _
                               ByVal t_si As Double, _
                               ByVal p_si As Double, _
                               ByVal pr_si As Double) As Double
                               
'this function computes absolute potential temperature of humid air

'returns   273.15 K + theta(A,T,P,Pr) potential temperature of humid air in K,

'a_si      mass fraction of dry air in kg/kg
't_si      absolute in-situ temperature in K,
'p_si      absolute in-situ pressure in Pa
'pr_si     absolute reference pressure in Pa

'Check value with default settings: air_pottemp_si(0.9, 300, 5e4, 1E5) = 363.65437317883  v. 1.0
'Check value with default settings: air_pottemp_si(0.9, 300, 5e4, 1E5) = 363.653905688047  v. 1.1

Dim S As Double

air_pottemp_si = ErrorReturn

If a_si < 0 Or a_si >= 1 Then Exit Function
If t_si < 0 Then Exit Function
If p_si < 0 Then Exit Function
If pr_si < 0 Then Exit Function

If p_si = pr_si Then
  air_pottemp_si = t_si
  Exit Function
End If

S = air_g_entropy_si(a_si, t_si, p_si)  'specific entropy in-situ
If S = ErrorReturn Then Exit Function

air_pottemp_si = air_temperature_si(a_si, S, pr_si)

End Function

'==========================================================================
Public Function air_potdensity_si(ByVal a_si As Double, _
                                  ByVal t_si As Double, _
                                  ByVal p_si As Double, _
                                  ByVal pr_si As Double) As Double
                               
'this function computes potential density of humid air

'returns   rho_theta(A,T,P,Pr) potential density of humid air in kg/m,

'a_si      mass fraction of dry air in kg/kg
't_si      absolute in-situ temperature in K,
'p_si      absolute in-situ pressure in Pa
'pr_si     absolute reference pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Air_3a
'      and on the iteration settings for temperature by set_it_ctrl_air_pottemp of this module

'Check value with default settings: air_potdensity_si(0.9, 300, 5e4, 1E5) = 0.903326577187662  v. 1.0
'Check value with default settings: air_potdensity_si(0.9, 300, 5e4, 1E5) = 0.903509489711487  v. 1.1

Dim S As Double, v As Double

air_potdensity_si = ErrorReturn

If a_si < 0 Or a_si >= 1 Then Exit Function
If t_si < 0 Then Exit Function
If p_si < 0 Then Exit Function
If pr_si < 0 Then Exit Function

S = air_g_entropy_si(a_si, t_si, p_si)  'specific entropy in-situ
If S = ErrorReturn Then Exit Function

v = air_h_si(0, 0, 1, a_si, S, pr_si) 'specific volume at reference pressure
If v = ErrorReturn Then Exit Function
If v <= 0 Then Exit Function

air_potdensity_si = 1 / v

End Function

'==========================================================================
Public Function air_potenthalpy_si(ByVal a_si As Double, _
                                   ByVal t_si As Double, _
                                   ByVal p_si As Double, _
                                   ByVal pr_si As Double) As Double
                               
'this function computes potential enthalpy of humid air

'returns   h_theta(A,T,P,Pr) potential enthalpy of humid air in J/kg,

'a_si      mass fraction of dry air in kg/kg
't_si      absolute in-situ temperature in K,
'p_si      absolute in-situ pressure in Pa
'pr_si     absolute reference pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Air_3a
'      and on the iteration settings for temperature by set_it_ctrl_air_pottemp of this module

'Check value with default settings: air_potenthalpy_si(0.9, 300, 5e4, 1E5) = 348892.581996519  v. 1.0
'Check value with default settings: air_potenthalpy_si(0.9, 300, 5e4, 1E5) = 348872.568665216  v. 1.1

Dim S As Double

air_potenthalpy_si = ErrorReturn

If a_si < 0 Or a_si >= 1 Then Exit Function
If t_si < 0 Then Exit Function
If p_si < 0 Then Exit Function
If pr_si < 0 Then Exit Function

S = air_g_entropy_si(a_si, t_si, p_si)  'specific entropy in-situ
If S = ErrorReturn Then Exit Function

air_potenthalpy_si = air_h_si(0, 0, 0, a_si, S, pr_si)

End Function

'=========================================================================
Public Function air_temperature_si(ByVal a_si As Double, _
                                   ByVal eta_si As Double, _
                                   ByVal p_si As Double) As Double
                                 
'returns   t(A,eta,P) (potential) temperature of humid air in K,

'a_si      mass fraction of dry air in kg/kg
'eta_si    entropy in J/(kg K),
'p_si      absolute (reference) pressure in Pa

'this is the inverse function to air_g_entropy_si(a_si, t_si, p_si) in air_3a

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Air_3a and on those made in this module

'Check value with default settings: air_temperature_si(0.9, 900, 1E5) = 297.395859294205  v. 1.0
'Check value with default settings: air_temperature_si(0.9, 900, 1E5) = 297.403043058409  v. 1.1

Dim t As Double, eps As Double, maxit As Long

air_temperature_si = ErrorReturn

init_it_ctrl_pottemp

Select Case ctrl_mode_pottemp
  Case 0:    t = aux_temperature_si(a_si, eta_si, p_si)  'ideal-gas approximation
  Case Else: t = ctrl_init_pottemp
End Select

Select Case ctrl_loop_maximum
  Case 0:      maxit = 100
  Case -1:     air_temperature_si = t
               Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else:   Exit Function
End Select

eps = ctrl_eps_exit_pottemp
If eps <= 0 Then Exit Function

'run the iteration
air_temperature_si = pottemp_iteration(a_si, eta_si, p_si, t, maxit, eps)

End Function

'==========================================================================
Private Function air_a_eta_p_derivatives_si(ByVal drv_a As Integer, _
                                            ByVal drv_eta As Integer, _
                                            ByVal drv_p As Integer, _
                                            ByVal a_si As Double, _
                                            ByVal t_si As Double, _
                                            ByVal p_si As Double) As Double
                                              
'this function computes humid air a-eta-p derivatives of h from a-t-p derivatives of g

Dim g As Double, gt As Double, gp As Double, ga As Double
Dim gaa As Double, gat As Double, gap As Double
Dim gtt As Double, gpp As Double, gtp As Double
Dim h As Double

'in one case we do not at all need to compute the Gibbs function:
If drv_a = 0 And drv_eta = 1 And drv_p = 0 Then
  air_a_eta_p_derivatives_si = t_si
  Exit Function
End If

air_a_eta_p_derivatives_si = ErrorReturn

Select Case drv_a

  Case 0:
    Select Case drv_eta
    
      Case 0:
        Select Case drv_p
        
          Case 0: g = air_g_si(0, 0, 0, a_si, t_si, p_si)
                  If g = ErrorReturn Then Exit Function
                  gt = air_g_si(0, 1, 0, a_si, t_si, p_si)
                  If gt = ErrorReturn Then Exit Function
                  h = g - t_si * gt                          'h
          
          Case 1: gp = air_g_si(0, 0, 1, a_si, t_si, p_si)
                  If gp = ErrorReturn Then Exit Function
                  h = gp                                      'dh/dp
                  
          Case 2: gtt = air_g_si(0, 2, 0, a_si, t_si, p_si)
                  If gtt = ErrorReturn Then Exit Function
                  If gtt = 0 Then Exit Function
                  gtp = air_g_si(0, 1, 1, a_si, t_si, p_si)
                  If gtp = ErrorReturn Then Exit Function
                  gpp = air_g_si(0, 0, 2, a_si, t_si, p_si)
                  If gpp = ErrorReturn Then Exit Function
                  h = (gtt * gpp - gtp ^ 2) / gtt              'd2h/dp2
                  
          Case Else: Exit Function
        End Select
        
      Case 1:
        Select Case drv_p
          Case 0: h = t_si                                     'dh/deta, has already been handled initially
          
          Case 1: gtt = air_g_si(0, 2, 0, a_si, t_si, p_si)
                  If gtt = ErrorReturn Then Exit Function
                  If gtt = 0 Then Exit Function
                  gtp = air_g_si(0, 1, 1, a_si, t_si, p_si)
                  If gtp = ErrorReturn Then Exit Function
                  h = -gtp / gtt                               'd2h/detadp
                  
          Case Else: Exit Function
        End Select

      Case 2:
        Select Case drv_p
          Case 0: gtt = air_g_si(0, 2, 0, a_si, t_si, p_si)
                  If gtt = ErrorReturn Then Exit Function
                  If gtt = 0 Then Exit Function
                  h = -1# / gtt                                'd2h/deta2
                  
          Case Else: Exit Function
        End Select

      Case Else: Exit Function
    End Select
    
  Case 1:
    Select Case drv_eta
    
      Case 0:
        Select Case drv_p
        
          Case 0: ga = air_g_si(1, 0, 0, a_si, t_si, p_si)
                  If ga = ErrorReturn Then Exit Function
                  h = ga                                        'dh/da
          
          Case 1: gtt = air_g_si(0, 2, 0, a_si, t_si, p_si)
                  If gtt = ErrorReturn Then Exit Function
                  If gtt = 0 Then Exit Function
                  gat = air_g_si(1, 1, 0, a_si, t_si, p_si)
                  If gat = ErrorReturn Then Exit Function
                  gap = air_g_si(1, 0, 1, a_si, t_si, p_si)
                  If gap = ErrorReturn Then Exit Function
                  h = (gtt * gap - gat * gtp) / gtt             'd2h/dadp
          
          Case Else: Exit Function
        End Select

      Case 1:
        Select Case drv_p
        
          Case 0: gtt = air_g_si(0, 2, 0, a_si, t_si, p_si)
                  If gtt = ErrorReturn Then Exit Function
                  If gtt = 0 Then Exit Function
                  gat = air_g_si(1, 1, 0, a_si, t_si, p_si)
                  If gat = ErrorReturn Then Exit Function
                  h = -gat / gtt                                'd2h/dadeta
          
          Case Else: Exit Function
        End Select

      Case Else: Exit Function
    End Select

  
  Case 2:
    Select Case drv_eta
    
      Case 0:
        Select Case drv_p
        
          Case 0: gtt = air_g_si(0, 2, 0, a_si, t_si, p_si)
                  If gtt = ErrorReturn Then Exit Function
                  If gtt = 0 Then Exit Function
                  gat = air_g_si(1, 1, 0, a_si, t_si, p_si)
                  If gat = ErrorReturn Then Exit Function
                  gaa = air_g_si(2, 0, 0, a_si, t_si, p_si)
                  If gaa = ErrorReturn Then Exit Function
                  h = (gtt * gaa - gat ^ 2) / gtt               'd2h/da2
          
          Case Else: Exit Function
        End Select

      Case Else: Exit Function
    End Select

  Case Else: Exit Function
End Select

air_a_eta_p_derivatives_si = h

End Function

'==========================================================================
Private Sub init_it_ctrl_pottemp()

If ctrl_initialized = -1 Then Exit Sub

ctrl_initialized = -1

'Set default values and modes for density iteration
ctrl_loop_maximum = 100
ctrl_mode_pottemp = 0           'default: theta = ideal-gas approximation
ctrl_init_pottemp = 273.15
ctrl_eps_exit_pottemp = 0.0001  'default = 0.1 mK

End Sub

'=========================================================================
Private Function pottemp_iteration(ByVal a_si As Double, _
                                   ByVal eta_si As Double, _
                                   ByVal p_si As Double, _
                                   ByVal t_si As Double, _
                                   ByVal maxit As Long, _
                                   ByVal eps As Double) As Double

'returns   theta =  potential temperature of humid air in K,
'          i.e. the temperature that solves eta_si = air_g_entropy_si(a_si, t_si, p_si)

'a_si      mass fraction of dry air in kg/kg
'eta_si    entropy in J/(kg K)
'p_si      absolute (reference) pressure in Pa
't_si      absolute (potential) temperature in K, initial value
'maxit     max. number of iterations
'eps       required tolerance in K

Dim i As Long
Dim S As Double, theta As Double, cp As Double, dt As Double

pottemp_iteration = ErrorReturn

If a_si < 0 Or a_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function
If eps <= 0 Then Exit Function
If maxit <= 0 Then Exit Function

check_limits = check_limits - 1

theta = t_si
For i = 1 To maxit

  'get entropy and its first derivative for Newton iteration
  S = air_g_entropy_si(a_si, theta, p_si)
  cp = air_g_cp_si(a_si, theta, p_si)

  If S = ErrorReturn Then Exit For
  If cp = ErrorReturn Then Exit For
  If cp <= 0 Then Exit For
  
  'next temperature improvement step
  dt = theta * (eta_si - S) / cp
  theta = theta + dt
  If theta <= 0 Then Exit For
  
  If Abs(dt) < eps Then
    pottemp_iteration = theta
    Exit For
  End If
  
Next i

check_limits = check_limits + 1
If check_limits = 1 Then
  If theta < dry_air_tmin Or theta > dry_air_tmax Then
    pottemp_iteration = ErrorReturn
  End If
  If a_si <> 1 And _
    (theta < mix_air_tmin Or theta > mix_air_tmax) Then
    pottemp_iteration = ErrorReturn
  End If
End If

End Function

'==========================================================================
Public Sub set_it_ctrl_air_pottemp(ByVal key As String, ByVal value As Double)

'this sub sets control parameters for the Newton iteration used to compute
'potential temperature from reference pressure

'key             value
'it_steps        0           set iteration number to default (100)
'it_steps        n > 0       set iteration number to n
'it_steps       -1           do not iterate, use initial value

'init_theta      0           use default potential temperature theta = ideal-gas approximation
'init_theta      t > 0       use value t as potential temperature to start

'tol_theta       0           use default exit accuracy for potential temperature (0.1 mK)
'tol_theta       eps > 0     use eps as exit accuracy for potential temperature

init_it_ctrl_pottemp

Select Case LCase(Trim(key))

  Case "it_steps":   'iteration steps
    Select Case value
      Case 0:      ctrl_loop_maximum = 100     'default = 100
      Case Is < 0: ctrl_loop_maximum = -1
      Case Else:   ctrl_loop_maximum = value
    End Select

  Case "init_theta":   'start theta
    Select Case CLng(value)
      Case 0:      ctrl_mode_pottemp = 0        'default: ideal-gas approximation
      Case Is > 0: ctrl_mode_pottemp = 1
                   ctrl_init_pottemp = value
    End Select

  Case "tol_theta":      'required theta tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_pottemp = 0.0001 'default = 0.1 mK
      Case Is > 0: ctrl_eps_exit_pottemp = value
    End Select

End Select

End Sub

'==========================================================================
Private Function aux_temperature_si(ByVal a_si As Double, _
                                    ByVal eta_si As Double, _
                                    ByVal p_si As Double) As Double
                                    
'returns   t(A,eta,P) ideal-gas approx. of temperature of humid air in K,

'a_si      mass fraction of dry air in kg/kg
'eta_si    entropy in J/(kg K),
'p_si      absolute (reference) pressure in Pa

'this is an inverse function estimate to air_g_entropy_si(a_si, t_si, p_si) in air_3a

Const Tt = TP_temperature_si
Const Pt = TP_pressure_IAPWS95_si

Const ra = Gas_constant_air_si ' R / ma
Const RW = Gas_constant_H2O_si ' R / mw

'properties at the triple point of water
Const cpa = 1003.69  'heat capacity of dry air, in J/(kg K)
Const cpv = 1884.352 'heat capacity of vapour, in J/(kg K)

Dim t As Double, etat As Double, rav As Double, xv As Double
Dim numer As Double, denom As Double

aux_temperature_si = ErrorReturn
If a_si < 0 Or a_si > 1 Then Exit Function
If p_si <= 0 Then Exit Function

If a_si = ErrorReturn Then Exit Function
If eta_si = ErrorReturn Then Exit Function
If p_si = ErrorReturn Then Exit Function

etat = air_g_entropy_si(a_si, Tt, Pt)  'entropy of humid air at the triple point
If etat = ErrorReturn Then Exit Function
rav = a_si * ra + (1 - a_si) * RW  'gas constant of humid air
xv = air_molfraction_vap_si(a_si)
If xv = ErrorReturn Then Exit Function
If xv < 0 Or xv > 1 Then Exit Function

denom = a_si * (cpa + ra) + (1 - a_si) * (cpv + RW)
If denom = 0 Then Exit Function

numer = eta_si - etat + rav * Log(p_si / Pt)
aux_temperature_si = Tt * Exp(numer / denom)

End Function




