Attribute VB_Name = "Sea_Vap_4_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules:
'     Constants_0_Mdl,  file Constants_0.bas
'     Flu_1_Mdl,        file Flu_1.bas
'     Sal_1_Mdl,        file Sal_1.bas
'     Sal_2_Mdl,        file Sal_2.bas
'     Flu_3a_Mdl,       file Flu_3a.bas
'     Flu_3b_Mdl,       file Flu_3b.bas
'     Sea_3a_Mdl,       file Sea_3a.bas
'     Maths_0_Mdl,      file Maths_0.bas

'#########################################################################

'=========================================================================
'This module implements the phase equilibrium properties of vapour with seawater
'computed from IAPWS-95 and IAPWS-08:

'Release on the IAPWS Formulation 1995 for the Thermodynamic Properties of
'Ordinary Water Substance for General and Scientific Use
'The International Association for the Properties of Water and Steam
'Fredericia, Denmark, September 1996

'Release on the IAPWS Formulation 2008 for the Thermodynamic Properties of Seawater
'The International Association for the Properties of Water and Steam
'Berlin, Germany, September 2008

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98
'Private Const IsOK = -1

'Control parameters of the seawater-vapour iteration
Private ctrl_initialized As Integer

Private ctrl_mode_sea As Integer
Private ctrl_mode_temperature As Integer
Private ctrl_mode_liquid As Integer
Private ctrl_mode_vapour As Integer

Private ctrl_loop_maximum As Long

Private ctrl_init_s_sea As Double    'initial brine salinity
Private ctrl_init_t As Double        'initial temperature
Private ctrl_init_d_liq As Double    'initial liquid density
Private ctrl_init_d_vap As Double    'initial vapour density

Private ctrl_eps_exit_s As Double    'salinity exit tolerance
Private ctrl_eps_exit_t As Double    'temperature exit tolerance
Private ctrl_eps_exit_p As Double    'pressure exit tolerance

'Properties of the current seawater-vapour equilibrium state
Private equi_sea_vap_done As Integer  'flag = IsOK if equilibrium was successfully computed

Private equi_sea_vap_t As Double      'equilibrium temperature
Private equi_sea_vap_p As Double      'equilibrium pressure
Private equi_sea_vap_s As Double      'equilibrium brine salinity
Private equi_sea_vap_d_liq As Double  'equilibrium density of liquid water
Private equi_sea_vap_d_vap As Double  'equilibrium density of water vapour

'IF97 coefficients from:
'Revised Release on the IAPWS Industrial Formulation 1997
'for the Thermodynamic Properties of Water and Steam
'The International Association for the Properties of Water and Steam
'Lucerne, Switzerland, August 2007

'Coefficients of region 1
Private i1i(34) As Integer
Private j1i(34) As Integer
Private n1i(34) As Double

'Coefficients of region 2, ideal part
Private j0i(9) As Integer
Private n0i(9) As Double

'Coefficients of region 2, residual part
Private iri(43) As Integer
Private jri(43) As Integer
Private nri(43) As Double

Private Const Version = "16 Sep 2009"

'==========================================================================
Public Function set_sea_vap_eq_at_s_p(ByVal sa_si As Double, _
                                               ByVal p_si As Double) As Double

'this function computes the seawater-vapour equilibrium at given salinity and pressure
'by Newton iteration

Const Tt = TP_temperature_si
Const dlt = TP_density_liq_IAPWS95_si   'triple point density of liquid water
Const dvt = TP_density_vap_IAPWS95_si   'triple point density of water vapour

Dim t As Double, d_liq As Double, d_vap As Double, eps As Double
Dim maxit As Long

If equi_sea_vap_done = IsOK And _
   sa_si = equi_sea_vap_s And p_si = equi_sea_vap_p Then
  'the requested state has already been computed earlier
  set_sea_vap_eq_at_s_p = IsOK
  Exit Function
End If

clear_sea_vap_state 'waste any previous state

set_sea_vap_eq_at_s_p = ErrorReturn

If sa_si <= 0 Then Exit Function
If sa_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function

init_it_ctrl_sea_vap

'set initial temperature guess
Select Case ctrl_mode_temperature
  Case 0:  t = aux_boilingtemperature_si(sa_si, p_si)
  Case -1: t = Tt
  Case 1:  t = ctrl_init_t
  Case Else: Exit Function
End Select
If t <= 0 Then Exit Function
If t = ErrorReturn Then Exit Function

'set initial liquid density guess
Select Case ctrl_mode_liquid
  Case 0:  d_liq = aux_liq_density_if97_si(t, p_si)
  Case -1: d_liq = dlt
  Case 1:  d_liq = ctrl_init_d_liq
  Case Else: Exit Function
End Select
If d_liq <= 0 Then Exit Function
If d_liq = ErrorReturn Then Exit Function

'set initial vapour density guess
Select Case ctrl_mode_vapour
  Case 0:  d_vap = aux_density_ideal_si(t, p_si)
  Case -1: d_vap = dvt
  Case 1:  d_vap = ctrl_init_d_vap
  Case Else: Exit Function
End Select
If d_vap <= 0 Then Exit Function
If d_vap = ErrorReturn Then Exit Function

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: set_sea_vap_state sa_si, t, p_si, d_liq, d_vap
           set_sea_vap_eq_at_s_p = IsOK
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit
eps = ctrl_eps_exit_t
If eps = 0 Then Exit Function

'run iteration loop
If sea_vap_iteration_at_s_p(sa_si, p_si, maxit, eps, d_liq, d_vap, t) = ErrorReturn Then
  Exit Function
End If

set_sea_vap_state sa_si, t, p_si, d_liq, d_vap

set_sea_vap_eq_at_s_p = IsOK

End Function

'==========================================================================
Public Function set_sea_vap_eq_at_s_t(ByVal sa_si As Double, _
                                               ByVal t_si As Double) As Double

'this function computes the seawater-vapour equilibrium at given temperature and salinity
'by Newton iteration

Const dlt = TP_density_liq_IAPWS95_si   'triple point density of liquid water
Const dvt = TP_density_vap_IAPWS95_si   'triple point density of water vapour

Dim eps As Double
Dim d_liq As Double, d_vap As Double, p_si As Double
Dim maxit As Long

If equi_sea_vap_done = IsOK And _
   sa_si = equi_sea_vap_s And t_si = equi_sea_vap_t Then
  'the requested state has already been computed earlier
  set_sea_vap_eq_at_s_t = IsOK
  Exit Function
End If

clear_sea_vap_state 'waste any previous state

set_sea_vap_eq_at_s_t = ErrorReturn

If sa_si <= 0 Then Exit Function
If sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function

init_it_ctrl_sea_vap

If ctrl_mode_liquid = 0 Or ctrl_mode_vapour = 0 Then
  p_si = aux_vapourpressure_WP80_si(sa_si, t_si)
End If

'set initial liquid density guess
Select Case ctrl_mode_liquid
  Case 0:  d_liq = aux_liq_density_if97_si(t_si, p_si)
  Case -1: d_liq = dlt
  Case 1:  d_liq = ctrl_init_d_liq
  Case Else: Exit Function
End Select
If d_liq <= 0 Then Exit Function
If d_liq = ErrorReturn Then Exit Function

'set initial vapour density guess
Select Case ctrl_mode_vapour
  Case 0:  d_vap = aux_density_ideal_si(t_si, p_si)
  Case -1: d_vap = dvt
  Case 1:  d_vap = ctrl_init_d_vap
  Case Else: Exit Function
End Select
If d_vap <= 0 Then Exit Function
If d_vap = ErrorReturn Then Exit Function

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: p_si = d_vap ^ 2 * flu_f_si(0, 1, t_si, d_vap)
           set_sea_vap_state sa_si, t_si, p_si, d_liq, d_vap
           set_sea_vap_eq_at_s_t = IsOK
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit
eps = ctrl_eps_exit_p
If eps = 0 Then Exit Function

'run iteration loop
If sea_vap_iteration_at_s_t(sa_si, t_si, maxit, eps, d_liq, d_vap, p_si) = ErrorReturn Then
  Exit Function
End If

set_sea_vap_state sa_si, t_si, p_si, d_liq, d_vap

set_sea_vap_eq_at_s_t = IsOK

End Function

'==========================================================================
Public Function set_sea_vap_eq_at_t_p(ByVal t_si As Double, _
                                               ByVal p_si As Double) As Double

'this function computes the seawater-ice equilibrium at given absolute temperature
't_si in K and absolute pressure p_si in Pa by Newton iteration

Const s_norm = SO_salinity_si  'kg/kg of KCl normalised seawater

Dim d_liq As Double, d_vap As Double
Dim S As Double, eps As Double
Dim maxit As Long

If equi_sea_vap_done = IsOK And _
   t_si = equi_sea_vap_t And p_si = equi_sea_vap_p Then
  'the requested state has already been computed earlier
  set_sea_vap_eq_at_t_p = IsOK
  Exit Function
End If

clear_sea_vap_state 'waste any previous state

set_sea_vap_eq_at_t_p = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

init_it_ctrl_sea_vap

'set initial brine salinity guess
Select Case ctrl_mode_sea
  Case 0:  S = aux_brinesalinity_si(t_si, p_si)
  Case -1: S = s_norm
  Case 1:  S = ctrl_init_s_sea
  Case Else: Exit Function
End Select

If S <= 0 Or S >= 1 Then Exit Function
If S = ErrorReturn Then Exit Function

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: d_liq = liq_density_si(t_si, p_si)
           d_vap = vap_density_si(t_si, p_si)
           set_sea_vap_state S, t_si, p_si, d_liq, d_vap
           set_sea_vap_eq_at_t_p = IsOK
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit
eps = ctrl_eps_exit_s
If eps = 0 Then Exit Function

'run iteration loop
If sea_vap_iteration_at_t_p(t_si, p_si, maxit, eps, S) = ErrorReturn Then
  Exit Function
End If

d_liq = liq_density_si(t_si, p_si) 'only for completeness of the state descriptor
d_vap = vap_density_si(t_si, p_si)
set_sea_vap_state S, t_si, p_si, d_liq, d_vap

set_sea_vap_eq_at_t_p = IsOK

End Function

'==========================================================================
Public Function sea_vap_boilingtemperature_si(ByVal sa_si As Double, _
                                              ByVal p_si As Double) As Double

'this function returns the absolute temperature in K at which seawater with
'absolute salinity sa_si in kg/kg boils at the absolute pressure p_si in Pa

'note: the accuracy of this function depends on the iteration settings of this module

'check value with default settings: sea_vap_boilingtemperature_si(0.035, 640) = 274.042416829483

sea_vap_boilingtemperature_si = ErrorReturn

If set_sea_vap_eq_at_s_p(sa_si, p_si) = ErrorReturn Then Exit Function

sea_vap_boilingtemperature_si = equi_sea_vap_t

End Function

'==========================================================================
Public Function sea_vap_brinesalinity_si(ByVal t_si As Double, _
                                         ByVal p_si As Double) As Double

'this function returns the absolute salinity sa_si in kg/kg with which seawater boils at the
'absolute temperature in K and the absolute pressure p_si in Pa

'note: the accuracy of this function depends on the iteration settings of this module
'and of those in Flu_3a

'check value with default settings: sea_vap_brinesalinity_si(274, 640) = 2.9439629829|3595E-02

Dim S As Double

sea_vap_brinesalinity_si = ErrorReturn

If set_sea_vap_eq_at_t_p(t_si, p_si) = ErrorReturn Then Exit Function

sea_vap_brinesalinity_si = equi_sea_vap_s

End Function

'=========================================================================
Public Function sea_vap_brinefraction_seavap_si(ByVal ssv_si As Double, _
                                                ByVal t_si As Double, _
                                                ByVal p_si As Double) As Double

'returns   w(Ssv,T,P) = Ssv/S(T,P)  mass fraction of brine in sea vapour (0 < w < 1)
'ssv_si    absolute sea-vapour salinity in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a and on those of this module

'check value with default settings:
'sea_vap_brinefraction_seavap_si(0.0035,274,640) = 0.118887364422|956

Dim w As Double, S As Double

sea_vap_brinefraction_seavap_si = ErrorReturn

If ssv_si < 0 Or ssv_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

S = sea_vap_brinesalinity_si(t_si, p_si)
If S = ErrorReturn Then Exit Function
If S <= ssv_si Then Exit Function

sea_vap_brinefraction_seavap_si = ssv_si / S

End Function

'==========================================================================
Public Function sea_vap_vapourpressure_si(ByVal sa_si As Double, _
                                          ByVal t_si As Double) As Double

'this function returns the absolute vapour pressure in Pa at the
'absolute temperature t_si in K and the ambient seawater salinity sa_si in kg/kg

'note: the accuracy of this function depends on the iteration settings of this module

'check value with default settings: sea_vap_vapourpressure_si(0.035, 274) = 638.044692614776

sea_vap_vapourpressure_si = ErrorReturn

If set_sea_vap_eq_at_s_t(sa_si, t_si) = ErrorReturn Then Exit Function

sea_vap_vapourpressure_si = equi_sea_vap_p

End Function

'==========================================================================
Public Function sea_vap_density_sea_si() As Double

'this function returns the density in kg/m3 of seawater at equilibrium with vapour,
'set by a previous call of either set_sea_vap_eq_at_s_p,
'set_sea_vap_eq_at_s_t or set_sea_vap_eq_at_t_p

'Note: the accuracy of this function depends on the iteration settings in this module,
'and, in the case of set_sea_vap_eq_at_t_p, of those in Flu_3a

'check values with default settings:
'set_sea_vap_eq_at_s_p 0.035, 640
'sea_vap_density_sea_si = 1027.87349555845

'set_sea_vap_eq_at_s_t 0.035, 274
'sea_vap_density_sea_si = 1027.87626132072

'set_sea_vap_eq_at_t_p 274, 640
'sea_vap_density_sea_si = 1023.42713046839

sea_vap_density_sea_si = ErrorReturn

If equi_sea_vap_done <> IsOK Then Exit Function

sea_vap_density_sea_si = sea_density_si(equi_sea_vap_s, equi_sea_vap_t, equi_sea_vap_p)

End Function

'==========================================================================
Public Function sea_vap_density_vap_si() As Double

'this function returns the density in kg/m3 of vapour at equilibrium with seawater,
'set by a previous call of either set_sea_vap_eq_at_s_p,
'set_sea_vap_eq_at_s_t or set_sea_vap_eq_at_t_p

'Note: the accuracy of this function depends on the iteration settings in this module
'and, in the case of set_sea_vap_eq_at_t_p, of those in Flu_3a

'check values with default settings:
'set_sea_vap_eq_at_s_p 0.035, 640
'sea_vap_density_vap_si = 5.06324890263994E-03

'set_sea_vap_eq_at_s_t 0.035, 274
'sea_vap_density_vap_si = 5.04855547810571E-03

'set_sea_vap_eq_at_t_p 274, 640
'sea_vap_density_vap_si = 5.06403699512862E-03

sea_vap_density_vap_si = ErrorReturn

If equi_sea_vap_done <> IsOK Then Exit Function

sea_vap_density_vap_si = vap_density_si(equi_sea_vap_t, equi_sea_vap_p)

End Function

'==========================================================================
Public Function sea_vap_enthalpy_sea_si() As Double

'this function returns the enthalpy in J/kg of seawater at equilibrium with vapour,
'set by a previous call of either set_sea_vap_eq_at_s_p,
'set_sea_vap_eq_at_s_t or set_sea_vap_eq_at_t_p

'Note: the accuracy of this function depends on the iteration settings in this module
'and, in the case of set_sea_vap_eq_at_t_p, of those in Flu_3a

'check values with default settings:
'set_sea_vap_eq_at_s_p 0.035, 640
'sea_vap_enthalpy_sea_si = 3465.1189614|4269

'set_sea_vap_eq_at_s_t 0.035, 274
'sea_vap_enthalpy_sea_si = 3295.9662929|8726

'set_sea_vap_eq_at_t_p 274, 640
'sea_vap_enthalpy_sea_si = 3405.9335372|9608

sea_vap_enthalpy_sea_si = ErrorReturn

If equi_sea_vap_done <> IsOK Then Exit Function

sea_vap_enthalpy_sea_si = sea_enthalpy_si(equi_sea_vap_s, equi_sea_vap_t, equi_sea_vap_p)

End Function

'==========================================================================
Public Function sea_vap_enthalpy_vap_si() As Double

'this function returns the enthalpy in J/kg of vapour at equilibrium with seawater,
'set by a previous call of either set_sea_vap_eq_at_s_p,
'set_sea_vap_eq_at_s_t or set_sea_vap_eq_at_t_p

'Note: the accuracy of this function depends on the iteration settings in this module
'and, in the case of set_sea_vap_eq_at_t_p, of those in Flu_3a

'check values with default settings:
'set_sea_vap_eq_at_s_p 0.035, 640
'sea_vap_enthalpy_vap_si = 2502546.89357828

'set_sea_vap_eq_at_s_t 0.035, 274
'sea_vap_enthalpy_vap_si = 2502469.07186831

'set_sea_vap_eq_at_t_p 274, 640
'sea_vap_enthalpy_vap_si = 2502466.96632615

sea_vap_enthalpy_vap_si = ErrorReturn

If equi_sea_vap_done <> IsOK Then Exit Function

sea_vap_enthalpy_vap_si = vap_enthalpy_si(equi_sea_vap_t, equi_sea_vap_p)

End Function

'==========================================================================
Public Function sea_vap_enthalpy_evap_si() As Double

'this function returns the evaporation enthalpy in J/kg of seawater,
'set by a previous call of either set_sea_vap_eq_at_s_p,
'set_sea_vap_eq_at_s_t or set_sea_vap_eq_at_t_p

'Note: the accuracy of this function depends on the iteration settings in this module
'and, in the case of set_sea_vap_eq_at_t_p, of those in Flu_3a

'check values with default settings:
'set_sea_vap_eq_at_s_p 0.035, 640
'sea_vap_enthalpy_evap_si = 2498295.3218674

'set_sea_vap_eq_at_s_t 0.035, 274
'sea_vap_enthalpy_evap_si = 2498395.40100976

'set_sea_vap_eq_at_t_p 274, 640
'sea_vap_enthalpy_evap_si = 2498551.1987509

Dim S As Double, t As Double, p As Double, d As Double

sea_vap_enthalpy_evap_si = ErrorReturn

If equi_sea_vap_done <> IsOK Then Exit Function

S = equi_sea_vap_s
t = equi_sea_vap_t
p = equi_sea_vap_p

d = sal_dilution_si(S, t, p)
If d = ErrorReturn Then Exit Function

sea_vap_enthalpy_evap_si = t * d * sea_vap_s_si(1, 0, t, p)

End Function

'==========================================================================
Public Function sea_vap_volume_evap_si() As Double

'this function returns the evaporation volume in m3/kg of seawater,
'set by a previous call of either set_sea_vap_eq_at_s_p,
'set_sea_vap_eq_at_s_t or set_sea_vap_eq_at_t_p

'Note: the accuracy of this function depends on the iteration settings in this module
'and, in the case of set_sea_vap_eq_at_t_p, of those in Flu_3a

'check values with default settings:
'set_sea_vap_eq_at_s_p 0.035, 640
'sea_vap_volume_evap_si = 197.500648109577

'set_sea_vap_eq_at_s_t 0.035, 274
'sea_vap_volume_evap_si = 198.075461154363

'set_sea_vap_eq_at_t_p 274, 640
'sea_vap_volume_evap_si = 197.469911653041

Dim S As Double, t As Double, p As Double, d As Double

sea_vap_volume_evap_si = ErrorReturn

If equi_sea_vap_done <> IsOK Then Exit Function

S = equi_sea_vap_s
t = equi_sea_vap_t
p = equi_sea_vap_p

d = sal_dilution_si(S, t, p)
If d = ErrorReturn Then Exit Function

sea_vap_volume_evap_si = -d * sea_vap_s_si(0, 1, t, p)

End Function

'==========================================================================
Public Function sea_vap_entropy_sea_si() As Double

'this function returns the entropy in J/(kg K) of seawater at equilibrium with vapour,
'set by a previous call of either set_sea_vap_eq_at_s_p,
'set_sea_vap_eq_at_s_t or set_sea_vap_eq_at_t_p

'Note: the accuracy of this function depends on the iteration settings in this module
'and, in the case of set_sea_vap_eq_at_t_p, of those in Flu_3a

'check values with default settings:
'set_sea_vap_eq_at_s_p 0.035, 640
'sea_vap_entropy_sea_si = 13.0616891215168

'set_sea_vap_eq_at_s_t 0.035, 274
'sea_vap_entropy_sea_si = 12.4443983378267

'set_sea_vap_eq_at_t_p 274, 640
'sea_vap_entropy_sea_si = 14.025681511|1694

sea_vap_entropy_sea_si = ErrorReturn

If equi_sea_vap_done <> IsOK Then Exit Function

sea_vap_entropy_sea_si = sea_entropy_si(equi_sea_vap_s, equi_sea_vap_t, equi_sea_vap_p)

End Function

'==========================================================================
Public Function sea_vap_entropy_vap_si() As Double

'this function returns the entropy in J/(kg K) of vapour at equilibrium with seawater,
'set by a previous call of either set_sea_vap_eq_at_s_p,
'set_sea_vap_eq_at_s_t or set_sea_vap_eq_at_t_p

'Note: the accuracy of this function depends on the iteration settings in this module
'and, in the case of set_sea_vap_eq_at_t_p, of those in Flu_3a

'check values with default settings:
'set_sea_vap_eq_at_s_p 0.035, 640
'sea_vap_entropy_vap_si = 9140.56256065021

'set_sea_vap_eq_at_s_t 0.035, 274
'sea_vap_entropy_vap_si = 9141.68990452172

'set_sea_vap_eq_at_t_p 274, 640
'sea_vap_entropy_vap_si = 9140.27087792674

sea_vap_entropy_vap_si = ErrorReturn

If equi_sea_vap_done <> IsOK Then Exit Function

sea_vap_entropy_vap_si = vap_entropy_si(equi_sea_vap_t, equi_sea_vap_p)

End Function

'==========================================================================
Public Function sea_vap_g_si(ByVal drv_s As Integer, _
                             ByVal drv_t As Integer, _
                             ByVal drv_p As Integer, _
                             ByVal ssv_si As Double, _
                             ByVal t_si As Double, _
                             ByVal p_si As Double) As Double

'returns the Ssv-T-P derivative (d/dSsv)^drv_s (d/dT)^drv_t (d/dP)^drv_p g(Ssv,T,P)
'of the Gibbs function g of sea vapour, as a function of sea-vapour salinity in kg/kg,
'absolute temperature in K and absolute pressure in Pa
'
' g(S,T,P) = (1-w) * gV(T,P) + w * gSW(S(T,P),T,P)
'
' w = mass fraction of brine, w(ssv, T, P) = Ssv/S(T,P)
' S = brine salinity
'
' ssv_si = absolute sea-vapour salinity in kg/kg, i.e. the mass fraction of salt in seawater + vapour
' t_si  = T = absolute temperature in K, ITS-90
' p_si  = P = absolute pressure in Pa
' sal_g_si  = specific Gibbs energy in J/kg

'check values with default settings:
'sea_vap_g_si( 0, 0, 0, 0.035, 274, 610) =-2748.82963244656
'sea_vap_g_si( 1, 0, 0, 0.035, 274, 610) = 151028.257424141
'sea_vap_g_si( 0, 1, 0, 0.035, 274, 610) =-6072.50817708997
'sea_vap_g_si( 0, 0, 1, 0.035, 274, 610) = 137.534028399354
'sea_vap_g_si( 2, 0, 0, 0.035, 274, 610) = 0
'sea_vap_g_si( 1, 1, 0, 0.035, 274, 610) = 14965.0677010982
'sea_vap_g_si( 1, 0, 1, 0.035, 274, 610) =-321.591932572033
'sea_vap_g_si( 0, 2, 0, 0.035, 274, 610) =-2760.11106420895
'sea_vap_g_si( 0, 1, 1, 0.035, 274, 610) = 63.1093348228985
'sea_vap_g_si( 0, 0, 2, 0.035, 274, 610) =-1.6502788587071

Dim g As Double, S As Double, w As Double, d As Double
Dim gb As Double, gv As Double
Dim gb_p As Double, gv_p As Double
Dim gb_t As Double, gv_t As Double
Dim gb_pp As Double, gv_pp As Double, gm_pp As Double
Dim gb_tp As Double, gv_tp As Double, gm_tp As Double
Dim gb_tt As Double, gv_tt As Double, gm_tt As Double
Dim gb_s As Double, gb_sp As Double, gb_st As Double

Dim s_p As Double, s_t As Double

sea_vap_g_si = ErrorReturn

If ssv_si < 0 Or ssv_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

If drv_s < 0 Then Exit Function
If drv_t < 0 Or drv_t > 2 Then Exit Function
If drv_p < 0 Or drv_p > 2 Then Exit Function

If drv_s > 1 Then    'g is linear in ssv
  sea_vap_g_si = 0
  Exit Function
End If

If set_sea_vap_eq_at_t_p(t_si, p_si) = ErrorReturn Then Exit Function

S = sea_vap_salinity_si
If S = ErrorReturn Then Exit Function
If S <= 0 Or S >= 1 Then Exit Function
If S < ssv_si Then Exit Function

w = ssv_si / S               'liquid mass fraction of sea vapour

If drv_t + drv_p > 1 Then
  d = sal_dilution_si(S, t_si, p_si)
End If

Select Case drv_s

  Case 0:

    Select Case drv_t

      Case 0:     'd/dt

        Select Case drv_p

          Case 0: gb = sea_g_si(0, 0, 0, S, t_si, p_si)
                  If gb = ErrorReturn Then Exit Function
                  gv = vap_g_si(0, 0, t_si, p_si)
                  If gv = ErrorReturn Then Exit Function
                  g = (1 - w) * gv + w * gb                      'g

          Case 1: gb_p = sea_g_si(0, 0, 1, S, t_si, p_si)
                  If gb_p = ErrorReturn Then Exit Function
                  gv_p = vap_g_si(0, 1, t_si, p_si)
                  If gv_p = ErrorReturn Then Exit Function
                  g = (1 - w) * gv_p + w * gb_p                  'v = g_p

          Case 2: s_p = sea_vap_s_si(0, 1, t_si, p_si)
                  If s_p = ErrorReturn Then Exit Function
                  gm_pp = -d * s_p ^ 2 / S                      'latent derivative
                  gb_pp = sea_g_si(0, 0, 2, S, t_si, p_si)
                  If gb_pp = ErrorReturn Then Exit Function
                  gv_pp = vap_g_si(0, 2, t_si, p_si)
                  If gv_pp = ErrorReturn Then Exit Function
                  g = (1 - w) * gv_pp + w * (gb_pp + gm_pp)      'g_pp

          Case Else: Exit Function
        End Select

      Case 1:     'd/dt

        Select Case drv_p

          Case 0: gb_t = sea_g_si(0, 1, 0, S, t_si, p_si)
                  If gb_t = ErrorReturn Then Exit Function
                  gv_t = vap_g_si(1, 0, t_si, p_si)
                  If gv_t = ErrorReturn Then Exit Function
                  g = (1 - w) * gv_t + w * gb_t                  '-eta = g_t

          Case 1: s_t = sea_vap_s_si(1, 0, t_si, p_si)
                  If s_t = ErrorReturn Then Exit Function
                  s_p = sea_vap_s_si(0, 1, t_si, p_si)
                  If s_p = ErrorReturn Then Exit Function
                  gm_tp = -d * s_t * s_p / S                     'latent derivative
                  gb_tp = sea_g_si(0, 1, 1, S, t_si, p_si)
                  If gb_tp = ErrorReturn Then Exit Function
                  gv_tp = vap_g_si(1, 1, t_si, p_si)
                  If gv_tp = ErrorReturn Then Exit Function
                  g = (1 - w) * gv_tp + w * (gb_tp + gm_tp)                 'g_tp

          Case Else: Exit Function
        End Select

      Case 2:     'd2/dt2

        Select Case drv_p

          Case 0: s_t = sea_vap_s_si(1, 0, t_si, p_si)
                  If s_t = ErrorReturn Then Exit Function
                  gm_tt = -d * s_t ^ 2 / S                      'latent derivative
                  gb_tt = sea_g_si(0, 2, 0, S, t_si, p_si)
                  If gb_tt = ErrorReturn Then Exit Function
                  gv_tt = vap_g_si(2, 0, t_si, p_si)
                  If gv_tt = ErrorReturn Then Exit Function
                  g = (1 - w) * gv_tt + w * (gb_tt + gm_tt)                 'g_tt

          Case Else: Exit Function
        End Select

       Case Else: Exit Function
    End Select

  Case 1:       'd/ds
  
    Select Case drv_t
    
      Case 0:
        Select Case drv_p

          Case 0: g = sal_g_si(1, 0, 0, S, t_si, p_si)                     'g_s

          Case 1: gb_sp = sal_g_si(1, 0, 1, S, t_si, p_si)                 'g_sp
                  If gb_sp = ErrorReturn Then Exit Function
                  gb_s = sal_g_si(1, 0, 0, S, t_si, p_si)
                  If gb_s = ErrorReturn Then Exit Function
                  s_p = sea_vap_s_si(0, 1, t_si, p_si)
                  If s_p = ErrorReturn Then Exit Function
                  g = gb_sp + gb_s * s_p

          Case Else: Exit Function
        End Select

      Case 1:   'd2/dsdt

        Select Case drv_p

          Case 0: gb_st = sal_g_si(1, 1, 0, S, t_si, p_si)                 'g_st
                  If gb_st = ErrorReturn Then Exit Function
                  gb_s = sal_g_si(1, 0, 0, S, t_si, p_si)
                  If gb_s = ErrorReturn Then Exit Function
                  s_t = sea_vap_s_si(1, 0, t_si, p_si)
                  If s_t = ErrorReturn Then Exit Function
                  g = gb_st + gb_s * s_t

          Case Else: Exit Function
        End Select

       Case Else: Exit Function
    End Select
End Select

sea_vap_g_si = g

End Function

'=========================================================================
Public Function sea_vap_cp_seavap_si(ByVal ssv_si As Double, _
                                     ByVal t_si As Double, _
                                     ByVal p_si As Double) As Double

'returns   cp(Ssv,T,P) = -T * (d2g/dT2)_Ssv_P isobaric heat capacity of sea vapour in J/(kg K),
'ssv_si    absolute sea-vapour salinity in kg/kg
't_si      absolute temperature in K
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings of this module

'check value with default settings: sea_vap_cp_seavap_si(0.035, 274, 610) = 756270.43159|3252

Dim g_tt As Double

sea_vap_cp_seavap_si = ErrorReturn

If ssv_si < 0 Or ssv_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g_tt = sea_vap_g_si(0, 2, 0, ssv_si, t_si, p_si)
If g_tt = ErrorReturn Then Exit Function

sea_vap_cp_seavap_si = -t_si * g_tt

End Function

'=========================================================================
Public Function sea_vap_density_seavap_si(ByVal ssv_si As Double, _
                                          ByVal t_si As Double, _
                                          ByVal p_si As Double) As Double

'returns   d(Ssv,T,P) = 1/(dg/dP)_Ssv_T  density of sea vapour in kg/m3,
'ssv_si    absolute sea-vapour salinity in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings of this module

'check value with default settings: sea_vap_density_seavap_si(0.035, 274, 610) = 7.2709278688204E-03

Dim g_p As Double

sea_vap_density_seavap_si = ErrorReturn

If ssv_si < 0 Or ssv_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g_p = sea_vap_g_si(0, 0, 1, ssv_si, t_si, p_si)
If g_p = ErrorReturn Then Exit Function
If g_p <= 0 Then Exit Function

sea_vap_density_seavap_si = 1# / g_p

End Function

'=========================================================================
Public Function sea_vap_enthalpy_seavap_si(ByVal ssv_si As Double, _
                                           ByVal t_si As Double, _
                                           ByVal p_si As Double) As Double

'returns   h(Ssv,T,P) = g - T * (dg/dT)_Ssv_P  enthalpy of sea vapour in J/kg,
'ssv_si    absolute sea-vapour salinity in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings of this module

'check value with default settings: sea_vap_enthalpy_seavap_si(0.035, 274, 610) = 1661118.41089021

Dim g As Double, g_t As Double

sea_vap_enthalpy_seavap_si = ErrorReturn

If ssv_si < 0 Or ssv_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g = sea_vap_g_si(0, 0, 0, ssv_si, t_si, p_si)
If g = ErrorReturn Then Exit Function
g_t = sea_vap_g_si(0, 1, 0, ssv_si, t_si, p_si)
If g_t = ErrorReturn Then Exit Function

sea_vap_enthalpy_seavap_si = g - t_si * g_t

End Function

'=========================================================================
Public Function sea_vap_entropy_seavap_si(ByVal ssv_si As Double, _
                                          ByVal t_si As Double, _
                                          ByVal p_si As Double) As Double

'returns   eta(Ssv,T,P) = - (dg/dT)_Ssv_P  entropy of sea vapour in J/(kg K),
'ssv_si    absolute sea-vapour salinity in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings of this module

'check value with default settings: sea_vap_entropy_seavap_si(0.035, 274, 610) = 6072.50817708997

Dim g_t As Double

sea_vap_entropy_seavap_si = ErrorReturn

If ssv_si < 0 Or ssv_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g_t = sea_vap_g_si(0, 1, 0, ssv_si, t_si, p_si)
If g_t = ErrorReturn Then Exit Function

sea_vap_entropy_seavap_si = -g_t

End Function

'=========================================================================
Public Function sea_vap_expansion_seavap_si(ByVal ssv_si As Double, _
                                            ByVal t_si As Double, _
                                            ByVal p_si As Double) As Double

'returns   alpha(Ssv,T,P) = (d2g/dTdP)_Ssv/(dg/dP)_Ssv_T  thermal expansion of sea vapour in 1/K,
'ssv_si    absolute sea-vapour salinity in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings of this module

'check value with default settings: sea_vap_expansion_seavap_si(0.035, 274, 610) = 0.458863421346531

Dim g_p As Double, g_tp As Double

sea_vap_expansion_seavap_si = ErrorReturn

If ssv_si < 0 Or ssv_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g_p = sea_vap_g_si(0, 0, 1, ssv_si, t_si, p_si)
If g_p = ErrorReturn Then Exit Function
If g_p <= 0 Then Exit Function
g_tp = sea_vap_g_si(0, 1, 1, ssv_si, t_si, p_si)
If g_tp = ErrorReturn Then Exit Function

sea_vap_expansion_seavap_si = g_tp / g_p

End Function

'=========================================================================
Public Function sea_vap_kappa_t_seavap_si(ByVal ssv_si As Double, _
                                          ByVal t_si As Double, _
                                          ByVal p_si As Double) As Double

'returns   kappa_t(Ssv,T,P) = - (d2g/dP2)_Ssv_T/(dg/dP)_Ssv_T  isothermal compressibility of sea vapour in 1/Pa,
'ssv_si    absolute sea-vapour salinity in kg/kg
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings of this module

'check value with default settings: sea_vap_kappa_t_seavap_si(0.035, 274, 610) = 1.19990585450986E-02

Dim g_p As Double, g_pp As Double

sea_vap_kappa_t_seavap_si = ErrorReturn

If ssv_si < 0 Or ssv_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g_p = sea_vap_g_si(0, 0, 1, ssv_si, t_si, p_si)
If g_p = ErrorReturn Then Exit Function
If g_p <= 0 Then Exit Function
g_pp = sea_vap_g_si(0, 0, 2, ssv_si, t_si, p_si)
If g_pp = ErrorReturn Then Exit Function

sea_vap_kappa_t_seavap_si = -g_pp / g_p

End Function

'==========================================================================
Public Function sea_vap_pressure_si() As Double

'this function returns the pressure in Pa of vapour at equilibrium with seawater,
'set by a previous call of either set_sea_vap_eq_at_s_p,
'set_sea_vap_eq_at_s_t or set_sea_vap_eq_at_t_p

'Note: the accuracy of this function depends on the iteration settings in this module
'and, in the case of set_sea_vap_eq_at_t_p, of those in Flu_3a

'check values with default settings:
'set_sea_vap_eq_at_s_p 0.035, 640
'sea_vap_pressure_si = 640

'set_sea_vap_eq_at_s_t 0.035, 274
'sea_vap_pressure_si = 638.044692614776

'set_sea_vap_eq_at_t_p 274, 640
'sea_vap_pressure_si = 640

sea_vap_pressure_si = ErrorReturn

If equi_sea_vap_done <> IsOK Then Exit Function

sea_vap_pressure_si = equi_sea_vap_p

End Function

'==========================================================================
Public Function sea_vap_salinity_si() As Double

'this function returns the brine salinity in kg/kg of seawater at equilibrium with vapour,
'set by a previous call of either set_sea_vap_eq_at_s_p,
'set_sea_vap_eq_at_s_t or set_sea_vap_eq_at_t_p

'Note: the accuracy of this function depends on the iteration settings in this module
'and, in the case of set_sea_vap_eq_at_t_p, of those in Flu_3a

'check values with default settings:
'set_sea_vap_eq_at_s_p 0.035, 640
'sea_vap_salinity_si = 0.035

'set_sea_vap_eq_at_s_t 0.035, 274
'sea_vap_salinity_si = 0.035

'set_sea_vap_eq_at_t_p 274, 640
'sea_vap_salinity_si = 2.9439629829|3595E-02

sea_vap_salinity_si = ErrorReturn

If equi_sea_vap_done <> IsOK Then Exit Function

sea_vap_salinity_si = equi_sea_vap_s

End Function

'==========================================================================
Public Function sea_vap_temperature_si() As Double

'this function returns the temperature in K of vapour at equilibrium with seawater,
'set by a previous call of either set_sea_vap_eq_at_s_p,
'set_sea_vap_eq_at_s_t or set_sea_vap_eq_at_t_p

'Note: the accuracy of this function depends on the iteration settings in this module
'and, in the case of set_sea_vap_eq_at_t_p, of those in Flu_3a

'check values with default settings:
'set_sea_vap_eq_at_s_p 0.035, 640
'sea_vap_temperature_si = 274.042416829483

'set_sea_vap_eq_at_s_t 0.035, 274
'sea_vap_temperature_si = 274

'set_sea_vap_eq_at_t_p 274, 640
'sea_vap_temperature_si = 274

sea_vap_temperature_si = ErrorReturn

If equi_sea_vap_done <> IsOK Then Exit Function

sea_vap_temperature_si = equi_sea_vap_t

End Function

'==========================================================================
Private Function sea_vap_iteration_at_s_p(ByVal sa_si As Double, _
                                          ByVal p_si As Double, _
                                          ByVal maxit As Long, _
                                          ByVal eps As Double, _
                                          ByRef d_liq_si As Double, _
                                          ByRef d_vap_si As Double, _
                                          ByRef t_si As Double) As Double

'this function computes the seawater-vapour phase equilibrium from
'equal chemical potentials of water at given salinity, sa_si, and pressure, p_si,
'from initial guesses for the temperature, t_si, the liquid water density, d_liq_si,
'and the vapour density, d_vap_si.
'The iteration limit eps refers to the error in temperature.

'output:    sea_vap_iteration_at_s_p = IsOK if successfully done
'           sea_vap_iteration_at_s_p = ErrorReturn is returned if
'           - the maximum number of iterations is exceeded without meeting the exit criterion
'           - the function call to a Gibbs/Helmholtz function has returned an error
'           - density or temperature has taken a zero or negative value during the iteration
' d_liq_si: density in kg/m3 of liquid water at (t_si, p_si)
' d_vap_si: density in kg/m3 of water vapour at (t_si, p_si)
'     t_si: absolute temperature in K of the seawater-vapour equilibrium

'input: sa_si: absolute salinity in kg/kg
'        p_si: absolute pressure in Pa
'       maxit: maximum number of iteration steps to be done
'         eps: required accuracy of temperature
'              eps > 0: absolute temperature tolerance in K
'              eps < 0: relative temperature tolerance
'    d_liq_si: initial guess for density in kg/m3 of liquid water at (t_si, p_si)
'    d_vap_si: initial guess for density in kg/m3 of water vapour at (t_si, p_si)
'        t_si: initial guess for the absolute temperature in K

Dim dl As Double, ddl As Double
Dim dv As Double, ddv As Double
Dim t As Double, dt As Double
Dim mus As Double, p As Double
Dim gl As Double, gv As Double
Dim gs_t As Double, gs_st As Double
Dim fl As Double, fl_t As Double, fl_td As Double
Dim fl_d As Double, fl_dd As Double
Dim fv As Double, fv_t As Double, fv_td As Double
Dim fv_d As Double, fv_dd As Double

Dim it As Long

Dim a(3, 3) As Double, b(3) As Double, x(3) As Double

sea_vap_iteration_at_s_p = ErrorReturn

If d_liq_si <= 0 Or _
   d_vap_si <= 0 Or _
   t_si <= 0 Then
  d_liq_si = ErrorReturn
  d_vap_si = ErrorReturn
  t_si = ErrorReturn
  Exit Function
End If

If check_limits <> 1 Then
  If p_si <= 0 Or _
     sa_si <= 0 Or sa_si >= 1 Then
    d_liq_si = ErrorReturn
    d_vap_si = ErrorReturn
    t_si = ErrorReturn
    Exit Function
  End If
Else
  'SAL_LIMITS
  If sa_si < sal_smin Or sa_si > sal_smax Or _
     p_si < sal_pmin Or p_si > sal_pmax Then
    d_liq_si = ErrorReturn
    d_vap_si = ErrorReturn
    t_si = ErrorReturn
    Exit Function
  End If
End If

check_limits = check_limits - 1

t = t_si
dv = d_vap_si
dl = d_liq_si

For it = 1 To maxit

  'Derivatives of the Helmholtz function of fluid water for 3D Newton iteration
  fl = flu_f_si(0, 0, t, dl)
  If fl = ErrorReturn Then Exit For
  fv = flu_f_si(0, 0, t, dv)
  If fv = ErrorReturn Then Exit For

  fl_t = flu_f_si(1, 0, t, dl)
  If fl_t = ErrorReturn Then Exit For
  fv_t = flu_f_si(1, 0, t, dv)
  If fv_t = ErrorReturn Then Exit For

  fl_d = flu_f_si(0, 1, t, dl)
  If fl_d = ErrorReturn Then Exit For
  fv_d = flu_f_si(0, 1, t, dv)
  If fv_d = ErrorReturn Then Exit For

  fl_td = flu_f_si(1, 1, t, dl)
  If fl_td = ErrorReturn Then Exit For
  fv_td = flu_f_si(1, 1, t, dv)
  If fv_td = ErrorReturn Then Exit For

  fl_dd = flu_f_si(0, 2, t, dl)
  If fl_dd = ErrorReturn Then Exit For
  fv_dd = flu_f_si(0, 2, t, dv)
  If fv_dd = ErrorReturn Then Exit For

  p = dv ^ 2 * fv_d
  gl = fl + dl * fl_d
  gv = fv + dv * fv_d

  'Derivatives of the saline Gibbs function for Newton iteration
  mus = sal_chempot_h2o_si(sa_si, t, p) 'saline chemical potential gs + s*dg/ds
  If mus = ErrorReturn Then Exit For
  gs_t = sal_g_si(0, 1, 0, sa_si, t, p)
  If gs_t = ErrorReturn Then Exit For
  gs_st = sal_g_si(1, 1, 0, sa_si, t, p)
  If gs_st = ErrorReturn Then Exit For

  'coefficient matrix
  a(1, 1) = fl_t + dl * fl_td + gs_t - sa_si * gs_st - fv_t - dv * fv_td
  a(1, 2) = 2 * fl_d + dl * fl_dd
  a(1, 3) = 2 * fv_d + dv * fv_dd

  a(2, 1) = dl ^ 2 * fl_td
  a(2, 2) = dl * (2 * fl_d + dl * fl_dd)
  a(2, 3) = 0

  a(3, 1) = dv ^ 2 * fv_td
  a(3, 2) = 0
  a(3, 3) = dv * (2 * fv_d + dv * fv_dd)

  'right-hand sides, must vanish at equilibrium
  b(1) = gv - gl - mus
  b(2) = p_si - dl ^ 2 * fl_d
  b(3) = p_si - dv ^ 2 * fv_d

  'solve equations
  If matrix_solve(a(), b(), x(), 3) <> 0 Then Exit For 'matrix singular
  dt = x(1)
  ddl = x(2)
  ddv = x(3)

  'update densities and temperature
  t = t + dt
  If t <= 0 Then Exit For
  dl = dl + ddl
  If dl <= 0 Then Exit For
  dv = dv + ddv
  If dv <= 0 Then Exit For

  'check absolute or relative error limit
  If (eps > 0 And Abs(dt) < eps) Or _
     (eps < 0 And Abs(dt) < -eps * Abs(t)) Then
    If t <= 0 Then Exit For
    sea_vap_iteration_at_s_p = IsOK
    t_si = t
    d_liq_si = dl
    d_vap_si = dv
    Exit For
  End If

Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Or _
     d_liq_si <= flu_dmin Or d_liq_si > flu_dmax Then
    t_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_vap_si = ErrorReturn
    sea_vap_iteration_at_s_p = ErrorReturn
    Exit Function
  End If
  'SAL_LIMITS
  If t_si < sal_tmin Or t_si > sal_tmax Or _
     sa_si < sal_smin Or sa_si > sal_smax Or _
     p_si < sal_pmin Or p_si > sal_pmax Then
    t_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_vap_si = ErrorReturn
    sea_vap_iteration_at_s_p = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Function sea_vap_iteration_at_s_t(ByVal sa_si As Double, _
                                          ByVal t_si As Double, _
                                          ByVal maxit As Long, _
                                          ByVal eps As Double, _
                                          ByRef d_liq_si As Double, _
                                          ByRef d_vap_si As Double, _
                                          ByRef p_si As Double) As Double

'this function computes the seawater-vapour phase equilibrium from
'equal chemical potentials of water at given salinity, sa_si, and temperature, t_si,
'from an initial guess for the liquid & vapour density, d_liq_si, d_vap_si.
'The iteration limit eps refers to the error in pressure.

'output:    sea_vap_iteration_at_s_t = IsOK if successfully done
'           sea_vap_iteration_at_s_t = ErrorReturn is returned if
'           - the maximum number of iterations is exceeded without meeting the exit criterion
'           - the function call to a Gibbs/Helmholtz function has returned an error
'           - density has taken a zero or negative value during the iteration
' d_liq_si: density in kg/m3 of liquid water at (t_si, p_si)
' d_vap_si: density in kg/m3 of water vapour at (t_si, p_si)
'     p_si: absolute pressure in Pa of the seawater-vapour equilibrium

'input: sa_si: absolute salinity in kg/kg
'        t_si: absolute temperature in K
'       maxit: maximum number of iteration steps to be done
'         eps: required accuracy of pressure
'              eps > 0: absolute pressure tolerance in Pa
'              eps < 0: relative pressure tolerance
'    d_liq_si: initial guess for density in kg/m3 of liquid water at (t_si, p_si)
'    d_vap_si: initial guess for density in kg/m3 of water vapour at (t_si, p_si)

Dim dl As Double, ddl As Double
Dim dv As Double, ddv As Double
Dim mus As Double, p As Double, p_old As Double
Dim gl As Double, gv As Double
Dim gs_p As Double, gs_sp As Double
Dim fl As Double, fl_d As Double, fl_dd As Double
Dim fv As Double, fv_d As Double, fv_dd As Double

Dim it As Long

Dim a(2, 2) As Double, b(2) As Double, x(2) As Double

sea_vap_iteration_at_s_t = ErrorReturn

If d_liq_si <= 0 Or _
   d_vap_si <= 0 Then
  t_si = ErrorReturn
  d_liq_si = ErrorReturn
  d_vap_si = ErrorReturn
  Exit Function
End If

If check_limits <> 1 Then
  If t_si <= 0 Or _
     sa_si < 0 Or sa_si > 1 Then
    t_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_vap_si = ErrorReturn
    Exit Function
  End If
Else
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Then
    t_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_vap_si = ErrorReturn
    Exit Function
  End If
  'SAL_LIMITS
  If sa_si < sal_smin Or sa_si > sal_smax Or _
     t_si < sal_tmin Or t_si > sal_tmax Then
    t_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_vap_si = ErrorReturn
    Exit Function
  End If
End If

check_limits = check_limits - 1

dv = d_vap_si
dl = d_liq_si

For it = 0 To maxit

  'Derivatives of the Helmholtz function of fluid water for 3D Newton iteration
  fl = flu_f_si(0, 0, t_si, dl)
  If fl = ErrorReturn Then Exit For
  fv = flu_f_si(0, 0, t_si, dv)
  If fv = ErrorReturn Then Exit For

  fl_d = flu_f_si(0, 1, t_si, dl)
  If fl_d = ErrorReturn Then Exit For
  fv_d = flu_f_si(0, 1, t_si, dv)
  If fv_d = ErrorReturn Then Exit For

  fl_dd = flu_f_si(0, 2, t_si, dl)
  If fl_dd = ErrorReturn Then Exit For
  fv_dd = flu_f_si(0, 2, t_si, dv)
  If fv_dd = ErrorReturn Then Exit For

  p_old = p
  p = dv ^ 2 * fv_d
  gl = fl + dl * fl_d
  gv = fv + dv * fv_d

  'Derivatives of the saline Gibbs function for Newton iteration
  mus = sal_chempot_h2o_si(sa_si, t_si, p) 'saline chemical potential gs + s*dg/ds
  If mus = ErrorReturn Then Exit For
  gs_p = sal_g_si(0, 0, 1, sa_si, t_si, p)
  If gs_p = ErrorReturn Then Exit For
  gs_sp = sal_g_si(1, 0, 1, sa_si, t_si, p)
  If gs_sp = ErrorReturn Then Exit For

  If it > 0 Then
    'check absolute or relative error limit
    If (eps > 0 And Abs(p - p_old) < eps) Or _
       (eps < 0 And Abs(p - p_old) < -eps * Abs(p)) Then
      If p <= 0 Then Exit For
      sea_vap_iteration_at_s_t = IsOK
      p_si = p
      d_liq_si = dl
      d_vap_si = dv
      Exit For
    End If
  End If

  If it = maxit Then Exit For

  'coefficient matrix
  a(1, 1) = (2 * fl_d + dl * fl_dd) * (1# + (gs_p - sa_si * gs_sp) * dl)
  a(1, 2) = -2 * fv_d - dv * fv_dd

  a(2, 1) = dl * (2 * fl_d + dl * fl_dd)
  a(2, 2) = -dv * (2 * fv_d + dv * fv_dd)

  'right-hand sides, must vanish at equilibrium
  b(1) = gv - gl - mus
  b(2) = dv ^ 2 * fv_d - dl ^ 2 * fl_d

  'solve equations
  If matrix_solve(a(), b(), x(), 2) <> 0 Then Exit For 'matrix singular
  ddl = x(1)
  ddv = x(2)

  'update densities
  dl = dl + ddl
  If dl <= 0 Then Exit For
  dv = dv + ddv
  If dv <= 0 Then Exit For

Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'FLU_LIMITS
  If d_liq_si <= flu_dmin Or d_liq_si > flu_dmax Then
    p_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_vap_si = ErrorReturn
    sea_vap_iteration_at_s_t = ErrorReturn
    Exit Function
  End If
  'SAL_LIMITS
  If p_si < sal_pmin Or p_si > sal_pmax Then
    p_si = ErrorReturn
    d_liq_si = ErrorReturn
    d_vap_si = ErrorReturn
    sea_vap_iteration_at_s_t = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Function sea_vap_iteration_at_t_p(ByVal t_si As Double, _
                                          ByVal p_si As Double, _
                                          ByVal maxit As Long, _
                                          ByVal eps As Double, _
                                          ByRef s_sea_si As Double) As Double

'this function computes the seawater-vapour phase equilibrium from
'equal chemical potentials of water at given temperature, t_si, and pressure, p_si,
'from an initial guess for the brine salinity, s_sea_si.
'The iteration limit eps refers to the error in brine salinity.

'output:    sea_vap_iteration_at_t_p = IsOK if successfully done
'           sea_vap_iteration_at_t_p = ErrorReturn is returned if
'           - the maximum number of iterations is exceeded without meeting the exit criterion
'           - the function call to a Gibbs function has returned an error
'           - salinity has taken a value <=0 or >=1 during the iteration
' s_sea_si: brine salinity in kg/kg at the seawater-vapour equilibrium

'input: t_si: absolute temperature in K
'       p_si: absolute pressure in Pa
'      maxit: maximum number of iteration steps to be done
'        eps: required accuracy of salinity
'             eps > 0: absolute salinity tolerance in kg/kg
'             eps < 0: relative salinity tolerance
'   s_sea_si: initial guess of brine salinity in kg/kg

Dim S As Double, ds As Double
Dim mus As Double, sgss As Double
Dim gl As Double, gv As Double

Dim it As Long

sea_vap_iteration_at_t_p = ErrorReturn

If s_sea_si <= 0 Or s_sea_si >= 1 Then
  s_sea_si = ErrorReturn
  Exit Function
End If

If check_limits <> 1 Then
  If p_si <= 0 Or _
     t_si <= 0 Then
    s_sea_si = ErrorReturn
    Exit Function
  End If
Else
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Then
    s_sea_si = ErrorReturn
    Exit Function
  End If
  'SAL_LIMITS
  If t_si < sal_tmin Or t_si > sal_tmax Or _
     p_si < sal_pmin Or p_si > sal_pmax Then
    s_sea_si = ErrorReturn
    Exit Function
  End If
End If

check_limits = check_limits - 1

'get the chemical potential of fluid water (this implies iteration in Flu_3a)
gl = liq_gibbs_energy_si(t_si, p_si)
If gl = ErrorReturn Then
  check_limits = check_limits + 1
  Exit Function
End If

gv = vap_gibbs_energy_si(t_si, p_si)
If gv = ErrorReturn Then
  check_limits = check_limits + 1
  Exit Function
End If

S = s_sea_si

For it = 1 To maxit

  'Derivatives of the Gibbs functions for Newton iteration
  mus = sal_chempot_h2o_si(S, t_si, p_si) 'saline chemical potential gs + s*dg/ds
  If mus = ErrorReturn Then Exit For
  sgss = sal_dilution_si(S, t_si, p_si)  'S* d2g/ds2
  If sgss = ErrorReturn Then Exit For
  If sgss <= 0 Then Exit For  'physically, this is impossible

  'brine salinity iteration step
  ds = (gl + mus - gv) / sgss

  'update brine salinity
  S = S + ds
  If S <= 0 Or S >= 1 Then Exit For

  'check absolute or relative error limit
  If (eps > 0 And Abs(ds) < eps) Or _
     (eps < 0 And Abs(ds) < -eps * S) Then
    If S >= 1 Then Exit For
    sea_vap_iteration_at_t_p = IsOK
    s_sea_si = S
    Exit For
  End If

Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'SAL_LIMITS
  If s_sea_si < sal_smin Or s_sea_si > sal_smax Then
    s_sea_si = ErrorReturn
    sea_vap_iteration_at_t_p = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Sub set_sea_vap_state(ByVal sa_si As Double, _
                              ByVal t_si As Double, _
                              ByVal p_si As Double, _
                              ByVal d_liq_si As Double, _
                              ByVal d_vap_si As Double)

'stores the actual properties as the current equilibrium state descriptor

equi_sea_vap_done = IsOK

equi_sea_vap_s = sa_si
equi_sea_vap_t = t_si
equi_sea_vap_p = p_si
equi_sea_vap_d_vap = d_vap_si   'density of water vapour
equi_sea_vap_d_liq = d_liq_si   'density of liquid water

End Sub

'==========================================================================
Private Sub clear_sea_vap_state()

'clears the current equilibrium state descriptor

equi_sea_vap_done = 0

End Sub

'==========================================================================
Private Sub init_it_ctrl_sea_vap()

Const s_norm = SO_salinity_si  'kg/kg of KCl normalised seawater
Const Tt = TP_temperature_si
Const dlt = TP_density_liq_IAPWS95_si  'triple point density of liquid water
Const dvt = TP_density_vap_IAPWS95_si  'triple point density of water vapour

If ctrl_initialized = IsOK Then Exit Sub

ctrl_initialized = IsOK

'Set default values and modes for the iteration
ctrl_mode_liquid = 0
ctrl_mode_vapour = 0
ctrl_mode_sea = 0
ctrl_mode_temperature = 0
ctrl_loop_maximum = 100
ctrl_init_s_sea = s_norm
ctrl_init_t = Tt
ctrl_init_d_liq = dlt
ctrl_init_d_vap = dvt

ctrl_eps_exit_s = -0.0000001 'relative, 0.1 ppm
ctrl_eps_exit_t = -0.0000001 'relative, 0.1 ppm
ctrl_eps_exit_p = -0.0000001 'relative, 0.1 ppm

End Sub

'==========================================================================
Public Sub set_it_ctrl_sea_vap(ByVal key As String, ByVal value As Double)

'this sub sets control parameters for the iteration used to compute
'seawater-vapour equilibrium

'key              value
'it_steps         0           set iteration number to default (100)
'it_steps         n > 0       set iteration number to n
'it_steps        -1           do not iterate, use initial values to compute equilibrium

'init_liq_dens    0           use default liquid density to start =
'                             liquid density correlation function, d = aux_liq_density_if97_si(T)
'init_liq_dens   -1           use triple point liquid density to start
'init_liq_dens    d > 0       use value d as liquid density to start

'init_vap_dens    0           use default vapour density to start =
'                             vapour density correlation function, d = aux_density_ideal_si(T)
'init_vap_dens   -1           use triple point vapour density to start
'init_vap_dens    d > 0       use value d as vapour density to start

'init_brine_sa    0           use default brine salinity to start ( = aux_brinesalinity_si(t, p))
'init_brine_sa   -1           use normal salinity to start
'init_brine_sa    s > 0       use value s as brine salinity to start

'init_temp        0           use default temperature to start ( = aux_boilingtemperature_si(s, p))
'init_temp       -1           use 0C temperature to start
'init_temp        t > 0       use value t as temperature to start

'tol_brine_sa     0           use default exit tolerance for brine salinity (0.1 ppm)
'tol_brine_sa     eps         use eps as exit tolerance for brine salinity (eps < 0 means relative error)

'tol_temp         0           use default exit tolerance for temperature (0.1 ppm)
'tol_temp         eps         use eps as exit tolerance for temperature (eps < 0 means relative error)

'tol_press        0           use default exit tolerance for pressure (0.1 ppm)
'tol_press        eps         use eps as exit tolerance for pressure (eps < 0 means relative error)

init_it_ctrl_sea_vap

clear_sea_vap_state

Select Case LCase(Trim(key))

  Case "it_steps":   'iteration steps
    Select Case value
      Case 0:      ctrl_loop_maximum = 100  'default = 100
      Case Is < 0: ctrl_loop_maximum = -1
      Case Else:   ctrl_loop_maximum = value
    End Select

  Case "init_liq_dens":   'starting liquid density
    Select Case CLng(value)
      Case 0:       ctrl_mode_liquid = 0    'default = aux polynomial
      Case Is < -1: 'ignore it
      Case Is < 0:  ctrl_mode_liquid = value
      Case Else:    ctrl_mode_liquid = 1
                    ctrl_init_d_liq = value
    End Select

  Case "init_vap_dens":   'starting vapour density
    Select Case CLng(value)
      Case 0:       ctrl_mode_vapour = 0    'default = aux polynomial
      Case Is < -1: 'ignore it
      Case Is < 0:  ctrl_mode_vapour = value
      Case Else:    ctrl_mode_vapour = 1
                    ctrl_init_d_vap = value
    End Select

  Case "init_brine_sa":   'starting brine salinity
    Select Case CLng(value)
      Case 0:       ctrl_mode_sea = 0    'default = aux polynomial
      Case Is < -1: 'ignore it
      Case Is < 0:  ctrl_mode_sea = value
      Case Else:    ctrl_mode_sea = 1
                    ctrl_init_s_sea = value
    End Select

  Case "init_temp":   'starting temperature
    Select Case CLng(value)
      Case 0:       ctrl_mode_temperature = 0    'default = aux polynomial
      Case Is < -1: 'ignore it
      Case Is < 0:  ctrl_mode_temperature = value
      Case Else:    ctrl_mode_temperature = 1
                    ctrl_init_t = value
    End Select

  Case "tol_brine_sa":      'required brine salinity tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_s = -0.0000001   'default = 0.1 ppm relative
      Case Else:   ctrl_eps_exit_s = value
    End Select

  Case "tol_temp":      'required temperature tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_t = -0.0000001   'default = 0.1 ppm relative
      Case Else:   ctrl_eps_exit_t = value
    End Select

  Case "tol_press":      'required pressure tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_p = -0.0000001   'default = 0.1 ppm relative
      Case Else:   ctrl_eps_exit_p = value
    End Select

End Select

End Sub

'==========================================================================
Private Function aux_boilingtemperature_si(ByVal sa_si As Double, _
                                           ByVal p_si As Double) As Double

'computes a boiling temperature estimate in K for seawater-vapour equilibrium at given
'brine salinity in kg/kg and absolute pressure in Pa from Clausius-Clapeyron
'and Raoult laws

Const Tt = TP_temperature_si
Const Pt = TP_pressure_IAPWS95_si

Const a1 = -19.8731005709116
Const a2 = -3.08975437352998

Const alpha = -0.57         'Raoult coefficient, (P(S)-P(0))/P(S) = alpha * S,  S in kg/kg

Dim p As Double, q As Double
Dim tr As Double, ps As Double

aux_boilingtemperature_si = ErrorReturn
If p_si <= 0 Then Exit Function
If sa_si < 0 Then Exit Function
If sa_si >= 1 Then Exit Function

'pressure is corrected for salinity by Raoult's law
ps = p_si / (1# - alpha * sa_si)

'boiling temperature of H2O on the saturation curve from a correlation polynomial
'as a first guess for liq_vap_iteration_at_p (Tt < T < 640 K with rms = 0.01 in ln(p/pt))

' Clausius-Clapeyron type equation:
' ln(p/pt) = a1 * (Tt/T-1) + a2 * (Tt/T-1)^2

p = a1 / a2
q = -Log(ps / Pt) / a2

tr = -0.5 * p + Sqr(0.25 * p ^ 2 - q)

aux_boilingtemperature_si = Tt / (tr + 1)

End Function

'==========================================================================
Private Function aux_brinesalinity_si(ByVal t_si As Double, _
                                      ByVal p_si As Double) As Double

'computes a brine salinity estimate in kg/kg for seawater-vapour equilibrium at given
'absolute temperature in K and absolute pressure in Pa from Weiss & Price 1980 and
'and Raoult laws

Const alpha = -0.57         'Raoult coefficient, (P(S)-P(0))/P(S) = alpha * S,  S in kg/kg

Dim P0 As Double

aux_brinesalinity_si = ErrorReturn
If p_si <= 0 Then Exit Function
If t_si <= 0 Then Exit Function

P0 = aux_vapourpressure_WP80_si(0, t_si)
aux_brinesalinity_si = (1 - P0 / p_si) / alpha

End Function

'==========================================================================
Private Function aux_density_ideal_si(ByVal t_si As Double, _
                                      ByVal p_si As Double) As Double

'This function returns the density of ideal-gas vapour as a function of temperature
'and pressure

'output:  aux_density_ideal_si: density in kg/m^3

'input:   t_si: absolute temperature in K
'         p_si: absolute pressure in Pa


Const R = Gas_constant_H2O_si         'specific gas constant of water in J/(kg K)

aux_density_ideal_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

aux_density_ideal_si = p_si / (R * t_si)

End Function

'==========================================================================
Private Function aux_vapourpressure_WP80_si(ByVal sa_si As Double, _
                                            ByVal t_si As Double) As Double

'computes a vapour pressure estimate in Pa for seawater-vapour equilibrium at given
'brine salinity in kg/kg and absolute temperature in K from Weiss & Price (1980)

Dim Spsu As Double
Dim tabs48_K As Double
Dim psw As Double

'crude conversion is sufficient for a guess
Spsu = 1000 * sa_si
tabs48_K = t_si

'Weiss, R.F. and B.A. Price, 1980. Nitrous oxide solubility in water and seawater. Marine Chem., 8, 347-359.
'saturation water vapour pressure of seawater over the temperature range 273 to 313 K and the
'salinity range 0 to 40:

psw = Exp(24.4543 - 67.4509 * (100# / tabs48_K) - 4.8489 * Log(tabs48_K / 100#) - 0.000544 * Spsu)

'psw is the water vapour pressure (in atm), tabs48_K is the temperature (in K), and Spsu is the salinity
'on the Practical Salinity Scale.

aux_vapourpressure_WP80_si = psw * 101325

End Function

'=========================================================================
Private Function sea_vap_s_si(ByVal drv_t As Integer, _
                              ByVal drv_p As Integer, _
                              ByVal t_si As Double, _
                              ByVal p_si As Double) As Double

'this function returns absolute brine salinity s in kg/kg, and its partial derivatives
'  (d/dt)^drv_t (d/dp)^drv_p s(t,p)
'as a function of absolute temperature, t_si, in K, and absolute pressure, p_si, in Pa

Dim d As Double  'dilution coeff
Dim S As Double  'brine salinity

Dim gb_p As Double, gv_p As Double, gb_sp As Double
Dim gb_t As Double, gv_t As Double, gb_st As Double

sea_vap_s_si = ErrorReturn

If drv_t < 0 Or drv_t > 1 Then Exit Function
If drv_p < 0 Or drv_p > 1 Then Exit Function

If t_si < 0 Then Exit Function
If p_si < 0 Then Exit Function

'this must be called outside before:
'If set_sea_ice_eq_at_t_p(t_si, p_si) = ErrorReturn Then Exit Function

'if not, exit here
If equi_sea_vap_done <> IsOK Then Exit Function

S = sea_vap_salinity_si
If S = ErrorReturn Then Exit Function
If S < 0 Or S >= 1 Then Exit Function

If drv_t > 0 Or drv_p > 0 Then
  d = sal_dilution_si(S, t_si, p_si)
  If d = ErrorReturn Then Exit Function
  If d <= 0 Then Exit Function
End If

Select Case drv_t

  Case 0:

    Select Case drv_p

      Case 0: sea_vap_s_si = S

      Case 1: gb_p = sea_g_si(0, 0, 1, S, t_si, p_si)
              If gb_p = ErrorReturn Then Exit Function
              gb_sp = sea_g_si(1, 0, 1, S, t_si, p_si)
              If gb_sp = ErrorReturn Then Exit Function
              gv_p = vap_g_si(0, 1, t_si, p_si)
              If gv_p = ErrorReturn Then Exit Function
              sea_vap_s_si = (gb_p - gv_p - S * gb_sp) / d
    End Select

  Case 1:

    Select Case drv_p

      Case 0: gb_t = sea_g_si(0, 1, 0, S, t_si, p_si)
              If gb_t = ErrorReturn Then Exit Function
              gb_st = sea_g_si(1, 1, 0, S, t_si, p_si)
              If gb_st = ErrorReturn Then Exit Function
              gv_t = vap_g_si(1, 0, t_si, p_si)
              If gv_t = ErrorReturn Then Exit Function
              sea_vap_s_si = (gb_t - gv_t - S * gb_st) / d
    End Select

End Select

End Function

'=========================================================================
Private Function aux_liq_g_if97_si(ByVal drv_t As Integer, _
                                  ByVal drv_p As Integer, _
                                  ByVal t_si As Double, _
                                  ByVal p_si As Double) As Double

'This function returns the Gibbs function g(t,p) and its 1st and 2nd derivatives
'with respect to temperature and pressure, as defined for the region 1 (liquid) in IAPWS-IF97
'
'output: aux_liq_g_if97_si: specific Gibbs energy in J/kg or its t-p derivative
'
'input:  drv_t: order of the temperature derivative (0-2)
'        drv_p: order of the pressure derivative (0-2)
'        t_si:  absolute temperature in K
'        p_si:  absolute pressure in Pa

Const tu = 1386#
Const pu# = 16530000#
Const R = 461.526         'J kg-1 K-1  specific gas constant
                          'note this deviates from Gas_Constant_H2O_si = 461.51805
                          
Dim g As Double, gt As Double, gtt As Double
Dim gp As Double, gtp As Double, gpp As Double
Dim PI As Double, tau As Double, RT As Double

aux_liq_g_if97_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

InitIF97_1

PI = p_si / pu
tau = tu / t_si
RT = R * t_si

Select Case drv_t

  Case 0:
    Select Case drv_p

      Case 0: g = gamma_1(0, 0, tau, PI)
              If g = ErrorReturn Then Exit Function
              aux_liq_g_if97_si = RT * g

      Case 1: gp = gamma_1(0, 1, tau, PI)
              If gp = ErrorReturn Then Exit Function
              aux_liq_g_if97_si = RT * gp / pu

      Case 2: gpp = gamma_1(0, 2, tau, PI)
              If gpp = ErrorReturn Then Exit Function
              aux_liq_g_if97_si = RT * gpp / pu ^ 2

      Case Else: Exit Function
    End Select

  Case 1:
    Select Case drv_p

      Case 0: g = gamma_1(0, 0, tau, PI)
              If g = ErrorReturn Then Exit Function
              gt = gamma_1(1, 0, tau, PI)
              If gt = ErrorReturn Then Exit Function
              aux_liq_g_if97_si = R * (g - tau * gt)

      Case 1: gp = gamma_1(0, 1, tau, PI)
              If gp = ErrorReturn Then Exit Function
              gtp = gamma_1(1, 1, tau, PI)
              If gtp = ErrorReturn Then Exit Function
              aux_liq_g_if97_si = R * (gp - tau * gtp) / pu

      Case Else: Exit Function
    End Select

  Case 2:
    Select Case drv_p

      Case 0: gtt = gamma_1(2, 0, tau, PI)
              If gtt = ErrorReturn Then Exit Function
              aux_liq_g_if97_si = R * tau ^ 2 * gtt / t_si

      Case Else: Exit Function
    End Select

  Case Else: Exit Function
End Select

End Function

'=========================================================================
Private Function aux_vap_g_if97_si(ByVal drv_t As Integer, _
                                  ByVal drv_p As Integer, _
                                  ByVal t_si As Double, _
                                  ByVal p_si As Double) As Double

'This function returns the Gibbs function g(t,p) and its 1st and 2nd derivatives
'with respect to temperature and pressure, as defined for the region 2 (vapour) in IAPWS-IF97
'
'output: aux_vap_g_if97_si: specific Gibbs energy in J/kg or its t-p derivative
'
'input:  drv_t: order of the temperature derivative (0-2)
'        drv_p: order of the pressure derivative (0-2)
'        t_si:  absolute temperature in K
'        p_si:  absolute pressure in Pa

Const tu = 540#
Const pu = 1000000#
Const R = 461.526         'J kg-1 K-1  specific gas constant
                          'note this deviates from Gas_Constant_H2O_si = 461.51805
                          
Dim g As Double, gt As Double, gtt As Double
Dim gp As Double, gtp As Double, gpp As Double
Dim PI As Double, tau As Double, RT As Double

aux_vap_g_if97_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

InitIF97_2

PI = p_si / pu
tau = tu / t_si
RT = R * t_si

Select Case drv_t

  Case 0:
    Select Case drv_p

      Case 0: g = gamma_2(0, 0, tau, PI)
              If g = ErrorReturn Then Exit Function
              aux_vap_g_if97_si = RT * g

      Case 1: gp = gamma_2(0, 1, tau, PI)
              If gp = ErrorReturn Then Exit Function
              aux_vap_g_if97_si = RT * gp / pu

      Case 2: gpp = gamma_2(0, 2, tau, PI)
              If gpp = ErrorReturn Then Exit Function
              aux_vap_g_if97_si = RT * gpp / pu ^ 2

      Case Else: Exit Function
    End Select

  Case 1:
    Select Case drv_p

      Case 0: g = gamma_2(0, 0, tau, PI)
              If g = ErrorReturn Then Exit Function
              gt = gamma_2(1, 0, tau, PI)
              If gt = ErrorReturn Then Exit Function
              aux_vap_g_if97_si = R * (g - tau * gt)

      Case 1: gp = gamma_2(0, 1, tau, PI)
              If gp = ErrorReturn Then Exit Function
              gtp = gamma_2(1, 1, tau, PI)
              If gtp = ErrorReturn Then Exit Function
              aux_vap_g_if97_si = R * (gp - tau * gtp) / pu

      Case Else: Exit Function
    End Select

  Case 2:
    Select Case drv_p

      Case 0: gtt = gamma_2(2, 0, tau, PI)
              If gtt = ErrorReturn Then Exit Function
              aux_vap_g_if97_si = R * tau ^ 2 * gtt / t_si

      Case Else: Exit Function
    End Select

  Case Else: Exit Function
End Select

End Function

'=========================================================================
Private Function aux_liq_density_if97_si(ByVal t_si As Double, _
                                        ByVal p_si As Double) As Double

'This function returns the density of liquid water as a function of temperature
'and pressure, in the region 1 (liquid) in IAPWS-IF97

'output:  aux_liq_density_if97_si: density in kg/m^3

'input:   t_si: absolute temperature in K
'         p_si: absolute pressure in Pa

Dim d As Double

aux_liq_density_if97_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = aux_liq_g_if97_si(0, 1, t_si, p_si)

If d = ErrorReturn Then Exit Function
If d <= 0 Then Exit Function

aux_liq_density_if97_si = 1# / d

End Function

'=========================================================================
Private Function aux_vap_density_if97_si(ByVal t_si As Double, _
                                        ByVal p_si As Double) As Double

'This function returns the density of water as a function of temperature
'and pressure, in the region 2 (vapour) in IAPWS-IF97

'output:  aux_vap_density_if97_si: density in kg/m^3 of liquid water

'input:   t_si: absolute temperature in K
'         p_si: absolute pressure in Pa

Dim d As Double

aux_vap_density_if97_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = aux_vap_g_if97_si(0, 1, t_si, p_si)

If d = ErrorReturn Then Exit Function
If d <= 0 Then Exit Function

aux_vap_density_if97_si = 1# / d

End Function

'=========================================================================
Private Sub InitIF97_1()

Dim i As Integer

If n1i(1) = 0.14632971213167 Then Exit Sub

'TablE-2. Numerical values of the coefficients and exponents of the dimensionless Gibbs free energy
'for region 1, Eq. (7)
'i Ii Ji ni i Ii Ji ni
i = 1: i1i(i) = 0:  j1i(i) = -2: n1i(i) = 0.14632971213167:     i = 18: i1i(i) = 2:  j1i(i) = 3:   n1i(i) = -4.4141845330846E-06
i = 2: i1i(i) = 0:  j1i(i) = -1: n1i(i) = -0.84548187169114:    i = 19: i1i(i) = 2:  j1i(i) = 17:  n1i(i) = -7.2694996297594E-16
i = 3: i1i(i) = 0:  j1i(i) = 0:  n1i(i) = -3.756360367204:      i = 20: i1i(i) = 3:  j1i(i) = -4:  n1i(i) = -3.1679644845054E-05
i = 4: i1i(i) = 0:  j1i(i) = 1:  n1i(i) = 3.3855169168385:      i = 21: i1i(i) = 3:  j1i(i) = 0:   n1i(i) = -2.8270797985312E-06
i = 5: i1i(i) = 0:  j1i(i) = 2:  n1i(i) = -0.95791963387872:    i = 22: i1i(i) = 3:  j1i(i) = 6:   n1i(i) = -8.5205128120103E-10
i = 6: i1i(i) = 0:  j1i(i) = 3:  n1i(i) = 0.15772038513228:     i = 23: i1i(i) = 4:  j1i(i) = -5:  n1i(i) = -2.2425281908E-06
i = 7: i1i(i) = 0:  j1i(i) = 4:  n1i(i) = -0.016616417199501:   i = 24: i1i(i) = 4:  j1i(i) = -2:  n1i(i) = -6.5171222895601E-07
i = 8: i1i(i) = 0:  j1i(i) = 5:  n1i(i) = 8.1214629983568E-04:  i = 25: i1i(i) = 4:  j1i(i) = 10:  n1i(i) = -1.4341729937924E-13
i = 9: i1i(i) = 1:  j1i(i) = -9: n1i(i) = 2.8319080123804E-04:  i = 26: i1i(i) = 5:  j1i(i) = -8:  n1i(i) = -4.0516996860117E-07
i = 10: i1i(i) = 1: j1i(i) = -7: n1i(i) = -6.0706301565874E-04: i = 27: i1i(i) = 8:  j1i(i) = -11: n1i(i) = -1.2734301741641E-09
i = 11: i1i(i) = 1: j1i(i) = -1: n1i(i) = -0.018990068218419:   i = 28: i1i(i) = 8:  j1i(i) = -6:  n1i(i) = -1.7424871230634E-10
i = 12: i1i(i) = 1: j1i(i) = 0:  n1i(i) = -0.032529748770505:   i = 29: i1i(i) = 21: j1i(i) = -29: n1i(i) = -6.8762131295531E-19
i = 13: i1i(i) = 1: j1i(i) = 1:  n1i(i) = -0.021841717175414:   i = 30: i1i(i) = 23: j1i(i) = -31: n1i(i) = 1.4478307828521E-20
i = 14: i1i(i) = 1: j1i(i) = 3:  n1i(i) = -5.283835796993E-05:  i = 31: i1i(i) = 29: j1i(i) = -38: n1i(i) = 2.6335781662795E-23
i = 15: i1i(i) = 2: j1i(i) = -3: n1i(i) = -4.7184321073267E-04: i = 32: i1i(i) = 30: j1i(i) = -39: n1i(i) = -1.1947622640071E-23
i = 16: i1i(i) = 2: j1i(i) = 0:  n1i(i) = -3.0001780793026E-04: i = 33: i1i(i) = 31: j1i(i) = -40: n1i(i) = 1.8228094581404E-24
i = 17: i1i(i) = 2: j1i(i) = 1:  n1i(i) = 4.7661393906987E-05:  i = 34: i1i(i) = 32: j1i(i) = -41: n1i(i) = -9.3537087292458E-26

End Sub

'=========================================================================
Private Sub InitIF97_2()

Dim i As Integer

If n0i(i) = -9.6927686500217 Then Exit Sub

'Table 10. Numerical values of the coefficients and exponents of the ideal-gas part gamma_0 of the
'dimensionless Gibbs free energy for region 2, Eq. (16)
'i Ji ni i Ji ni
i = 1: j0i(i) = 0:  n0i(i) = -9.6927686500217
i = 2: j0i(i) = 1:  n0i(i) = 10.086655968018
i = 3: j0i(i) = -5: n0i(i) = -0.005608791128302
i = 4: j0i(i) = -4: n0i(i) = 0.071452738081455
i = 5: j0i(i) = -3: n0i(i) = -0.40710498223928
i = 6: j0i(i) = -2: n0i(i) = 1.4240819171444
i = 7: j0i(i) = -1: n0i(i) = -4.383951131945
i = 8: j0i(i) = 2:  n0i(i) = -0.28408632460772
i = 9: j0i(i) = 3:  n0i(i) = 0.021268463753307
'If Eq. (16) is incorporated into Eq. (18), instead of thE-values for n0i(1) and n0i(1)
'given above, the following values
'n0i(1) = -0.96937268393049E1
'n0i(2) = 0.10087275970006E2
'should be used


'Table 11. Numerical values of the coefficients and exponents of the
'residual part gamma_r of the dimensionless Gibbs free energy for
'region 2, Eq. (17)
'i Ii Ji ni
i = 1:  iri(i) = 1:  jri(i) = 0:  nri(i) = -1.7731742473213E-03
i = 2:  iri(i) = 1:  jri(i) = 1:  nri(i) = -0.017834862292358
i = 3:  iri(i) = 1:  jri(i) = 2:  nri(i) = -0.045996013696365
i = 4:  iri(i) = 1:  jri(i) = 3:  nri(i) = -0.057581259083432
i = 5:  iri(i) = 1:  jri(i) = 6:  nri(i) = -0.05032527872793
i = 6:  iri(i) = 2:  jri(i) = 1:  nri(i) = -3.3032641670203E-05
i = 7:  iri(i) = 2:  jri(i) = 2:  nri(i) = -1.8948987516315E-04
i = 8:  iri(i) = 2:  jri(i) = 4:  nri(i) = -3.9392777243355E-03
i = 9:  iri(i) = 2:  jri(i) = 7:  nri(i) = -0.043797295650573
i = 10: iri(i) = 2:  jri(i) = 36: nri(i) = -2.6674547914087E-05
i = 11: iri(i) = 3:  jri(i) = 0:  nri(i) = 2.0481737692309E-08
i = 12: iri(i) = 3:  jri(i) = 1:  nri(i) = 4.3870667284435E-07
i = 13: iri(i) = 3:  jri(i) = 3:  nri(i) = -3.227767723857E-05
i = 14: iri(i) = 3:  jri(i) = 6:  nri(i) = -1.5033924542148E-03
i = 15: iri(i) = 3:  jri(i) = 35: nri(i) = -0.040668253562649
i = 16: iri(i) = 4:  jri(i) = 1:  nri(i) = -7.8847309559367E-10
i = 17: iri(i) = 4:  jri(i) = 2:  nri(i) = 1.2790717852285E-08
i = 18: iri(i) = 4:  jri(i) = 3:  nri(i) = 4.8225372718507E-07
i = 19: iri(i) = 5:  jri(i) = 7:  nri(i) = 2.2922076337661E-06
i = 20: iri(i) = 6:  jri(i) = 3:  nri(i) = -1.6714766451061E-11
i = 21: iri(i) = 6:  jri(i) = 16: nri(i) = -2.1171472321355E-03
i = 22: iri(i) = 6:  jri(i) = 35: nri(i) = -23.895741934104
i = 23: iri(i) = 7:  jri(i) = 0:  nri(i) = -5.905956432427E-18
i = 24: iri(i) = 7:  jri(i) = 11: nri(i) = -1.2621808899101E-06
i = 25: iri(i) = 7:  jri(i) = 25: nri(i) = -0.038946842435739
i = 26: iri(i) = 8:  jri(i) = 8:  nri(i) = 1.1256211360459E-11
i = 27: iri(i) = 8:  jri(i) = 36: nri(i) = -0.082311340897998
i = 28: iri(i) = 9:  jri(i) = 13: nri(i) = 1.9809712802088E-08
i = 29: iri(i) = 10: jri(i) = 4:  nri(i) = 1.0406965210174E-19
i = 30: iri(i) = 10: jri(i) = 10: nri(i) = -1.0234747095929E-13
i = 31: iri(i) = 10: jri(i) = 14: nri(i) = -1.0018179379511E-09
i = 32: iri(i) = 16: jri(i) = 29: nri(i) = -8.0882908646985E-11
i = 33: iri(i) = 16: jri(i) = 50: nri(i) = 0.10693031879409
i = 34: iri(i) = 18: jri(i) = 57: nri(i) = -0.33662250574171
i = 35: iri(i) = 20: jri(i) = 20: nri(i) = 8.9185845355421E-25
i = 36: iri(i) = 20: jri(i) = 35: nri(i) = 3.0629316876232E-13
i = 37: iri(i) = 20: jri(i) = 48: nri(i) = -4.2002467698208E-06
i = 38: iri(i) = 21: jri(i) = 21: nri(i) = -5.9056029685639E-26
i = 39: iri(i) = 22: jri(i) = 53: nri(i) = 3.7826947613457E-06
i = 40: iri(i) = 23: jri(i) = 39: nri(i) = -1.2768608934681E-15
i = 41: iri(i) = 24: jri(i) = 26: nri(i) = 7.3087610595061E-29
i = 42: iri(i) = 24: jri(i) = 40: nri(i) = 5.5414715350778E-17
i = 43: iri(i) = 24: jri(i) = 58: nri(i) = -9.436970724121E-07

End Sub

'=========================================================================
Private Function gamma_0(ByVal drv_t As Integer, _
                         ByVal drv_p As Integer, _
                         ByVal tau As Double, _
                         ByVal PI As Double) As Double

'this function implements the derivatives of gamma_0 as given in Table 13 of IF-97

Dim g As Double, pwrt As Double
Dim i As Integer, k As Integer

gamma_0 = ErrorReturn

If PI <= 0 Then Exit Function
If tau <= 0 Then Exit Function
If drv_t < 0 Then Exit Function
If drv_p < 0 Then Exit Function

g = 0
If drv_t = 0 Then
  If drv_p = 0 Then
    g = Log(PI)
  Else
    g = 1# / PI
    For k = 2 To drv_p
      g = (1 - k) * g / PI
    Next k
  End If
End If

If drv_p = 0 Then
  For i = 1 To 9
    pwrt = tau ^ (j0i(i) - drv_t)
    For k = 0 To drv_t - 1
      pwrt = pwrt * (j0i(i) - k)
    Next k
    g = g + n0i(i) * pwrt
  Next i
End If

gamma_0 = g

End Function

'=========================================================================
Private Function gamma_1(ByVal drv_t As Integer, _
                         ByVal drv_p As Integer, _
                         ByVal tau As Double, _
                         ByVal PI As Double) As Double

'this function implements the derivatives of gamma as given in Table 4 of IF-97

Dim i As Integer, k As Integer
Dim g As Double, pp As Double, Tt As Double
Dim pwrt As Double, pwrp As Double

gamma_1 = ErrorReturn

If PI <= 0 Then Exit Function
If tau <= 0 Then Exit Function
If drv_t < 0 Then Exit Function
If drv_p < 0 Then Exit Function

pp = 7.1 - PI
Tt = tau - 1.222

g = 0
For i = 1 To 34

  If Tt = 0 Then
    Select Case j1i(i) - drv_t
      Case 0: pwrt = 1
      Case Is > 0: pwrt = 0
      Case Else: Exit Function
    End Select
  Else
    pwrt = Tt ^ (j1i(i) - drv_t)
  End If
  For k = 0 To drv_t - 1
    pwrt = pwrt * (j1i(i) - k)
  Next k

  If pp = 0 Then
    Select Case i1i(i) - drv_p
      Case 0: pwrp = 1
      Case Is > 0: pwrp = 0
      Case Else: Exit Function
    End Select
  Else
    pwrp = pp ^ (i1i(i) - drv_p)
  End If
  For k = 0 To drv_p - 1
    pwrp = -pwrp * (i1i(i) - k)
  Next k

  g = g + n1i(i) * pwrp * pwrt

Next i

gamma_1 = g

End Function

'=========================================================================
Private Function gamma_2(ByVal drv_t As Integer, _
                         ByVal drv_p As Integer, _
                         ByVal tau As Double, _
                         ByVal PI As Double) As Double

'this function implements the derivatives of gamma as given in Eq. 15 of IF-97

Dim g0 As Double, gr As Double

InitIF97_2

gamma_2 = ErrorReturn

g0 = gamma_0(drv_t, drv_p, tau, PI)
If g0 = ErrorReturn Then Exit Function

gr = gamma_r(drv_t, drv_p, tau, PI)
If gr = ErrorReturn Then Exit Function

gamma_2 = g0 + gr

End Function

'=========================================================================
Private Function gamma_r(ByVal drv_t As Integer, _
                         ByVal drv_p As Integer, _
                         ByVal tau As Double, _
                         ByVal PI As Double) As Double
                         
'this function implements the derivatives of gamma_r as given in Table 14 of IF-97

Dim i As Integer, k As Integer
Dim g As Double, Tt As Double
Dim pwrt As Double, pwrp As Double

gamma_r = ErrorReturn

If PI <= 0 Then Exit Function
If tau <= 0 Then Exit Function
If drv_t < 0 Then Exit Function
If drv_p < 0 Then Exit Function

Tt = tau - 0.5

g = 0
For i = 1 To 43

  If Tt = 0 Then
    Select Case jri(i) - drv_t
      Case 0: pwrt = 1
      Case Is > 0: pwrt = 0
      Case Else: Exit Function
    End Select
  Else
    pwrt = Tt ^ (jri(i) - drv_t)
  End If
  For k = 0 To drv_t - 1
    pwrt = pwrt * (jri(i) - k)
  Next k

  pwrp = PI ^ (iri(i) - drv_p)
  For k = 0 To drv_p - 1
    pwrp = pwrp * (iri(i) - k)
  Next k

  g = g + nri(i) * pwrp * pwrt

Next i

gamma_r = g

End Function


