Attribute VB_Name = "Sea_Liq_4_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules:
'     Constants_0_Mdl, file Constants_0.bas
'     Flu_1_Mdl,  file Flu_1.bas
'     Flu_2_Mdl,  file Flu_2.bas
'     Sal_1_Mdl,  file Sal_1.bas
'     Sal_2_Mdl,  file Sal_2.bas
'     Flu_3a_Mdl, file Flu_3a.bas

'#########################################################################

'=========================================================================
'This module implements the phase equilibrium properties of liquid water with seawater
'computed from IAPWS-95 and IAPWS-08:

'Release on the IAPWS Formulation 1995 for the Thermodynamic Properties of
'Ordinary Water Substance for General and Scientific Use
'The International Association for the Properties of Water and Steam
'Fredericia, Denmark, September 1996

'Release on the IAPWS Formulation 2008 for the Thermodynamic Properties of Seawater
'The International Association for the Properties of Water and Steam
'Berlin, Germany, September 2008

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98
'Private Const IsOK = -1

'Control parameters of the seawater-liquid iteration
Private ctrl_initialized As Integer

Private ctrl_mode_liquid As Integer

Private ctrl_loop_maximum As Long

Private ctrl_init_d_liq As Double    'initial liquid density

Private ctrl_eps_exit_p As Double    'pressure exit tolerance

'Properties of the current seawater-liquid equilibrium state
Private equi_sea_liq_done As Integer  'flag = IsOK if equilibrium was successfully computed

Private equi_sea_liq_d_liq As Double  'equilibrium density of liquid water
Private equi_sea_liq_d_sea As Double  'equilibrium density of liquid water under seawater pressure
Private equi_sea_liq_s As Double      'equilibrium salinity
Private equi_sea_liq_t As Double      'equilibrium temperature
Private equi_sea_liq_p_liq As Double  'equilibrium pressure of liquid water
Private equi_sea_liq_p_sea As Double  'equilibrium pressure of seawater

Private Const Version = "16 Sep 2009"
'==========================================================================
Public Function sea_liq_osmoticpressure_si(ByVal sa_si As Double, _
                                           ByVal t_si As Double, _
                                           ByVal p_si As Double) As Double


'this function returns the osmotic pressure in Pa at the absolute temperature t_si in K,
'the absolute pressure p_si in Pa and the absolute salinity sa_si in kg/kg

'Note: the accuracy of this function depends on the iteration settings in this module

'check value with default settings: sea_liq_osmoticpressure_si(0.035, 300, 1E5) = 2594603.2096|8294

sea_liq_osmoticpressure_si = ErrorReturn

If set_sea_liq_eq_at_s_t_p(sa_si, t_si, p_si) = ErrorReturn Then Exit Function

sea_liq_osmoticpressure_si = equi_sea_liq_p_sea - equi_sea_liq_p_liq

End Function

'==========================================================================
Public Function set_sea_liq_eq_at_s_t_p(ByVal sa_si As Double, _
                                                 ByVal t_si As Double, _
                                                 ByVal p_liq_si As Double) As Double

'this function computes the seawater-water equilibrium at given temperature and salinity
'by Newton iteration

Dim S As Double, eps As Double
Dim d_liq As Double
Dim d_sea As Double, p_sea As Double
Dim maxit As Long

If equi_sea_liq_done = IsOK And _
   sa_si = equi_sea_liq_s And t_si = equi_sea_liq_t And p_liq_si = equi_sea_liq_p_liq Then
  'the requested state has already been computed earlier
  set_sea_liq_eq_at_s_t_p = IsOK
  Exit Function
End If

clear_sea_liq_state 'waste any previous state

set_sea_liq_eq_at_s_t_p = ErrorReturn

If sa_si <= 0 Then Exit Function
If sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_liq_si <= 0 Then Exit Function

d_liq = liq_density_si(t_si, p_liq_si)
If d_liq = ErrorReturn Then Exit Function

init_it_ctrl_sea_liq

'set initial density guess for water under seawater pressure
Select Case ctrl_mode_liquid
  Case 0:  d_sea = d_liq
  Case 1:  d_sea = ctrl_init_d_liq
  Case Else: Exit Function
End Select
If d_sea <= 0 Then Exit Function
If d_sea = ErrorReturn Then Exit Function

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: p_sea = d_sea ^ 2 * flu_f_si(0, 1, t_si, d_sea)
           set_sea_liq_state d_liq, d_sea, sa_si, t_si, p_liq_si, p_sea
           set_sea_liq_eq_at_s_t_p = IsOK
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit
eps = ctrl_eps_exit_p
If eps = 0 Then Exit Function

'run iteration loop
If sea_liq_iteration_at_s_t_d(sa_si, t_si, d_liq, maxit, eps, d_sea, p_sea) = ErrorReturn Then
  Exit Function
End If

set_sea_liq_state d_liq, d_sea, sa_si, t_si, p_liq_si, p_sea

set_sea_liq_eq_at_s_t_p = IsOK

End Function

'==========================================================================
Public Sub set_it_ctrl_sea_liq(ByVal key As String, ByVal value As Double)

'this sub sets control parameters for the iteration used to compute
'seawater-water equilibrium

'key             value
'it_steps        0           set iteration number to default (100)
'it_steps        n > 0       set iteration number to n
'it_steps       -1           do not iterate, use initial values to compute equlibrium

'init_liq_dens   0           use default liquid density to start =
'                            liquid density IAPWS95, d = liq_density_si(T,P)
'init_liq_dens   d > 0       use value d as liquid density to start

'tol_liq_press   0           use default exit accuracy for seawater pressure (0.1 ppm)
'tol_liq_press   eps         use eps as exit accuracy for pressure (eps < 0 means relative error)

init_it_ctrl_sea_liq

clear_sea_liq_state

Select Case LCase(Trim(key))

  Case "it_steps":   'iteration steps
    Select Case value
      Case 0:      ctrl_loop_maximum = 100  'default = 100
      Case Is < 0: ctrl_loop_maximum = -1
      Case Else:   ctrl_loop_maximum = value
    End Select

  Case "init_liq_dens":   'starting liquid density
    Select Case CLng(value)
      Case 0:       ctrl_mode_liquid = 0    'default = pure water density
      Case Is < 0: 'ignore it
      Case Else:    ctrl_mode_liquid = 1
                    ctrl_init_d_liq = value
    End Select

  Case "tol_liq_press":      'required pressure tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_p = -0.0000001   'default = 0.1 ppm relative
      Case Else:   ctrl_eps_exit_p = value
    End Select

End Select

End Sub

'==========================================================================
Private Function sea_liq_iteration_at_s_t_d(ByVal sa_si As Double, _
                                            ByVal t_si As Double, _
                                            ByVal d_liq_si As Double, _
                                            ByVal maxit As Long, _
                                            ByVal eps As Double, _
                                            ByRef d_sea_si As Double, _
                                            ByRef p_sea_si As Double) As Double

'this function computes the seawater-water phase equilibrium from
'equal chemical potentials of water at given salinity, sa_si, and temperature, t_si,
'and liquid density, d_liq_si, from an initial guess for the density, d_sea_si.
'd_liq_si is the density in kg/m3 of liquid water under the pressure of the pure-water compartment
'd_sea_si is the density in kg/m3 of liquid water under the pressure of the seawater compartment
'The iteration limit eps refers to the error in pressure.

'output:    sea_liq_iteration_at_s_t_d = IsOK if successfully done
'           sea_liq_iteration_at_s_t_d = ErrorReturn is returned if
'           - the maximum number of iterations is exceeded without meeting the exit criterion
'           - the function call to a Gibbs/Helmholtz function has returned an error
'           - density has taken a zero or negative value during the iteration
' d_sea_si: density of liquid water in kg/m3 at (t_si, p_sea_si)
' p_sea_si: pressure in Pa on seawater = pressure on pure water + osmotic pressure

'input: sa_si: absolute salinity in kg/kg
'        t_si: absolute temperature in K
'    d_liq_si: density of liquid water in kg/m3
'       maxit: maximum number of iteration steps to be done
'         eps: required accuracy of pressure
'              eps > 0: absolute pressure uncertainty in Pa
'              eps < 0: relative pressure uncertainty
'    d_sea_si: initial guess for density in kg/m3 of liquid water under seawater pressure

Dim p As Double, d As Double, dd As Double
Dim mus As Double, p_old As Double
Dim gl As Double, gw As Double
Dim gs_p As Double, gs_sp As Double
Dim f As Double, f_d As Double, f_dd As Double

Dim it As Long

sea_liq_iteration_at_s_t_d = ErrorReturn

If d_sea_si <= 0 Then
  d_sea_si = ErrorReturn
  p_sea_si = ErrorReturn
  Exit Function
End If

If check_limits <> 1 Then
  If d_liq_si <= 0 Or _
     t_si <= 0 Or _
    sa_si <= 0 Or sa_si >= 1 Then
    d_sea_si = ErrorReturn
    p_sea_si = ErrorReturn
    Exit Function
  End If
Else
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Or _
     d_liq_si <= flu_dmin Or d_liq_si > flu_dmax Then
    d_sea_si = ErrorReturn
    p_sea_si = ErrorReturn
    Exit Function
  End If
  'SAL_LIMITS
  If t_si < sal_tmin Or t_si > sal_tmax Or _
     sa_si < sal_smin Or sa_si > sal_smax Then
    d_sea_si = ErrorReturn
    p_sea_si = ErrorReturn
    Exit Function
  End If
End If

check_limits = check_limits - 1

gw = flu_gibbs_energy_si(t_si, d_liq_si)
If gw = ErrorReturn Then
  check_limits = check_limits + 1
  Exit Function
End If

d = d_sea_si

For it = 0 To maxit

  'Derivatives of the Helmholtz function of liquid water for Newton iteration
  f = flu_f_si(0, 0, t_si, d)
  If f = ErrorReturn Then Exit For
  f_d = flu_f_si(0, 1, t_si, d)
  If f_d = ErrorReturn Then Exit For
  f_dd = flu_f_si(0, 2, t_si, d)
  If f_dd = ErrorReturn Then Exit For
  p_old = p
  p = d ^ 2 * f_d
  gl = f + d * f_d

  'Derivatives of the saline Gibbs function for Newton iteration
  mus = sal_chempot_h2o_si(sa_si, t_si, p) 'saline chemical potential gs + s*dg/ds
  If mus = ErrorReturn Then Exit For
  gs_p = sal_g_si(0, 0, 1, sa_si, t_si, p)
  If gs_p = ErrorReturn Then Exit For
  gs_sp = sal_g_si(1, 0, 1, sa_si, t_si, p)
  If gs_sp = ErrorReturn Then Exit For

  If it > 0 Then
    'check absolute or relative error limit
    If (eps > 0 And Abs(p - p_old) < eps) Or _
       (eps < 0 And Abs(p - p_old) < -eps * Abs(p)) Then
      If p <= 0 Then Exit For
      sea_liq_iteration_at_s_t_d = IsOK
      p_sea_si = p
      d_sea_si = d
      Exit For
    End If
  End If

  If it = maxit Then Exit For

  'density iteration step
  dd = (2 * f_d + d * f_dd) * (1 + d * (gs_p - sa_si * gs_sp))
  If dd = 0 Then Exit For
  dd = (gw - gl - mus) / dd

  'update liquid density
  d = d + dd
  If d <= 0 Then Exit For

Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'FLU_LIMITS
  If d_sea_si <= flu_dmin Or d_sea_si > flu_dmax Then
    d_sea_si = ErrorReturn
    p_sea_si = ErrorReturn
    sea_liq_iteration_at_s_t_d = ErrorReturn
    Exit Function
  End If
  'SAL_LIMITS
  If p_sea_si < sal_pmin Or p_sea_si > sal_pmax Then
    d_sea_si = ErrorReturn
    p_sea_si = ErrorReturn
    sea_liq_iteration_at_s_t_d = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Sub clear_sea_liq_state()

'clears the current equilibrium state descriptor

equi_sea_liq_done = 0

End Sub

'==========================================================================
Private Sub init_it_ctrl_sea_liq()

If ctrl_initialized = IsOK Then Exit Sub

ctrl_initialized = IsOK

'Set default values and modes for density iteration
ctrl_mode_liquid = 0
ctrl_loop_maximum = 100
ctrl_init_d_liq = 1000
ctrl_eps_exit_p = -0.0000001 'relative, 0.1 ppm

End Sub

'==========================================================================
Private Sub set_sea_liq_state(ByVal d_liq As Double, _
                              ByVal d As Double, _
                              ByVal S As Double, _
                              ByVal t As Double, _
                              ByVal p_liq As Double, _
                              ByVal p As Double)

equi_sea_liq_d_liq = d_liq    'equilibrium density of liquid water
equi_sea_liq_d_sea = d        'equilibrium density of liquid water under seawater pressure
equi_sea_liq_s = S            'equilibrium salinity
equi_sea_liq_t = t            'equilibrium temperature
equi_sea_liq_p_liq = p_liq    'equilibrium pressure of liquid water
equi_sea_liq_p_sea = p        'equilibrium pressure of seawater

End Sub


