Attribute VB_Name = "Sea_Ice_4_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules:
'     Constants_0_Mdl, file Constants_0.bas
'     Maths_0_Mdl,     file Maths_0.bas
'     Convert_0_Mdl,   file Convert_0.bas
'     Flu_1_Mdl,       file Flu_1.bas
'     Flu_3a_Mdl,      file Flu_3a.bas
'     Flu_3b_Mdl,      file Flu_3b.bas
'     Ice_1_Mdl,       file Ice_1.bas
'     Ice_2_Mdl,       file Ice_2.bas
'     Sal_1_Mdl,       file Sal_1.bas
'     Sal_2_Mdl,       file Sal_2.bas
'     Sea_3a_Mdl,      file Sea_3a.bas

'#########################################################################

'=========================================================================
'This module implements the phase equilbria properties of ice with seawater
'computed from IAPWS-95, IAPWS-06 and IAPWS-08:

'Release on the IAPWS Formulation 1995 for the Thermodynamic Properties of
'Ordinary Water Substance for General and Scientific Use
'The International Association for the Properties of Water and Steam
'Fredericia, Denmark, September 1996

'Release on an Equation of State for H2O Ice Ih
'The International Association for the Properties of Water and Steam
'Witney, UK, September 2006

'Release on the IAPWS Formulation 2008 for the Thermodynamic Properties of Seawater
'The International Association for the Properties of Water and Steam
'Berlin, Germany, September 2008

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98
'Private Const IsOK = -1

'Control parameters of the sea-ice iteration
Private ctrl_initialized As Integer

Private ctrl_mode_sea As Integer
Private ctrl_mode_temperature As Integer
Private ctrl_mode_liquid As Integer

Private ctrl_loop_maximum As Long

Private ctrl_init_s_sea As Double    'initial brine salinity
Private ctrl_init_t As Double        'initial temperature
Private ctrl_init_d_liq As Double    'initial liquid density

Private ctrl_eps_exit_s As Double    'salinity exit tolerance
Private ctrl_eps_exit_t As Double    'temperature exit tolerance
Private ctrl_eps_exit_p As Double    'pressure exit tolerance

'Properties of the current ice-seawater equilibrium state
Private equi_sea_ice_done As Integer  'flag = IsOK if equilibrium was successfully computed

Private equi_sea_ice_t As Double      'equilibrium temperature
Private equi_sea_ice_p As Double      'equilibrium pressure
Private equi_sea_ice_s As Double      'equilibrium brine salinity
Private equi_sea_ice_d_liq As Double  'equilibrium density of liquid water at t, p

Private Const Version = "07 Jun 2010"

'==========================================================================
Public Function sea_ice_brinesalinity_si(ByVal t_si As Double, _
                                         ByVal p_si As Double) As Double

'this function returns the absolute salinity in kg/kg of brine pockets at the
'absolute temperature t_si in K and the absolute pressure p_si in Pa

'note: the accuracy of this function depends on the iteration settings of this module

'check value with default settings: sea_ice_brinesalinity_si(270, 1E5) = 0.056026415032|218

Dim S As Double

sea_ice_brinesalinity_si = ErrorReturn

If set_sea_ice_eq_at_t_p(t_si, p_si) = ErrorReturn Then Exit Function

sea_ice_brinesalinity_si = equi_sea_ice_s

End Function

'=========================================================================
Public Function sea_ice_brinefraction_seaice_si(ByVal ssi_si As Double, _
                                                ByVal t_si As Double, _
                                                ByVal p_si As Double) As Double

'returns   w(ssi,T,P) = ssi/S(T,P)  mass fraction of brine in sea ice (0 < w < 1)
'ssi_si    absolute sea-ice salinity in kg/kg
't_si      absolute temperature in K
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings of this module

'check value with default settings: sea_ice_brinefraction_seaice_si(0.035, 270, 1E5) = 0.62470532836|8293

Dim S As Double

sea_ice_brinefraction_seaice_si = ErrorReturn

If ssi_si < 0 Or ssi_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

S = sea_ice_brinesalinity_si(t_si, p_si)
If S = ErrorReturn Then Exit Function
If S <= ssi_si Then Exit Function

sea_ice_brinefraction_seaice_si = ssi_si / S

End Function

'==========================================================================
Public Function sea_ice_meltingpressure_si(ByVal sa_si As Double, _
                                           ByVal t_si As Double) As Double

'this function returns the absolute pressure in Pa at which ice melts at the
'absolute temperature t_si in K and the ambient seawater salinity sa_si in kg/kg

'note: the accuracy of this function depends on the iteration settings of this module

'check value with default settings: sea_ice_meltingpressure_si(0.035, 270) = 16132047.4384824

sea_ice_meltingpressure_si = ErrorReturn

If set_sea_ice_eq_at_s_t(sa_si, t_si) = ErrorReturn Then Exit Function

sea_ice_meltingpressure_si = equi_sea_ice_p

End Function

'==========================================================================
Public Function sea_ice_freezingtemperature_si(ByVal sa_si As Double, _
                                               ByVal p_si As Double) As Double

'this function returns the absolute temperature in K at which seawater with
'absolute salinity sa_si in kg/kg freezes at the absolute pressure p in Pa

'note: the accuracy of this function depends on the iteration settings of this module

'check value with default settings: sea_ice_freezingtemperature_si(0.035, 1E5) = 271.240373585159

sea_ice_freezingtemperature_si = ErrorReturn

If set_sea_ice_eq_at_s_p(sa_si, p_si) = ErrorReturn Then Exit Function

sea_ice_freezingtemperature_si = equi_sea_ice_t

End Function

'==========================================================================
Public Function sea_ice_dtfdp_si(ByVal sa_si As Double, _
                                 ByVal p_si As Double) As Double

'Clausius-Clapeyron: Freezing point lowering due to pressure, dt/dp

'Checkvalue: sea_ice_dtfdp_si(0.035,1E5) = 7.48210942878832E-08

Dim tf As Double, s_t As Double, s_p As Double

sea_ice_dtfdp_si = ErrorReturn

tf = sea_ice_freezingtemperature_si(sa_si, p_si)
If tf = ErrorReturn Then Exit Function
s_t = sea_ice_s_si(1, 0, tf, p_si)
If s_t = ErrorReturn Or s_t = 0 Then Exit Function
s_p = sea_ice_s_si(0, 1, tf, p_si)
If s_p = ErrorReturn Then Exit Function

sea_ice_dtfdp_si = s_p / s_t

End Function

'==========================================================================
Public Function sea_ice_dtfds_si(ByVal sa_si As Double, _
                                ByVal p_si As Double) As Double

'Raoult's law: Freezing point lowering due to salt, dt/ds

'Checkvalue: sea_ice_dtfds_si(0.035,1E5) = -56.8751336296247

Dim tf As Double, s_t As Double

sea_ice_dtfds_si = ErrorReturn

tf = sea_ice_freezingtemperature_si(sa_si, p_si)
If tf = ErrorReturn Then Exit Function
s_t = sea_ice_s_si(1, 0, tf, p_si)
If s_t = ErrorReturn Or s_t = 0 Then Exit Function

sea_ice_dtfds_si = 1 / s_t

End Function

'==========================================================================
Public Function sea_ice_g_si(ByVal drv_s As Integer, _
                             ByVal drv_t As Integer, _
                             ByVal drv_p As Integer, _
                             ByVal s_si As Double, _
                             ByVal t_si As Double, _
                             ByVal p_si As Double) As Double

'returns the S-T-P derivative (d/dS)^drv_s (d/dT)^drv_t (d/dP)^drv_p gSI(S,T,P)
'of the Gibbs function gSI of sea ice, as a function of sea-ice salinity S in kg/kg,
'absolute temperature in K and absolute pressure in Pa
'
' gSI(S,T,P) = (1-w) * gIh(T,P) + w * gSW(SB(T,P),T,P)
'
' w = mass fraction of brine, w(S, T, P) = S/SB(T,P)
' SB = brine salinity
'
' s_si = S = absolute sea-ice salinity in kg/kg, i.e. the mass fraction of salt in sea ice
' t_si  = T = absolute temperature in K, ITS-90
' p_si  = P = absolute pressure in Pa
' sea_ice_g_si  = specific Gibbs energy in J/kg

'note: the accuracy of this function depends on the iteration settings of this module

'check values with default settings:
'sea_ice_g_si( 0, 0, 0, 0.035, 270, 1E5) =-414.0175745|46827
'sea_ice_g_si( 1, 0, 0, 0.035, 270, 1E5) = 96363.773049|4649
'sea_ice_g_si( 0, 1, 0, 0.035, 270, 1E5) = 500.44544418|0504
'sea_ice_g_si( 0, 0, 1, 0.035, 270, 1E5) = 1.00689072299954E-03
'sea_ice_g_si( 2, 0, 0, 0.035, 270, 1E5) = 0
'sea_ice_g_si( 1, 1, 0, 0.035, 270, 1E5) =-1144.02883418563
'sea_ice_g_si( 1, 0, 1, 0.035, 270, 1E5) =-8.62856321467455E-04
'sea_ice_g_si( 0, 2, 0, 0.035, 270, 1E5) =-232.84778338|013
'sea_ice_g_si( 0, 1, 1, 0.035, 270, 1E5) =-1.6586644669|3812E-05
'sea_ice_g_si( 0, 0, 2, 0.035, 270, 1E5) =-1.5759193211|7997E-12

Dim g As Double, S As Double, w As Double, d As Double
Dim gb As Double, gi As Double
Dim gb_p As Double, gi_p As Double
Dim gb_t As Double, gi_t As Double
Dim gb_pp As Double, gi_pp As Double, gm_pp As Double
Dim gb_tp As Double, gi_tp As Double, gm_tp As Double
Dim gb_tt As Double, gi_tt As Double, gm_tt As Double
Dim gb_s As Double, gb_sp As Double, gb_st As Double

Dim s_p As Double, s_t As Double

sea_ice_g_si = ErrorReturn

If s_si < 0 Or s_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

If drv_s < 0 Then Exit Function
If drv_t < 0 Then Exit Function
If drv_p < 0 Then Exit Function
If drv_s + drv_t + drv_p > 2 Then Exit Function

If drv_s > 1 Then    'g is linear in s_si
  sea_ice_g_si = 0
  Exit Function
End If

If set_sea_ice_eq_at_t_p(t_si, p_si) = ErrorReturn Then Exit Function

S = sea_ice_salinity_si     'brine salinity
If S = ErrorReturn Then Exit Function
If S <= 0 Or S >= 1 Then Exit Function
If S < s_si Then Exit Function

w = s_si / S               'liquid mass fraction of sea ice

If drv_t + drv_p > 1 Then
  d = sal_dilution_si(S, t_si, p_si)
End If

Select Case drv_s

  Case 0:

    Select Case drv_t

      Case 0:     'd/dt

        Select Case drv_p

          Case 0: gb = sea_g_si(0, 0, 0, S, t_si, p_si)
                  If gb = ErrorReturn Then Exit Function
                  gi = ice_g_si(0, 0, t_si, p_si)
                  If gi = ErrorReturn Then Exit Function
                  g = (1 - w) * gi + w * gb                      'g

          Case 1: gb_p = sea_g_si(0, 0, 1, S, t_si, p_si)
                  If gb_p = ErrorReturn Then Exit Function
                  gi_p = ice_g_si(0, 1, t_si, p_si)
                  If gi_p = ErrorReturn Then Exit Function
                  g = (1 - w) * gi_p + w * gb_p                  'v = g_p

          Case 2: s_p = sea_ice_s_si(0, 1, t_si, p_si)
                  If s_p = ErrorReturn Then Exit Function
                  gm_pp = -d * s_p ^ 2 / S                       'latent derivative
                  gb_pp = sea_g_si(0, 0, 2, S, t_si, p_si)
                  If gb_pp = ErrorReturn Then Exit Function
                  gi_pp = ice_g_si(0, 2, t_si, p_si)
                  If gi_pp = ErrorReturn Then Exit Function
                  g = (1 - w) * gi_pp + w * (gb_pp + gm_pp)      'g_pp

          Case Else: Exit Function
        End Select

      Case 1:     'd/dt

        Select Case drv_p

          Case 0: gb_t = sea_g_si(0, 1, 0, S, t_si, p_si)
                  If gb_t = ErrorReturn Then Exit Function
                  gi_t = ice_g_si(1, 0, t_si, p_si)
                  If gi_t = ErrorReturn Then Exit Function
                  g = (1 - w) * gi_t + w * gb_t                  '-eta = g_t

          Case 1: s_t = sea_ice_s_si(1, 0, t_si, p_si)
                  If s_t = ErrorReturn Then Exit Function
                  s_p = sea_ice_s_si(0, 1, t_si, p_si)
                  If s_p = ErrorReturn Then Exit Function
                  gm_tp = -d * s_t * s_p / S                     'latent derivative
                  gb_tp = sea_g_si(0, 1, 1, S, t_si, p_si)
                  If gb_tp = ErrorReturn Then Exit Function
                  gi_tp = ice_g_si(1, 1, t_si, p_si)
                  If gi_tp = ErrorReturn Then Exit Function
                  g = (1 - w) * gi_tp + w * (gb_tp + gm_tp)       'g_tp

          Case Else: Exit Function
        End Select

      Case 2:     'd2/dt2

        Select Case drv_p

          Case 0: s_t = sea_ice_s_si(1, 0, t_si, p_si)
                  If s_t = ErrorReturn Then Exit Function
                  gm_tt = -d * s_t ^ 2 / S                       'latent derivative
                  gb_tt = sea_g_si(0, 2, 0, S, t_si, p_si)
                  If gb_tt = ErrorReturn Then Exit Function
                  gi_tt = ice_g_si(2, 0, t_si, p_si)
                  If gi_tt = ErrorReturn Then Exit Function
                  g = (1 - w) * gi_tt + w * (gb_tt + gm_tt)      'g_tt

          Case Else: Exit Function
        End Select

       Case Else: Exit Function
    End Select

  Case 1:       'd/ds
  
    Select Case drv_t
    
      Case 0:
        Select Case drv_p

          Case 0: g = sal_g_si(1, 0, 0, S, t_si, p_si)            'g_s

          Case 1: gb_sp = sal_g_si(1, 0, 1, S, t_si, p_si)        'g_sp
                  If gb_sp = ErrorReturn Then Exit Function
                  gb_s = sal_g_si(1, 0, 0, S, t_si, p_si)
                  If gb_s = ErrorReturn Then Exit Function
                  s_p = sea_ice_s_si(0, 1, t_si, p_si)
                  If s_p = ErrorReturn Then Exit Function
                  g = gb_sp + gb_s * s_p

          Case Else: Exit Function
        End Select

      Case 1:   'd2/dsdt

        Select Case drv_p

          Case 0: gb_st = sal_g_si(1, 1, 0, S, t_si, p_si)          'g_st
                  If gb_st = ErrorReturn Then Exit Function
                  gb_s = sal_g_si(1, 0, 0, S, t_si, p_si)
                  If gb_s = ErrorReturn Then Exit Function
                  s_t = sea_ice_s_si(1, 0, t_si, p_si)
                  If s_t = ErrorReturn Then Exit Function
                  g = gb_st + gb_s * s_t

          Case Else: Exit Function
        End Select

       Case Else: Exit Function
    End Select
End Select

sea_ice_g_si = g

End Function

'=========================================================================
Public Function sea_ice_cp_seaice_si(ByVal ssi_si As Double, _
                                     ByVal t_si As Double, _
                                     ByVal p_si As Double) As Double

'returns   cp(Ssi,T,P) = -T * (d2g/dT2)_Ssi_P isobaric heat capacity of sea ice in J/(kg K),
'ssi_si    absolute sea-ice salinity in kg/kg
't_si      absolute temperature in K
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings of this module

'check value with default settings: sea_ice_cp_seaice_si(0.035, 270, 1E5) = 62868.901512|635

Dim g_tt As Double

sea_ice_cp_seaice_si = ErrorReturn

If ssi_si < 0 Or ssi_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g_tt = sea_ice_g_si(0, 2, 0, ssi_si, t_si, p_si)
If g_tt = ErrorReturn Then Exit Function

sea_ice_cp_seaice_si = -t_si * g_tt

End Function

'=========================================================================
Public Function sea_ice_density_seaice_si(ByVal ssi_si As Double, _
                                          ByVal t_si As Double, _
                                          ByVal p_si As Double) As Double

'returns   d(Ssi,T,P) = 1/(dg/dp)_Ssi_T  density of sea ice in kg/m3,
'ssi_si    absolute sea-ice salinity in kg/kg
't_si      absolute temperature in K
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings of this module

'check value with default settings: sea_ice_density_seaice_si(0.035, 270, 1E5) = 993.156434117286

Dim g_p As Double

sea_ice_density_seaice_si = ErrorReturn

If ssi_si < 0 Or ssi_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g_p = sea_ice_g_si(0, 0, 1, ssi_si, t_si, p_si)
If g_p = ErrorReturn Then Exit Function
If g_p <= 0 Then Exit Function

sea_ice_density_seaice_si = 1# / g_p

End Function

'=========================================================================
Public Function sea_ice_enthalpy_seaice_si(ByVal ssi_si As Double, _
                                           ByVal t_si As Double, _
                                           ByVal p_si As Double) As Double

'returns   h(Ssi,T,P) = g - T * (dg/dT)_Ssi_P  enthalpy of sea ice in J/kg,
'ssi_si    absolute sea-ice salinity in kg/kg
't_si      absolute temperature in K
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings of this module

'check value with default settings: sea_ice_enthalpy_seaice_si(0.035, 270, 1E5) = -135534.28750|3283

Dim g As Double, g_t As Double

sea_ice_enthalpy_seaice_si = ErrorReturn

If ssi_si < 0 Or ssi_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g = sea_ice_g_si(0, 0, 0, ssi_si, t_si, p_si)
If g = ErrorReturn Then Exit Function
g_t = sea_ice_g_si(0, 1, 0, ssi_si, t_si, p_si)
If g_t = ErrorReturn Then Exit Function

sea_ice_enthalpy_seaice_si = g - t_si * g_t

End Function

'=========================================================================
Public Function sea_ice_entropy_seaice_si(ByVal ssi_si As Double, _
                                          ByVal t_si As Double, _
                                          ByVal p_si As Double) As Double

'returns   eta(Ssi,T,P) = - (dg/dT)_Ssi_P  entropy of sea ice in J/(kg K),
'ssi_si    absolute sea-ice salinity in kg/kg
't_si      absolute temperature in K
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings of this module

'check value with default settings: sea_ice_entropy_seaice_si(0.035, 270, 1E5) = -500.44544418|0504

Dim g_t As Double

sea_ice_entropy_seaice_si = ErrorReturn

If ssi_si < 0 Or ssi_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g_t = sea_ice_g_si(0, 1, 0, ssi_si, t_si, p_si)
If g_t = ErrorReturn Then Exit Function

sea_ice_entropy_seaice_si = -g_t

End Function

'=========================================================================
Public Function sea_ice_expansion_seaice_si(ByVal ssi_si As Double, _
                                            ByVal t_si As Double, _
                                            ByVal p_si As Double) As Double

'returns   alpha(Ssi,T,P) = (d2g/dTdP)_Ssi/(dg/dP)_Ssi_T  thermal expansion of sea ice in 1/K,
'ssi_si    absolute sea-ice salinity in kg/kg
't_si      absolute temperature in K
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings of this module

'check value with default settings: sea_ice_expansion_seaice_si(0.035, 270, 1E5) = -1.6473132873|8132E-02

Dim g_p As Double, g_tp As Double

sea_ice_expansion_seaice_si = ErrorReturn

If ssi_si < 0 Or ssi_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g_p = sea_ice_g_si(0, 0, 1, ssi_si, t_si, p_si)
If g_p = ErrorReturn Then Exit Function
If g_p <= 0 Then Exit Function
g_tp = sea_ice_g_si(0, 1, 1, ssi_si, t_si, p_si)
If g_tp = ErrorReturn Then Exit Function

sea_ice_expansion_seaice_si = g_tp / g_p

End Function

'=========================================================================
Public Function sea_ice_kappa_t_seaice_si(ByVal ssi_si As Double, _
                                          ByVal t_si As Double, _
                                          ByVal p_si As Double) As Double

'returns   kappa_t(Ssi,T,P) = - (d2g/dP2)_Ssi_T/(dg/dP)_Ssi_T  isothermal compressibility of sea ice in 1/Pa,
'ssi_si    absolute sea-ice salinity in kg/kg
't_si      absolute temperature in K
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings of this module

'check value with default settings: sea_ice_kappa_t_seaice_si(0.035, 270, 1E5) = 1.5651344134|7963E-09

Dim g_p As Double, g_pp As Double

sea_ice_kappa_t_seaice_si = ErrorReturn

If ssi_si < 0 Or ssi_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

g_p = sea_ice_g_si(0, 0, 1, ssi_si, t_si, p_si)
If g_p = ErrorReturn Then Exit Function
If g_p <= 0 Then Exit Function
g_pp = sea_ice_g_si(0, 0, 2, ssi_si, t_si, p_si)
If g_pp = ErrorReturn Then Exit Function

sea_ice_kappa_t_seaice_si = -g_pp / g_p

End Function

'=========================================================================
Private Function sea_ice_s_si(ByVal drv_t As Integer, _
                              ByVal drv_p As Integer, _
                              ByVal t_si As Double, _
                              ByVal p_si As Double) As Double

'this function returns absolute brine salinity s in kg/kg, and its partial derivatives
'  (d/dt)^drv_t (d/dp)^drv_p s(t,p)
'as a function of absolute temperature, t_si, in K, and absolute pressure, p_si, in Pa

'as defined in eq. (4.10) of
'R. Feistel, E. Hagen
'A Gibbs thermodynamic potential of sea ice
'Cold Regions Science and Technology 28 (1998) 83142

Dim d As Double  'dilution coeff
Dim S As Double  'brine salinity

Dim gb_p As Double, gi_p As Double, gb_sp As Double
Dim gb_t As Double, gi_t As Double, gb_st As Double

sea_ice_s_si = ErrorReturn

If drv_t < 0 Or drv_t > 1 Then Exit Function
If drv_p < 0 Or drv_p > 1 Then Exit Function

If t_si < 0 Then Exit Function
If p_si < 0 Then Exit Function

'this must be called outside before
'If set_sea_ice_eq_at_t_p(t_si, p_si) = ErrorReturn Then Exit Function

'if not, exit here
If equi_sea_ice_done <> IsOK Then Exit Function

S = sea_ice_salinity_si
If S = ErrorReturn Then Exit Function
If S < 0 Or S >= 1 Then Exit Function

If drv_t > 0 Or drv_p > 0 Then
  d = sal_dilution_si(S, t_si, p_si)
  If d = ErrorReturn Then Exit Function
  If d <= 0 Then Exit Function
End If

Select Case drv_t

  Case 0:

    Select Case drv_p

      Case 0: sea_ice_s_si = S

      Case 1: gb_p = sea_g_si(0, 0, 1, S, t_si, p_si)
              If gb_p = ErrorReturn Then Exit Function
              gb_sp = sea_g_si(1, 0, 1, S, t_si, p_si)
              If gb_sp = ErrorReturn Then Exit Function
              gi_p = ice_g_si(0, 1, t_si, p_si)
              If gi_p = ErrorReturn Then Exit Function
              sea_ice_s_si = (gb_p - gi_p - S * gb_sp) / d
    End Select

  Case 1:

    Select Case drv_p

      Case 0: gb_t = sea_g_si(0, 1, 0, S, t_si, p_si)
              If gb_t = ErrorReturn Then Exit Function
              gb_st = sea_g_si(1, 1, 0, S, t_si, p_si)
              If gb_st = ErrorReturn Then Exit Function
              gi_t = ice_g_si(1, 0, t_si, p_si)
              If gi_t = ErrorReturn Then Exit Function
              sea_ice_s_si = (gb_t - gi_t - S * gb_st) / d
    End Select

End Select

End Function

'==========================================================================
Public Function set_sea_ice_eq_at_t_p(ByVal t_si As Double, _
                                               ByVal p_si As Double) As Double

'this function computes the seawater-ice equilibrium at given absolute temperature t_si
'in K and absolute pressure p_si in Pa by Newton iteration

Const s_norm = SO_salinity_si  'kg/kg of KCl normalised seawater

Dim d As Double
Dim S As Double, eps As Double
Dim maxit As Long

If equi_sea_ice_done = IsOK And _
   t_si = equi_sea_ice_t And p_si = equi_sea_ice_p Then
  'the requested state has already been computed earlier
  set_sea_ice_eq_at_t_p = IsOK
  Exit Function
End If

clear_sea_ice_state 'waste any previous state

set_sea_ice_eq_at_t_p = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

init_it_ctrl_sea_ice

'set initial brine salinity guess
Select Case ctrl_mode_sea
  Case 0:  S = aux_brinesalinity_si(t_si, p_si)
  Case -1: S = s_norm
  Case 1:  S = ctrl_init_s_sea
  Case Else: Exit Function
End Select

If S <= 0 Or S >= 1 Then Exit Function
If S = ErrorReturn Then Exit Function

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: d = liq_density_si(t_si, p_si)
           set_sea_ice_state S, t_si, p_si, d
           set_sea_ice_eq_at_t_p = IsOK
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit
eps = ctrl_eps_exit_s
If eps = 0 Then Exit Function

'run iteration loop
If sea_ice_iteration_at_t_p(t_si, p_si, maxit, eps, S) = ErrorReturn Then
  Exit Function
End If

d = liq_density_si(t_si, p_si)     'only for completeness of the state descriptor
set_sea_ice_state S, t_si, p_si, d

set_sea_ice_eq_at_t_p = IsOK

End Function

'==========================================================================
Public Function set_sea_ice_eq_at_s_p(ByVal sa_si As Double, _
                                               ByVal p_si As Double) As Double

'this function computes the seawater-ice equilibrium at given absolute salinity sa_si in kg/kg
'and absolute pressure p_si in Pa by Newton iteration

Const Tt = TP_temperature_si
Const dt = TP_density_liq_IAPWS95_si  'triple point density of liquid water

Dim t As Double, d As Double, eps As Double
Dim maxit As Long

If equi_sea_ice_done = IsOK And _
   sa_si = equi_sea_ice_s And p_si = equi_sea_ice_p Then
  'the requested state has already been computed earlier
  set_sea_ice_eq_at_s_p = IsOK
  Exit Function
End If

clear_sea_ice_state 'waste any previous state

set_sea_ice_eq_at_s_p = ErrorReturn

If sa_si <= 0 Then Exit Function
If sa_si >= 1 Then Exit Function
If p_si <= 0 Then Exit Function

init_it_ctrl_sea_ice

'set initial temperature guess
Select Case ctrl_mode_temperature
  Case 0:  t = aux_freezingtemperature_si(sa_si, p_si)
  Case -1: t = Tt
  Case 1:  t = ctrl_init_t
  Case Else: Exit Function
End Select
If t <= 0 Then Exit Function
If t = ErrorReturn Then Exit Function

'set initial liquid density guess
Select Case ctrl_mode_liquid
  Case 0:  d = aux_density_EOS80_si(t, p_si)
  Case -1: d = dt
  Case 1:  d = ctrl_init_d_liq
  Case Else: Exit Function
End Select
If d <= 0 Then Exit Function
If d = ErrorReturn Then Exit Function

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: set_sea_ice_state sa_si, t, p_si, d
           set_sea_ice_eq_at_s_p = IsOK
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit
eps = ctrl_eps_exit_t
If eps = 0 Then Exit Function

'run iteration loop
If sea_ice_iteration_at_s_p(sa_si, p_si, maxit, eps, d, t) = ErrorReturn Then
  Exit Function
End If

set_sea_ice_state sa_si, t, p_si, d

set_sea_ice_eq_at_s_p = IsOK

End Function

'==========================================================================
Public Function set_sea_ice_eq_at_s_t(ByVal sa_si As Double, _
                                               ByVal t_si As Double) As Double

'this function computes the seawater-ice equilibrium at given absolute salinity
'sa_si in kg/kg and absolute temperature in kg/kg by Newton iteration

Const dt = TP_density_liq_IAPWS95_si  'triple point density of liquid water

Dim S As Double, eps As Double
Dim d_si As Double, p_si As Double
Dim maxit As Long

If equi_sea_ice_done = IsOK And _
   sa_si = equi_sea_ice_s And t_si = equi_sea_ice_t Then
  'the requested state has already been computed earlier
  set_sea_ice_eq_at_s_t = IsOK
  Exit Function
End If

clear_sea_ice_state 'waste any previous state

set_sea_ice_eq_at_s_t = ErrorReturn

If sa_si <= 0 Then Exit Function
If sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function

init_it_ctrl_sea_ice

'set initial liquid water density guess
Select Case ctrl_mode_liquid
  Case 0:  d_si = aux_liq_density_correlation_si(t_si)
           If d_si = ErrorReturn Then Exit Function
           If d_si <= 0 Then Exit Function
  Case -1: d_si = dt
  Case 1:  d_si = ctrl_init_d_liq
  Case Else: Exit Function
End Select
If d_si <= 0 Then Exit Function
If d_si = ErrorReturn Then Exit Function

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: p_si = d_si ^ 2 * flu_f_si(0, 1, t_si, d_si)
           set_sea_ice_state sa_si, t_si, p_si, d_si
           set_sea_ice_eq_at_s_t = IsOK
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit
eps = ctrl_eps_exit_p
If eps = 0 Then Exit Function

'run iteration loop
If sea_ice_iteration_at_s_t(sa_si, t_si, maxit, eps, d_si, p_si) = ErrorReturn Then
  Exit Function
End If

set_sea_ice_state sa_si, t_si, p_si, d_si

set_sea_ice_eq_at_s_t = IsOK

End Function

'==========================================================================
Public Sub set_it_ctrl_sea_ice(ByVal key As String, ByVal value As Double)

'this sub sets control parameters for the iteration used to compute
'seawater-ice equilibrium

'key              value
'it_steps         0           set iteration number to default (100)
'it_steps         n > 0       set iteration number to n
'it_steps        -1           do not iterate, use initial values to compute equilibrium

'init_liq_dens    0           use default liquid density to start =
'                             liquid density correlation function, d = aux_liq_density_correlation_si(T)
'init_liq_dens   -1           use triple point liquid density to start
'init_liq_dens    d > 0       use value d as liquid density to start

'init_brine_sa    0           use default brine salinity to start ( = aux_brinesalinity_si(t, p))
'init_brine_sa   -1           use normal salinity to start
'init_brine_sa    s > 0       use value s as temperature to start

'init_temp        0           use default temperature to start ( = aux_freezingtemperature_si(s, p))
'init_temp       -1           use 0C temperature to start
'init_temp        t > 0       use value t as temperature to start

'tol_brine_sa     0           use default exit accuracy for brine salinity (0.1 ppm)
'tol_brine_sa     eps         use eps as exit accuracy for brine salinity (eps < 0 means relative error)

'tol_temp         0           use default exit accuracy for freezing temperature (0.1 ppm)
'tol_temp         eps         use eps as exit accuracy for temperature (eps < 0 means relative error)

'tol_press        0           use default exit accuracy for melting pressure (0.1 ppm)
'tol_press        eps         use eps as exit accuracy for pressure (eps < 0 means relative error)

init_it_ctrl_sea_ice

clear_sea_ice_state

Select Case LCase(Trim(key))

  Case "it_steps":   'iteration steps
    Select Case value
      Case 0:      ctrl_loop_maximum = 100  'default = 100
      Case Is < 0: ctrl_loop_maximum = -1
      Case Else:   ctrl_loop_maximum = value
    End Select

  Case "init_liq_dens":   'starting liquid density
    Select Case CLng(value)
      Case 0:       ctrl_mode_liquid = 0    'default = aux polynomial
      Case Is < -1: 'ignore it
      Case Is < 0:  ctrl_mode_liquid = value
      Case Else:    ctrl_mode_liquid = 1
                    ctrl_init_d_liq = value
    End Select

  Case "init_brine_sa":   'starting brine salinity
    Select Case CLng(value)
      Case 0:       ctrl_mode_sea = 0    'default = aux polynomial
      Case Is < -1: 'ignore it
      Case Is < 0:  ctrl_mode_sea = value
      Case Else:    ctrl_mode_sea = 1
                    ctrl_init_s_sea = value
    End Select

  Case "init_temp":   'starting temperature
    Select Case CLng(value)
      Case 0:       ctrl_mode_temperature = 0    'default = aux polynomial
      Case Is < -1: 'ignore it
      Case Is < 0:  ctrl_mode_temperature = value
      Case Else:    ctrl_mode_temperature = 1
                    ctrl_init_t = value
    End Select

  Case "tol_brine_sa":      'required brine salinity tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_s = -0.0000001   'default = 0.1 ppm relative
      Case Else:   ctrl_eps_exit_s = value
    End Select

  Case "tol_temp":      'required temperature tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_t = -0.0000001   'default = 0.1 ppm relative
      Case Else:   ctrl_eps_exit_t = value
    End Select

  Case "tol_press":      'required pressure tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_p = -0.0000001   'default = 0.1 ppm relative
      Case Else:   ctrl_eps_exit_p = value
    End Select

End Select

End Sub

'==========================================================================
Public Function sea_ice_density_ice_si() As Double

'this function returns the density in kg/m3 of ice at equilibrium with seawater,
'set by a previous call of either set_sea_ice_eq_at_s_p,
'set_sea_ice_eq_at_s_t or set_sea_ice_eq_at_t_p

'note: the accuracy of this function depends on the iteration settings of this module

'check values with default settings:
'set_sea_ice_eq_at_s_p 0.035, 1E5
'sea_ice_density_ice_si = 917.000739687017

'set_sea_ice_eq_at_s_t 0.035, 270
'sea_ice_density_ice_si = 918.898527655119

'set_sea_ice_eq_at_t_p 270, 1E5
'sea_ice_density_ice_si = 917.181167191815

sea_ice_density_ice_si = ErrorReturn

If equi_sea_ice_done <> IsOK Then Exit Function

sea_ice_density_ice_si = ice_density_si(equi_sea_ice_t, equi_sea_ice_p)

End Function

'==========================================================================
Public Function sea_ice_density_sea_si() As Double

'this function returns the density in kg/m3 of seawater at equilibrium with ice,
'set by a previous call of either set_sea_ice_eq_at_s_p,
'set_sea_ice_eq_at_s_t or set_sea_ice_eq_at_t_p

'note: the accuracy of this function depends on the iteration settings of this module

'check values with default settings:
'set_sea_ice_eq_at_s_p 0.035, 1E5
'sea_ice_density_sea_si = 1028.05199645

'set_sea_ice_eq_at_s_t 0.035, 270
'sea_ice_density_sea_si = 1035.73670169174

'set_sea_ice_eq_at_t_p 270, 1E5
'sea_ice_density_sea_si = 1045.16805917549

sea_ice_density_sea_si = ErrorReturn

If equi_sea_ice_done <> IsOK Then Exit Function

sea_ice_density_sea_si = sea_density_si(equi_sea_ice_s, equi_sea_ice_t, equi_sea_ice_p)

End Function

'==========================================================================
Public Function sea_ice_enthalpy_melt_si() As Double

'this function returns the melting enthalpy in J/kg of ice in seawater,
'set by a previous call of either set_sea_ice_eq_at_s_p,
'set_sea_ice_eq_at_s_t or set_sea_ice_eq_at_t_p

'note: the accuracy of this function depends on the iteration settings of this module

'check values with default settings:
'set_sea_ice_eq_at_s_p 0.035, 1E5
'sea_ice_enthalpy_melt_si = 329942.976284916

'set_sea_ice_eq_at_s_t 0.035, 270
'sea_ice_enthalpy_melt_si = 326829.393605264

'set_sea_ice_eq_at_t_p 270, 1E5
'sea_ice_enthalpy_melt_si = 328249.119579055

Dim S As Double, t As Double, p As Double, d As Double

sea_ice_enthalpy_melt_si = ErrorReturn

If equi_sea_ice_done <> IsOK Then Exit Function

S = equi_sea_ice_s
t = equi_sea_ice_t
p = equi_sea_ice_p

d = sal_dilution_si(S, t, p)
If d = ErrorReturn Then Exit Function

sea_ice_enthalpy_melt_si = -t * d * sea_ice_s_si(1, 0, t, p)

End Function

'==========================================================================
Public Function sea_ice_volume_melt_si() As Double

'this function returns the melting volume in m3/kg of ice in seawater,
'set by a previous call of either set_sea_ice_eq_at_s_p,
'set_sea_ice_eq_at_s_t or set_sea_ice_eq_at_t_p

'note: the accuracy of this function depends on the iteration settings of this module

'check values with default settings:
'set_sea_ice_eq_at_s_p 0.035, 1E5
'sea_ice_volume_melt_si = -9.10140854473048E-05

'set_sea_ice_eq_at_s_t 0.035, 270
'sea_ice_volume_melt_si = -9.67135426847657E-05

'set_sea_ice_eq_at_t_p 270, 1E5
'sea_ice_volume_melt_si = -9.1818691790039E-05

Dim S As Double, t As Double, p As Double, d As Double

sea_ice_volume_melt_si = ErrorReturn

If equi_sea_ice_done <> IsOK Then Exit Function

S = equi_sea_ice_s
t = equi_sea_ice_t
p = equi_sea_ice_p

d = sal_dilution_si(S, t, p)
If d = ErrorReturn Then Exit Function

sea_ice_volume_melt_si = d * sea_ice_s_si(0, 1, t, p)

End Function

'==========================================================================
Public Function sea_ice_enthalpy_ice_si() As Double

'this function returns the enthalpy in J/kg of ice at equilibrium with seawater,
'set by a previous call of either set_sea_ice_eq_at_s_p,
'set_sea_ice_eq_at_s_t or set_sea_ice_eq_at_t_p

'note: the accuracy of this function depends on the iteration settings of this modulea

'check values with default settings:
'set_sea_ice_eq_at_s_p 0.035, 1E5
'sea_ice_enthalpy_ice_si = -337351.999357552

'set_sea_ice_eq_at_s_t 0.035, 270
'sea_ice_enthalpy_ice_si = -323205.968289297

'set_sea_ice_eq_at_t_p 270, 1E5
'sea_ice_enthalpy_ice_si = -339929.55549867

sea_ice_enthalpy_ice_si = ErrorReturn

If equi_sea_ice_done <> IsOK Then Exit Function

sea_ice_enthalpy_ice_si = ice_enthalpy_si(equi_sea_ice_t, equi_sea_ice_p)

End Function

'==========================================================================
Public Function sea_ice_enthalpy_sea_si() As Double

'this function returns the enthalpy in J/kg of seawater at equilibrium with ice,
'set by a previous call of either set_sea_ice_eq_at_s_p,
'set_sea_ice_eq_at_s_t or set_sea_ice_eq_at_t_p

'note: the accuracy of this function depends on the iteration settings of this modulea

'check values with default settings:
'set_sea_ice_eq_at_s_p 0.035, 1E5
'sea_ice_enthalpy_sea_si = -7613.19337918|826

'set_sea_ice_eq_at_s_t 0.035, 270
'sea_ice_enthalpy_sea_si = 2832.9491040|7065

'set_sea_ice_eq_at_t_p 270, 1E5
'sea_ice_enthalpy_sea_si = -12742.8664892136

sea_ice_enthalpy_sea_si = ErrorReturn

If equi_sea_ice_done <> IsOK Then Exit Function

sea_ice_enthalpy_sea_si = sea_enthalpy_si(equi_sea_ice_s, equi_sea_ice_t, equi_sea_ice_p)

End Function

'==========================================================================
Public Function sea_ice_entropy_ice_si() As Double

'this function returns the entropy in J/(kg K) of ice at equilibrium with seawater,
'set by a previous call of either set_sea_ice_eq_at_s_p,
'set_sea_ice_eq_at_s_t or set_sea_ice_eq_at_t_p

'note: the accuracy of this function depends on the iteration settings of this module

'check values with default settings:
'set_sea_ice_eq_at_s_p 0.035, 1E5
'sea_ice_entropy_ice_si = -1235.44872811514

'set_sea_ice_eq_at_s_t 0.035, 270
'sea_ice_entropy_ice_si = -1247.71314646436

'set_sea_ice_eq_at_t_p 270, 1E5
'sea_ice_entropy_ice_si = -1244.97335506441

sea_ice_entropy_ice_si = ErrorReturn

If equi_sea_ice_done <> IsOK Then Exit Function

sea_ice_entropy_ice_si = ice_entropy_si(equi_sea_ice_t, equi_sea_ice_p)

End Function

'==========================================================================
Public Function sea_ice_entropy_sea_si() As Double

'this function returns the enthalpy in J/kg of seawater at equilibrium with ice,
'set by a previous call of either set_sea_ice_eq_at_s_p,
'set_sea_ice_eq_at_s_t or set_sea_ice_eq_at_t_p

'note: the accuracy of this function depends on the iteration settings of this module

'check values with default settings:
'set_sea_ice_eq_at_s_p 0.035, 1E5
'sea_ice_entropy_sea_si = -27.9264598102|728

'set_sea_ice_eq_at_s_t 0.035, 270
'sea_ice_entropy_sea_si = -46.736116956|0176

'set_sea_ice_eq_at_t_p 270, 1E5
'sea_ice_entropy_sea_si = -53.166791114|353

sea_ice_entropy_sea_si = ErrorReturn

If equi_sea_ice_done <> IsOK Then Exit Function

sea_ice_entropy_sea_si = sea_entropy_si(equi_sea_ice_s, equi_sea_ice_t, equi_sea_ice_p)

End Function

'==========================================================================
Public Function sea_ice_pressure_si() As Double

'this function returns the absolute pressure in Pa of ice at equilibrium with seawater,
'set by a previous call of either set_sea_ice_eq_at_s_p,
'set_sea_ice_eq_at_s_t or set_sea_ice_eq_at_t_p

'note: the accuracy of this function depends on the iteration settings of this module

'check values with default settings:
'set_sea_ice_eq_at_s_p 0.035, 1E5
'sea_ice_pressure_si = 100000

'set_sea_ice_eq_at_s_t 0.035, 270
'sea_ice_pressure_si = 16132047.4384824

'set_sea_ice_eq_at_t_p 270, 1E5
'sea_ice_pressure_si = 100000

sea_ice_pressure_si = ErrorReturn

If equi_sea_ice_done <> IsOK Then Exit Function

sea_ice_pressure_si = equi_sea_ice_p

End Function

'==========================================================================
Public Function sea_ice_temperature_si() As Double

'this function returns the absolute temperature in K of ice at equilibrium with seawater,
'set by a previous call of either set_sea_ice_eq_at_s_p,
'set_sea_ice_eq_at_s_t or set_sea_ice_eq_at_t_p

'note: the accuracy of this function depends on the iteration settings of this module3a

'check values with default settings:
'set_sea_ice_eq_at_s_p 0.035, 1E5
'sea_ice_temperature_si = 271.240373585159

'set_sea_ice_eq_at_s_t 0.035, 270
'sea_ice_temperature_si = 270

'set_sea_ice_eq_at_t_p 270, 1E5
'sea_ice_temperature_si = 270

sea_ice_temperature_si = ErrorReturn

If equi_sea_ice_done <> IsOK Then Exit Function

sea_ice_temperature_si = equi_sea_ice_t

End Function

'==========================================================================
Public Function sea_ice_salinity_si() As Double

'this function returns the absolute brine salinity in kg/kg of ice at equilibrium with seawater,
'set by a previous call of either set_sea_ice_eq_at_s_p,
'set_sea_ice_eq_at_s_t or set_sea_ice_eq_at_t_p

'note: the accuracy of this function depends on the iteration settings of this module

'check values with default settings:
'set_sea_ice_eq_at_s_p 0.035, 1E5
'sea_ice_salinity_si = 0.035

'set_sea_ice_eq_at_s_t 0.035, 270
'sea_ice_salinity_si = 0.035

'set_sea_ice_eq_at_t_p 270, 1E5
'sea_ice_salinity_si = 0.056026415032|218

sea_ice_salinity_si = ErrorReturn

If equi_sea_ice_done <> IsOK Then Exit Function

sea_ice_salinity_si = equi_sea_ice_s

End Function

'==========================================================================
Private Function sea_ice_iteration_at_t_p(ByVal t_si As Double, _
                                          ByVal p_si As Double, _
                                          ByVal maxit As Long, _
                                          ByVal eps As Double, _
                                          ByRef s_sea_si As Double) As Double

'this function computes the seawater-ice phase equilibrium from
'equal chemical potentials of water in the two phases at given absolute temperature t_si in K
'and absolute pressure p_si in K, from an initial guess for absolute brine salinity s_sea_si
'in kg/kg
'The iteration limit eps refers to the error in brine salinity.

'output:    sea_ice_iteration_at_t_p = IsOK if successfully done
'           sea_ice_iteration_at_t_p = ErrorReturn is returned if
'           - the maximum number of iterations is exceeded without meeting the exit criterion
'           - the function call to a Gibbs function has returned an error
'           - salinity has taken a zero or negative value during the iteration
'           - t-p values do not permit the stable existence of ice
' s_sea_si: brine salinity in kg/kg in the seawater-ice equilibrium

'input: t_si: absolute temperature in K
'       p_si: absolute pressure in Pa
'      maxit: maximum number of iteration steps to be done
'        eps: required accuracy of salinity
'             eps > 0: absolute salinity tolerance in kg/kg
'             eps < 0: relative salinity tolerance
'   s_sea_si: initial guess of brine salinity in kg/kg

Dim S As Double, ds As Double
Dim mus As Double, sgss As Double
Dim gl As Double, gi As Double

Dim it As Long

sea_ice_iteration_at_t_p = ErrorReturn

If s_sea_si <= 0 Or s_sea_si >= 1 Then
  s_sea_si = ErrorReturn
  Exit Function
End If

If check_limits <> 1 Then
  If p_si <= 0 Or t_si <= 0 Then
    s_sea_si = ErrorReturn
    Exit Function
  End If
Else
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Then
    s_sea_si = ErrorReturn
    Exit Function
  End If
  'SAL_LIMITS
  If t_si < sal_tmin Or t_si > sal_tmax Or _
     p_si < sal_pmin Or p_si > sal_pmax Then
    s_sea_si = ErrorReturn
    Exit Function
  End If
  'ICE_LIMITS
  If t_si <= ice_tmin Or t_si > ice_tmax Or _
     p_si <= ice_pmin Or p_si > ice_pmax Then
    s_sea_si = ErrorReturn
    Exit Function
  End If
End If

check_limits = check_limits - 1

'get the chemical potential of liquid water gl(t,p) (this implies iteration)
gl = liq_gibbs_energy_si(t_si, p_si)
If gl = ErrorReturn Then
  check_limits = check_limits + 1
  Exit Function
End If
'get the chemical potential of ice gi(t,p)
gi = ice_g_si(0, 0, t_si, p_si)
If gi = ErrorReturn Then
  check_limits = check_limits + 1
  Exit Function
End If

If gl <= gi Then
  check_limits = check_limits + 1
  Exit Function
End If

S = s_sea_si

For it = 1 To maxit

  'Derivatives of the Gibbs functions for Newton iteration
  mus = sal_chempot_h2o_si(S, t_si, p_si) 'saline chemical potential gs + s*dg/ds
  If mus = ErrorReturn Then Exit For
  sgss = sal_dilution_si(S, t_si, p_si)  'S* d2g/ds2
  If sgss = ErrorReturn Then Exit For
  If sgss <= 0 Then Exit For  'physically, this is impossible

  'brine salinity iteration step
  ds = (gl + mus - gi) / sgss

  'update brine salinity
  S = S + ds
  If S <= 0 Then Exit For

  'check absolute or relative error limit
  If (eps > 0 And Abs(ds) < eps) Or _
     (eps < 0 And Abs(ds) < -eps * S) Then
    If S >= 1 Then Exit For
    sea_ice_iteration_at_t_p = IsOK
    s_sea_si = S
    Exit For
  End If

Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'SAL_LIMITS
  If s_sea_si < sal_smin Or s_sea_si > sal_smax Then
    s_sea_si = ErrorReturn
    sea_ice_iteration_at_t_p = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Function sea_ice_iteration_at_s_t(ByVal sa_si As Double, _
                                          ByVal t_si As Double, _
                                          ByVal maxit As Long, _
                                          ByVal eps As Double, _
                                          ByRef d_si As Double, _
                                          ByRef p_si As Double) As Double

'this function computes the seawater-ice phase equilibrium from
'equal chemical potentials of water in the two phases at given absolute salinity sa_si in kg/kg
'and absolute temperature t_si in K, from an initial guess for the liquid density d_si in kg/m3.
'The iteration limit eps refers to the error in pressure.

'output: sea_ice_iteration_at_s_t = IsOK if successfully done
'        sea_ice_iteration_at_s_t = ErrorReturn is returned if
'        - the maximum number of iterations is exceeded without meeting the exit criterion
'        - the function call to a Gibbs/Helmholtz function has returned an error
'        - density has taken a zero or negative value during the iteration
'        - s-t values do not permit the stable existence of ice
' d_si: density of water at (t_si, p_si) in kg/m3
' p_si: pressure in Pa at the seawater-ice equilibrium

'input: sa_si: absolute salinity in kg/kg
'        t_si: absolute temperature in K
'       maxit: maximum number of iteration steps to be done
'         eps: required accuracy of pressure
'              eps > 0: absolute pressure tolerance in Pa
'              eps < 0: relative pressure tolerance
'        d_si: initial guess for density of water in kg/m3 at (t_si, p_si)

Dim p As Double, d As Double, dd As Double
Dim mus As Double, p_old As Double
Dim gl As Double, gi As Double, gi_p As Double
Dim gs_p As Double, gs_sp As Double
Dim f As Double, f_d As Double, f_dd As Double

Dim it As Long

sea_ice_iteration_at_s_t = ErrorReturn

If d_si <= 0 Then
  p_si = ErrorReturn
  d_si = ErrorReturn
  Return
End If

If check_limits <> 1 Then
  If t_si <= 0 Or _
     sa_si <= 0 Or sa_si >= 1 Then
    p_si = ErrorReturn
    d_si = ErrorReturn
    Exit Function
  End If
Else
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Then
    p_si = ErrorReturn
    d_si = ErrorReturn
    Exit Function
  End If
  'SAL_LIMITS
  If t_si < sal_tmin Or t_si > sal_tmax Or _
     sa_si < sal_smin Or sa_si > sal_smax Then
    p_si = ErrorReturn
    d_si = ErrorReturn
    Exit Function
  End If
  'ICE_LIMITS
  If t_si <= ice_tmin Or t_si > ice_tmax Then
    p_si = ErrorReturn
    d_si = ErrorReturn
    Exit Function
  End If
End If

check_limits = check_limits - 1

d = d_si

For it = 0 To maxit

  'Derivatives of the Helmholtz function of fluid water for Newton iteration
  f = flu_f_si(0, 0, t_si, d)
  If f = ErrorReturn Then Exit For
  f_d = flu_f_si(0, 1, t_si, d)
  If f_d = ErrorReturn Then Exit For
  f_dd = flu_f_si(0, 2, t_si, d)
  If f_dd = ErrorReturn Then Exit For
  p_old = p
  p = d ^ 2 * f_d
  gl = f + d * f_d

  'Derivatives of the Gibbs function of ice for Newton iteration
  gi = ice_g_si(0, 0, t_si, p)
  If gi = ErrorReturn Then Exit For
  gi_p = ice_g_si(0, 1, t_si, p)
  If gi_p = ErrorReturn Then Exit For

  'Derivatives of the saline Gibbs function for Newton iteration
  mus = sal_chempot_h2o_si(sa_si, t_si, p) 'saline chemical potential gs + s*dg/ds
  If mus = ErrorReturn Then Exit For
  gs_p = sal_g_si(0, 0, 1, sa_si, t_si, p)
  If gs_p = ErrorReturn Then Exit For
  gs_sp = sal_g_si(1, 0, 1, sa_si, t_si, p)
  If gs_sp = ErrorReturn Then Exit For

  If it > 0 Then
    'check absolute or relative error limit
    If (eps > 0 And Abs(p - p_old) < eps) Or _
       (eps < 0 And Abs(p - p_old) < -eps * Abs(p)) Then
      If p <= 0 Then Exit For
      sea_ice_iteration_at_s_t = IsOK
      p_si = p
      d_si = d
      Exit For
    End If
  End If

  If it = maxit Then Exit For

  'density iteration step
  dd = (2 * f_d + d * f_dd) * (1 + d * (-gi_p + gs_p - sa_si * gs_sp))
  If dd = 0 Then Exit For
  dd = (gi - gl - mus) / dd

  'update liquid density
  d = d + dd
  If d <= 0 Then Exit For

Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'FLU_LIMITS
  If d_si <= flu_dmin Or d_si > flu_dmax Then
    p_si = ErrorReturn
    d_si = ErrorReturn
    sea_ice_iteration_at_s_t = ErrorReturn
    Exit Function
  End If
  'SAL_LIMITS
  If p_si < sal_pmin Or p_si > sal_pmax Then
    p_si = ErrorReturn
    d_si = ErrorReturn
    sea_ice_iteration_at_s_t = ErrorReturn
    Exit Function
  End If
  'ICE_LIMITS
  If p_si <= ice_pmin Or p_si > ice_pmax Then
    p_si = ErrorReturn
    d_si = ErrorReturn
    sea_ice_iteration_at_s_t = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Function sea_ice_iteration_at_s_p(ByVal sa_si As Double, _
                                          ByVal p_si As Double, _
                                          ByVal maxit As Long, _
                                          ByVal eps As Double, _
                                          ByRef d_si As Double, _
                                          ByRef t_si As Double) As Double

'this function computes the seawater-ice phase equilibrium from equal chemical potentials
'of water at given absolute salinity sa_si in kg/kg and absolute pressure p_si in Pa,
'from an initial guesses for the absolute temperature t_si in K and the liquid density d_si
'in kg/m3.
'The iteration limit eps refers to the error in temperature.

'output: sea_ice_iteration_at_s_p = IsOK if successfully done
'        sea_ice_iteration_at_s_p = ErrorReturn is returned if
'        - the maximum number of iterations is exceeded without meeting the exit criterion
'        - the function call to a Gibbs/Helmholtz function has returned an error
'        - density or temperature has taken a zero or negative value during the iteration
'        - s-p values do not permit the stable existence of ice
'  d_si: density of water at (t_si, p_si)
'  t_si: temperature in K of the seawater-ice equilibrium

'input: sa_si: absolute salinity in kg/kg
'        p_si: absolute pressure in Pa
'       maxit: maximum number of iteration steps to be done
'         eps: required accuracy of pressure
'              eps > 0: absolute temperature tolerance in K
'              eps < 0: relative temperature tolerance
'        d_si: initial guess for density in kg/m3 of water at (t_si, p_si)
'        t_si: initial guess for the absolute freezing temperature in K

Dim d As Double, dd As Double, t As Double, dt As Double
Dim mus As Double, p As Double
Dim gl As Double, gi As Double, gi_t As Double
Dim gs_t As Double, gs_st As Double
Dim f As Double, f_t As Double, f_td As Double
Dim f_d As Double, f_dd As Double

Dim it As Long

Dim a(2, 2) As Double, b(2) As Double, x(2) As Double

sea_ice_iteration_at_s_p = ErrorReturn

If d_si <= 0 Or _
   t_si <= 0 Then
  d_si = ErrorReturn
  t_si = ErrorReturn
  Exit Function
End If

If check_limits <> 1 Then
  If p_si <= 0 Or _
     sa_si <= 0 Or sa_si >= 1 Then
    d_si = ErrorReturn
    t_si = ErrorReturn
    Exit Function
  End If
Else
  'SAL_LIMITS
  If sa_si < sal_smin Or sa_si > sal_smax Or _
     p_si < sal_pmin Or p_si > sal_pmax Then
    d_si = ErrorReturn
    t_si = ErrorReturn
    Exit Function
  End If
  'ICE_LIMITS
  If p_si <= ice_pmin Or p_si > ice_pmax Then
    d_si = ErrorReturn
    t_si = ErrorReturn
    Exit Function
  End If
End If

check_limits = check_limits - 1

t = t_si
d = d_si

For it = 1 To maxit

  'Derivatives of the Helmholtz function of fluid water for Newton iteration
  f = flu_f_si(0, 0, t, d)
  If f = ErrorReturn Then Exit For
  f_t = flu_f_si(1, 0, t, d)
  If f_t = ErrorReturn Then Exit For
  f_d = flu_f_si(0, 1, t, d)
  If f_d = ErrorReturn Then Exit For
  f_td = flu_f_si(1, 1, t, d)
  If f_td = ErrorReturn Then Exit For
  f_dd = flu_f_si(0, 2, t, d)
  If f_dd = ErrorReturn Then Exit For
  p = d ^ 2 * f_d
  gl = f + d * f_d

  'Derivatives of the Gibbs function of ice for Newton iteration
  gi = ice_g_si(0, 0, t, p_si)
  If gi = ErrorReturn Then Exit For
  gi_t = ice_g_si(1, 0, t, p_si)
  If gi_t = ErrorReturn Then Exit For

  'Derivatives of the saline Gibbs function for Newton iteration
  mus = sal_chempot_h2o_si(sa_si, t, p_si) 'saline chemical potential gs + s*dg/ds
  If mus = ErrorReturn Then Exit For
  gs_t = sal_g_si(0, 1, 0, sa_si, t, p_si)
  If gs_t = ErrorReturn Then Exit For
  gs_st = sal_g_si(1, 1, 0, sa_si, t, p_si)
  If gs_st = ErrorReturn Then Exit For

  'coefficient matrix
  a(1, 1) = f_t + d * f_td + gs_t - sa_si * gs_st - gi_t
  a(1, 2) = 2 * f_d + d * f_dd
  a(2, 1) = d ^ 2 * f_td
  a(2, 2) = d * (2 * f_d + d * f_dd)
  
  'right-hand sides, must vanish at equilibrium
  b(1) = gi - gl - mus
  b(2) = p_si - p

  'solve equations
  If matrix_solve(a(), b(), x(), 2) <> 0 Then Exit For 'matrix singular
  dt = x(1)
  dd = x(2)

  'update density and temperature
  d = d + dd
  If d <= 0 Then Exit For
  t = t + dt
  If t <= 0 Then Exit For

  'check absolute or relative error limit
  If (eps > 0 And Abs(dt) < eps) Or _
     (eps < 0 And Abs(dt) < -eps * Abs(t)) Then
    If t <= 0 Then Exit For
    sea_ice_iteration_at_s_p = IsOK
    t_si = t
    d_si = d
    Exit For
  End If

Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Or _
     d_si <= flu_dmin Or d_si > flu_dmax Then
    t_si = ErrorReturn
    d_si = ErrorReturn
    sea_ice_iteration_at_s_p = ErrorReturn
    Exit Function
  End If
  'SAL_LIMITS
  If t_si < sal_tmin Or t_si > sal_tmax Then
    t_si = ErrorReturn
    d_si = ErrorReturn
    sea_ice_iteration_at_s_p = ErrorReturn
    Exit Function
  End If
  'ICE_LIMITS
  If t_si <= ice_tmin Or t_si > ice_tmax Then
    t_si = ErrorReturn
    d_si = ErrorReturn
    sea_ice_iteration_at_s_p = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Sub set_sea_ice_state(ByVal sa_si As Double, _
                              ByVal t_si As Double, _
                              ByVal p_si As Double, _
                              ByVal d_si As Double)

'stores the actual properties as the current equilibrium state descriptor

equi_sea_ice_done = IsOK

equi_sea_ice_s = sa_si
equi_sea_ice_t = t_si
equi_sea_ice_p = p_si
equi_sea_ice_d_liq = d_si   'density of pure water

End Sub

'==========================================================================
Private Sub clear_sea_ice_state()

'clears the current equilibrium state descriptor

equi_sea_ice_done = 0

End Sub

'==========================================================================
Private Sub init_it_ctrl_sea_ice()

Const s_norm = SO_salinity_si  'kg/kg of KCl normalised seawater
Const Tt = TP_temperature_si
Const dt = TP_density_liq_IAPWS95_si  'triple point density of liquid water

If ctrl_initialized = IsOK Then Exit Sub

ctrl_initialized = IsOK

'Set default values and modes for the iteration
ctrl_mode_liquid = 0
ctrl_mode_sea = 0
ctrl_mode_temperature = 0
ctrl_loop_maximum = 100
ctrl_init_s_sea = s_norm
ctrl_init_t = Tt
ctrl_init_d_liq = dt

ctrl_eps_exit_s = -0.0000001 'relative, 0.1 ppm
ctrl_eps_exit_t = -0.0000001 'relative, 0.1 ppm
ctrl_eps_exit_p = -0.0000001 'relative, 0.1 ppm

End Sub

'==========================================================================
Private Function aux_brinesalinity_si(ByVal t_si As Double, _
                                      ByVal p_si As Double) As Double

'computes a brine salinity estimate in kg/kg for seawater-ice equilibrium at given
'absolute temperature in K and absolute pressure in Pa from Clausius-Clapeyron
'and Raoult laws

Const Tt = TP_temperature_si
Const Pt = TP_pressure_IAPWS95_si  'IAPWS-95 triple point pressure in Pa

Const alpha = -0.217         'Raoult cofficient, (T-Tt)/T = alpha * S,  in kg/kg
Const chi = -0.0000000743    'Clausius-Clapeyron coefficient,  (T-Tt) = chi * (P-Pt), in K/Pa

aux_brinesalinity_si = ErrorReturn
If p_si <= 0 Then Exit Function
If t_si <= 0 Then Exit Function

aux_brinesalinity_si = (t_si - Tt - chi * (p_si - Pt)) / (alpha * t_si)

End Function

'==========================================================================
Private Function aux_density_EOS80_si(ByVal t_si As Double, _
                                      ByVal p_si As Double) As Double

'This function returns the density of liquid water computed from the
'International Equation of State of Seawater 1980, EOS-80, as a function of temperature
'and pressure

'output:  aux_density_EOS80_si: density in kg/m^3

'input:   t_si: absolute temperature in K
'         p_si: absolute pressure in Pa

Dim AW As Double, BW As Double, KW As Double, RW As Double
Dim t As Double, p As Double

aux_density_EOS80_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

'R.C.Millard Jr.: International Oceanographic Tables Vol.4
'UNESCO technical Papers in Marine Science 40
'UNESCO 1987

't = t_c_from_t_k(t68_k_from_t_si(t_si))            'T68 in C
t = cnv_temperature("DEGC(T68)", t_si, "K(T90)")

'P = psea_bar_from_psea_pa(psea_pa_from_p_si(p_si))  'in bar
p = 0.1 * cnv_pressure("DBAR", p_si, "PA")

BW = 0.0000850935 + t * (-0.00000612293 + t * 0.000000052787)
AW = 3.239908 + t * (0.00143713 + t * (0.000116092 - t * 0.000000577905))
KW = 0.01360477 - t * 0.00005155288
KW = 19652.21 + t * (148.4206 + t * (-2.327105 + t * KW))
KW = KW + p * (AW + p * BW)
If KW = 0 Then Exit Function

KW = 1# - p / KW
If KW = 0 Then Exit Function

RW = 0.0001001685 + t * (-0.000001120083 + t * 0.000000006536332)
RW = 999.842594 + t * (0.06793952 + t * (-0.00909529 + t * RW))

aux_density_EOS80_si = RW / KW

End Function


'==========================================================================
Private Function aux_freezingtemperature_si(ByVal sa_si As Double, _
                                            ByVal p_si As Double) As Double

'computes a freezing temperature estimate in K for seawater-ice equilibrium at given
'brine salinity in kg/kg and absolute pressure in Pa from Clausius-Clapeyron
'and Raoult laws

Const Tt = TP_temperature_si
Const Pt = TP_pressure_IAPWS95_si 'IAPWS-95 triple point pressure in Pa

Const alpha = -0.217         'Raoult cofficient, (T-Tt)/T = alpha * S,  in kg/kg
Const chi = -0.0000000743    'Clausius-Clapeyron coefficient,  (T-Tt) = chi * (P-Pt), in K/Pa

aux_freezingtemperature_si = ErrorReturn
If p_si <= 0 Then Exit Function
If sa_si < 0 Then Exit Function
If sa_si >= 1 Then Exit Function

aux_freezingtemperature_si = (Tt + chi * (p_si - Pt)) / (1 - alpha * sa_si)

End Function

'==========================================================================
Private Function aux_liq_density_correlation_si(ByVal t_si As Double) As Double

'this function implements the density in kg/m3 of the freezing liquid
'as a function of absolute temperature t_si in K from an approximate
'correlation fit between 252 and 273 K with rms = 1.2 E-4 in d/dt

Const Tt = TP_temperature_si
Const dt = TP_density_liq_IAPWS95_si 'triple point density of liquid water

Const a1 = -1.78582981492113
Const a2 = -12.2325084306734
Const a3 = -52.8236936433529

Dim tr As Double, dr As Double

aux_liq_density_correlation_si = ErrorReturn

If t_si <= 0 Then Exit Function

tr = t_si / Tt - 1#
dr = a1 * tr + a2 * tr ^ 2 + a3 * tr ^ 3

aux_liq_density_correlation_si = dt * (dr + 1#)

End Function

'==========================================================================
Private Function aux_meltingpressure_si(ByVal sa_si As Double, _
                                        ByVal t_si As Double) As Double

'computes a melting pressure estimate in Pa for seawater-ice equilibrium at given
'absolute temperature in K and brine salinity in kg/kg from Clausius-Clapeyron
'and Raoult laws

Const Tt = TP_temperature_si
Const Pt = TP_pressure_IAPWS95_si  'IAPWS-95 triple point pressure in Pa

Const alpha = -0.217         'Raoult cofficient, (T-Tt)/T = alpha * S, in kg/kg
Const chi = -0.0000000743    'Clausius-Clapeyron coefficient,  (T-Tt) = chi * (P-Pt), in K/Pa

aux_meltingpressure_si = ErrorReturn
If sa_si < 0 Then Exit Function
If sa_si >= 1 Then Exit Function
If t_si <= 0 Then Exit Function

aux_meltingpressure_si = Pt + (t_si - Tt - alpha * t_si * sa_si) / chi

End Function
