Attribute VB_Name = "Sea_5a_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules:
'     Constants_0_Mdl, file Constants_0.bas
'     Sea_3a_Mdl,      file Sea_3a.bas
'     Sea_3b_Mdl,      file Sea_3b.bas
'     Sea_3c_Mdl,      file Sea_3c.bas

'#########################################################################

'=========================================================================
'This module implements seawater properties related to potential temperature
'and conservative temperature. VB Code adapted from D.R. Jackett and D.G.Wright.

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98

Private Const CP0 = 3991.86795711963       'J/(kg K), in concrete 02/12/08
Private Const p_ref = 101325#              'Pa, reference pressure

Private Const Version = "22 Oct 2009"

'==========================================================================
Public Function sea_ctmp_from_ptmp0_si(ByVal sa_si As Double, _
                                       ByVal tpot_si As Double) As Double
'==========================================================================

'   conservative temperature from potential temperature of seawater
'
'   sa_si         : Absolute Salinity                        [kg/kg]
'   pt0           : potential temperature with               [K]
'                   reference pressure of 101325 Pa
'
'   result        : conservative temperature                 [K]

'Checkvalue: sea_ctmp_from_ptmp0_si(0.035, 300) =   300.010069445349

Dim ct As Double

sea_ctmp_from_ptmp0_si = ErrorReturn

ct = sea_enthalpy_si(sa_si, tpot_si, p_ref)
If ct = ErrorReturn Then Exit Function

sea_ctmp_from_ptmp0_si = 273.15 + ct / CP0

End Function

'==========================================================================
Public Function sea_alpha_ct_si(ByVal sa_si As Double, _
                                ByVal t_si As Double, _
                                ByVal p_si As Double) As Double
'==========================================================================

' THERMAL EXPANSION COEFFICIENT OF SEAWATER WRT CONSERVATIVE TEMPERATURE
'
' SA_SI                  : ABSOLUTE SALINITY                  [KG/KG]
' T_SI                   : IN SITU TEMPERATURE                [K]
' P_SI                   : ABSOLUTE PRESSURE                  [PA]
'
' SEA_ALPHA_CT_SI        : THERMAL EXPANSION COEFFICIENT      [1/K]
'                          WRT CONSERVATIVE TEMPERATURE


'check value with default settings:
'sea_alpha_ct_si(0.035, 300, 1E8) = 3.91772847589188E-04

Dim ah As Double

sea_alpha_ct_si = ErrorReturn

ah = sea_eta_expansion_h_si(sa_si, t_si, p_si, p_ref, "t")
If ah = ErrorReturn Then Exit Function

sea_alpha_ct_si = CP0 * ah

End Function

'==========================================================================
Public Function sea_alpha_pt0_si(ByVal sa_si As Double, _
                                 ByVal t_si As Double, _
                                 ByVal p_si As Double) As Double
'==========================================================================

' THERMAL EXPANSION COEFFICIENT OF SEAWATER WRT POTENTIAL TEMPERATURE
'
' SA_SI                  : ABSOLUTE SALINITY                  [KG/KG]
' T_SI                   : IN SITU TEMPERATURE                [K]
' P_SI                   : ABSOLUTE PRESSURE                  [PA]
'
' SEA_ALPHA_PT0_SI       : THERMAL EXPANSION COEFFICIENT      [1/K]
'                          WRT POTENTIAL TEMPERATURE

'check value with default settings:
'sea_alpha_pt0_si(0.035, 300, 1e8) = 3.92515634063559E-04

sea_alpha_pt0_si = sea_eta_expansion_theta_si(sa_si, t_si, p_si, p_ref, "t")

End Function

'==========================================================================
Public Function sea_alpha_t_si(ByVal sa_si As Double, _
                               ByVal t_si As Double, _
                               ByVal p_si As Double) As Double
'==========================================================================

' THERMAL EXPANSION COEFFICIENT OF SEAWATER WRT IN SITU TEMPERATURE
'
' SA_SI                  : ABSOLUTE SALINITY                  [KG/KG]
' T_SI                   : IN SITU TEMPERATURE                [K]
' P_SI                   : ABSOLUTE PRESSURE                  [PA]
'
' SEA_ALPHA_T_SI         : THERMAL EXPANSION COEFFICIENT      [1/K]
'                          WRT IN SITU TEMPERATURE

'check value with default settings:
'sea_alpha_t_si(0.035, 300, 1e8) = 3.73608885177539E-04

sea_alpha_t_si = sea_g_expansion_t_si(sa_si, t_si, p_si)

End Function

'==========================================================================
Public Function sea_beta_ct_si(ByVal sa_si As Double, _
                               ByVal t_si As Double, _
                               ByVal p_si As Double) As Double
'==========================================================================

' HALINE CONTRACTION COEFFICIENT OF SEAWATER WRT CONSTANT CONSERVATIVE TEMPERATURE
'
' SA_SI                  : ABSOLUTE SALINITY                  [KG/KG]
' T_SI                   : IN SITU TEMPERATURE                [K]
' P_SI                   : ABSOLUTE PRESSURE                  [PA]
'
' RESULT                 : HALINE CONTRACTION COEFFICIENT     [KG/KG]
'                          WRT CONSTANT CONSERVATIVE TEMPERATURE

'check value with default settings:
'sea_beta_ct_si(0.035, 300, 1e8) = 0.649596383653744

sea_beta_ct_si = sea_eta_contraction_h_si(sa_si, t_si, p_si, p_ref, "t")

End Function

'==========================================================================
Public Function sea_beta_pt0_si(ByVal sa_si As Double, _
                                ByVal t_si As Double, _
                                ByVal p_si As Double) As Double
'==========================================================================

' HALINE CONTRACTION COEFFICIENT OF SEAWATER WRT CONSTANT POTENTIAL TEMPERATURE
'
' SA_SI                  : ABSOLUTE SALINITY                  [KG/KG]
' T_SI                   : IN SITU TEMPERATURE                [K]
' P_SI                   : ABSOLUTE PRESSURE                  [PA]
'
' RESULT                 : HALINE CONTRACTION COEFFICIENT     [KG/KG]
'                          WRT CONSTANT POTENTIAL TEMPERATURE

'check value with default settings:
'sea_beta_pt0_si(0.035, 300, 1e8) = 0.663973579411448

sea_beta_pt0_si = sea_eta_contraction_theta_si(sa_si, t_si, p_si, p_ref, "t")

End Function

'==========================================================================
Public Function sea_beta_t_si(ByVal sa_si As Double, _
                              ByVal t_si As Double, _
                              ByVal p_si As Double) As Double
'==========================================================================

' HALINE CONTRACTION COEFFICIENT OF SEAWATER WRT IN SITU TEMPERATURE
'
' SA_SI                  : ABSOLUTE SALINITY                  [KG/KG]
' T_SI                   : IN SITU TEMPERATURE                [K]
' P_SI                   : ABSOLUTE PRESSURE                  [PA]
'
' RESULT                 : HALINE CONTRACTION COEFFICIENT     [KG/KG]
'                          WRT CONSTANT IN SITU TEMPERATURE

'check value with default settings:
'sea_beta_t_si(0.035, 300, 1e8) = 0.666238827368374

sea_beta_t_si = sea_g_contraction_t_si(sa_si, t_si, p_si)

End Function

'==========================================================================
Public Function sea_ptmp0_from_ctmp_si(ByVal sa_si As Double, _
                                       ByVal ct_si As Double) As Double
'==========================================================================

' potential temperature of seawater from conservative temperature
'
' sa_si               : Absolute Salinity                  [kg/kg]
' ct_si               : conservative temperature           [K]
'
' result              : potential temperature with         [K]
'                       reference pressure of  101325 Pa

'Checkvalue: sea_ptmp0_from_ctmp_si(0.035, 0.300010069445E+03) =   299.999999999654

Dim nloops&, n&
Dim s1#, ct1#, P0#
Dim a0#, a1#, a2#, a3#, a4#, a5#, b0#, b1#, b2#, b3#
Dim a5ct#, b3ct#, ct_factor#, th0_num#, rec_th0_den#
Dim th0#, ct0#, dth_dct#, theta#, dct#, dct_dth#, factor#

sea_ptmp0_from_ctmp_si = ErrorReturn

s1 = sa_si * 35# / 35.16504: ct1 = ct_si - 273.15: P0 = 101325#

a0 = -1.44601364634479E-02:     b0 = 1#
a1 = -3.30530899585292E-03:     b1 = 6.5060971156358E-04
a2 = 1.06241592912898E-04:      b2 = 3.8302894868509E-03
a3 = 0.947756667379449:         b3 = 1.24781176036803E-06
a4 = 2.16659194773661E-03
a5 = 3.8288429550399E-03

a5ct = a5 * ct1: b3ct = b3 * ct1

ct_factor = (a3 + a4 * s1 + a5ct)

th0_num = a0 + s1 * (a1 + a2 * s1) + ct1 * ct_factor

rec_th0_den = 1# / (b0 + b1 * s1 + ct1 * (b2 + b3ct))

th0 = th0_num * rec_th0_den

ct0 = sea_ctmp_from_ptmp0_si(sa_si, 273.15 + th0)

dth_dct = (ct_factor + a5ct - (b2 + b3ct + b3ct) * th0) * rec_th0_den

theta = th0 - (ct0 - ct_si) * dth_dct


nloops = 1                  ' default

'    NOTE: nloops = 1 gives theta with a maximum error of &

n = 0

Do While n <= nloops
  factor = sea_ctmp_from_ptmp0_si(sa_si, 273.15 + theta)
  If factor = ErrorReturn Then Exit Function
  dct = factor - ct_si

  factor = sea_g_si(0, 2, 0, sa_si, 273.15 + theta, P0)
  If factor = ErrorReturn Then Exit Function
  dct_dth = -(theta + 273.15) * factor / CP0

  theta = theta - dct / dct_dth
  n = n + 1
Loop

sea_ptmp0_from_ctmp_si = theta + 273.15

End Function

'==========================================================================
Public Function sea_cabb_ct_si#(ByVal sa_si#, ByVal t_si#, ByVal p_si#)
'==========================================================================

' cabbeling coefficient of seawater wrt conservative temperature
'
' sa_si                  : Absolute Salinity                  [kg/kg]
' t_si                   : in situ temperature                [K]
' p_si                   : sea (gauge) pressure               [Pa]
'
' result                 : cabbeling coefficient              [1/(K**2)]
'                          wrt conservative temperature

'check value:
'sea_cabb_ct_si(0.035, 300, 1E5) = 8.612525|67438267E-06  'DEFAULT TOLERANCE
'sea_cabb_ct_si(0.035, 300, 1E5) = 8.612525|64750277E-06  'TOLERANCE RESET TO -1D-12

Dim th#, dct#, th_l#, th_u#, t_l#, t_u#, alpha#, beta#, alpha_ct#, ratio#
Dim ct#, ct_l#, ct_u#, dsa#, sa_l#, sa_u#, alpha_sa#, beta_sa#, factor1#, factor2#

sea_cabb_ct_si = ErrorReturn

th = sea_pottemp_si(sa_si, t_si, p_si, p_ref) ' NOTE: ct = const <==> theta = const
'here, p_ref is commonly specified at the module level
If th = ErrorReturn Then Exit Function
ct = sea_ctmp_from_ptmp0_si(sa_si, th)
If ct = ErrorReturn Then Exit Function

'ct derivative: The modification made here is to ensure centered differences
dct = 0.001
ct_l = ct - dct
ct_u = ct + dct

th_l = sea_ptmp0_from_ctmp_si(sa_si, ct_l)
If th_l = ErrorReturn Then Exit Function
th_u = sea_ptmp0_from_ctmp_si(sa_si, ct_u)
If th_u = ErrorReturn Then Exit Function

t_l = sea_pottemp_si(sa_si, th_l, p_ref, p_si)
If t_l = ErrorReturn Then Exit Function
t_u = sea_pottemp_si(sa_si, th_u, p_ref, p_si)
If t_u = ErrorReturn Then Exit Function

alpha = sea_alpha_ct_si(sa_si, t_si, p_si)
If alpha = ErrorReturn Then Exit Function
beta = sea_beta_ct_si(sa_si, t_si, p_si)
If beta = ErrorReturn Or beta = 0 Then Exit Function
ratio = alpha / beta

factor1 = sea_alpha_ct_si(sa_si, t_u, p_si)
If (factor1 = ErrorReturn) Then Exit Function
factor2 = sea_alpha_ct_si(sa_si, t_l, p_si)
If (factor2 = ErrorReturn) Then Exit Function

'calculate d(alpha_ct)/dct
If (ct_u = ct_l) Then Exit Function
alpha_ct = (factor1 - factor2) / (ct_u - ct_l)

'sa derivatives

'dsa = Min(sa_si, 1E-5):
dsa = IIf(sa_si < 0.00001, sa_si, 0.00001)

If (sa_si > dsa) Then
  sa_l = sa_si - dsa
  sa_u = sa_si + dsa
ElseIf (sa_si >= 0#) Then
  sa_l = 0#
  sa_u = dsa
Else
  Exit Function
End If

If (sa_u = sa_l) Then Exit Function

factor1 = sea_alpha_ct_si(sa_u, t_si, p_si)
If (factor1 = ErrorReturn) Then Exit Function
factor2 = sea_alpha_ct_si(sa_l, t_si, p_si)
If (factor2 = ErrorReturn) Then Exit Function
alpha_sa = (factor1 - factor2) / (sa_u - sa_l)

factor1 = sea_beta_ct_si(sa_u, t_si, p_si)
If (factor1 = ErrorReturn) Then Exit Function
factor2 = sea_beta_ct_si(sa_l, t_si, p_si)
If (factor2 = ErrorReturn) Then Exit Function
beta_sa = (factor1 - factor2) / (sa_u - sa_l)

'cabbeling coefficient

sea_cabb_ct_si = alpha_ct + ratio * (2# * alpha_sa - ratio * beta_sa)

End Function

'==========================================================================
Public Function sea_cabb_pt0_si#(ByVal sa_si#, ByVal t_si#, ByVal p_si#)
'==========================================================================

' cabbeling coefficient of seawater wrt potential temperature
'
' sa_si                  : Absolute Salinity                  [kg/kg]
' t_si                   : in situ temperature                [K]
' p_si                   : sea (gauge) pressure               [Pa]
'
' result                 : cabbeling coefficient              [1/(K**2)]
'                          wrt potential temperature

'check value:
'sea_cabb_pt0_si(0.035, 300, 1E5) = 8.3387453|7690444E-06  'DEFAULT TOLERANCES
'sea_cabb_pt0_si(0.035, 300, 1E5) = 8.3387453|0126243E-06  'TOLERANCE RESET TO -1D-12

Dim th#, dth#, th_l#, th_u#, t_l#, t_u#, alpha#, beta#, alpha_pt0#, ratio#
Dim dsa#, sa_l#, sa_u#, alpha_sa#, beta_sa#, factor1#, factor2#

sea_cabb_pt0_si = ErrorReturn

th = sea_pottemp_si(sa_si, t_si, p_si, p_ref):
If (th = ErrorReturn) Then Exit Function

'th derivative
dth = 0.001
th_l = th - dth
th_u = th + dth

t_l = sea_pottemp_si(sa_si, th_l, p_ref, p_si)
If t_l = ErrorReturn Then Exit Function
t_u = sea_pottemp_si(sa_si, th_u, p_ref, p_si)
If t_u = ErrorReturn Or th_u = th_l Then Exit Function

alpha = sea_alpha_pt0_si(sa_si, t_si, p_si)
If alpha = ErrorReturn Then Exit Function
beta = sea_beta_pt0_si(sa_si, t_si, p_si)
If beta = ErrorReturn Or beta = 0 Then Exit Function
ratio = alpha / beta

factor1 = sea_alpha_pt0_si(sa_si, t_u, p_si)
If (factor1 = ErrorReturn) Then Exit Function
factor2 = sea_alpha_pt0_si(sa_si, t_l, p_si)
If (factor2 = ErrorReturn) Then Exit Function
alpha_pt0 = (factor1 - factor2) / (th_u - th_l)

'sa derivatives
'dsa = Min(sa_si, 1E-5):
dsa = IIf(sa_si < 0.00001, sa_si, 0.00001)
If (sa_si >= dsa) Then
  sa_l = sa_si - dsa
  sa_u = sa_si + dsa:
ElseIf (sa_si >= 0#) Then
  sa_l = 0#
  sa_u = dsa
Else
  Exit Function
End If
If sa_u = sa_l Then Exit Function

factor1 = sea_alpha_pt0_si(sa_u, t_si, p_si)
If (factor1 = ErrorReturn) Then Exit Function
factor2 = sea_alpha_pt0_si(sa_l, t_si, p_si)
If (factor2 = ErrorReturn) Then Exit Function

alpha_sa = (factor1 - factor2) / (sa_u - sa_l)

beta_sa = (sea_beta_pt0_si(sa_u, t_si, p_si) - sea_beta_pt0_si(sa_l, t_si, p_si)) / (sa_u - sa_l)

'cabbeling coefficient
sea_cabb_pt0_si = alpha_pt0 + ratio * (2# * alpha_sa - ratio * beta_sa)

End Function

'==========================================================================
Public Function sea_thrmb_ct_si#(ByVal sa_si#, ByVal t_si#, ByVal p_si#)
'==========================================================================

' thermobaric coefficient of seawater wrt conservative temperature
'
' sa_si                  : Absolute Salinity                  [kg/kg]
' t_si                   : in situ temperature                [K]
' p_si                   : sea (gauge) pressure               [Pa]
'
' result                 : thermobaric coefficient            [1/(K Pa)]
'                          wrt conservative temperature

'check value:
'sea_thrmb_ct_si(0.035, 300, 1E5) = 1.4810927|1668362E-12  'DEFAULT TOLERANCES
'sea_thrmb_ct_si(0.035, 300, 1E5) = 1.4810927|5172403E-12  'TOLERANCE RESET TO -1D-12

Dim theta#, dp#, p_l#, p_u#, t_l#, t_u#, alpha#, beta#, alpha_p#, beta_p#
Dim alpha_l#, alpha_u#, beta_l#, beta_u#

sea_thrmb_ct_si = ErrorReturn
 
dp = 1000#

theta = sea_pottemp_si(sa_si, t_si, p_si, p_ref)
'here, p_ref is commonly specified at the module level
If theta = ErrorReturn Then Exit Function

If (p_si >= dp) Then
  p_l = p_si - dp
  p_u = p_si + dp
Else
  p_l = 0#
  p_u = dp
End If
If p_u = p_l Then Exit Function

t_l = sea_pottemp_si(sa_si, theta, p_ref, p_l) ' ct = const ==> theta = const
If t_l = ErrorReturn Then Exit Function
t_u = sea_pottemp_si(sa_si, theta, p_ref, p_u)
If t_u = ErrorReturn Then Exit Function

alpha = sea_alpha_ct_si(sa_si, t_si, p_si)
If alpha = ErrorReturn Then Exit Function

beta = sea_beta_ct_si(sa_si, t_si, p_si)
If beta = ErrorReturn Then Exit Function

alpha_u = sea_alpha_ct_si(sa_si, t_u, p_u)
If (alpha_u = ErrorReturn) Then Exit Function
beta_u = sea_beta_ct_si(sa_si, t_u, p_u)
If (beta_u = ErrorReturn Or beta_u = 0) Then Exit Function

alpha_l = sea_alpha_ct_si(sa_si, t_l, p_l)
If (alpha_l = ErrorReturn) Then Exit Function
beta_l = sea_beta_ct_si(sa_si, t_l, p_l)
If (beta_l = ErrorReturn Or beta_l = 0) Then Exit Function

sea_thrmb_ct_si = beta * (alpha_u / beta_u - alpha_l / beta_l) / (p_u - p_l)

End Function

'==========================================================================
Public Function sea_thrmb_pt0_si#(ByVal sa_si#, ByVal t_si#, ByVal p_si#)
'==========================================================================

' thermobaric coefficient of seawater wrt potential temperature
'
' sa_si                  : Absolute Salinity                  [kg/kg]
' t_si                   : in situ temperature                [K]
' p_si                   : sea (gauge) pressure               [Pa]
'
' result                 : thermobaric coefficient            [1/(K Pa)]
'                          wrt potential temperature

'check value:
'sea_thrmb_pt0_si(0.035, 300, 1E5) = 1.4594101|0702991E-12  'DEFAULT TOLERANCES
'sea_thrmb_pt0_si(0.035, 300, 1E5) = 1.4594101|3482853E-12  'TOLERANCE RESET TO -1D-12

Dim theta#, dp#, p_l#, p_u#, t_l#, t_u#, alpha#, beta#, alpha_p#, beta_p#
Dim alpha_l#, alpha_u#, beta_l#, beta_u#

sea_thrmb_pt0_si = ErrorReturn

dp = 1000#

theta = sea_pottemp_si(sa_si, t_si, p_si, p_ref)
'here, p_ref is commonly specified at the module level
If theta = ErrorReturn Then Exit Function

If (p_si >= dp) Then
  p_l = p_si - dp
  p_u = p_si + dp:
Else
  p_l = 0#
  p_u = dp
End If
If p_u = p_l Then Exit Function

t_l = sea_pottemp_si(sa_si, theta, p_ref, p_l)
If (t_l = ErrorReturn) Then Exit Function

t_u = sea_pottemp_si(sa_si, theta, p_ref, p_u)
If (t_u = ErrorReturn) Then Exit Function

beta = sea_beta_pt0_si(sa_si, t_si, p_si)
If (beta = ErrorReturn) Then Exit Function

alpha_u = sea_alpha_pt0_si(sa_si, t_u, p_u)
If (alpha_u = ErrorReturn) Then Exit Function
beta_u = sea_beta_pt0_si(sa_si, t_u, p_u)
If (beta_u = ErrorReturn Or beta_u = 0) Then Exit Function

alpha_l = sea_alpha_pt0_si(sa_si, t_l, p_l)
If (alpha_l = ErrorReturn) Then Exit Function
beta_l = sea_beta_pt0_si(sa_si, t_l, p_l)
If (beta_l = ErrorReturn Or beta_l = 0) Then Exit Function

sea_thrmb_pt0_si = beta * (alpha_u / beta_u - alpha_l / beta_l) / (p_u - p_l)

End Function



 
