Attribute VB_Name = "Ice_Liq_4_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules:
'     Constants_0_Mdl, file Constants_0.bas
'     Flu_1_Mdl,       file Flu_1.bas
'     Ice_1_Mdl,       file Ice_1.bas
'     Flu_2_Mdl,       file Flu_2.bas
'     Ice_2_Mdl,       file Ice_2.bas
'     Maths_0_Mdl,     file Maths_0.bas

'#########################################################################

'=========================================================================
'This module implements the phase equilibrium properties of ice with liquid water
'computed from IAPWS-95 and IAPWS-06:

'Release on the IAPWS Formulation 1995 for the Thermodynamic Properties of
'Ordinary Water Substance for General and Scientific Use
'The International Association for the Properties of Water and Steam
'Fredericia, Denmark, September 1996

'Release on an Equation of State for H2O Ice Ih
'The International Association for the Properties of Water and Steam
'Witney, UK, September 2006

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98
'Private Const IsOK = -1

'Control parameters of the liquid pressure iteration
Private ctrl_initialized As Integer

Private ctrl_mode_ice As Integer
Private ctrl_mode_liquid As Integer
Private ctrl_mode_temperature As Integer
Private ctrl_loop_maximum As Long
Private ctrl_init_d_liq As Double
Private ctrl_init_t As Double
Private ctrl_eps_exit_p As Double

'Properties of the current ice-liquid equilibrium state
Private equi_ice_liq_done As Integer

Private equi_ice_liq_t As Double
Private equi_ice_liq_p As Double
Private equi_ice_liq_g_ice As Double
Private equi_ice_liq_g_liq As Double
Private equi_ice_liq_d_ice As Double
Private equi_ice_liq_d_liq As Double

Private Const Version = "16 Sep 2009"

'==========================================================================
Public Function ice_liq_meltingpressure_si(ByVal t_si As Double) As Double

'returns the absolute pressure of water at liquid-ice equilibrium in Pa
'as a function of absolute temperature t_si in K

'note: the accuracy of this function depends on the iteration settings in ths module

'check value with default settings: ice_liq_meltingpressure_si(270) = 39313338.8824903

ice_liq_meltingpressure_si = ErrorReturn

If set_ice_liq_eq_at_t(t_si) = ErrorReturn Then Exit Function

ice_liq_meltingpressure_si = equi_ice_liq_p

End Function

'==========================================================================
Public Function ice_liq_meltingtemperature_si(ByVal p_si As Double) As Double

'returns the absolute temperature of liquid-ice equilibrium in K
'as a function of absolute pressure p_si in Pa

'note: the accuracy of this function depends on the iteration settings in ths module

'check value with default settings: ice_liq_meltingtemperature_si(1E7) = 272.401648869143

ice_liq_meltingtemperature_si = ErrorReturn

If set_ice_liq_eq_at_p(p_si) = ErrorReturn Then Exit Function

ice_liq_meltingtemperature_si = equi_ice_liq_t

End Function

'==========================================================================
Public Function ice_liq_pressure_liq_si() As Double

'returns the absolute pressure in Pa of water at liquid-ice equilibrium
'after execution of set_ice_liq_eq_at_t or set_ice_liq_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_ice_liq_eq_at_p 1E7
'ice_liq_pressure_liq_si = 10000000.0069817

'set_ice_liq_eq_at_t 270
'ice_liq_pressure_liq_si = 39313338.8824903

ice_liq_pressure_liq_si = ErrorReturn

If equi_ice_liq_done <> IsOK Then Exit Function

ice_liq_pressure_liq_si = equi_ice_liq_p

End Function

'==========================================================================
Public Function ice_liq_temperature_si() As Double

'returns the absolute temperature in K of water at liquid-ice equilibrium
'after execution of set_ice_liq_eq_at_t or set_ice_liq_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_ice_liq_eq_at_p 1E7
'ice_liq_temperature_si = 272.401648869143

'set_ice_liq_eq_at_t 270
'ice_liq_temperature_si = 270

ice_liq_temperature_si = ErrorReturn

If equi_ice_liq_done <> IsOK Then Exit Function

ice_liq_temperature_si = equi_ice_liq_t

End Function

'==========================================================================
Public Function ice_liq_chempot_si() As Double

'returns the chemical potential in J/kg of water liquid at liquid-ice equilibrium
'after execution of set_ice_liq_eq_at_t or set_ice_liq_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_ice_liq_eq_at_p 1E7
'ice_liq_chempot_si = 9972.8817138|1369

'set_ice_liq_eq_at_t 270
'ice_liq_chempot_si = 38870.060519|24

ice_liq_chempot_si = ErrorReturn

If equi_ice_liq_done <> IsOK Then Exit Function

ice_liq_chempot_si = equi_ice_liq_g_liq


End Function

'==========================================================================
Public Function ice_liq_density_ice_si() As Double

'returns the density in kg/m3 of ice at liquid-ice equilibrium
'after execution of set_ice_liq_eq_at_t or set_ice_liq_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_ice_liq_eq_at_p 1E7
'ice_liq_density_ice_si = 917.89669083078

'set_ice_liq_eq_at_t 270
'ice_liq_density_ice_si = 921.35942851403

ice_liq_density_ice_si = ErrorReturn

If equi_ice_liq_done <> IsOK Then Exit Function

ice_liq_density_ice_si = equi_ice_liq_d_ice

End Function

'==========================================================================
Public Function ice_liq_density_liq_si() As Double

'returns the density in kg/m3 of water at liquid-ice equilibrium
'after execution of set_ice_liq_eq_at_t or set_ice_liq_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_ice_liq_eq_at_p 1E7
'ice_liq_density_liq_si = 1004.793536602

'set_ice_liq_eq_at_t 270
'ice_liq_density_liq_si = 1019.05568894323

ice_liq_density_liq_si = ErrorReturn

If equi_ice_liq_done <> IsOK Then Exit Function

ice_liq_density_liq_si = equi_ice_liq_d_liq

End Function

'==========================================================================
Public Function ice_liq_entropy_ice_si() As Double

'returns the specific entropy in J/(kg K) of ice at liquid-ice equilibrium
'after execution of set_ice_liq_eq_at_t or set_ice_liq_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_ice_liq_eq_at_p 1E7
'ice_liq_entropy_ice_si = -1228.2446413855

'set_ice_liq_eq_at_t 270
'ice_liq_entropy_ice_si = -1251.57777462393

ice_liq_entropy_ice_si = ErrorReturn

If equi_ice_liq_done <> IsOK Then Exit Function

ice_liq_entropy_ice_si = ice_entropy_si(equi_ice_liq_t, equi_ice_liq_p)

End Function

'==========================================================================
Public Function ice_liq_entropy_liq_si() As Double

'returns the specific entropy in J/(kg K) of water at liquid-ice equilibrium
'after execution of set_ice_liq_eq_at_t or set_ice_liq_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_ice_liq_eq_at_p 1E7
'ice_liq_entropy_liq_si = -11.112101272|3401

'set_ice_liq_eq_at_t 270
'ice_liq_entropy_liq_si = -47.2567126290783

ice_liq_entropy_liq_si = ErrorReturn

If equi_ice_liq_done <> IsOK Then Exit Function

ice_liq_entropy_liq_si = flu_entropy_si(equi_ice_liq_t, equi_ice_liq_d_liq)

End Function

'==========================================================================
Public Function ice_liq_enthalpy_ice_si() As Double

'returns the specific enthalpy in J/kg of ice at liquid-ice equilibrium
'after execution of set_ice_liq_eq_at_t or set_ice_liq_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_ice_liq_eq_at_p 1E7
'ice_liq_enthalpy_ice_si = -324602.983812453

'set_ice_liq_eq_at_t 270
'ice_liq_enthalpy_ice_si = -299055.938629222

ice_liq_enthalpy_ice_si = ErrorReturn

If equi_ice_liq_done <> IsOK Then Exit Function

ice_liq_enthalpy_ice_si = ice_enthalpy_si(equi_ice_liq_t, equi_ice_liq_p)

End Function

'==========================================================================
Public Function ice_liq_enthalpy_liq_si() As Double

'returns the specific enthalpy in J/kg of water at liquid-ice equilibrium
'after execution of set_ice_liq_eq_at_t or set_ice_liq_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_ice_liq_eq_at_p 1E7
'ice_liq_enthalpy_liq_si = 6945.9270048|2733

'set_ice_liq_eq_at_t 270
'ice_liq_enthalpy_liq_si = 26110.7481093889

ice_liq_enthalpy_liq_si = ErrorReturn

If equi_ice_liq_done <> IsOK Then Exit Function

ice_liq_enthalpy_liq_si = flu_enthalpy_si(equi_ice_liq_t, equi_ice_liq_d_liq)

End Function

'==========================================================================
Public Function ice_liq_volume_melt_si() As Double

'returns the specific melting volume in m3/kg at liquid-ice equilibrium
'after execution of set_ice_liq_eq_at_t or set_ice_liq_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_ice_liq_eq_at_p 1E7
'ice_liq_volume_melt_si = -9.4217890328823E-05

'set_ice_liq_eq_at_t 270
'ice_liq_volume_melt_si = -1.04052121182316E-04

ice_liq_volume_melt_si = ErrorReturn

If equi_ice_liq_done <> IsOK Then Exit Function

If equi_ice_liq_d_ice <= 0 Then Exit Function
If equi_ice_liq_d_liq <= 0 Then Exit Function

ice_liq_volume_melt_si = 1# / equi_ice_liq_d_liq - 1# / equi_ice_liq_d_ice

End Function

'==========================================================================
Public Function ice_liq_entropy_melt_si() As Double

'returns the specific melting entropy in J/(kg K) of water
'after execution of set_ice_liq_eq_at_t or set_ice_liq_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_ice_liq_eq_at_p 1E7
'ice_liq_entropy_melt_si = 1217.13254011316

'set_ice_liq_eq_at_t 270
'ice_liq_entropy_melt_si = 1204.32106199485

Dim si As Double, sl As Double

ice_liq_entropy_melt_si = ErrorReturn

If equi_ice_liq_done <> IsOK Then Exit Function

si = ice_entropy_si(equi_ice_liq_t, equi_ice_liq_p)
If si = ErrorReturn Then Exit Function
sl = flu_entropy_si(equi_ice_liq_t, equi_ice_liq_d_liq)
If sl = ErrorReturn Then Exit Function

ice_liq_entropy_melt_si = sl - si

End Function

'==========================================================================
Public Function ice_liq_enthalpy_melt_si() As Double

'returns the specific melting enthalpy of water in J/kg
'after execution of set_ice_liq_eq_at_t or set_ice_liq_eq_at_p

'note: the accuracy of this function depends on the iteration settings in ths module

'check values with default settings:
'set_ice_liq_eq_at_p 1E7
'ice_liq_enthalpy_melt_si = 331548.910817281

'set_ice_liq_eq_at_t 270
'ice_liq_enthalpy_melt_si = 325166.68673861

Dim hi As Double, hl As Double

ice_liq_enthalpy_melt_si = ErrorReturn

If equi_ice_liq_done <> IsOK Then Exit Function

hi = ice_enthalpy_si(equi_ice_liq_t, equi_ice_liq_p)
If hi = ErrorReturn Then Exit Function
hl = flu_enthalpy_si(equi_ice_liq_t, equi_ice_liq_d_liq)
If hl = ErrorReturn Then Exit Function

ice_liq_enthalpy_melt_si = hl - hi

End Function

'==========================================================================
Public Function set_ice_liq_eq_at_p(ByVal p_si As Double) As Double

'this function computes the ice-liquid equilibrium at given pressure
'by 2-dimensional Newton iteration

'triple point
Const dl_triple = TP_density_liq_IAPWS95_si
Const t_triple = TP_temperature_si

Dim t As Double
Dim pl As Double
Dim gi As Double, gl As Double
Dim di As Double, dl As Double

Dim fl_d As Double, eps As Double
Dim maxit As Long

If equi_ice_liq_done = IsOK And _
   p_si = equi_ice_liq_p Then
  'the requested state has already been computed earlier
  set_ice_liq_eq_at_p = IsOK
  Exit Function
End If

clear_ice_liq_state 'waste any previous state

set_ice_liq_eq_at_p = ErrorReturn

If p_si <= 0 Then Exit Function

init_it_ctrl_ice_liq

'set initial temperature guess
Select Case ctrl_mode_temperature
  Case 0:  t = aux_temperature_correlation(p_si)
           If t = ErrorReturn Then Exit Function
  Case -1: t = t_triple
  Case 1:  t = ctrl_init_t
  Case Else: Exit Function
End Select

'set initial liquid density guess
Select Case ctrl_mode_liquid
  Case 0:  dl = aux_liq_density_correlation_si(t)
  Case -1: dl = dl_triple
  Case 1:  dl = ctrl_init_d_liq
  Case Else: Exit Function
End Select

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: pl = flu_pressure_si(t, dl)
           gi = ice_g_si(0, 0, t, pl)
           gl = flu_gibbs_energy_si(t, dl)
           di = ice_g_si(0, 1, t, pl)
           If di = ErrorReturn Or di <= 0 Then Exit Function
           di = 1# / di
           set_ice_liq_state t, pl, gi, gl, di, dl
           set_ice_liq_eq_at_p = IsOK
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit
eps = ctrl_eps_exit_p
If eps = 0 Then Exit Function

'run iteration loop
If ice_liq_iteration_at_p(p_si, maxit, eps, t, di, dl, gi, gl, pl) = ErrorReturn Then
  Exit Function
End If

set_ice_liq_state t, pl, gi, gl, di, dl

set_ice_liq_eq_at_p = IsOK

End Function

'==========================================================================
Public Function set_ice_liq_eq_at_t(ByVal t_si As Double) As Double

'this function computes the ice-liquid equilibrium at given temperature
'by Newton iteration

'triple point
Const t_triple = TP_temperature_si
Const dl_triple = TP_density_liq_IAPWS95_si

Dim pl As Double
Dim gi As Double, gl As Double
Dim di As Double, dl As Double

Dim fl_d As Double, eps As Double
Dim maxit As Long

If equi_ice_liq_done = IsOK And _
   t_si = equi_ice_liq_t Then
  'the requested state has already been computed earlier
  set_ice_liq_eq_at_t = IsOK
  Exit Function
End If

clear_ice_liq_state 'waste any previous state

set_ice_liq_eq_at_t = ErrorReturn

If t_si <= 0 Then Exit Function
If t_si > t_triple Then Exit Function

init_it_ctrl_ice_liq

'set initial liquid density guess
Select Case ctrl_mode_liquid
  Case 0:  dl = aux_liq_density_correlation_si(t_si)
  Case -1: dl = dl_triple
  Case 1:  dl = ctrl_init_d_liq
  Case Else: Exit Function
End Select

'set max. iteration number
Select Case ctrl_loop_maximum
  Case 0: maxit = 100
  Case -1: pl = flu_pressure_si(t_si, dl)
           gi = ice_g_si(0, 0, t_si, pl)
           gl = flu_gibbs_energy_si(t_si, dl)
           di = ice_g_si(0, 1, t_si, pl)
           If di = ErrorReturn Or di <= 0 Then Exit Function
           di = 1# / di
           set_ice_liq_state t_si, pl, gi, gl, di, dl
           set_ice_liq_eq_at_t = IsOK
           Exit Function
  Case Is > 0: maxit = ctrl_loop_maximum
  Case Else: Exit Function
End Select

'set iteration accuracy limit
eps = ctrl_eps_exit_p
If eps = 0 Then Exit Function

'run iteration loop
If ice_liq_iteration_at_t(t_si, maxit, eps, di, dl, gi, gl, pl) = ErrorReturn Then
  Exit Function
End If

set_ice_liq_state t_si, pl, gi, gl, di, dl

set_ice_liq_eq_at_t = IsOK

End Function

'==========================================================================
Private Function ice_liq_iteration_at_p(ByVal p_si As Double, _
                                        ByVal maxit As Long, _
                                        ByVal eps As Double, _
                                        ByRef t_si As Double, _
                                        ByRef d_ice_si As Double, _
                                        ByRef d_liq_si As Double, _
                                        ByRef g_ice_si As Double, _
                                        ByRef g_liq_si As Double, _
                                        ByRef p_liq_si As Double) As Double

'this function returns the ice-liquid phase equilibrium from equal pressures,
'temperatures and chemical potentials of the two phases at given pressure, p_si,
'from initial guesses for the liquid density, d_liq_si, and the temperature, t_si
'The iteration limit eps refers to the error in liquid pressure

'output:    ice_liq_iteration_at_p = IsOK if successful
'           ice_liq_iteration_at_p = ErrorReturn is returned if
'           - the maximum number of iterations is exceeded without meeting the exit criterion
'           - the function call to flu_f_si or to ice_g_si has returned an error
'           - density or temperature has taken a zero or negative value during the iteration
'     t_si: absolute temperature in K
' d_ice_si: ice density at the ice-liquid equilibrium in kg/m3
' d_liq_si: liquid density at the ice-liquid equilibrium in kg/m3
' g_ice_si: ice chemical potential at the ice-liquid equilibrium in J/kg
' g_liq_si: liquid chemical potential at the ice-liquid equilibrium in J/kg
' p_liq_si: liquid pressure at the ice-liquid equilibrium in Pa

'input: p_si: absolute pressure in Pa
'      maxit: maximum number of iteration steps to be done
'        eps: required accuracy of liquid pressure
'             eps > 0: absolute liquid pressure tolerance in Pa
'             eps < 0: relative liquid pressure tolerance
'       t_si: initial guess of absolute temperature in K
'   d_liq_si: initial guess of liquid density in kg/m3

Const Tt = TP_temperature_si 'triple point temperature in K

Dim vi As Double, dl As Double
Dim pl As Double, gl As Double
Dim gi As Double, gi_t As Double
Dim t As Double
Dim fl As Double, fl_d As Double, fl_dd As Double
Dim fl_t As Double, fl_td As Double

Dim it As Long

Dim a(2, 2) As Double, b(2) As Double, x(2) As Double
Dim ddl As Double, dt As Double, pl_old As Double

ice_liq_iteration_at_p = ErrorReturn

If t_si <= 0 Or d_liq_si <= 0 Then
  t_si = ErrorReturn
  d_ice_si = ErrorReturn
  d_liq_si = ErrorReturn
  g_ice_si = ErrorReturn
  g_liq_si = ErrorReturn
  p_liq_si = ErrorReturn
  Exit Function
End If

If check_limits = 1 Then
  'ICE_LIMITS
  If p_si <= ice_pmin Or p_si > ice_pmax Then
    t_si = ErrorReturn
    d_ice_si = ErrorReturn
    d_liq_si = ErrorReturn
    g_ice_si = ErrorReturn
    g_liq_si = ErrorReturn
    p_liq_si = ErrorReturn
    Exit Function
  End If
Else
  If p_si <= 0 Then
    t_si = ErrorReturn
    d_ice_si = ErrorReturn
    d_liq_si = ErrorReturn
    g_ice_si = ErrorReturn
    g_liq_si = ErrorReturn
    p_liq_si = ErrorReturn
    Exit Function
  End If
End If

check_limits = check_limits - 1

dl = d_liq_si
t = t_si

For it = 0 To maxit

  'Derivatives of the Gibbs & Helmholtz functions for 2D-Newton iteration
  fl = flu_f_si(0, 0, t, dl)
  If fl = ErrorReturn Then Exit For
  
  fl_d = flu_f_si(0, 1, t, dl)
  If fl_d = ErrorReturn Then Exit For
  fl_dd = flu_f_si(0, 2, t, dl)
  If fl_dd = ErrorReturn Then Exit For
  
  fl_t = flu_f_si(1, 0, t, dl)
  If fl_t = ErrorReturn Then Exit For
  fl_td = flu_f_si(1, 1, t, dl)
  If fl_td = ErrorReturn Then Exit For

  pl_old = pl  'keep previous pl to check accuracy
  'liquid pressure and ice + liquid chemical potentials
  pl = dl ^ 2 * fl_d     'pressure of liquid
  gl = fl + dl * fl_d    'chem. pot. of liquid
  
  gi = ice_g_si(0, 0, t, pl)
  If gi = ErrorReturn Then Exit For
  gi_t = ice_g_si(1, 0, t, pl)
  If gi_t = ErrorReturn Then Exit For

  If it > 0 Then
    'check absolute or relative error limit
    If (eps > 0 And Abs(pl - pl_old) < eps) Or _
       (eps < 0 And Abs(pl - pl_old) < -eps * pl) Then
      p_liq_si = pl
      g_ice_si = gi
      g_liq_si = gl
      vi = ice_g_si(0, 1, t, pl)
      If vi = ErrorReturn Or vi <= 0 Then Exit For
      d_ice_si = 1# / vi
      d_liq_si = dl
      t_si = t
      ice_liq_iteration_at_p = IsOK
      Exit For
    End If
  End If
  
  If it = maxit Then Exit For
  
  'coefficient matrix
  a(1, 1) = dl * (2 * fl_d + dl * fl_dd)
  a(1, 2) = dl ^ 2 * fl_td
  a(2, 1) = (2 * fl_d + dl * fl_dd)
  a(2, 2) = -gi_t + fl_t + dl * fl_td

  'right-hand sides, must vanish in equilibrium
  b(1) = p_si - pl
  b(2) = gi - gl

  'solve equations
  If matrix_solve(a(), b(), x(), 2) <> 0 Then Exit For 'matrix singular
  ddl = x(1)
  dt = x(2)
  
  'update density & temperature
  dl = dl + ddl
  If dl <= 0 Then Exit For
  t = t + dt
  If t <= 0 Then Exit For
  
Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Or _
     p_liq_si <= 0 Or _
     d_liq_si <= flu_dmin Or d_liq_si > flu_dmax Then
    t_si = ErrorReturn
    d_ice_si = ErrorReturn
    d_liq_si = ErrorReturn
    g_ice_si = ErrorReturn
    g_liq_si = ErrorReturn
    p_liq_si = ErrorReturn
    ice_liq_iteration_at_p = ErrorReturn
    Exit Function
  End If
  'ICE_LIMITS
  If t_si <= ice_tmin Or t_si > ice_tmax Or _
     d_ice_si <= 0 Then
    t_si = ErrorReturn
    d_ice_si = ErrorReturn
    d_liq_si = ErrorReturn
    g_ice_si = ErrorReturn
    g_liq_si = ErrorReturn
    p_liq_si = ErrorReturn
    ice_liq_iteration_at_p = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Function ice_liq_iteration_at_t(ByVal t_si As Double, _
                                        ByVal maxit As Long, _
                                        ByVal eps As Double, _
                                        ByRef d_ice_si As Double, _
                                        ByRef d_liq_si As Double, _
                                        ByRef g_ice_si As Double, _
                                        ByRef g_liq_si As Double, _
                                        ByRef p_liq_si As Double) As Double

'this function returns the ice-liquid phase equilibrium from equal pressures
'and chemical potentials of the two phases at given temperature, t_si,
'from an initial guess for the liquid density, d_liq_si.
'The iteration limit eps refers to the error in liquid pressure

'output:    ice_liq_iteration_at_t = IsOK if successfully done
'           ice_liq_iteration_at_t = ErrorReturn is returned if
'           - the maximum number of iterations is exceeded without meeting the exit criterion
'           - the function call to flu_f_si has returned an error
'           - density has taken a zero or negative value during the iteration
' d_liq_si: liquid density at the ice-liquid equilibrium in kg/m3
' d_ice_si: ice density at the ice-liquid equilibrium in kg/m3
' g_liq_si: liquid chemical potential at the ice-liquid equilibrium in J/kg
' g_ice_si: ice chemical potential at the ice-liquid equilibrium in J/kg
' p_liq_si: liquid pressure at the ice-liquid equilibrium in Pa

'input: t_si: absolute temperature in K
'      maxit: maximum number of iteration steps to be done
'        eps: required accuracy of pressure
'             eps > 0: absolute pressure tolerance in kg/m^3
'             eps < 0: relative pressure tolerance
'   d_liq_si: initial guess of liquid density in kg/m3

Dim vi As Double, dl As Double
Dim pl As Double
Dim gi As Double, gl As Double
Dim fl As Double, fl_d As Double, fl_dd As Double

Dim it As Long

Dim ddl As Double, pl_old As Double

ice_liq_iteration_at_t = ErrorReturn

If t_si <= 0 Then Exit Function

If d_liq_si <= 0 Then
  d_ice_si = ErrorReturn
  d_liq_si = ErrorReturn
  g_ice_si = ErrorReturn
  g_liq_si = ErrorReturn
  p_liq_si = ErrorReturn
  Exit Function
End If

If check_limits = 1 Then
  'FLU_LIMITS
  If t_si < flu_tmin Or t_si > flu_tmax Then
    d_ice_si = ErrorReturn
    d_liq_si = ErrorReturn
    g_ice_si = ErrorReturn
    g_liq_si = ErrorReturn
    p_liq_si = ErrorReturn
    Exit Function
  End If
  'ICE_LIMITS
  If t_si <= ice_tmin Or t_si > ice_tmax Then
    d_ice_si = ErrorReturn
    d_liq_si = ErrorReturn
    g_ice_si = ErrorReturn
    g_liq_si = ErrorReturn
    p_liq_si = ErrorReturn
    Exit Function
  End If
Else
  If t_si <= 0 Then
    d_ice_si = ErrorReturn
    d_liq_si = ErrorReturn
    g_ice_si = ErrorReturn
    g_liq_si = ErrorReturn
    p_liq_si = ErrorReturn
    Exit Function
  End If
End If

check_limits = check_limits - 1

dl = d_liq_si

For it = 0 To maxit

  'Derivatives of the Helmholtz function for Newton iteration
  fl = flu_f_si(0, 0, t_si, dl)
  If fl = ErrorReturn Then Exit For
  fl_d = flu_f_si(0, 1, t_si, dl)
  If fl_d = ErrorReturn Then Exit For
  fl_dd = flu_f_si(0, 2, t_si, dl)
  If fl_dd = ErrorReturn Then Exit For
  
  pl_old = pl  'keep previous pl to check accuracy
  'liquid pressure and ice + liquid chemical potentials
  pl = dl ^ 2 * fl_d     'pressure of liquid
  gl = fl + dl * fl_d    'chem. pot. of liquid
  
  'Derivatives of the Gibbs function of ice for Newton iteration
  gi = ice_g_si(0, 0, t_si, pl)
  If gi = ErrorReturn Then Exit For
  vi = ice_g_si(0, 1, t_si, pl)
  If vi = ErrorReturn Then Exit For
  If vi <= 0 Then Exit For

  If it > 0 Then
    'check absolute or relative error limit
    If (eps > 0 And Abs(pl - pl_old) < eps) Or _
       (eps < 0 And Abs(pl - pl_old) < -eps * pl) Then
      ice_liq_iteration_at_t = IsOK
      p_liq_si = pl
      g_ice_si = gi
      g_liq_si = gl
      d_ice_si = 1# / vi
      d_liq_si = dl
      Exit For
    End If
  End If

  If it = maxit Then Exit For
  
  'do iteration step
  ddl = (2 * fl_d + dl * fl_dd) * (vi * dl - 1#)
  If ddl = 0 Then Exit For
  ddl = (gl - gi) / ddl
  
  'update liquid density
  dl = dl + ddl
  If dl <= 0 Then Exit For

Next it

check_limits = check_limits + 1

If check_limits = 1 Then
  'FLU_LIMITS
  If d_liq_si <= flu_dmin Or d_liq_si > flu_dmax Or _
     p_liq_si <= 0 Then
    d_ice_si = ErrorReturn
    d_liq_si = ErrorReturn
    g_ice_si = ErrorReturn
    g_liq_si = ErrorReturn
    p_liq_si = ErrorReturn
    ice_liq_iteration_at_t = ErrorReturn
    Exit Function
  End If
  'ICE_LIMITS
  If d_ice_si <= 0 Then
    d_ice_si = ErrorReturn
    d_liq_si = ErrorReturn
    g_ice_si = ErrorReturn
    g_liq_si = ErrorReturn
    p_liq_si = ErrorReturn
    ice_liq_iteration_at_t = ErrorReturn
  End If
End If

End Function

'==========================================================================
Private Sub init_it_ctrl_ice_liq()

'triple point
Const dl_triple = TP_density_liq_IAPWS95_si
Const t_triple = TP_temperature_si

If ctrl_initialized = IsOK Then Exit Sub

ctrl_initialized = IsOK

'Set default values and modes for the iteration
ctrl_mode_ice = 0
ctrl_mode_liquid = 0
ctrl_mode_temperature = 0
ctrl_loop_maximum = 100
ctrl_init_d_liq = dl_triple
ctrl_init_t = t_triple
ctrl_eps_exit_p = -0.0000001 'relative, 0.1 ppm

End Sub

'==========================================================================
Public Sub set_it_ctrl_ice_liq(ByVal key As String, ByVal value As Double)

'this sub sets control parameters for the iteration used to compute
'ice-liquid equilibrium

'key             value
'it_steps        0           set iteration number to default (100)
'it_steps        n > 0       set iteration number to n
'it_steps       -1           do not iterate, use initial liquid density to compute liquid pressure

'init_liq_dens   0           use default liquid density to start =
'                            liquid density correlation function, d = aux_liq_density_correlation_si(T)
'init_liq_dens  -1           use triple point liquid density to start
'init_liq_dens   d > 0       use value d as liquid density to start

'init_temp       0           use default temperature to start ( = aux_melt_temperature(p))
'init_temp      -1           use triple point temperature to start
'init_temp       t > 0       use value t as temperature to start

'tol_liq_press   0           use default exit tolerance for liquid pressure (0.1 ppm)
'tol_liq_press   eps         use eps as exit tolerance for liquid pressure (eps < 0 means relative error)


init_it_ctrl_ice_liq

clear_ice_liq_state

Select Case LCase(Trim(key))

  Case "it_steps":   'iteration steps
    Select Case value
      Case 0:      ctrl_loop_maximum = 100  'default = 100
      Case Is < 0: ctrl_loop_maximum = -1
      Case Else:   ctrl_loop_maximum = value
    End Select

  Case "init_liq_dens":   'start liquid density
    Select Case CLng(value)
      Case 0:       ctrl_mode_liquid = 0    'default = aux polynomial
      Case Is < -1: 'ignore it
      Case Is < 0:  ctrl_mode_liquid = value
      Case Else:    ctrl_mode_liquid = 1
                    ctrl_init_d_liq = value
    End Select

  Case "init_temp":   'start temperature
    Select Case CLng(value)
      Case 0:       ctrl_mode_temperature = 0    'default = aux polynomial
      Case Is < -1: 'ignore it
      Case Is < 0:  ctrl_mode_temperature = value
      Case Else:    ctrl_mode_temperature = 1
                    ctrl_init_t = value
    End Select

  Case "tol_liq_press":      'required liquid pressure tolerance
    Select Case value
      Case 0:      ctrl_eps_exit_p = -0.0000001   'default = 0.1 ppm relative
      Case Else:   ctrl_eps_exit_p = value
    End Select

End Select

End Sub

'==========================================================================
Private Sub set_ice_liq_state(ByVal t As Double, _
                              ByVal pl As Double, _
                              ByVal gi As Double, _
                              ByVal gl As Double, _
                              ByVal di As Double, _
                              ByVal dl As Double)

'stores the actual properties as the current equilibrium state descriptor

equi_ice_liq_done = IsOK

equi_ice_liq_t = t
equi_ice_liq_p = pl
equi_ice_liq_g_ice = gi
equi_ice_liq_g_liq = gl
equi_ice_liq_d_ice = di
equi_ice_liq_d_liq = dl

End Sub

'==========================================================================
Private Sub clear_ice_liq_state()

'clears the current equilibrium state descriptor

equi_ice_liq_done = 0

End Sub

'==========================================================================
Private Function aux_liq_density_correlation_si(ByVal t_si As Double) As Double

'this function implements the density in kg/m3 of the freezing liquid
'as a function of absolute temperature t_si in K from an approximate
'correlation fit between 252 and 273 K with rms = 1.2 E-4 in d/dt

Const Tt = TP_temperature_si          'triple point temperature of water
Const dt = TP_density_liq_IAPWS95_si  'triple point density of liquid water

Const a1 = -1.78582981492113
Const a2 = -12.2325084306734
Const a3 = -52.8236936433529

Dim tr As Double, dr As Double

aux_liq_density_correlation_si = ErrorReturn

If t_si <= 0 Then Exit Function

tr = t_si / Tt - 1#
dr = a1 * tr + a2 * tr ^ 2 + a3 * tr ^ 3

aux_liq_density_correlation_si = dt * (dr + 1#)

End Function

'==========================================================================
Private Function aux_temperature_correlation(ByVal p_si As Double) As Double

'this function implements the melting temperature in K of ice as an approximate
'function of the absolute pressure in Pa from a correlation fit
'between 273 K and 252 K with rms = 1.4E-5 in T/Tt

Const Pt = TP_pressure_IAPWS95_si  'IAPWS-95 triple point pressure in Pa
Const Tt = TP_temperature_si       'triple point temperature of water

Const a1 = -1.67329759176351E-07
Const a2 = -2.02262929999658E-13

Dim tr As Double, pr As Double

aux_temperature_correlation = ErrorReturn

If p_si <= 0 Then Exit Function

pr = p_si / Pt - 1#
tr = a1 * pr + a2 * pr ^ 2

aux_temperature_correlation = Tt * (tr + 1#)

End Function

