Attribute VB_Name = "GSW_Library_5_Mdl"
Option Explicit

'#########################################################################

'This module requires the library module
'     Constants_0_Mdl, file Constants_0.bas
'     Maths_0_Mdl,      file Maths_0.bas
'     convert_5_Mdl, file convert_5.bas
'     liq_f03_5_Mdl, file liq_f03_5.bas
'     flu_1_Mdl,     file flu_1.bas
'     flu_3a_Mdl,    file flu_3a.bas
'     sal_1_Mdl,     file sal_1.bas
'     sal_2_Mdl,     file sal_2.bas
'     sea_3a_Mdl,    file sea_3a.bas
'     sea_3b_Mdl,    file sea_3b.bas
'     sea_5a_Mdl,    file sea_5a.bas

'#########################################################################

'=====================================
'This module implements the Gibbs Seawater (GSW) Library, a subset of
'taylored functions based on the general Sea-Ice-Air (SIA) Library

'Implementation in VB6 by Rainer Feistel
'from original code in Fortran by David Jackett and Dan Wright:

'======================================
' This is the gsw thermodynamic Gibbs function library implemented in Fortran
'======================================
'
' consisiting of the sum of pure water and saline components defined by
'
' Feistel, R., 2003: A new extended Gibbs thermodynamic potential of seawater,
'                      Progr. Oceanogr., 58, 43-114.
'
' for the pure water component and by
'
' Feistel, R., 2008: A Gibbs function for seawater thermodynamics for -6 to 80C
'                      and salinity up to 120 G/KG, Deep-Sea Res. I, 55, 1639-1671, and
'
'   IAPWS 2008: Release on the IAPWS Formulation 2008 for the Thermodynamic Properties of Seawater,
'               The International Association for the Properties of Water and Steam,
'               Berlin, Germany, September 2008, available at http://www.iapws.org
'
' for the saline componentr.,
'
' The basic routines are copies of those written by David Jackett for the GSW library
' on the TEOS-10 site.  However, they been modified to use the routines and approaches
' used in the SIA library in order to provide a cross-check between the two libraries.
'
' September 2009
'======================================

'Private Const ErrorReturn = 9.99999999E+98

Private Const Version = "07 Jun 2010"

'======================================
Public Function gsw_g#(ByVal drv_s&, _
                       ByVal drv_t&, _
                       ByVal drv_p&, _
                       ByVal sa#, _
                       ByVal t#, _
                       ByVal p#)
'======================================

' SEAWATER SPECIFIC GIBBS FREE ENERGY AND DERIVATIVES UP TO ORDER 2
'
' DRV_S               : ORDER OF S DERIVATIVE
' DRV_T               : ORDER OF T DERIVATIVE
' DRV_P               : ORDER OF P DERIVATIVE
' SA                  : ABSOLUTE SALINITY                  [G/KG]
' T                   : TEMPERATURE                        [DEG C]
' P                   : SEA (GAUGE) PRESSURE               [DBAR]
'
'OUTPUT: gsw_g        : SPECIFIC GIBBS ENERGY OR ITS DERIVATIVE

'GSW_G IS THE S-T-P DERIVATIVE
'(D/DS)^DRV_S (D/DT)^DRV_T (D/DP)^DRV_P G(S,T,P)
'OF THE SPECIFIC GIBBS ENERGY OF SEAWATER, G(S,T,P), IN J/KG
'T AND P DERIVATIVES ARE IN UNITS OF 1/K AND 1/PA
'S DERIVATIVES ARE IN UNITS OF 1/(G/KG)


'integer drv_s, drv_t, drv_p
'real*8 gsw_g, sa, t, p
Dim sa_si#, t_si#, p_si#
Dim gw#, gs#

sa_si = sa * 0.001
t_si = cnv_temperature("K(T90)", t, "DEGC(T90)")
p_si = cnv_pressure("PA", p, "DBAR")

gsw_g = ErrorReturn

If (sa_si < 0# Or sa_si >= 1#) Then Exit Function
If (t_si <= 0#) Then Exit Function
If (p_si <= 0#) Then Exit Function

If (drv_s < 0) Then Exit Function
If (drv_t < 0) Then Exit Function
If (drv_p < 0) Then Exit Function
If (drv_s + drv_t + drv_p > 2) Then Exit Function

'WATER PART
gw = 0#
If (drv_s = 0) Then
  gw = fit_liq_g_f03_si(drv_t, drv_p, t_si, p_si)
  If (gw = ErrorReturn) Then Exit Function
End If

'SALINE PART
gs = sal_g_si(drv_s, drv_t, drv_p, sa_si, t_si, p_si)
If (gs = ErrorReturn) Then Exit Function

gsw_g = gw + gs
    
'It appears that the gsw library works with gsw_g in G/KG, DEGC, Pa
gsw_g = gsw_g * (0.001) ^ drv_s

End Function

'======================================
Public Function gsw_alpha_t#(ByVal sa#, _
                             ByVal t#, _
                             ByVal p#)
'======================================

' THERMAL EXPANSION COEFFICIENT OF SEAWATER
'
' SA                  : ABSOLUTE SALINITY                  [G/KG]
' T                   : TEMPERATURE                        [DEG C]
' P                   : SEA PRESSURE                       [DBAR]
'
' GSW_ALPHA_T         : THERMAL EXPANSION COEFFICIENT      [1/K]
'                       WRT (IN SITU) TEMPERATURE

'CHECKVALUE: gsw_alpha_t(35.7d0, 25.5d0, 1023d0) = 0.0003098378393192645d0

Dim gtp#, gp#

gsw_alpha_t = ErrorReturn

gtp = gsw_g(0, 1, 1, sa, t, p)
If gtp = ErrorReturn Then Exit Function
gp = gsw_g(0, 0, 1, sa, t, p)
If gp = ErrorReturn Or gp <= 0 Then Exit Function

gsw_alpha_t = gtp / gp

End Function

'======================================
Public Function gsw_beta_t#(ByVal sa#, _
                            ByVal t#, _
                            ByVal p#)
'======================================

' HALINE CONTRACTION COEFFICIENT OF SEAWATER
'
' SA                  : ABSOLUTE SALINITY                  [G/KG]
' T                   : TEMPERATURE                        [DEG C]
' P                   : SEA PRESSURE                       [DBAR]
'
' RESULT              : HALINE CONTRACTION COEFFICIENT     [KG/G]

'CHECKVALUE: gsw_beta_t(35.7d0, 25.5d0, 1023d0) = 0.0007257297978386655d0

Dim gsp#, gp#

gsw_beta_t = ErrorReturn

gsp = gsw_g(1, 0, 1, sa, t, p)
If gsp = ErrorReturn Then Exit Function
gp = gsw_g(0, 0, 1, sa, t, p)
If gp = ErrorReturn Or gp <= 0 Then Exit Function

gsw_beta_t = -gsp / gp

End Function

'======================================
Public Function gsw_cp#(ByVal sa#, _
                        ByVal t#, _
                        ByVal p#)
'======================================

' ISOBARIC HEAT CAPACITY OF SEAWATER
'
' SA                  : ABSOLUTE SALINITY                  [G/KG]
' T                   : TEMPERATURE                        [DEG C]
' P                   : SEA PRESSURE                       [DBAR]
'
' RESULT              : HEAT CAPACITY                      [J/(KG K)]

'CHECKVALUE: gsw_cp(35.7d0, 25.5d0, 1023d0) = 3974.42541259729d0

Dim gtt#

gsw_cp = ErrorReturn

gtt = gsw_g(0, 2, 0, sa, t, p)
If gtt = ErrorReturn Then Exit Function

gsw_cp = -(t + 273.15) * gtt

End Function

'======================================
Public Function gsw_dens#(ByVal sa#, _
                          ByVal t#, _
                          ByVal p#)
'======================================

' DENSITY OF SEAWATER
'
' SA                  : ABSOLUTE SALINITY                  [G/KG]
' T                   : TEMPERATURE                        [DEG C]
' P                   : SEA PRESSURE                       [DBAR]
'
' RESULT              : DENSITY                            [KG/M^3]

'CHECKVALUE: gsw_dens(35.7d0, 25.5d0, 1023d0) = 1027.95249315662d0

Dim gp#

gsw_dens = ErrorReturn

gp = gsw_g(0, 0, 1, sa, t, p)
If gp = ErrorReturn Or gp <= 0 Then Exit Function

gsw_dens = 1# / gp

End Function

'======================================
Public Function gsw_enthalpy#(ByVal sa#, _
                              ByVal t#, _
                              ByVal p#)
'======================================

' SPECIFIC ENTHALPY OF SEAWATER
'
' SA                  : ABSOLUTE SALINITY                  [G/KG]
' T                   : TEMPERATURE                        [DEG C]
' P                   : SEA PRESSURE                       [DBAR]
'
' RESULT              : SPECIFIC ENTHALPY                  [J/KG]

'CHECKVALUE: gsw_enthalpy(35.7d0, 25.5d0, 1023d0) = 110776.712408975d0

Dim g#, gt#

gsw_enthalpy = ErrorReturn

g = gsw_g(0, 0, 0, sa, t, p)
If g = ErrorReturn Then Exit Function
gt = gsw_g(0, 1, 0, sa, t, p)
If gt = ErrorReturn Then Exit Function

gsw_enthalpy = g - (t + 273.15) * gt

End Function

'======================================
Public Function gsw_entropy#(ByVal sa#, _
                             ByVal t#, _
                             ByVal p#)
'======================================

' SPECIFIC ENTROPY OF SEAWATER
'
' SA                  : ABSOLUTE SALINITY                  [G/KG]
' T                   : TEMPERATURE                        [DEGC]
' P                   : SEA PRESSURE                       [DBAR]
'
' RESULT              : SPECIFIC ENTROPY                   [J/(KG K)]

'CHECKVALUE: gsw_entropy(35.7d0, 25.5d0, 1023d0) = 352.81879771528d0

Dim gt#

gsw_entropy = ErrorReturn

gt = gsw_g(0, 1, 0, sa, t, p)
If gt = ErrorReturn Then Exit Function

gsw_entropy = -gt

End Function

'======================================
Public Function gsw_kappa#(ByVal sa#, _
                           ByVal t#, _
                           ByVal p#)
'======================================

' ISENTROPIC COMPRESSIBILITY OF SEAWATER
'
' SA                  : ASOLUTE SALINITY                   [G/KG]
' T                   : TEMPERATURE                        [DEGC]
' P                   : SEA PRESSURE                       [DBAR]
'
' RESULT              : ISENTROPIC COMPRESSIBILITY         [1/DBAR]

'CHECKVALUE: gsw_kappa(35.7d0, 25.5d0, 1023d0) = 4.033862685464779d-6

Dim g_tt#, g_tp#, g_pp#, g_p#

gsw_kappa = ErrorReturn

g_tt = gsw_g(0, 2, 0, sa, t, p)
If g_tt = ErrorReturn Or g_tt >= 0 Then Exit Function
g_tp = gsw_g(0, 1, 1, sa, t, p)
If g_tp = ErrorReturn Then Exit Function
g_pp = gsw_g(0, 0, 2, sa, t, p)
If g_pp = ErrorReturn Then Exit Function
g_p = gsw_g(0, 0, 1, sa, t, p)
If g_p = ErrorReturn Or g_p <= 0 Then Exit Function

gsw_kappa = 10000# * (g_tp * g_tp - g_tt * g_pp) / (g_p * g_tt)

End Function

'======================================
Public Function gsw_kappa_t#(ByVal sa#, _
                             ByVal t#, _
                             ByVal p#)
'======================================

' ISOTHERMAL COMPRESSIBILITY OF SEAWATER
'
' SA                  : ASOLUTE SALINITY                   [G/KG]
' T                   : TEMPERATURE                        [DEGC]
' P                   : SEA PRESSURE                       [DBAR]
'
' RESULT              : ISOTHERMAL COMPRESSIBILITY         [1/DBAR]

'CHECKVALUE: gsw_kappa_t(35.7d0, 25.5d0, 1023d0) = 4.104037946151349d-6

Dim g_pp#, g_p#

gsw_kappa_t = ErrorReturn

g_pp = gsw_g(0, 0, 2, sa, t, p)
If g_pp = ErrorReturn Then Exit Function
g_p = gsw_g(0, 0, 1, sa, t, p)
If g_p = ErrorReturn Or g_p <= 0 Then Exit Function

gsw_kappa_t = -10000# * g_pp / g_p

End Function

'======================================
Public Function gsw_specvol#(ByVal sa#, _
                             ByVal t#, _
                             ByVal p#)
'======================================

' SPECIFIC VOLUME OF SEAWATER
'
' SA                  : ASOLUTE SALINITY                   [G/KG]
' T                   : TEMPERATURE                        [DEGC]
' P                   : SEA PRESSURE                       [DBAR]
'
' RESULT              : SPECIFIC VOLUME                    [M^3/KG]

'CHECKVALUE: gsw_specvol(35.7d0, 25.5d0, 1023d0) = 0.0009728076021579713d0

gsw_specvol = gsw_g(0, 0, 1, sa, t, p)

End Function

'======================================
Public Function gsw_svel#(ByVal sa#, _
                          ByVal t#, _
                          ByVal p#)
'======================================

' SOUND SPEED OF SEAWATER
'
' sa                  : ASOLUTE SALINITY                   [G/KG]
' T                   : TEMPERATURE                        [DEGC]
' P                   : SEA PRESSURE                       [DBAR]
'
' RESULT              : SOUND SPEED                        [M/S]

'CHECKVALUE: gsw_svel(35.7d0, 25.5d0, 1023d0) = 1552.93372863425d0

Dim g_tt#, g_tp#, g_pp#, g_p#

gsw_svel = ErrorReturn

g_tt = gsw_g(0, 2, 0, sa, t, p)
If g_tt = ErrorReturn Or g_tt >= 0 Then Exit Function
g_tp = gsw_g(0, 1, 1, sa, t, p)
If g_tp = ErrorReturn Then Exit Function
g_pp = gsw_g(0, 0, 2, sa, t, p)
If g_pp = ErrorReturn Then Exit Function
g_p = gsw_g(0, 0, 1, sa, t, p)
If g_p = ErrorReturn Or g_p <= 0 Then Exit Function
If g_tp * g_tp >= g_tt * g_pp Then Exit Function

gsw_svel = g_p * Sqr(g_tt / (g_tp * g_tp - g_tt * g_pp))

End Function

'======================================
Public Function gsw_ctmp_from_ptmp0#(ByVal sa#, ByVal tpot#)
'======================================

'CONSERVATIVE TEMPERATURE FROM POTENTIAL TEMPERATURE OF SEAWATER
'BASED ON sea_ctmp_from_ptmp0_si BUT USES GSW_G
'
'SA         : ABSOLUTE SALINITY                        [G/KG]
'TPOT       : POTENTIAL TEMPERATURE WITH               [DEGC]
'             REFERENCE PRESSURE OF 101325 PA
'
'RESULT     : CONSERVATIVE TEMPERATURE                 [DEGC]

'CHECKVALUE: gsw_ctmp_from_ptmp0(35.7d0, 25.5d0) = 25.4805463842239d0

Dim CP0#, pr#, ct#

gsw_ctmp_from_ptmp0 = ErrorReturn

CP0 = 3991.86795711963       ' IN CONCRETE 02/12/08

pr = 0#

ct = gsw_enthalpy(sa, tpot, pr)
If (ct = ErrorReturn) Then Exit Function

gsw_ctmp_from_ptmp0 = ct / CP0

End Function

'======================================
Public Function gsw_ptmp0_from_ctmp#(ByVal sa#, ByVal ct#)
'======================================

' POTENTIAL TEMPERATURE OF SEAWATER FROM CONSERVATIVE TEMPERATURE
' THIS CODE IS CONSISTENT WITH sea_ptmp0_from_ctmp_si IN SEA_5A
' BUT USED GSW ROUTINES AND DIFFERENT UNITS
'
' SA                  : ASOLUTE SALINITY                   [G/KG]
' CT                  : CONSERVATIVE TEMPERATURE           [DEGC]
'
' RESULT              : POTENTIAL TEMPERATURE WITH         [DEGC]
'                       REFERENCE PRESSURE OF  0 DBAR

'CHECKVALUE: gsw_ptmp0_from_ctmp(35.7d0, 25.4805463842239d0) = 25.5d0

Dim n0&, n2&, nloops&, n&
Dim s1#, ct1#, P0#, CP0#
Dim a0#, a1#, a2#, a3#, a4#, a5#, b0#, b1#, b2#, b3#
Dim a5ct#, b3ct#, ct_factor#, th0_num#, rec_th0_den#
Dim th0#, ct0#, dth_dct#, theta#, dct#, dct_dth#, gtt#

gsw_ptmp0_from_ctmp = ErrorReturn

CP0 = 3991.86795711963

n0 = 0: n2 = 2

s1 = sa * 35# / 35.16504: ct1 = ct: P0 = 0#

a0 = -1.44601364634479E-02:     b0 = 1#
a1 = -3.30530899585292E-03:     b1 = 6.5060971156358E-04
a2 = 1.06241592912898E-04:      b2 = 3.8302894868509E-03
a3 = 0.947756667379449:         b3 = 1.24781176036803E-06
a4 = 2.16659194773661E-03
a5 = 3.8288429550399E-03

a5ct = a5 * ct1: b3ct = b3 * ct1
ct_factor = (a3 + a4 * s1 + a5ct)
th0_num = a0 + s1 * (a1 + a2 * s1) + ct1 * ct_factor
rec_th0_den = 1# / (b0 + b1 * s1 + ct1 * (b2 + b3ct))
th0 = th0_num * rec_th0_den
ct0 = gsw_ctmp_from_ptmp0(sa, th0)
If ct0 = ErrorReturn Then Exit Function
dth_dct = (ct_factor + a5ct - (b2 + b3ct + b3ct) * th0) * rec_th0_den
theta = th0 - (ct0 - ct) * dth_dct

nloops = 1                  ' default

'    NOTE: nloops = 1 gives theta with a maximum error of _

n = 0:

Do While (n <= nloops)
    dct = gsw_ctmp_from_ptmp0(sa, theta)
    If dct = ErrorReturn Then Exit Function
    dct = dct - ct
    gtt = gsw_g(n0, n2, n0, sa, theta, P0)
    If gtt = ErrorReturn Or gtt >= 0 Then Exit Function
    dct_dth = -(theta + 273.15) * gtt / CP0
    If dct_dth = 0 Then Exit Function
    theta = theta - dct / dct_dth
    n = n + 1
Loop

gsw_ptmp0_from_ctmp = theta

End Function

'======================================
Public Function gsw_ptmp#(ByVal sa#, _
                          ByVal t#, _
                          ByVal p#, _
                          ByVal pr#)
'======================================
                               
'THIS FUNCTION COMPUTES POTENTIAL TEMPERATURE OF SEAWATER
'BASED ON SEA_POTTEMP_SI BUT USES GSW_G

'OUTPUT:
'THETA(S,T,P,PR) POTENTIAL TEMPERATURE OF SEAWATER IN DEGC

'INPUT:
'SA     ABSOLUTE SALINITY IN G/KG
'T      ABSOLUTE IN-SITU TEMPERATURE IN DEGC
'P      SEA PRESSURE IN DBAR
'PR     REFERENCE SEA PRESSURE IN DBAR

'CHECKVALUE: gsw_ptmp(35.7d0, 25.2720983155409d0, 0d0, 1023d0) = 25.5
'CHECKVALUE: gsw_ptmp(35.7d0, 25.5d0, 1023d0, 0d0) = 25.2720983155409d0

Dim maxit&
Dim sp#, S#, eps#, theta#

gsw_ptmp = ErrorReturn

If (p = pr) Then
  gsw_ptmp = t
  Exit Function
End If

S = gsw_entropy(sa, t, p)  'SPECIFIC ENTROPY IN-SITU
If S = ErrorReturn Then Exit Function

'use David's initial estimate for theta
sp = sa * 35# / 35.16504
theta = t + (p - pr) * (8.65483913395442E-06 - _
            sp * 1.41636299744881E-06 - _
            (p + pr) * 7.38286467135737E-09 + _
            t * (-8.38241357039698E-06 + _
            sp * 2.83933368585534E-08 + _
            t * 1.77803965218656E-08 + _
            (p + pr) * 1.71155619208233E-10))

maxit = 100
eps = 0.000000000001

'RUN THE ITERATION
gsw_ptmp = gsw_pottemp_iteration(sa, S, pr, theta, maxit, eps)

End Function

'======================================
Public Function gsw_pden#(sa#, t#, p#, pr#)
'======================================

' POTENTIAL DENSITY OF SEAWATER
'
' SA                  : ASOLUTE SALINITY                   [G/KG]
' T                   : TEMPERATURE                        [DEGC]
' P                   : SEA (GAUGE) PRESSURE               [DBAR]
' PR                  : REFERENCE (GAUGE) PRESSURE         [DBAR]
'
' RESULT              : POTENTIAL DENSITY                  [KG/M^3]

'CHECKVALUE: gsw_pden(35.7d0, 25.5d0, 1023d0, 0d0) = 1023.66254941185d0

Dim theta#

gsw_pden = ErrorReturn

theta = gsw_ptmp(sa, t, p, pr)
If theta = ErrorReturn Then Exit Function

gsw_pden = gsw_dens(sa, theta, pr)

End Function

'======================================
Private Function gsw_pottemp_iteration#(sa#, eta#, p#, theta#, maxit&, eps#)
'======================================

'BASED ON POTTEMP_ITERATION FROM SEA_3B

'OUTPUT:
'THETA =   POTENTIAL TEMPERATURE OF SEAWATER IN DEGC
'          I.E. THE TEMPERATURE THAT SOLVES ETA_SI = GSW_ENTROPY_SI(SA, T, P)

'INPUT:
'SA     ABSOLUTE SALINITY IN G/KG
'ETA    ENTROPY IN J/(KG K)
'P      (REFERENCE) SEA PRESSURE IN DBAR
'T      (POTENTIAL) SEA TEMPERATURE IN DEGC, INITIAL VALUE
'MAXIT  MAX. NUMBER OF ITERATIONS
'EPS    REQUIRED TOLERANCE IN K

Dim i&
Dim S#, cp#, dt#

gsw_pottemp_iteration = ErrorReturn

If (eps <= 0#) Then Exit Function
If (maxit <= 0#) Then Exit Function

For i = 1 To maxit

  'get entropy and its first derivative for newton iteration
  S = gsw_entropy(sa, theta, p)
  If S = ErrorReturn Then Exit Function
  cp = gsw_cp(sa, theta, p)
  If cp = ErrorReturn Or cp = 0 Then Exit Function
  
  'next temperature improvement step
  dt = (273.15 + theta) * (eta - S) / cp
  theta = theta + dt
  If (theta <= -273.15) Then Exit Function
  
  If (Abs(dt) < eps) Then
    gsw_pottemp_iteration = theta
    Exit Function
  End If
  
Next i

End Function

'==========================================================================
Public Function gsw_alpha_ct(ByVal sa_gkg As Double, _
                             ByVal t_c As Double, _
                             ByVal p_dbar As Double) As Double
'==========================================================================

' THERMAL EXPANSION COEFFICIENT OF SEAWATER WRT CONSERVATIVE TEMPERATURE
'
' SA_GKG                : ABSOLUTE SALINITY                  [G/KG]
' T_C                   : IN SITU TEMPERATURE                [C]
' P_DBAR                : SEA PRESSURE                       [DBAR]
'
' gsw_alpha_ct        : THERMAL EXPANSION COEFFICIENT      [1/C]
'                         WRT CONSERVATIVE TEMPERATURE

'check value with default settings:
'gsw_alpha_ct(35, 20, 1000) = 2.6941860986082E-04

Dim sa_si As Double, p_si As Double, t_si As Double

'convert practical input units to basic SI units
sa_si = sa_gkg * 0.001
p_si = SO_pressure_si + p_dbar * 10000#
t_si = SO_temperature_si + t_c

'and call the SI version
gsw_alpha_ct = sea_alpha_ct_si(sa_si, t_si, p_si)

End Function

'==========================================================================
Public Function gsw_alpha_pt0(ByVal sa_gkg As Double, _
                              ByVal t_c As Double, _
                              ByVal p_dbar As Double) As Double
'==========================================================================

' THERMAL EXPANSION COEFFICIENT OF SEAWATER WRT POTENTIAL TEMPERATURE
'
' SA_GKG                 : ABSOLUTE SALINITY                  [G/KG]
' T_C                    : IN SITU TEMPERATURE                [C]
' P_DBAR                 : SEA PRESSURE                       [DBAR]
'
' gsw_alpha_pt0        : THERMAL EXPANSION COEFFICIENT      [1/C]
'                          WRT POTENTIAL TEMPERATURE

'check value with default settings:
'gsw_alpha_pt0(35, 20, 1000) = 2.6975373331|6792E-04

Dim sa_si As Double, p_si As Double, t_si As Double

'convert practical input units to basic SI units
sa_si = sa_gkg * 0.001
p_si = SO_pressure_si + p_dbar * 10000#
t_si = SO_temperature_si + t_c

'and call the SI version
gsw_alpha_pt0 = sea_alpha_pt0_si(sa_si, t_si, p_si)

End Function

'==========================================================================
Public Function gsw_beta_ct(ByVal sa_gkg As Double, _
                            ByVal t_c As Double, _
                            ByVal p_dbar As Double) As Double
'==========================================================================

' HALINE CONTRACTION COEFFICIENT OF SEAWATER WRT CONSTANT CONSERVATIVE TEMPERATURE
'
' SA_GKG                 : ABSOLUTE SALINITY                  [G/KG]
' T_C                    : IN SITU TEMPERATURE                [C]
' P_DBAR                 : SEA PRESSURE                       [DBAR]
'
' RESULT                 : HALINE CONTRACTION COEFFICIENT     [KG/G]
'                          WRT CONSTANT CONSERVATIVE TEMPERATURE

'check value with default settings:
'gsw_beta_ct(35, 20, 1000) = 7.23213672953544E-04

Dim sa_si As Double, p_si As Double, t_si As Double, bt As Double

gsw_beta_ct = ErrorReturn

'convert practical input units to basic SI units
sa_si = sa_gkg * 0.001
p_si = SO_pressure_si + p_dbar * 10000#
t_si = SO_temperature_si + t_c

'call the SI version
bt = sea_beta_ct_si(sa_si, t_si, p_si)
If bt = ErrorReturn Then Exit Function

'and convert the result to kg/g, the unit of 1/sa_gkg
gsw_beta_ct = bt * 0.001 '1 kg/kg = 0.001 kg/g

End Function

'==========================================================================
Public Function gsw_beta_pt0(ByVal sa_gkg As Double, _
                             ByVal t_c As Double, _
                             ByVal p_dbar As Double) As Double
'==========================================================================

' HALINE CONTRACTION COEFFICIENT OF SEAWATER WRT CONSTANT POTENTIAL TEMPERATURE
'
' SA_GKG                 : ABSOLUTE SALINITY                  [G/KG]
' T_C                    : IN SITU TEMPERATURE                [C]
' P_DBAR                 : SEA PRESSURE                       [DBAR]
'
' RESULT                 : HALINE CONTRACTION COEFFICIENT     [KG/G]
'                          WRT CONSTANT POTENTIAL TEMPERATURE

'check value with default settings:
'gsw_beta_pt0(35, 20, 1000) = 7.31582583383175E-04

Dim sa_si As Double, p_si As Double, t_si As Double, bt As Double

gsw_beta_pt0 = ErrorReturn

'convert practical input units to basic SI units
sa_si = sa_gkg * 0.001
p_si = SO_pressure_si + p_dbar * 10000#
t_si = SO_temperature_si + t_c

'call the SI version
bt = sea_beta_pt0_si(sa_si, t_si, p_si)
If bt = ErrorReturn Then Exit Function

'and convert the result to kg/g, the unit of 1/sa_gkg
gsw_beta_pt0 = bt * 0.001 '1 kg/kg = 0.001 kg/g

End Function

'==========================================================================
Public Function gsw_cabb_ct(ByVal sa_gkg As Double, _
                            ByVal t_c As Double, _
                            ByVal p_dbar As Double) As Double
'==========================================================================

'CABBELING COEFFICIENT OF SEAWATER WRT CONSERVATIVE TEMPERATURE

'SA                  : ABSOLUTE SALINITY                  [G/KG]
'T                   : IN SITU TEMPERATURE                [DEGC]
'P                   : SEA PRESSURE                       [DBAR]
'
'RESULT              : CABBELING COEFFICIENT              [1/(DEGC**2)]
'                      WRT CONSERVATIVE TEMPERATURE

'check value:
'gsw_cabb_ct(35, 20, 1000#) = 8.969073|83083E-06

Dim sa_si As Double, t_si As Double, p_si As Double

'convert practical input units to basic SI units
sa_si = sa_gkg * 0.001
p_si = SO_pressure_si + p_dbar * 10000#
t_si = SO_temperature_si + t_c

'call the SI version
gsw_cabb_ct = sea_cabb_ct_si(sa_si, t_si, p_si)

End Function

'==========================================================================
Public Function gsw_cabb_pt0(ByVal sa_gkg As Double, _
                             ByVal t_c As Double, _
                             ByVal p_dbar As Double) As Double
'==========================================================================

'CABBELING COEFFICIENT OF SEAWATER WRT POTENTIAL TEMPERATURE

'SA                  : ABSOLUTE SALINITY                  [G/KG]
'T                   : IN SITU TEMPERATURE                [DEGC]
'P                   : SEA PRESSURE                       [DBAR]
'
'RESULT              : CABBELING COEFFICIENT              [1/(DEGC**2)]
'                      WRT POTENTIAL TEMPERATURE

'check value:
'gsw_cabb_pt0(35, 20, 1000#) = 8.759631|54048E-06

Dim sa_si As Double, t_si As Double, p_si As Double

gsw_cabb_pt0 = ErrorReturn

'convert practical input units to basic SI units
sa_si = sa_gkg * 0.001
p_si = SO_pressure_si + p_dbar * 10000#
t_si = SO_temperature_si + t_c

'call the SI version
gsw_cabb_pt0 = sea_cabb_pt0_si(sa_si, t_si, p_si)

End Function

'==========================================================================
Public Function gsw_thrmb_ct(ByVal sa_gkg As Double, _
                             ByVal t_c As Double, _
                             ByVal p_dbar As Double) As Double
'==========================================================================

'THERMOBARIC COEFFICIENT OF SEAWATER WRT CONSERVATIVE TEMPERATURE
'
'SA                  : ABSOLUTE SALINITY                  [G/KG]
'T                   : IN SITU TEMPERATURE                [DEGC]
'P                   : SEA (GAUGE) PRESSURE               [DBAR]
'
'RESULT              : THERMOBARIC COEFFICIENT            [1/(DEGC DBAR)]
'                      WRT CONSERVATIVE TEMPERATURE

'check value:
'gsw_thrmb_ct(35, 20, 1000#) = 1.727083|65652E-08

Dim sa_si As Double, t_si As Double, p_si As Double, x As Double

gsw_thrmb_ct = ErrorReturn

'convert practical input units to basic SI units
sa_si = sa_gkg * 0.001
p_si = SO_pressure_si + p_dbar * 10000#
t_si = SO_temperature_si + t_c

'call the SI version
x = sea_thrmb_ct_si(sa_si, t_si, p_si)
If x = ErrorReturn Then Exit Function

'and convert the result from 1/Pa to 1/dbar
gsw_thrmb_ct = 10000# * x

End Function

'==========================================================================
Public Function gsw_thrmb_pt0(ByVal sa_gkg As Double, _
                              ByVal t_c As Double, _
                              ByVal p_dbar As Double) As Double
'==========================================================================

'THERMOBARIC COEFFICIENT OF SEAWATER WRT POTENTIAL TEMPERATURE
'
'SA                  : ABSOLUTE SALINITY                  [G/KG]
'T                   : IN SITU TEMPERATURE                [DEGC]
'P                   : SEA (GAUGE) PRESSURE               [DBAR]
'
'RESULT              : THERMOBARIC COEFFICIENT            [1/(DEGC DBAR)]
'                      WRT POTENTIAL TEMPERATURE

'check value:
'gsw_thrmb_pt0(35, 20, 1000#) = 1.709450|45984E-08

Dim sa_si As Double, t_si As Double, p_si As Double, x As Double

gsw_thrmb_pt0 = ErrorReturn

'convert practical input units to basic SI units
sa_si = sa_gkg * 0.001
p_si = SO_pressure_si + p_dbar * 10000#
t_si = SO_temperature_si + t_c

'call the SI version
x = sea_thrmb_pt0_si(sa_si, t_si, p_si)
If x = ErrorReturn Then Exit Function

'and convert the result from 1/Pa to 1/dbar
gsw_thrmb_pt0 = 10000# * x

End Function

'=====================================================
Public Function gsw_asal_from_psal(ByVal sp As Double, _
                                   ByVal longs0 As Double, _
                                   ByVal lats0 As Double, _
                                   ByVal p_dbar As Double) As Double
'=====================================================

' CONVERT PRACTICAL SALINITY TO ABSOLUTE SALINITY
'
' SP                  : PRACTICAL SALINITY                 [PSU]
' P_DBAR              : SEA PRESSURE                       [DBAR]
' LONGS0              : LONGITUDE                          [DEG E]
' LATS0               : LATITUDE                           [DEG N]
'
' RESULT              : ABSOLUTE SALINITY                  [G/KG]

'CHECK VALUE:
'gsw_asal_from_psal(35.5276443777339, 201, -21, 1023) = 35.7000000000001

Dim sa As Double, p_si As Double

gsw_asal_from_psal = ErrorReturn

p_si = cnv_pressure("PA", p_dbar, "DBAR")
If p_si = ErrorReturn Then Exit Function

sa = asal_from_psal(sp, longs0, lats0, p_si)
If sa = ErrorReturn Then Exit Function

gsw_asal_from_psal = 1000 * sa

End Function

'=====================================================
Public Function gsw_psal_from_asal(ByVal sa_gkg As Double, _
                                   ByVal longs0 As Double, _
                                   ByVal lats0 As Double, _
                                   ByVal p_dbar As Double) As Double
'=====================================================

' CONVERT ABSOLUTE SALINITY TO PRACTICAL SALINITY
'
' SA_SI               : ABSOLUTE SALINITY                  [G/KG]
' P_DBAR              : SEA PRESSURE                       [DBAR]
' LONGS0              : LONGITUDE                          [DEG E]
' LATS0               : LATITUDE                           [DEG N]
'
' PSAL_FROM_ASAL  : PRACTICAL SALINITY                     [PSU]


'CHECK value
'gsw_psal_from_asal(35.7, 201, -21, 1023) = 35.5276443777339

Dim sa_si As Double, p_si As Double

gsw_psal_from_asal = ErrorReturn

p_si = cnv_pressure("PA", p_dbar, "DBAR")
If p_si = ErrorReturn Then Exit Function

sa_si = 0.001 * sa_gkg

gsw_psal_from_asal = psal_from_asal(sa_si, longs0, lats0, p_si)

End Function


