Attribute VB_Name = "Flu_IF97_5_Mdl"
Option Explicit


'#########################################################################

'This module requires the library module
'     Constants_0_Mdl, file Constants_0.bas
'     Maths_0_Mdl,     file Maths_0.bas

'#########################################################################


'=========================================================================
'This module implements the Gibbs potentials of liquid water and vapour and its
'first and second partial derivatives with respect to temperature and
'density as defined in IAPWS-IF97 for the regions 1 and 2:

'Revised Release on the IAPWS Industrial Formulation 1997
'for the Thermodynamic Properties of Water and Steam
'(The revision only relates to the extension of region 5 to 50 MPa)
'The International Association for the Properties of Water and Steam
'Lucerne , Switzerland, August 2007

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98

Private Const Version = "19 Dec 2009"

'Coefficients of region 1
Private i1i(34) As Integer
Private j1i(34) As Integer
Private n1i(34) As Double

'Coefficients of region 2, ideal part
Private j0i(9) As Integer
Private n0i(9) As Double

'Coefficients of region 2, residual part
Private iri(43) As Integer
Private jri(43) As Integer
Private nri(43) As Double


'=========================================================================
Public Function fit_liq_g_if97_si(ByVal drv_t As Integer, _
                                  ByVal drv_p As Integer, _
                                  ByVal t_si As Double, _
                                  ByVal p_si As Double) As Double

'This function returns the Gibbs function g(t,p) and its 1st and 2nd derivatives
'with respect to temperature and pressure, as defined for the region 1 (liquid) in IAPWS-IF97
'
'output: fit_liq_g_if97_si: specific Gibbs energy in J/kg or its t-p derivative
'
'input:  drv_t: order of the temperature derivative (0-2)
'        drv_p: order of the pressure derivative (0-2)
'        t_si:  absolute temperature in K
'        p_si:  absolute pressure in Pa

Const tu = 1386#
Const pu# = 16530000#
Const R = 461.526         'J kg-1 K-1  specific gas constant
                          'note this deviates from Gas_Constant_H2O_si = 461.51805
                          
Dim g As Double, gt As Double, gtt As Double
Dim gp As Double, gtp As Double, gpp As Double
Dim PI As Double, tau As Double, RT As Double

fit_liq_g_if97_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

InitIF97_1

PI = p_si / pu
tau = tu / t_si
RT = R * t_si

Select Case drv_t

  Case 0:
    Select Case drv_p

      Case 0: g = gamma_1(0, 0, tau, PI)
              If g = ErrorReturn Then Exit Function
              fit_liq_g_if97_si = RT * g

      Case 1: gp = gamma_1(0, 1, tau, PI)
              If gp = ErrorReturn Then Exit Function
              fit_liq_g_if97_si = RT * gp / pu

      Case 2: gpp = gamma_1(0, 2, tau, PI)
              If gpp = ErrorReturn Then Exit Function
              fit_liq_g_if97_si = RT * gpp / pu ^ 2

      Case Else: Exit Function
    End Select

  Case 1:
    Select Case drv_p

      Case 0: g = gamma_1(0, 0, tau, PI)
              If g = ErrorReturn Then Exit Function
              gt = gamma_1(1, 0, tau, PI)
              If gt = ErrorReturn Then Exit Function
              fit_liq_g_if97_si = R * (g - tau * gt)

      Case 1: gp = gamma_1(0, 1, tau, PI)
              If gp = ErrorReturn Then Exit Function
              gtp = gamma_1(1, 1, tau, PI)
              If gtp = ErrorReturn Then Exit Function
              fit_liq_g_if97_si = R * (gp - tau * gtp) / pu

      Case Else: Exit Function
    End Select

  Case 2:
    Select Case drv_p

      Case 0: gtt = gamma_1(2, 0, tau, PI)
              If gtt = ErrorReturn Then Exit Function
              fit_liq_g_if97_si = R * tau ^ 2 * gtt / t_si

      Case Else: Exit Function
    End Select

  Case Else: Exit Function
End Select

End Function

'=========================================================================
Public Function fit_vap_g_if97_si(ByVal drv_t As Integer, _
                                  ByVal drv_p As Integer, _
                                  ByVal t_si As Double, _
                                  ByVal p_si As Double) As Double

'This function returns the Gibbs function g(t,p) and its 1st and 2nd derivatives
'with respect to temperature and pressure, as defined for the region 2 (vapour) in IAPWS-IF97
'
'output: fit_vap_g_if97_si: specific Gibbs energy in J/kg or its t-p derivative
'
'input:  drv_t: order of the temperature derivative (0-2)
'        drv_p: order of the pressure derivative (0-2)
'        t_si:  absolute temperature in K
'        p_si:  absolute pressure in Pa

Const tu = 540#
Const pu = 1000000#
Const R = 461.526         'J kg-1 K-1  specific gas constant
                          'note this deviates from Gas_Constant_H2O_si = 461.51805
                          
Dim g As Double, gt As Double, gtt As Double
Dim gp As Double, gtp As Double, gpp As Double
Dim PI As Double, tau As Double, RT As Double

fit_vap_g_if97_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

InitIF97_2

PI = p_si / pu
tau = tu / t_si
RT = R * t_si

Select Case drv_t

  Case 0:
    Select Case drv_p

      Case 0: g = gamma_2(0, 0, tau, PI)
              If g = ErrorReturn Then Exit Function
              fit_vap_g_if97_si = RT * g

      Case 1: gp = gamma_2(0, 1, tau, PI)
              If gp = ErrorReturn Then Exit Function
              fit_vap_g_if97_si = RT * gp / pu

      Case 2: gpp = gamma_2(0, 2, tau, PI)
              If gpp = ErrorReturn Then Exit Function
              fit_vap_g_if97_si = RT * gpp / pu ^ 2

      Case Else: Exit Function
    End Select

  Case 1:
    Select Case drv_p

      Case 0: g = gamma_2(0, 0, tau, PI)
              If g = ErrorReturn Then Exit Function
              gt = gamma_2(1, 0, tau, PI)
              If gt = ErrorReturn Then Exit Function
              fit_vap_g_if97_si = R * (g - tau * gt)

      Case 1: gp = gamma_2(0, 1, tau, PI)
              If gp = ErrorReturn Then Exit Function
              gtp = gamma_2(1, 1, tau, PI)
              If gtp = ErrorReturn Then Exit Function
              fit_vap_g_if97_si = R * (gp - tau * gtp) / pu

      Case Else: Exit Function
    End Select

  Case 2:
    Select Case drv_p

      Case 0: gtt = gamma_2(2, 0, tau, PI)
              If gtt = ErrorReturn Then Exit Function
              fit_vap_g_if97_si = R * tau ^ 2 * gtt / t_si

      Case Else: Exit Function
    End Select

  Case Else: Exit Function
End Select

End Function

'=========================================================================
Public Function fit_liq_density_if97_si(ByVal t_si As Double, _
                                        ByVal p_si As Double) As Double

'This function returns the density of liquid water as a function of temperature
'and pressure, in the region 1 (liquid) in IAPWS-IF97

'output:  fit_liq_density_if97_si: density in kg/m^3

'input:   t_si: absolute temperature in K
'         p_si: absolute pressure in Pa

Dim d As Double

fit_liq_density_if97_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = fit_liq_g_if97_si(0, 1, t_si, p_si)

If d = ErrorReturn Then Exit Function
If d <= 0 Then Exit Function

fit_liq_density_if97_si = 1# / d

End Function

'=========================================================================
Public Function fit_vap_density_if97_si(ByVal t_si As Double, _
                                        ByVal p_si As Double) As Double

'This function returns the density of liquid water as a function of temperature
'and pressure, in the region 2 (vapour) in IAPWS-IF97

'output:  fit_vap_density_if97_si: density in kg/m^3

'input:   t_si: absolute temperature in K
'         p_si: absolute pressure in Pa

Dim d As Double

fit_vap_density_if97_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = fit_vap_g_if97_si(0, 1, t_si, p_si)

If d = ErrorReturn Then Exit Function
If d <= 0 Then Exit Function

fit_vap_density_if97_si = 1# / d

End Function

'=========================================================================
Private Sub InitIF97_1()

Dim i As Integer

If n1i(1) = 0.14632971213167 Then Exit Sub

'TablE-2. Numerical values of the coefficients and exponents of the dimensionless Gibbs free energy
'for region 1, Eq. (7)
'i Ii Ji ni i Ii Ji ni
i = 1: i1i(i) = 0:  j1i(i) = -2: n1i(i) = 0.14632971213167:     i = 18: i1i(i) = 2:  j1i(i) = 3:   n1i(i) = -4.4141845330846E-06
i = 2: i1i(i) = 0:  j1i(i) = -1: n1i(i) = -0.84548187169114:    i = 19: i1i(i) = 2:  j1i(i) = 17:  n1i(i) = -7.2694996297594E-16
i = 3: i1i(i) = 0:  j1i(i) = 0:  n1i(i) = -3.756360367204:      i = 20: i1i(i) = 3:  j1i(i) = -4:  n1i(i) = -3.1679644845054E-05
i = 4: i1i(i) = 0:  j1i(i) = 1:  n1i(i) = 3.3855169168385:      i = 21: i1i(i) = 3:  j1i(i) = 0:   n1i(i) = -2.8270797985312E-06
i = 5: i1i(i) = 0:  j1i(i) = 2:  n1i(i) = -0.95791963387872:    i = 22: i1i(i) = 3:  j1i(i) = 6:   n1i(i) = -8.5205128120103E-10
i = 6: i1i(i) = 0:  j1i(i) = 3:  n1i(i) = 0.15772038513228:     i = 23: i1i(i) = 4:  j1i(i) = -5:  n1i(i) = -2.2425281908E-06
i = 7: i1i(i) = 0:  j1i(i) = 4:  n1i(i) = -0.016616417199501:   i = 24: i1i(i) = 4:  j1i(i) = -2:  n1i(i) = -6.5171222895601E-07
i = 8: i1i(i) = 0:  j1i(i) = 5:  n1i(i) = 8.1214629983568E-04:  i = 25: i1i(i) = 4:  j1i(i) = 10:  n1i(i) = -1.4341729937924E-13
i = 9: i1i(i) = 1:  j1i(i) = -9: n1i(i) = 2.8319080123804E-04:  i = 26: i1i(i) = 5:  j1i(i) = -8:  n1i(i) = -4.0516996860117E-07
i = 10: i1i(i) = 1: j1i(i) = -7: n1i(i) = -6.0706301565874E-04: i = 27: i1i(i) = 8:  j1i(i) = -11: n1i(i) = -1.2734301741641E-09
i = 11: i1i(i) = 1: j1i(i) = -1: n1i(i) = -0.018990068218419:   i = 28: i1i(i) = 8:  j1i(i) = -6:  n1i(i) = -1.7424871230634E-10
i = 12: i1i(i) = 1: j1i(i) = 0:  n1i(i) = -0.032529748770505:   i = 29: i1i(i) = 21: j1i(i) = -29: n1i(i) = -6.8762131295531E-19
i = 13: i1i(i) = 1: j1i(i) = 1:  n1i(i) = -0.021841717175414:   i = 30: i1i(i) = 23: j1i(i) = -31: n1i(i) = 1.4478307828521E-20
i = 14: i1i(i) = 1: j1i(i) = 3:  n1i(i) = -5.283835796993E-05:  i = 31: i1i(i) = 29: j1i(i) = -38: n1i(i) = 2.6335781662795E-23
i = 15: i1i(i) = 2: j1i(i) = -3: n1i(i) = -4.7184321073267E-04: i = 32: i1i(i) = 30: j1i(i) = -39: n1i(i) = -1.1947622640071E-23
i = 16: i1i(i) = 2: j1i(i) = 0:  n1i(i) = -3.0001780793026E-04: i = 33: i1i(i) = 31: j1i(i) = -40: n1i(i) = 1.8228094581404E-24
i = 17: i1i(i) = 2: j1i(i) = 1:  n1i(i) = 4.7661393906987E-05:  i = 34: i1i(i) = 32: j1i(i) = -41: n1i(i) = -9.3537087292458E-26

End Sub

'=========================================================================
Private Sub InitIF97_2()

Dim i As Integer

If n0i(i) = -9.6927686500217 Then Exit Sub

'Table 10. Numerical values of the coefficients and exponents of the ideal-gas part gamma_0 of the
'dimensionless Gibbs free energy for region 2, Eq. (16)
'i Ji ni i Ji ni
i = 1: j0i(i) = 0:  n0i(i) = -9.6927686500217
i = 2: j0i(i) = 1:  n0i(i) = 10.086655968018
i = 3: j0i(i) = -5: n0i(i) = -0.005608791128302
i = 4: j0i(i) = -4: n0i(i) = 0.071452738081455
i = 5: j0i(i) = -3: n0i(i) = -0.40710498223928
i = 6: j0i(i) = -2: n0i(i) = 1.4240819171444
i = 7: j0i(i) = -1: n0i(i) = -4.383951131945
i = 8: j0i(i) = 2:  n0i(i) = -0.28408632460772
i = 9: j0i(i) = 3:  n0i(i) = 0.021268463753307
'If Eq. (16) is incorporated into Eq. (18), instead of thE-values for n0i(1) and n0i(1)
'given above, the following values
'n0i(1) = -0.96937268393049E1
'n0i(2) = 0.10087275970006E2
'should be used


'Table 11. Numerical values of the coefficients and exponents of the
'residual part gamma_r of the dimensionless Gibbs free energy for
'region 2, Eq. (17)
'i Ii Ji ni
i = 1:  iri(i) = 1:  jri(i) = 0:  nri(i) = -1.7731742473213E-03
i = 2:  iri(i) = 1:  jri(i) = 1:  nri(i) = -0.017834862292358
i = 3:  iri(i) = 1:  jri(i) = 2:  nri(i) = -0.045996013696365
i = 4:  iri(i) = 1:  jri(i) = 3:  nri(i) = -0.057581259083432
i = 5:  iri(i) = 1:  jri(i) = 6:  nri(i) = -0.05032527872793
i = 6:  iri(i) = 2:  jri(i) = 1:  nri(i) = -3.3032641670203E-05
i = 7:  iri(i) = 2:  jri(i) = 2:  nri(i) = -1.8948987516315E-04
i = 8:  iri(i) = 2:  jri(i) = 4:  nri(i) = -3.9392777243355E-03
i = 9:  iri(i) = 2:  jri(i) = 7:  nri(i) = -0.043797295650573
i = 10: iri(i) = 2:  jri(i) = 36: nri(i) = -2.6674547914087E-05
i = 11: iri(i) = 3:  jri(i) = 0:  nri(i) = 2.0481737692309E-08
i = 12: iri(i) = 3:  jri(i) = 1:  nri(i) = 4.3870667284435E-07
i = 13: iri(i) = 3:  jri(i) = 3:  nri(i) = -3.227767723857E-05
i = 14: iri(i) = 3:  jri(i) = 6:  nri(i) = -1.5033924542148E-03
i = 15: iri(i) = 3:  jri(i) = 35: nri(i) = -0.040668253562649
i = 16: iri(i) = 4:  jri(i) = 1:  nri(i) = -7.8847309559367E-10
i = 17: iri(i) = 4:  jri(i) = 2:  nri(i) = 1.2790717852285E-08
i = 18: iri(i) = 4:  jri(i) = 3:  nri(i) = 4.8225372718507E-07
i = 19: iri(i) = 5:  jri(i) = 7:  nri(i) = 2.2922076337661E-06
i = 20: iri(i) = 6:  jri(i) = 3:  nri(i) = -1.6714766451061E-11
i = 21: iri(i) = 6:  jri(i) = 16: nri(i) = -2.1171472321355E-03
i = 22: iri(i) = 6:  jri(i) = 35: nri(i) = -23.895741934104
i = 23: iri(i) = 7:  jri(i) = 0:  nri(i) = -5.905956432427E-18
i = 24: iri(i) = 7:  jri(i) = 11: nri(i) = -1.2621808899101E-06
i = 25: iri(i) = 7:  jri(i) = 25: nri(i) = -0.038946842435739
i = 26: iri(i) = 8:  jri(i) = 8:  nri(i) = 1.1256211360459E-11
i = 27: iri(i) = 8:  jri(i) = 36: nri(i) = -8.2311340897998
i = 28: iri(i) = 9:  jri(i) = 13: nri(i) = 1.9809712802088E-08
i = 29: iri(i) = 10: jri(i) = 4:  nri(i) = 1.0406965210174E-19
i = 30: iri(i) = 10: jri(i) = 10: nri(i) = -1.0234747095929E-13
i = 31: iri(i) = 10: jri(i) = 14: nri(i) = -1.0018179379511E-09
i = 32: iri(i) = 16: jri(i) = 29: nri(i) = -8.0882908646985E-11
i = 33: iri(i) = 16: jri(i) = 50: nri(i) = 0.10693031879409
i = 34: iri(i) = 18: jri(i) = 57: nri(i) = -0.33662250574171
i = 35: iri(i) = 20: jri(i) = 20: nri(i) = 8.9185845355421E-25
i = 36: iri(i) = 20: jri(i) = 35: nri(i) = 3.0629316876232E-13
i = 37: iri(i) = 20: jri(i) = 48: nri(i) = -4.2002467698208E-06
i = 38: iri(i) = 21: jri(i) = 21: nri(i) = -5.9056029685639E-26
i = 39: iri(i) = 22: jri(i) = 53: nri(i) = 3.7826947613457E-06
i = 40: iri(i) = 23: jri(i) = 39: nri(i) = -1.2768608934681E-15
i = 41: iri(i) = 24: jri(i) = 26: nri(i) = 7.3087610595061E-29
i = 42: iri(i) = 24: jri(i) = 40: nri(i) = 5.5414715350778E-17
i = 43: iri(i) = 24: jri(i) = 58: nri(i) = -9.436970724121E-07

End Sub

'=========================================================================
Private Function gamma_0(ByVal drv_t As Integer, _
                         ByVal drv_p As Integer, _
                         ByVal tau As Double, _
                         ByVal PI As Double) As Double

'this function implements the derivatives of gamma_0 as given in Table 13 of IF-97

Dim g As Double, pwrt As Double
Dim i As Integer, k As Integer

gamma_0 = ErrorReturn

If PI <= 0 Then Exit Function
If tau <= 0 Then Exit Function
If drv_t < 0 Then Exit Function
If drv_p < 0 Then Exit Function

g = 0
If drv_t = 0 Then
  If drv_p = 0 Then
    g = Log(PI)
  Else
    g = 1# / PI
    For k = 2 To drv_p
      g = (1 - k) * g / PI
    Next k
  End If
End If

If drv_p = 0 Then
  For i = 1 To 9
    pwrt = tau ^ (j0i(i) - drv_t)
    For k = 0 To drv_t - 1
      pwrt = pwrt * (j0i(i) - k)
    Next k
    g = g + n0i(i) * pwrt
  Next i
End If

gamma_0 = g

End Function

'=========================================================================
Private Function gamma_1(ByVal drv_t As Integer, _
                         ByVal drv_p As Integer, _
                         ByVal tau As Double, _
                         ByVal PI As Double) As Double

'this function implements the derivatives of gamma as given in Table 4 of IF-97

Dim i As Integer, k As Integer
Dim g As Double, pp As Double, Tt As Double
Dim pwrt As Double, pwrp As Double

gamma_1 = ErrorReturn

If PI <= 0 Then Exit Function
If tau <= 0 Then Exit Function
If drv_t < 0 Then Exit Function
If drv_p < 0 Then Exit Function

pp = 7.1 - PI
Tt = tau - 1.222

g = 0
For i = 1 To 34

  If Tt = 0 Then
    Select Case j1i(i) - drv_t
      Case 0: pwrt = 1
      Case Is > 0: pwrt = 0
      Case Else: Exit Function
    End Select
  Else
    pwrt = Tt ^ (j1i(i) - drv_t)
  End If
  For k = 0 To drv_t - 1
    pwrt = pwrt * (j1i(i) - k)
  Next k

  If pp = 0 Then
    Select Case i1i(i) - drv_p
      Case 0: pwrp = 1
      Case Is > 0: pwrp = 0
      Case Else: Exit Function
    End Select
  Else
    pwrp = pp ^ (i1i(i) - drv_p)
  End If
  For k = 0 To drv_p - 1
    pwrp = -pwrp * (i1i(i) - k)
  Next k

  g = g + n1i(i) * pwrp * pwrt

Next i

gamma_1 = g

End Function

'=========================================================================
Private Function gamma_2(ByVal drv_t As Integer, _
                         ByVal drv_p As Integer, _
                         ByVal tau As Double, _
                         ByVal PI As Double) As Double

'this function implements the derivatives of gamma as given in Eq. 15 of IF-97

Dim g0 As Double, gr As Double

InitIF97_2

gamma_2 = ErrorReturn

g0 = gamma_0(drv_t, drv_p, tau, PI)
If g0 = ErrorReturn Then Exit Function

gr = gamma_r(drv_t, drv_p, tau, PI)
If gr = ErrorReturn Then Exit Function

gamma_2 = g0 + gr

End Function

'=========================================================================
Private Function gamma_r(ByVal drv_t As Integer, _
                         ByVal drv_p As Integer, _
                         ByVal tau As Double, _
                         ByVal PI As Double) As Double
                         
'this function implements the derivatives of gamma_r as given in Table 14 of IF-97

Dim i As Integer, k As Integer
Dim g As Double, Tt As Double
Dim pwrt As Double, pwrp As Double

gamma_r = ErrorReturn

If PI <= 0 Then Exit Function
If tau <= 0 Then Exit Function
If drv_t < 0 Then Exit Function
If drv_p < 0 Then Exit Function

Tt = tau - 0.5

g = 0
For i = 1 To 43

  If Tt = 0 Then
    Select Case jri(i) - drv_t
      Case 0: pwrt = 1
      Case Is > 0: pwrt = 0
      Case Else: Exit Function
    End Select
  Else
    pwrt = Tt ^ (jri(i) - drv_t)
  End If
  For k = 0 To drv_t - 1
    pwrt = pwrt * (jri(i) - k)
  Next k

  pwrp = PI ^ (iri(i) - drv_p)
  For k = 0 To drv_p - 1
    pwrp = pwrp * (iri(i) - k)
  Next k

  g = g + nri(i) * pwrp * pwrt

Next i

gamma_r = g

End Function

'=========================================================================
Public Function chk_IAPWS97_Table5() As String

'Table 5. Thermodynamic property values calculated from Eq. (7) for selected values of T and p
'                   T = 300 K,              T = 300 K,          T = 500 K,
'                   p = 3 MPa               p = 80 MPa          p = 3 MPa
'v / (m3 kg-1)      0.100 215 168 E-2       0.971 180 894 E-3   0.120 241 800 E-2
'h / (kJ kg-1)      0.115 331 273 E3        0.184 142 828 E3    0.975 542 239 E3
'u / (kJ kg-1)      0.112 324 818 E3        0.106 448 356 E3    0.971 934 985 E3
's / (kJ kg-1 K-1)  0.392 294 792           0.368 563 852       0.258 041 912 E1
'cp / (kJ kg-1 K-1) 0.417 301 218 E1        0.401 008 987 E1    0.465 580 682 E1
'w / (m s-1)        0.150 773 921 E4        0.163 469 054 E4    0.124 071 337 E4

Dim CRLF As String, TB As String
Dim txt As String, a As String
Dim p As Double, t As Double, d As Double

Dim i As Integer, j As Integer, row As String
Dim g As Double, gt As Double, gtt As Double, gtp As Double, gp As Double, gpp As Double

CRLF = Chr(13) + Chr(10)

txt = "Implementation of IAPWS-IF97 in Visual Basic" + CRLF
txt = txt + "for Publication in Ocean Science, 2009" + CRLF
txt = txt + "R. Feistel, IOW, Version " + Version + CRLF
txt = txt + "Compiled on " + CStr(Now) + CRLF + CRLF

txt = txt + "Function values as given in Table 5 of IF-97:" + CRLF
txt = txt + "Table 5. Thermodynamic property values calculated from Eq. (7) for selected values of T and p" + CRLF + CRLF

txt = txt + "                    T = 300 K       T = 300 K       T = 500 K" + CRLF
txt = txt + "                    p = 3 MPa       p = 80 MPa      p = 3 MPa" + CRLF + CRLF

For i = 1 To 6
  Select Case i
    Case 1:  row = "v / (m3 kg-1)       0.100215168E-2  0.971180894E-3  0.120241800E-2"
    Case 2:  row = "h / (kJ kg-1)       0.115331273E+3  0.184142828E+3  0.975542239E+3"
    Case 3:  row = "u / (kJ kg-1)       0.112324818E+3  0.106448356E+3  0.971934985E+3"
    Case 4:  row = "s / (kJ kg-1 K-1)   0.392294792     0.368563852     0.258041912E+1"
    Case 5:  row = "cp / (kJ kg-1 K-1)  0.417301218E+1  0.401008987E+1  0.465580682E+1"
    Case 6:  row = "w / (m s-1)         0.150773921E+4  0.163469054E+4  0.124071337E+4"
  End Select

  txt = txt + row + CRLF
  txt = txt + "this code:         "
  
  For j = 1 To 3
    'get temperature and pressure input
    t = Choose(j, 300, 300, 500)
    p = Choose(j, 3, 80, 3) * 1000000#

    g = fit_liq_g_if97_si(0, 0, t, p)
    gt = fit_liq_g_if97_si(1, 0, t, p)
    gp = fit_liq_g_if97_si(0, 1, t, p)
    gtt = fit_liq_g_if97_si(2, 0, t, p)
    gtp = fit_liq_g_if97_si(1, 1, t, p)
    gpp = fit_liq_g_if97_si(0, 2, t, p)

    Select Case i
      Case 1:  txt = txt + Left(EFormat(gp, 9) + Space(16), 16)
      Case 2:  txt = txt + Left(EFormat(0.001 * (g - t * gt), 9) + Space(16), 16)
      Case 3:  txt = txt + Left(EFormat(0.001 * (g - t * gt - p * gp), 9) + Space(16), 16)
      Case 4:  txt = txt + Left(EFormat(-0.001 * gt, 9) + Space(16), 16)
      Case 5:  txt = txt + Left(EFormat(-0.001 * t * gtt, 9) + Space(16), 16)
      Case 6:  d = gtp * gtp - gtt * gpp
               a = "invalid value"
               If d <> 0 Then
                 d = gtt / d
                 If d >= 0 Then
                   a = Left(EFormat(gp * Sqr(d), 9) + Space(16), 16)
                 End If
               End If
               txt = txt + Left(a + Space(16), 16)
    End Select

  Next j

  txt = txt + CRLF + CRLF
  
Next i

chk_IAPWS97_Table5 = txt

End Function

'=========================================================================
Public Function chk_IAPWS97_Table15() As String

'Table 15. Thermodynamic property values calculated from Eq. (15) for selected values of T and p a
'                   T = 300 K,       T = 700 K,       T = 700 K
'                   p = 0.0035 MPa   p = 0.0035 MPa   p = 30 MPa
'v / (m3 kg-1)      0.394 913 866 E2 0.923 015 898 E2 0.542 946 619 E-2
'h / (kJ kg-1)      0.254 991 145 E4 0.333 568 375 E4 0.263 149 474 E4
'u / (kJ kg-1)      0.241 169 160 E4 0.301 262 819 E4 0.246 861 076 E4
's / (kJ kg-1 K-1)  0.852 238 967 E1 0.101 749 996 E2 0.517 540 298 E1
'cp / (kJ kg-1 K-1) 0.191 300 162 E1 0.208 141 274 E1 0.103 505 092 E2
'w / (m s-1)        0.427 920 172 E3 0.644 289 068 E3 0.480 386 523 E3

Dim CRLF As String, TB As String
Dim txt As String, a As String
Dim p As Double, t As Double, d As Double

Dim i As Integer, j As Integer, row As String
Dim g As Double, gt As Double, gtt As Double, gtp As Double, gp As Double, gpp As Double

CRLF = Chr(13) + Chr(10)
TB = Chr(9)

txt = "Implementation of IAPWS-IF97 in Visual Basic" + CRLF
txt = txt + "for Publication in Ocean Science, 2009" + CRLF
txt = txt + "R. Feistel, IOW, Version " + Version + CRLF
txt = txt + "Compiled on " + CStr(Now) + CRLF + CRLF

txt = txt + "Function values as given in Table 15 of IF-97:" + CRLF
txt = txt + "Table 15. Thermodynamic property values calculated from Eq. (15) for selected values of T and p" + CRLF + CRLF

txt = txt + "                    T = 300 K       T = 700 K       T = 700 K" + CRLF
txt = txt + "                    p = 3500 Pa     p = 3500 Pa     p = 30 MPa" + CRLF + CRLF

For i = 1 To 6
  Select Case i
    Case 1:  row = "v / (m3 kg-1)       0.394913866E+2  0.923015898E+2  0.542946619E-2"
    Case 2:  row = "h / (kJ kg-1)       0.254991145E+4  0.333568375E+4  0.263149474E+4"
    Case 3:  row = "u / (kJ kg-1)       0.241169160E+4  0.301262819E+4  0.246861076E+4"
    Case 4:  row = "s / (kJ kg-1 K-1)   0.852238967E+1  0.101749996E+2  0.517540298E+1"
    Case 5:  row = "cp / (kJ kg-1 K-1)  0.191300162E+1  0.208141274E+1  0.103505092E+2"
    Case 6:  row = "w / (m s-1)         0.427920172E+3  0.644289068E+3  0.480386523E+3"
  End Select

  txt = txt + row + CRLF
  txt = txt + "this code:         "
  
  For j = 1 To 3
    'get temperature and pressure input
    t = Choose(j, 300#, 700#, 700#)
    p = Choose(j, 3500#, 3500#, 30000000#)

    g = fit_vap_g_if97_si(0, 0, t, p)
    gt = fit_vap_g_if97_si(1, 0, t, p)
    gp = fit_vap_g_if97_si(0, 1, t, p)
    gtt = fit_vap_g_if97_si(2, 0, t, p)
    gtp = fit_vap_g_if97_si(1, 1, t, p)
    gpp = fit_vap_g_if97_si(0, 2, t, p)

    Select Case i
      Case 1:  txt = txt + Left(EFormat(gp, 9) + Space(16), 16)
      Case 2:  txt = txt + Left(EFormat(0.001 * (g - t * gt), 9) + Space(16), 16)
      Case 3:  txt = txt + Left(EFormat(0.001 * (g - t * gt - p * gp), 9) + Space(16), 16)
      Case 4:  txt = txt + Left(EFormat(-0.001 * gt, 9) + Space(16), 16)
      Case 5:  txt = txt + Left(EFormat(-0.001 * t * gtt, 9) + Space(16), 16)
      Case 6:  d = gtp * gtp - gtt * gpp
               a = "invalid value"
               If d <> 0 Then
                 d = gtt / d
                 If d >= 0 Then
                   a = Left(EFormat(gp * Sqr(d), 9) + Space(16), 16)
                 End If
               End If
               txt = txt + Left(a + Space(16), 16)
    End Select

  Next j

  txt = txt + CRLF + CRLF
  
Next i

chk_IAPWS97_Table15 = txt

End Function

'=========================================================================
Public Function chk_IAPWS97_Table(ByVal number As Integer) As String

Select Case number
  Case 5:    chk_IAPWS97_Table = chk_IAPWS97_Table5
  Case 15:   chk_IAPWS97_Table = chk_IAPWS97_Table15
  Case Else: chk_IAPWS97_Table = "Only Table 5 or 15 is available"
End Select

End Function

