Attribute VB_Name = "Flu_3b_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules:
'     Constants_0_Mdl, file Constants_0.bas
'     Flu_2_Mdl,     file Flu_2.bas
'     Flu_3a_Mdl,    file Flu_3a.bas

'#########################################################################

'=========================================================================
'This module implements thermodynamic properties of liquid water and vapour
'as functions of temperature and pressure, computed numerically from
'the Helmholtz function of fluid water.

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98

Private Const Version = "11 Jun 2009"

'=========================================================================
Public Function liq_cp_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   cp(T,P) = T * (deta/dT)_P  isobaric heat capacity of liquid water in J/(kg K),
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: liq_cp_si(300, 1e5) = 4180.63952202258

Dim d As Double

liq_cp_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = liq_density_si(t_si, p_si)
If d = ErrorReturn Then Exit Function

liq_cp_si = flu_cp_si(t_si, d)

End Function

'=========================================================================
Public Function liq_cv_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   cv(T,P) = T * (deta/dT)_D  isochoric heat capacity of liquid water in J/(kg K),
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: liq_cv_si(300, 1e5) = 4130.17861503357

Dim d As Double

liq_cv_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = liq_density_si(t_si, p_si)
If d = ErrorReturn Then Exit Function

liq_cv_si = flu_cv_si(t_si, d)

End Function

'=========================================================================
Public Function liq_enthalpy_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   h(T,P) = f - T*(df/dT)_D + D*(df/dD)_T enthalpy of liquid water in J/kg,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: liq_enthalpy_si(300, 1e5) = 112653.679688599

Dim d As Double

liq_enthalpy_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = liq_density_si(t_si, p_si)
If d = ErrorReturn Then Exit Function

liq_enthalpy_si = flu_enthalpy_si(t_si, d)

End Function

'=========================================================================
Public Function liq_entropy_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   eta(T,P) = - (df/dT)_D entropy of liquid water in J/(kg K),
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: liq_entropy_si(300, 1e5) = 393.062433814569

Dim d As Double

liq_entropy_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = liq_density_si(t_si, p_si)
If d = ErrorReturn Then Exit Function

liq_entropy_si = flu_entropy_si(t_si, d)

End Function

'=========================================================================
Public Function liq_expansion_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   alpha(T,P) = -(1/D) * (dD/dT)_P  thermal expansion of liquid water in 1/K,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: liq_expansion_si(300, 1e5) = 2.74803716255683E-04

Dim d As Double

liq_expansion_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = liq_density_si(t_si, p_si)
If d = ErrorReturn Then Exit Function

liq_expansion_si = flu_expansion_si(t_si, d)

End Function

'=========================================================================
Public Function liq_gibbs_energy_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   g(T,P) = f + D*(df/dD)_T   Gibbs energy of liquid water in J/kg,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: liq_gibbs_energy_si(300, 1e5) = -5265.0504557718

Dim d As Double

liq_gibbs_energy_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = liq_density_si(t_si, p_si)
If d = ErrorReturn Then Exit Function

liq_gibbs_energy_si = flu_gibbs_energy_si(t_si, d)

End Function

'=========================================================================
Public Function liq_internal_energy_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   u(T,P) = f - T*(df/dT)_D internal energy of liquid water in J/kg,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: liq_internal_energy_si(300, 1e5) = 112553.334132656

Dim d As Double

liq_internal_energy_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = liq_density_si(t_si, p_si)
If d = ErrorReturn Then Exit Function

liq_internal_energy_si = flu_internal_energy_si(t_si, d)

End Function

'=========================================================================
Public Function liq_kappa_s_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   kappa_s(T,P) = (1/D) * (dD/dP)_eta  isentropic compressibility of liquid water in 1/Pa,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: liq_kappa_s_si(300, 1e5) = 4.45077521252688E-10

Dim d As Double

liq_kappa_s_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = liq_density_si(t_si, p_si)
If d = ErrorReturn Then Exit Function

liq_kappa_s_si = flu_kappa_s_si(t_si, d)

End Function

'=========================================================================
Public Function liq_kappa_t_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   kappa_t(T,P) = (1/D) * (dD/dP)_T  isothermal compressibility of liquid water in 1/K,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: liq_kappa_t_si(300, 1e5) = 4.50515304335745E-10

Dim d As Double

liq_kappa_t_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = liq_density_si(t_si, p_si)
If d = ErrorReturn Then Exit Function

liq_kappa_t_si = flu_kappa_t_si(t_si, d)

End Function

'=========================================================================
Public Function liq_lapserate_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   gamma(T,P) = (dT/dP)_eta  adiabatic lapse rate of liquid water in K/Pa,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: liq_lapserate_si(300, 1e5) = 1.97878804448096E-08

Dim d As Double

liq_lapserate_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = liq_density_si(t_si, p_si)
If d = ErrorReturn Then Exit Function

liq_lapserate_si = flu_lapserate_si(t_si, d)

End Function

'=========================================================================
Public Function liq_soundspeed_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   c(T,P) = sqr[ (dP/dD)_eta ] sound speed of liquid water in m/s,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: liq_soundspeed_si(300, 1e5) = 1501.52041505665

Dim d As Double

liq_soundspeed_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = liq_density_si(t_si, p_si)
If d = ErrorReturn Then Exit Function

liq_soundspeed_si = flu_soundspeed_si(t_si, d)

End Function

'=========================================================================
Public Function vap_cp_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   cp(T,P) = T * (deta/dT)_P  isobaric heat capacity of water vapour in J/(kg K),
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: vap_cp_si(300, 1e3) = 1874.12149028065

Dim d As Double

vap_cp_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = vap_density_si(t_si, p_si)
If d = ErrorReturn Then Exit Function

vap_cp_si = flu_cp_si(t_si, d)

End Function

'=========================================================================
Public Function vap_cv_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   cv(T,P) = T * (deta/dT)_D  isochoric heat capacity of water vapour in J/(kg K),
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: vap_cv_si(300, 1e3) = 1410.22845788565

Dim d As Double

vap_cv_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = vap_density_si(t_si, p_si)
If d = ErrorReturn Then Exit Function

vap_cv_si = flu_cv_si(t_si, d)

End Function

'=========================================================================
Public Function vap_enthalpy_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   h(T,P) = f - T*(df/dT)_D + D*(df/dD)_T enthalpy of water vapour in J/kg,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: vap_enthalpy_si(300, 1e3) = 2551013.47892339

Dim d As Double

vap_enthalpy_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = vap_density_si(t_si, p_si)
If d = ErrorReturn Then Exit Function

vap_enthalpy_si = flu_enthalpy_si(t_si, d)

End Function

'=========================================================================
Public Function vap_entropy_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   eta(T,P) = - (df/dT)_D entropy of water vapour in J/(kg K),
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: vap_entropy_si(300, 1e3) = 9103.67940087138

Dim d As Double

vap_entropy_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = vap_density_si(t_si, p_si)
If d = ErrorReturn Then Exit Function

vap_entropy_si = flu_entropy_si(t_si, d)

End Function

'=========================================================================
Public Function vap_expansion_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   alpha(T,P) = -(1/D) * (dD/dT)_P  thermal expansion of water vapour in 1/K,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: vap_expansion_si(300, 1e3) = 3.34352010566687E-03

Dim d As Double

vap_expansion_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = vap_density_si(t_si, p_si)
If d = ErrorReturn Then Exit Function

vap_expansion_si = flu_expansion_si(t_si, d)

End Function

'=========================================================================
Public Function vap_gibbs_energy_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   g(T,P) = f + D*(df/dD)_T   Gibbs energy of water vapour in J/kg,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: vap_gibbs_energy_si(300, 1e3) = -180090.341338025

Dim d As Double

vap_gibbs_energy_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = vap_density_si(t_si, p_si)
If d = ErrorReturn Then Exit Function

vap_gibbs_energy_si = flu_gibbs_energy_si(t_si, d)

End Function

'=========================================================================
Public Function vap_internal_energy_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   u(T,P) = f - T*(df/dT)_D internal energy of water vapour in J/kg,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: vap_internal_energy_si(300, 1e3) = 2412625.00085396

Dim d As Double

vap_internal_energy_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = vap_density_si(t_si, p_si)
If d = ErrorReturn Then Exit Function

vap_internal_energy_si = flu_internal_energy_si(t_si, d)

End Function

'=========================================================================
Public Function vap_kappa_s_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   kappa_s(T,P) = (1/D) * (dD/dP)_eta  isentropic compressibility of water vapour in 1/Pa,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: vap_kappa_s_si(300, 1e3) = 7.52840457971025E-04

Dim d As Double

vap_kappa_s_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = vap_density_si(t_si, p_si)
If d = ErrorReturn Then Exit Function

vap_kappa_s_si = flu_kappa_s_si(t_si, d)

End Function

'=========================================================================
Public Function vap_kappa_t_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   kappa_t(T,P) = (1/D) * (dD/dP)_T  isothermal compressibility of water vapour in 1/K,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: vap_kappa_t_si(300, 1e3) = 1.00048646242156E-03

Dim d As Double

vap_kappa_t_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = vap_density_si(t_si, p_si)
If d = ErrorReturn Then Exit Function

vap_kappa_t_si = flu_kappa_t_si(t_si, d)

End Function

'=========================================================================
Public Function vap_lapserate_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   gamma(T,P) = (dT/dP)_eta  adiabatic lapse rate of water vapour in K/Pa,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: vap_lapserate_si(300, 1e3) = 7.40674488634917E-02

Dim d As Double

vap_lapserate_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = vap_density_si(t_si, p_si)
If d = ErrorReturn Then Exit Function

vap_lapserate_si = flu_lapserate_si(t_si, d)

End Function

'=========================================================================
Public Function vap_soundspeed_si(ByVal t_si As Double, ByVal p_si As Double) As Double

'returns   c(T,P) = sqr[ (dP/dD)_eta ] sound speed of water vapour in m/s,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in Flu_3a

'Check value with default settings: vap_soundspeed_si(300, 1e3) = 428.744430494758

Dim d As Double

vap_soundspeed_si = ErrorReturn

If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = vap_density_si(t_si, p_si)
If d = ErrorReturn Then Exit Function

vap_soundspeed_si = flu_soundspeed_si(t_si, d)

End Function



