Attribute VB_Name = "Convert_5_Mdl"
Option Explicit

'#########################################################################

'This module requires the library module
'     Constants_0_Mdl, file Constants_0.bas
'     Convert_0_Mdl,   file Convert_0.bas

'#########################################################################

'=========================================================================
'This module implements conversion formulas between different scales and units

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98

Private Const Version = "07 Jun 2010"

'==========================================================================
Public Function cnv_pressure(ByVal unit_out As String, _
                             ByVal p_in As Double, _
                             ByVal unit_in As String, _
                    Optional ByVal lat As Double = 45#) As Double

'NOTE: UNIT_IN AND UNIT_OUT ARE CASE SENSITIVE. mPa IS NOT THE SAME AS MPA.
'      ALL UNITS MUST BE IN CAPITAL LETTERS

'check values:
'cnv_pressure("MPa",4000,"dbar") = 40.101325

'cnv_pressure("DBAR", 10331325, "PA") = 1023#
'cnv_pressure("PA", 10331325, "DBAR") = 10331325#
'cnv_pressure("TORR", 10331325, "PA") = 77491.3101406
'cnv_pressure("PA", 77491.3101406, "TORR") = 10331325#
'cnv_pressure("KGF", 10331325, "PA") = 105350.19604
'cnv_pressure("PA", 105350.19604, "KGF") = 10331325#
'cnv_pressure("ATM", 10331325, "PA") = 105350.19604
'cnv_pressure("PA", 105350.19604, "ATM") = 10331325#
'cnv_pressure("LBF/IN2", 10331325, "PA") = 1498.42272437
'cnv_pressure("PA", 1498.42272437, "LBF/IN2") = 10331325#
'cnv_pressure("PSI", 10331325, "PA") = 1498.42272437
'cnv_pressure("PA", 1498.42272437, "PSI") = 10331325#
'cnv_pressure("MPA", 4000, "DBAR") = 40.101325
'cnv_pressure("M", 1023#, "DBAR") = 1011.94563591
'cnv_pressure("DBAR", 1011.94563591, "M") = 1023

Dim p_si As Double

If unit_in = unit_out Then
  cnv_pressure = p_in
  Exit Function
End If

cnv_pressure = ErrorReturn

'convert input to SI
Select Case unit_in
  Case ("PA"):      p_si = p_in
  Case ("HPA"):     p_si = p_in * 100#
  Case ("MPA"):     p_si = p_in * 1000000#
  Case ("KPA"):     p_si = p_in * 1000#
  Case ("DBAR"):    p_si = p_in * 10000# + 101325#    'SEA PRESSURE ASSUMED
  'Case ("DB"):      p_si = p_in * 10000# + 101325#    'SEA PRESSURE ASSUMED
  Case ("MBAR"):    p_si = p_in * 100#
  'Case ("MB"):      p_si = p_in * 100#
  Case ("BAR"):     p_si = p_in * 100000#
  Case ("KBAR"):    p_si = p_in * 100000000#
  Case ("TORR"):    p_si = p_in * 101325# / 760#
  Case ("MMHG"):    p_si = p_in * 101325# / 760#
  Case ("KGF"):     p_si = p_in * 98.0665   'ATM = KG/CM2 GAUGE
  Case ("ATM"):     p_si = p_in * 98.0665   'ATM = KG/CM2 GAUGE
  Case ("LBF/IN2"): p_si = p_in * 6894.8    'POUND PER SQUARE INCH
  Case ("PSI"):     p_si = p_in * 6894.8    'POUND PER SQUARE INCH
  Case ("M"):       p_si = 101325# + 10000# * sw_pres(p_in, lat)
  Case Else: Exit Function
End Select

'convert SI to output
Select Case unit_out
  Case ("PA"):      cnv_pressure = p_si
  Case ("HPA"):     cnv_pressure = p_si / 100#
  Case ("MPA"):     cnv_pressure = p_si / 1000000#
  Case ("KPA"):     cnv_pressure = p_si / 1000#
  Case ("DBAR"):    cnv_pressure = (p_si - 101325#) / 10000#   'SEA PRESSURE ASSUMED
  'Case ("DB"):      cnv_pressure = (p_si - 101325#) / 10000#   'SEA PRESSURE ASSUMED
  Case ("MBAR"):    cnv_pressure = p_si / 100#
  'Case ("MB"):      cnv_pressure = p_si / 100#
  Case ("BAR"):     cnv_pressure = p_si / 100000#
  Case ("KBAR"):    cnv_pressure = p_si / 100000000#
  Case ("TORR"):    cnv_pressure = p_si / 101325# * 760#
  Case ("MMHG"):    cnv_pressure = p_si / 101325# * 760#
  Case ("KGF"):     cnv_pressure = p_si / 98.0665   'ATM = KG/CM2 GAUGE
  Case ("ATM"):     cnv_pressure = p_si / 98.0665   'ATM = KG/CM2 GAUGE
  Case ("LBF/IN2"): cnv_pressure = p_si / 6894.8    'POUND PER SQUARE INCH
  Case ("PSI"):     cnv_pressure = p_si / 6894.8    'POUND PER SQUARE INCH
  Case ("M"):       cnv_pressure = sw_dpth((p_si - 101325#) / 10000#, lat)
  Case Else: Exit Function
End Select
                         
End Function

'==========================================================================
Public Function cnv_temperature(ByVal unit_out As String, _
                                ByVal t_in As Double, _
                                ByVal unit_in As String) As Double
'==========================================================================

'NOTE: UNIT_IN AND UNIT_OUT ARE CASE SENSITIVE. DegC IS NOT THE SAME AS DEGC.
'      ALL UNITS MUST BE IN CAPITAL LETTERS
 
Dim t_si As Double, t_c As Double

'CHECK VALUE:
'cnv_temperature("DEGF(T48)",300,"K(T48)") = 80.33
'cnv_temperature("DEGF(T68)",300,"K(T48)") = 80.3139948157844
'cnv_temperature("DEGF(T90)",300,"K(T48)") = 80.3018730496216
'cnv_temperature("DEGC(T48)", 300, "K(T48)") = 26.85
'cnv_temperature("DEGC(T68)", 300, "K(T48)") = 26.841108231|2
'cnv_temperature("DEGC(T90)", 300, "K(T48)") = 26.834373916|7
'cnv_temperature("K(T68)", 300, "K(T48)") = 299.991108231
'cnv_temperature("K(T90)", 300, "K(T48)") = 299.98437391|7
'cnv_temperature("DEGF(T68)",299.991108231,"K(T68)") = 80.3139916337851
'cnv_temperature("DEGF(T90)",299.991108231,"K(T68)") = 80.3018698684571
'cnv_temperature("DEGC(T68)", 299.991108231, "K(T68)") = 26.8411064632
'cnv_temperature("DEGC(T90)", 299.991108231, "K(T68)") = 26.8343721491
'cnv_temperature("K(T90)", 299.991108231, "K(T68)") = 299.984372149
'cnv_temperature("DEGF(T90)",299.984372149,"K(T90)") = 80.3018698682001
'cnv_temperature("DEGC(T90)", 299.984373917, "K(T90)") = 26.834373917

If unit_in = unit_out Then
  cnv_temperature = t_in
  Exit Function
End If

cnv_temperature = ErrorReturn

'CONVERT INPUT TO T90
Select Case (unit_in)
  Case ("DEGF(T48)"): t_c = (t_in - 32#) / 1.8
                      t_si = t_si_from_t48_k(t_k_from_t_c(t_c))
  Case ("DEGF(T68)"): t_c = (t_in - 32#) / 1.8
                      t_si = t_si_from_t68_k(t_k_from_t_c(t_c))
  Case ("DEGF(T90)"): t_c = (t_in - 32#) / 1.8
                      t_si = t_k_from_t_c(t_c)
  Case ("DEGC(T48)"): t_si = t_si_from_t48_k(t_k_from_t_c(t_in))
  Case ("DEGC(T68)"): t_si = t_si_from_t68_k(t_k_from_t_c(t_in))
  Case ("DEGC(T90)"): t_si = t_k_from_t_c(t_in)
  Case "C", "C":     t_si = t_k_from_t_c(t_in)
  Case ("K(T48)"):    t_si = t_si_from_t48_k(t_in)
  Case ("K(T68)"):    t_si = t_si_from_t68_k(t_in)
  Case ("K(T90)"):    t_si = t_in
  Case ("K"):         t_si = t_in
  Case Else: Exit Function
End Select

'CONVERT T90 TO OUTPUT
Select Case (unit_out)
  Case ("DEGF(T48)")
    If (unit_in = "K(T48)") Then
       cnv_temperature = 32# + 1.8 * t_c_from_t_k(t_in)
    ElseIf (unit_in = "DEGC(T48)") Then
       cnv_temperature = 32# + 1.8 * t_in
    Else
      Exit Function  't48_f is not a valid output except for t48_k or t48_c input
    End If
  Case ("DEGF(T68)"): cnv_temperature = 32# + 1.8 * t_c_from_t_k(t68_k_from_t_si(t_si))
  Case ("DEGF(T90)"): cnv_temperature = 32# + 1.8 * t_c_from_t_k(t_si)
  Case ("DEGC(T48)")
    If (UCase(unit_in) = "K(T48)") Then
       cnv_temperature = t_c_from_t_k(t_in)
    Else
       Exit Function  't48_c is not a valid output except for t48_k input
    End If
  Case ("DEGC(T68)"): cnv_temperature = t_c_from_t_k(t68_k_from_t_si(t_si))
  Case ("DEGC(T90)"): cnv_temperature = t_c_from_t_k(t_si)
  Case "C", "C":     cnv_temperature = t_c_from_t_k(t_si)
  Case ("K(T68)"):    cnv_temperature = t68_k_from_t_si(t_si)
  Case ("K(T90)"):    cnv_temperature = t_si
  Case ("K"):         cnv_temperature = t_si
  Case Else: Exit Function
End Select
                         
End Function

'==========================================================================
Public Function cnv_salinity(ByVal unit_out As String, _
                             ByVal s_in As Double, _
                             ByVal unit_in As String, _
                    Optional ByVal t_si As Double = 298.15, _
                    Optional ByVal p_si As Double = ErrorReturn, _
                    Optional ByVal lon As Double = ErrorReturn, _
                    Optional ByVal lat As Double = ErrorReturn) As Double
'==========================================================================

'NOTE: UNIT_IN AND UNIT_OUT ARE CASE SENSITIVE. Cl IS NOT THE SAME AS CL.
 
' CHECK VALUES (Fortran version):
'CNV_SALINITY("CL",0.0357,"KG/KG(ABS)",,101325+1023E4,201,-21)  = 0.196660177563E+02
'CNV_SALINITY("KN",0.0357,"KG/KG(ABS)",,101325+1023E4,201,-21)  = 0.355271620502E+02
'CNV_SALINITY("CND",0.0357,"KG/KG(ABS)",273.15+25.5,101325+1023E4,201,-21)  = 0.127556680822E+01
'CNV_SALINITY("PSU",0.0357,"KG/KG(ABS)",,101325+1023E4,201,-21)  = 0.355276443777E+02
'CNV_SALINITY("KG/KG(REF)",0.0357d0,"KG/KG(ABS)",,101325+1023E4,201,-21)  = 0.356951724471E-01
'CNV_SALINITY("KG/KG(ABS)",0.0356951724471,"KG/KG(REF)",,101325+1023E4,201,-21)  = 0.0357
'CNV_SALINITY("CL", 35.5276443777339, "PSU") = 19.6660177563
'CNV_SALINITY("KN", 35.5276443777339, "PSU") = 35.5271620502
'CNV_SALINITY("CND",35.52764437773386,"PSU",273.15+25.5,101325+1023E4) = 1.27556680821579
'CNV_SALINITY("KG/KG(REF)", 35.5276443777339, "PSU") = 0.0356951724471
'CNV_SALINITY("KG/KG(ABS)",35.52764437773386d0,"PSU",,101325+1023E4,201,-21)  = 0.0357
'CNV_SALINITY("PSU", 35.5271620502, "KN") = 35.5276443777
'CNV_SALINITY("KG/KG(ABS)",0.355271620502E+02,"KN",,101325+1023E4,201,-21)  = 0.357000000000E-01
'CNV_SALINITY("PSU",0.127556680822E+01,"CND",273.15+25.5,101325+1023E4) = 35.5276443778661
'CNV_SALINITY("KG/KG(ABS)",0.127556680822E+01,"CND",273.15+25.5,101325+1023E4,201,-21)  = 0.0357

Const ups = SO_salinity_si / 35  'SR = SP * ups
Const ucl = 1.80655 * ups        'SR = Cl * ucl
Const c35 = 4.2914               'C(35,15,0) = 42.914 mS/cm = 4.2914 S/m,  Culkin & Smith

Dim s_si As Double, psal As Double, t_c As Double, t68_c As Double
Dim p_dbar As Double, c_ratio As Double, p_loc As Double

If unit_in = unit_out Then
  cnv_salinity = s_in
  Exit Function
End If

cnv_salinity = ErrorReturn

p_loc = p_si
If p_si = ErrorReturn Or _
   lon = ErrorReturn Or _
   lat = ErrorReturn Then
  p_loc = 101325
  lon = -20
  lat = 50
End If
If p_si = ErrorReturn Then p_si = 101325

'CONVERT INPUT TO REFERENCE SALINITY IN KG/KG
Select Case unit_in

  'Chlorinity
  Case "CL": s_si = s_in * ucl

  'Knudsen Salinity
  Case "KN": s_si = (s_in - 0.03) / 1.805 * ucl

  'Conductivity
  Case "MS/CM", "MMHO/CM" 'conductivity in mS/cm = mmho/cm
    'Temperature and pressure is needed to convert from CONDUCTIVITY
    'If missing, proceeding with t90 = 25C and atmospheric pressure
     t68_c = cnv_temperature("DEGC(T68)", t_si, "K(T90)")
     p_dbar = cnv_pressure("DBAR", p_si, "PA")
     c_ratio = 0.1 * s_in / c35
     s_si = sal78_from_cnd(c_ratio, t68_c, p_dbar) * ups
     
  Case "S/M"    'conductivity in S/m
     t68_c = cnv_temperature("DEGC(T68)", t_si, "K(T90)")
     p_dbar = cnv_pressure("DBAR", p_si, "PA")
     c_ratio = s_in / c35
     s_si = sal78_from_cnd(c_ratio, t68_c, p_dbar) * ups

  Case "CND"    'conductivity ratio
     t68_c = cnv_temperature("DEGC(T68)", t_si, "K(T90)")
     p_dbar = cnv_pressure("DBAR", p_si, "PA")
     c_ratio = s_in
     s_si = sal78_from_cnd(c_ratio, t68_c, p_dbar) * ups

  'PRACTICAL SALINITY
  Case "PSU", "PSS", "ONE": s_si = s_in * ups

  'Reference SALINITY
  Case "KG/KG(REF)": s_si = s_in

  Case "G/KG(REF)": s_si = 0.001 * s_in
  
  'ABSOLUTE SALINITY
  Case "KG/KG(ABS)"
        'Position and pressure are needed to convert from ABSOLUTE SALINITY
        'in order to determine the ABSOLUTE SALINITY anomaly
        'If missing, proceeding value from surface North Atlantic (near zero)
        'Pressure is needed to convert from ABSOLUTE SALINITY
        'If missing, proceeding with atmospheric pressure
     s_si = psal_from_asal(s_in, lon, lat, p_loc) * ups
     
  Case "G/KG(ABS)"
     s_si = psal_from_asal(0.001 * s_in, lon, lat, p_loc) * ups
     
  Case Else: Exit Function
  
End Select

'CONVERT SI REFERENCE SALINITY TO OUTPUT
Select Case unit_out

  'Chlorinity
  Case "CL": cnv_salinity = s_si / ucl

  'Knudsen Salinity
  Case "KN": cnv_salinity = 0.03 + 1.805 * s_si / ucl

  'Conductivity
  Case "MS/CM", "MMHO/CM" 'conductivity in mS/cm = mmho/cm
    'Temperature and pressure is needed to convert to CONDUCTIVITY
    'If missing, proceeding with t90 = 25C and atmospheric pressure
     t68_c = cnv_temperature("DEGC(T68)", t_si, "K(T90)")
     p_dbar = cnv_pressure("DBAR", p_si, "PA")
     psal = s_si / ups
     cnv_salinity = 10 * c35 * cnd_from_sal78(psal, t68_c, p_dbar)
     
  Case "S/M"  'conductivity in S/m
     t68_c = cnv_temperature("DEGC(T68)", t_si, "K(T90)")
     p_dbar = cnv_pressure("DBAR", p_si, "PA")
     psal = s_si / ups
     cnv_salinity = c35 * cnd_from_sal78(psal, t68_c, p_dbar)
     
  Case "CND"    'conductivity ratio
     t68_c = cnv_temperature("DEGC(T68)", t_si, "K(T90)")
     p_dbar = cnv_pressure("DBAR", p_si, "PA")
     psal = s_si / ups
     cnv_salinity = cnd_from_sal78(psal, t68_c, p_dbar)

  'PRACTICAL SALINITY
  Case "PSU", "PSS", "ONE": cnv_salinity = s_si / ups

  'Reference SALINITY
  Case "KG/KG(REF)": cnv_salinity = s_si

  Case "G/KG(REF)": cnv_salinity = 1000# * s_si
  
  'ABSOLUTE SALINITY
  Case "KG/KG(ABS)"
        'Position and pressure are needed to convert to ABSOLUTE SALINITY
        'in order to determine the ABSOLUTE SALINITY anomaly
        'If missing, proceeding value from surface North Atlantic (near zero)
        'Pressure is needed to convert from ABSOLUTE SALINITY
        'If missing, proceeding with atmospheric pressure
     psal = s_si / ups
     cnv_salinity = asal_from_psal(psal, lon, lat, p_loc)
     
  Case "G/KG(ABS)"
     psal = s_si / ups
     cnv_salinity = 1000# * asal_from_psal(psal, lon, lat, p_loc)

  Case Else: Exit Function
  
End Select
                         
End Function

'==========================================================================
Private Function present(ByVal x As Double) As Boolean

present = (x <> ErrorReturn)

End Function

'==========================================================================
Private Function t_k_from_t_c(ByVal t_c As Double) As Double

'this function computes temperature in K from temperature in C (ITS-90, IPTS-68, IPTS-48)

t_k_from_t_c = 273.15 + t_c

End Function

'==========================================================================
Private Function t_c_from_t_k(ByVal T_K As Double) As Double

'this function computes temperature in C from temperature in K (ITS-90, IPTS-68, IPTS-48)

t_c_from_t_k = T_K - 273.15

End Function

'==========================================================================
Private Function t68_k_from_t48_k(ByVal t48_k)

'this function returns IPTS-68 temperature in K computed from IPTS-48 temperature in K.

'VB code adapted from:

'C^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
'C
'C                          RUHR-UNIVERSITAET BOCHUM
'C                        Lehrstuhl fuer Thermodynamik
'C                          Prof. Dr.-Ing. W. Wagner
'C                              D - 44780 BOCHUM
'C                             Tel.: 0234/700-3033
'C                             FAX : 0234/7094 -163
'C
'C^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

'c
'C PROGRAMM ZUR TEMPERATURUMRECHNUNG: IPTS-48 IN IPTS-68
'c
'C DER WERT 't48_k' IN DER IPTS-48 (IN KELVIN) WIRD VOM RUFENDEN PROGRAMM UEBENOMMEN
'C UND IN t68_k_from_t48_k (IN KELVIN) WIEDER UEBERGEBEN.
'c

Const A48A4 = 0.003984517
Const B48 = -0.000000585502
Const C48C4 = -4.35716E-12
Const A48A68 = -0.00000019539
Const B48B68 = 0.0000000019539
Const A68 = 0.003984712
Const B68 = -0.000000587456

Dim t48 As Double, t68 As Double, T68ALT As Double, TSTRI As Double
Dim HG1 As Double, HG2 As Double, HG3 As Double, FKT As Double
Dim i As Integer, k As Integer

Dim a(21)

a(1) = 273.15
a(2) = 250.846209678803
a(3) = 135.099869965
a(4) = 52.78567590085
a(5) = 27.6768548854105
a(6) = 39.1053205376684
a(7) = 65.5613230578069
a(8) = 80.8035868559867
a(9) = 70.5242118234052
a(10) = 44.7847589638966
a(11) = 21.2525653556078
a(12) = 7.67976358170846
a(13) = 2.1368945938285
a(14) = 0.459843348928069
a(15) = 7.63614629231648E-02
a(16) = 9.69328620373121E-03
a(17) = 9.23069154007008E-04
a(18) = 6.38116590952654E-05
a(19) = 3.02293237874619E-06
a(20) = 8.7755139130376E-08
a(21) = 1.17702613125477E-09

      If (t48_k < 90#) Then
        'MsgBox "ACHTUNG: Keine Umrechnung T48T68 fuer t48_k < 90 K '"
        t68_k_from_t48_k = t48_k
        Exit Function
      End If


      If (t48_k > 273.15) Then GoTo 10

      t48 = t48_k - 273.15
      HG1 = 1# + A48A4 * t48 + B48 * t48 * t48 + C48C4 * t48 * t48 * t48 * (t48 - 100#)
      HG1 = Log(HG1)
      t68 = 0#
      For k = 1 To 21
        i = 22 - k
        t68 = t68 * HG1 + a(i)
      Next k

      t68_k_from_t48_k = t68
      Exit Function

10:
      t48 = t48_k - 273.15
      HG1 = A68 + 2# * B68 * t48
      HG2 = A48A68 * t48 + B48B68 * t48 * t48
      t68 = t48
      T68ALT = t48
      TSTRI = t48

11:   FKT = 0.045 * TSTRI / 100# * (TSTRI / 100# - 1#) * (TSTRI / 419.58 - 1#) * (TSTRI / 630.74 - 1#)
      HG3 = B68 * (t68 - t48 - FKT) * (t68 - t48 - FKT)
      t68 = t48 + FKT + HG2 / HG1 - HG3 / HG1
      If (Abs(t68 - T68ALT) < 0.000000000001) Then GoTo 12   'NOTE 1D-5 REPLACED BY 1D-12
      T68ALT = t68
      TSTRI = t68
      GoTo 11

12:
99:   t68_k_from_t48_k = t68 + 273.15

End Function

'==========================================================================
Private Function t_si_from_t48_k(ByVal t48_k As Double) As Double

t_si_from_t48_k = t_si_from_t68_k(t68_k_from_t48_k(t48_k))

End Function

'==========================================================================
Private Function t_si_from_t68_k(ByVal t68_k As Double) As Double

'this function returns ITS-90 temperature in K computed from IPTS-68 temperature in K.

'VB code adapted from:

'C^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
'c
'C                          RUHR-UNIVERSITAET BOCHUM
'C                        Lehrstuhl fuer Thermodynamik
'C                          Prof. Dr.-Ing. W. Wagner
'C                              D - 44780 BOCHUM
'C                             Tel.: 0234/700-3033
'C                             FAX : 0234/7094 -163
'c
'C^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
'c
'C   REGION 1:  13.8 K  < t68_k <   83.8 K
'c
'C   REGION 2:  83.8 K  < t68_k <  903.75 K
'c
'C   REGION 3: 903.89 K < T68 < 1337.58 K
'c
'C   REGION 4:     1337.58 < T68
'c
'C   INPUT:  t68_k      TEMPERATURE IN K (IPTS-68)
'c
'C   OUTPUT: t68_k_from_t68_k   TEMPERATURE IN K (ITS-90)
'c
'c From:
'c
'C  Supplementary Information for the ITS-90, Bureau International des Poids et
'C  Mesures, Pav Breteuil, 92312 Sevres Cedex, France, 1990.
'c
'C  R.L.Rusby: The Conversion of Thermal Reference Values to the
'C             ITS-90. J. Chem. Thermodynamics, 1991, 23, 1153 - 1161.
'c
'C  Rusby et al.: International Report. Revised Values for (t90-t68) from
'C                630C to 1064C. Metrologia, 1994, 31, 149 - 153.
'c
'C^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

Dim DELTAT As Double, Tn As Double
Dim j As Integer

Dim a(0 To 12), b(1 To 8), c(0 To 5)

'c
a(0) = -0.005903
a(1) = 0.008174
a(2) = -0.061924
a(3) = -0.193388
a(4) = 1.490793
a(5) = 1.252347
a(6) = -9.835868
a(7) = 1.411912
a(8) = 25.277595
a(9) = -19.183815
a(10) = -18.437089
a(11) = 27.000895
a(12) = -8.716324
'c
b(1) = -0.148759
b(2) = -0.267408
b(3) = 1.08076
b(4) = 1.269056
b(5) = -4.089591
b(6) = -1.871251
b(7) = 7.438081
b(8) = -3.536296
'c
c(0) = 78.687209
c(1) = -0.47135991
c(2) = 0.0010954715
c(3) = -0.0000012357884
c(4) = 6.7736583E-10
c(5) = -1.4458081E-13
'c

      DELTAT = 0#
'c
      If (t68_k < 13.8) Then GoTo 999
'c
'C     ^* REGION 4 ^*
'c
      If (t68_k >= 1337.58) Then
       DELTAT = -0.25 * (t68_k / 1337.33) ^ 2
       GoTo 1000
'c
'C     ^* REGION 3 ^*
'c
'C DIE GLEICHUNG IST IN ABHAENGIGKEIT VON T90 FORMULIERT, FUER DIE
'C BERECHNUNG VON DELTAT WIRD HIER T68=T90 GESETZT /AK 16.7.96
'c
      ElseIf (t68_k >= 903.89) Then
       Tn = t68_k - 273.15
       For j = 1 To 5
         DELTAT = DELTAT + c(j) * Tn ^ j
       Next j

       DELTAT = DELTAT + c(0)
       GoTo 1000
'c
'C     ^* REGION 2 ^*
'c
      ElseIf (t68_k >= 83.8) Then
        Tn = (t68_k - 273.15) / 630#
        For j = 1 To 8
          DELTAT = DELTAT + b(j) * Tn ^ j
        Next j
       GoTo 1000
'c
'C     ^* REGION 1 ^*
'c
      ElseIf (t68_k >= 13.8) Then
       Tn = (t68_k - 40#) / 40#
       For j = 1 To 12
         DELTAT = DELTAT + a(j) * Tn ^ j
       Next j

       DELTAT = DELTAT + a(0)
       GoTo 1000
      End If
'c
999:  'MsgBox "Temperature out of range of validity, no conversion."
      t_si_from_t68_k = t68_k
      Exit Function

'c
1000:  t_si_from_t68_k = t68_k + DELTAT

End Function

'==========================================================================
Private Function t68_k_from_t_si(ByVal t_si As Double) As Double

'this function returns IPTS-68 temperature in K computed from ITS-90 temperature in K.

'VB code adapted from:

'C^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
'C
'C                          RUHR-UNIVERSITAET BOCHUM
'C                        Lehrstuhl fuer Thermodynamik
'C                          Prof. Dr.-Ing. W. Wagner
'C                              D - 44780 BOCHUM
'C                             Tel.: 0234/700-3033
'C                             FAX : 0234/7094 -163
'C
'C^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
'C
'C   REGION 1:  13.8 K  < t_si <   83.8 K
'C
'C   REGION 2:  83.8 K  < t_si <  903.75 K
'C
'C   REGION 3: 903.765 K < t_si < 1337.33 K
'C
'C   REGION 4:     1337.33 < t_si
'C
'C   INPUT:  t_si      TEMPERATURE IN K (ITS-90)
'C
'C   OUTPUT: t68_k_from_t_si   TEMPERATURE IN K (IPTS-68)
'C
'C From:
'C
'C  Supplementary Information for the ITS-90, Bureau International des Poids et
'C  Mesures, Pav Breteuil, 92312 Sevres Cedex, France, 1990.
'C
'C  R.L.Rusby: The Conversion of Thermal Reference Values to the
'C             ITS-90. J. Chem. Thermodynamics, 1991, 23, 1153 - 1161.
'C
'C  Rusby et al.: International Report. Revised Values for (t90-t68) from
'C                630C to 1064C. Metrologia, 1994, 31, 149 - 153.
'C
'C^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

Dim i As Integer, j As Integer
Dim DELTAT As Double, Tn As Double

Dim a(0 To 12), b(1 To 8), c(0 To 5)

a(0) = -0.005903
a(1) = 0.008174
a(2) = -0.061924
a(3) = -0.193388
a(4) = 1.490793
a(5) = 1.252347
a(6) = -9.835868
a(7) = 1.411912
a(8) = 25.277595
a(9) = -19.183815
a(10) = -18.437089
a(11) = 27.000895
a(12) = -8.716324

b(1) = -0.148759
b(2) = -0.267408
b(3) = 1.08076
b(4) = 1.269056
b(5) = -4.089591
b(6) = -1.871251
b(7) = 7.438081
b(8) = -3.536296

c(0) = 78.687209
c(1) = -0.47135991
c(2) = 0.0010954715
c(3) = -0.0000012357884
c(4) = 6.7736583E-10
c(5) = -1.4458081E-13

      DELTAT = 0
      If (t_si < 13.8) Then GoTo 999

'C
'C     ^* REGION 4 ^*
'C
      If (t_si >= 1337.33) Then
       DELTAT = -0.25 * (t_si / 1337.33) ^ 2
       GoTo 1000

'C
'C     ^* REGION 3 ^*
'C
      ElseIf (t_si >= 903.765) Then
       Tn = t_si - 273.15

       For j = 1 To 5
         DELTAT = DELTAT + c(j) * Tn ^ j
       Next j

       DELTAT = DELTAT + c(0)
       GoTo 1000
'C
'C     ^* REGION 2 ^*
'C
      ElseIf (t_si >= 83.8) Then
       Tn = (t_si - 273.15) / 630#

       For j = 1 To 8
        DELTAT = DELTAT + b(j) * Tn ^ j
       Next j
       GoTo 1000
'C
'C     ^* REGION 1 ^*
'C
      ElseIf (t_si >= 13.8) Then
       Tn = (t_si - 40#) / 40#
       For j = 1 To 12
         DELTAT = DELTAT + a(j) * Tn ^ j
       Next j

       DELTAT = DELTAT + a(0)
       GoTo 1000
      End If

999:   'MsgBox "Temperature out of range of validity, no conversion."
       t68_k_from_t_si = t_si
       Exit Function

1000:  t68_k_from_t_si = t_si - DELTAT

End Function

'==========================================================================
Private Function p_si_from_psea_pa(ByVal psea_pa As Double) As Double

'this function computes absolute pressure in Pa from sea pressure in Pa

p_si_from_psea_pa = psea_pa + 101325#

End Function

'==========================================================================
Private Function psea_pa_from_p_si(ByVal p_si As Double) As Double

'this function computes sea pressure in Pa from absolute pressure in Pa

psea_pa_from_p_si = p_si - 101325#

End Function

'==========================================================================
Private Function psea_pa_from_psea_bar(ByVal psea_bar As Double) As Double

'this function computes sea pressure in Pa from absolute pressure in Pa

psea_pa_from_psea_bar = psea_bar * 100000#

End Function

'==========================================================================
Private Function psea_bar_from_psea_pa(ByVal psea_pa As Double) As Double

'this function computes sea pressure in Pa from absolute pressure in Pa

psea_bar_from_psea_pa = psea_pa * 0.00001

End Function

'=====================================================
Function sal78_from_cnd(ByVal cnd As Double, _
                        ByVal t68_c As Double, _
                        ByVal p_dbar As Double) As Double
'=====================================================

'THE CONDUCTIVITY RATIO (CND)=1.0000000 FOR SALINITY=35 PSS-78
'TEMPERATURE=15.0 DEG. CELSIUS AND ATMOSPHERIC PRESSURE.
'FUNCTION TO CONVERT CONDUCTIVITY RATIO TO SALINITY

'REFERENCES: ALSO LOCATED IN UNESCO REPORT NO. 37 1981
'PRACTICAL SALINITY SCALE 1978: E.L. LEWIS IEEE OCEAN ENG. JAN. 1980

'----------------------------------------------------------
'UNITS:
'  PRESSURE      p_dbar   DECIBARS
'  TEMPERATURE   t68_c    DEG CELSIUS IPTS-68
'  CONDUCTIVITY  CND      RATIO
'  SALINITY      SAL78    PSS-78
'----------------------------------------------------------
'CHECKVALUES:
'  SAL78=40.00000 for CND=1.8880911555787682, T=40 DEG C, P=10000 DECIBARS
'----------------------------------------------------------
'SAL78 RATIO: RETURNS ZERO FOR CONDUCTIVITY RATIO: < 0.0005
'----------------------------------------------------------
'ORIGINAL FORTRAN CODE IS FOUND IN:
'  UNESCO TECHNICAL PAPERS IN MARINE SCIENCE 44 (1983) -
'  'ALGORITHMS FOR COMPUTATION OF FUNDAMENTAL PROPERTIES OF SEAWATER'
'----------------------------------------------------------
'TRANSLATED TO OBJECT PASCAL BY:
'  DR. JAN SCHULZ, 19 MAY 2008, WWW.CODE10.INFO
'----------------------------------------------------------
'TRANSLATED BACK TO FORTRAN BY:
'  D. WRIGHT, BIO, 23 SEPT., 2009
'----------------------------------------------------------

'real*8 sal78_from_cnd, cnd, t68_c, p_dbar
Dim dt#, res#, RT#

sal78_from_cnd = 0#

' ZERO SALINITY/CONDUCTIVITY TRAP
If (cnd <= 0.0005) Then Exit Function

dt = t68_c - 15#

' CONVERT CONDUCTIVITY TO SALINITY
res = cnd
RT = res / (rt35(t68_c) * (1# + c(p_dbar) / (b(t68_c) + a(t68_c) * res)))
RT = Sqr(Abs(RT))
sal78_from_cnd = sal(RT, dt)

End Function

'=====================================================
Function cnd_from_sal78(ByVal sal78 As Double, _
                        ByVal t68_c As Double, _
                        ByVal p_dbar As Double) As Double
'=====================================================

'THE CONDUCTIVITY RATIO (CND)=1.0000000 FOR SALINITY=35 PSS-78
'TEMPERATURE=15.0 DEG. CELSIUS AND ATMOSPHERIC PRESSURE.
'Function to convert SALINITY TO CONDUCTIVITY RATIO

'REFERENCES: ALSO LOCATED IN UNESCO REPORT NO. 37 1981
'PRACTICAL SALINITY SCALE 1978: E.L. LEWIS IEEE OCEAN ENG. JAN. 1980

'----------------------------------------------------------
'UNITS:
'  PRESSURE      p_dbar      DECIBARS
'  TEMPERATURE   t68_c      DEG CELSIUS IPTS-68
'  CONDUCTIVITY  CND    RATIO
'  SALINITY      SAL78  PSS-78
'----------------------------------------------------------
'CHECKVALUES:
'  CND=1.8880911555787682 for SAL78=40.0000, T=40 DEG C, P=10000 DECIBARS
'----------------------------------------------------------
'CND: RETURNS ZERO FOR SALINITY: < 0.02
'----------------------------------------------------------
'ORIGINAL FORTRAN CODE IS FOUND IN:
'  UNESCO TECHNICAL PAPERS IN MARINE SCIENCE 44 (1983) -
'  'ALGORITHMS FOR COMPUTATION OF FUNDAMENTAL PROPERTIES OF SEAWATER'
'----------------------------------------------------------
'TRANSLATED TO OBJECT PASCAL BY:
'  DR. JAN SCHULZ, 19 MAY 2008, WWW.CODE10.INFO
'----------------------------------------------------------
'TRANSLATED BACK TO FORTRAN BY:
'  D. WRIGHT, BIO, 23 SEPT., 2009
'----------------------------------------------------------

Dim n&
Dim cnd#
Dim dt#, RT#, si#, dels#, rtt#, at#, bt#, cp#, res#

cnd = 0#

'ZERO SALINITY/CONDUCTIVITY TRAP
If (sal78 <= 0.2) Then Exit Function

dt = t68_c - 15#

'INVERT SALINITY TO CONDUCTIVITY BY THE
'NEWTON-RAPHSON ITERATIVE METHOD

'FIRST APPROXIMATION
RT = Sqr(sal78 / 35#)
si = sal(RT, dt)

'ITERATION LOOP BEGINS HERE WITH A MAXIMUM OF 20 CYCLES
For n = 1 To 20
   RT = RT + (sal78 - si) / dsal(RT, dt)
   si = sal(RT, dt)
   dels = Abs(si - sal78)
   If (n > 1 And dels < 1E-20) Then Exit For
Next n


'COMPUTE CONDUCTIVITY RATIO
rtt = rt35(t68_c) * RT * RT
at = a(t68_c)
bt = b(t68_c)
cp = c(p_dbar)
cp = rtt * (cp + bt)
bt = bt - rtt * at

'SOLVE QUADRATIC EQUATION FOR R: R=RT35*RT*(1+C/AR+B)
res = Sqr(Abs(bt * bt + 4# * at * cp)) - bt

'RETURN CONDUCTIVITY
cnd_from_sal78 = 0.5 * res / at

End Function

'=====================================================
Private Function sal(ByVal xr As Double, _
                     ByVal xt As Double) As Double
'=====================================================

'PRACTICAL SALINITY SCALE 1978 DEFINITION WITH TEMPERATURE
'CORRECTION;XT :=T-15.0; XR:=SQRT(RT)

'real*8 sal, xr, xt

sal = ((((2.7081 * xr - 7.0261) * xr + 14.0941) * xr + 25.3851) * xr _
    - 0.1692) * xr + 0.008 _
    + (xt / (1# + 0.0162 * xt)) * (((((-0.0144 * xr _
    + 0.0636) * xr - 0.0375) * xr - 0.0066) * xr _
    - 0.0056) * xr + 0.0005)

End Function

'=====================================================
Private Function dsal(ByVal xr As Double, _
                      ByVal xt As Double) As Double
'=====================================================

'FUNCTION FOR DERIVATIVE OF SAL(XR,XT) WITH XR

'real*8 dsal, xr, xt

dsal = ((((13.5405 * xr - 28.1044) * xr + 42.2823) * xr + 50.7702) _
     * xr - 0.1692) + (xt / (1# + 0.0162 * xt)) * ((((-0.072 _
     * xr + 0.2544) * xr - 0.1125) * xr - 0.0132) * xr - 0.0056)

End Function

'=====================================================
Private Function rt35(ByVal xt As Double) As Double
'=====================================================

'FUNCTION RT35: C(35,T,0)/C(35,15,0) VARIATION WITH TEMPERATURE

'real*8 rt35, xt

rt35 = (((0.0000000010031 * xt - 0.00000069698) * xt + 0.0001104259) * xt _
     + 0.0200564) * xt + 0.6766097

End Function

'=====================================================
Private Function c(ByVal xp As Double) As Double
'=====================================================

'C(XP) POLYNOMIAL CORRESPONDS TO A1-A3 CONSTANTS: LEWIS 1980

'real*8 c, xp

c = ((3.989E-15 * xp - 0.000000000637) * xp + 0.0000207) * xp

End Function

'=====================================================
Private Function b(ByVal xt As Double) As Double
'=====================================================

'real*8 b, xt

b = (0.0004464 * xt + 0.03426) * xt + 1#

End Function

'=====================================================
Private Function a(ByVal xt As Double) As Double
'=====================================================

'A(XT) POLYNOMIAL CORRESPONDS TO B3 AND B4 CONSTANTS: LEWIS 1980

'integer n, iter
'real*8 a, xt
'real*8 dt, res, rt, si, dels, rtt, at, bt, cp

a = -0.003107 * xt + 0.4215

End Function

'=========================================================================
Function sw_dpth(ByVal p As Double, _
                 Optional ByVal lat As Double = 45#) As Double
'=========================================================================

' SW_DPTH    DEPTH FROM PRESSURE
'=========================================================================
' SW_DPTH   IS A MODIFICATION OF THE MATLAB ROUTINE WRITTEN BY PHIL MORGAN
'           OF CSIRO FOR THE SEAWATER LIBRARY DEVELOPED BY HIM AROUND 1992.
'
' USAGE:  DPTH = SW_DPTH(P,LAT)
'
' DESCRIPTION:
'    CALCULATES DEPTH IN METRES FROM PRESSURE IN DBARS.
'    BASED ON EQN 25, P26.  UNESCO 1983.
'
' INPUT:
'   P   = PRESSURE    [DB]
'   LAT = LATITUDE IN DECIMAL DEGRESS NORTH [-90..+90]
'
' OUTPUT:
'  DPTH = DEPTH [METRES]
'
' DISCLAIMER:
'   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND.
'
' REFERENCES:
'    UNESCO 1983. ALGORITHMS FOR COMPUTATION OF FUNDAMENTAL PROPERTIES OF
'    SEAWATER, 1983. _UNESCO TECH. PAP. IN MAR. SCI._, NO. 44, 53 PP.
'=========================================================================

'real*8 p, sw_dpth
Dim deg2rad#, c1#, c2#, c3#, c4#, gam_dash#, x#
'real*8, optional :: lat

'If (Not present(lat)) Then lat = 45#

sw_dpth = ErrorReturn

' CHECK INPUTS
If (p < 0# Or Abs(lat) > 90#) Then Exit Function

' The following commented code is the original
' It is not an exact inverse of sw_press
'deg2rad = 3.1415926535 / 180#
'c1 = 9.72659
'c2 = -0.000022512
'c3 = 0.0000000002279
'c4 = -1.82E-15
'gam_dash = 0.000002184
'x   = sin(lat*deg2rad)^2  ' CONVERT TO RADIANS
'sw_dpth = (((c4 * P + c3) * P + c2) * P + c1) * P
'sw_dpth = sw_dpth / (9.780318 * (1# + (0.0052788 + 0.0000236 * x) * x) + gam_dash * 0.5 * P)

' The following code is simply the analytical inverse of swpres
deg2rad = 3.1415926535 / 180#
x = Sin(Abs(lat) * deg2rad)      ' convert to radians
c1 = 0.00592 + x ^ 2 * 0.00525
sw_dpth = ((1# - c1) ^ 2 - (1# - c1 - 0.00000442 * p) ^ 2) / 0.00000884

End Function

'===========================================================================
Function sw_pres(ByVal depth As Double, _
                 Optional ByVal lat As Double = 45#) As Double
'===========================================================================

' SW_PRES    PRESSURE FROM DEPTH
'===========================================================================
' SW_PRES   IS A MODIFICATION OF THE MATLAB ROUTINE WRITTEN BY PHIL MORGAN
'           OF CSIRO FOR THE SEAWATER LIBRARY DEVELOPED BY HIM AROUND 1992.
'
' USAGE:  PRES = SW_PRES(DEPTH,LAT)
'
' DESCRIPTION:
'    CALCULATES PRESSURE IN DBARS FROM DEPTH IN METERS.
'
' INPUT:
'   DEPTH = DEPTH [METRES]
'   LAT   = LATITUDE IN DECIMAL DEGRESS NORTH [-90..+90]
'
' OUTPUT:
'  PRES   = PRESSURE    [DB]
'
' DISCLAIMER:
'   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND.
'
' REFERENCES:
'    SAUNDERS, P.M. 1981
'    "PRACTICAL CONVERSION OF PRESSURE TO DEPTH"
'    JOURNAL OF PHYSICAL OCEANOGRAPHY, 11, 573-574
'
' CHECK VALUE:
'    P=7500.00 DB for LAT=30 DEG, DEPTH=7321.45 METERS
'=========================================================================

'real*8 depth, sw_pres
Dim deg2rad#, c1#, x#
'real*8, optional :: lat

'If (Not present(lat)) Then lat = 45#

sw_pres = ErrorReturn

' CHECK INPUTS
If (depth < 0# Or Abs(lat) > 90#) Then Exit Function

deg2rad = 3.1415926535 / 180#
x = Sin(Abs(lat) * deg2rad)      ' convert to radians
c1 = 0.00592 + x ^ 2 * 0.00525
sw_pres = ((1# - c1) - Sqr((1# - c1) ^ 2 - 0.00000884 * depth)) / 0.00000442

End Function
