Attribute VB_Name = "Convert_0_Mdl"
Option Explicit

'#########################################################################

'This module requires the library module
'     Constants_0_Mdl, file Constants_0.bas

'#########################################################################

'=========================================================================
'This module implements conversion formulas between different scales and units

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98

Private Const mw = Molar_mass_H2O_si       'molar mass of H2O in kg/mol
Private Const ma = Molar_mass_air_L2000    'molar mass of air in kg/mol used by Lemmon et al. 2000

Private Const Version = "01 Jun 2010"

'==========================================================================
Public Function air_molar_mass_si(ByVal a_si As Double) As Double

'returns the molar mass of humid air in kg mol-1
'as a function of the air mass fraction, a_si, in kg/kg

'check value:
'air_molar_mass_si(0.5) = 2.22122197773792E-02  v. 1.0
'air_molar_mass_si(0.5) = 2.22142374908826E-02  v. 1.1

air_molar_mass_si = ErrorReturn

If a_si < 0 Or a_si > 1# Then Exit Function

air_molar_mass_si = 1# / ((1# - a_si) / mw + a_si / ma)

End Function

'==========================================================================
Public Function air_molfraction_vap_si(ByVal a_si As Double) As Double

'returns the mole fraction of vapour in humid air in mol mol-1
'as a function of the air mass fraction, a_si, in kg/kg

'check value:
'air_molfraction_vap_si(0.5) = 0.616483190185658   v. 1.0
'air_molfraction_vap_si(0.5) = 0.616539190282449   v. 1.1

air_molfraction_vap_si = ErrorReturn

If a_si < 0 Or a_si > 1# Then Exit Function

air_molfraction_vap_si = (1# - a_si) / (1# - a_si * (1# - mw / ma))

End Function
                             
'==========================================================================
Public Function air_molfraction_air_si(ByVal a_si As Double) As Double

'returns the mole fraction of air in humid air in mol mol-1
'as a function of the air mass fraction, a_si, in kg/kg

'check value:
'air_molfraction_air_si(0.5) = 0.383516809814342  v. 1.0
'air_molfraction_air_si(0.5) = 0.383460809717551  v. 1.1

air_molfraction_air_si = ErrorReturn

If a_si < 0 Or a_si > 1# Then Exit Function

air_molfraction_air_si = (a_si * mw / ma) / (1# - a_si * (1# - mw / ma))

End Function

'==========================================================================
Public Function air_massfraction_air_si(ByVal x_si As Double) As Double

'returns the mass fraction a_si in kg/kg of air in humid air
'as a function of the mole fraction x_si in mol/mol of air in humid air

'check value:
'air_massfraction_air_si(0.5) = 0.616483190185658  v. 1.0
'air_massfraction_air_si(0.5) = 0.616539190282449  v. 1.1

air_massfraction_air_si = ErrorReturn

If x_si < 0 Or x_si > 1# Then Exit Function

air_massfraction_air_si = x_si / (1# - (1# - x_si) * (1# - mw / ma))

End Function

'==========================================================================
Public Function air_massfraction_vap_si(ByVal x_si As Double) As Double

'returns the mass fraction of vapour in humid air in kg/kg
'as a function of the mole fraction of air, x_si, in mol mol-1

'check value:
'air_massfraction_vap_si(0.5) = 0.383516809814342  v. 1.0
'air_massfraction_vap_si(0.5) = 0.383460809717551  v. 1.1

air_massfraction_vap_si = ErrorReturn

If x_si < 0 Or x_si > 1# Then Exit Function

air_massfraction_vap_si = (1# - x_si) / (1# - x_si * (1# - ma / mw))

End Function

'==========================================================================
' THIS IS THE GSW ABSOLUTE SALINITY ALGORITHM TRANSLATED FROM FORTRAN TO VB
'==========================================================================
'
' AS DEFINED IN
'
' MCDOUGALL, T.J., JACKETT, D.R. AND MILLERO, F.J., 2009:
' AN ALGORITHM FOR ESTIMATING ABSOLUTE SALINITY IN THE GLOBAL OCEAN,
' Ocean Sci. Discuss., 6, 215-242, 2009
'
'
' DAVID JACKETT
' JANUARY 2009
' adapted by D.G.Wright and translated to VB by R. Feistel

'=====================================================
Public Function asal_from_psal(ByVal sp As Double, _
                               ByVal longs0 As Double, _
                               ByVal lats0 As Double, _
                               ByVal p_si As Double) As Double
'=====================================================

On Error GoTo Errortrap

' CONVERT PRACTICAL SALINITY TO ABSOLUTE SALINITY
'
' SP                  : PRACTICAL SALINITY                 [PSU]
' P_SI                : ABSOLUTE PRESSURE                  [PA]
' LONGS0              : LONGITUDE                          [DEG E]
' LATS0               : LATITUDE                           [DEG N]
'
' RESULT              : ABSOLUTE SALINITY                  [KG/KG]

'CHECK VALUE:
'asal_from_psal(35.5276443777339, 201, -21, 101325 + 10230000#) = 3.57000000000001E-02
'asal_from_psal(35, 180, 40, 101325 + 20000000#) = 3.51888478029404E-02 'north Pacific silicate
'asal_from_psal(8, 20, 57, 101325) = 8.13338057142857E-03               'central Baltic Sea

'if the file "gsw_data.dat" is missing, SA = SR is computed in the ocean:
'asal_from_psal(35.5276443777339, 201, -21, 101325 + 10230000#) = 3.56951724471082E-02
'asal_from_psal(35, 180, 40, 101325 + 20000000#) = 0.03516504            'north Pacific (without silicate)
'asal_from_psal(8, 20, 57, 101325) = 8.13338057142857E-03                'central Baltic Sea

Dim flag&
Dim sa#, P0#

P0 = (p_si - 101325#) / 10000#
sa = (35.16504 / 35#) * sp + gsw_delta_sa(longs0, lats0, P0)
flag = 1
asal_from_psal = 0.001 * gsw_adjust_baltic(sa, sp, longs0, lats0, flag)

Exit Function

Errortrap:
asal_from_psal = ErrorReturn

End Function

'=====================================================
Public Function psal_from_asal(ByVal sa_si As Double, _
                               ByVal longs0 As Double, _
                               ByVal lats0 As Double, _
                               ByVal p_si As Double) As Double
'=====================================================

On Error GoTo Errortrap

' CONVERT ABSOLUTE SALINITY TO PRACTICAL SALINITY
'
' SA_SI               : ABSOLUTE SALINITY                  [KG/KG]
' P_SI                : ABSOLUTE PRESSURE                  [PA]
' LONGS0              : LONGITUDE                          [DEG E]
' LATS0               : LATITUDE                           [DEG N]
'
' PSAL_FROM_ASAL  : PRACTICAL SALINITY                     [PSU]


'CHECK value
'psal_from_asal(0.0357, 201, -21, 101325 + 10230000#) = 35.5276443777339

'if the file "gsw_data.dat" is missing, SA = SR is computed in the ocean:
'psal_from_asal(0.0357, 201, -21, 101325 + 10230000#) = 35.532449273483

Dim flag&
Dim sa#, P0#, psal#

P0 = (p_si - 101325#) / 10000#
sa = 1000# * sa_si
psal = (35# / 35.16504) * (sa - gsw_delta_sa(longs0, lats0, P0))
flag = 2
psal_from_asal = gsw_adjust_baltic(sa, psal, longs0, lats0, flag)

Exit Function

Errortrap:
psal_from_asal = ErrorReturn

End Function

'=====================================================
Private Function gsw_delta_sa(ByVal lon0 As Double, _
                              ByVal lat0 As Double, _
                              ByVal P0 As Double) As Double
'=====================================================

' CALCULATE THE ABSOLUTE SALINITY ANOMALY
'
' P0                  : ABSOLUTE pressure                  [DBAR]
' lon0                : LONGITUDE                          [DEG E]
' lat0                : LATITUDE                           [DEG N]
'
' RESULT              : ABSOLUTE SALINITY ANOMALY          [G/KG]

'integer nx, ny, nz
'parameter(nx=91,ny=44,nz=45)

Const nx = 91, ny = 44, nz = 45

Dim indx0&, indy&, indz&, i&, j&, lint&
Dim k&, deli&(4), delj&(4), nmean&, d#
Dim p0_original#, sa_upper#, sa_lower#
Dim dlongs#, dlats#
Dim r1#, s1#, t1#, dsa_mean#, dsa#(4), ndepth_max#

Static icalled&, longs#(nx), lats#(ny), p#(nz), del_sa#(nz, ny, nx), ndepth#(ny, nx)

gsw_delta_sa = 0

If lat0 < -82 Or lat0 > 90 Then Exit Function

lint = Int(lon0)
lon0 = lon0 - lint
lint = ((lint Mod 360) + 360) Mod 360  'results in 0 <= lint < 360
lon0 = lon0 + lint
If lon0 = 0 Then lon0 = 0.000000000000001

'data deli/0,1,1,0/, delj/0,0,1,1/
deli(2) = 1
deli(3) = 1
delj(3) = 1
delj(4) = 1

'data icalled/0/

If (icalled <> 1) Then
   icalled = 1
   If Read_gsw_data(nx, ny, nz, longs(), lats(), p(), ndepth(), del_sa()) = ErrorReturn Then
     gsw_delta_sa = 0
     icalled = 0
     Exit Function
   End If
'   open(10,file='gsw_data.dat',status='unknown')
'   read(10,*) (longs(i), i=1,nx)
'   read(10,*) (lats(i), i=1,ny)
'   read(10,*) (p(i), i=1,nz)
'   read(10,*) ((ndepth(j,i), j=1,ny), i=1,nx)
'   read(10,*) (((del_sa(k,j,i), k=1,nz), j=1,ny), i=1,nx)
'   Close (10)
End If

dlongs = longs(2) - longs(1)
dlats = lats(2) - lats(1)

'indx0 = floor(1 + (nx - 1) * (lon0 - longs(1)) / (longs(nx) - longs(1)))
indx0 = Int(1 + (nx - 1) * (lon0 - longs(1)) / (longs(nx) - longs(1)))
If (indx0 = nx) Then
   indx0 = nx - 1
End If

'indy = floor(1 + (ny - 1) * (lat0 - lats(1)) / (lats(ny) - lats(1)))
indy = Int(1 + (ny - 1) * (lat0 - lats(1)) / (lats(ny) - lats(1)))
If (indy = ny) Then
   indy = ny - 1
End If

ndepth_max = -1#
For k = 1 To 4
   If (ndepth(indy + delj(k), indx0 + deli(k)) > 0#) Then
      ndepth_max = max(ndepth_max, ndepth(indy + delj(k), indx0 + deli(k)))
   End If
Next k

If (ndepth_max = -1#) Then
   gsw_delta_sa = 0#   '/0.
   Exit Function
End If

p0_original = P0
If (P0 > p(Int(ndepth_max))) Then P0 = p(Int(ndepth_max))
Call indx(p, nz, P0, indz)
    
r1 = (lon0 - longs(indx0)) / (longs(indx0 + 1) - longs(indx0))
s1 = (lat0 - lats(indy)) / (lats(indy + 1) - lats(indy))
t1 = (P0 - p(indz)) / (p(indz + 1) - p(indz))

For k = 1 To 4
   dsa(k) = del_sa(indz, indy + delj(k), indx0 + deli(k))
Next k

If (260# <= lon0 And lon0 <= 291.999 And 3.4 <= lat0 And lat0 <= 19.55) Then
   dsa_add_barrier dsa(), lon0, lat0, longs(indx0), lats(indy), dlongs, dlats
ElseIf (Abs(sum(dsa)) >= 10000000000#) Then
   dsa_add_mean dsa(), lon0, lat0
End If

sa_upper = (1# - s1) * (dsa(1) + r1 * (dsa(2) - dsa(1))) + s1 * (dsa(4) + r1 * (dsa(3) - dsa(4)))

For k = 1 To 4
   dsa(k) = del_sa(indz + 1, indy + delj(k), indx0 + deli(k))
Next k

If (260# <= lon0 And lon0 <= 291.999 And 3.4 <= lat0 And lat0 <= 19.55) Then
  dsa_add_barrier dsa(), lon0, lat0, longs(indx0), lats(indy), dlongs, dlats
ElseIf (Abs(sum(dsa)) >= 10000000000#) Then
  dsa_add_mean dsa(), lon0, lat0
End If

sa_lower = (1# - s1) * (dsa(1) + r1 * (dsa(2) - dsa(1))) + s1 * (dsa(4) + r1 * (dsa(3) - dsa(4)))
If (Abs(sa_lower) >= 10000000000#) Then sa_lower = sa_upper

d = sa_upper + t1 * (sa_lower - sa_upper)
gsw_delta_sa = d
If (Abs(d) >= 10000000000#) Then gsw_delta_sa = 0#
P0 = p0_original

End Function

'=====================================================
Private Function gsw_adjust_baltic(ByVal sa As Double, _
                                   ByVal sp As Double, _
                                   ByVal longs As Double, _
                                   ByVal lats As Double, _
                                   ByVal flag As Long) As Double
'=====================================================

' FOR THE BALTIC SEA, OVERWRITE ABSOLUTE SALINITY WITH A VALUE
' COMPUTED ANALYTICALLY FROM PRACTICAL SALINITY, OR VICE VERSA
'
' SA                  : ABSOLUTE SALINITY                  [G/KG]
' SP                  : PRACTICAL SALINITY                 [PSU]
' LONGS               : LONGITUDE                          [DEG E]
' LATS                : LATITUDE                           [DEG N]
' FLAG                : FLAG - 1 or 2
'
' GSW_ADJUST_BALTIC   : ABSOLUTE SALINITY                  [G/KG]
'                         WHEN FLAG = 1
'                     : PRACTICAL SALINITY                 [PSU]
'                         WHEN FLAG = 2

Dim n2&, n3&
Dim xb_left#(3), xb_right#(2), yb_left#(3), yb_right#(2), xx_left#, xx_right#

'data xb_left/12.6d0, 7.d0, 26.d0/, yb_left/50.d0, 59.d0, 69.d0/
xb_left(1) = 12.6
xb_left(2) = 7#
xb_left(3) = 26#
yb_left(1) = 50#
yb_left(2) = 59#
yb_left(3) = 69#

'data xb_right/45.d0, 26.d0/, yb_right/50.d0, 69.d0/
xb_right(1) = 45#
xb_right(2) = 26#
yb_right(1) = 50#
yb_right(2) = 69#

n2 = 2
n3 = 3

If (flag = 1) Then
   gsw_adjust_baltic = sa
ElseIf (flag = 2) Then
   gsw_adjust_baltic = sp
End If

If (xb_left(2) < longs And longs < xb_right(1) And yb_left(1) < lats And lats < yb_left(3)) Then
   xx_left = xinterp1(yb_left(), xb_left(), n3, lats)
   xx_right = xinterp1(yb_right(), xb_right(), n2, lats)
   If (xx_left <= longs And longs <= xx_right) Then
      If (flag = 1) Then
         gsw_adjust_baltic = (35.16504 / 35#) * sp + 0.124 * (1# - sp / 35#)
      ElseIf (flag = 2) Then
         gsw_adjust_baltic = (35# / 35.04104) * (sa - 0.124)
      End If
   End If
End If

End Function

'=====================================================
Private Sub dsa_add_barrier(ByRef dsa() As Double, _
                            ByVal longs0 As Double, _
                            ByVal lats0 As Double, _
                            ByVal longs As Double, _
                            ByVal lats As Double, _
                            ByVal dlongs As Double, _
                            ByVal dlats As Double)
'=====================================================

' ADD A BARRIER THROUGH CENTRAL AMERICA (PANAMA) AND THEN AVERAGE
' OVER THE APPROPRIATE SIDE OF THE BARRIER
'
' DSA                 : ABSOLUTE SALINITY ANOMALY                [G/KG]
' LONGS0              : LONGITUDES OF DATA                       [DEG E]
' LATS0               : LATITUDES OF DATA                        [DEG N]
' LONGS               : LONGITUDES OF REGULAR GRID               [DEG E]
' LATS                : LATITUDES OF REGULAR GRID                [DEG N]
' DLONGS              : LONGITUDE DIFFERENCE OF REGULAR GRID     [DEG LONGITUDE]
' DLATS               : LATITUDES DIFFERENCE OF REGULAR GRID     [DEG LATITUDE]
'
' RESULT              : ABSOLUTE SALINITY ANOMALY OF DATA        [G/KG]

'integer n4, n6
'parameter(n4=4, n6=6)
Const n4 = 4
Const n6 = 6

Dim k&, nmean&, above_line0&, above_line&(n4)
Dim longs_pan#(n6), lats_pan#(n6), R#, lats_line#
Dim dsa_mean#

For k = 1 To n6
 'data longs_pan/260.0000, 272.5900, 276.5000, 278.6500, 280.7300, 292.000/
  longs_pan(k) = Choose(k, 260#, 272.59, 276.5, 278.65, 280.73, 292#)
 'data  lats_pan/ 19.5500,  13.9700,   9.6000,   8.1000,   9.3300,   3.400/
  lats_pan(k) = Choose(k, 19.55, 13.97, 9.6, 8.1, 9.33, 3.4)
Next k

Call indx(longs_pan, n6, longs0, k)                         '   the long0/lat0 point
R = (longs0 - longs_pan(k)) / (longs_pan(k + 1) - longs_pan(k))
lats_line = lats_pan(k) + R * (lats_pan(k + 1) - lats_pan(k))
If (lats_line <= lats0) Then
   above_line0 = 1
Else
   above_line0 = 0
End If

''print *, 'above_line = ', above_line0

Call indx(longs_pan, n6, longs, k)                                  '  the 1 and 4 long/lat points
R = (longs - longs_pan(k)) / (longs_pan(k + 1) - longs_pan(k))
lats_line = lats_pan(k) + R * (lats_pan(k + 1) - lats_pan(k))
If (lats_line <= lats) Then
   above_line(1) = 1
Else
   above_line(1) = 0
End If
If (lats_line <= lats + dlats) Then
   above_line(4) = 1
Else
   above_line(4) = 0
End If

Call indx(longs_pan, n6, longs + dlongs, k)                         '  the 2 and 3 long/lat points
R = (longs + dlongs - longs_pan(k)) / (longs_pan(k + 1) - longs_pan(k))
lats_line = lats_pan(k) + R * (lats_pan(k + 1) - lats_pan(k))
If (lats_line <= lats) Then
   above_line(2) = 1
Else
   above_line(2) = 0
End If
If (lats_line <= lats + dlats) Then
   above_line(3) = 1
Else
   above_line(3) = 0
End If

nmean = 0
dsa_mean = 0#
For k = 1 To n4
   If ((Abs(dsa(k)) <= 100#) And above_line0 = above_line(k)) Then
      nmean = nmean + 1
      dsa_mean = dsa_mean + dsa(k)
   End If
Next k

If (nmean = 0) Then
   dsa_mean = ErrorReturn
Else
   dsa_mean = dsa_mean / nmean
End If

For k = 1 To n4
   If ((Abs(dsa(k)) >= 10000000000#) Or above_line0 <> above_line(k)) Then
      dsa(k) = dsa_mean
   End If
Next k

End Sub

'=====================================================
Private Sub dsa_add_mean(ByRef dsa() As Double, _
                         ByVal longs0 As Double, _
                         ByVal lats0 As Double)
'=====================================================

' REPLACE NANS WITH NAMEAN
'
' DSA           : ABSOLUTE SALINITY ANOMALY                [G/KG]
'                 OF THE 4 ADJACENT NEIGHBOURS
'
' RESULT        : NANMEAN OF THE 4 ADJACENT NEIGHBOURS     [G/KG]


Dim k&, nmean&
Dim dsa_mean#

nmean = 0
dsa_mean = 0#

For k = 1 To 4
   If (Abs(dsa(k)) <= 100#) Then
      nmean = nmean + 1
      dsa_mean = dsa_mean + dsa(k)
   End If
Next k

If (nmean = 0) Then
   dsa_mean = ErrorReturn
Else
   dsa_mean = dsa_mean / nmean
End If

For k = 1 To 4
   If (Abs(dsa(k)) >= 100#) Then
      dsa(k) = dsa_mean
   End If
Next k

End Sub

'=====================================================
Private Sub indx(ByRef x() As Double, _
                 ByVal n As Long, _
                 ByVal z As Double, _
                 ByRef k As Long)
'=====================================================

'DESCRIPTION: FIND THE INDEX OF A REAL NUMBER IN A
'   MONOTONICALLY INCREASING REAL ARRAY
'
'INPUT:           X   ARRAY OF INCREASING VALUES
'                 N   LENGTH OF ARRAY
'           Z   REAL NUMBER
'
'OUTPUT:    K   INDEX K - IF X(K) <= Z < X(K+1), OR
'       N-1         - IF Z = X(N)
'
'CREATED:   JUNE 1993
'

Dim ku&, kl&, km&

If (x(1) < z And z < x(n)) Then
   kl = 1
   ku = n
   Do While (ku - kl > 1)
      km = (ku + kl) / 2
      If (z > x(km)) Then
         kl = km
      Else
         ku = km
      End If
   Loop
   k = kl
   If (z = x(k + 1)) Then k = k + 1
Else
   If (z = x(1)) Then
      k = 1
   ElseIf (z = x(n)) Then
      k = n - 1
   Else
     MsgBox "ERROR in indx.f : out of range" + Chr(13) + _
     "z=" + Str(z) + ", n=" + Str(n) + ", x(1)=" + Str(x(1)) + ", x(n)=" + Str(x(n))
      'print *, z,n,x
      'pause
   End If
End If

End Sub

'=====================================================
Private Function xinterp1(ByRef x() As Double, _
                          ByRef y() As Double, _
                          ByVal n As Long, _
                          ByVal x0 As Double) As Double
'=====================================================

' LINEARLY INTERPOLATE A REAL ARRAY
'
' X                   : X ARRAY (MONOTONIC)
' Y                   : Y ARRAY
' N                   : LENGTH OF X AND Y ARRAYS
' X0                  : X VALUE TO BE INTERPOLATED
'
' RESULT              : INTERPOLATED VALUE

Dim k&
Dim R#

Call indx(x(), n, x0, k)
R = (x0 - x(k)) / (x(k + 1) - x(k))
xinterp1 = y(k) + R * (y(k + 1) - y(k))

End Function

'=====================================================
Private Function max(ByVal a As Double, ByVal b As Double) As Double
'=====================================================

max = IIf(a > b, a, b)

End Function

'=====================================================
Private Function sum(ByRef a() As Double) As Double
'=====================================================

Dim S As Double, i As Long

For i = LBound(a) To UBound(a)
  S = S + a(i)
Next i
sum = S

End Function

'=====================================================
Private Function Read_gsw_data(ByVal nx As Long, _
                               ByVal ny As Long, _
                               ByVal nz As Long, _
                               ByRef lon() As Double, _
                               ByRef lat() As Double, _
                               ByRef pres() As Double, _
                               ByRef ndpt() As Double, _
                               ByRef dels() As Double) As Double
'=====================================================

Dim f$, h%, i&, j&, k&

f = App.path
If Right(f, 1) <> "\" Then f = f + "\"
f = f + "gsw_data.dat"

'check whether file exists:
If Dir(f) = "" Then
  Read_gsw_data = ErrorReturn
  Exit Function
End If

h = FreeFile
Open f For Binary Access Read As #h

For i = 1 To nx
  lon(i) = Read_next(h)
Next i

For i = 1 To ny
  lat(i) = Read_next(h)
Next i

For i = 1 To nz
  pres(i) = Read_next(h)
Next i

For i = 1 To nx
  For j = 1 To ny
    ndpt(j, i) = Read_next(h)
  Next j
Next i

For i = 1 To nx
  For j = 1 To ny
    For k = 1 To nz
      dels(k, j, i) = Read_next(h)
    Next k
  Next j
Next i

Close #h

End Function

'=====================================================
Private Function Read_next(ByVal h As Integer) As Double
'=====================================================

'File contains chr(10) as separator -> read as binary the bytes till the next linefeed

Dim addr&, bytes$, L&

addr = Seek(h) 'current read pointer

'read l chars
L = 64
If addr - 1 + L > LOF(h) Then 'beyond end of file?
  L = LOF(h) - addr + 1
End If
If L <= 0 Then Exit Function 'passed end of file

bytes = Space(L)

Get #h, , bytes   'read l chars

L = InStr(bytes, Chr(10))  'find the linefeed in them
If L > 0 Then
  bytes = Left(bytes, L - 1)  'cut off the next line
  addr = addr + L
Else
  addr = addr + Len(bytes) 'no linefeed, take all bytes (this is the final line)
End If

Seek #h, addr  'update read pointer to the line's end

Read_next = Val(bytes)

End Function


