Attribute VB_Name = "Complex_0_Mdl"
Option Explicit

'========================================================
'Implementation of elementary complex arithmetics in VB6
' R.Feistel, 5 Nov 2003
'
' This module requires the module
'     Constants_0_Mdl, file Constants_0.bas
'========================================================

'Private Const ErrorReturn = 9.99999999E+98

'Private Const PI = 3.14159265358979

'Define Complex Number Type
Public Type CplxType
  Re  As Double
  Im  As Double
End Type

Private Const Version = "03 Nov 2009"

'========================================================
Private Function Acs(ByVal x As Double) As Double
'Returns real function arccos(x)
Dim S As Double

S = Asn(x)
If S = ErrorReturn Then
  Acs = ErrorReturn
Else
  Acs = 0.5 * PI - S
End If

End Function

'========================================================
Private Function Asn(ByVal x As Double) As Double
'Returns real function arcsin(x) between -PI/2 and +PI/2
Select Case Abs(x)

  'For invalid arguments > 1, accept a certain rounding tolerance
  Case Is > 1.00000001: Asn = ErrorReturn
  
  'compute regular values from intrinsic arctan
  Case Is < 1: Asn = Atn(x / Sqr(1# - x * x))

  'values -PI/2 and +PI/2
  Case Else: Asn = 0.5 * PI * Sgn(x)

End Select
End Function

'========================================================
Public Function Cplx_Ampl(a As CplxType) As Double
'returns Amplitude A = |a| of complex number a
Cplx_Ampl = Sqr(Cplx_Mult(a, Cplx_Conj(a)).Re)
End Function

'========================================================
Public Function Cplx_Arg(a As CplxType) As Double
'returns argument = phase angle arg(a) of complex number a,
'-PI < arg <= +PI
Select Case a.Re
  Case 0:      Cplx_Arg = Sgn(a.Im) * PI * 0.5
  Case Is > 0: Cplx_Arg = Atn(a.Im / a.Re)
  Case Else:   Cplx_Arg = Atn(a.Im / a.Re) + IIf(a.Im < 0, -PI, PI)
End Select
End Function

'========================================================
Public Function Cplx_Conj(a As CplxType) As CplxType
'returns complex conjugate Re(a) - i*Im(a) of complex number a
Cplx_Conj.Re = a.Re
Cplx_Conj.Im = -a.Im
End Function

'========================================================
Public Function Cplx_Inv(a As CplxType) As CplxType
'returns complex reciprocal 1/a
Cplx_Inv = Cplx_Div(Cplx_Num(1, 0), a)
End Function

'========================================================
Public Function Cplx_Div(a As CplxType, b As CplxType) As CplxType
'returns complex fraction a / b
Dim ab As CplxType, bb As Double
bb = Cplx_Mult(b, Cplx_Conj(b)).Re
If bb = 0 Then
  MsgBox "Complex Division by Zero"
  Cplx_Div.Re = 0
  Cplx_Div.Im = 0
  Exit Function
End If
ab = Cplx_Mult(a, Cplx_Conj(b))
Cplx_Div.Re = ab.Re / bb
Cplx_Div.Im = ab.Im / bb
End Function

'========================================================
Public Function Cplx_Log(a As CplxType) As CplxType
'returns complex logarithm ln(a), principal value -PI < Im(ln(a)) <= PI
Dim R#
R# = Cplx_Ampl(a)
If R > 0 Then
  Cplx_Log.Re = Log(R)
  Cplx_Log.Im = Cplx_Arg(a)
Else
  MsgBox "Invalid Argument of Complex Log"
  Cplx_Log.Re = 0
  Cplx_Log.Im = 0
End If
End Function

'========================================================
Public Function Cplx_Exp(a As CplxType) As CplxType
'returns complex exponential exp(a)
Dim amp#
amp# = Exp(a.Re)
Cplx_Exp.Re = amp# * Cos(a.Im)
Cplx_Exp.Im = amp# * Sin(a.Im)
End Function

'========================================================
Public Function Cplx_Mult(a As CplxType, b As CplxType) As CplxType
'returns complex product a*b
Cplx_Mult.Re = a.Re * b.Re - a.Im * b.Im
Cplx_Mult.Im = a.Re * b.Im + a.Im * b.Re
End Function

'========================================================
Public Function Cplx_Num(ByVal x As Double, Optional ByVal y As Double = 0) As CplxType
'assigns real x (and optionally imaginary y) part to a complex number x + iy
Cplx_Num.Re = x
Cplx_Num.Im = y
End Function

'========================================================
Public Function Cplx_Factor(a As CplxType, ByVal b As Double) As CplxType
'returns product a*b of complex a with real b
Cplx_Factor.Re = a.Re * b
Cplx_Factor.Im = a.Im * b
End Function

'========================================================
Public Function Cplx_Power(a As CplxType, n As Integer) As CplxType
'returns complex integer power a^n by multiplication
Dim an As CplxType, i As Long, p As CplxType
If n = 0 Then
  Cplx_Power = Cplx_Num(1)
  Exit Function
End If

If n > 0 Then
  an = a
Else
  an = Cplx_Inv(a)
End If

p = an
For i = 1 To Abs(n) - 1
  p = Cplx_Mult(p, an)
Next i
Cplx_Power = p

End Function

'========================================================
Public Function Cplx_Sum(a As CplxType, b As CplxType) As CplxType
'returns the complex sum a + b
Cplx_Sum.Re = a.Re + b.Re
Cplx_Sum.Im = a.Im + b.Im
End Function

'========================================================
Public Function Cplx_Diff(a As CplxType, b As CplxType) As CplxType
'returns complex difference a - b
Cplx_Diff.Re = a.Re - b.Re
Cplx_Diff.Im = a.Im - b.Im
End Function

'========================================================
Public Function Cplx_Root(z As CplxType, ByVal i As Integer, ByVal n As Integer) As CplxType
'returns the i-th root of n complex n-th roots of z = (x+iy)
Cplx_Root = Cplx_Rootxy(z.Re, z.Im, i, n)
End Function

'========================================================
Public Function Cplx_Rootxy(ByVal x As Double, _
                            ByVal y As Double, _
                            ByVal i As Integer, _
                            ByVal n As Integer) As CplxType
'returns the i-th root of n complex n-th roots of z = (x+iy)

Dim R#, cph#, sph#, phi#

If n < 1 Then
  Cplx_Rootxy.Re = ErrorReturn
  Cplx_Rootxy.Im = 0
  Exit Function
End If

R# = Sqr(x ^ 2 + y ^ 2)
If R = 0 Then
  Cplx_Rootxy.Re = 0
  Cplx_Rootxy.Im = 0
  Exit Function
End If

cph# = x / R
sph# = y / R
phi# = Acs(cph)
If phi = ErrorReturn Then
  Cplx_Rootxy.Re = ErrorReturn
  Cplx_Rootxy.Im = 0
  Exit Function
End If

If sph < 0 Then phi = -phi
R = R ^ (1 / n)
x = R * Cos((phi + i * 2 * PI) / n)
y = R * Sin((phi + i * 2 * PI) / n)
Cplx_Rootxy.Re = x
Cplx_Rootxy.Im = y
End Function


