Attribute VB_Name = "Air_3b_Mdl"
Option Explicit

'#########################################################################

'This module requires the library modules:
'     Constants_0_Mdl, file Constants_0.bas
'     Convert_0_Mdl,   file Convert_0.bas
'     air_1_Mdl,       file air_1.bas
'     air_2_Mdl,       file air_2.bas
'     air_3a_Mdl,      file air_3a.bas

'#########################################################################

'=========================================================================
'This module implements thermodynamic properties of humid air as functions
'of air fraction, temperature and pressure, computed numerically from
'the Gibbs function of humid air.

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009

'Feistel, R., Kretzschmar, H.-J., Span, R., Hagen, E., Wright, D.G., Herrmann, S.:
'Thermodynamic Properties of Sea Air.
'Ocean Science Discussion 6(2009)21932325.
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98

Private Const Version = "28 May 2010"

'=========================================================================
Public Function air_g_chempot_vap_si(ByVal a_si As Double, _
                                     ByVal t_si As Double, _
                                     ByVal p_si As Double) As Double

'this function computes the chemical potential of vapour in humid air
'from the mass fraction of air, temperature and pressure

'input:  a_si:  mass fraction of dry air in humid air, in kg/kg
'        t_si:  absolute temperature in K
'        p_si:  absolute pressure in Pa
'output: chemical potential in J/kg

'check value with default settings:
'air_g_chempot_vap_si(0.9,300,1E5) = 193678.583379401  v 1.0
'air_g_chempot_vap_si(0.9,300,1E5) = 193705.688575676  v 1.1

Dim g As Double, g_a As Double

air_g_chempot_vap_si = ErrorReturn

If a_si < 0 Or a_si > 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

'g
g = air_g_si(0, 0, 0, a_si, t_si, p_si)
If g = ErrorReturn Then Exit Function

'dg/dA
g_a = air_g_si(1, 0, 0, a_si, t_si, p_si)
If g = ErrorReturn Then Exit Function

air_g_chempot_vap_si = g - a_si * g_a 'g - A(dg/dA)

End Function

'=========================================================================
Public Function air_g_contraction_si(ByVal a_si As Double, _
                                     ByVal t_si As Double, _
                                     ByVal p_si As Double) As Double

'returns   beta = -(1/v) * (dv/dA)_T_P  contraction coefficient of humid air in kg/kg,
'          i.e. the relative density incease if some vapour is replaced by air
'a_si      air fraction in kg/kg,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in air_3a

'Check value with default settings: air_g_contraction_si(0.9, 300, 1E5) = 0.529966294385666  v. 1.0
'Check value with default settings: air_g_contraction_si(0.9, 300, 1E5) = 0.530280645259538  v. 1.1

Dim d As Double, g_ap As Double

air_g_contraction_si = ErrorReturn

If a_si < 0 Or a_si > 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = air_density_si(a_si, t_si, p_si)
If d = ErrorReturn Then Exit Function

g_ap = air_g_si(1, 0, 1, a_si, t_si, p_si)
If g_ap = ErrorReturn Then Exit Function

air_g_contraction_si = -d * g_ap

End Function

'=========================================================================
Public Function air_g_compressibilityfactor_si(ByVal a_si As Double, _
                                               ByVal t_si As Double, _
                                               ByVal p_si As Double) As Double

'returns   Z = P * v/(RAV * T)  compressibility factor of humid air,

'a_si      mass fraction of dry air in kg/kg,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in air_3a

'Check value with default settings:
'air_g_compressibilityfactor_si(0.9, 300, 1E5) = 0.99782665440423  v 1.0
'air_g_compressibilityfactor_si(0.9, 300, 1E5) = 0.997825670991282  v 1.1

Dim d As Double, rav As Double

air_g_compressibilityfactor_si = ErrorReturn

If a_si < 0 Or a_si > 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = air_density_si(a_si, t_si, p_si)
If d = ErrorReturn Then Exit Function
If d <= 0 Then Exit Function

rav = air_molar_mass_si(a_si)
If rav = ErrorReturn Then Exit Function
If rav <= 0 Then Exit Function
rav = Gas_constant_molar_L2000 / rav

air_g_compressibilityfactor_si = p_si / (d * rav * t_si)

End Function

'=========================================================================
Public Function air_g_cp_si(ByVal a_si As Double, _
                            ByVal t_si As Double, _
                            ByVal p_si As Double) As Double

'returns   cp = T * (deta/dT)_A_P  isobaric heat capacity of humid air in J/(kg K),

'a_si      mass fraction of dry air in kg/kg,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in air_3a

'Check value with default settings: air_g_cp_si(0.9, 300, 1E5) = 1246.47586190592  v. 1.0
'Check value with default settings: air_g_cp_si(0.9, 300, 1E5) = 1246.34991644392  v. 1.1

Dim d As Double

air_g_cp_si = ErrorReturn

If a_si < 0 Or a_si > 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = air_density_si(a_si, t_si, p_si)
If d = ErrorReturn Then Exit Function

air_g_cp_si = air_f_cp_si(a_si, t_si, d)

End Function

'=========================================================================
Public Function air_g_cv_si(ByVal a_si As Double, _
                            ByVal t_si As Double, _
                            ByVal p_si As Double) As Double

'returns   cv = T * (deta/dT)_A_D  isochoric heat capacity of humid air in J/(kg K),

'a_si      mass fraction of dry air in kg/kg,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in air_3a

'Check value with default settings: air_g_cv_si(0.9, 300, 1E5) = 920.676202550767  v. 1.0
'Check value with default settings: air_g_cv_si(0.9, 300, 1E5) = 920.60078101248  v. 1.1

Dim d As Double

air_g_cv_si = ErrorReturn

If a_si < 0 Or a_si > 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = air_density_si(a_si, t_si, p_si)
If d = ErrorReturn Then Exit Function

air_g_cv_si = air_f_cv_si(a_si, t_si, d)

End Function

'=========================================================================
Public Function air_g_enthalpy_si(ByVal a_si As Double, _
                                  ByVal t_si As Double, _
                                  ByVal p_si As Double) As Double

'returns   h = g - T*(dg/dT)_A_P  enthalpy of humid air in J/kg,

'a_si      mass fraction of dry air in kg/kg,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in air_3a

'Check value with default settings: air_g_enthalpy_si(0.9, 300, 1E5) = 277935.361122372  v. 1.0
'Check value with default settings: air_g_enthalpy_si(0.9, 300, 1E5) = 277928.954795303  v. 1.1

Dim d As Double

air_g_enthalpy_si = ErrorReturn

If a_si < 0 Or a_si > 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = air_density_si(a_si, t_si, p_si)
If d = ErrorReturn Then Exit Function

air_g_enthalpy_si = air_f_enthalpy_si(a_si, t_si, d)

End Function

'=========================================================================
Public Function air_g_entropy_si(ByVal a_si As Double, _
                                 ByVal t_si As Double, _
                                 ByVal p_si As Double) As Double

'returns   eta = - (dg/dT)_A_P  entropy of humid air in J/(kg K),

'a_si      mass fraction of dry air in kg/kg,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in air_3a

'Check value with default settings: air_g_entropy_si(0.9, 300, 1E5) = 911.203092171182  v. 1.0
'Check value with default settings: air_g_entropy_si(0.9, 300, 1E5) = 911.170080461387  v. 1.1

Dim d As Double

air_g_entropy_si = ErrorReturn

If a_si < 0 Or a_si > 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = air_density_si(a_si, t_si, p_si)
If d = ErrorReturn Then Exit Function

air_g_entropy_si = air_f_entropy_si(a_si, t_si, d)

End Function

'=========================================================================
Public Function air_g_expansion_si(ByVal a_si As Double, _
                                   ByVal t_si As Double, _
                                   ByVal p_si As Double) As Double

'returns   alpha = (1/v) * (dv/dT)_A_P  thermal expansion of humid air in 1/K,

'a_si      mass fraction of dry air in kg/kg,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in air_3a

'Check value with default settings: air_g_expansion_si(0.9, 300, 1E5) = 3.45696232116875E-03  v. 1.0
'Check value with default settings: air_g_expansion_si(0.9, 300, 1E5) = 3.45704654419841E-03  v. 1.1

Dim d As Double

air_g_expansion_si = ErrorReturn

If a_si < 0 Or a_si > 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = air_density_si(a_si, t_si, p_si)
If d = ErrorReturn Then Exit Function

air_g_expansion_si = air_f_expansion_si(a_si, t_si, d)

End Function

'=========================================================================
Public Function air_g_density_si(ByVal a_si As Double, _
                                 ByVal t_si As Double, _
                                 ByVal p_si As Double) As Double

'this simply wraps the density call air_density_si() of Air_3a
'for a uniform appearance air_g_... if a, t, p are the arguments

'returns   rho = 1/(dg/dP)_A_T  density of humid air in 1/K,

'a_si      mass fraction of dry air in kg/kg,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in air_3a

'Check value with default settings: air_g_density_si(0.9, 300, 1E5) = 1.09686619009421  v. 1.0
'Check value with default settings: air_g_density_si(0.9, 300, 1E5) = 1.09708772444253  v. 1.1

air_g_density_si = air_density_si(a_si, t_si, p_si)

End Function

'=========================================================================
Public Function air_g_gibbs_energy_si(ByVal a_si As Double, _
                                      ByVal t_si As Double, _
                                      ByVal p_si As Double) As Double

'returns   g = f + D*(df/dD)_A_T   Gibbs energy of humid air in J/kg,

'a_si      mass fraction of dry air in kg/kg,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in air_3a

'Check value with default settings: air_g_gibbs_energy_si(0.9, 300, 1E5) = 4574.43347101731  v. 1.0
'Check value with default settings: air_g_gibbs_energy_si(0.9, 300, 1E5) = 4577.93065688699  v. 1.1

Dim d As Double

air_g_gibbs_energy_si = ErrorReturn

If a_si < 0 Or a_si > 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = air_density_si(a_si, t_si, p_si)
If d = ErrorReturn Then Exit Function

air_g_gibbs_energy_si = air_f_gibbs_energy_si(a_si, t_si, d)

End Function

'=========================================================================
Public Function air_g_internal_energy_si(ByVal a_si As Double, _
                                         ByVal t_si As Double, _
                                         ByVal p_si As Double) As Double

'returns   u = g - T*(dg/dT)_A_P - P*(dg/dP)_A_T   internal energy of humid air in J/kg,

'a_si      mass fraction of dry air in kg/kg,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in air_3a

'Check value with default settings: air_g_internal_energy_si(0.9, 300, 1E5) = 186766.537702434  v. 1.0
'Check value with default settings: air_g_internal_energy_si(0.9, 300, 1E5) = 186778.541048021  v. 1.1

Dim d As Double

air_g_internal_energy_si = ErrorReturn

If a_si < 0 Or a_si > 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = air_density_si(a_si, t_si, p_si)
If d = ErrorReturn Then Exit Function

air_g_internal_energy_si = air_f_internal_energy_si(a_si, t_si, d)

End Function

'=========================================================================
Public Function air_g_kappa_s_si(ByVal a_si As Double, _
                                 ByVal t_si As Double, _
                                 ByVal p_si As Double) As Double

'returns   kappa_s = -(1/v) * (dv/dP)_A_eta  isentropic compressibility of humid air in 1/Pa,

'a_si      mass fraction of dry air in kg/kg,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in air_3a

'Check value with default settings: air_g_kappa_s_si(0.9, 300, 1E5) = 7.41018952703296E-06  v. 1.0
'Check value with default settings: air_g_kappa_s_si(0.9, 300, 1E5) = 7.4103450544927E-06  v. 1.1

Dim d As Double

air_g_kappa_s_si = ErrorReturn

If a_si < 0 Or a_si > 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = air_density_si(a_si, t_si, p_si)
If d = ErrorReturn Then Exit Function

air_g_kappa_s_si = air_f_kappa_s_si(a_si, t_si, d)

End Function

'=========================================================================
Public Function air_g_kappa_t_si(ByVal a_si As Double, _
                                 ByVal t_si As Double, _
                                 ByVal p_si As Double) As Double

'returns   kappa_t = -(1/v) * (dv/dP)_A_T  isothermal compressibility of humid air in 1/K,

'a_si      mass fraction of dry air in kg/kg,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in air_3a

'Check value with default settings: air_g_kappa_t_si(0.9, 300, 1E5) = 1.00324330660489E-05  v. 1.0
'Check value with default settings: air_g_kappa_t_si(0.9, 300, 1E5) = 1.00324517749485E-05  v. 1.1

Dim d As Double

air_g_kappa_t_si = ErrorReturn

If a_si < 0 Or a_si > 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = air_density_si(a_si, t_si, p_si)
If d = ErrorReturn Then Exit Function

air_g_kappa_t_si = air_f_kappa_t_si(a_si, t_si, d)

End Function

'=========================================================================
Public Function air_g_lapserate_si(ByVal a_si As Double, _
                                   ByVal t_si As Double, _
                                   ByVal p_si As Double) As Double

'returns   gamma = (dT/dP)_A_eta  "dry"-adiabatic lapse rate of humid air in K/Pa,

'a_si      mass fraction of dry air in kg/kg,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in air_3a

'Check value with default settings: air_g_lapserate_si(0.9, 300, 1E5) = 7.58539809056811E-04  v. 1.0
'Check value with default settings: air_g_lapserate_si(0.9, 300, 1E5) = 7.58481752250688E-04  v. 1.1

Dim d As Double

air_g_lapserate_si = ErrorReturn

If a_si < 0 Or a_si > 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = air_density_si(a_si, t_si, p_si)
If d = ErrorReturn Then Exit Function

air_g_lapserate_si = air_f_lapserate_si(a_si, t_si, d)

End Function

'=========================================================================
Public Function air_g_soundspeed_si(ByVal a_si As Double, _
                                    ByVal t_si As Double, _
                                    ByVal p_si As Double) As Double

'returns   c = sqr[ (dP/dD)_A_eta ] sound speed in humid air in m/s,

'a_si      mass fraction of dry air in kg/kg,
't_si      absolute temperature in K,
'p_si      absolute pressure in Pa

'note: the accuracy of this function depends on the iteration settings for
'      density computed in air_3a

'Check value with default settings: air_g_soundspeed_si(0.9, 300, 1E5) = 350.758752725391  v. 1.0
'Check value with default settings: air_g_soundspeed_si(0.9, 300, 1E5) = 350.719656182404  v. 1.1

Dim d As Double

air_g_soundspeed_si = ErrorReturn

If a_si < 0 Or a_si > 1 Then Exit Function
If t_si <= 0 Then Exit Function
If p_si <= 0 Then Exit Function

d = air_density_si(a_si, t_si, p_si)
If d = ErrorReturn Then Exit Function

air_g_soundspeed_si = air_f_soundspeed_si(a_si, t_si, d)

End Function

'==========================================================================
Public Function chk_lemmon_etal_2000(ByVal mode As Integer) As String

'comparison of computed selected values from the paper
'Lemmon, E.W., Jacobsen, R.T., Penoncello, S.G., Friend, D.G.
'Thermodynamic Properties of Air and Mixtures of Nitrogen, Argon and Oxygen
'From 60 to 2000 K at Pressures to 2000 MPa.
'J. Phys. Chem. Ref. Data 29(2000)331-362.

'Temperature Density  Pressure Int.Energy Enthalpy Entropy    Cv         Cp          Soundspeed
'K           mol/dm3  MPa      J/mol      J/mol    J/(mol K)  J/(mol K)  J/(mol K)   m/s
'
'140         0.087718 0.101325 2873.2     4028.3   176.60     20.81      29.38       236.4
'270         0.045164 0.101325 5578.9     7822.4   195.78     20.76      29.13       329.6
'400         0.030461 0.101325 8294.3     11621.0  207.26     21.04      29.38       400.5
'2000        0.006092 0.101325 48610.0    65242.0  259.62     27.90      36.21       863.5
'140         19.810   10.0     -329.46    175.34   118.30     25.82      72.88       418.1
'270         4.6064   10.0     4911.3     7082.2   155.25     21.61      35.28       349.7
'400         2.9202   10.0     7923.0     11347.0  168.19     21.38      31.50       425.6
'2000        0.59094  10.0     48600.0    65522.0  221.44     27.93      36.25       878.6
'270         47.327   2000.0   4354.8     46614.0  96.232     37.64      45.74       2899.8
'400         45.208   2000.0   8076.2     52316.0  113.52     34.45      42.27       2822.9
'2000        32.893   2000.0   53433.0    114240.0 176.45     31.86      38.21       2472.1

'mode = 0: check the Helmholtz function
'mode = 1: check the Gibbs function


Const ma = Molar_mass_air_L2000    'molar mass of air in kg/mol

Dim CRLF As String, TB As String
Dim txt As String
Dim d As Double, t As Double, x As Double, p As Double

Dim i As Integer, j As Integer, row As String
Dim lrow(9) As Integer

CRLF = Chr(13) + Chr(10)
TB = Chr(9)

If mode = 0 Then
  txt = "Implementation of the Dry-Air Helmholtz function in Visual Basic" + CRLF
Else
  txt = "Implementation of the Dry-Air Gibbs function in Visual Basic" + CRLF
End If
txt = txt + "for Publication in Ocean Science, 2009" + CRLF
txt = txt + "R. Feistel, IOW, Version " + Version + CRLF
txt = txt + "Compiled on " + CStr(Now) + CRLF + CRLF

txt = txt + "Selected function values as given in Lemmon et al., " + CRLF
txt = txt + "J. Phys. Chem. Ref. Data 29(2000)331-362:" + CRLF

dry_init_Lemmon2000  'set original adjustable constants

For i = -1 To 11
  If i > 0 Then
    txt = txt + CRLF + "Table value:"
  Else
    txt = txt + CRLF + "            "
  End If
  For j = 1 To 9
    Select Case i
      Case -1: row = Choose(j, "Temperature ", "Density  ", "Pressure ", "Int.Energy ", "Enthalpy ", "Entropy    ", "Cv         ", "Cp          ", "Soundspeed")
      Case 0:  row = Choose(j, "K           ", "mol/dm3  ", "MPa      ", "J/mol      ", "J/mol    ", "J/(mol K)  ", "J/(mol K)  ", "J/(mol K)   ", "m/s")
      Case 1:  row = Choose(j, "140         ", "0.087718 ", "0.101325 ", "2873.2     ", "4028.3   ", "176.60     ", "20.81      ", "29.38       ", "236.4")
      Case 2:  row = Choose(j, "270         ", "0.045164 ", "0.101325 ", "5578.9     ", "7822.4   ", "195.78     ", "20.76      ", "29.13       ", "329.6")
      Case 3:  row = Choose(j, "400         ", "0.030461 ", "0.101325 ", "8294.3     ", "11621.0  ", "207.26     ", "21.04      ", "29.38       ", "400.5")
      Case 4:  row = Choose(j, "2000        ", "0.006092 ", "0.101325 ", "48610.0    ", "65242.0  ", "259.62     ", "27.90      ", "36.21       ", "863.5")
      Case 5:  row = Choose(j, "140         ", "19.810   ", "10.0     ", "-329.46    ", "175.34   ", "118.30     ", "25.82      ", "72.88       ", "418.1")
      Case 6:  row = Choose(j, "270         ", "4.6064   ", "10.0     ", "4911.3     ", "7082.2   ", "155.25     ", "21.61      ", "35.28       ", "349.7")
      Case 7:  row = Choose(j, "400         ", "2.9202   ", "10.0     ", "7923.0     ", "11347.0  ", "168.19     ", "21.38      ", "31.50       ", "425.6")
      Case 8:  row = Choose(j, "2000        ", "0.59094  ", "10.0     ", "48600.0    ", "65522.0  ", "221.44     ", "27.93      ", "36.25       ", "878.6")
      Case 9:  row = Choose(j, "270         ", "47.327   ", "2000.0   ", "4354.8     ", "46614.0  ", "96.232     ", "37.64      ", "45.74       ", "2899.8")
      Case 10: row = Choose(j, "400         ", "45.208   ", "2000.0   ", "8076.2     ", "52316.0  ", "113.52     ", "34.45      ", "42.27       ", "2822.9")
      Case 11: row = Choose(j, "2000        ", "32.893   ", "2000.0   ", "53433.0    ", "114240.0 ", "176.45     ", "31.86      ", "38.21       ", "2472.1")
    End Select
    
    lrow(j) = Len(row)
    If j = 1 Then t = Val(row)
    If j = 2 Then d = Val(row)
    If j = 3 Then p = Val(row)
    txt = txt + TB + row
  Next j
  
  If i > 0 Then
    txt = txt + CRLF + "this code:  "
    For j = 1 To 9
    
      If mode = 0 Then
        Select Case j
          Case 1: x = t
          Case 2: x = d
          Case 3: x = 0.000001 * air_f_pressure_si(1, t, d * ma * 1000)
                  If x < 1 Then x = Round(x, 6) Else x = Round(x, 1)
          Case 4: x = ma * air_f_internal_energy_si(1, t, d * ma * 1000)
                  If Abs(x) < 1000 Then x = Round(x, 2) Else x = Round(x, 1)
          Case 5: x = ma * air_f_enthalpy_si(1, t, d * ma * 1000)
                  If Abs(x) < 1000 Then x = Round(x, 2) Else x = Round(x, 1)
          Case 6: x = ma * air_f_entropy_si(1, t, d * ma * 1000)
                  If x < 100 Then x = Round(x, 3) Else x = Round(x, 2)
          Case 7: x = Round(ma * air_f_cv_si(1, t, d * ma * 1000), 2)
          Case 8: x = Round(ma * air_f_cp_si(1, t, d * ma * 1000), 2)
          Case 9: x = Round(air_f_soundspeed_si(1, t, d * ma * 1000), 1)
        End Select
      Else
        Select Case j
          Case 1: x = t
          Case 2: x = 0.001 * air_g_density_si(1, t, p * 1000000#) / ma
                  If x < 0.1 Then
                    x = Round(x, 6)
                  ElseIf x < 1 Then
                    x = Round(x, 5)
                  ElseIf x < 10 Then
                    x = Round(x, 4)
                  Else
                    x = Round(x, 3)
                  End If
          Case 3: x = p
          Case 4: x = ma * air_g_internal_energy_si(1, t, p * 1000000#)
                  If Abs(x) < 1000 Then x = Round(x, 2) Else x = Round(x, 1)
          Case 5: x = ma * air_g_enthalpy_si(1, t, p * 1000000#)
                  If Abs(x) < 1000 Then x = Round(x, 2) Else x = Round(x, 1)
          Case 6: x = ma * air_g_entropy_si(1, t, p * 1000000#)
                  If x < 100 Then x = Round(x, 3) Else x = Round(x, 2)
          Case 7: x = Round(ma * air_g_cv_si(1, t, p * 1000000#), 2)
          Case 8: x = Round(ma * air_g_cp_si(1, t, p * 1000000#), 2)
          Case 9: x = Round(air_g_soundspeed_si(1, t, p * 1000000#), 1)
        End Select
      End If
      
      row = Trim(Str(x))
      If Left(row, 1) = "." Then row = "0" + row
      txt = txt + TB + Left(row + Space(lrow(j)), lrow(j))
    Next j
  End If
Next i

dry_init_clear  'restore sea-air adjustable coefficients

txt = txt + CRLF + CRLF + "Note: For this comparison, the originally published adjustable coefficients were used"

'added in v. 1.1:
txt = txt + CRLF + CRLF + "Note: The molar mass of air used in SIA Version 1.1 results in minor offsets"

txt = txt + CRLF + CRLF + "Note: For this comparison, check_limits =" + Str(check_limits) + _
                          " was initilialized in Property Get check_limits"

chk_lemmon_etal_2000 = txt

'v. 1.1:
'=================================
'Print chk_lemmon_etal_2000(0):

'Implementation of the Dry-Air Helmholtz function in Visual Basic
'for Publication in Ocean Science, 2009
'R. Feistel, IOW, Version 28 May 2010
'Compiled on 29.05.2010 09:58:01
'
'Selected function values as given in Lemmon et al.,
'J. Phys. Chem. Ref. Data 29(2000)331-362:
'
'              Temperature   Density   Pressure  Int.Energy  Enthalpy  Entropy     Cv          Cp            Soundspeed
'              K             mol/dm3   MPa       J/mol       J/mol     J/(mol K)   J/(mol K)   J/(mol K)     m/s
'Table value:  140           0.087718  0.101325  2873.2      4028.3    176.60      20.81       29.38         236.4
'this code:    140           0.087718  0.101324  2873.2      4028.3    176.6       20.81       29.38         236.4
'Table value:  270           0.045164  0.101325  5578.9      7822.4    195.78      20.76       29.13         329.6
'this code:    270           0.045164  0.101324  5578.9      7822.4    195.78      20.76       29.13         329.5
'Table value:  400           0.030461  0.101325  8294.3      11621.0   207.26      21.04       29.38         400.5
'this code:    400           0.030461  0.101326  8294.3      11620.8   207.26      21.04       29.38         400.4
'Table value:  2000          0.006092  0.101325  48610.0     65242.0   259.62      27.90       36.21         863.5
'this code:    2000          0.006092  0.101322  48610.4     65242.4   259.62      27.9        36.21         863.4
'Table value:  140           19.810    10.0      -329.46     175.34    118.30      25.82       72.88         418.1
'this code:    140           19.81     10        -329.5      175.32    118.3       25.82       72.88         418.1
'Table value:  270           4.6064    10.0      4911.3      7082.2    155.25      21.61       35.28         349.7
'this code:    270           4.6064    10        4911.3      7082.2    155.25      21.61       35.28         349.7
'Table value:  400           2.9202    10.0      7923.0      11347.0   168.19      21.38       31.50         425.6
'this code:    400           2.9202    10        7923        11347.4   168.19      21.38       31.5          425.5
'Table value:  2000          0.59094   10.0      48600.0     65522.0   221.44      27.93       36.25         878.6
'this code:    2000          0.59094   10        48599.8     65522     221.44      27.93       36.25         878.5
'Table value:  270           47.327    2000.0    4354.8      46614.0   96.232      37.64       45.74         2899.8
'this code:    270           47.327    2000      4354.9      46615     96.232      37.64       45.74         2899.5
'Table value:  400           45.208    2000.0    8076.2      52316.0   113.52      34.45       42.27         2822.9
'this code:    400           45.208    2000      8076.1      52315.6   113.52      34.45       42.27         2822.5
'Table value:  2000          32.893    2000.0    53433.0     114240.0  176.45      31.86       38.21         2472.1
'this code:    2000          32.893    2000      53432.6     114236.2  176.45      31.86       38.21         2471.8
'
'Note: For this comparison, the originally published adjustable coefficients were used
'
'Note: The molar mass of air used in SIA Version 1.1 results in minor offsets
'
'Note: For this comparison, check_limits = 0 was initilialized in Property Get check_limits

'v. 1.1:
'=================================
'Print chk_lemmon_etal_2000(1):

'Implementation of the Dry-Air Gibbs function in Visual Basic
'for Publication in Ocean Science, 2009
'R. Feistel, IOW, Version 28 May 2010
'Compiled on 29.05.2010 09:58:31
'
'Selected function values as given in Lemmon et al.,
'J. Phys. Chem. Ref. Data 29(2000)331-362:
'
'              Temperature   Density   Pressure  Int.Energy  Enthalpy  Entropy     Cv          Cp            Soundspeed
'              K             mol/dm3   MPa       J/mol       J/mol     J/(mol K)   J/(mol K)   J/(mol K)     m/s
'Table value:  140           0.087718  0.101325  2873.2      4028.3    176.60      20.81       29.38         236.4
'this code:    140           0.087718  0.101325  2873.2      4028.3    176.6       20.81       29.38         236.4
'Table value:  270           0.045164  0.101325  5578.9      7822.4    195.78      20.76       29.13         329.6
'this code:    270           0.045164  0.101325  5578.9      7822.4    195.78      20.76       29.13         329.5
'Table value:  400           0.030461  0.101325  8294.3      11621.0   207.26      21.04       29.38         400.5
'this code:    400           0.030461  0.101325  8294.3      11620.8   207.26      21.04       29.38         400.4
'Table value:  2000          0.006092  0.101325  48610.0     65242.0   259.62      27.90       36.21         863.5
'this code:    2000          0.006092  0.101325  48610.4     65242.4   259.62      27.9        36.21         863.4
'Table value:  140           19.810    10.0      -329.46     175.34    118.30      25.82       72.88         418.1
'this code:    140           19.81     10        -329.46     175.34    118.3       25.82       72.88         418.1
'Table value:  270           4.6064    10.0      4911.3      7082.2    155.25      21.61       35.28         349.7
'this code:    270           4.6064    10        4911.3      7082.2    155.25      21.61       35.28         349.7
'Table value:  400           2.9202    10.0      7923.0      11347.0   168.19      21.38       31.50         425.6
'this code:    400           2.9202    10        7923        11347.4   168.19      21.38       31.5          425.5
'Table value:  2000          0.59094   10.0      48600.0     65522.0   221.44      27.93       36.25         878.6
'this code:    2000          0.59094   10        48599.8     65522     221.44      27.93       36.25         878.5
'Table value:  270           47.327    2000.0    4354.8      46614.0   96.232      37.64       45.74         2899.8
'this code:    270           47.327    2000      4354.8      46614.2   96.232      37.64       45.74         2899.5
'Table value:  400           45.208    2000.0    8076.2      52316.0   113.52      34.45       42.27         2822.9
'this code:    400           45.208    2000      8076.2      52316     113.52      34.45       42.27         2822.5
'Table value:  2000          32.893    2000.0    53433.0     114240.0  176.45      31.86       38.21         2472.1
'this code:    2000          32.893    2000      53432.6     114235.9  176.45      31.86       38.21         2471.8
'
'Note: For this comparison, the originally published adjustable coefficients were used
'
'Note: The molar mass of air used in SIA Version 1.1 results in minor offsets
'
'Note: For this comparison, check_limits = 0 was initilialized in Property Get check_limits


'v. 1.0:
'=================================
'Print chk_lemmon_etal_2000(0):

'Implementation of the Dry-Air Helmholtz function in Visual Basic
'for Publication in Ocean Science, 2009
'R. Feistel, IOW, Version 12 Jun 2009
'Compiled on 31.07.2009 15:27:41
'
'Selected function values as given in Lemmon et al.,
'J. Phys. Chem. Ref. Data 29(2000)331-362:
'
'              Temperature   Density   Pressure  Int.Energy  Enthalpy  Entropy     Cv          Cp            Soundspeed
'              K             mol/dm3   MPa       J/mol       J/mol     J/(mol K)   J/(mol K)   J/(mol K)     m/s
'Table value:  140           0.087718  0.101325  2873.2      4028.3    176.60      20.81       29.38         236.4
'Computed:     140           0.087718  0.101324  2873.2      4028.3    176.6       20.81       29.38         236.4
'Table value:  270           0.045164  0.101325  5578.9      7822.4    195.78      20.76       29.13         329.6
'Computed:     270           0.045164  0.101324  5578.9      7822.4    195.78      20.76       29.13         329.6
'Table value:  400           0.030461  0.101325  8294.3      11621.0   207.26      21.04       29.38         400.5
'Computed:     400           0.030461  0.101326  8294.3      11620.8   207.26      21.04       29.38         400.5
'Table value:  2000          0.006092  0.101325  48610.0     65242.0   259.62      27.90       36.21         863.5
'Computed:     2000          0.006092  0.101322  48610.4     65242.4   259.62      27.9        36.21         863.5
'Table value:  140           19.810    10.0      -329.46     175.34    118.30      25.82       72.88         418.1
'Computed:     140           19.81     10        -329.5      175.32    118.3       25.82       72.88         418.1
'Table value:  270           4.6064    10.0      4911.3      7082.2    155.25      21.61       35.28         349.7
'Computed:     270           4.6064    10        4911.3      7082.2    155.25      21.61       35.28         349.7
'Table value:  400           2.9202    10.0      7923.0      11347.0   168.19      21.38       31.50         425.6
'Computed:     400           2.9202    10        7923        11347.4   168.19      21.38       31.5          425.6
'Table value:  2000          0.59094   10.0      48600.0     65522.0   221.44      27.93       36.25         878.6
'Computed:     2000          0.59094   10        48599.8     65522     221.44      27.93       36.25         878.6
'Table value:  270           47.327    2000.0    4354.8      46614.0   96.232      37.64       45.74         2899.8
'Computed:     270           47.327    2000      4354.9      46615     96.232      37.64       45.74         2899.9
'Table value:  400           45.208    2000.0    8076.2      52316.0   113.52      34.45       42.27         2822.9
'Computed:     400           45.208    2000      8076.1      52315.6   113.52      34.45       42.27         2822.9
'Table value:  2000          32.893    2000.0    53433.0     114240.0  176.45      31.86       38.21         2472.1
'Computed:     2000          32.893    2000      53432.6     114236.2  176.45      31.86       38.21         2472.1
'
'Note: For this comparison, the originally published adjustable coefficients were used
'Note: For this comparison, check_limits_value = 0 was initilialized in Property Get check_limits

'v. 1.0:
'=================================
'Print chk_lemmon_etal_2000(1):

'Implementation of the Dry-Air Gibbs function in Visual Basic
'for Publication in Ocean Science, 2009
'R. Feistel, IOW, Version 12 Jun 2009
'Compiled on 31.07.2009 15:29:11
'
'Selected function values as given in Lemmon et al.,
'J. Phys. Chem. Ref. Data 29(2000)331-362:
'
'              Temperature   Density   Pressure  Int.Energy  Enthalpy  Entropy     Cv          Cp            Soundspeed
'              K             mol/dm3   MPa       J/mol       J/mol     J/(mol K)   J/(mol K)   J/(mol K)     m/s
'Table value:  140           0.087718  0.101325  2873.2      4028.3    176.60      20.81       29.38         236.4
'Computed:     140           0.087718  0.101325  2873.2      4028.3    176.6       20.81       29.38         236.4
'Table value:  270           0.045164  0.101325  5578.9      7822.4    195.78      20.76       29.13         329.6
'Computed:     270           0.045164  0.101325  5578.9      7822.4    195.78      20.76       29.13         329.6
'Table value:  400           0.030461  0.101325  8294.3      11621.0   207.26      21.04       29.38         400.5
'Computed:     400           0.030461  0.101325  8294.3      11620.8   207.26      21.04       29.38         400.5
'Table value:  2000          0.006092  0.101325  48610.0     65242.0   259.62      27.90       36.21         863.5
'Computed:     2000          0.006092  0.101325  48610.4     65242.4   259.62      27.9        36.21         863.5
'Table value:  140           19.810    10.0      -329.46     175.34    118.30      25.82       72.88         418.1
'Computed:     140           19.81     10        -329.46     175.34    118.3       25.82       72.88         418.1
'Table value:  270           4.6064    10.0      4911.3      7082.2    155.25      21.61       35.28         349.7
'Computed:     270           4.6064    10        4911.3      7082.2    155.25      21.61       35.28         349.7
'Table value:  400           2.9202    10.0      7923.0      11347.0   168.19      21.38       31.50         425.6
'Computed:     400           2.9202    10        7923        11347.4   168.19      21.38       31.5          425.6
'Table value:  2000          0.59094   10.0      48600.0     65522.0   221.44      27.93       36.25         878.6
'Computed:     2000          0.59094   10        48599.8     65522     221.44      27.93       36.25         878.6
'Table value:  270           47.327    2000.0    4354.8      46614.0   96.232      37.64       45.74         2899.8
'Computed:     270           47.327    2000      4354.8      46614.2   96.232      37.64       45.74         2899.8
'Table value:  400           45.208    2000.0    8076.2      52316.0   113.52      34.45       42.27         2822.9
'Computed:     400           45.208    2000      8076.2      52316     113.52      34.45       42.27         2822.9
'Table value:  2000          32.893    2000.0    53433.0     114240.0  176.45      31.86       38.21         2472.1
'Computed:     2000          32.893    2000      53432.6     114235.9  176.45      31.86       38.21         2472.1
'
'Note: For this comparison, the originally published adjustable coefficients were used
'Note: For this comparison, check_limits_value = 0 was initilialized in Property Get check_limits

End Function


