Attribute VB_Name = "Air_2_Mdl"
Option Explicit


'#########################################################################

'This module requires the library modules
'     Constants_0_Mdl, file Constants_0.bas
'     Maths_0_Mdl,     file Maths_0.bas
'     Flu_1_Mdl,       file flu_1.bas
'     Air_1_Mdl,       file air_1.bas

'#########################################################################

'=========================================================================
'This module implements the air-water mixing Helmholtz function
'air_f_mix_si(drv_a, drv_t, drv_d, a_si, a_si, t_si, d_si),
'and its first and second partial derivatives with respect to air fraction,
'temperature and density,

'the Helmholtz potential of humid air as a function of the dry-air fraction
'in kg/kg, absolute temperature in K and humid-air density in kg/m^3,
'air_f_si(drv_a, drv_t, drv_d, a_si, a_si, t_si, d_si),
'and its first and second partial derivatives with respect to air fraction,
'temperature and density,

'and thermodynamic properties of humid air computed from the Helmholtz potential.

'Implementation in VB6 by Rainer Feistel
'for publication in Ocean Science, as described in the papers

'Feistel, R., Wright, D.G., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part I: Background and Equations.
'Ocean Science, 2009

'Wright, D.G., Feistel, R., Jackett, D.R., Miyagawa, K., Reissmann, J.H.,
'Wagner, W., Overhoff, U., Guder, C., Feistel, A., Marion, G.M.:
'Numerical implementation and oceanographic application of the thermodynamic
'potentials of water, vapour, ice, seawater and air. Part II: The Library Routines,
'Ocean Science, 2009

'Feistel, R., Kretzschmar, H.-J., Span, R., Hagen, E., Wright, D.G., Herrmann, S.:
'Thermodynamic Properties of Sea Air.
'Ocean Science Discussion 6(2009)2193-2325.
'==========================================================================

'Private Const ErrorReturn = 9.99999999E+98

Private Const Version = "29 May 2010"

'==========================================================================
Public Function air_f_mix_si(ByVal drv_a As Integer, _
                             ByVal drv_t As Integer, _
                             ByVal drv_d As Integer, _
                             ByVal a_si As Double, _
                             ByVal t_si As Double, _
                             ByVal d_si As Double) As Double

'this function implements the specific mixing Helmholtz energy of humid air and its
'derivatives with respect to the air fraction, temperature, and humid-air density

'Output: air_f_mix_si = mixing Helmholtz energy in J kg-1 K^-drv_t (kg/m3)^-drv_d

'Input:  drv_a  order of derivative with respect to the air mass fraction
'        drv_t  order of derivative with respect to temperature
'        drv_d  order of derivative with respect to humid-air density
'        a_si   mass fraction of air in humid air in kg/kg
'        t_si   absolute temperature in K
'        d_si   humid-air density in kg/m3

'check values, v. 1.0 and v. 1.1:
'air_f_mix_si( 0, 0, 0, 0.9,300,1) =-25.8379179181588
'air_f_mix_si( 1, 0, 0, 0.9,300,1) = 233.827370433721
'air_f_mix_si( 0, 1, 0, 0.9,300,1) = 0.1641959520604
'air_f_mix_si( 0, 0, 1, 0.9,300,1) =-26.2357498619055
'air_f_mix_si( 2, 0, 0, 0.9,300,1) = 500.273928155059
'air_f_mix_si( 1, 1, 0, 0.9,300,1) =-1.53932744055467
'air_f_mix_si( 1, 0, 1, 0.9,300,1) = 241.520643317113
'air_f_mix_si( 0, 2, 0, 0.9,300,1) =-0.687329742958611E-3
'air_f_mix_si( 0, 1, 1, 0.9,300,1) = 0.172192606102968
'air_f_mix_si( 0, 0, 2, 0.9,300,1) =-0.795663887493488

Const mw = Molar_mass_H2O_si    'molar mass of H2O in kg/mol
Const ma = Molar_mass_air_si    'molar mass of air in kg/mol
Const Raw = Gas_constant_molar_si / (ma * mw)

Dim bawt As Double, caawt As Double, cawwt As Double
Dim baw As Double, caaw As Double, caww As Double

air_f_mix_si = ErrorReturn

If drv_a < 0 Then Exit Function
If drv_t < 0 Then Exit Function
If drv_d < 0 Then Exit Function
If drv_a + drv_t + drv_d > 2 Then Exit Function

If a_si < 0 Or a_si > 1 Then Exit Function

If check_limits = 1 Then
  If t_si < mix_air_tmin Or t_si > mix_air_tmax Then Exit Function
  If d_si <= mix_air_dmin Or d_si > mix_air_dmax Then Exit Function
Else
  If t_si <= 0 Then Exit Function
  If d_si <= 0 Then Exit Function
End If

'temperature derivatives of the virial coefficients
Select Case drv_t

  Case 0: baw = air_baw_m3mol(0, t_si)
          If baw = ErrorReturn Then Exit Function
          bawt = 2 * Raw * t_si * baw

          caaw = air_caaw_m6mol2(0, t_si)
          If caaw = ErrorReturn Then Exit Function
          caawt = 1.5 * Raw * t_si * caaw / ma

          caww = air_caww_m6mol2(0, t_si)
          If caww = ErrorReturn Then Exit Function
          cawwt = 1.5 * Raw * t_si * caww / mw
          
  Case 1: baw = air_baw_m3mol(1, t_si)
          If baw = ErrorReturn Then Exit Function
          bawt = 2 * Raw * t_si * baw
          baw = air_baw_m3mol(0, t_si)
          If baw = ErrorReturn Then Exit Function
          bawt = bawt + 2 * Raw * baw

          caaw = air_caaw_m6mol2(1, t_si)
          If caaw = ErrorReturn Then Exit Function
          caawt = 1.5 * Raw * t_si * caaw / ma
          caaw = air_caaw_m6mol2(0, t_si)
          If caaw = ErrorReturn Then Exit Function
          caawt = caawt + 1.5 * Raw * caaw / ma

          caww = air_caww_m6mol2(1, t_si)
          If caww = ErrorReturn Then Exit Function
          cawwt = 1.5 * Raw * t_si * caww / mw
          caww = air_caww_m6mol2(0, t_si)
          If caww = ErrorReturn Then Exit Function
          cawwt = cawwt + 1.5 * Raw * caww / mw

  Case 2: baw = air_baw_m3mol(2, t_si)
          If baw = ErrorReturn Then Exit Function
          bawt = 2 * Raw * t_si * baw
          baw = air_baw_m3mol(1, t_si)
          If baw = ErrorReturn Then Exit Function
          bawt = bawt + 4 * Raw * baw

          caaw = air_caaw_m6mol2(2, t_si)
          If caaw = ErrorReturn Then Exit Function
          caawt = 1.5 * Raw * t_si * caaw / ma
          caaw = air_caaw_m6mol2(1, t_si)
          If caaw = ErrorReturn Then Exit Function
          caawt = caawt + 1.5 * Raw * 2 * caaw / ma

          caww = air_caww_m6mol2(2, t_si)
          If caww = ErrorReturn Then Exit Function
          cawwt = 1.5 * Raw * t_si * caww / mw
          caww = air_caww_m6mol2(1, t_si)
          If caww = ErrorReturn Then Exit Function
          cawwt = cawwt + 1.5 * Raw * 2 * caww / mw
          
  Case Else: Exit Function
End Select

'density derivatives of the virial coefficients
Select Case drv_d

  Case 0: bawt = bawt * d_si
          caawt = caawt * d_si ^ 2
          cawwt = cawwt * d_si ^ 2
          
  Case 1: caawt = 2 * caawt * d_si
          cawwt = 2 * cawwt * d_si
          
  Case 2: bawt = 0
          caawt = 2 * caawt
          cawwt = 2 * cawwt
          
  Case Else: Exit Function
End Select

'air-fraction derivatives of the virial coefficients
Select Case drv_a

  Case 0: bawt = bawt * a_si * (1 - a_si)
          caawt = caawt * a_si ^ 2 * (1 - a_si)
          cawwt = cawwt * a_si * (1 - a_si) ^ 2
          
  Case 1: bawt = bawt * (1 - 2 * a_si)
          caawt = caawt * a_si * (2 - 3 * a_si)
          cawwt = cawwt * (1 - 4 * a_si + 3 * a_si ^ 2)
          
  Case 2: bawt = -2 * bawt
          caawt = caawt * (2 - 6 * a_si)
          cawwt = cawwt * (-4 + 6 * a_si)
          
  Case Else: Exit Function
End Select

air_f_mix_si = bawt + caawt + cawwt

End Function

'==========================================================================
Public Function air_f_si(ByVal drv_a As Integer, _
                         ByVal drv_t As Integer, _
                         ByVal drv_d As Integer, _
                         ByVal a_si As Double, _
                         ByVal t_si As Double, _
                         ByVal d_si As Double) As Double

'This function implements the Helmholtz potential of humid air and its first and second
'derivatives with respect to the air fraction, temperature, and humid-air density

'Output:  air_f_si = Helmholtz energy in J/kg, or its derivatives, in J kg-1 K^-drv_t (kg/m3)^-drv_d

'Input:   drv_a  order of the air fraction derivative
'         drv_t  order of the temperature derivative
'         drv_d  order of the density derivative
'         a_si mass fraction of dry air in humid air in kg/kg
'         t_si absolute temperature in K
'         d_si density of humid air in kg/m3

'check values, v. 1.0:
'air_f_si( 0, 0, 0, 0.9,300,1) =-95024.7900037553
'air_f_si( 1, 0, 0, 0.9,300,1) =-205630.937995868
'air_f_si( 0, 1, 0, 0.9,300,1) =-940.150652240964
'air_f_si( 0, 0, 1, 0.9,300,1) = 91193.7358459761
'air_f_si( 2, 0, 0, 0.9,300,1) = 1447791.10862457
'air_f_si( 1, 1, 0, 0.9,300,1) = 7443.19342187571
'air_f_si( 1, 0, 1, 0.9,300,1) =-48827.5293333158
'air_f_si( 0, 2, 0, 0.9,300,1) =-2.96533251320995
'air_f_si( 0, 1, 1, 0.9,300,1) = 312.124501057728
'air_f_si( 0, 0, 2, 0.9,300,1) =-91439.8037141926

'check values, v. 1.1:
'air_f_si( 0, 0, 0, 0.9,300,1) =-95019.594323089
'air_f_si( 1, 0, 0, 0.9,300,1) =-205645.554994923
'air_f_si( 0, 1, 0, 0.9,300,1) =-940.175394022848
'air_f_si( 0, 0, 1, 0.9,300,1) = 91175.3848661597
'air_f_si( 2, 0, 0, 0.9,300,1) = 1447768.46379203
'air_f_si( 1, 1, 0, 0.9,300,1) = 7443.09771949878
'air_f_si( 1, 0, 1, 0.9,300,1) =-48847.909682604
'air_f_si( 0, 2, 0, 0.9,300,1) =-2.96482218054432
'air_f_si( 0, 1, 1, 0.9,300,1) = 312.063110700377
'air_f_si( 0, 0, 2, 0.9,300,1) =-91421.4440689192

Dim f As Double, da As Double, dw As Double
Dim fw As Double, fa As Double, fm As Double

air_f_si = ErrorReturn

If drv_a < 0 Then Exit Function
If drv_t < 0 Then Exit Function
If drv_d < 0 Then Exit Function
If drv_a + drv_t + drv_d > 2 Then Exit Function

If a_si < 0 Or a_si > 1 Then Exit Function
If (a_si = 0 Or a_si = 1) And drv_a > 0 Then Exit Function
If t_si <= 0 Then Exit Function
If d_si <= 0 Then Exit Function

da = a_si * d_si          'partial density of air
dw = (1 - a_si) * d_si    'partial density of vapour

Select Case drv_a

  Case 0: f = 0
          If dw > 0 Then
            fw = flu_f_si(drv_t, drv_d, t_si, dw)
            If fw = ErrorReturn Then Exit Function
            f = f + (1 - a_si) ^ (drv_d + 1) * fw
          End If
          If da > 0 Then
            fa = dry_f_si(drv_t, drv_d, t_si, da)
            If fa = ErrorReturn Then Exit Function
            f = f + a_si ^ (drv_d + 1) * fa
          End If
          If dw > 0 And da > 0 Then
            fm = air_f_mix_si(0, drv_t, drv_d, a_si, t_si, d_si)
            If fm = ErrorReturn Then Exit Function
            f = f + fm
          End If

  Case 1:

  Select Case drv_d
    Case 0: f = 0
            If dw > 0 Then
              fw = flu_f_si(drv_t, 0, t_si, dw)
              If fw = ErrorReturn Then Exit Function
              f = f - fw
              fw = flu_f_si(drv_t, 1, t_si, dw)
              If fw = ErrorReturn Then Exit Function
              f = f - dw * fw
            End If
            If da > 0 Then
              fa = dry_f_si(drv_t, 0, t_si, da)
              If fa = ErrorReturn Then Exit Function
              f = f + fa
              fa = dry_f_si(drv_t, 1, t_si, da)
              If fa = ErrorReturn Then Exit Function
              f = f + da * fa
            End If
            If dw > 0 And da > 0 Then
              fm = air_f_mix_si(1, drv_t, 0, a_si, t_si, d_si)
              If fm = ErrorReturn Then Exit Function
              f = f + fm
            End If

    Case 1: f = 0
            If dw > 0 Then
              fw = flu_f_si(drv_t, 1, t_si, dw)
              If fw = ErrorReturn Then Exit Function
              f = f - (1 - a_si) * 2 * fw
              fw = flu_f_si(drv_t, 2, t_si, dw)
              If fw = ErrorReturn Then Exit Function
              f = f - (1 - a_si) * dw * fw
            End If
            If da > 0 Then
              fa = dry_f_si(drv_t, 1, t_si, da)
              If fa = ErrorReturn Then Exit Function
              f = f + a_si * 2 * fa
              fa = dry_f_si(drv_t, 2, t_si, da)
              If fa = ErrorReturn Then Exit Function
              f = f + a_si * da * fa
            End If
            If dw > 0 And da > 0 Then
              fm = air_f_mix_si(1, drv_t, drv_d, a_si, t_si, d_si)
              If fm = ErrorReturn Then Exit Function
              f = f + fm
            End If
            
    Case Else: Exit Function
    
  End Select

  Case 2: f = 0
          If dw > 0 Then
            fw = flu_f_si(drv_t, 1, t_si, dw)
            If fw = ErrorReturn Then Exit Function
            f = f + 2 * d_si * fw
            fw = flu_f_si(drv_t, 2, t_si, dw)
            If fw = ErrorReturn Then Exit Function
            f = f + dw * d_si * fw
          End If
          If da > 0 Then
            fa = dry_f_si(drv_t, 1, t_si, da)
            If fa = ErrorReturn Then Exit Function
            f = f + 2 * d_si * fa
            fa = dry_f_si(drv_t, 2, t_si, da)
            If fa = ErrorReturn Then Exit Function
            f = f + da * d_si * fa
          End If
          If dw > 0 And da > 0 Then
            fm = air_f_mix_si(2, drv_t, 0, a_si, t_si, d_si)
            If fm = ErrorReturn Then Exit Function
            f = f + fm
          End If
          
    Case Else: Exit Function
    
End Select

air_f_si = f

End Function

'==========================================================================
Public Function air_2_example_call(ByVal a_si As Double, ByVal t_si As Double, ByVal d_si As Double) As String

'standard ocean conditions of dry air at t_si = 273.15, d_si = 1.29275937468394

Const ma = Molar_mass_air_L2000    'molar mass of air in kg/mol

Dim CRLF As String, TB As String
Dim txt As String

CRLF = Chr(13) + Chr(10)
TB = Chr(9)

txt = " Implementation of thermodynamic properties of humid air in Visual Basic" + CRLF
txt = txt + " for Publication in Ocean Science, 2009" + CRLF
txt = txt + " R. Feistel, IOW, Version " + Version + CRLF
txt = txt + " Compiled on " + CStr(Now) + CRLF + CRLF

If a_si < 0 Or a_si > 1 Then
  txt = txt + "incorrect: dry-air mass fraction between 0 and 1 rrequired"
  air_2_example_call = txt
  Exit Function
End If

If d_si <= 0 Then
  txt = txt + "incorrect: positive density required"
  air_2_example_call = txt
  Exit Function
End If

If t_si <= 0 Then
  txt = txt + "incorrect: positive temperature required"
  air_2_example_call = txt
  Exit Function
End If

txt = txt + " Dry air fraction:           " + TB + Str(a_si) + TB + "kg/kg" + CRLF
txt = txt + " Absolute temperature:       " + TB + Str(t_si) + TB + "K" + CRLF
txt = txt + " Mass density:               " + TB + Str(d_si) + TB + "kg/m3" + CRLF
txt = txt + " Molar density:              " + TB + Str(0.001 * d_si / ma) + TB + "mol/l" + CRLF + CRLF

txt = txt + " Specific volume:            " + TB + Str(1 / d_si) + TB + "m3/kg" + CRLF
txt = txt + " Molar volume:               " + TB + Str(1 / (0.001 * d_si / ma)) + TB + "l/mol" + CRLF
txt = txt + " Specific heat capacity cp:  " + TB + Str(air_f_cp_si(a_si, t_si, d_si)) + TB + "J/(kg K)" + CRLF
txt = txt + " Molar heat capacity cp:     " + TB + Str(air_f_cp_si(a_si, t_si, d_si) * ma) + TB + "J/(mol K)" + CRLF
txt = txt + " Specific heat capacity cv:  " + TB + Str(air_f_cv_si(a_si, t_si, d_si)) + TB + "J/(kg K)" + CRLF
txt = txt + " Molar heat capacity cv:     " + TB + Str(air_f_cv_si(a_si, t_si, d_si) * ma) + TB + "J/(mol K)" + CRLF
txt = txt + " Specific enthalpy:          " + TB + Str(air_f_enthalpy_si(a_si, t_si, d_si)) + TB + "J/kg" + CRLF
txt = txt + " Molar enthalpy:             " + TB + Str(air_f_enthalpy_si(a_si, t_si, d_si) * ma) + TB + "J/mol" + CRLF
txt = txt + " Specific entropy:           " + TB + Str(air_f_entropy_si(a_si, t_si, d_si)) + TB + "J/(kg K)" + CRLF
txt = txt + " Molar entropy:              " + TB + Str(air_f_entropy_si(a_si, t_si, d_si) * ma) + TB + "J/(mol K)" + CRLF
txt = txt + " Thermal expansion:          " + TB + Str(air_f_expansion_si(a_si, t_si, d_si)) + TB + "1/K" + CRLF
txt = txt + " Specific Gibbs energy:      " + TB + Str(air_f_gibbs_energy_si(a_si, t_si, d_si)) + TB + "J/kg" + CRLF
txt = txt + " Molar Gibbs energy:         " + TB + Str(air_f_gibbs_energy_si(a_si, t_si, d_si) * ma) + TB + "J/mol" + CRLF
txt = txt + " Specific internal energy:   " + TB + Str(air_f_internal_energy_si(a_si, t_si, d_si)) + TB + "J/kg" + CRLF
txt = txt + " Molar internal energy:      " + TB + Str(air_f_internal_energy_si(a_si, t_si, d_si) * ma) + TB + "J/mol" + CRLF
txt = txt + " Adiabatic compressibility:  " + TB + Str(air_f_kappa_s_si(a_si, t_si, d_si)) + TB + "1/Pa" + CRLF
txt = txt + " Isothermal compressibility: " + TB + Str(air_f_kappa_t_si(a_si, t_si, d_si)) + TB + "1/Pa" + CRLF
txt = txt + " Adiabatic lapse rate:       " + TB + Str(air_f_lapserate_si(a_si, t_si, d_si)) + TB + "K/Pa" + CRLF
txt = txt + " Adiabatic lapse rate:       " + TB + Str(100 * air_f_lapserate_si(a_si, t_si, d_si) * d_si * 9.81) + TB + "K/(100 m)" + CRLF
txt = txt + " Absolute pressure:          " + TB + Str(air_f_pressure_si(a_si, t_si, d_si)) + TB + "Pa" + CRLF
txt = txt + " Sound speed:                " + TB + Str(air_f_soundspeed_si(a_si, t_si, d_si)) + TB + "m/s" + CRLF + CRLF

air_2_example_call = txt

End Function

'==========================================================================
Public Function air_f_cp_si(ByVal a_si As Double, ByVal t_si As Double, ByVal d_si As Double) As Double

'returns   cp = T * (deta/dT)_A_P  isobaric heat capacity in J/kg,
'a_si      mass fraction of dry air in humid air
't_si      absolute temperature in K,
'd_si      humid-air density in kg m-3

'check value: air_f_cp_si(0.9,300,1) = 1210.95501131063  v. 1.0
'check value: air_f_cp_si(0.9,300,1) = 1210.7403105811   v. 1.1

Dim f_d As Double
Dim f_tt As Double
Dim f_td As Double
Dim f_dd As Double
Dim x As Double

air_f_cp_si = ErrorReturn

f_d = air_f_si(0, 0, 1, a_si, t_si, d_si)
If f_d = ErrorReturn Then Exit Function
f_tt = air_f_si(0, 2, 0, a_si, t_si, d_si)
If f_tt = ErrorReturn Then Exit Function
f_td = air_f_si(0, 1, 1, a_si, t_si, d_si)
If f_td = ErrorReturn Then Exit Function
f_dd = air_f_si(0, 0, 2, a_si, t_si, d_si)
If f_dd = ErrorReturn Then Exit Function

x = 2 * f_d + d_si * f_dd
If x = 0 Then Exit Function

air_f_cp_si = t_si * (f_td ^ 2 * d_si / x - f_tt)

End Function

'==========================================================================
Public Function air_f_cv_si(ByVal a_si As Double, ByVal t_si As Double, ByVal d_si As Double) As Double

'returns   cv = T * (deta/dT)_A_D  isochoric heat capacity in J/kg,
'a_si      mass fraction of dry air in humid air
't_si      absolute temperature in K,
'd_si      humid-air density in kg m-3

'check value: air_f_cv_si(0.9,300,1) = 889.599753962985  v. 1.0
'check value: air_f_cv_si(0.9,300,1) = 889.446654163297  v. 1.1

Dim f_tt As Double

air_f_cv_si = ErrorReturn

f_tt = air_f_si(0, 2, 0, a_si, t_si, d_si)
If f_tt = ErrorReturn Then Exit Function

air_f_cv_si = -t_si * f_tt

End Function

'==========================================================================
Public Function air_f_enthalpy_si(ByVal a_si As Double, ByVal t_si As Double, ByVal d_si As Double) As Double

'returns   h = f - T*(df/dT)_A_D + D*(df/dD)_A_T  enthalpy in J/kg,
'a_si      mass fraction of dry air in humid air
't_si      absolute temperature in K,
'd_si      humid-air density in kg m-3

'check value: air_f_enthalpy_si(0.9,300,1) = 278214.14151451  v. 1.0
'check value: air_f_enthalpy_si(0.9,300,1) = 278208.408749925  v. 1.1

Dim f As Double
Dim f_t As Double
Dim f_d As Double

air_f_enthalpy_si = ErrorReturn

f = air_f_si(0, 0, 0, a_si, t_si, d_si)
If f = ErrorReturn Then Exit Function
f_t = air_f_si(0, 1, 0, a_si, t_si, d_si)
If f_t = ErrorReturn Then Exit Function
f_d = air_f_si(0, 0, 1, a_si, t_si, d_si)
If f_d = ErrorReturn Then Exit Function

air_f_enthalpy_si = f - t_si * f_t + d_si * f_d

End Function

'==========================================================================
Public Function air_f_entropy_si(ByVal a_si As Double, ByVal t_si As Double, ByVal d_si As Double) As Double

'returns   eta = - (df/dT)_A_D entropy in J/kg K,
'a_si      mass fraction of dry air in humid air
't_si      absolute temperature in K,
'd_si      humid-air density in kg m-3

'check value: air_f_entropy_si(0.9,300,1) = 940.150652240964  v. 1.0
'check value: air_f_entropy_si(0.9,300,1) = 940.175394022848  v. 1.1

Dim f_t As Double

air_f_entropy_si = ErrorReturn

f_t = air_f_si(0, 1, 0, a_si, t_si, d_si)
If f_t = ErrorReturn Then Exit Function

air_f_entropy_si = -f_t

End Function

'==========================================================================
Public Function air_f_expansion_si(ByVal a_si As Double, ByVal t_si As Double, ByVal d_si As Double) As Double

'returns   alpha = -(1/D) * (dD/dT)_A_P  thermal expansion in 1/K,
'a_si      mass fraction of dry air in humid air
't_si      absolute temperature in K,
'd_si      humid-air density in kg m-3

'check value: air_f_expansion_si(0.9,300,1) = 3.43191318697755E-03  v. 1.0
'check value: air_f_expansion_si(0.9,300,1) = 3.4319303307666E-03  v. 1.1

Dim f_d As Double
Dim f_td As Double
Dim f_dd As Double
Dim x As Double

air_f_expansion_si = ErrorReturn

f_d = air_f_si(0, 0, 1, a_si, t_si, d_si)
If f_d = ErrorReturn Then Exit Function
f_td = air_f_si(0, 1, 1, a_si, t_si, d_si)
If f_td = ErrorReturn Then Exit Function
f_dd = air_f_si(0, 0, 2, a_si, t_si, d_si)
If f_dd = ErrorReturn Then Exit Function

x = 2 * f_d + d_si * f_dd
If x = 0 Then Exit Function

air_f_expansion_si = f_td / x

End Function

'==========================================================================
Public Function air_f_gibbs_energy_si(ByVal a_si As Double, ByVal t_si As Double, ByVal d_si As Double) As Double

'returns   g = f + D*(df/dD)_A_T   Gibbs energy in J/kg,
'a_si      mass fraction of dry air in humid air
't_si      absolute temperature in K,
'd_si      humid-air density in kg m-3

'check value: air_f_gibbs_energy_si(0.9,300,1) = -3831.0541577792  v. 1.0
'check value: air_f_gibbs_energy_si(0.9,300,1) = -3844.20945692934  v. 1.1

Dim f As Double
Dim f_d As Double

air_f_gibbs_energy_si = ErrorReturn

f = air_f_si(0, 0, 0, a_si, t_si, d_si)
If f = ErrorReturn Then Exit Function
f_d = air_f_si(0, 0, 1, a_si, t_si, d_si)
If f_d = ErrorReturn Then Exit Function

air_f_gibbs_energy_si = f + d_si * f_d

End Function

'==========================================================================
Public Function air_f_internal_energy_si(ByVal a_si As Double, ByVal t_si As Double, ByVal d_si As Double) As Double

'returns   u = f - T*(df/dT)_A_D internal energy in J/kg,
'a_si      mass fraction of dry air in humid air
't_si      absolute temperature in K,
'd_si      humid-air density in kg m-3

'check value: air_f_internal_energy_si(0.9,300,1) = 187020.405668534  v. 1.0
'check value: air_f_internal_energy_si(0.9,300,1) = 187033.023883765  v. 1.1

Dim f As Double
Dim f_t As Double

air_f_internal_energy_si = ErrorReturn

f = air_f_si(0, 0, 0, a_si, t_si, d_si)
If f = ErrorReturn Then Exit Function
f_t = air_f_si(0, 1, 0, a_si, t_si, d_si)
If f_t = ErrorReturn Then Exit Function

air_f_internal_energy_si = f - t_si * f_t

End Function

'==========================================================================
Public Function air_f_kappa_s_si(ByVal a_si As Double, ByVal t_si As Double, ByVal d_si As Double) As Double

'returns   kappa_s = (1/D) * (dD/dP)_A_eta  isentropic compressibility in 1/Pa,
'a_si      mass fraction of dry air in humid air
't_si      absolute temperature in K,
'd_si      humid-air density in kg m-3

'check value: air_f_kappa_s_si(0.9,300,1) = 8.07746486644041E-06  v. 1.0
'check value: air_f_kappa_s_si(0.9,300,1) = 8.0791362681566E-06   v. 1.1

Dim f_d As Double
Dim f_tt As Double
Dim f_td As Double
Dim f_dd As Double
Dim x As Double

air_f_kappa_s_si = ErrorReturn

f_d = air_f_si(0, 0, 1, a_si, t_si, d_si)
If f_d = ErrorReturn Then Exit Function
f_tt = air_f_si(0, 2, 0, a_si, t_si, d_si)
If f_tt = ErrorReturn Then Exit Function
f_td = air_f_si(0, 1, 1, a_si, t_si, d_si)
If f_td = ErrorReturn Then Exit Function
f_dd = air_f_si(0, 0, 2, a_si, t_si, d_si)
If f_dd = ErrorReturn Then Exit Function

x = d_si ^ 2 * (f_tt * (2 * f_d + d_si * f_dd) - d_si * f_td ^ 2)
If x = 0 Then Exit Function

air_f_kappa_s_si = f_tt / x

End Function

'==========================================================================
Public Function air_f_kappa_t_si(ByVal a_si As Double, ByVal t_si As Double, ByVal d_si As Double) As Double

'returns   kappa_t = (1/D) * (dD/dP)_A_T  isothermal compressibility in 1/Pa,
'a_si      mass fraction of dry air in humid air
't_si      absolute temperature in K,
'd_si      humid-air density in kg m-3

'check value: air_f_kappa_t_si(0.9,300,1) = 1.09953341546321E-05  v. 1.0
'check value: air_f_kappa_t_si(0.9,300,1) = 1.0997552139579E-05  v. 1.1

Dim f_d As Double
Dim f_dd As Double
Dim x As Double

air_f_kappa_t_si = ErrorReturn

f_d = air_f_si(0, 0, 1, a_si, t_si, d_si)
If f_d = ErrorReturn Then Exit Function
f_dd = air_f_si(0, 0, 2, a_si, t_si, d_si)
If f_dd = ErrorReturn Then Exit Function

x = d_si ^ 2 * (2 * f_d + d_si * f_dd)
If x = 0 Then Exit Function

air_f_kappa_t_si = 1 / x

End Function

'==========================================================================
Public Function air_f_lapserate_si(ByVal a_si As Double, ByVal t_si As Double, ByVal d_si As Double) As Double

'returns   gamma = (dT/dP)_A_eta  "dry" adiabatic lapse rate in K/Pa,
'a_si      mass fraction of dry air in humid air
't_si      absolute temperature in K,
'd_si      humid-air density in kg m-3

'check value: air_f_lapserate_si(0.9,300,1) = 8.50216520413048E-04  v. 1.0
'check value: air_f_lapserate_si(0.9,300,1) = 8.50371537341341E-04  v. 1.1

Dim f_d As Double
Dim f_tt As Double
Dim f_td As Double
Dim f_dd As Double
Dim x As Double

air_f_lapserate_si = ErrorReturn

f_d = air_f_si(0, 0, 1, a_si, t_si, d_si)
If f_d = ErrorReturn Then Exit Function
f_tt = air_f_si(0, 2, 0, a_si, t_si, d_si)
If f_tt = ErrorReturn Then Exit Function
f_td = air_f_si(0, 1, 1, a_si, t_si, d_si)
If f_td = ErrorReturn Then Exit Function
f_dd = air_f_si(0, 0, 2, a_si, t_si, d_si)
If f_dd = ErrorReturn Then Exit Function

x = d_si * (f_tt * (2 * f_d + d_si * f_dd) - d_si * f_td ^ 2)
If x = 0 Then Exit Function

air_f_lapserate_si = -f_td / x

End Function

'==========================================================================
Public Function air_f_pressure_si(ByVal a_si As Double, ByVal t_si As Double, ByVal d_si As Double) As Double

'returns   P = D^2*(df/dD)_A_T  pressure in Pa,
'a_si      mass fraction of dry air in humid air
't_si      absolute temperature in K,
'd_si      humid-air density in kg m-3

'check value: air_f_pressure_si(0.9,300,1) = 91193.7358459761  v. 1.0
'check value: air_f_pressure_si(0.9,300,1) = 91175.3848661597  v. 1.1

Dim f_d As Double

air_f_pressure_si = ErrorReturn

f_d = air_f_si(0, 0, 1, a_si, t_si, d_si)
If f_d = ErrorReturn Then Exit Function

air_f_pressure_si = d_si ^ 2 * f_d

End Function

'==========================================================================
Public Function air_f_soundspeed_si(ByVal a_si As Double, ByVal t_si As Double, ByVal d_si As Double) As Double

'returns   c = sqr[ (dP/dD)_A_eta ]  sound speed in m/s,
'a_si      mass fraction of dry air in humid air
't_si      absolute temperature in K,
'd_si      humid-air density in kg m-3

'check value: air_f_soundspeed_si(0.9,300,1) = 351.853974518412  v. 1.0
'check value: air_f_soundspeed_si(0.9,300,1) = 351.817577078371  v. 1.1

Dim f_d As Double
Dim f_tt As Double
Dim f_td As Double
Dim f_dd As Double
Dim x As Double

air_f_soundspeed_si = ErrorReturn

f_d = air_f_si(0, 0, 1, a_si, t_si, d_si)
If f_d = ErrorReturn Then Exit Function
f_tt = air_f_si(0, 2, 0, a_si, t_si, d_si)
If f_tt = ErrorReturn Then Exit Function
f_td = air_f_si(0, 1, 1, a_si, t_si, d_si)
If f_td = ErrorReturn Then Exit Function
f_dd = air_f_si(0, 0, 2, a_si, t_si, d_si)
If f_dd = ErrorReturn Then Exit Function

If f_tt = 0 Then Exit Function
x = 2 * d_si * f_d + d_si ^ 2 * (f_dd - f_td ^ 2 / f_tt)
If x < 0 Then Exit Function

air_f_soundspeed_si = Sqr(x)

End Function

'=========================================================================
Public Function chk_IAPWS10_Table13() As String

'TABLE 13  Numerical check values for the Helmholtz function f and its derivatives,
'Table 6, for saturated air, A = Asat, with respect to ice at 200 K, and with respect
'to liquid water at 300 and 400 K. The numerical functions evaluated here at given
'points (A, T, d) are defined in Tables 5, 6 and 12.

'Version 1.0 (this checktable was in a preliminary version of IAPWS-10):
'Quantity         Value            Value            Value           Unit
'A                0.892224944      0.977600624      0.825531379     kg kg-1
'T                200              300              400             K
'd                0.163445112E-4   0.114587678E+1   0.793198757E+1  kg m-3
'p                0.100000000E+1   0.100000000E+6   0.100000000E+7  Pa
'f               -0.682237558E+6  -0.927933016E+5   0.240393277E+5  J kg-1
'(df/dA)_T,d     -0.572842114E+6  -0.264760901E+3   0.311145187E+6  J kg-1
'(df/dT)_A,d     -0.405403634E+4  -0.296780108E+3  -0.106912744E+4  J kg-1 K-1
'(df/dd)_A,T      0.374331286E+10  0.761595138E+5   0.158941003E+5  m3 kg-1
'(d2f/dA2)_T,d    0.920803547E+6   0.624746061E+7   0.113770932E+7  J kg-1
'(d2f/dAdT)_d     0.915588345E+4   0.822737763E+4   0.702643265E+4  J kg-1 K-1
'(d2f/dAdd)_T    -0.213404004E+10 -0.449931362E+5  -0.727775043E+4  m3 kg-1
'(d2f/dT2)_A,d   -0.394095200E+1  -0.244799624E+1  -0.222492603E+1  J kg-1 K-2
'(d2f/dTdd)_A     0.187166127E+8   0.254574163E+3   0.414512281E+2  m3 kg-1 K-1
'(d2f/dd2)_A,T   -0.229025760E+15 -0.664927281E+5  -0.202004713E+4  m3 kg-1 Pa-1
'h                0.189752330E+6   0.835101493E+5   0.577762111E+6  J kg-1
'g               -0.621054939E+6  -0.552388319E+4   0.150111134E+6  J kg-1
's                0.405403634E+4   0.296780108E+3   0.106912744E+4  J kg-1 K-1
'w              -0.109950916E+6  -0.526505277E+4  -0.106748982E+6  J kg-1
'cp               0.109410518E+4   0.102705097E+4   0.123576406E+4  J kg-1 K-1
'w                0.291425752E+3   0.349274633E+3   0.416697519E+3  m s-1

'Version 1.1:
'Quantity         Value            Value            Value           Unit
'A                0.892247719      0.977605798      0.825565291     kg kg-1
'T                200              300              400             K
'd                0.163479657E-4   0.114614216E+1   0.793354063E+1  kg m-3
'p                0.999999998      0.100000000E+6   0.100000000E+7  Pa
'f               -0.682093392E+6  -0.927718178E+5   0.240345570E+5  J kg-1
'(df/dA)_T,d     -0.572680404E+6  -0.263453864E+3   0.311096733E+6  J kg-1
'(df/dT)_A,d     -0.405317966E+4  -0.296711481E+3  -0.106891931E+4  J kg-1 K-1
'(df/dd)_A,T      0.374173101E+10  0.761242496E+5   0.158878781E+5  m3 kg-1
'(d2f/dA2)_T,d    0.920967684E+6   0.624886233E+7   0.113786423E+7  J kg-1
'(d2f/dAdT)_d     0.915653743E+4   0.822733446E+4   0.702631471E+4  J kg-1 K-1
'(d2f/dAdd)_T    -0.213442099E+10 -0.450004399E+5  -0.727972651E+4  m3 kg-1
'(d2f/dT2)_A,d   -0.394011921E+1  -0.244742952E+1  -0.222449294E+1  J kg-1 K-2
'(d2f/dTdd)_A     0.187087034E+8   0.254456302E+3   0.414350772E+2  m3 kg-1 K-1
'(d2f/dd2)_A,T   -0.228880603E+15 -0.664465525E+5  -0.201886184E+4  m3 kg-1 Pa-1
'h                0.189712231E+6   0.834908383E+5   0.577649408E+6  J kg-1
'g               -0.620923701E+6  -0.552260595E+4   0.150081684E+6  J kg-1
's                0.405317966E+4   0.296711481E+3   0.106891931E+4  J kg-1 K-1
'w              -0.109950917E+6  -0.526505193E+4  -0.106748981E+6  J kg-1
'cp               0.109387397E+4   0.102681324E+4   0.123552454E+4  J kg-1 K-1
'w                0.291394959E+3   0.349234196E+3   0.416656820E+3  m s-1


Dim CRLF As String
Dim txt As String, row As String
Dim a As Double, t As Double, d As Double, q(3) As Double
Dim i As Integer, j As Integer

CRLF = Chr(13) + Chr(10)

txt = " Implementation of IAPWS-10 in Visual Basic" + CRLF
txt = txt + " for Publication in Ocean Science, 2009" + CRLF
txt = txt + " R. Feistel, IOW, Version " + Version + CRLF
txt = txt + " Compiled on " + CStr(Now) + CRLF + CRLF

txt = txt + " Function values as given in Table 13 of IAPWS-10:" + CRLF
txt = txt + " Numerical check values for the Helmholtz function f and its derivatives," + CRLF
txt = txt + " Table 6, for saturated air, A = Asat, with respect to ice at 200 K, and with respect" + CRLF
txt = txt + " to liquid water at 300 and 400 K. The numerical functions evaluated here at given" + CRLF
txt = txt + " points (A, T, d) are defined in Tables 5, 6 and 12." + CRLF + CRLF

txt = txt + "Quantity         Value             Value             Value            Unit" + CRLF

For i = 1 To 20
  Select Case i

    Case 1:  row = "A                0.892247719       0.977605798       0.825565291      kg kg-1"
    Case 2:  row = "T                200               300               400              K"
    Case 3:  row = "d                0.163479657E-4    0.114614216E+1    0.793354063E+1   kg m-3"
    Case 4:  row = "p                0.999999998       0.100000000E+6    0.100000000E+7   Pa"
    Case 5:  row = "f               -0.682093392E+6   -0.927718178E+5    0.240345570E+5   J kg-1"
    Case 6:  row = "(df/dA)_T,d     -0.572680404E+6   -0.263453864E+3    0.311096733E+6   J kg-1"
    Case 7:  row = "(df/dT)_A,d     -0.405317966E+4   -0.296711481E+3   -0.106891931E+4   J kg-1 K-1"
    Case 8:  row = "(df/dd)_A,T      0.374173101E+10   0.761242496E+5    0.158878781E+5   m3 kg-1"
    Case 9:  row = "(d2f/dA2)_T,d    0.920967684E+6    0.624886233E+7    0.113786423E+7   J kg-1"
    Case 10: row = "(d2f/dAdT)_d     0.915653743E+4    0.822733446E+4    0.702631471E+4   J kg-1 K-1"
    Case 11: row = "(d2f/dAdd)_T    -0.213442099E+10  -0.450004399E+5   -0.727972651E+4   m3 kg-1"
    Case 12: row = "(d2f/dT2)_A,d   -0.394011921E+1   -0.244742952E+1   -0.222449294E+1   J kg-1 K-2"
    Case 13: row = "(d2f/dTdd)_A     0.187087034E+8    0.254456302E+3    0.414350772E+2   m3 kg-1 K-1"
    Case 14: row = "(d2f/dd2)_A,T   -0.228880603E+15  -0.664465525E+5   -0.201886184E+4   m3 kg-1 Pa-1"
    Case 15: row = "h                0.189712231E+6    0.834908383E+5    0.577649408E+6   J kg-1"
    Case 16: row = "g               -0.620923701E+6   -0.552260595E+4    0.150081684E+6   J kg-1"
    Case 17: row = "s                0.405317966E+4    0.296711481E+3    0.106891931E+4   J kg-1 K-1"
    Case 18: row = "w              -0.109950917E+6   -0.526505193E+4   -0.106748981E+6   J kg-1"
    Case 19: row = "cp               0.109387397E+4    0.102681324E+4    0.123552454E+4   J kg-1 K-1"
    Case 20: row = "w                0.291394959E+3    0.349234196E+3    0.416656820E+3   m s-1"
  End Select

  txt = txt + row + CRLF
  If i > 3 Then txt = txt + "this code:      "
  
  For j = 1 To 3
    a = Choose(j, 0.892247719, 0.977605798, 0.825565291)
    t = Choose(j, 200, 300, 400)
    d = Choose(j, 0.0000163479657, 1.14614216, 7.93354063)
    q(j) = 0
    Select Case i
      Case 1: q(j) = a
      Case 2: q(j) = t
      Case 3: q(j) = d
      Case 4: q(j) = air_f_pressure_si(a, t, d)
      Case 5: q(j) = air_f_si(0, 0, 0, a, t, d)
      Case 6: q(j) = air_f_si(1, 0, 0, a, t, d)
      Case 7: q(j) = air_f_si(0, 1, 0, a, t, d)
      Case 8: q(j) = air_f_si(0, 0, 1, a, t, d)
      Case 9: q(j) = air_f_si(2, 0, 0, a, t, d)
      Case 10: q(j) = air_f_si(1, 1, 0, a, t, d)
      Case 11: q(j) = air_f_si(1, 0, 1, a, t, d)
      Case 12: q(j) = air_f_si(0, 2, 0, a, t, d)
      Case 13: q(j) = air_f_si(0, 1, 1, a, t, d)
      Case 14: q(j) = air_f_si(0, 0, 2, a, t, d)

      Case 15: q(j) = air_f_enthalpy_si(a, t, d)
      Case 16: q(j) = air_f_gibbs_energy_si(a, t, d)
      Case 17: q(j) = air_f_entropy_si(a, t, d)
      Case 18: q(j) = air_f_gibbs_energy_si(a, t, d) - a * air_f_si(1, 0, 0, a, t, d)
      Case 19: q(j) = air_f_cp_si(a, t, d)
      Case 20: q(j) = air_f_soundspeed_si(a, t, d)
    End Select
    
    If i > 3 Then txt = txt + Left(EFormat(q(j), 9) + Space(18), 18)
  Next j
  
  If i > 3 Then txt = txt + CRLF + CRLF
Next i

chk_IAPWS10_Table13 = txt

End Function

'=========================================================================
Public Function chk_IAPWS10_Table14() As String

'TABLE 14  Numerical check values for the dry-air part, fA, and the water-vapor part, fV,
'of the Helmholtz function fAV and its derivatives, Table 6, for saturated air, A = Asat,
'with respect to ice at 200 K, and with respect to liquid water at 300 and 400 K,
'corresponding to Table 13. The numerical functions evaluated here at given points (A, T, d)
'are defined in Tables 7 and 8.

'Version 1.0 (this checktable was in a preliminary version of IAPWS-10):
'Quantity         Value            Value            Value           Unit
'A                0.892224944      0.977600624      0.825531379     kg kg-1
'T                  200              300              400           K
'd                0.163445112E-4   0.114587678E+1   0.793198757E+1  kg m-3
'dA = A*d         0.145829806E-4   0.112020986E+1   0.654810464E+1  kg m-3
'fA              -0.740216453E+6  -0.916320468E+5   0.895773442E+5  J kg-1
'(dfA/dT)_dA     -0.304846375E+4  -0.108501917E+3   0.193317180E+3  J kg-1 K-1
'(dfA/ddA)_T      0.393770146E+10  0.768690856E+5   0.175643300E+5  m3 kg-1
'(d2fA/dT2)_dA   -0.357762609E+1  -0.239376632E+1  -0.181852945E+1  J kg-1 K-2
' d2fA /dTddA     0.196885083E+8   0.256804931E+3   0.442979471E+2  m3 kg-1 K-1
'[d2fA/d(dA)2]_T -0.270020351E+15 -0.686404948E+5  -0.267826171E+4  m3 kg-1 Pa-1
'dV = (1-A)*d     0.176153061E-5   0.256669248E-1   0.138388293E+1  kg m-3
'fV              -0.202254350E+6  -0.143157503E+6  -0.285137783E+6  J kg-1
'(dfV/dT)_dV     -0.123787544E+5  -0.851598239E+4  -0.705288118E+4  J kg-1 K-1
'(dfV/ddV)_T      0.523995669E+11  0.538480918E+7   0.129645224E+6  m3 kg-1
'(d2fV/dT2)_dV   -0.694877601E+1  -0.480816998E+1  -0.411710547E+1  J kg-1 K-2
' d2fV/dTddV      0.262001883E+9   0.181489601E+5   0.361784532E+3  m3 kg-1 K-1
'[d2fV/d(dV)2]T  -0.297466666E+17 -0.210185225E+9  -0.965542132E+5  m3 kg-1 Pa-1

'Version 1.1:
'Quantity         Value            Value            Value           Unit
'A                0.892247719      0.977605798      0.825565291     kg kg-1
'T                  200              300              400           K
'd                0.163479657E-4   0.114614216E+1   0.793354063E+1  kg m-3
'dA = A*d         0.145864351E-4   0.112047522E+1   0.654965578E+1  kg m-3
'fA              -0.740041144E+6  -0.916103453E+5   0.895561286E+5  J kg-1
'(dfA/dT)_dA     -0.304774177E+4  -0.108476220E+3   0.193271394E+3  J kg-1 K-1
'(dfA/ddA)_T      0.393583654E+10  0.768326795E+5   0.175560114E+5  m3 kg-1
'(d2fA/dT2)_dA   -0.357677878E+1  -0.239319940E+1  -0.181809877E+1  J kg-1 K-2
' d2fA /dTddA     0.196791837E+8   0.256683306E+3   0.442769673E+2  m3 kg-1 K-1
'[d2fA/d(dA)2]_T -0.269828549E+15 -0.685917373E+5  -0.267635928E+4  m3 kg-1 Pa-1
'dV = (1-A)*d     0.176153059E-5   0.256669391E-1   0.138388485E+1  kg m-3
'fV              -0.202254351E+6  -0.143157426E+6  -0.285137534E+6  J kg-1
'(dfV/dT)_dV     -0.123787544E+5  -0.851598213E+4  -0.705288048E+4  J kg-1 K-1
'(dfV/ddV)_T      0.523995674E+11  0.538480619E+7   0.129645039E+6  m3 kg-1
'(d2fV/dT2)_dV   -0.694877601E+1  -0.480817011E+1  -0.411710659E+1  J kg-1 K-2
' d2fV/dTddV      0.262001885E+9   0.181489502E+5   0.361784086E+3  m3 kg-1 K-1
'[d2fV/d(dV)2]T  -0.297466671E+17 -0.210184992E+9  -0.965539462E+5  m3 kg-1 Pa-1

Dim CRLF As String
Dim txt As String, row As String
Dim a As Double, t As Double, d As Double, q(3) As Double
Dim i As Integer, j As Integer

CRLF = Chr(13) + Chr(10)

txt = " Implementation of IAPWS-10 in Visual Basic" + CRLF
txt = txt + " for Publication in Ocean Science, 2009" + CRLF
txt = txt + " R. Feistel, IOW, Version " + Version + CRLF
txt = txt + " Compiled on " + CStr(Now) + CRLF + CRLF

txt = txt + " Function values as given in Table 14 of IAPWS-10:" + CRLF
txt = txt + " Numerical check values for the dry-air part, fA, and the water-vapor part, fV," + CRLF
txt = txt + " of the Helmholtz function fAV and its derivatives, Table 6, for saturated air, A = Asat," + CRLF
txt = txt + " with respect to ice at 200 K, and with respect to liquid water at 300 and 400 K," + CRLF
txt = txt + " corresponding to Table 13. The numerical functions evaluated here at given points (A, T, d)" + CRLF
txt = txt + " are defined in Tables 7 and 8." + CRLF + CRLF

txt = txt + "Quantity         Value             Value             Value            Unit" + CRLF

For i = 1 To 17
  Select Case i
    Case 1:  row = "A                0.892247719      0.977605798      0.825565291     kg kg-1"
    Case 2:  row = "T                  200              300              400           K"
    Case 3:  row = "d                0.163479657E-4   0.114614216E+1   0.793354063E+1  kg m-3"
    Case 4:  row = "dA = A*d         0.145864351E-4   0.112047522E+1   0.654965578E+1  kg m-3"
    Case 5:  row = "fA              -0.740041144E+6  -0.916103453E+5   0.895561286E+5  J kg-1"
    Case 6:  row = "(dfA/dT)_dA     -0.304774177E+4  -0.108476220E+3   0.193271394E+3  J kg-1 K-1"
    Case 7:  row = "(dfA/ddA)_T      0.393583654E+10  0.768326795E+5   0.175560114E+5  m3 kg-1"
    Case 8:  row = "(d2fA/dT2)_dA   -0.357677878E+1  -0.239319940E+1  -0.181809877E+1  J kg-1 K-2"
    Case 9:  row = " d2fA /dTddA     0.196791837E+8   0.256683306E+3   0.442769673E+2  m3 kg-1 K-1"
    Case 10: row = "[d2fA/d(dA)2]_T -0.269828549E+15 -0.685917373E+5  -0.267635928E+4  m3 kg-1 Pa-1"
    Case 11: row = "dV = (1-A)*d     0.176153059E-5   0.256669391E-1   0.138388485E+1  kg m-3"
    Case 12: row = "fV              -0.202254351E+6  -0.143157426E+6  -0.285137534E+6  J kg-1"
    Case 13: row = "(dfV/dT)_dV     -0.123787544E+5  -0.851598213E+4  -0.705288048E+4  J kg-1 K-1"
    Case 14: row = "(dfV/ddV)_T      0.523995674E+11  0.538480619E+7   0.129645039E+6  m3 kg-1"
    Case 15: row = "(d2fV/dT2)_dV   -0.694877601E+1  -0.480817011E+1  -0.411710659E+1  J kg-1 K-2"
    Case 16: row = " d2fV/dTddV      0.262001885E+9   0.181489502E+5   0.361784086E+3  m3 kg-1 K-1"
    Case 17: row = "[d2fV/d(dV)2]T  -0.297466671E+17 -0.210184992E+9  -0.965539462E+5  m3 kg-1 Pa-1"
  End Select

  txt = txt + row + CRLF
  If i = 3 Then txt = txt + CRLF
  If i > 3 Then txt = txt + "this code:      "
  
  For j = 1 To 3
    a = Choose(j, 0.892247719, 0.977605798, 0.825565291)
    t = Choose(j, 200, 300, 400)
    d = Choose(j, 0.0000163479657, 1.14614216, 7.93354063)
    q(j) = 0
    Select Case i
      Case 1: q(j) = a
      Case 2: q(j) = t
      Case 3: q(j) = d

      Case 4:  q(j) = a * d
      Case 5:  q(j) = dry_f_si(0, 0, t, a * d)
      Case 6:  q(j) = dry_f_si(1, 0, t, a * d)
      Case 7:  q(j) = dry_f_si(0, 1, t, a * d)
      Case 8:  q(j) = dry_f_si(2, 0, t, a * d)
      Case 9:  q(j) = dry_f_si(1, 1, t, a * d)
      Case 10: q(j) = dry_f_si(0, 2, t, a * d)

      Case 11: q(j) = (1 - a) * d
      Case 12: q(j) = flu_f_si(0, 0, t, (1 - a) * d)
      Case 13: q(j) = flu_f_si(1, 0, t, (1 - a) * d)
      Case 14: q(j) = flu_f_si(0, 1, t, (1 - a) * d)
      Case 15: q(j) = flu_f_si(2, 0, t, (1 - a) * d)
      Case 16: q(j) = flu_f_si(1, 1, t, (1 - a) * d)
      Case 17: q(j) = flu_f_si(0, 2, t, (1 - a) * d)
    End Select
    
    If i > 3 Then txt = txt + Left(EFormat(q(j), 9) + Space(17), 17)
  Next j
  If i > 3 Then txt = txt + CRLF + CRLF
Next i

chk_IAPWS10_Table14 = txt

End Function

'=========================================================================
Public Function chk_IAPWS10_Table15() As String

'TABLE 15  Numerical check values for the cross-virial part, fmix, and
'the cross-virial coefficients, BAW, CAAW, CAWW, of the Helmholtz function fAV
'and its derivatives, Table 6, for saturated air, A = Asat, with respect to ice
'at 200 K, and with respect to liquid water at 300 and 400 K, corresponding to
'Table 13. The numerical functions evaluated here at given points (A, T, d) are
'defined in Tables 10 and 11.
'
'Quantity           Value            Value            Value            Unit
'A                  0.892247719      0.977605798      0.825565291      kg kg-1
'T                    200              300              400            K
'd                  0.163479657E-4   0.114614216E+1   0.793354063E+1   kg m-3
'fmix              -0.786231899E-3  -0.711677596E+1  -0.161991543E+3   J kg-1
'(dfmix/dA)_T,d     0.641550398E-2   0.311844020E+3   0.831044354E+3   J kg-1
'(dfmix/dT)_A,d     0.456438658E-5   0.441247962E-1   0.178968942E+1   J kg-1 K-1
'(dfmix/dd)_A,T    -0.480937188E+2  -0.623030392E+1  -0.223330257E+2   m3 kg-1
'(d2fmix/dA2)_T,d   0.163552956E-1   0.534234669E+3   0.135814949E+4   J kg-1
'(d2fmix/dAdT)_d   -0.372455576E-4  -0.195073372E+1  -0.916854756E+1   J kg-1 K-1
'(d2fmix/dAdd)_T    0.392437132E+3   0.274155508E+3   0.125834930E+3   m3 kg-1
'(d2fmix/dT2)_A,d  -0.378875706E-7  -0.148783177E-3  -0.536741578E-2   J kg-1 K-2
'(d2fmix/dTdd)_A    0.279209778      0.390012443E-1   0.249580143      m3 kg-1 K-1
'(d2fmix/dd2)_A,T  -0.192042557E+2  -0.365975429E-1  -0.482623664      m3 kg-1 Pa-1
'Baw               -0.784874278E-4  -0.295672747E-4  -0.100804610E-4   m3 mol-1
'dBaw/dT            0.848076624E-6   0.280097360E-6   0.135021228E-6   m3 mol-1 K-1
'd2Baw/dT2         -0.122622146E-7  -0.242599241E-8  -0.839901729E-9   m3 mol-1 K-2
'Caaw               0.105493575E-8   0.801977741E-9   0.672018172E-9   m6 mol-2
'dCaaw/dT          -0.152535000E-11 -0.196103457E-11 -0.812416406E-12  m6 mol-2 K-1
'd2Caaw/dT2        -0.113436375E-12  0.170055638E-13  0.683147461E-14  m6 mol-2 K-2
'Caww              -0.349872634E-5  -0.115552784E-6  -0.200806021E-7   m6 mol-2
'dCaww/dT           0.188025052E-6   0.261363278E-8   0.274535403E-9   m6 mol-2 K-1
'd2Caww/dT2        -0.124996856E-7  -0.751334582E-10 -0.491763910E-11  m6 mol-2 K-2

Dim CRLF As String
Dim txt As String, row As String
Dim a As Double, t As Double, d As Double, q(3) As Double
Dim i As Integer, j As Integer

CRLF = Chr(13) + Chr(10)

txt = " Implementation of IAPWS-10 in Visual Basic" + CRLF
txt = txt + " for Publication in Ocean Science, 2009" + CRLF
txt = txt + " R. Feistel, IOW, Version " + Version + CRLF
txt = txt + " Compiled on " + CStr(Now) + CRLF + CRLF

txt = txt + " Function values as given in Table 15 of IAPWS-10:" + CRLF
txt = txt + " Numerical check values for the cross-virial part, fmix, and" + CRLF
txt = txt + " the cross-virial coefficients, BAW, CAAW, CAWW, of the Helmholtz function fAV" + CRLF
txt = txt + " and its derivatives, Table 6, for saturated air, A = Asat, with respect to ice" + CRLF
txt = txt + " at 200 K, and with respect to liquid water at 300 and 400 K, corresponding to" + CRLF
txt = txt + " Table 13. The numerical functions evaluated here at given points (A, T, d) are" + CRLF
txt = txt + " defined in Tables 10 and 11." + CRLF + CRLF

txt = txt + "Quantity           Value            Value            Value            Unit" + CRLF

For i = 1 To 22
  Select Case i
    Case 1:  row = "A                  0.892247719      0.977605798      0.825565291      kg kg-1"
    Case 2:  row = "T                    200              300              400            K"
    Case 3:  row = "d                  0.163479657E-4   0.114614216E+1   0.793354063E+1   kg m-3"
    Case 4:  row = "fmix              -0.786231899E-3  -0.711677596E+1  -0.161991543E+3   J kg-1"
    Case 5:  row = "(dfmix/dA)_T,d     0.641550398E-2   0.311844020E+3   0.831044354E+3   J kg-1"
    Case 6:  row = "(dfmix/dT)_A,d     0.456438658E-5   0.441247962E-1   0.178968942E+1   J kg-1 K-1"
    Case 7:  row = "(dfmix/dd)_A,T    -0.480937188E+2  -0.623030392E+1  -0.223330257E+2   m3 kg-1"
    Case 8:  row = "(d2fmix/dA2)_T,d   0.163552956E-1   0.534234669E+3   0.135814949E+4   J kg-1"
    Case 9:  row = "(d2fmix/dAdT)_d   -0.372455576E-4  -0.195073372E+1  -0.916854756E+1   J kg-1 K-1"
    Case 10: row = "(d2fmix/dAdd)_T    0.392437132E+3   0.274155508E+3   0.125834930E+3   m3 kg-1"
    Case 11: row = "(d2fmix/dT2)_A,d  -0.378875706E-7  -0.148783177E-3  -0.536741578E-2   J kg-1 K-2"
    Case 12: row = "(d2fmix/dTdd)_A    0.279209778      0.390012443E-1   0.249580143      m3 kg-1 K-1"
    Case 13: row = "(d2fmix/dd2)_A,T  -0.192042557E+2  -0.365975429E-1  -0.482623664      m3 kg-1 Pa-1"
    Case 14: row = "Baw               -0.784874278E-4  -0.295672747E-4  -0.100804610E-4   m3 mol-1"
    Case 15: row = "dBaw/dT            0.848076624E-6   0.280097360E-6   0.135021228E-6   m3 mol-1 K-1"
    Case 16: row = "d2Baw/dT2         -0.122622146E-7  -0.242599241E-8  -0.839901729E-9   m3 mol-1 K-2"
    Case 17: row = "Caaw               0.105493575E-8   0.801977741E-9   0.672018172E-9   m6 mol-2"
    Case 18: row = "dCaaw/dT          -0.152535000E-11 -0.196103457E-11 -0.812416406E-12  m6 mol-2 K-1"
    Case 19: row = "d2Caaw/dT2        -0.113436375E-12  0.170055638E-13  0.683147461E-14  m6 mol-2 K-2"
    Case 20: row = "Caww              -0.349872634E-5  -0.115552784E-6  -0.200806021E-7   m6 mol-2"
    Case 21: row = "dCaww/dT           0.188025052E-6   0.261363278E-8   0.274535403E-9   m6 mol-2 K-1"
    Case 22: row = "d2Caww/dT2        -0.124996856E-7  -0.751334582E-10 -0.491763910E-11  m6 mol-2 K-2"
  End Select

  txt = txt + row + CRLF
  If i = 3 Then txt = txt + CRLF
  If i > 3 Then txt = txt + "this code:        "
  
  For j = 1 To 3
    a = Choose(j, 0.892247719, 0.977605798, 0.825565291)
    t = Choose(j, 200, 300, 400)
    d = Choose(j, 0.0000163479657, 1.14614216, 7.93354063)
    q(j) = 0
    Select Case i
      Case 1: q(j) = a
      Case 2: q(j) = t
      Case 3: q(j) = d

      Case 4: q(j) = air_f_mix_si(0, 0, 0, a, t, d)
      Case 5: q(j) = air_f_mix_si(1, 0, 0, a, t, d)
      Case 6: q(j) = air_f_mix_si(0, 1, 0, a, t, d)
      Case 7: q(j) = air_f_mix_si(0, 0, 1, a, t, d)
      Case 8: q(j) = air_f_mix_si(2, 0, 0, a, t, d)
      Case 9: q(j) = air_f_mix_si(1, 1, 0, a, t, d)
      Case 10: q(j) = air_f_mix_si(1, 0, 1, a, t, d)
      Case 11: q(j) = air_f_mix_si(0, 2, 0, a, t, d)
      Case 12: q(j) = air_f_mix_si(0, 1, 1, a, t, d)
      Case 13: q(j) = air_f_mix_si(0, 0, 2, a, t, d)

      Case 14: q(j) = air_baw_m3mol(0, t)
      Case 15: q(j) = air_baw_m3mol(1, t)
      Case 16: q(j) = air_baw_m3mol(2, t)

      Case 17: q(j) = air_caaw_m6mol2(0, t)
      Case 18: q(j) = air_caaw_m6mol2(1, t)
      Case 19: q(j) = air_caaw_m6mol2(2, t)

      Case 20: q(j) = air_caww_m6mol2(0, t)
      Case 21: q(j) = air_caww_m6mol2(1, t)
      Case 22: q(j) = air_caww_m6mol2(2, t)

    End Select
    
    If i > 3 Then txt = txt + Left(EFormat(q(j), 9) + Space(17), 17)
  Next j
  
  If i > 3 Then txt = txt + CRLF + CRLF
Next i

chk_IAPWS10_Table15 = txt

End Function

'=========================================================================
Public Function chk_IAPWS10_Table(ByVal number As Integer) As String

Select Case number
  Case 13: chk_IAPWS10_Table = chk_IAPWS10_Table13
  Case 14: chk_IAPWS10_Table = chk_IAPWS10_Table14
  Case 15: chk_IAPWS10_Table = chk_IAPWS10_Table15
  Case Else: chk_IAPWS10_Table = "Only Table 13, 14 or 15 is available"
End Select

End Function


